/*
Copyright (C) 2003 Hotsprings Inc.
For conditions of distribution and use, see copyright notice

Location: 
	www.HotspringsInc.com 

History:
	2003Aug27-GiuseppeG: code write
*/

#if TARGET_API_Win32 || TARGET_API_Win32_Console
#include "XSP_Core.h"
#include "XSP_File.h"
#include "XSP_GUI.h"

namespace XSP
{

UIEvent::UIEvent(UINT msg, WPARAM wp, LPARAM lp)
: origin(0)
, modifiers(0)
, code(0)
, count(1)
{
	switch (msg)
	{
	case WM_MOUSEMOVE:
	case 0x02A1: //WM_MOUSEHOVER                   
		origin = MouseMove;
		break;
	case WM_LBUTTONDOWN:
		origin = MouseClick;
		code = LeftMouse;
		break;
	case WM_RBUTTONDOWN:
		origin = MouseClick;
		code = RightMouse;
		break;
	case WM_MBUTTONDOWN:
		origin = MouseClick;
		code = MiddleMouse;
		break;
	case WM_LBUTTONUP:
		origin = MouseUp;
		code = LeftMouse;
		break;
	case WM_RBUTTONUP:
		origin = MouseUp;
		code = RightMouse;
		break;
	case WM_MBUTTONUP:
		origin = MouseUp;
		code = MiddleMouse;
		break;
	case WM_LBUTTONDBLCLK:
		origin = MouseDClick;
		code = LeftMouse;
		break;
	case WM_RBUTTONDBLCLK:
		origin = MouseDClick;
		code = RightMouse;
		break;
	case WM_MBUTTONDBLCLK:
		origin = MouseDClick;
		code = MiddleMouse;
		break;
	case 0x20A: // WM_MOUSEWHEEL:
		origin = MouseWheel;
		count = 1[(sint16*)&wp]; // 120 per line
		break;
	case 0x20B: //WM_XBUTTONDOWN
		origin = MouseClick;
		code = X1Mouse-1 +1[(sint16*)&wp];
		break;
	case 0x20C: //WM_XBUTTONUP    
		origin = MouseUp;
		code = X1Mouse-1 +1[(sint16*)&wp];
		break;
	case 0x20D: //WM_XBUTTONDBLCLK
		origin = MouseDClick;
		code = X1Mouse-1 +1[(sint16*)&wp];
		break;
	case 0x02A3: //WM_MOUSELEAVE
		origin = 0;
		return;
	case WM_HOTKEY: 
		// only comes to the THREAD that registered for it with the system
		// will not be dispatched to a window
		origin = KeyDown;
		code = wp;
		count = 1;
		break;
	case WM_KEYDOWN:
	case WM_SYSKEYDOWN:
		origin = KeyDown;
		code = wp;
		count = 0[(sint16*)&lp];
		break;
	case WM_KEYUP:
	case WM_SYSKEYUP:
		origin = KeyUp;
		code = wp;
		count = 0[(sint16*)&lp];
		break;
	case WM_SYSDEADCHAR: // ALT-char
	case WM_DEADCHAR:	 // first for 2 key chars
	case WM_CHAR:
		origin = AsciiCode;
		code = wp;
		count = 0[(sint16*)&lp];
		break;
	}
    if (origin < MouseMove)
    {
		mousePos.x = -1;
		mousePos.y = -1;
    }
    else
    {
		mousePos.x = 0[(sint16*)&lp];
		mousePos.y = 1[(sint16*)&lp];
	}
	BYTE keyState[256];
	if (::GetKeyboardState(keyState))
	{	
		if ( 0x80 & keyState[VK_CONTROL] )
			modifiers |= CtrlState;
		if ( 0x80 & keyState[VK_SHIFT] )
			modifiers |= ShiftState;
		if ( 0x80 & keyState[VK_MENU])
			modifiers |= AltOptionState;
		if ( keyState[VK_CAPITAL])
			modifiers |= CapsState;
		if ( 0x80 & (keyState[VK_LWIN] | keyState[VK_RWIN]) )
			modifiers |= OSState;
		if ( 0x80 & keyState[VK_APPS] )
			modifiers |= ContextState;
		if ( keyState[VK_NUMLOCK] )
			modifiers |= NumLockState;
		if ( keyState[VK_SCROLL] )
			modifiers |= ScrollState;

		if ( 0x80 & keyState[VK_LBUTTON] )
			modifiers |= MouseLeftState;
		if ( 0x80 & keyState[VK_RBUTTON] )
			modifiers |= MouseRightState;
		if ( 0x80 & keyState[VK_MBUTTON] )
			modifiers |= MouseMiddleState;
		if ( 0x80 & keyState[0x05] ) // VK_XBUTTON1
			modifiers |= MouseX1State;
		if ( 0x80 & keyState[0x06] ) // VK_XBUTTON2
			modifiers |= MouseX2State;
	}
}
WinMgr::_Data::_Data()
: refcount(0)
, wndClassAtom(0)
, moduleInstance(0)
, usedLastChar(false)
{	
	// initialize OLE for DND and clipboard operations
	// we do not care about the return code, if for some reason 
	// it fails we will simply not have DND available but the 
	// rest of the GUI will still work
	::OleInitialize(NULL);
	
	// the event processor does not own us
	processEvents = new EventProcessor(this);
}

WinMgr::_Data::~_Data()
{
	// the event processor is not allowed to know about us any more
	// and should be canceled so we don't get called any more
	processEvents->Cancel();
	
	// we're done with ole support
	::OleUninitialize();
	
    ASSERT(windowPool.empty());
    ::UnregisterClass((LPCTSTR)(uint32)wndClassAtom, moduleInstance);
}

WinMgr::WinMgr()
: data(new _Data)
{
	data->moduleInstance = (HINSTANCE) ::GetModuleHandle(NULL);				 
	data->arrowCursor  = MouseCursor(new MouseCursor::_Data(
		data->moduleInstance, MAKEINTRESOURCE(32512))); //IDC_ARROW
	data->textCursor  = MouseCursor(new MouseCursor::_Data(
		data->moduleInstance, MAKEINTRESOURCE(32513))); //IDC_IBEAM
	data->grabCursor  = MouseCursor(new MouseCursor::_Data(
		data->moduleInstance, MAKEINTRESOURCE(32649))); //IDC_HAND

	String className(String::From_c_str("WCLASS#")+
					 String::From_uint32((uint32)this, 16));
					 
    WNDCLASSEX wc; //  = data->windowClass;
	wc.cbSize = sizeof(wc);
	wc.style =  CS_DBLCLKS | CS_PARENTDC | CS_HREDRAW | CS_VREDRAW; //??
				//??	CS_DROPSHADOW = 0x020000 on WinXP
    wc.lpfnWndProc = WndProc;//	 WNDPROC     
    wc.cbClsExtra = 0;         
    wc.cbWndExtra = 0;         
    wc.hInstance  = data->moduleInstance;//	HINSTANCE   
    wc.hCursor 	  = NULL; //::LoadCursor(NULL, IDC_ARROW); // HCURSOR     
    wc.hbrBackground = NULL; //	 HBRUSH ::GetStockObject(WHITE_BRUSH); 
    wc.lpszMenuName  = NULL; //	 LPCSTR      
    wc.lpszClassName = className.c_str();	// LPCSTR      
//    wc.hIcon   = ::LoadIcon(NULL, IDI_APPLICATION); // ToDo resource IDI_APPLICATION	HICON       
    wc.hIcon   = (HICON) ::LoadImage(data->moduleInstance,
    						 MAKEINTRESOURCE(100), IMAGE_ICON,
    						 ::GetSystemMetrics(SM_CXICON), 
    						 ::GetSystemMetrics(SM_CYICON),
    						 LR_DEFAULTCOLOR ); // HICON
    wc.hIconSm = (HICON) ::LoadImage(data->moduleInstance,
    						 MAKEINTRESOURCE(100), IMAGE_ICON,
    						 ::GetSystemMetrics(SM_CXSMICON),
    						 ::GetSystemMetrics(SM_CYSMICON),
    						 LR_DEFAULTCOLOR ); // HICON
    data->wndClassAtom = ::RegisterClassEx(&wc);
    VERIFY(data->wndClassAtom != 0);

	RequestEvents(true);
}

uint32 WinMgr::GetWindowCount() const
{
	return data->windowPool.size();
}

void WinMgr::ProcessEvents()
{
    CheckMouseStatus(); // every once in a while

	uint32 eventCount = 0;
	for(; eventCount<100; ++eventCount) // process a bunch of msgs at a time
	{
		MSG msg;
		if (0 == ::PeekMessage(&msg, (HWND)NULL, 0,0, PM_REMOVE))
		{
			break; // no more msgs
		}

		if (msg.hwnd == 0)
		{	// message for this thread
			ThreadProc(msg);
		}
		else
		{	// message for a window
			if (0 != ::TranslateMessage(&msg))
			{  // the message was translated to an ascii char code
			   // and posted back to the message queue
			}

			// dispatch the message to the appropriate window
			{
				LONG ret = ::DispatchMessage(&msg);
			}
		}
	}
	// if we processed some events then we do not allow for sleep to kick in
	// if however no event was processed then we do not interfere with the sleep
	// process and that means the app will go to sleep for a short amount of time
	RequestEvents(eventCount>0);	
}

void WinMgr::ThreadProc(const MSG& msg)
{
	switch (msg.message)
	{
	case WM_SYSCOLORCHANGE:
	case WM_WININICHANGE:
	case WM_ACTIVATEAPP:
	case WM_USERCHANGED:
	case WM_NOTIFY:
	case WM_DISPLAYCHANGE:
	case WM_STYLECHANGED:
	case WM_ENTERIDLE:
	case WM_POWERBROADCAST:
	case WM_DEVICECHANGE:
	case WM_HOTKEY:
		::DispatchMessage(&msg); // ToDo
		break;
	
	case WM_QUIT:
		{
			XSP::CoreModule::_instance->mainLoop->PostEvent(
				XSP::CoreModule::_instance->mainLoop->CreateQuitEvent());
			XSP::AppExitCode::SetExitCode(msg.wParam);
			break;
		}
	};
}

LRESULT CALLBACK WinMgr::WndProc(HWND hwnd, UINT msg, WPARAM wp, LPARAM lp)
{
    Window::_Data* wdata = (Window::_Data*)::GetWindowLong(hwnd, GWL_USERDATA);
    if (wdata == 0)
		return ::DefWindowProc(hwnd, msg, wp, lp);

	_Data::WindowPool& pool = wdata->winMgr.data->windowPool;
	_Data::WindowPool::iterator w = pool.find(hwnd);
	if (w == pool.end()) // not our window ?!
		return ::DefWindowProc(hwnd, msg, wp, lp);

	
//	#if DEBUG
//		UnitTest::Log("WM_[%d = %08XH]\n", msg,msg);
//    #endif

	Window wnd = w->second;
	ASSERT(wnd.data->hwnd == hwnd);
	switch (msg)
	{
	case WM_PAINT:
		wnd.data->ProcPAINT(msg, wp,lp);
		return 0;
	case WM_NCPAINT:
		break;
	case WM_ERASEBKGND:
		return 1;
/*	case WM_PAINTICON:
		::OutputDebugString("WM_PAINTICON");
		wnd.data->ProcPAINT(msg, wp,lp);
		return 0;
	case WM_PRINT:
		::OutputDebugString("WM_PRINT");
		wnd.data->ProcPAINT(msg, wp,lp);
		return 0;
	case WM_PRINTCLIENT:
		::OutputDebugString("WM_PRINTCLIENT");
		wnd.data->ProcPAINT(msg, wp,lp);
		return 0;
//	case WM_GETICON:
*/		

	case WM_MOUSEMOVE:
	case WM_LBUTTONDOWN:
	case WM_LBUTTONUP:
	case WM_LBUTTONDBLCLK:
	case WM_RBUTTONDOWN:
	case WM_RBUTTONUP:
	case WM_RBUTTONDBLCLK:
	case WM_MBUTTONDOWN:
	case WM_MBUTTONUP:
	case WM_MBUTTONDBLCLK:
	case 0x20B: //WM_XBUTTONDOWN
	case 0x20C: //WM_XBUTTONUP    
	case 0x20D: //WM_XBUTTONDBLCLK
	case 0x02A1: //WM_MOUSEHOVER                   
	case 0x02A3: //WM_MOUSELEAVE
	{
		UIEvent ev(msg, wp, lp);
		wnd.data->WindowMouseEvent(ev);
		return 0;
    }
	case 0x20A: // WM_MOUSEWHEEL:  
	{
		UIEvent ev(msg, wp, lp);
		// wheel events come with the mouse in screen coordinates
		// we need to offset the stuff back to our client area
		Rect2D off(wnd.GetContentBoundsOnScreen());
		ev.mousePos.x -= off.x;
		ev.mousePos.y -= off.y;
		
		wnd.data->WindowMouseEvent(ev);
		return 0;
    }

	case WM_NCMOUSEMOVE:
	case WM_NCLBUTTONDOWN:
	case WM_NCLBUTTONUP:
	case WM_NCLBUTTONDBLCLK:
	case WM_NCRBUTTONDOWN:
	case WM_NCRBUTTONUP:
	case WM_NCRBUTTONDBLCLK:
	case WM_NCMBUTTONDOWN:
	case WM_NCMBUTTONUP:
	case WM_NCMBUTTONDBLCLK:
	case 0xAB: //WM_NCXBUTTONDOWN
	case 0xAC: //WM_NCXBUTTONUP    
	case 0xAD: //WM_NCXBUTTONDBLCLK
	case 0x02A0: //WM_NCMOUSEHOVER                   
	case 0x02A2: //WM_NCMOUSELEAVE
		break; // outside of client so ... the usual DefWindowProc

	case WM_HOTKEY: 
	case WM_KEYDOWN:
	case WM_SYSKEYDOWN:
	case WM_KEYUP:
	case WM_SYSKEYUP:
	case WM_SYSDEADCHAR: // ALT-char
	case WM_DEADCHAR:	 // first for 2 key chars
	{
		UIEvent ev(msg, wp, lp);
		if (wnd.data->WindowKeyEvent(ev))
		{
			wdata->winMgr.data->usedLastChar = true;
			return 0; // used
		}
		wdata->winMgr.data->usedLastChar = false;
		break; // let the system do something ?
    }
	case WM_CHAR:
	{
		if (wdata->winMgr.data->usedLastChar)
		{
			wdata->winMgr.data->usedLastChar = false;
			return 0;
		}
		if (wp < ' ') // control characters will not be sent as ascii
			return 0;
		UIEvent ev(msg, wp, lp);
		if (wnd.data->WindowKeyEvent(ev))
		{
			return 0; // used
		}
		break; // let the system do something ?
    }

	// SETFOCUS, KILLFOCUS, ACTIVATE are similar but not identical
	// one can lose focus but still be active if a child window is gaining focus
	// for now we only care about activate ...
    case WM_SETFOCUS:	 
		//UnitTest::Log("WM_SETFOCUS %08X", hwnd);
		//	HWND oldFocus = (HWND) wp;
		wnd.data->WindowFocused(true);
		break;
    case WM_KILLFOCUS:
		//HWND newFocus = (HWND) wp;
		//UnitTest::Log("WM_KILLFOCUS %08X", hwnd);
		wnd.data->WindowFocused(false);
		wnd.data->SetCaret(Rect2D()); // just in case the views did not do it
		break;
	//case WM_ACTIVATEAPP:
	case WM_ACTIVATE:
		wnd.data->WindowActivated(wp == WA_INACTIVE ? false : true);
		break;
	case WM_MOUSEACTIVATE:
		if (wnd.GetContent()->IsFocusable())
		{	//Activates the window, and discards the mouse message. 
	//		return MA_ACTIVATEANDEAT; 
			return MA_ACTIVATE; 
		}
		else
		{	//Does not activate the window, and does not discard the mouse message. 
			return MA_NOACTIVATE; 
		}
		break;
    	
	case WM_MOVE:
	case WM_SIZE:
		wnd.data->WindowBoundsChanged();
		break;

	case WM_CAPTURECHANGED:
		{
			HWND newCapturer = (HWND) lp;
			wnd.data->WindowCaptureMouseBreak();
			break;
		}

	case WM_GETMINMAXINFO: 
		wnd.data->ProcGETMINMAXINFO(msg, wp, lp);
		return 0;
	// as a DDE server we need to process these
	case WM_DDE_INITIATE:
	case WM_DDE_EXECUTE:
	case WM_DDE_TERMINATE:
		return 0;
/*	case WM_SYSCOMMAND:
		switch (wp)
		{
			case SC_SCREENSAVE:	  // Screensaver trying to start
			case SC_MONITORPOWER: // Monitor trying to powersave
				return 0;		  // Prevent From Happening
		}
		break;*/
	case WM_DESTROY:
	case WM_NCDESTROY:
		wnd.data->ProcDESTROY(msg, wp, lp);
		break;		
	};
	return ::DefWindowProc(hwnd, msg, wp, lp);
}

void WinMgr::CheckMouseStatus()
{
	// if it's captured it's not subject to check
	if (data->mouseCaptureView != 0) 
		return;
	// if nobody hovered no need to do anything
	if (data->oldHoveredView == 0)
		return;
	// do check if the hover is still in effect
	Window w = data->oldHoveredView->GetParentWindow();
	if (w == 0)
	{
		data->oldHoveredView->MouseEnter(false);
		data->oldHoveredView = 0;
		return;
	}
	UIEvent ev; // null event
	w.data->WindowMouseEvent(ev);
}


Window WinMgr::GetActiveWindow()
{
	HWND hwnd = ::GetActiveWindow();
	_Data::WindowPool& pool = data->windowPool;
	_Data::WindowPool::iterator w = pool.find(hwnd);
	if (w == pool.end()) // not our window ?!
		return Window();
	Window& wnd = w->second;
	ASSERT(wnd.data->hwnd == hwnd);
	return wnd;
}

Rect2D WinMgr::GetScreenBounds()
{
	return Rect2D(data->fullVirtualSize);
/*	
	HWND desk = ::GetDesktopWindow();
//	HWND desk = ::GetShellWindow(); // W2K only
	RECT drect;
	::GetWindowRect(desk, &drect);

	Rect2D sB;
	sB.ConvertFromRect(drect);
	return sB;     */
}

const ViewSkin::owner&  WinMgr::GetSysSkin()
{
	ASSERT(data!=0);
	if (data->sysSkin != 0)
		return data->sysSkin;
	ColorRGB body;   body.FromCOLORREF(::GetSysColor(COLOR_BTNFACE));
//	light.Blend(ColorRGB(0xFF,0xFF,0xFF,0x40), body);
//	 dark.Blend(ColorRGB(0x00,0x00,0x00,0x40), body);
	ColorRGB light; light.FromCOLORREF(::GetSysColor(COLOR_BTNHIGHLIGHT));
	ColorRGB  dark;  dark.FromCOLORREF(::GetSysColor(COLOR_BTNSHADOW));
	ColorRGB selet; selet.FromCOLORREF(::GetSysColor(COLOR_HIGHLIGHT));
	ColorRGB edit;   edit.FromCOLORREF(::GetSysColor(COLOR_WINDOW));

	data->sysSkin = new ViewSkin();
	
	refc<StateSkin> sbody(new SolidColorSkin(body));
	refc<StateSkin> spushed(new SolidButtonSkin(body,dark,light));
	refc<StateSkin> sraised(new SolidButtonSkin(body,light,dark));
	{ // button
		ViewSkin::owner skin(new ViewSkin());
		skin->AddStateSkin( ButtonView::State3RD|View::StateCapturing, 
							ButtonView::State3RD,
							spushed);
		skin->AddStateSkin( ButtonView::State3RD|View::StateCapturing, 
							ButtonView::State3RD|View::StateCapturing,
							sbody);
		skin->AddStateSkin(	View::StateCapturing,	
							View::StateCapturing,
							spushed);
//		skin->AddStateSkin( View::StateHovered,	 	
//							View::StateHovered,
//							new SolidColorSkin(body));
		// ToDo ... remove this ?
		skin->AddStateSkin(	View::StateFocused,	
							View::StateFocused,
							new SolidColorSkin(selet));
		skin->AddStateSkin(	0, 0, sraised);
		data->sysSkin->AddPart(String::From_c_str("Sys.Button"), skin);
	}
	{ // edit
		ViewSkin::owner skin(new ViewSkin());
		skin->AddStateSkin( View::StateDisabled, View::StateDisabled, sbody);
		skin->AddStateSkin(	0, 0, new SolidButtonSkin(edit,dark,light));
		data->sysSkin->AddPart(String::From_c_str("Sys.Edit"), skin);
	}
	{ // embed edit
		ViewSkin::owner skin(new ViewSkin());
		skin->AddStateSkin( View::StateDisabled, View::StateDisabled, sbody);
		skin->AddStateSkin(	0, 0, new SolidColorSkin(edit));
		data->sysSkin->AddPart(String::From_c_str("Sys.EmbedEdit"), skin);
	}
	{ // dialog
		ViewSkin::owner skin(new ViewSkin());
		skin->AddStateSkin(	0, 0,
							sbody);
		data->sysSkin->AddPart(String::From_c_str("Sys.Dialog"), skin);
	}
		
	return data->sysSkin;
}

#if 0
#pragma mark -
#endif


Window::_Data::_Data(const WinMgr& d)
: winMgr(d)
, hwnd(NULL)
, maxMode(MaximizeStandard)//, minmaxInfo(0)
, dropTarget(0)
{
}

Window::_Data::~_Data()
{
	// this should not be necessary , The window is owned by the winMgr so 
	// the only way to make it go away is to ask the system to DestroyWindow it
	// that will tell the winMgr to remove it from its list
	// However ... we need to make double sure so we don't have memory leaks in GDI
	if (hwnd != NULL)
		::DestroyWindow(hwnd);
	// just in case DestroyWindow does not do it's job 
	if (hwnd != NULL)
		Close(); 
}

void Window::_Data::Init(HWND parentHWND, DWORD af, DWORD ef)
{
	hwnd = ::CreateWindowEx(ef & ~WS_EX_ACCEPTFILES,  // extended flags
							(LPCSTR)(uint32) winMgr.data->wndClassAtom,
							NULL, // window title 
							af, // normal flags
							CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
							parentHWND,// parent wnd
							(HMENU) NULL, 		// menu / child ID
							winMgr.data->moduleInstance, // the application
							(LPSTR) 0 ); 		// creation data
	if( hwnd == NULL )
	{
		Exception(XSPMSG(0,"Failed to create new window"))
			.Raise();
	}
    ::SetWindowLong(hwnd, GWL_USERDATA, (uint32)this);
    VERIFY(::GetWindowLong(hwnd, GWL_USERDATA) == (uint32)this);
    
	if (ef & WS_EX_ACCEPTFILES)
	{   // register for DND
	    VERIFY(dropTarget == 0);
		dropTarget = new DropTargetWin32(this);
	    HRESULT hr = ::RegisterDragDrop(hwnd, dropTarget);
	    VERIFY( SUCCEEDED(hr) );
	}
}

void Window::_Data::Close()
{
	if (dropTarget != 0)
	{
		::RevokeDragDrop(hwnd); // unregister the DND interface
		dropTarget->Release(); // OLE release our reference
	}
	dropTarget = 0; // forget about this
	
	// since the winMgr owns a copy of this window
	// we can only get here after the window has been destroyed
	if (hwnd != NULL)
	{
		WinMgr::_Data::WindowPool& pool = winMgr.data->windowPool;
		pool.erase(hwnd);
		::SetWindowLong(hwnd, GWL_USERDATA, 0);
		hwnd = NULL;
	}
	
	BackgroundView::Close();
}

void Window::_Data::Refresh(const Rect2D& r)
{
	if (hwnd != NULL)
	{
		RECT rect;
		r.ConvertToRect(rect);
		::InvalidateRect(hwnd, &rect, FALSE);
	}
}

void Window::_Data::ProcPAINT(UINT , WPARAM wp, LPARAM )
{
	RECT ur;
	if (wp != 0)
	{
		Graphics g((HDC)wp);
		::GetClientRect(hwnd,&ur);
		WindowPaint(g, Rect2D(ur));
	}
	else if(0 != ::GetUpdateRect(hwnd,&ur,FALSE))
	{
		Graphics g(hwnd, (PAINTSTRUCT*)0);
		WindowPaint(g, Rect2D(ur));
	}
}
void Window::_Data::ProcGETMINMAXINFO(UINT msg, WPARAM wp, LPARAM lp)
{
	::DefWindowProc(hwnd, msg, wp,lp);
	MINMAXINFO* infod = (MINMAXINFO*) lp;
	
	// remember the full virtual screen size 
	Size2D& fvZ = winMgr.data->fullVirtualSize;
	// ToDo : I understand why 2*maxpos needs to be 
	// subtracted ... but why the hack 3*
	fvZ.w = infod->ptMaxTrackSize.x +3*infod->ptMaxPosition.x;
	fvZ.h = infod->ptMaxTrackSize.y +3*infod->ptMaxPosition.y;

	switch (maxMode)
	{
	case MaximizeStandard:
		// windows by default does move the border area out of 
		// the screen, so we need to do nothing extra
		break;
	case MaximizeFullMultiScreen:
		// ToDo ... windows has a problem with this, the resize works as expected
		// but later when WM_PAINT comes, only one paint with the DC set up for
		// only one of the monitors comes... I have to look into this deeper
		infod->ptMaxSize.x = infod->ptMaxTrackSize.x+infod->ptMaxPosition.x;
		infod->ptMaxSize.y = infod->ptMaxTrackSize.y+infod->ptMaxPosition.y;
		// now we can act exactly as if we were doing a fullscreen
	case MaximizeFullScreen:
		// in addition to maximize standard here we must move 
		// the entire non client area out of the screen

		// first undo the spec for standard maximize and put the border back in
		infod->ptMaxSize.x += 2*infod->ptMaxPosition.x;; 
		infod->ptMaxSize.y += 2*infod->ptMaxPosition.y;
		infod->ptMaxPosition.x = 0; 
		infod->ptMaxPosition.y = 0;
		// next compute the client edges
		RECT crect;
		{  
			RECT wrect;
			::GetWindowRect(hwnd, &wrect);
			::GetClientRect(hwnd, &crect);
			::ClientToScreen(hwnd, 0+(POINT*)&crect);
			crect.right = wrect.right - (crect.left+crect.right);
			crect.bottom = wrect.bottom - (crect.top+crect.bottom);
			crect.left = crect.left - wrect.left;			
			crect.top = crect.top - wrect.top;			
		}
		// now oversize the window so the nonclient area is out of this world :-)
		infod->ptMaxPosition.x = -crect.left; 
		infod->ptMaxPosition.y = -crect.top;
		infod->ptMaxSize.x += crect.left+crect.right; 
		infod->ptMaxSize.y += crect.top+crect.bottom;
		infod->ptMaxTrackSize.x = infod->ptMaxSize.x;
		infod->ptMaxTrackSize.y = infod->ptMaxSize.y;
		break;
	}
	
/*	MINMAXINFO* infos = minmaxInfo;
	if (infos != 0)
	{
		infod->ptMaxSize.x = infos->ptMaxSize.x;
		infod->ptMaxSize.y = infos->ptMaxSize.y;
		infod->ptMaxTrackSize.x = infos->ptMaxTrackSize.x;
		infod->ptMaxTrackSize.y = infos->ptMaxTrackSize.y;
		infod->ptMaxPosition.x = infos->ptMaxPosition.x;
		infod->ptMaxPosition.y = infos->ptMaxPosition.y;
	} */
}
void Window::_Data::ProcDESTROY(UINT , WPARAM , LPARAM )
{   
	::ShowWindow(hwnd, SW_HIDE);
	// to safely manipulate the pool deletion
	refc<_Data> tmp(this);
	WindowClosed(); // notify the window (this call could delete us)
	tmp->Close();
}

void Window::_Data::WindowPaint(Graphics& g, const Rect2D& area)
{
	if (area.isEmpty())
		return;
	OffscreenImage oi(area.w, area.h, OffscreenImage::RGBA32);
	{
		Graphics goi(oi);
		goi.SetOrigin(Point2D(-area.x, -area.y));
		DrawDeep(goi, area);
    }
	Rect2D aoi(area.GetSize());
	g.DrawImage(oi, aoi, area);
}
void Window::_Data::WindowClosed()
{
	Window w(this);
	for (WindowListener::each e(listeners); !e.empty(); e.pop()) 
		if (!e->IsCanceled())
			e->Closed(w);
	listeners.CancelAllListeners();
}

void Window::_Data::WindowBoundsChanged()
{
	Window w(this);
	
	SetBounds(Rect2D(w.GetContentSize()));
	for (WindowListener::each e(listeners); !e.empty(); e.pop()) 
		if (!e->IsCanceled())
			e->BoundsChanged(w);
}

void Window::_Data::WindowActivated(bool on)
{
	SetActivated(on); // do the view's equivalent
}
void Window::_Data::WindowFocused(bool on)
{
	Children c;
	if (on)
	{
		if (!IsFocused()) // focus
		{
			View::owner t = this;
			while(t != 0)
			{
				if (t->IsFocused())
					break; // this and it's children are all already set up
				t->StateSetReset(StateFocused, StateOldFocus);
//				t->viewState |= StateFocused;
//				t->viewState &= ~StateOldFocus;
				
				c.push_back(t);
				
				View::owner ct = t->GetChild(StateFocused|StateOldFocus);
				if (ct == 0)
					ct = t->GetChild(StateFocusable);
				t = ct;
			}
		}
	}
	else if (IsFocused()) // unfocus
	{
		for(View::owner cv = this; 
			cv != 0; 
			cv = cv->GetChild(StateFocused))
		{
			cv->StateSetReset(StateOldFocus, StateFocused);
//			cv->viewState &= ~StateFocused;
//			cv->viewState |= StateOldFocus;
			c.push_back(cv);
		}
	}

	Window w(this);

	for (Children::iterator v=c.begin(); v!=c.end(); ++v)
	{
		View::owner vi = *v;
		vi->HandleFocused(on);
	}

	for (WindowListener::each e(listeners); !e.empty(); e.pop()) 
		if (!e->IsCanceled())
			e->Activated(w, on);
}

void Window::_Data::WindowDropEvent(
							  DragDropType::TypeSet& availableTypes, 
							  DragDropType::Effect&  effect,
							  POINTL* 			     screenPoint,
							  bool 					 onlySimulate,
							  DragDropType::FatData& theFatData )
{
	View::owner hovered;

	// need to setup mouse pos relative to this window
	POINT sp; 
	sp.x = screenPoint->x;
	sp.y = screenPoint->y;
	::ScreenToClient(hwnd, &sp);
	Point2D mousePos(sp.x,sp.y);
	// check if the point is inside the client window
	RECT r;
	::GetClientRect(hwnd, &r);
	if ( (mousePos.x >= r.left) && (mousePos.x < r.right)
	   &&(mousePos.y >= r.top)  && (mousePos.y < r.bottom))
	{
		FindDeepChildAt(mousePos, hovered);
	}
    if (hovered == 0)
    { 
    	effect = 0;
    	return;
    }
    hovered->HandleDrop(availableTypes, effect, mousePos, onlySimulate, theFatData);	
}

void Window::_Data::WindowMouseEvent(UIEvent& ev)
{
	if (winMgr.data->mouseCaptureView != 0)
	{
		ev.mousePos += winMgr.data->mouseCaptureOriginDelta;
		winMgr.data->mouseCaptureView->HandleMouseEvent(ev);
		return;
	}	

	View::owner hovered;

	if (ev.origin == 0)
	{   // this is a synthethized event to make the hover change 
		// when views/windows are unobscured
		// need to setup mouse pos relative to this window
		POINT sp; 
		::GetCursorPos(&sp);
		::ScreenToClient(hwnd, &sp);
		ev.mousePos.x = sp.x;
		ev.mousePos.y = sp.y;
		// check if the synthetized event is inside the client window
		RECT r;
		::GetClientRect(hwnd, &r);
		if ( (ev.mousePos.x >= r.left) && (ev.mousePos.x < r.right)
		   &&(ev.mousePos.y >= r.top) && (ev.mousePos.y < r.bottom))
		{
			FindDeepChildAt(ev.mousePos, hovered);
		}
	}
	else // normal OS generated mouse event
		FindDeepChildAt(ev.mousePos, hovered);

	if (winMgr.data->oldHoveredView != hovered)
	{
		if (winMgr.data->oldHoveredView != 0)
			winMgr.data->oldHoveredView->MouseEnter(false);
	    winMgr.data->oldHoveredView = hovered;
		if (hovered != 0)
			hovered->MouseEnter(true);
	}
	if (ev.origin != 0)
	{
		// if the leaf view has no  use for the event then maybe 
		// one of the parents does
		while (hovered != 0)
		{
			if (hovered->HandleMouseEvent(ev))
				break;
			else
				hovered = hovered->GetParent();
		}
	}
}

void Window::_Data::StartMouseCapture(const owner& view, const Point2D& origin)
{
	VERIFY(view != 0);
	if (view != winMgr.data->mouseCaptureView)
	{
		VERIFY (winMgr.data->mouseCaptureView == 0);
		if (hwnd != ::GetCapture())
		{
			HWND oldCapturer = ::SetCapture(hwnd);
			ASSERT(oldCapturer == NULL);
			VERIFY(hwnd == ::GetCapture());
		}
		winMgr.data->mouseCaptureView = view;

		if (view == this)
			StateSetReset(StateCapturing, 0);
	}

	POINT sp;
	::GetCursorPos(&sp);
	::ScreenToClient(hwnd, &sp);
	winMgr.data->mouseCaptureOriginDelta.w = origin.x-sp.x;
	winMgr.data->mouseCaptureOriginDelta.h = origin.y-sp.y;
}
void Window::_Data::EndMouseCapture(const owner& view)
{
	if ((view == winMgr.data->mouseCaptureView) && 
		(winMgr.data->mouseCaptureView != 0))
	{
		if (hwnd == ::GetCapture())
			::ReleaseCapture();
		winMgr.data->mouseCaptureView = 0;
	}

	if (view == this)
		StateSetReset(0, StateCapturing);
//		viewState &=~ StateCapturing;
}
void Window::_Data::WindowCaptureMouseBreak()
{
	if (winMgr.data->mouseCaptureView != 0)
	{
		View::owner tmp = winMgr.data->mouseCaptureView;
		winMgr.data->mouseCaptureView= 0;
		tmp->MouseCaptureBroken();
	}	
}

void Window::_Data::SetCaret(const Rect2D& r)
{
	if ((caretRect.w != r.w) || (caretRect.h!=r.h))
	{   // caret size changed
		if (r.isEmpty())
		{
			::DestroyCaret();
//			UnitTest::Log("Caret destroyed");
		}
		else
		{
			VERIFY(0 != ::CreateCaret(hwnd, NULL, r.w, r.h)); 
			VERIFY(0 != ::SetCaretPos(r.x, r.y));
			VERIFY(0 != ::ShowCaret(hwnd));
//			UnitTest::Log("Caret created [%d,%d %dx%d]", r.x, r.y, r.w, r.h);
		}
		caretRect = r;
	}
	else if ((r.w > 0) && (r.h > 0))
	{   // caret size the same
		::SetCaretPos(caretRect.x=r.x, caretRect.y=r.y);
//		UnitTest::Log("Caret [%d,%d %dx%d]", r.x, r.y, r.w, r.h);
	}
}

													
bool Window::_Data::WindowKeyEvent(UIEvent& ev)
{
	if (ev.origin == UIEvent::KeyDown)
	{
		ev.origin = UIEvent::PriorityKey;
		if (DispatchKeyEvent(ev))
			return true;
		ev.origin = UIEvent::KeyDown;
		if (DispatchKeyEvent(ev))
			return true;
		ev.origin = UIEvent::UnusedKey;
		return DispatchKeyEvent(ev);
	}
	else // AsciiCode, KeyUp
		return DispatchKeyEvent(ev);
}

#if 0
#pragma mark -
#endif

static const DWORD af_main = WS_OVERLAPPED | WS_CLIPCHILDREN  | WS_CLIPSIBLINGS | WS_SYSMENU
				  		   | WS_MINIMIZEBOX | WS_MAXIMIZEBOX 
				  		   | WS_THICKFRAME; // resizable
static const DWORD ef_main = WS_EX_APPWINDOW // will show on the taskbar
						   | WS_EX_ACCEPTFILES; // we use OLE, this is not necessary but we use it to mark windows with DND support 

static const DWORD af_dlg  = WS_OVERLAPPED | WS_CLIPCHILDREN  | WS_CLIPSIBLINGS | WS_SYSMENU
						   | WS_DLGFRAME; // not resizable
static const DWORD ef_dlg  = 0; //WS_EX_APPWINDOW;

static const DWORD af_tool = WS_OVERLAPPED | WS_CLIPCHILDREN  | WS_CLIPSIBLINGS | WS_SYSMENU;
static const DWORD ef_tool = WS_EX_CLIENTEDGE // WS_EX_WINDOWEDGE // 
						   | WS_EX_TOOLWINDOW; // to prevent it from appearing on the taskbar
						   //| WS_EX_TOPMOST | WS_EX_TRANSPARENT; 
static const DWORD ef_drop = WS_EX_TOOLWINDOW; // to prevent it from appearing on the taskbar


Window::Window(const WinMgr& winMgr)
: data(new _Data(winMgr))
{
	data->Init(NULL, af_main, ef_main);
	winMgr.data->windowPool[data->hwnd] = *this;
	SetBgStyle(BackgroundView::Edit);
}

Window Window::PopupChild() const
{
	Window w;
	w.data = new _Data(data->winMgr);
	w.data->Init(data->hwnd, af_tool, ef_tool);
	w.data->winMgr.data->windowPool[w.data->hwnd] = w;
	w.SetBgStyle(BackgroundView::Dialog);
	return w;
}

Window Window::DropDownChild() const
{
	Window w;
	w.data = new _Data(data->winMgr);
	w.data->Init(data->hwnd, af_tool, ef_drop);
	w.data->winMgr.data->windowPool[w.data->hwnd] = w;
	w.SetBgStyle(BackgroundView::Edit);
	w.ShowFrame(false);
	return w;
}

Window Window::PopupDialog() const
{
	Window w;
	w.data = new _Data(data->winMgr);
	w.data->Init(data->hwnd, af_dlg, ef_dlg);
	w.data->winMgr.data->windowPool[w.data->hwnd] = w;
	w.SetBgStyle(BackgroundView::Dialog);
	return w;
}

View::owner Window::GetContent()
{
	return data.operator-> ();
}

WinMgr& Window::GetWinMgr()
{
	return data->winMgr;
}

void Window::Close()
{
	::SendMessage(data->hwnd, WM_CLOSE, 0,0);
}
bool Window::IsClosed() const
{
	return ((data == 0) || (data->hwnd == NULL));
}

void Window::ShowFrame(bool on)
{
	::SetLastError(ERROR_SUCCESS);
	LONG wndStyle = ::GetWindowLong( data->hwnd, GWL_STYLE );
	if (on)
		wndStyle |= WS_CAPTION | WS_THICKFRAME;
	else
		wndStyle &= ~WS_CAPTION & ~WS_THICKFRAME;
	VERIFY(0 != ::SetWindowLong( data->hwnd, GWL_STYLE, wndStyle ) );
    // flush looks 
	::SetWindowPos( data->hwnd
				  , HWND_NOTOPMOST, 0,0,0,0 
				  , SWP_NOMOVE | SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE | SWP_FRAMECHANGED);
}

void Window::Show(bool on)
{
	int how = SW_HIDE;
	if (on)
	{
		// if the window has no frame it must be a drop down 
		// child so we won't activate it
//		LONG wndStyle = ::GetWindowLong( data->hwnd, GWL_STYLE );
//		if (0 == (wndStyle & (WS_CAPTION | WS_THICKFRAME)))
//			how = SW_SHOWNOACTIVATE;
//		else
			how = SW_SHOW;
	}
	::ShowWindow(data->hwnd, how);
}

void Window::SetParent(const Window& p)
{
	HWND desk = ::GetDesktopWindow();
	if ((p.data == 0) || (p.data->hwnd == desk))
	{	// to desktop
		::SetParent(data->hwnd, NULL);
/*
		DWORD style = ::GetWindowLong(data->hwnd, GWL_STYLE);
		style &= ~ WS_CHILD;
		style |= ~ WS_POPUP;
		::SetWindowLong(data->hwnd, GWL_STYLE, style);
*/
		return;
	}
//	HWND pwnd = ::GetParent(data->hwnd);

//	VERIFY(NULL != ::SetParent(data->hwnd, p.data->hwnd));
	::SetParent(data->hwnd, p.data->hwnd);
}

void Window::SetMouseCursor(MouseCursor& curs)
{
	::SetCursor(curs.data->handle);
}

void Window::SetTitle(const String& title)
{
	::SetWindowText(data->hwnd, title.c_str());
}

String Window::GetTitle() const
{
	int len = ::GetWindowTextLength(data->hwnd);
	String s;
	if (len > 0)
	{
		s.resize(len);
		// we ask for len+1 since the system will need to copy the 
		// ASCIIZ char at the end of the string, we have room for it
		// because we also have a reserved space for such a char 
		::GetWindowText(data->hwnd, s.begin(), len+1);
	}
	return s;
}

void Window::SetBounds(const Rect2D& bounds)
{
	::MoveWindow(data->hwnd, bounds.x, bounds.y, bounds.w, bounds.h, TRUE);
}
void Window::SetContentBoundsOnScreen(const Rect2D& bounds)
{
	RECT wrect, crect;
	::GetWindowRect(data->hwnd, &wrect);
	::GetClientRect(data->hwnd, &crect);
	::ClientToScreen(data->hwnd, 0+(POINT*)&crect);
	crect.right +=crect.left;
	crect.bottom+=crect.top; 
	crect.right  =wrect.right-crect.right;    // right edge
	crect.bottom =wrect.bottom-crect.bottom; // bottom edge   
	crect.left  -= wrect.left;
	crect.top   -= wrect.top;

	Rect2D r(bounds);
	r.x -= crect.left;
	r.y -= crect.top;
	r.w += crect.left+crect.right;
	r.h += crect.top+crect.bottom;
	SetBounds(r);
}

Rect2D Window::GetBounds() const
{
	RECT rect;
	::GetWindowRect(data->hwnd, &rect);

	Rect2D r;
	r.ConvertFromRect(rect);
	return r;
}

Size2D Window::GetContentSize() const
{
	RECT rect;
	::GetClientRect(data->hwnd, &rect);
	return Size2D(rect.right-rect.left, rect.bottom-rect.top);
}

Rect2D Window::GetContentBounds() const
{
	RECT wrect, crect;
	::GetWindowRect(data->hwnd, &wrect);
	::GetClientRect(data->hwnd, &crect);
	::ClientToScreen(data->hwnd, 0+(POINT*)&crect);
	// crect relative to window frame
	crect.left-=wrect.left;
	crect.top-=wrect.top;    
	// now that crect is no longer (0,0) based, offset the SE corner too
	crect.right+=crect.left;
	crect.bottom+=crect.top; 

	Rect2D r;
	r.ConvertFromRect(crect);
	return r;
}
Rect2D Window::GetContentBoundsOnScreen() const
{
	RECT crect;
	::GetClientRect(data->hwnd, &crect);
	::ClientToScreen(data->hwnd, 0+(POINT*)&crect);
	// now that crect is no longer (0,0) based, offset the SE corner too
	crect.right+=crect.left;
	crect.bottom+=crect.top; 

	Rect2D r;
	r.ConvertFromRect(crect);
	return r;
}

void Window::BringToFront()
{
	::SetWindowPos(data->hwnd, HWND_TOP, 0,0,0,0, SWP_NOACTIVATE|SWP_NOMOVE|SWP_NOSIZE);
	::SetActiveWindow(data->hwnd);
}
bool Window::IsActive() const
{
	return data->hwnd == ::GetActiveWindow();
}
bool Window::IsVisible() const
{
	return 0 != ::IsWindowVisible(data->hwnd);
}

bool Window::IsMinimized() const
{
	WINDOWPLACEMENT wp;
	if (0 == ::GetWindowPlacement(data->hwnd, &wp))
		return true; // report true if failure
	return wp.showCmd == SW_SHOWMINIMIZED;
}

bool Window::IsMaximized() const
{
	WINDOWPLACEMENT wp;
	if (0 == ::GetWindowPlacement(data->hwnd, &wp))
		return false; // report false if failure
	return wp.showCmd == SW_SHOWMAXIMIZED;
}

void Window::SetMaximizeMode(MaximizeMode mode)
{
	data->maxMode = mode;
}

void Window::Maximize()
{
	::ShowWindow(data->hwnd, SW_MAXIMIZE);
}

void Window::Minimize()
{
	::ShowWindow(data->hwnd, SW_MINIMIZE);
}

void Window::RestoreSize()
{
	::ShowWindow(data->hwnd, SW_RESTORE);
}

/*
void Window::ShowFullScreen(bool on)
{
	if (on)
	{	
		// if we have a limit set for WM_GETMINMAXINFO then 
		// we will refuze going to full screen 
		if (data->minmaxInfo != 0)
			return;
			
		Rect2D full;
		{   
			HWND desk = ::GetDesktopWindow();
			RECT drect;
			::GetWindowRect(desk, &drect);
			// to avoid desktop toolbars
			// ::SystemParametersInfo(SPI_GETWORKAREA,0,&drect,0);
			
			// for full main monitor client area
			// ::GetSystemMetrics(SM_CXFULLSCREEN), SM_CXSCREEN
			// ::GetSystemMetrics(SM_CYFULLSCREEN), SM_CYSCREEN

			full.ConvertFromRect(drect);
			Rect2D cb = GetContentBounds();
			Rect2D wb = GetBounds();
			full.EnlargeBy(cb.x,cb.y, Rect2D::NW);
			full.EnlargeBy(wb.w-cb.GetEast(),wb.h-cb.GetSouth(), Rect2D::SE);
		}
		MINMAXINFO info;
		info.ptMaxTrackSize.x = info.ptMaxSize.x = full.w;
		info.ptMaxTrackSize.y = info.ptMaxSize.y = full.h;
		info.ptMaxPosition.x = full.x;
		info.ptMaxPosition.y = full.y;
		
		// during WM_GETMINMAXINFO we will return this expanded size
		// so that the frame around the window is outside the desktop
		data->minmaxInfo = &info; // start using this
		// with the change in place maximize will go to full screen
		::ShowWindow(data->hwnd, SW_MAXIMIZE);
		// now we set the bounds to the full screen
		//::SetWindowPos(data->hwnd, HWND_TOP, full.x,full.y,full.w,full.h, SWP_SHOWWINDOW);
		// cancel interference with	WM_GETMINMAXINFO
		data->minmaxInfo = 0;     // end of use
		
	}
	else
		::ShowWindow(data->hwnd, SW_RESTORE);
}
*/
void Window::Refresh()
{
	if (data->hwnd != NULL)
		::InvalidateRect(data->hwnd, NULL, FALSE);
}

void Window::Refresh(const Rect2D& r)
{
	data->Refresh(r);
}





} // namespace XSP
#endif
