/*
Copyright (C) 2003 Hotsprings Inc.
For conditions of distribution and use, see copyright notice

Location: 
	www.HotspringsInc.com 

History:
	2003Aug27-GiuseppeG: code write
*/

#include "XSP_Core.h"
#include "XSP_File.h"
#include "XSP_GUI.h"

namespace XSP
{
GUI_Module* GUI_Module::_instance = 0;

GUI_Module::GUI_Module()
{
	_instance = this;
}

GUI_Module::~GUI_Module()
{
	if (_instance == this)
		_instance = 0;

	codecFactory = 0;
}




Point2D Point2D::operator- () const
{
	return Point2D(-x, -y);
}	
Size2D Point2D::operator- (const Point2D& p) const
{
	return Size2D(x-p.x, y-p.y);
}	
Point2D Point2D::operator- (const Size2D& p) const
{
	return Point2D(x-p.w, y-p.h);
}	
Point2D Point2D::operator+ (const Size2D& p) const
{
	return Point2D(x+p.w, y+p.h);
}	
Point2D& Point2D::operator-= (const Size2D& p)
{
	x-=p.w;
	y-=p.h;
	return *this;
}	
Point2D& Point2D::operator+= (const Size2D& p)
{
	x+=p.w;
	y+=p.h;
	return *this;
}	
Point2D Point2D::operator+ (const Point2D& p) const
{
	return Point2D(x+p.x, y+p.y);
}	
Point2D& Point2D::operator+= (const Point2D& p)
{
	x+=p.x;
	y+=p.y;
	return *this;
}	

void Size2D::Normalize() 
{
	if (w<0) w = 0;
	if (h<0) h = 0;
}

Size2D  Size2D::operator- (const Size2D& p) const
{
	return Size2D(w-p.w, h-p.h);
}	
Size2D  Size2D::operator+ (const Size2D& p) const
{
	return Size2D(w+p.w, h+p.h);
}	
Size2D& Size2D::operator-= (const Size2D& p)
{
	w-=p.w;
	h-=p.h;
	return *this;
}	
Size2D& Size2D::operator+= (const Size2D& p)
{
	w+=p.w;
	h+=p.h;
	return *this;
}	


Rect2D::Rect2D(const Point2D& nw, const Point2D& se)
: x(nw.x), y(nw.y), w(se.x-nw.x), h(se.y-nw.y)
{
}

bool Rect2D::operator == (const Rect2D& p) const
{
	return (x==p.x) && (y==p.y) && (w==p.w) && (h==p.h);
}

bool Rect2D::operator != (const Rect2D& p) const
{
	return (x!=p.x) || (y!=p.y) || (w!=p.w) || (h!=p.h);
}

void Rect2D::Normalize() 
{
	if (w<0) w = 0;
	if (h<0) h = 0;
}

bool Rect2D::Contains(const Point2D& p) const
{
	return (x<=p.x) && (y<=p.y) && (x+w>p.x) && (y+h>p.y);
}

Point2D Rect2D::GetCenter() const 
{ 
	return Point2D(x+(w>>1), y+(h>>1)); 
}

bool Rect2D::Intersects(const Rect2D& r) const
{
	Point2D nw(std::max(x, r.x), std::max(y, r.y));
	Point2D se(std::min(x+w, r.x+r.w), std::min(y+h, r.y+r.h));
	return (nw.x<=se.x) && (nw.y<=se.y);
}  


// rectangle bounding both input rectangles
Rect2D Rect2D::Union(const Rect2D& r) const
{
	Point2D nw(std::min(x, r.x), std::min(y, r.y));
	Point2D se(std::max(x+w, r.x+r.w), std::max(y+h, r.y+r.h));
	return Rect2D(nw, se);  
}
Rect2D Rect2D::Intersect(const Rect2D& p) const
{
	Rect2D rez(*this);
	rez.ClipTo(p);
	return rez;  
}  
void Rect2D::ClipTo(const Rect2D& r)
{
	Point2D nw(std::max(x, r.x), std::max(y, r.y));
	Point2D se(std::min(x+w, r.x+r.w), std::min(y+h, r.y+r.h));
	w = std::max(0L, se.x-nw.x); 
	h = std::max(0L, se.y-nw.y);
	x = nw.x;
	y = nw.y;
} 


Rect2D Rect2D::Align(const Rect2D& r, Alignment algn, Alignment ovr) const
{
	Rect2D rez(*this);
	rez.x+=r.x;
	rez.y+=r.y;

	switch ((w < r.w) ? ovr : algn)
	{
	case C: case N: case S: // horizontally centered
		rez.x += (w-r.w) >> 1;
		break;
	case E: case NE: case SE: // horizontally aligned right
		rez.x += (w-r.w);
		break;
	}	
	switch ((h < r.h) ? ovr : algn)
	{
	case C: case E: case W: // vertically centered
		rez.y += (h-r.h) >> 1;
		break;
	case SW: case S: case SE: // vertically aligned bottom
		rez.y += (h-r.h);
		break;
	}	
	
	return rez;
}

void Rect2D::RemoveBorder(const Rect2D& border, Alignment a)
{
	sint32 t;
	switch (a)
	{
	case C: 
		if (w >= border.w) { x+=border.x; w-=border.w; }
		if (h >= border.h) { y+=border.y; h-=border.h; }
		break;

	case NW: 
		if (w >= border.w) 		w = border.x; 
		else if (w >= border.x) w = border.x;
		else 					{} // keep w

		if (h >= border.h) 		h = border.y; 
		else if (h >= border.y) h = border.y;
		else 					{} // keep h
		break;
	case NE: 
		t = border.w-border.x;
		x+=w-t;
		if (w >= border.w) 		w=t;
		else if (w >= border.x) w-=border.x;
		else 					w=0;

		if (h >= border.h) 		h = border.y; 
		else if (h >= border.y) h = border.y;
		else 					{} // keep h
		break;
	case SE:
		t = border.w-border.x;
		x+=w-t;
		if (w >= border.w) 		w=t;
		else if (w >= border.x) w-=border.x;
		else 					w=0;

		t = border.h-border.y;
		y+=h-t;
		if (h >= border.h) 		h=t;
		else if (h >= border.y) h-=border.y;
		else 					h=0;
		break;
	case SW:
		if (w >= border.w) 		w = border.x; 
		else if (w >= border.x) w = border.x;
		else 					{} // keep w

		t = border.h-border.y;
		y+=h-t;
		if (h >= border.h) 		h=t;
		else if (h >= border.y) h-=border.y;
		else 					h=0;
		break;

	case N: 
		if (w >= border.w) { x+=border.x; w-=border.w; }

		if (h >= border.h) 		h = border.y; 
		else if (h >= border.y) h = border.y;
		else 					{} // keep h
		break;
	case E: 
		t = border.w-border.x;
		x+=w-t;
		if (w >= border.w) 		w=t;
		else if (w >= border.x) w-=border.x;
		else 					w=0;

		if (h >= border.h) { y+=border.y; h-=border.h; }
		break;
	case S: 
		if (w >= border.w) { x+=border.x; w-=border.w; }

		t = border.h-border.y;
		y+=h-t;
		if (h >= border.h) 		h=t;
		else if (h >= border.y) h-=border.y;
		else 					h=0;
		break;
	case W: 
		if (w >= border.w) 		w = border.x; 
		else if (w >= border.x) w = border.x;
		else 					{} // keep w

		if (h >= border.h) { y+=border.y; h-=border.h; }
		break;
	}	
}

void Rect2D::OffsetBy(sint32 u, sint32 v, Alignment a)
{
	switch (a)
	{
	case C: case N: case S:
		x += u>>1;
		break;
	case E: case NE: case SE:
		w += u;
		break;
	case W: case NW: case SW: 
		x += u;
		w -= u;
		break;
	}	
	switch (a)
	{
	case C: case E: case W:
		y += v>>1;
		break;
	case S: case SE: case SW:
		h += v;
		break;
	case N: case NW: case NE: 
		y += v;
		h -= v;
		break;
	}	
}

void Rect2D::EnlargeBy(sint32 u, sint32 v, Alignment a)
{
	switch (a)
	{
	case C: case N: case S:
		x -= u >> 1;
		break;
	case W: case NW: case SW: 
		x -= u;
		break;
	}	
	w += u;

	switch (a)
	{
	case C: case E: case W:
		y -= v >> 1;
		break;
	case N: case NW: case NE: 
		y -= v;
		break;
	}	
	h += v;
}

Rect2D Rect2D::Scale(const Rect2D& factor, const Rect2D& divider) const
{
	VERIFY(! divider.isEmpty());

	Rect2D rez(*this);
	// align origins
	rez.x -= divider.x;
	rez.y -= divider.y;
	// !!! the function for all 4 corners must have the same looks
	if (factor.w != divider.w)
	{
		sint64 fx = sint64(factor.w);
		sint32 r=x+w;
		rez.x = sint32( (x * fx) / divider.w );
	    	r = sint32( (r * fx) / divider.w );
		rez.w = r-rez.x;
    }
	if (factor.h != divider.h)
	{
		sint64 fy = sint64(factor.h);
		sint32 b=y+h;
		rez.y = sint32( (y * fy) / divider.h );
			b = sint32( (b * fy) / divider.h );
	    rez.h = b-rez.y;
	}
	// offset origins back according to the factor
	rez.x += factor.x;
	rez.y += factor.y;
	return rez;
}


void ColorRGB::Blend(const ColorRGB& a, const ColorRGB& b)
{
	uint32 alf = a.Alpha();
	if (alf == 0xffU) 
		rgba = a.rgba;
	else if (alf == 0U) 
		rgba = b.rgba;
	else
	{
		uint64 n = 
			( 
				(0x100U-alf) *
	        	(  (((uint64) (b.rgba & 0x00ff00ff)) << 32) 
	        	 + (b.rgba & 0xff00ff00))
			) >> 8;
		rgba = a.rgba + ( 1[(uint32*)&n] | 0[(uint32*)&n] );
	}
}

void LockOffscreenImage::Fill(ColorRGB clr, Rect2D dst)
{
	// ToDo more bpp
	ASSERT(bpp == 32);
	uint8* row = memory;
	row += rowBytes*dst.y;
	for(sint32 Y = dst.y+dst.h; dst.y<Y; ++dst.y)
	{
		uint32* pix = ((uint32*)row) + dst.x;
		uint32* pex = pix + dst.w;
		while( pix<pex )
			*pix++ = clr.rgba;
		row += rowBytes;
	}
}

OffscreenImage::OffscreenImage()
{}

OffscreenImage::~OffscreenImage()
{}

OffscreenImage::OffscreenImage(const OffscreenImage& i) 
: data(i.data) 
{}

OffscreenImage& OffscreenImage::operator = (const OffscreenImage& i) 
{ 
	data = i.data; 
	return *this;
}
OffscreenImage& OffscreenImage::operator = (_Data* i)
{
	data = i;
	return *this;
}

bool OffscreenImage::operator == (const OffscreenImage& i) const 
{ 
	return data == i.data; 
}

bool OffscreenImage::operator != (const OffscreenImage& i) const 
{ 
	return data != i.data; 
}

bool OffscreenImage::operator == (const _Data* i) const 
{ 
	return data == i; 
}

bool OffscreenImage::operator != (const _Data* i) const 
{ 
	return data != i; 
}


/*
void LockOffscreenImage::_DrawImage(const LockOffscreenImage& dstI
								   ,const LockOffscreenImage& srcI
								   ,const Rect2D& dstR
								   ,const Rect2D& srcR)
{
	Rect2D dst_r = dstR.Intersect(dstI.bounds);
	Rect2D src_r = srcR.Intersect(srcI.bounds);

    ImageBlenders::LineBlender lineBlender = 0;
    switch((dstI.bpp<<8)|srcI.bpp)
    {
    case 0x2020: lineBlender = ImageBlenders::LineBlender_32_32; break;
    case 0x2010: lineBlender = ImageBlenders::LineBlender_32_16; break;
    case 0x1020: lineBlender = ImageBlenders::LineBlender_16_32; break;
    case 0x1010: lineBlender = ImageBlenders::LineBlender_16_16; break;
    };

	uint8* dst_L = dstI.memory + dstI.rowBytes*dst_r.y + (dstI.bpp>>3)*dst_r.x;
	uint8* src_L = srcI.memory + srcI.rowBytes*src_r.y + (srcI.bpp>>3)*src_r.x;
	
	sint32 rows = std::min(dst_r.h, src_r.h);
	sint32 cols = std::min(dst_r.w, src_r.w);
	
	for(sint32 y=0; y<rows; ++y)
	{
		lineBlender(dst_L, dst_cx, dst_kx, dst_b
					src_L, src_cx, src_kx, src_b
					,cols);
		
		uint8* dst_p = dst_L;
		uint8* src_p = src_L;
		for(sint32 x=cols; x>0; --x)
		{
			pixelBlender(dst_p, src_p);
			if (--dst_cx == 0)
			{
				dst_p += dst_b;
				dst_cx = dst_kx;
			}
			if (--src_cx == 0)
			{
				src_p += src_b;
				src_cx = src_kx;
			}
		}
		if (--dst_cy == 0)
		{
			dst_L += dstI.rowBytes;
			dst_cy = dst_ky;
		}
		if (--src_cy == 0)
		{
			src_L += srcI.rowBytes;
			src_cy = src_ky;
		}
	}
}
*/

UIEvent::UIEvent()
: origin(0)
, modifiers(0)
, code(0)
, count(1)
{}

UIEvent::UIEvent(const UIEvent& ev)
: origin(ev.origin), code(ev.code)
, modifiers(ev.modifiers), count(ev.count)
, mousePos(ev.mousePos)
{}

UIEvent& UIEvent::operator= (const UIEvent& ev)
{
  origin = ev.origin;
  code = ev.code;
  modifiers = ev.modifiers;
  count = ev.count;
  mousePos = ev.mousePos;
  return *this;
}

} // namespace XSP
