/*
Copyright (C) 2003 Hotsprings Inc.
For conditions of distribution and use, see copyright notice

Location: 
	www.HotspringsInc.com 

History:
	2003Apr14-GiuseppeG: code write
*/

#include "XSP_Core.h"
#include "XSP_File.h"

namespace XSP
{
FileModule* FileModule::_instance = 0;

FileModule::FileModule()
{
	// singleton: only allow one of these to be instantiated
	VERIFY(_instance == 0);
	if (_instance == 0)
		_instance = this;
}

FileModule::~FileModule()
{
	_instance = 0;
}

#if 0
#pragma mark -
#endif

class FileName::_Data : public ref_obj
{
public:
	String filename;
	NameType nameType; 
	FileName parent;

	_Data();
	virtual ~_Data();
}; // class FileName::_Data


FileName::_Data::_Data()
: nameType(FileName::Relative)
{
}

FileName::_Data::~_Data()
{
}

// build with ref==0
FileName::FileName()
{
}

FileName::~FileName()
{
}

FileName::FileName(const FileName& fn)
: ref(fn.ref)
{
}

FileName& FileName::operator =(const FileName& fn)
{
	ref = fn.ref;
	return *this;
}

FileName& FileName::operator=(_Data* fn) 
{ 
	ref = fn; 
	return *this; 
}
bool FileName::operator == (_Data* fn) const 
{ 
	return ref == fn; 
}
bool FileName::operator != (_Data* fn) const 
{ 
	return ref != fn; 
}



bool FileName::operator == (const FileName& fn) const
{
	if (ref == fn.ref)
		return true;
	if (ref == 0 || fn.ref == 0)
		return false;	
	if (ref->filename != fn.ref->filename)
		return false;
	if (ref->nameType != fn.ref->nameType)
		return false;

	return ref->parent == fn.ref->parent; 
}

int FileName::cmp(const FileName& fn) const
{
	if (ref == fn.ref)
		return 0;
	if (ref == 0)
		return 1;
	if (fn.ref == 0)
		return -1;
	int c = ref->parent.cmp(fn.ref->parent);
	if (c != 0)
		return c;
	if (ref->nameType != fn.ref->nameType)
		return ((ref->nameType < fn.ref->nameType) ? 1 : -1);
	return ref->filename.cmp(fn.ref->filename);
}

const String& FileName::GetName() const 
{
	VERIFY(ref != 0); 
	return ref->filename; 
}


bool FileName::SplitNameExtension(String& baseName, String& extension) const
{	
	VERIFY(ref != 0); 
    NameType t = ref->nameType;
    VERIFY (t == Relative || t == Root);
    
	String::const_iterator b(ref->filename.begin()); 
	String::const_iterator e(ref->filename.end());
	if (b != e)
	{
		for(String::const_iterator p(e - 1); ; --p)
		{
			if (*p == '.')
			{
				baseName = String(b, p);
				extension = String(p+1, e);
				return true;
			}
			if (p == b)
				break;
		} 
	}

	baseName = ref->filename;
	extension.clear();
	return false;
}

FileName FileName::From(const FileName& path, const String& fname)
{
    FileName fn;
    fn.ref = new _Data();
    fn.ref->nameType = Relative;
    fn.ref->filename.assign(fname);
	fn.ref->parent = path;
	return fn;
}
FileName FileName::From(const FileName& path, const String& basename, const String& extension)
{
	return From(path, basename+String::From_c_str(".")+extension);
}

bool FileName::HasParent() const 
{ 
	VERIFY(ref != 0); 
	if (ref->parent.ref == 0)
		return false;
	switch(ref->nameType)
	{
	case Root:
		return false;
	}
	switch(ref->parent.ref->nameType)
	{
	case Drive:
	case Machine:
		return false;
	}
	return true; 
}

const FileName& FileName::GetParent() const
{
	VERIFY(ref != 0 && ref->parent.ref != 0);
	return ref->parent;
}

void FileName::ToPath(FileName::Path& p) const
{
	p.clear();
	//p.reserve(256);	
	for( FileName r(*this); r.ref != 0; r = r.ref->parent)
		p.push_front(r);
}


String FileName::GetFullPathNameOS(bool dir) const
{
	#if TARGET_API_MAC_OS8 || TARGET_API_MAC_Carbon 
		return GetFullPathNameMac(dir);
	#elif TARGET_API_Win32 || TARGET_API_Win32_Console 
		return GetFullPathNameWin32(dir);
	#elif TARGET_API_Darwin || TARGET_API_MAC_Mach0 || TARGET_API_Linux
		return GetFullPathNameLinux(dir);
	#endif
}

FileName FileName::FromPathNameOS(const String& fullPathNameOS)
{
	#if TARGET_API_MAC_OS8 || TARGET_API_MAC_Carbon 
		return FromPathNameMac(fullPathNameOS);
	#elif TARGET_API_Win32 || TARGET_API_Win32_Console 
		return FromPathNameWin32(fullPathNameOS);
	#elif TARGET_API_Darwin || TARGET_API_MAC_Mach0 || TARGET_API_Linux
		return FromPathNameLinux(fullPathNameOS);
	#endif
}


static const char _machinePrefix[] = "\\\\";
static const char _drivePostfix[] = ":";
static const char _pathSepWin32[] = "\\";
static const char _pathSepLinux[] = "/";
static const char _pathSepMac[] = ":";

String FileName::GetFullPathNameWin32(bool dir) const
{
	VERIFY(ref != 0); 

	String s;
	s.reserve(1024);
		
	Path path;
	ToPath(path);

	Path::const_iterator b(path.begin());
	Path::const_iterator e(path.end());
	VERIFY (b != e);
	{	 
		const String& rt = (* b).GetName();
		switch((*b).ref->nameType)
		{
		case Drive: // is something like "C:..."
			VERIFY(rt.length() == 1);
			s.append(rt);
			s.append(_drivePostfix, _drivePostfix+sizeof(_drivePostfix)-1);
			++b;	
			break;
		case Machine: // must be UNC name "\\machine\path"
			VERIFY(rt.length() > 0);
			s.append(_machinePrefix, _machinePrefix+sizeof(_machinePrefix)-1);
			s.append(rt);
			++b;	
			break;
		}
	}
	if (b != e)
	{
		const String& rt = (*b).GetName();
		switch((*b).ref->nameType)
		{
		case Relative: // must be a relative path like "name\name"
			VERIFY(rt.length() > 0);
			s.append(rt);
			++b;	
			break;
		}
    }
    bool endAtSlash = false;
	for(; b != e; ++b)
	{
		VERIFY((*b).ref->nameType == Relative || (*b).ref->nameType == Root);
		if (!endAtSlash)
			s.append(_pathSepWin32, _pathSepWin32+sizeof(_pathSepWin32)-1);
		
		const String& rt = (*b).GetName();
		s.append(rt);
		
		endAtSlash = (rt.length() == 0);
	}
	if (dir && !endAtSlash && !s.isEmpty())
		s.append(_pathSepWin32, _pathSepWin32+sizeof(_pathSepWin32)-1);
	s.compact();
	return s;
}

FileName FileName::FromPathNameWin32(const String& s)
{
	FileName rfn;
	
	String::const_iterator b(s.begin());
	String::const_iterator e(s.end());
	
	if ((e-b >= 1+sizeof(_drivePostfix)-1) && 
		std::equal(_drivePostfix, _drivePostfix+sizeof(_drivePostfix)-1, b+1))
    {   // is something like "C:\path"
        rfn = FileName::From(rfn, String(b, b+1));
    	rfn.ref->nameType = Drive;
    	b = b + 1 + sizeof(_drivePostfix)-1;
    }
    else if ((e-b >= sizeof(_machinePrefix)-1) && 
    		std::equal(_machinePrefix, _machinePrefix+sizeof(_machinePrefix)-1, b))
    {
    	b += sizeof(_machinePrefix)-1;
    	String::const_iterator sep = std::search(b, e,
    			    _pathSepWin32, _pathSepWin32+sizeof(_pathSepWin32)-1);
    	VERIFY(b != sep);		     
        rfn = FileName::From(rfn, String(b, sep));
    	rfn.ref->nameType = Machine;
    	
    	if (sep != e)  b = sep+1;
    	else           b = e;
    }
    
    if ((e-b >= sizeof(_pathSepWin32)-1) && 
    		std::equal(_pathSepWin32, _pathSepWin32+sizeof(_pathSepWin32)-1, b))
    {
    	b += sizeof(_pathSepWin32)-1;
        rfn = FileName::From(rfn, String::kEmpty);
    	rfn.ref->nameType = Root;
    }
    
    while(b != e)
    {
    	String::const_iterator sep = std::search(b, e,
    			    _pathSepWin32, _pathSepWin32+sizeof(_pathSepWin32)-1);
    	if(b != sep) // skip the empty folders	     
        	rfn = FileName::From(rfn, String(b, sep));
    	if (sep != e)  b = sep+1;
    	else           b = e;
    }
   
    return rfn;
}

String FileName::GetFullPathNameLinux(bool dir) const
{
	VERIFY(ref != 0); 

  	String s;
	s.reserve(1024);
		
	Path path;
	ToPath(path);

	Path::const_iterator b(path.begin());
	Path::const_iterator e(path.end());
	VERIFY (b != e);
	for(; b != e; ++b)
	{
		NameType t = (*b).ref->nameType;
		VERIFY(t == Relative || t == Root);
		if ((t == Root) || !s.isEmpty())
			s.append(_pathSepLinux, _pathSepLinux+sizeof(_pathSepLinux)-1);
		s.append((*b).GetName());
	}
	if (dir && !s.isEmpty())
		s.append(_pathSepLinux, _pathSepLinux+sizeof(_pathSepLinux)-1);
	s.compact();
	return s;
}

FileName FileName::FromPathNameLinux(const String& s)
{
	FileName rfn;
	
	String::const_iterator b(s.begin());
	String::const_iterator e(s.end());
	
    if ((e-b >= sizeof(_pathSepLinux)-1) && 
    	std::equal(_pathSepLinux, _pathSepLinux+sizeof(_pathSepLinux)-1, b))
    {
    	b += sizeof(_pathSepLinux)-1;
    	String::const_iterator sep = std::search(b, e,
    			    _pathSepLinux, _pathSepLinux+sizeof(_pathSepLinux)-1);
        rfn = FileName::From(rfn, String(b, sep));
    	rfn.ref->nameType = Root;
    	
    	if (sep != e)  b = sep+1;
    	else           b = e;
    }
    
    while(b != e)
    {
    	String::const_iterator sep = std::search(b, e,
    			    _pathSepLinux, _pathSepLinux+sizeof(_pathSepLinux)-1);
    	VERIFY(b != sep);		     
        rfn = FileName::From(rfn, String(b, sep));
    	if (sep != e)  b = sep+1;
    	else           b = e;
    }
   
    return rfn;
}

String FileName::GetFullPathNameMac(bool dir) const
{
	VERIFY(ref != 0); 
	String s;
//	if (ref == 0)
//		return s;
	s.reserve(1024);
		
	Path path;
	ToPath(path);

	Path::const_iterator b(path.begin());
	Path::const_iterator e(path.end());
	VERIFY ((b != e) && ((*b).ref->nameType == Root));
	for(; b != e; ++b)
	{
		NameType t = (*b).ref->nameType;
		VERIFY(t == Relative || t == Root);
		if (!s.isEmpty())
			s.append(_pathSepMac, _pathSepMac+sizeof(_pathSepMac)-1);
		s.append((*b).GetName());
	}
	if (dir && !s.isEmpty())
		s.append(_pathSepMac, _pathSepMac+sizeof(_pathSepMac)-1);
	
	s.compact();
	return s;
}

FileName FileName::FromPathNameMac(const String& s)
{
	FileName rfn;
	
	String::const_iterator b(s.begin());
	String::const_iterator e(s.end());
	
 	VERIFY(b != e);		     
    {
    	String::const_iterator sep = std::search(b, e,
    			    _pathSepMac, _pathSepMac+sizeof(_pathSepMac)-1);
        rfn = FileName::From(rfn, String(b, sep));
    	rfn.ref->nameType = Root;
    	
    	if (sep != e)  b = sep+1;
    	else           b = e;
    }
    
    while(b != e)
    {
    	String::const_iterator sep = std::search(b, e,
    			    _pathSepMac, _pathSepMac+sizeof(_pathSepMac)-1);
    	VERIFY(b != sep);		     
        rfn = FileName::From(rfn, String(b, sep));
    	if (sep != e)  b = sep+1;
    	else           b = e;
    }
   
    return rfn;
}



#if 0
#pragma mark -
#endif

FileList::FileList()
: ref(new _Data())
{
}

FileList::~FileList()
{
}

FileList::FileList(const FileList& fn)
: ref(fn.ref)
{
}

FileList& FileList::operator=(const FileList& fn)
{
	ref = fn.ref;
	return *this;
}
			  
const FileList::ContentType& FileList::GetContent() const
{
	return ref->items;
}

FileList::iterator FileList::begin() const
{
	return ref->items.begin();
}

FileList::iterator FileList::end() const
{
	return ref->items.end();
}

FileList::size_type FileList::size() const
{
	return ref->items.size();
}

void FileList::Add(const FileName& fname)
{
	FileProperties props(fname);
	Add(props);
}

void FileList::Add(const FileProperties& props)
{
	if (ref == 0)
		ref = new _Data();
    ref->items.push_back(props);
}


#if 0
#pragma mark -
#endif

FileForkList::FileForkList()
: ref(new _Data())
{
}
FileForkList::~FileForkList()
{
}
FileForkList::FileForkList(const FileForkList& fn)
: ref(fn.ref)
{
}
FileForkList& FileForkList::operator=(const FileForkList& fn)
{
	ref = fn.ref;
	return *this;
}

const FileName& FileForkList::GetFileName() const
{
	return ref->filename;
}
const FileForkList::ContentType& FileForkList::GetContent() const
{
	return ref->items;
}

FileForkList::iterator FileForkList::begin() const
{
	return ref->items.begin();
}
FileForkList::iterator FileForkList::end() const
{
	return ref->items.end();
}
FileForkList::size_type FileForkList::size() const
{
	return ref->items.size();
}


#if 0
#pragma mark -
#endif

const FileProperties::PropertyID FileProperties::CreationTime;
const FileProperties::PropertyID FileProperties::ModifyTime;
const FileProperties::PropertyID FileProperties::AccessTime;
const FileProperties::PropertyID FileProperties::BackupTime;
const FileProperties::PropertyID FileProperties::Size;
const FileProperties::PropertyID FileProperties::MacType;
const FileProperties::PropertyID FileProperties::MacCreator;
const FileProperties::PropertyID FileProperties::IsHidden;
const FileProperties::PropertyID FileProperties::IsDir;
const FileProperties::PropertyID FileProperties::IsLink;
const FileProperties::PropertyID FileProperties::IsReadOnly;
const FileProperties::PropertyID FileProperties::IconLocation;
const FileProperties::PropertyID FileProperties::ResourceForkSize;
const FileProperties::PropertyID FileProperties::UID;
const FileProperties::PropertyID FileProperties::GID;
const FileProperties::PropertyID FileProperties::Mode;
const FileProperties::PropertyID FileProperties::ShortName;


FileProperties::_Data::_Data()
: refcount(0)
{
}
FileProperties::_Data::~_Data()
{
	ASSERT(refcount == 0);
}
void FileProperties::_Data::addref()
{
	++refcount;
}
void FileProperties::_Data::release()
{
	if (0==--refcount)
		delete this;
}

FileProperties::FileProperties()
: ref(new _Data())
{}

FileProperties::~FileProperties()
{}

FileProperties::FileProperties(const FileProperties& fn)
: ref(fn.ref)
{}

FileProperties& FileProperties::operator=(const FileProperties& fn)
{
	ref = fn.ref;
	return *this;
}

const FileName& FileProperties::GetFileName() const
{
	return ref->filename;
}

void FileProperties::CopyOnWrite()
{
	if (ref->refcount != 1)
	{ // copy on write
		refc<_Data> oref(ref);
		ref = new _Data();
		ref->filename = oref->filename;
		ref->properties = oref->properties;
		ASSERT(ref->refcount == 1);
	}
}

void FileProperties::SetFileName(const FileName& fname)
{
	CopyOnWrite();
	ref->filename = fname;
}

void FileProperties::SetProperty(const PropertyID& pid, const String& prop)
{
	CopyOnWrite();
	ref->properties.insert(Properties::value_type(&pid,prop));
}

bool FileProperties::HasProperty(const PropertyID& pid) const
{
    return ref->properties.find(&pid) != ref->properties.end();
}

const String& FileProperties::operator[] (const PropertyID& pid) const
{
    Properties::const_iterator w = ref->properties.find(&pid);
    if (w == ref->properties.end())
    	return String::kEmpty;
    return w->second;
} 

#if 0
#pragma mark -
#endif

bool FileFilterPredicate::Match(const char* pattern, const char* fname)
{
	for(;;)
	{
		switch (*pattern)
		{
		case '*':
			while (* ++pattern == '*')
			{} // skip the *

			if (*pattern == 0)
				return true;

			for(;;)
			{
				if (*fname == 0)
					return false;
				if (Match(pattern, fname))
					return true;
				++fname;
			}	
			break;

		case '?':
			if (*fname == 0)
				return false;
			++fname;
			++pattern;
			break;

		case 0:
			return (*fname == 0);
			
		default:
			if (*fname != *pattern)
				return false;
			++fname;
			++pattern;
			break;
		}
	}
}

#if 0
#pragma mark -
#endif

const SpecialFileNames::FileID SpecialFileNames::SystemDir;	
const SpecialFileNames::FileID SpecialFileNames::MachineRoot;
const SpecialFileNames::FileID SpecialFileNames::UserHomeDir;
const SpecialFileNames::FileID SpecialFileNames::UserDesktopDir;
const SpecialFileNames::FileID SpecialFileNames::TempDir;
const SpecialFileNames::FileID SpecialFileNames::ProcessWorkingDir;
const SpecialFileNames::FileID SpecialFileNames::AppName;
const SpecialFileNames::FileID SpecialFileNames::AppDir;
const SpecialFileNames::FileID SpecialFileNames::UserPreferencesDir;
const SpecialFileNames::FileID SpecialFileNames::MachinePreferencesDir;
const SpecialFileNames::FileID SpecialFileNames::RecycleBinDir;
const SpecialFileNames::FileID SpecialFileNames::StartupDir;
const SpecialFileNames::FileID SpecialFileNames::ShutdownDir;
const SpecialFileNames::FileID SpecialFileNames::MenuDir;

SpecialFileNames::~SpecialFileNames()
{
	ref->specials.clear();
}

void SpecialFileNames::SetFileName(const FileID& pid, const FileName& prop)
{
	ref->specials.insert(Specials::value_type(&pid,prop));
}

bool SpecialFileNames::HasFileName(const FileID& pid) const
{
    return ref->specials.find(&pid) != ref->specials.end();
}

const FileName& SpecialFileNames::operator[] (const FileID& pid) const
{
    Specials::const_iterator w = ref->specials.find(&pid);
    VERIFY(w != ref->specials.end());
    return w->second;
}

#if 0
#pragma mark -
#endif

const OpenedFile::AccessType OpenedFile::ReadOnlyShared;
const OpenedFile::AccessType OpenedFile::ReadWriteExclusive;
const OpenedFile::AccessType OpenedFile::ReadWriteShared;

bool OpenedFile::operator == (const OpenedFile& of) const
{
	return ref == of.ref;
}

bool OpenedFile::operator != (const OpenedFile& of) const
{
	return ref != of.ref;
}

} // namespace XSP
