/*
Copyright (C) 2003 Hotsprings Inc.
For conditions of distribution and use, see copyright notice

Location: 
	www.HotspringsInc.com 

History:
	2003Apr10-GiuseppeG: code write
*/

#include "XSP_Core.h"

namespace XSP
{

LocalTime::TData::TData()
: refcount(0)
, year(0)
, second(0)
, nanosec(0)
{
}

void LocalTime::TData::release()
{
	if (0== --refcount) 
		delete this;
}



static const uint32 _NanosecsInSecond = 1000000000U;
static const uint32 _SecondsInMinute = 60U;
static const uint32 _SecondsInHour   = _SecondsInMinute * 60U;
static const uint32 _SecondsInDay    = _SecondsInHour * 24U;
static const uint32 _DaysInNormalYear = 365U;
static const uint32 _DaysInLeapYear = _DaysInNormalYear + 1U;
static const uint32 _SecondsInNormalYear = _SecondsInDay * _DaysInNormalYear;
static const uint32 _SecondsInLeapYear = _SecondsInNormalYear + _SecondsInDay;
static const uint32 _DaysInMonth[12] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
static const uint32 _DaysToMonth[12] = {  0, 31, 59, 90,120,151,181,212,243,273,304,334 };
static sint32 _daysTo(sint32 year)
{
	sint32 days = year * (sint32)_DaysInNormalYear;
	--year;
	days += (year>=0) ? 1:0;
	days += year/4 -year/100 + year/400;
	return days;
}
static const sint32 _DaysTo1904 = _daysTo(1904);
static const sint32 _DaysTo1970 = _daysTo(1970);
static const sint32 _DaysTo1601 = _daysTo(1601);


LocalTime::LocalTime()
{
} 

LocalTime::LocalTime(const LocalTime& t)
: tdata(t.tdata)
{
}
 
LocalTime& LocalTime::operator= (const LocalTime& t)
{
	tdata = t.tdata;
	return *this;
}

LocalTime::~LocalTime()
{
}
 
LocalTime::LocalTime( sint32 yr, uint32 month, uint32 dayofmonth
 		  , uint32 hr, uint32 mn, uint32 sec
 		  , uint32 nano)
: tdata(new TData())
{
	VERIFY(month>=1 && month<=12 && dayofmonth>=1 && dayofmonth<=31);
//	tdata->year = static_cast<sint16>(yr);
//	VERIFY(tdata->year == yr);
	tdata->year = yr;
	
	uint32 days= _DaysToMonth[month-1] + dayofmonth-1;
	if (month>=3 && IsLeapYear())
		++days;
	tdata->second = days * _SecondsInDay 
		   		  + hr * _SecondsInHour
		   		  + mn * _SecondsInMinute
		   		  + sec;
	tdata->nanosec = nano;
}				 	  
 
sint32 LocalTime::Year() const
{
	if (tdata==0)
		return 0;
	return tdata->year;
} 
uint32 LocalTime::DayOfYear() const
{
	if (tdata==0)
		return 0;
	return tdata->second / _SecondsInDay;
}
uint32 LocalTime::Month() const
{
	uint32 day = DayOfYear();
	if ((day > 31+28) && IsLeapYear())
		--day;
	uint32 month = 0;
	while (day >= _DaysInMonth[month])
		day -= _DaysInMonth[month++];
	return month+1; // January is month number 1 not 0	
	
} 
uint32 LocalTime::DayOfMonth() const
{
	uint32 day = DayOfYear();
	if ((day > 31+28) && IsLeapYear())
		--day;
	uint32 month = 0;
	while (day >= _DaysInMonth[month])
		day -= _DaysInMonth[month++];
	return day+1;  // first day of month is 1
} 
uint32 LocalTime::Week() const
{
	uint32 day = DayOfYear();
	// is this correct ... when does a week start?
	return day/7;
} 
uint32 LocalTime::DayOfWeek() const
{
	if (tdata==0)
		return 0;
    // days since year 0
    sint32 day = _daysTo(tdata->year) + (sint32)(DayOfYear()-1);
    day = day % 7;
    // this needs some offset adjustment
    return (uint32)day;
} 
uint32 LocalTime::Hour() const
{
	if (tdata==0)
		return 0;
	return (tdata->second % _SecondsInDay)/_SecondsInHour;
}
uint32 LocalTime::Minute() const 
{
	if (tdata==0)
		return 0;
	return (tdata->second % _SecondsInHour)/_SecondsInMinute;
}
uint32 LocalTime::Second() const
{
	if (tdata==0)
		return 0;
	return tdata->second % _SecondsInMinute;
} 
uint32 LocalTime::NanoSecond() const
{
	if (tdata==0)
		return 0;
	return tdata->nanosec;
} 

bool LocalTime::IsLeapYear() const
{
	if (tdata==0)
		return true; // year 0 was leap
	sint32 year = tdata->year;
	if (year%400 == 0) return true;
	if (year%100 == 0) return false;
	if (year%4 == 0) return true;
    return false;
}
/*
LocalTime LocalTime::operator+ (const LocalTime& t)
{
	if (tdata==0)
		return t;
	if (t.tdata==0)
		return *this;
	LocalTime r;
	TData* nt = r.tdata = new TData();
	nt->year = tdata->year + t.tdata->year;
	nt->second = tdata->second + t.tdata->second;
	if ((nt->nanosec = tdata->nanosec+t.tdata->nanosec) >= _NanosecsInSecond)
	{
		++nt->second;
		nt->nanosec -= _NanosecsInSecond;
	}	
	if (nt->second >= _SecondsInNormalYear)
	{  // the extra time has pushed us accross an year boundary
	   if (IsLeapYear())
	   {
			if (nt->second >= _SecondsInLeapYear)
			{
		   	   ++ nt->year;  // leap and boundary crossed
		   	   nt->second -= _SecondsInLeapYear;
			}
	   }
	   else
	   {
	   	  ++ nt->year;  // not leap and boundary crossed
	   	  nt->second -= _SecondsInNormalYear;
	   }
	}	
	return r;
}
LocalTime LocalTime::operator- (const LocalTime& t)
{
	if (t.tdata==0)
		return *this;
	LocalTime r;
	TData* nt = r.tdata = new TData();
	if (tdata==0)
	{
		nt->year    = 0;
		nt->second  = 0;
		nt->nanosec = 0;
	}
	else
	{
		nt->year    = tdata->year;
		nt->second  = tdata->second;
		nt->nanosec = tdata->nanosec;
	}

	nt->year -= t.tdata->year;
	if ((nt->nanosec -= t.tdata->nanosec) >= _NanosecsInSecond)
	{   // fix overflow
		nt->nanosec = _NanosecsInSecond - nt->nanosec;
		--nt->second;
	}
	if ((nt->second -= t.tdata->second) >= _SecondsInLeapYear)
	{   // fix overflow
	    --nt->year;
	    nt->second = (r.IsLeapYear() ? _SecondsInLeapYear : _SecondsInNormalYear) 
	    		   - nt->second;
	}
	
	return r;
}
*/

/*
LocalTime LocalTime::ToGM() const
{
	if (tdata->timeZoneBiasMinutes == 0)
		return *this;
	
	LocalTime t;
	TData* nt = new TData();
	t.tdata = nt;
	nt->nanosec = tdata->nanosec;
	nt->timeZoneBiasMinutes = 0; // GM time
	nt->second = tdata->second + tdata->timeZoneBiasMinutes;
	nt->year = tdata->year;
	if (nt->second >= _SecondsInNormalYear)
	{  // the extra time has pushed us accross an year boundary
	   if (IsLeapYear())
	   {
			if (nt->second >= _SecondsInLeapYear)
			{
		   	   ++ nt->year;  // leap and boundary crossed
		   	   nt->second -= _SecondsInLeapYear;
			}
	   }
	   else
	   {
	   	  ++ nt->year;  // not leap and boundary crossed
	   	  nt->second -= _SecondsInNormalYear;
	   }
	}	
	return t;
}
*/

LocalTime LocalTime::FromSecondsSince_1904Jan1(uint32 s) // Mac style
{
    return LocalTime(_DaysTo1904*(sint64)_SecondsInDay+s);
}
LocalTime LocalTime::FromSecondsSince_1970Jan1(uint32 s) // Linux style
{
    return LocalTime(_DaysTo1970*(sint64)_SecondsInDay+s);
}
LocalTime LocalTime::From100NanoSince_1601Jan1(uint64 ms) // Windows style
{
    LocalTime ltm(_DaysTo1601*(sint64)_SecondsInDay+ms/10000000);
    ltm.tdata->nanosec = (ms % 10000000) * 100;
    return ltm;
}

uint32 LocalTime::GetSecondsSince_1904Jan1() const // Mac style
{
	VERIFY (tdata!=0 && tdata->year>=1904);
    return (uint32)((_daysTo(tdata->year) - _DaysTo1904) * _SecondsInDay + tdata->second);
}
uint32 LocalTime::GetSecondsSince_1970Jan1() const // Unix style
{
	VERIFY (tdata!=0 && tdata->year>=1970);
    return (uint32)((_daysTo(tdata->year) - _DaysTo1970) * _SecondsInDay + tdata->second);
}
uint64 LocalTime::Get100NanoSince_1601Jan1() const // Windows style
{
	VERIFY (tdata!=0 && tdata->year>=1601);
    return ((_daysTo(tdata->year) - _DaysTo1601) * static_cast<uint64>(_SecondsInDay)  
    		+ tdata->second)  * 10000000 
    		+ tdata->nanosec / 100;
}

sint64 LocalTime::NormalizedSeconds() const 
{
	if (tdata==0)
		return 0;
    return _daysTo(tdata->year)  * static_cast<sint64>(_SecondsInDay) + tdata->second;
}

bool LocalTime::operator == (const LocalTime& t) const
{
	if (tdata == t.tdata)
		return true;
	if (tdata == 0 || t.tdata == 0)
		return false;
	if (tdata->year != t.tdata->year)
		return false;
	if (tdata->second != t.tdata->second)
		return false;
	return tdata->nanosec == t.tdata->nanosec;
}
bool LocalTime::operator <  (const LocalTime& t) const
{
	if (tdata == t.tdata)
		return false;
	if (tdata == 0)
		return (t.tdata->year >= 0);
	if (t.tdata == 0)
		return (tdata->year < 0);
	if (tdata->year != t.tdata->year)
		return (tdata->year < t.tdata->year);
	if (tdata->second != t.tdata->second)
		return tdata->second < t.tdata->second;
	return tdata->nanosec < t.tdata->nanosec;
}
bool LocalTime::operator <= (const LocalTime& t) const
{
	if (tdata == t.tdata)
		return true;
	if (tdata == 0)
		return (t.tdata->year >= 0);
	if (t.tdata == 0)
		return (tdata->year < 0);
	if (tdata->year != t.tdata->year)
		return (tdata->year < t.tdata->year);
	if (tdata->second != t.tdata->second)
		return tdata->second < t.tdata->second;
	return tdata->nanosec <= t.tdata->nanosec;
}

LocalTime::LocalTime(sint64 totalsecs)
: tdata(new TData())
{
	sint32 days = static_cast<sint32>(totalsecs / _SecondsInDay);
    sint32 tmp = days - (31+28);
	sint32 leapdays = 1+tmp/1461-tmp/36524+tmp/146097;
	tdata->year = (days-leapdays)/365;
	uint32 yday = static_cast<uint32>(days - _daysTo(tdata->year));
	tdata->second = yday*_SecondsInDay 
				  + static_cast<uint32>(totalsecs % _SecondsInDay);
	tdata->nanosec = 0;
}

LocalTime& LocalTime::operator += (sint64 t)
{
	sint64 os = NormalizedSeconds();

	if (tdata==0)
	{
		tdata = new TData();
		tdata->year = 0;
		tdata->second = 0;
		tdata->nanosec= 0;
	}
 	else if (tdata->refcount != 1)
 	{
 		refc<TData> o(tdata);
		tdata = new TData();
		tdata->year   = 0;
		tdata->second = 0;
		tdata->nanosec= o->nanosec;
 	}
	if ((tdata->nanosec += t % _NanosecsInSecond) > _NanosecsInSecond)
	{
	   os += ((t>0) ? 1:-1);
	   tdata->nanosec %= _NanosecsInSecond;
	}
	
	os += t / _NanosecsInSecond;

	sint32 days = static_cast<sint32>(os / _SecondsInDay);
    sint32 tmp = days - (31+28);
	sint32 leapdays = 1+tmp/1461-tmp/36524+tmp/146097;
	tdata->year = (days-leapdays)/365;
	uint32 yday = static_cast<uint32>(days - _daysTo(tdata->year));
	tdata->second = yday*_SecondsInDay 
				  + static_cast<uint32>(os % _SecondsInDay);
	return *this;
}


String LocalTime::ToString() const
{
	String s(String::From_c_str("yyyy.0m.0d 0h:0m:0s"));
    String v;
	String::iterator b,e;
	
	v = String::From_uint32(Month());
	e = s.begin()+7;
	b = e - v.length();
	s.replace(b,e, v.begin(), v.end());
	
	v = String::From_uint32(DayOfMonth());
	e = s.begin()+10;
	b = e - v.length();
	s.replace(b,e, v.begin(), v.end());

	v = String::From_uint32(Hour());
	e = s.begin()+13;
	b = e - v.length();
	s.replace(b,e, v.begin(), v.end());

	v = String::From_uint32(Minute());
	e = s.begin()+16;
	b = e - v.length();
	s.replace(b,e, v.begin(), v.end());

	v = String::From_uint32(Second());
	e = s.begin()+19;
	b = e - v.length();
	s.replace(b,e, v.begin(), v.end());
	
	b = s.begin();
	e = b+4;
    v = String::From_sint32(Year());
	s.replace(b,e, v.begin(), v.end());
	
	return s;
}



} // namespace XSP
