/*David Nishimoto
davepamn@relia.net
icoshedron
*/
#ifdef WIN32
#include "windows.h"
#endif

#include <stdio.h>
#include <string.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>
#include <GL/glaux.h> 
#include <math.h>
#include <stdlib.h>
#include <string.h>

/*Dictionary
2D classes
Point class - used to store a x and y coordinate

3D classes
Reflection Spline
REFLECTION_SPLINE_PATH_STRUCT - Store vector information
about the incident vector, intersection point, and
reflection ray.

Polygon Linked list
POLY_STRUCT-Linked list of polygons with four vertexes

Lathe Structure
LATHE_CONTROL_POINT_STRUCT-Stores the control points on the spline
CATMULL_ROM_CURVE_STRUCT_TYPE-List of spline control points.
VECTOR_STRUCT-A vector structure

DISPLAY_LIST_TYPE-List of Display List ids
*/

#define DISPLAY_LIST_SIZE 10
#define TEXTURE_MAP_LIST_SIZE 10
#define TRUE 1
#define FALSE 0
#define YES 1
#define NO 0
#define glRGB(x, y, z)	glColor3ub((GLubyte)x, (GLubyte)y, (GLubyte)z)

/*Camera Coordinates*/
class Camera
{
public:
GLfloat m_Translate_x;
GLfloat m_Translate_y;
GLfloat	m_Translate_z;

GLfloat m_TranslationUnit;
GLfloat m_camera_radsFromEast;
GLfloat m_camera_direction_y;
GLfloat m_camera_height_y;
};

typedef double MATRIX4X4 [4][4];
typedef double MATRIX4X1 [4][1];
typedef double MATRIX1X4 [1][4];

class Point
{
public:
	Point(){x=y=0.0f;}
	Point(GLint xx,GLint yy) {x=xx; y=yy;}
	void set(GLfloat xx, GLfloat yy) {x=xx;y=yy;}
	GLfloat getX() {return x;}
	GLfloat getY() {return y;}
private:
	GLfloat x,y;
};
typedef struct DISPLAY_LIST_STRUCT
{
	char name[40];
	GLint id;
	GLint available;
}DISPLAY_LIST_TYPE;

typedef struct POLYGON_STRUCT
{
	GLfloat vertexes[4][3];
struct POLYGON_STRUCT *next;
} POLYGON_STRUCT_TYPE;

typedef struct REFLECTION_SPLINE_PATH_STRUCT
{
GLfloat m_t;
GLfloat ray_begin_x;
GLfloat ray_begin_y;
GLfloat ray_begin_z;
GLfloat ray_end_x;
GLfloat ray_end_y;
GLfloat ray_end_z;
GLfloat ray_intersection_x;
GLfloat ray_intersection_y;
GLfloat ray_intersection_z;
GLfloat ray_reflection_x;
GLfloat ray_reflection_y;
GLfloat ray_reflection_z;
}REFLECTION_SPLINE_PATH_STRUCT_TYPE;

typedef struct LATHE_CONTROL_POINT_STRUCT
{
GLfloat Px[4];
GLfloat Py[4];
GLfloat Pz[4];
struct LATHE_CONTROL_POINT_STRUCT *next;
} LATHE_CONTROL_POINT_STRUCT_TYPE;

typedef struct LATHE_CATMULL_ROM_CURVE_STRUCT
{
	LATHE_CONTROL_POINT_STRUCT_TYPE *head;
	GLfloat m_t;
} LATHE_CATMULL_ROM_CURVE_STRUCT_TYPE;


typedef struct VECTOR_STRUCT
{
	GLfloat m_x;
	GLfloat m_y;
	GLfloat m_z;
} VECTOR_STRUCT_TYPE;


class Canvas{
public:
	GLint miWidth;
	GLint miHeight;
	GLint m_CurrDegree;
	POLYGON_STRUCT_TYPE *current;
	POLYGON_STRUCT_TYPE *head;
	REFLECTION_SPLINE_PATH_STRUCT_TYPE spline;
	LATHE_CATMULL_ROM_CURVE_STRUCT_TYPE lathe_spline;
	Camera myCamera;

	DISPLAY_LIST_TYPE display_list[DISPLAY_LIST_SIZE];
	DISPLAY_LIST_STRUCT texture_map_list[TEXTURE_MAP_LIST_SIZE];
		
	Canvas(GLint x, GLint y, GLint width, GLint height, char *windowTitle);
	~Canvas();
	void setPerspectiveWindow(GLdouble fovy, GLdouble aspect,GLdouble zNear,GLdouble zFar);
	void setWindow(GLfloat fLeft, GLfloat fRight, GLfloat fBottom, GLfloat fTop);
	void setViewport(GLint iX, GLint iY, GLsizei iWidth, GLsizei iHeight);
	void setBackgroundColor(GLfloat red, GLfloat green, GLfloat blue);
	void setColor(GLfloat red, GLfloat green, GLfloat blue);
	void setLighting();
	void setBlending();
	void setDepthBuffer();
	void forward(GLfloat dist, int isVisible);
	void moveTo(GLfloat x, GLfloat y);
	void lineTo(GLfloat x, GLfloat y);
	void turn(GLint angle) {m_CurrDegree=(m_CurrDegree+angle)%360;}
	void nGON(int n, GLfloat cx, GLfloat cy, GLfloat radius, GLfloat rotAngle);
	void drawArch(Point center, float radius, float startAngle, float sweep);
	void reduceToUnit(GLfloat vector[3]);
	void calcNormal(GLfloat v[3][3], GLfloat out[3]);
	void calcNormal(VECTOR_STRUCT_TYPE *vertex1, VECTOR_STRUCT_TYPE *vertex2,VECTOR_STRUCT_TYPE *vertex3,GLfloat out[3]);
	void setDiffuseMaterialColor(GLfloat red, GLfloat green, GLfloat blue,GLfloat alpha);
	void setAmbientMaterialColor(GLfloat red, GLfloat green, GLfloat blue,GLfloat alpha);
	void setMaterialShininess(GLint intensity);
	void setSpecularMaterialColor(GLfloat red, GLfloat green, GLfloat blue,GLfloat alpha);
	GLfloat getPoint(GLfloat x, GLfloat z, GLfloat normal[3],GLfloat vertexes[4][3]);
	void Cylinder(GLfloat radius,GLfloat height);
	void Sphere(GLfloat radius);
	void Box(GLfloat w,GLfloat h, GLfloat d);
	void Matrix4X1_4X4(MATRIX4X1 *result, MATRIX4X1 *matrix1, MATRIX4X4 *matrix2);
	double Matrix4X1_1X4(MATRIX4X1 *matrix1, MATRIX1X4 *matrix2);
	void Matrix1X4_4X4(MATRIX1X4 *result, MATRIX1X4 *matrix1, MATRIX4X4 *matrix2);
	VECTOR_STRUCT_TYPE* ParameterizeSplineCurve(LATHE_CONTROL_POINT_STRUCT_TYPE *ctrl, GLfloat t);
	VECTOR_STRUCT_TYPE* Rotate(GLfloat angle,GLfloat ux,GLfloat uy,GLfloat uz, VECTOR_STRUCT_TYPE *vector);
	GLint ReserveDisplayListId(char *name);
	GLint FindDisplayListId(char *name);
	GLint ReserveTextureMap(char *name,char *filename);
	GLint FindTextureMap(char *name);
	void Sinc(char *Name);
	GLubyte * LoadDIBitmap(const char *filename,BITMAPINFO **info);
	GLuint TextureLoad(char *filename,GLboolean alpha,GLenum minfilter,GLenum magfilter,GLenum wrap);      
	void TextureSphere(GLfloat radius);
	void TextureCylinder(GLfloat radius,GLfloat height);
	void setTextureMapping();
	void buildGrid(char *Name);
private:
	Point CP;
};
Canvas::Canvas(GLint x, GLint y, GLint width, GLint height, char *windowTitle)
{
	char * argv[1];
	char dummyString[8];
	argv[0]=dummyString;
	int argc=1;
	int i;

	glutInit(&argc,argv);
	glutInitDisplayMode(GLUT_RGB | GLUT_DEPTH | GLUT_DOUBLE);
	glutInitWindowSize(width,height);
	glutInitWindowPosition(x,y);
	if (glutCreateWindow(windowTitle) == GL_FALSE)
		exit(-1);
	miWidth=width;
	miHeight=height;
	m_CurrDegree=0;

	for(i=0;i<DISPLAY_LIST_SIZE; i++)
	{
		display_list[i].available=YES;
		display_list[i].id=-1;
	}

	for(i=0;i<TEXTURE_MAP_LIST_SIZE; i++)
	{
		texture_map_list[i].available=YES;
		texture_map_list[i].id=-1;
	}

	/*Camera Initialization*/

	myCamera.m_Translate_x=0;
	myCamera.m_Translate_y=10;
	myCamera.m_Translate_z=50;
	myCamera.m_TranslationUnit=5;
	myCamera.m_camera_radsFromEast=1.57;

}
Canvas::~Canvas()
{
	POLYGON_STRUCT_TYPE *a_current;
	POLYGON_STRUCT_TYPE *temp;
	LATHE_CONTROL_POINT_STRUCT_TYPE *a_current2;
	LATHE_CONTROL_POINT_STRUCT_TYPE *temp2;
	/*User defined Polygon List*/
	a_current=head;

	while(a_current!=NULL)
	{
		temp=a_current;
		a_current=a_current->next;
		free(temp);
	}
	head=NULL;

	/*Lathe Spline Curve List*/
	a_current2=lathe_spline.head;
	while(a_current2!=NULL)
	{
		temp2=a_current2;
		a_current2=a_current2->next;
		free(temp2);
	}
	lathe_spline.head=NULL;

}
GLubyte * Canvas::LoadDIBitmap(const char *filename, /* I - File to load */
             BITMAPINFO **info)    /* O - Bitmap information */
    {
    FILE             *fp;          /* Open file pointer */
    GLubyte          *bits;        /* Bitmap pixel bits */
    int              bitsize;      /* Size of bitmap */
    int              infosize;     /* Size of header information */
    BITMAPFILEHEADER header;       /* File header */


    /* Try opening the file; use "rb" mode to read this *binary* file. */
    if ((fp = fopen(filename, "rb")) == NULL)
        return (NULL);

    /* Read the file header and any following bitmap information... */
    if (fread(&header, sizeof(BITMAPFILEHEADER), 1, fp) < 1)
        {
        /* Couldn't read the file header - return NULL... */
	fclose(fp);
        return (NULL);
        }

    if (header.bfType != 'MB')	/* Check for BM reversed... */
        {
        /* Not a bitmap file - return NULL... */
        fclose(fp);
        return (NULL);
        }

    infosize = header.bfOffBits - sizeof(BITMAPFILEHEADER);
    if ((*info = (BITMAPINFO *)malloc(infosize)) == NULL)
        {
        /* Couldn't allocate memory for bitmap info - return NULL... */
        fclose(fp);
        return (NULL);
        }

    if (fread(*info, 1, infosize, fp) < infosize)
        {
        /* Couldn't read the bitmap header - return NULL... */
        free(*info);
        fclose(fp);
        return (NULL);
        }

    /* Now that we have all the header info read in, allocate memory for *
     * the bitmap and read *it* in...                                    */
    if ((bitsize = (*info)->bmiHeader.biSizeImage) == 0)
        bitsize = ((*info)->bmiHeader.biWidth *
                   (*info)->bmiHeader.biBitCount + 7) / 8 *
  	           abs((*info)->bmiHeader.biHeight);

    if ((bits = (GLubyte *) malloc(bitsize)) == NULL)
        {
        /* Couldn't allocate memory - return NULL! */
        free(*info);
        fclose(fp);
        return (NULL);
        }

    if (fread(bits, 1, bitsize, fp) < bitsize)
        {
        /* Couldn't read bitmap - free memory and return NULL! */
        free(*info);
        free(bits);
        fclose(fp);
        return (NULL);
        }

    /* OK, everything went fine - return the allocated bitmap... */
    fclose(fp);
    return (bits);
}
GLuint Canvas::TextureLoad(char      *filename, /* I - Bitmap file to load */
            GLboolean alpha,     /* I - Generate alpha for bitmap */
            GLenum    minfilter, /* I - Minification filter */
	    GLenum    magfilter, /* I - Magnification filter */
	    GLenum    wrap)      /* I - Repeat or clamp */
    {
    int         i;               /* Looping var */
    BITMAPINFO	*info;           /* Bitmap information */
    GLubyte	*bits;           /* Bitmap RGB pixels */
    GLubyte     *ptr;            /* Pointer into bit buffer */
    GLubyte     temp;            /* Swapping variable */
    GLenum      type;            /* Texture type */
    GLuint      texture;         /* Texture object */

    /* Try loading the bitmap file... */
    bits = LoadDIBitmap(filename, &info);
    if (bits == NULL)
        return (0);

    /*
     * Convert the bitmap data from BGR to RGB. Since texture images
     * must be a power of two, and since we can figure that we won't
     * have a texture image as small as 2x2 pixels, we ignore any
     * alignment concerns...
     */

    for (i = info->bmiHeader.biWidth * info->bmiHeader.biHeight, ptr = bits;
         i > 0;
	 i --, ptr += 3)
        {
	/* Swap red and blue */
	temp   = ptr[0];
	ptr[0] = ptr[2];
	ptr[2] = temp;
	}

    /* Figure out the type of texture... */
    if (info->bmiHeader.biHeight == 1)
        type = GL_TEXTURE_1D;
    else
        type = GL_TEXTURE_2D;

    /* Create and bind a texture object */
    glGenTextures(1, &texture);
    glBindTexture(type, texture);

    /* Set texture parameters */
    glTexParameteri(type, GL_TEXTURE_MAG_FILTER, magfilter);
    glTexParameteri(type, GL_TEXTURE_MIN_FILTER, minfilter);
    glTexParameteri(type, GL_TEXTURE_WRAP_S, wrap);
    glTexParameteri(type, GL_TEXTURE_WRAP_T, wrap);
    glTexEnvi(type, GL_TEXTURE_ENV_MODE, GL_MODULATE);

    /*
     * Set texture image; if the minification filter uses mip-mapping
     * then use gluBuild2D/1DMipmaps() to load the texture...
     */

    if (minfilter == GL_LINEAR || minfilter == GL_NEAREST)
        glTexImage2D(type, 0, 3, info->bmiHeader.biWidth,
                     info->bmiHeader.biHeight, 0, GL_RGB,
                     GL_UNSIGNED_BYTE, bits);
    else if (type == GL_TEXTURE_1D)
        gluBuild1DMipmaps(type, 3, info->bmiHeader.biWidth,
                          GL_RGB, GL_UNSIGNED_BYTE, bits);
    else
        gluBuild2DMipmaps(type, 3, info->bmiHeader.biWidth,
                          info->bmiHeader.biHeight, GL_RGB,
                          GL_UNSIGNED_BYTE, bits);

    /* Free the bitmap and return... */
    free(info);
    free(bits);

    return (texture);
}
void Canvas::setTextureMapping()
{
  glEnable(GL_TEXTURE_2D);


}
void Canvas::setWindow(GLfloat fLeft, GLfloat fRight, GLfloat fBottom, GLfloat fTop)
{
	GLfloat zNear=1.0f;
	GLfloat zFar=2000.0f;

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	//gluOrtho2D(fLeft,fRight,fBottom,fTop);
	glOrtho(fLeft,fRight,fBottom,fTop, zNear, zFar);

}
void Canvas::setPerspectiveWindow(GLdouble fovy, GLdouble aspect,GLdouble zNear,GLdouble zFar)
{
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(fovy, aspect,zNear,  zFar);
}
void Canvas::setViewport(GLint iX, GLint iY, GLsizei iWidth, GLsizei iHeight)
{
	glViewport(iX,iY,iWidth,iHeight);
}
void Canvas::setBackgroundColor(GLfloat red, GLfloat green, GLfloat blue)
{
	glClearColor(red,green,blue,1);
}
void Canvas::setColor(GLfloat red, GLfloat green, GLfloat blue)
{
	glColor3f(red,green,blue);
}
void Canvas::setLighting()
{
  GLfloat light0pos[4]=     {0.7071, 0.7071, 5.0, 0.0};
  GLfloat light0ambient[4]= {0.6, 0.6, 0.6, 1.0};
  GLfloat light0diffuse[4]= {0.6, 0.6, 0.6, 1.0};
  GLfloat light0specular[4]={0.8, 0.8, 0.8, 1.0};
  GLfloat lmodel_ambient[]= {0.6f,0.6f,0.6f,1.0f};

  glLightModelfv(GL_LIGHT_MODEL_AMBIENT,lmodel_ambient);

  //glDisable(GL_DITHER);
  glShadeModel(GL_SMOOTH);
  
  glLightfv(GL_LIGHT0, GL_POSITION, light0pos);
  glLightfv(GL_LIGHT0, GL_AMBIENT, light0ambient);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, light0diffuse);
  glLightfv(GL_LIGHT0, GL_SPECULAR, light0specular);
  
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);

}
void Canvas::setBlending()
{
	glEnable(GL_BLEND);
	/*GL_SRC_ALPHA = Source color multiplied by source alpha
	  GL_ONE_MINUS_SRC_ALPHA =  Destination color is multiplied by 1,1,1,1 - source color)
	*/
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

}
void Canvas::setDepthBuffer()
{
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LESS);
  glEnable(GL_CULL_FACE);
  glCullFace(GL_BACK);
  glFrontFace(GL_CCW);
}
void Canvas::forward(GLfloat dist, int isVisible)
{
	const float RadPerDeg=0.017453393;
	float x=CP.getX() + dist * cos(RadPerDeg*m_CurrDegree);
	float y=CP.getY() + dist * sin(RadPerDeg*m_CurrDegree);
	if(isVisible)
	{
		lineTo(x,y);
	}
	else
	{
		moveTo(x,y);
	}
}
void Canvas::moveTo(GLfloat x, GLfloat y)
{
	CP.set(x,y);
}
void Canvas::lineTo(GLfloat x, GLfloat y)
{
	glBegin(GL_LINES);
		glVertex2f(CP.getX(),CP.getY());
		glVertex2f(x,y);
	glEnd();
	CP.set(x,y);
	glFinish();
}
void Canvas::nGON(int n, GLfloat cx, GLfloat cy, GLfloat radius, GLfloat rotAngle)
{
	if(n<3) return;
	GLdouble angle= rotAngle * 3.14159265/180.0f;

	double angleInc=2*3.14159265/n;
	moveTo(radius*cos(angle)+cx,radius*sin(angle+cy));
	for(int k=0; k<n; k++)
	{
		angle+=angleInc;
		lineTo(radius*cos(angle)+cx,radius*sin(angle)+cy);
	}

}
void Canvas::drawArch(Point center, float radius, float startAngle, float sweep)
{
	const int n=36;
	float angle=startAngle * 3.14159265/180.0f;
	float angleInc=sweep * 3.14159265/(180.0f*n);
	float cx=center.getX(), cy=center.getY();
	moveTo(cx+radius*cos(angle), cy+radius*sin(angle));
	for(int k=1; k<=(n+1); k++, angle += angleInc)
	{
		lineTo(cx+radius*cos(angle),cy+radius*sin(angle));
	}
}
void Canvas::reduceToUnit(GLfloat vector[3])
{
float length;
length = (float)sqrt((vector[0]*vector[0]) + 
(vector[1]*vector[1]) +
(vector[2]*vector[2]));
if(length == 0.0f)
length = 1.0f;
vector[0] /= length;
vector[1] /= length;
vector[2] /= length;
}
void Canvas::calcNormal(VECTOR_STRUCT_TYPE *vertex1, VECTOR_STRUCT_TYPE *vertex2,VECTOR_STRUCT_TYPE *vertex3,GLfloat out[3])
{
float v1[3],v2[3];
static const int x = 0;
static const int y = 1;
static const int z = 2;
v1[x] = vertex1->m_x - vertex2->m_x;
v1[y] = vertex1->m_y - vertex2->m_y;
v1[z] = vertex1->m_z - vertex2->m_z;
v2[x] = vertex2->m_x - vertex3->m_x;
v2[y] = vertex2->m_y - vertex3->m_y;
v2[z] = vertex2->m_z - vertex3->m_z;
/*Cross Product*/
out[x] = v1[y]*v2[z] - v1[z]*v2[y];
out[y] = v1[z]*v2[x] - v1[x]*v2[z];
out[z] = v1[x]*v2[y] - v1[y]*v2[x];
reduceToUnit(out);

}
void Canvas::calcNormal(GLfloat v[3][3], GLfloat out[3])
{
float v1[3],v2[3];
static const int x = 0;
static const int y = 1;
static const int z = 2;
v1[x] = v[0][x] - v[1][x];
v1[y] = v[0][y] - v[1][y];
v1[z] = v[0][z] - v[1][z];
v2[x] = v[1][x] - v[2][x];
v2[y] = v[1][y] - v[2][y];
v2[z] = v[1][z] - v[2][z];
/*Cross Product*/
out[x] = v1[y]*v2[z] - v1[z]*v2[y];
out[y] = v1[z]*v2[x] - v1[x]*v2[z];
out[z] = v1[x]*v2[y] - v1[y]*v2[x];
reduceToUnit(out);
}
void Canvas::setDiffuseMaterialColor(GLfloat red, GLfloat green, GLfloat blue,GLfloat alpha)
{
	GLfloat diff[4];
	diff[0]=red;
	diff[1]=green;
	diff[2]=blue;
	diff[3]=alpha;
	glMaterialfv(GL_FRONT, GL_DIFFUSE,diff); 

}
void Canvas::setAmbientMaterialColor(GLfloat red, GLfloat green, GLfloat blue,GLfloat alpha)
{
	GLfloat amb[4];
	amb[0]=red;
	amb[1]=green;
	amb[2]=blue;
	amb[3]=alpha;
	glMaterialfv(GL_FRONT, GL_AMBIENT,amb); 

}
void Canvas::setSpecularMaterialColor(GLfloat red, GLfloat green, GLfloat blue,GLfloat alpha)
{
	GLfloat spec[4];
	spec[0]=red;
	spec[1]=green;
	spec[2]=blue;
	spec[3]=alpha;
	glMaterialfv(GL_FRONT, GL_SPECULAR,spec); 

}
void Canvas::setMaterialShininess(GLint shininess)
{
	glMateriali(GL_FRONT, GL_SHININESS, shininess);
}
GLfloat Canvas::getPoint(GLfloat x, GLfloat z, GLfloat normal[3],GLfloat vertexes[4][3])
{
  GLfloat a,b,y,c,d;
  
  a=normal[0];
  b=normal[1];
  c=normal[2];
  d=sqrt(vertexes[0][0]*vertexes[0][0]+vertexes[0][1]*vertexes[0][1]+vertexes[0][2]*vertexes[0][2]);
  d-=sqrt(vertexes[1][0]*vertexes[1][0]+vertexes[1][1]*vertexes[1][1]+vertexes[1][2]*vertexes[1][2]);
  d-=sqrt(vertexes[2][0]*vertexes[2][0]+vertexes[2][1]*vertexes[2][1]+vertexes[2][2]*vertexes[2][2]);
  d-=sqrt(vertexes[3][0]*vertexes[3][0]+vertexes[3][1]*vertexes[3][1]+vertexes[3][2]*vertexes[3][2]);

  //fudge factor
  d*=1.1f;

  d/=4;

  //equation of a plane is ax+by+cz-d=0  where d is the distance from the orign
  if (b>0)
  {
	  y=(-(a*x)-(c*z)-d)/b;
  }
  else
  {
	  y=0;
  }
	
  return(y);
}
void Canvas::Cylinder(GLfloat radius,GLfloat height)
{
GLUquadricObj *obj = gluNewQuadric();
gluQuadricDrawStyle(obj,GLU_FILL);
gluQuadricNormals(obj,GLU_SMOOTH);
gluQuadricOrientation(obj,GLU_OUTSIDE);
gluQuadricTexture(obj, GL_TRUE);

/*Base Cylinder*/
glPushMatrix();
/* Rotate about the X axis */
glRotatef(90.0f,1.0f,0.0f,0.0f);
/* qobj = quadric object
baseRadius
topRadius
height,
slices (Subdivisions around the Z axis)
stacks (Subdivisions around the Z axis)
*/
gluCylinder(obj,radius,radius,height,20,10);
glPopMatrix(); 
/* Lid*/
glPushMatrix(); 
glRotatef(90.0f,1.0f,0.0f,0.0f);
/* qobj = quadric object
innerRadius = Inner radius of the disk
outerRadius = Outer radius of the disk
slices = number of subdivisions around the Z axis
loops = number of concentric rings about the origin
*/
gluDisk(obj,0.0f,radius,20,20);
glPopMatrix();
/*Bottom*/
glPushMatrix(); 
glTranslatef(0.0f,-height,0.0f);
glRotatef(90.0f,1.0f,0.0f,0.0f);
/* qobj = quadric object
innerRadius = Inner radius of the disk
outerRadius = Outer radius of the disk
slices = number of subdivisions around the Z axis
loops = number of concentric rings about the origin
*/
gluDisk(obj,0.0f,radius,20,20);
glPopMatrix();
gluDeleteQuadric(obj);

}

void Canvas::Sphere(GLfloat radius)
{
	GLUquadricObj *obj = gluNewQuadric();
	gluQuadricDrawStyle(obj,GLU_FILL);
	gluQuadricNormals(obj,GLU_SMOOTH);
	gluQuadricOrientation(obj,GLU_OUTSIDE);
	gluQuadricTexture(obj, GL_TRUE);

	gluSphere(obj, radius, 20, 20);
	gluDeleteQuadric(obj);
}
void Canvas::Box(GLfloat w,GLfloat h,GLfloat d)
{

	glBegin(GL_POLYGON);
		glNormal3f(0,0,1);
		glTexCoord2f(0.0f, 1.0f);
		glVertex3f(1.0f*w,	1.0f*h,	1.0f*d);
		glTexCoord2f(0.0f, 0.0f);
		glVertex3f(-1.0f*w,	1.0f*h,	1.0f*d);
		glTexCoord2f(1.0f, 0.0f);
		glVertex3f(-1.0f*w,	-1.0f*h,	1.0f*d);
		glTexCoord2f(1.0f, 1.0f);
		glVertex3f(1.0f*w,	-1.0f*h,	1.0f*d);
	glEnd();

	glBegin(GL_POLYGON);
		glNormal3f(0,0,-1);
		glTexCoord2f(0.0f, 1.0f);
		glVertex3f(1.0f*w,	1.0f*h,	-1.0f*d);
		glTexCoord2f(0.0f, 0.0f);
		glVertex3f(1.0f*w,	-1.0f*h,	-1.0f*d);
		glTexCoord2f(1.0f, 0.0f);
		glVertex3f(-1.0f*w,	-1.0f*h,	-1.0f*d);
		glTexCoord2f(1.0f, 1.0f);
		glVertex3f(-1.0f*w,	1.0f*h,	-1.0f*d);
	glEnd();

	glBegin(GL_POLYGON);
		glNormal3f(-1,0,0);
		glTexCoord2f(0.0f, 1.0f);
		glVertex3f(-1.0f*w,	1.0f*h,	1.0f*d);
		glTexCoord2f(0.0f, 0.0f);
		glVertex3f(-1.0f*w,	1.0f*h,	-1.0f*d);
		glTexCoord2f(1.0f, 0.0f);
		glVertex3f(-1.0f*w,	-1.0f*h,	-1.0f*d);
		glTexCoord2f(1.0f, 1.0f);
		glVertex3f(-1.0f*w,	-1.0f*h,	1.0f*d);
	glEnd();

	glBegin(GL_POLYGON);
		glNormal3f(1,0,0);
		glTexCoord2f(0.0f, 1.0f);
		glVertex3f(1.0f*w,	1.0f*h,	1.0f*d);
		glTexCoord2f(0.0f, 0.0f);
		glVertex3f(1.0f*w,	-1.0f*h,	1.0f*d);
		glTexCoord2f(1.0f, 0.0f);
		glVertex3f(1.0f*w,	-1.0f*h,	-1.0f*d);
		glTexCoord2f(1.0f, 1.0f);
		glVertex3f(1.0f*w,	1.0f*h,	-1.0f*d);
	glEnd();

	glBegin(GL_POLYGON);
		glNormal3f(0,1,0);
		glTexCoord2f(0.0f, 1.0f);
		glVertex3f(-1.0f*w,	1.0f*h,	-1.0f*d);
		glTexCoord2f(0.0f, 0.0f);
		glVertex3f(-1.0f*w,	1.0f*h,	1.0f*d);
		glTexCoord2f(1.0f, 0.0f);
		glVertex3f(1.0f*w,	1.0f*h,	1.0f*d);
		glTexCoord2f(1.0f, 1.0f);
		glVertex3f(1.0f*w,	1.0f*h,	-1.0f*d);
	glEnd();

	glBegin(GL_POLYGON);
		glNormal3f(0,-1,0);
		glTexCoord2f(0.0f, 1.0f);
		glVertex3f(-1.0f*w,	-1.0f*h,	-1.0f*d);
		glTexCoord2f(0.0f, 0.0f);
		glVertex3f(1.0f*w,	-1.0f*h,	-1.0f*d);
		glTexCoord2f(1.0f, 0.0f);
		glVertex3f(1.0f*w,	-1.0f*h,	1.0f*d);
		glTexCoord2f(1.0f, 1.0f);
		glVertex3f(-1.0f*w,	-1.0f*h,	1.0f*d);
	glEnd();

}

VECTOR_STRUCT_TYPE* Canvas::ParameterizeSplineCurve(LATHE_CONTROL_POINT_STRUCT_TYPE *ctrl, GLfloat t)
{
VECTOR_STRUCT_TYPE *vector;
MATRIX4X4 Mcr;
MATRIX4X1 Mt;
MATRIX4X1 MPar;
MATRIX1X4 Px;
MATRIX1X4 Py;
MATRIX1X4 Pz;

	vector= (VECTOR_STRUCT_TYPE *) malloc (sizeof(VECTOR_STRUCT_TYPE));
	Mcr[0][0]= -0.5; Mcr[1][0]= 1.5; Mcr[2][0]= -1.5; Mcr[3][0]= 0.5;
	Mcr[0][1]= 1.0; Mcr[1][1]= -2.5; Mcr[2][1]= 2.0; Mcr[3][1]= -0.5;
	Mcr[0][2]= -0.5; Mcr[1][2]= 0.0; Mcr[2][2]= 0.5; Mcr[3][2]= 0.0;
	Mcr[0][3]= 0.0; Mcr[1][3]= 1.0; Mcr[2][3]= 0.0; Mcr[3][3]= 0.0;
	
	Px[0][0]=ctrl->Px[0];
	Px[0][1]=ctrl->Px[1];
	Px[0][2]=ctrl->Px[2];
	Px[0][3]=ctrl->Px[3];
	Py[0][0]=ctrl->Py[0];
	Py[0][1]=ctrl->Py[1];
	Py[0][2]=ctrl->Py[2];
	Py[0][3]=ctrl->Py[3];
	Pz[0][0]=ctrl->Pz[0];
	Pz[0][1]=ctrl->Pz[1];
	Pz[0][2]=ctrl->Pz[2];
	Pz[0][3]=ctrl->Pz[3];

	MPar[0][0]=t*t*t;
	MPar[0][1]=t*t;
	MPar[0][2]=t;
	MPar[0][3]=1;
	
	Matrix4X1_4X4(&Mt,&MPar,&Mcr); 
	vector->m_x = (GLfloat) Matrix4X1_1X4(&Mt,&Px); 
	vector->m_y = (GLfloat) Matrix4X1_1X4(&Mt,&Py); 
	vector->m_z = (GLfloat) Matrix4X1_1X4(&Mt,&Pz); 

	return (vector);
}
double Canvas::Matrix4X1_1X4(MATRIX4X1 *matrix1, MATRIX1X4 *matrix2)
{
register int k;
double result=0.0;
	for (k = 0 ; k < 4 ; k++)
		result += (*matrix1)[k][0] * (*matrix2)[0][k];
	return(result);
}
void Canvas::Matrix4X1_4X4(MATRIX4X1 *result, MATRIX4X1 *matrix1, MATRIX4X4 *matrix2)
{
register int i, k;
MATRIX4X1 temp_matrix;
	for (i = 0 ; i < 4 ; i++)
	{
	temp_matrix[i][0] = 0.0;
		for (k = 0 ; k < 4 ; k++)
		{
			temp_matrix[i][0] += (*matrix1)[k][0] * (*matrix2)[i][k];
		}
	}
	for (i = 0 ; i < 4 ; i++)
		(*result)[i][0] = temp_matrix[i][0];
}
void Canvas::Matrix1X4_4X4(MATRIX1X4 *result, MATRIX1X4 *matrix1, MATRIX4X4 *matrix2)
{
register int i, k;
MATRIX4X1 temp_matrix;
	for (i = 0 ; i < 4 ; i++)
	{
		temp_matrix[i][0] = 0.0;
			for (k = 0 ; k < 4 ; k++)
			{
				temp_matrix[i][0] += (*matrix1)[0][k] * (*matrix2)[i][k];
			}
	}
	for (i = 0 ; i < 4 ; i++)
		(*result)[0][i] = temp_matrix[i][0];
}

VECTOR_STRUCT_TYPE* Canvas::Rotate(GLfloat angle,GLfloat ux,GLfloat uy,GLfloat uz, VECTOR_STRUCT_TYPE *vector)
{
MATRIX4X4 Mt;
MATRIX1X4 Mp;
MATRIX1X4 Mr;

float sinTheta;
float cosTheta;
VECTOR_STRUCT_TYPE *ret_vector;

ret_vector= (VECTOR_STRUCT_TYPE *) malloc (sizeof(VECTOR_STRUCT_TYPE));

sinTheta=sin(angle/180.0f*3.142);
cosTheta=cos(angle/180.0f*3.142);

Mt[0][0]= cosTheta+(1-cosTheta)*pow(ux,2);
Mt[0][1]= (1-cosTheta)*uy*ux-sinTheta*uz; 
Mt[0][2]= (1-cosTheta)*uz*ux+sinTheta*uy;
Mt[0][3]= 0.0f;  

Mt[1][0]= (1-cosTheta)*ux*uy+sinTheta*uz;
Mt[1][1]= cosTheta+(1-cosTheta)*pow(uy,2);
Mt[1][2]= (1-cosTheta)*uz*uy-sinTheta*ux;
Mt[1][3]= 0.0f; 

Mt[2][0]= (1-cosTheta)*ux*uz-sinTheta*uy;
Mt[2][1]= (1-cosTheta)*uy*uz+sinTheta*ux;
Mt[2][2]= cosTheta+(1-cosTheta)*pow(uz,2);
Mt[2][3]= 0.0f;

Mt[3][0]=0;
Mt[3][1]=0;
Mt[3][2]=0;
Mt[3][3]=1;

Mp[0][0]=vector->m_x;
Mp[0][1]=vector->m_y;
Mp[0][2]=vector->m_z;
Mp[0][3]=1;


Matrix1X4_4X4(&Mr,&Mp,&Mt); 

ret_vector->m_x=Mr[0][0];
ret_vector->m_y=Mr[0][1];
ret_vector->m_z=Mr[0][2];
return(ret_vector);

}
GLint Canvas::ReserveDisplayListId(char *name)
{
	int i;
	
	for(i=0;i<DISPLAY_LIST_SIZE; i++)
	{
		if(display_list[i].available==YES)
		{
			display_list[i].available=NO;
			strcpy(display_list[i].name,name);
			display_list[i].id=glGenLists(1);
			return(display_list[i].id);
		}
	}
	return(-1);
}
GLint Canvas::FindDisplayListId(char *name)
{
	int i;
	
	for(i=0;i<DISPLAY_LIST_SIZE; i++)
	{
		if(strcmp(display_list[i].name,name)==0)
		{
			return(display_list[i].id);
		}
	}
	return(-1);
}

GLint Canvas::ReserveTextureMap(char *name,char *filename)
{

	int i;
	
	for(i=0;i<TEXTURE_MAP_LIST_SIZE; i++)
	{
		if(texture_map_list[i].available==YES)
		{
			texture_map_list[i].available=NO;
			strcpy(texture_map_list[i].name,name);

			texture_map_list[i].id=TextureLoad(filename, GL_FALSE, GL_NEAREST,
                                GL_NEAREST, GL_REPEAT);
			return(texture_map_list[i].id);
		}
	}
	return(-1);

}
GLint Canvas::FindTextureMap(char *name)
{
	int i;
	
	for(i=0;i<TEXTURE_MAP_LIST_SIZE; i++)
	{
		if(strcmp(texture_map_list[i].name,name)==0)
		{
			return(texture_map_list[i].id);
		}
	}
	return(-1);
}

void Canvas::Sinc(char *Name)
{
	float vertexes[4][3];
	float normal[4];
	GLint sinc_id;
	GLfloat x,z,y;
	GLfloat xt,zt;
	GLfloat square_size;

	sinc_id=ReserveDisplayListId(Name);

	if(sinc_id==-1)
	{
		printf("\nNo more Display List ids available\n");
		return;
	}

	glNewList(sinc_id,GL_COMPILE);

	setAmbientMaterialColor(0.0f,0.0f,0.5f,1.0f);
	setDiffuseMaterialColor(0.0f,0.0f,0.8f,1.0f);
	
	square_size=0.5;

	for(z=-20; z<=20; z+=square_size)
		for(x=-20;x<=20; x+=square_size)
	{

		/*Vertex 1*/
		xt=x;
		zt=z;
		if (xt==0 && zt==0)
		{
			y=0;
		}
		else
		{
		y=3*sin(sqrt(xt*xt+zt*zt))/sqrt(xt*xt+zt*zt);
		}
		
		
		vertexes[0][0]=xt;
		vertexes[0][1]=y;
		vertexes[0][2]=zt;

		/*Vertex 2*/
	
		xt=x;
		zt=z+square_size;
		if (xt==0 && zt==0)
		{
			y=0;
		}
		else
		{
		y=3*sin(sqrt(xt*xt+zt*zt))/sqrt(xt*xt+zt*zt);
		}
		
		
		vertexes[1][0]=xt;
		vertexes[1][1]=y;
		vertexes[1][2]=zt;

		/*Vertex 3*/
	
		xt=x+square_size;
		zt=z+square_size;
		if (xt==0 && zt==0)
		{
			y=0;
		}
		else
		{
		y=3*sin(sqrt(xt*xt+zt*zt))/sqrt(xt*xt+zt*zt);
		}
		
		vertexes[2][0]=xt;
		vertexes[2][1]=y;
		vertexes[2][2]=zt;

		/*Vertex 4*/
		xt=x+square_size;
		zt=z;
		if (xt==0 && zt==0)
		{
			y=0;
		}
		else
		{
		y=3*sin(sqrt(xt*xt+zt*zt))/sqrt(xt*xt+zt*zt);
		}
		
		vertexes[3][0]=xt;
		vertexes[3][1]=y;
		vertexes[3][2]=zt;

		calcNormal(vertexes,normal);
		glBegin(GL_POLYGON);
			glNormal3f(normal[0],normal[1],normal[2]);
			glVertex3f(vertexes[0][0],vertexes[0][1],vertexes[0][2]);
			glVertex3f(vertexes[1][0],vertexes[1][1],vertexes[1][2]);
			glVertex3f(vertexes[2][0],vertexes[2][1],vertexes[2][2]);
			glVertex3f(vertexes[3][0],vertexes[3][1],vertexes[3][2]);
		glEnd();


	}
	glEndList();

}
void Canvas::TextureCylinder(GLfloat radius,GLfloat height)
{
	GLUquadricObj *obj = gluNewQuadric();

gluQuadricDrawStyle(obj,GLU_FILL);
gluQuadricNormals(obj,GLU_SMOOTH);
gluQuadricOrientation(obj,GLU_OUTSIDE);
gluQuadricTexture(obj, GL_TRUE);

/*Base Cylinder*/
glPushMatrix();
/* Rotate about the X axis */
glRotatef(90.0f,1.0f,0.0f,0.0f);
/* qobj = quadric object
baseRadius
topRadius
height,
slices (Subdivisions around the Z axis)
stacks (Subdivisions around the Z axis)
*/
gluCylinder(obj,radius,radius,height,10,10);
glPopMatrix(); 
/* Lid*/
glPushMatrix(); 
glTranslatef(0.0f,0.0f,0.0f);
glRotatef(-90.0f,1.0f,0.0f,0.0f);
/* qobj = quadric object
innerRadius = Inner radius of the disk
outerRadius = Outer radius of the disk
slices = number of subdivisions around the Z axis
loops = number of concentric rings about the origin
*/
gluDisk(obj,0.0f,radius,20,20);
glPopMatrix();
/*Bottom*/
glPushMatrix(); 
glTranslatef(0.0f,-height,0.0f);
glRotatef(90.0f,1.0f,0.0f,0.0f);
/* qobj = quadric object
innerRadius = Inner radius of the disk
outerRadius = Outer radius of the disk
slices = number of subdivisions around the Z axis
loops = number of concentric rings about the origin
*/
gluDisk(obj,0.0f,radius,20,20);
glPopMatrix();
gluDeleteQuadric(obj);
}
void Canvas::TextureSphere(GLfloat radius)
{
	GLUquadricObj *obj = gluNewQuadric();

	gluQuadricDrawStyle(obj,GLU_FILL);
	gluQuadricNormals(obj,GLU_SMOOTH);
	gluQuadricTexture(obj, GL_TRUE);
	gluQuadricOrientation(obj,GLU_OUTSIDE);
	gluSphere(obj, radius, 20.0f, 20.0f);
	gluDeleteQuadric(obj);
}
void Canvas::buildGrid(char *Name)
{
int r,c;
int nStep = 5;
int size=200;
GLint iGrid_Id;


  //glPushAttrib(GL_LIGHTING_BIT);
  //glDisable(GL_LIGHTING);

  glPushMatrix();

	iGrid_Id=ReserveDisplayListId(Name);

	glNewList(iGrid_Id,GL_COMPILE_AND_EXECUTE);

	/* Just draw a bunch of horizontal and vertical lines */
	glBegin(GL_LINES);

	for(r = -size; r  <= size; r += nStep)
		{
		glVertex3f((float)r, 0.0f, (float)-size);
		glVertex3f((float)r, 0.0f, (float)size);
		}

	for(c = -size; c <= size; c += nStep)
		{
		glVertex3f((float)size, 0.0f, (float)c);
		glVertex3f((float)-size, 0.0f, (float)c);
		}

	glEnd();

	glEndList();

  glPopMatrix();
  //glPopAttrib();
  //glEnable(GL_LIGHTING);

}


Canvas cvs(0,0,640,480,"Texture Mapping");

void myPolyhedra(char *Name)
{
	GLint poly_id;
	GLfloat normal[3];
	VECTOR_STRUCT_TYPE vertex[12];
	GLfloat t;

	poly_id=cvs.ReserveDisplayListId(Name);

	if(poly_id==-1)
	{
		printf("\nNo more Display List ids available\n");
		return;
	}

	glNewList(poly_id,GL_COMPILE_AND_EXECUTE);

	cvs.setAmbientMaterialColor(0.0f,0.0f,0.5f,1.0f);
	cvs.setDiffuseMaterialColor(0.0f,0.0f,0.8f,1.0f);

	if(strcmp(Name,"Icoshedron")==0)
	{
	t=(float) sqrt((float)(5)-1)/2;

	/*Icoshedron*/
	vertex[0].m_x=0;
	vertex[0].m_y=1;
	vertex[0].m_z=t;

	vertex[1].m_x=0;
	vertex[1].m_y=1;
	vertex[1].m_z=-t;

	vertex[2].m_x=1;
	vertex[2].m_y=t;
	vertex[2].m_z=0;

	vertex[3].m_x=1;
	vertex[3].m_y=-t;
	vertex[3].m_z=0;

	vertex[4].m_x=0;
	vertex[4].m_y=-1;
	vertex[4].m_z=-t;

	vertex[5].m_x=0;
	vertex[5].m_y=-1;
	vertex[5].m_z=t;

	vertex[6].m_x=t;
	vertex[6].m_y=0;
	vertex[6].m_z=1;

	vertex[7].m_x=-t;
	vertex[7].m_y=0;
	vertex[7].m_z=1;

	vertex[8].m_x=t;
	vertex[8].m_y=0;
	vertex[8].m_z=-1;

	vertex[9].m_x=-t;
	vertex[9].m_y=0;
	vertex[9].m_z=-1;

	vertex[10].m_x=-1;
	vertex[10].m_y=t;
	vertex[10].m_z=0;

	vertex[11].m_x=-1;
	vertex[11].m_y=-t;
	vertex[11].m_z=0;



	/*Face 0 - 6,2,0*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[1],&vertex[2],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[6].m_x,vertex[6].m_y,vertex[6].m_z);
		glVertex3f(vertex[2].m_x,vertex[2].m_y,vertex[2].m_z);
		glVertex3f(vertex[0].m_x,vertex[0].m_y,vertex[0].m_z);
	glEnd();

	/*Face 1 - 2,6,3*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[3],&vertex[2],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[2].m_x,vertex[2].m_y,vertex[2].m_z);
		glVertex3f(vertex[6].m_x,vertex[6].m_y,vertex[6].m_z);
		glVertex3f(vertex[3].m_x,vertex[3].m_y,vertex[3].m_z);
	glEnd();

	/*Face 2 - 5,3,6*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[1],&vertex[3],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[5].m_x,vertex[5].m_y,vertex[5].m_z);
		glVertex3f(vertex[3].m_x,vertex[3].m_y,vertex[3].m_z);
		glVertex3f(vertex[6].m_x,vertex[6].m_y,vertex[6].m_z);
	glEnd();

	/*Face 3 - 5,6,7*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[5].m_x,vertex[5].m_y,vertex[5].m_z);
		glVertex3f(vertex[6].m_x,vertex[6].m_y,vertex[6].m_z);
		glVertex3f(vertex[7].m_x,vertex[7].m_y,vertex[7].m_z);
	glEnd();

	/*Face 4 - 0,7,6*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[0].m_x,vertex[0].m_y,vertex[0].m_z);
		glVertex3f(vertex[7].m_x,vertex[7].m_y,vertex[7].m_z);
		glVertex3f(vertex[6].m_x,vertex[6].m_y,vertex[6].m_z);
	glEnd();


	/*Face 5 - 3,8,2*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[3].m_x,vertex[3].m_y,vertex[3].m_z);
		glVertex3f(vertex[8].m_x,vertex[8].m_y,vertex[8].m_z);
		glVertex3f(vertex[2].m_x,vertex[2].m_y,vertex[2].m_z);
	glEnd();

	/*Face 6 - 2,8,1*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[2].m_x,vertex[2].m_y,vertex[2].m_z);
		glVertex3f(vertex[8].m_x,vertex[8].m_y,vertex[8].m_z);
		glVertex3f(vertex[1].m_x,vertex[1].m_y,vertex[1].m_z);
	glEnd();


	/*Face 7 - 1,0,2*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[1].m_x,vertex[1].m_y,vertex[1].m_z);
		glVertex3f(vertex[0].m_x,vertex[0].m_y,vertex[0].m_z);
		glVertex3f(vertex[2].m_x,vertex[2].m_y,vertex[2].m_z);
	glEnd();

	/*Face 8 - 0,1,10*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[0].m_x,vertex[0].m_y,vertex[0].m_z);
		glVertex3f(vertex[1].m_x,vertex[1].m_y,vertex[1].m_z);
		glVertex3f(vertex[10].m_x,vertex[10].m_y,vertex[10].m_z);
	glEnd();

		/*Face 9 - 9,10,1*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[9].m_x,vertex[9].m_y,vertex[9].m_z);
		glVertex3f(vertex[10].m_x,vertex[10].m_y,vertex[10].m_z);
		glVertex3f(vertex[1].m_x,vertex[1].m_y,vertex[1].m_z);
	glEnd();

		/*Face 10 - 9,1,8*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[9].m_x,vertex[9].m_y,vertex[9].m_z);
		glVertex3f(vertex[1].m_x,vertex[1].m_y,vertex[1].m_z);
		glVertex3f(vertex[8].m_x,vertex[8].m_y,vertex[8].m_z);
	glEnd();

		/*Face 11 - 8,3,4*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[8].m_x,vertex[8].m_y,vertex[8].m_z);
		glVertex3f(vertex[3].m_x,vertex[3].m_y,vertex[3].m_z);
		glVertex3f(vertex[4].m_x,vertex[4].m_y,vertex[4].m_z);
	glEnd();

		/*Face 12 - 3,5,4*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[3].m_x,vertex[3].m_y,vertex[3].m_z);
		glVertex3f(vertex[5].m_x,vertex[5].m_y,vertex[5].m_z);
		glVertex3f(vertex[4].m_x,vertex[4].m_y,vertex[4].m_z);
	glEnd();

			/*Face 13 - 4,5,11*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[4].m_x,vertex[4].m_y,vertex[4].m_z);
		glVertex3f(vertex[5].m_x,vertex[5].m_y,vertex[5].m_z);
		glVertex3f(vertex[11].m_x,vertex[11].m_y,vertex[11].m_z);
	glEnd();

			/*Face 14 - 7,10,11*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[7].m_x,vertex[7].m_y,vertex[7].m_z);
		glVertex3f(vertex[10].m_x,vertex[10].m_y,vertex[10].m_z);
		glVertex3f(vertex[11].m_x,vertex[11].m_y,vertex[11].m_z);
	glEnd();

	/*Face 15 - 10,7,0*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[10].m_x,vertex[10].m_y,vertex[10].m_z);
		glVertex3f(vertex[7].m_x,vertex[7].m_y,vertex[7].m_z);
		glVertex3f(vertex[0].m_x,vertex[0].m_y,vertex[0].m_z);
	glEnd();

	/*Face 16 - 11,9,4*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[11].m_x,vertex[11].m_y,vertex[11].m_z);
		glVertex3f(vertex[9].m_x,vertex[9].m_y,vertex[9].m_z);
		glVertex3f(vertex[4].m_x,vertex[4].m_y,vertex[4].m_z);
	glEnd();

	/*Face 17 - 8,4,9*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[8].m_x,vertex[8].m_y,vertex[8].m_z);
		glVertex3f(vertex[4].m_x,vertex[4].m_y,vertex[4].m_z);
		glVertex3f(vertex[9].m_x,vertex[9].m_y,vertex[9].m_z);
	glEnd();

	/*Face 18 - 7,11,5*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[7].m_x,vertex[7].m_y,vertex[7].m_z);
		glVertex3f(vertex[11].m_x,vertex[11].m_y,vertex[11].m_z);
		glVertex3f(vertex[5].m_x,vertex[5].m_y,vertex[5].m_z);
	glEnd();
	
	/*Face 19 - 10,9,11*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[10].m_x,vertex[10].m_y,vertex[10].m_z);
		glVertex3f(vertex[9].m_x,vertex[9].m_y,vertex[9].m_z);
		glVertex3f(vertex[11].m_x,vertex[11].m_y,vertex[11].m_z);
	glEnd();
	}
	else if(strcmp(Name,"Tetrahedron")==0)
	{
	/*Tetrahedron*/
	vertex[0].m_x=1;
	vertex[0].m_y=1;
	vertex[0].m_z=1;

	vertex[1].m_x=1;
	vertex[1].m_y=-1;
	vertex[1].m_z=-1;

	vertex[2].m_x=-1;
	vertex[2].m_y=-1;
	vertex[2].m_z=1;

	vertex[3].m_x=-1;
	vertex[3].m_y=1;
	vertex[3].m_z=-1;

	
	/*Face 0 - 1,2,3*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[1],&vertex[2],&vertex[3],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[1].m_x,vertex[1].m_y,vertex[1].m_z);
		glVertex3f(vertex[2].m_x,vertex[2].m_y,vertex[2].m_z);
		glVertex3f(vertex[3].m_x,vertex[3].m_y,vertex[3].m_z);
	glEnd();

	/*Face 1 - 0,3,2*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[3],&vertex[2],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[0].m_x,vertex[0].m_y,vertex[0].m_z);
		glVertex3f(vertex[3].m_x,vertex[3].m_y,vertex[3].m_z);
		glVertex3f(vertex[2].m_x,vertex[2].m_y,vertex[2].m_z);
	glEnd();

	/*Face 2 - 0,1,3*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[1],&vertex[3],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[0].m_x,vertex[0].m_y,vertex[0].m_z);
		glVertex3f(vertex[1].m_x,vertex[1].m_y,vertex[1].m_z);
		glVertex3f(vertex[3].m_x,vertex[3].m_y,vertex[3].m_z);
	glEnd();

	/*Face 3 - 0,2,1*/
	glBegin(GL_POLYGON);
		cvs.calcNormal(&vertex[0],&vertex[2],&vertex[1],normal);
		glNormal3f(normal[0],normal[1],normal[2]);
		glVertex3f(vertex[0].m_x,vertex[0].m_y,vertex[0].m_z);
		glVertex3f(vertex[2].m_x,vertex[2].m_y,vertex[2].m_z);
		glVertex3f(vertex[1].m_x,vertex[1].m_y,vertex[1].m_z);
	glEnd();

	}
	
	glEndList();
}

void myReshape(GLsizei W, GLsizei H)
{
	cvs.setViewport(0,0,W,H);
}
void drawNurb(void)
{
	GLUnurbsObj * pNurbs = NULL;
	
	GLfloat texpts[2][2][2] = 
	{
		{
			{0.0f, 0.0f},{0.0f, 1.0f}
		}, 
		{
			{1.0f, 0.0f},{1.0f, 1.0f}
		}
	};
	GLfloat KnotsA[4] = {0.0f, 0.0f, 1.0f, 1.0f};

	GLfloat ctrlPoints[4][4][3] = {
		{{-6.0f, -6.0f, 0.0f}, {-6.0f, -2.0f, 0.0f}, {-6.0f, 2.0f, 0.0f}, {-6.0f, 6.0f, 0.0f}},
		{{-2.0f, -6.0f, 0.0f}, {-2.0f, -2.0f, 8.0f}, {-2.0f, 2.0f, 8.0f}, {-2.0f, 6.0f, 0.0f}},
		{{2.0f, -6.0f, 0.0f}, {2.0f, -2.0f, 8.0f}, {2.0f, 2.0f, 8.0f}, {2.0f, 6.0f, 0.0f}},
		{{6.0f, -6.0f, 0.0f}, {6.0f, 2.0f, 0.0f}, {6.0f, 2.0f, 0.0f}, {6.0f, 6.0f, 0.0f}}};

	GLfloat KnotsB[8] = {0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 1.0f, 1.0f, 1.0f};


	pNurbs = gluNewNurbsRenderer();

	gluNurbsProperty(pNurbs, GLU_SAMPLING_TOLERANCE, 25.0f);
	gluNurbsProperty(pNurbs, GLU_DISPLAY_MODE, (GLfloat)GLU_FILL);

	gluBeginSurface(pNurbs);
		gluNurbsSurface(pNurbs, 8, KnotsB, 8, KnotsB, 4*3, 3, &ctrlPoints[0][0][0], 4,4, GL_MAP2_VERTEX_3);
		gluNurbsSurface(pNurbs, 4, KnotsA, 4, KnotsA, 2*2, 2, &texpts[0][0][0], 2,2, GL_MAP2_TEXTURE_COORD_2);
	gluEndSurface(pNurbs);

	if(pNurbs)
		gluDeleteNurbsRenderer(pNurbs);

}

void ProcessGeometry()
{
	static int spin=0.0f;

	
	glLoadIdentity();

			gluLookAt(cvs.myCamera.m_Translate_x, /*eye x*/
			  cvs.myCamera.m_Translate_y, /*eye y*/
			  cvs.myCamera.m_Translate_z, /*eye z*/
			  cvs.myCamera.m_Translate_x + (100.0f*(GLfloat)cos(cvs.myCamera.m_camera_radsFromEast)), 
			  cvs.myCamera.m_Translate_y, 
			  cvs.myCamera.m_Translate_z - (100.0f*(GLfloat)sin(cvs.myCamera.m_camera_radsFromEast)), 
			  0.0f, /*orientation x*/
			  1.0f, /*orientation y*/
			  0.0f  /*orientation z*/
			  );


	spin=(spin+5)%360;

	glCallList(cvs.FindDisplayListId("Grid"));	

	cvs.setDiffuseMaterialColor(1,1,1,1);

	glPushMatrix();

		glTranslatef(0,10,0);
		glRotatef(spin,0,1,0);

	//glCallList(cvs.FindDisplayListId("Sinc"));	

	//glCallList(cvs.FindDisplayListId("Icoshedron"));	
	//glTranslatef(5,0,0);


		glBindTexture(GL_TEXTURE_2D, cvs.FindTextureMap("Mona"));

		glPushMatrix();
			cvs.Box(5,5,5);
			glTranslatef(5,10,0);
			glRotatef(spin,0,1,0);
			cvs.TextureSphere(5);
		glPopMatrix();

		glPushMatrix();
			glTranslatef(-10,10,0);
			glRotatef(spin,1,0,0);
			cvs.TextureCylinder(5,5);
		glPopMatrix();

		glPushMatrix();
			glTranslatef(10,-10,0);
			glRotatef(spin,0,1,0);
			glRotatef(-90,1,0,0);
			drawNurb();
		glPopMatrix();

		glBindTexture(GL_TEXTURE_2D, cvs.FindTextureMap("Image1"));
		glPushMatrix();
			glTranslatef(-20,0,0);
			glRotatef(spin,0,1,0);
			glEnable(GL_TEXTURE_GEN_S);
			glEnable(GL_TEXTURE_GEN_T);
			glTexGeni(GL_S,GL_TEXTURE_GEN_MODE,GL_SPHERE_MAP);
			glTexGeni(GL_T,GL_TEXTURE_GEN_MODE,GL_SPHERE_MAP);
			glutSolidTorus(2.0,4.0,20,20);
			glDisable(GL_TEXTURE_GEN_S);
			glDisable(GL_TEXTURE_GEN_T);
		glPopMatrix();

	glPopMatrix();

	glutPostRedisplay();
	glutSwapBuffers();

	glFinish();
	//glFlush();
}

void init() {
 
  cvs.setPerspectiveWindow(60,cvs.miWidth/cvs.miHeight,5,2000);
   //cvs.setWindow(-cvs.miWidth/2,cvs.miWidth/2,-cvs.miHeight/2,cvs.miHeight/2);
	//cvs.setWindow(-5,5,-5,5);
  cvs.setViewport(0,0,cvs.miWidth,cvs.miHeight);
  cvs.setBackgroundColor(1,1,1);
  cvs.setLighting();
  //cvs.setBlending();
  cvs.setDepthBuffer();
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  //myPolyhedra("Icoshedron");
  //myPolyhedra("Tetrahedron");
  //cvs.Sinc("Sinc");

  cvs.setTextureMapping();

  cvs.ReserveTextureMap("Mona","mona.bmp");
  cvs.ReserveTextureMap("Image1","text3.bmp");
  cvs.buildGrid("Grid");

}

static void key (unsigned char k, int x, int y) {
  switch(k) {
  case 27:
    exit(0);
    break;
  case 'x':
	/* insert your function */
    break;
  }
}

void MoveCamera(GLdouble dStep)
{
	GLdouble xChange, zChange;

	xChange = dStep*cos(cvs.myCamera.m_camera_radsFromEast);
	zChange = -dStep*sin(cvs.myCamera.m_camera_radsFromEast);

	if ((cvs.myCamera.m_Translate_x<200+xChange) && (cvs.myCamera.m_Translate_x+xChange)>-200)
	{
		cvs.myCamera.m_Translate_x += (GLfloat)xChange;
	}
	if ((cvs.myCamera.m_Translate_z+zChange)<200 && (cvs.myCamera.m_Translate_z+zChange)>-200)
	{
		cvs.myCamera.m_Translate_z += (GLfloat)zChange;
	}

}
void CameraLeft(void)
{
	cvs.myCamera.m_camera_radsFromEast += (GLfloat) 10.0f/180.0f*3.142f;
	if(cvs.myCamera.m_camera_radsFromEast > (6.28f))		/* Keep in bounds */
		cvs.myCamera.m_camera_radsFromEast = 0.0f;
}

void CameraRight(void)
{
	cvs.myCamera.m_camera_radsFromEast -= (GLfloat) 10.0f/180.0f*3.142f; 
	if (cvs.myCamera.m_camera_radsFromEast <0.0f)		/* Keep in bounds */
	   cvs.myCamera.m_camera_radsFromEast = (GLfloat)(6.28);
}
void CameraForward(void)
{
	MoveCamera((double)cvs.myCamera.m_TranslationUnit);
}
void CameraBack(void)
{
	MoveCamera((GLdouble)(-1.0f*cvs.myCamera.m_TranslationUnit));
}

static void special (int k, int x, int y) {
  switch (k) {
  case GLUT_KEY_UP:
	CameraForward();
    break;
  case GLUT_KEY_DOWN:
	CameraBack();
	break;
  case GLUT_KEY_LEFT:
	CameraLeft();
	break;
  case GLUT_KEY_RIGHT:
	CameraRight();
	break;
  }
}
static void draw(GLvoid)
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	ProcessGeometry();
	glutSwapBuffers();
}

int main(int argc, char **argv) 
{
	init();
	glutKeyboardFunc(key);
	glutSpecialFunc(special);
	glutDisplayFunc(draw);
	glutReshapeFunc(myReshape);
	//glutIdleFunc(draw);
	glutMainLoop();

	return(0);
}
