/*
 *     MYSTERY.T
 *     by David M. Tuller (tulled@rpi.edu)
 * 
 *     This file is intended to assist in writing mysteries for TADS.
 *     It includes a system for keeping track of time, as well as other
 *     useful stuff. You need TADS 2.1 or later because of the use of
 *     the keywords modify and replace. This file is public domain. All 
 *     I ask is that you let me know if you find any bugs in the code. 
 *
 *     Changes needed to adv.t - The following line should be added to 
 *     adv.t in the appropriate place.
 *
 *     compoundWord 'wait' 'until' 'waituntil';
 *
 *     You need to change 'of' to 'ov' in specialWords because TADS
 *     gets confused with commands like "accuse mike of murder" otherwise.
 *     
 *     Whenever you see a call to scoreStatus in adv.t, you need to change
 *     it to scoreStatus(0) since I have altered scoreStatus in this file.
 *     This way, the call does nothing. You should also call scoreStatus
 *     in init to set up the time on the status line.
 *
 *     I have included madv.t which already has these changes for you.        
 */

/*
 *     Function declarations
 */
runEnding: function;
options: function;
timeevents: function;
intsay: function;
closePassage: function;

/*
 *     The preparse function takes the commands the user enters and finds
 *     any use of a time and stores this information in the global object
 *     as temphour and tempmin. For example, if you typed in "wait until
 *     3:00", preparse will find the time, make temphour=3 and tempmin=0.
 *     This probably will cause some problems if you only type "wait until".  
 */
preparse: function(inline)
{
        local a, b, c1, c2, d, e, f, g, l := lower(inline);
        if (find(l,':'))
        {
                a := find(l,':'); 
                b := substr(l,a-1,1);
                c1 := substr(l,a+1,1);
                c2 := substr(l,a+2,1);
                d := substr(l,a-2,1);
                f := length(l);
                if ( d = ' ' ) 
                { 
                        e := substr(l,1,a-2);
                        e += substr(l,a+3,f-a-2);
                        global.temphour := cvtnum(b);
                        global.tempmin := 10*cvtnum(c1)+cvtnum(c2);  
                }
                else 
                {
                        e := substr(l,1,a-3);
                        e += substr(l,a+3,f-a-1);
                        global.temphour := 10*cvtnum(d)+cvtnum(b);
                        global.tempmin := 10*cvtnum(c1)+cvtnum(c2); 
                }       
        }
        else e := inline;
        return e;
}

/*
 *     The following few functions are fairly standard. I have made
 *     the options function separate since it is useful to have sometimes.
 */
die: function
{
    "\b*** You have died ***\b";
    scoreRank();
    options();
}

options: function
{
    "\bYou may restore a saved game, start over, quit, or undo
    the current command.\n";
    while ( 1 )
    {
        local resp;

        "\nPlease enter RESTORE, RESTART, QUIT, or UNDO: >";
        resp := upper(input());
        if ( resp = 'RESTORE' )
        {
            resp := askfile( 'File to restore' );
            if ( resp = nil ) "Restore failed. ";
            else if ( restore( resp )) "Restore failed. ";
            else
            {
                Me.location.lookAround(true);
                setscore( 0 );
                abort;
            }
        }
        else if ( resp = 'RESTART' )
        {
            restart();
        }
        else if ( resp = 'QUIT' )
        {
            terminate();
            quit();
            abort;
        }
        else if (resp = 'UNDO')
        {
            if (undo())
            {
                "(Undoing one command)\b";
                Me.location.lookAround(true);
                scoreStatus( 0 );
                abort;
            }
            else
                "Sorry, no undo information is available. ";
        }
    }
}

preinit: function
{
    local o;

    global.lamplist := [];
    o := firstobj();
    while( o <> nil )
    {
        if ( o.islamp ) global.lamplist := global.lamplist + o;
        o := nextobj( o );
    }
    initSearch();
}

terminate: function
{
}

pardon: function
{
    "I beg your pardon? ";
}

goToSleep: function
{
        "You can't sleep now. You have investigating to do. ";
}

numObj: basicNumObj  // use default definition from adv.t
;

strObj: basicStrObj     // use default definition from adv.t
;

Me: basicMe
;

darkTravel: function
{
    "You stumble around in the dark, and don't get anywhere. ";
}

/*
 *     The following are a few verbs that you might want to use in your
 *     mystery. Nothing fancy here.
 */
accuseVerb: deepverb
        verb = 'accuse' 'charge'
        sdesc = "accuse"
        ioAction( ofPrep ) = 'ChargeWith'
        ioAction( withPrep ) = 'ChargeWith'
        prepDefault = ofPrep
;

analyzeVerb: deepverb
        verb = 'analyze' 'check' 'analise'
        sdesc = "analyze"
        ioAction( forPrep ) = 'CheckFor'
        prepDefault = forPrep
;

arrestVerb: deepverb
        verb = 'arrest'
        sdesc = "arrest"
        doAction = 'Arrest'
;

printVerb: deepverb
        verb = 'print' 'fingerprint' 'dust'
        sdesc = "fingerprint"
        doAction = 'Print'
;

ofPrep: Prep
        preposition = 'of'
        sdesc = "of"
;

forPrep: Prep
        preposition = 'for'
        sdesc = "for"
;

/*
 *      This should be used when you want a piece of text to interrupt
 *      the waituntil command. If you pass nil as the argument, it 
 *      interrupts waituntil without asking if you want to stop waiting.
 */
intsay: function( line )
{
        global.interrupt := true;
        if (line = nil) 
        {       
                global.dontask := true;
                return;
        }
        "\b";
        say(line);
}

/*
 *     This is the implementation of "wait until". All it does is run the
 *     daemons and fuses until the time specified by global.temphour and
 *     global.tempmin. The only exception is when intsay is called. Then
 *     the player will be asked if he/she would like to stop waiting.
 */
waitUntilVerb: sysverb
        verb = 'waituntil'
        action(actor) = 
        {
                local m, h;
                m := global.tempmin - 1;
                h := global.temphour;
                if ( m < 0 )
                {
                        m := m + 60;
                        h := h - 1;
                        if ( h < 1 )
                        {
                                h := h + 12;
                        }                       
                } 
                while ( not ( h = global.hour ) or
                        not ( m = global.minute ) )
                {
                        global.interrupt := nil;
                        global.dontask := nil;
                        rundaemons();
                        runfuses();
                        if (global.dontask)
                        {
                                return nil;
                        }
                        if (global.interrupt)
                        {
                                "\bDo you want to stop waiting?\ <Y/N> ";
                                if (yorn())
                                {
                                        return nil;
                                }
                        }

                }
        }
;

/*
 *     This command displays the current time. It should already appear
 *     on the status line, but if you don't have a status line, you can
 *     use this instead.
 */
timeVerb: sysverb
        verb = 'time' 't'
        action(actor) = 
        {
                "The time is now "; say(global.hour); ":";
                if ( global.minute > 9 ) say(global.minute);
                else { "0"; say(global.minute); }
                if ( global.isAM ) " a.m.\ on ";
                else " p.m.\ on ";
                say( global.day ); ".";
                abort;
        }
;

/*
 *     This is a modification of the scoreRank function so it displays the
 *     time as well as your score.
 */
scoreRank: function
{
        "By "; say( global.hour ); ":"; 
        if ( global.minute > 9 ) say( global.minute );
        else { "0"; say( global.minute ); }
        if ( global.isAM ) " a.m.\ on ";
        else " p.m.\ on ";
        say( global.day ); 
        ",\ you have achieved a score of ";
        say( global.score ); " points out of a possible ";
        say( global.maxscore ); ".\n";
}

global: object
    turnsofar = 0                            // no turns have transpired so far
    score = 0                            // no points have been accumulated yet
    maxscore = 100                                    // maximum possible score
    verbose = true                          // we are currently in VERBOSE mode
    lamplist = []              // list of all known light providers in the game
    printlist = []            // list of all people who have been fingerprinted
    hour = 10                                               // the current hour
    minute = 30                                           // the current minute
    isAM = true                         // it is the afternoon, not the morning
    day = 'Monday'                                       // the day of the week
    temphour = 0                                        // hour to stop waiting
    tempmin = 0                                       // minute to stop waiting                  
    awakeTime = 1100
    sleepTime = 1440
    endhour = 12                                     // when to stop the game
    endmin = 0
    endday = 'Monday'
    endisAM = nil
    interrupt = nil
    dontask = nil
;

/*
 *     This function displays the ending to the mystery. The input is
 *     the person to be arrested and whether or not you have enough to
 *     arrest the person. It is called whenever you arrest someone.
 */
runEnding: function( actor, canArrest )
{
        scoreRank();
        quit();
}

/*
 *     This is my variation of the actor to give default responses to  
 *     the new verbs defined above. Instead of a sdesc, use the name 
 *     characteristic to name the actor. It also has a characteristic for
 *     whether or not the character is dead. Unfortunately, you need to
 *     have an alive actor and a dead actor for each one that can be killed
 *     because TADS does not allow for adjectives that vary with time e.g.
 *     you cannot add 'dead' as an adjective after the actor died without
 *     replacing the actors altogether.   
 */
class myActor: Actor
        adesc = { sdesc; }
        thedesc = { sdesc; }
        isalive = true
        name = ""
        sdesc = { name; }
        ignoremsg = 
        {
                caps(); name; " just ignores you. ";
        }
/*
 *      The following lists are there to keep track of what clues have
 *      not been shown to the actor. When all clues have been shown, the
 *      actor will confess if accused.
 */
        murderlist = []                        
        bmaillist = []
        adulterylist = []
        fraudlist = []
        theftlist = []
        gamblinglist = []
/*
 *      The following are the confessions that the actor will give.
 */
        murderConf = nil
        bmailConf = nil
        adulteryConf = nil
        fraudConf = nil
        theftConf = nil
        gamblingConf = nil
        useasclue = nil
        canArrest = nil
        verDoPrint( actor ) = { return true; }
        doPrint( actor ) = 
        {
                if (self.isalive)
                {
                        self.name; " does not want to be fingerprinted. ";
                }
                else 
                {
                        local l;
                        l := [];
                        l += self;
                        if (not (intersect(global.printlist,l) = nil))
                                global.printlist += self;
                        "%You% fingerprint "; self.name; ". ";
                }
        }
        verDoArrest( actor ) = { return true; }
        doArrest( actor ) = 
        {
                if (self.isalive) 
                {
                        "%You% arrest "; name; " for murder. ";
                        self.moveInto(nil);
                        runEnding(self, canArrest);
                }
                else
                {
                        "%You% can't arrest a dead body. ";
                }
        } 
        verDoCheckFor( actor, iobj ) = { return true; } 
        doCheckFor( actor, iobj ) = 
        {
                if (self.isalive)
                {
                        self.name; " refuses to let you do that. ";
                }
                else 
                {
                        "%You% can't do that. ";
                }       
        } 
        verIoCheckFor( actor ) = { return true; }
        ioCheckFor( actor, dobj ) = 
        {
                "Don't be ridiculous. ";
        }
        verDoChargeWith( actor, iobj ) = { return true; }
        doChargeWith( actor, iobj ) = 
        {
                if (self.isalive)
                {
                        if ((iobj = murder) and (length(murderlist) = 0))
                        { 
                                if (murderConf = nil)
                                {
                                        ignoremsg;
                                }
                                else
                                { 
                                        murderConf := nil;
                                        canArrest := true;
                                }
                        }
                        else if ((iobj = blackmail) and (length(bmaillist) = 0))
                        {
                                if (bmailConf = nil )
                                {
                                        ignoremsg;
                                }
                                else 
                                { 
                                        bmailConf := nil; 
                                }
                        }
                        else if ((iobj = adultery) and (length(adulterylist) = 0))
                        {
                                if (adulteryConf = nil)
                                {
                                        ignoremsg;
                                }
                                else
                                { 
                                        adulteryConf := nil; 
                                }
                        }
                        else if ((iobj = theft) and (length(theftlist) = 0))
                        {
                                if (theftConf = nil)
                                {
                                        ignoremsg;
                                }
                                else
                                { 
                                        theftConf := nil; 
                                }
                        }
                        else if ((iobj = gambling) and (length(gamblinglist) = 0))
                        {
                                if (gamblingConf = nil)
                                {
                                        ignoremsg;
                                }
                                else
                                { 
                                        gamblingConf := nil; 
                                }
                        }
                        else if ((iobj = fraud) and (length(fraudlist) = 0))
                        {
                                if (fraudConf = nil)
                                {
                                        ignoremsg;
                                }
                                else
                                { 
                                        fraudConf := nil; 
                                }
                        }
                        else 
                        {
                                self.name; " looks at %you% and says, \"%You% don't have
                                enough evidence to accuse me of anything.\"";
                        }
                }
                else
                {
                        "Did %you% think a dead body would be able to respond?";
                } 
        }
        verDoAttackWith( actor, iobj ) = 
        {
                if (self.isalive)
                { 
                        "You're supposed to be solving the crimes, not 
                        committing them."; 
                }
                else 
                {
                        "%You% can't kill someone who's already dead.";
                }
        }
        doAttackWith( actor, iobj ) = 
        {
                return nil;
        }
        verDoAskAbout( actor, iobj ) = {}
        doAskAbout( actor, iobj ) = 
        {
                if (self.isalive)
                {
                        self.name; " laughs and says \"Why should I tell you
                        anything?\"";
                }
                else
                {
                        "That's pointless, isn't it?";
                } 
        }
        verIoShowTo( actor ) = 
        { 
                if (self.isalive) return true; 
                else 
                {       
                        "What's the point? ";
                }
        }
        ioShowTo( actor, dobj ) = 
        { 
                local l;
                l := [];
                l += self;
                if (dobj.useasclue)
                {
                        if (dobj.shown)
                        {
                                "%You% have already shown that to ";
                                self.name; ". ";
                                return(nil);
                        }
                        else if ((intersect(l,dobj.prints)=[]) or 
                                not (intersect(l,dobj.knownprints)=[]))
                        {
                                caps(); self.name; " looks at "; 
                                dobj.thedesc; " guiltily. ";
                                self.murderlist -= dobj;                        
                                self.bmaillist -= dobj;         
                                self.adulterylist -= dobj;
                                self.fraudlist -= dobj;
                                self.theftlist -= dobj;
                                self.gamblinglist -= dobj;
                                dobj.shown := true;
                        }
                        else
                        {
                                caps(); self.name; " looks at "; 
                                dobj.thedesc; ". ";
                        }
                }
                else
                {
                        caps(); self.name; " looks at "; dobj.thedesc; ". ";   
                }
        } 
        verDoTellAbout( actor, io ) = 
        { 
                if (self.isalive) return true; 
                else 
                {
                        "What's the point? ";        
                }
        }
        doTellAbout( actor, io ) =
        {
                local l;
                l := [];
                l += self;
                if (io.useasclue)
                {
                        if (io.shown)
                        {
                                caps(); self.name; 
                                " already knows about that. ";
                                return(nil);
                        }
                        else if ((intersect(l,io.prints)=[]) or 
                                not (intersect(l,io.knownprints)=[]))
                        {
                                caps(); self.name; 
                                " looks at %you% guiltily. ";
                                self.murderlist -= io;                        
                                self.bmaillist -= io;         
                                self.adulterylist -= io;
                                self.fraudlist -= io;
                                self.theftlist -= io;
                                self.gamblinglist -= io;
                                io.shown := true;
                        }
                        else
                        {
                                caps(); self.name; 
                                " listens to %you%. ";
                        }
                }
                else
                {
                        caps(); self.name; 
                        " listens to %you%. ";   
                }
        }
;

/*
 *     This is the class to use for all your dead bodies.
 */
class mydeadActor: searchHider, myActor
        isalive = nil
        autoTake = nil
        sdesc = 
        {
                name;
                "'s body";
        }
        verGrab( item ) = { }
;

class mActor: myActor
        isHim = true
;

class fActor: myActor
        isHer = true
;

class mdeadActor: mydeadActor, mActor
;

class fdeadActor: mydeadActor, fActor
;
 
/*
 *     Modify the item class to allow for default answers to the new verbs.
 */
modify item
        prints = []
        knownprints = []
        printscore = 0
        shown = nil
        useasclue = true
        verDoPrint( actor ) = { return true; }
        doPrint( actor ) = 
        {
                if ( length(self.prints) = 0 )
                {
                        "%You% can't find any clear prints. ";
                }
                else if ( intersect(self.prints,global.printlist) = [] )
                {
                        "%You% find some prints but cannot tell
                        whose they are. ";
                }
                else 
                {
                        local l := intersect(self.prints,global.printlist);
                        local ind;
                        local len := length(l);
                        local len2 := length(self.prints);
                        "On "; self.thedesc;
                        ",\ %you% find prints from:\n";
                        for(ind := 1; ind <= len; ind++)
                        {
                                "\t";
                                if (isclass(l[ind],myActor))
                                {
                                        local m := [];
                                        m += l[ind]; 
                                        l[ind].name;
                                        if (intersect(m,self.knownprints) = [])
                                        {
                                                knownprints += l[ind];
                                                incscore(printscore);
                                        }
                                }
                                else l[ind].sdesc;
                                "\n";
                        }
                        if (len2 > len)
                        {
                                "\bUnfortunately, some prints cannot be 
                                identified yet. ";
                        }
                }
        }
        verDoCheckFor( actor, dobj ) = { return true; }
        doCheckFor( actor, dobj ) = 
        {
                "%You% can't check "; self.thedesc; " for "; dobj.sdesc;
        } 
        verDoShowTo( actor, io ) = { return true; }
        doShowTo( actor, io ) = 
        { 
                if (isclass(io,myActor))
                {
                        io.ioShowTo( actor, self );
                }
                else pass doShowTo;
        } 
;

/*
 *      This is used for all hiddenItems which should give the player
 *      points when he/she takes it.
 */
hiddenClue: hiddenItem, item
        findpts = 0
        taken = nil
        autoTake = nil
        verDoTake( actor ) = {}
        doTake( actor ) =
        {
                if (not taken)
                {
                        incscore(findpts);
                        taken := true;
                }
                pass doTake;
        }
;

/*
 *      This is a class for crimes. You can't do much with them except
 *      accuse people of them or ask people about them. The crimes have to
 *      be a floatingItem so you can use them everywhere.
 */
class crime: floatingItem, item
        ldesc = "It is impossible to look at abstractions. "
        sdesc = "crime"
        location = 
        { 
                return( Me.location ); 
        }
        locationOK = true
        adesc = { sdesc; }
        thedesc = { sdesc; }
        verIoChargeWith( actor ) = { return true; }
        ioChargeWith( actor, dobj ) = { dobj.doChargeWith( actor, self ); }
        verIoAskAbout( actor ) = {}
        ioAskAbout( actor, dobj ) =
        {
                dobj.doAskAbout( actor, self ); 
        }
        dobjGen(actor, verb, iobj, prep) = 
        {
                "Crimes are not physical objects so you can't do that.";
                exit;
        }
        iobjGen(actor, verb, dobj, prep) = 
        {
                self.dobjGen(actor, verb, dobj, prep);
        }
;

/*
 *      Some common crimes that you might want to use.
 */
murder: crime
        noun = 'murder' 'killing'
        sdesc = "murder" 
;

blackmail: crime
        noun = 'blackmail' 'mail'
        adjective = 'black'
        sdesc = "blackmail"
;

adultery: crime
        noun = 'adultery' 'affair'
        adjective = 'having' 'an'
        sdesc = "adultery"
;

gambling: crime
        noun = 'gambling' 'betting' 
        sdesc = "gambling"
;

fraud: crime
        noun = 'fraud'
        sdesc = "fraud"
;

theft: crime
        noun = 'theft' 'stealing' 'robbery'
        sdesc = "theft"
;

/*
 *      Used for rooms which have access to a secret door. The only
 *      difference is that it displays a special message if the door
 *      is open.
 */
passageRoom: room
        secretdoor = nil
        roomdesc = ""
        ldesc = 
        {
                roomdesc;
                if (secretdoor.isopen)
                {
                        secretdoor.opendesc;
                }
        }
;

/*
 *      The secret door. It is originally closed and when you want to
 *      open it, you must call opendoor which opens the door and then
 *      sets up a fuse that will close the door automatically.  
 */
secretDoor: doorway, floatingItem
        isopen = nil
        timeopen = 5
        sdesc = "secret passage"
        ldesc = 
        {
                if (self.isopen) 
                {
                        opendesc;
                }
                else 
                {
                        "I don't see any <<self.sdesc>> here. ";
                }
        }
        opendesc = "%You% can see a secret passage. "
        openmsg = "A secret passage appears. "
        closemsg = "The secret passage closes. "
        opendoor = 
        {
                if (not self.isopen)
                {
                        self.isopen := true;
                        self.otherside.isopen := true;
                        if (Me.location = self.location)
                        {
                                "\n";
                                self.openmsg;
                        } 
                        else if (Me.location = self.otherside.location)
                        { 
                                "\n";
                                self.otherside.openmsg;
                        }
                        setfuse(closePassage,timeopen,self); 
                }     
                else
                {
                        "Nothing happens. ";
                }  
        }
;

/*
 *      This is called as a fuse to shut the secret doorways.
 */
closePassage: function( d )
{
        if (Me.location = d.location)
        {
                "\n";
                d.closemsg;
        } 
        else if (Me.location = d.otherside.location)
        { 
                "\n";
                d.otherside.closemsg;
        }
        d.isopen := nil;
        d.otherside.isopen := nil;
}

/*
 *      This triggers the secret door, but could be used for other 
 *      things by changing the trigger property.
 */
triggerItem: item
        triggerobj = nil
        trigger = 
        {
                self.triggerobj.opendoor;
        }
;

/*
 *     This is the same as the original except it calls scoreStatus to 
 *     change the time as well as the number of turns.     
 */
replace turncount: function( parm )
{
    incturn();
    global.turnsofar := global.turnsofar + 1;
    scoreStatus( 1 );
}

/*
 *      This function is evaluated once each time scoreStatus changes.
 *      You can add code here to do anything at a given time, whether 
 *      you are waiting or not. 
 */
timeevents: function
{
}

/* 
 *     This function is used to change the time. Each turn, you need to
 *     call this at least once to change the time. The timeinc is always
 *     in minutes. If you need anything to change depending on the time 
 *     spent you need to either make it a daemon or alter this function to
 *     change it whenever it is called. For example, awakeTime is changed
 *     accordingly here. Warning: It isn't safe to call scoreStatus with
 *     anything but 0 or 1 as an argument. Otherwise, it will mess up the
 *     daemons and fuses.
 */
replace scoreStatus: function( timeinc )
{
        local amorpm, tmp;
        
        global.minute := global.minute + timeinc;
        global.awakeTime := global.awakeTime + timeinc; 
        if ( global.minute > 59 ) 
        {
                global.minute := global.minute - 60;
                global.hour := global.hour + 1;
                if ( global.hour = 12 )
                if ( global.isAM ) 
                        {
                                global.isAM := nil;
                        }
                        else 
                        {
                                global.isAM := true;
                                if ( global.day = 'Friday' )
                                        global.day := 'Saturday';
                                else if ( global.day = 'Saturday' )
                                        global.day := 'Sunday';
                                else if ( global.day = 'Sunday' )
                                        global.day := 'Monday';
                                else if ( global.day = 'Monday' )
                                        global.day := 'Tuesday';
                                else if ( global.day = 'Tuesday' )
                                        global.day := 'Wednesday';
                                else if ( global.day = 'Wednesday' )
                                        global.day := 'Thursday';
                                else if ( global.day = 'Thursday' )
                                        global.day := 'Friday';
                        }
                if ( global.hour > 12)
                {
                        global.hour := global.hour - 12;
                }
        }
        if ( global.isAM = true ) amorpm := 'a.m.';
        else amorpm := 'p.m.';
        if (global.minute > 9)
        setscore(global.day + ' ' + cvtstr(global.hour) + ':' + 
                cvtstr(global.minute) + ' ' + amorpm);     
        else
        setscore(global.day + ' ' + cvtstr(global.hour) + ':0' + 
                cvtstr(global.minute) + ' ' + amorpm);   

        timeevents();

        /*
         *      This checks to see if you have reached the end of a 
         *      time limit as prescribed by the global.end* properties.
         *      The game then calls runEnding which should terminate
         *      the game. 
         */
        if ((global.day = global.endday) and 
            (global.minute = global.endmin) and
            (global.hour = global.endhour) and
            (global.isAM = global.endisAM))
        {
                runEnding(nil,nil);
        } 
}

