/* interp.c
 *
 *  foblub -- a Z-machine for TI calculators
 *  based on pinfocom by InfoTaskForce and Paul Smith
 *  Ported and extended by Nils Gesbert, 2003
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; see the file COPYING.  If not, write to the
 *  Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "infocom.h"
#ifndef VERSION_CINQ
#define call_1n(x)
#define catch()
#endif

static void
execute A3(word, opcode, word, num, word *, param)
{
    switch (opcode)
    {
        case 0x01 : compare(num, param);
                    break;
        case 0x02 : LTE(param[0], param[1]);
                    break;
        case 0x03 : GTE(param[0], param[1]);
                    break;
        case 0x04 : dec_chk(param[0], param[1]);
                    break;
        case 0x05 : inc_chk(param[0], param[1]);
                    break;
        case 0x06 : check_loc(param[0], param[1]);
                    break;
        case 0x07 : bit(param[0], param[1]);
                    break;
        case 0x08 : or(param[0], param[1]);
                    break;
        case 0x09 : and(param[0], param[1]);
                    break;
        case 0x0a : test_attr(param[0], param[1]);
                    break;
        case 0x0b : set_attr(param[0], param[1]);
                    break;
        case 0x0c : clr_attr(param[0], param[1]);
                    break;
        case 0x0d : put_var(param[0], param[1]);
                    break;
        case 0x0e : transfer(param[0], param[1]);
                    break;
        case 0x0f : load_word_array(param[0], param[1]);
                    break;
        case 0x10 : load_byte_array(param[0], param[1]);
                    break;
        case 0x11 : get_prop(param[0], param[1]);
                    break;
        case 0x12 : get_prop_addr(param[0], param[1]);
                    break;
        case 0x13 : next_prop(param[0], param[1]);
                    break;
        case 0x14 : plus(param[0], param[1]);
                    break;
        case 0x15 : minus(param[0], param[1]);
                    break;
        case 0x16 : multiply(param[0], param[1]);
                    break;
        case 0x17 : divide(param[0], param[1]);
                    break;
        case 0x18 : mod(param[0], param[1]);
                    break;
#ifdef VERSION_QUATRE
        case 0x19 : if (STANDARD) goto illegal; /* version  2 oprandes de gosub */
#endif
        /*
         * Missing opcodes
         */

        case 0x20 : gosub(num, param);
                    break;
        case 0x21 : save_word_array(param[0], param[1], param[2]);
                    break;
        case 0x22 : save_byte_array(param[0], param[1], param[2]);
                    break;
        case 0x23 : put_prop(param[0], param[1], param[2]);
                    break;
        case 0x24 : input(param[0], param[1]);
                    break;
        case 0x25 : print_char(param[0]);
                    break;
        case 0x26 : print_num(param[0]);
                    break;
        case 0x27 : pi_random(param[0]);
                    break;
        case 0x28 : push(param[0]);
                    break;
        case 0x29 : pop(param[0]);
                    break;
        case 0x2a : scr_window(param[0]);
                    break;
        case 0x2b : scr_set_win(param[0]);
                    break;

		    /* Ici des opcodes V4...*/

        case 0x33 : output_stream(param);
	            break;
        case 0x34 : /* input_stream(param[0]); */
	            break;
        case 0x35 : scr_putsound(param[0], param[1], param[2], num);
                    break;

        default :

#ifdef VERSION_QUATRE
  /* 0x2c : forme de gosub avec jusqu' 8 arguments */
      if (ENHANCED) switch (opcode) {

        case 0x2d : erase_window(param[0]);
                    return;
        case 0x2e : erase_line(param[0]);
                    return;
        case 0x2f : set_cursor(param[0], param[1]);
	            return;
        case 0x30 : get_cursor(param[0]);
                    return;
        case 0x31 : scr_font_style(param[0]);
                    return;
        case 0x32 : /* buffer_mode(param[0]); */
                    return;

		    /* Ici des opcodes V3... */

        case 0x36 : read_char(num, param);
                    return;
        case 0x37 : scan_table(num, param);
                    return;
      }
#ifdef VERSION_CINQ
      if (ADVANCED) switch (opcode) {

        case 0x1a : call_2n(param[0], param[1]);
                    return;
        case 0x1b : set_colour(param[0], param[1]);
                    return;
        case 0x1c : throw(param[0], param[1]);
                    return;


        case 0x38 : not(param[0]);
                    return;
        case 0x39 : call_vn(num, param);
	            return;
        case 0x3a : call_vn2(num, param);
	            return;
        case 0x3b : tokenise(param[0], param[1], param[2], param[3]);
	            return;
        case 0x3c : encode_text(param[0], param[1], param[2], param[3]);
	            return;
        case 0x3d : copy_table(param[0], param[1], param[2]);
	            return;
        case 0x3e : print_table(num, param);
	            return;
        case 0x3f : check_arg_count(param[0], num);
	            return;
      }
#endif /* VERSION_CINQ */
#endif /* VERSION_QUATRE */
      illegal : error("execute: Invalid Z-code instruction: $%02x", opcode);
    }
}

static inline void
oper1 A1(word, opcode)
{
    word    param1;

    param1 = load((opcode >> 4) & 0x03);
    switch (opcode & 0x0F)
    {
        case 0x00 : cp_zero(param1);
                    break;
        case 0x01 : get_link(param1);
                    break;
        case 0x02 : get_holds(param1);
                    break;
        case 0x03 : get_loc(param1);
                    break;
        case 0x04 : get_prop_len(param1);
                    break;
        case 0x05 : inc_var(param1);
                    break;
        case 0x06 : dec_var(param1);
                    break;
        case 0x07 : print1(param1);
                    break;
#ifdef VERSION_QUATRE
        case 0x08 : if (ENHANCED) gosub(1, &param1); else goto illegal;
	            break;    
#endif
        case 0x09 : cut_obj(param1);
                    break;
        case 0x0a : p_obj(param1);
                    break;
        case 0x0b : rtn(param1);
                    break;
        case 0x0c : jump(param1);
                    break;
        case 0x0d : print2(param1);
                    break;
        case 0x0e : get_var(param1);
                    break;
        case 0x0f : if (ADVANCED) call_1n(param1); else not(param1);
                    break;
        default   : 
	illegal   :   error("oper1: Invalid Z-code instruction: $%02x", opcode);
    }
}

static inline void
oper2 A1(word, opcode)
{
    word    param[2];
    int     mode;

    mode = 1 + ((opcode & 0x40) != 0);
    param[0] = load(mode);

    mode = 1 + ((opcode & 0x20) != 0);
    param[1] = load(mode);

    execute((opcode & 0x1F), 2, param);
}

static inline void
oper3 A1(word, opcode)
{
  word    *pp;
  word    num_params;
  int     addr_mode;
  int     i;
  
  if (ENHANCED && (opcode == 0xEC)) { /* gosub  8 arguments */
    word    param[8];
    word    modes = next_word();
    
    num_params = 0;
    
    for (i=14, pp=param; i >= 0; ++pp, i-=2)
      {
        addr_mode = (modes >> i) & 0x03;
        if (addr_mode == 3)
	  break;
	
        ++num_params;
        *pp = load(addr_mode);
      }
    gosub (num_params, param);
  }

  else {
    word    param[4];
    byte    modes;
    
    NEXT_BYTE(modes);
    num_params = 0;
    
    for (i=6, pp=param; i >= 0; ++pp, i-=2)
      {
        addr_mode = (modes >> i) & 0x03;
        if (addr_mode == 3)
	  break;
	
        ++num_params;
        *pp = load(addr_mode);
      }

    execute((opcode & 0x3F), num_params, param);
  }
}

__attribute__ ((noreturn)) inline void
interp()
{
    word            opcode;

    while (TRUE)
    {
        NEXT_BYTE(opcode);
#ifdef DEBUG4
	if (tracing) {
	  char buf[50];
	  sprintf (buf, "opcode : 0x%02x", opcode);
	  scr_putline (buf);
	}
#endif
        if (opcode < 0x80)
        {
	    oper2(opcode);
            continue;
        }
        if (opcode < 0xb0)
        {
	    oper1(opcode);
            continue;
        }
        if (opcode >= 0xc0)
        {
            oper3(opcode);
            continue;
        }

        switch (opcode & 0x0F)
        {
            case 0x00 : rtn(1);
                        break;
            case 0x01 : rtn(0);
                        break;
            case 0x02 : wrt();
                        break;
            case 0x03 : writeln();
                        break;
            case 0x04 : null();
                        break;
       	    case 0x05 : if (ADVANCED) goto illegal;
	                save();
                        break;
            case 0x06 : if (ADVANCED) goto illegal;
	                restore();
                        break;
            case 0x07 : restart();
                        break;
            case 0x08 : rts();
                        break;
            case 0x09 : if (ADVANCED) catch(); else pop_stack();
                        break;
            case 0x0a : quit();
                        break;
            case 0x0b : new_line();
                        break;
	    case 0x0c : if (ENHANCED) goto illegal;
	                set_score();
                        scr_putscore();
                        break;
            case 0x0d : verify();
                        break;
#ifdef VERSION_CINQ
	    case 0x0e : if (ADVANCED) {
	                  extended_opcode();
	                  break;
	                }
	    case 0x0f : if (ADVANCED) {
	                  piracy();
	                  break;
	                }
#endif
            default :
	    illegal : error("interp: Invalid Z-code instruction: $%02x",opcode);
        }
    }
}
