#define INFO "\
zorkword -- an Infocom(tm) game vocabulary dumper\n\
Copyleft (c) 1991 by Mike Threepoint. All rights reversed.\n\
Release 10 / Serial number 970824\n\
\n\
Display vocabulary lists from Infocom(tm) adventure game data files.\n\
\n\
Usage:  %s [-1] [-w] [-#] [-f] file...\n\
\n\
\t-1   one-column list\n\
\t-w   wide multiple-column list (default)\n\
\t-#   number vocabulary words\n\
\t-f   print vocabulary flags\n\
\n"

/* In 1992, there were no Z-machines greater than 6, and with
 * Infocom dead, no reason to expect any. Over the last 5 years,
 * Graham Nelson's outstanding Inform compiler has completely
 * changed Z-machineology, and introducted the v7 and z8 game.
 * Accordingly, I have relaxed the sanity check to allow greater
 * range in Z-machine versions.
 *
 * Zorkword is rather obsolete now. More versatile programs can
 * do the same task. The Z-machine specification supercedes the
 * comments in the source. But since it's still preserved for
 * posterity on the if-archive (as a historical artifact?), I
 * feel it is my duty to make the one-line change that will
 * enable it to function with the new breed of Z-machine games.
 *
 * Changes in release 10 (970824):
 *      Dump the vocabulary of Inform-made v7 and v8 Z-machine files.
 *	Add new info about internal structures.-
 *      Try adding .z3-.z8 extensions if a file name is not found.
 *	Bring my contact info up to date.
 *
 * Changes in release 9 (920920):
 *	Added new info about internal structures.
 *      In show-flags mode, enclose in brackets any vocabulary words without
 *      the end-of-string bit set, since they can't really be used.
 *
 * Changes in release 8:
 *	Add a sanity check to make sure a file is really a data file, before
 *	trying to frob it and usually crashing.
 *      Try adding .data extension if a file name is not found.
 *      Revise some comments.
 *
 * Changes in release 7:
 *	Add comments about the file format.
 *      Correct my mistaken assumption that every entry in the vocabulary
 *      table has 3 data bytes -- Sherlock has only 2, and Shogun has 4.
 *      Add support for non-standard text compression alphabets in the
 *      data file, which Shogun uses.
 *      Make -# and -f ordinary switches instead of toggles.
 *      Rename program to something more unique than "vocab".
 *      Show version number of pix data files.
 *
 * Disclaimer:	This program works for me, at the moment. I think it will work
 *		for you. It's portable ANSI C, and should work whether your
 *		machine is big-endian or little-endian.
 *
 * Warning:	Using this program on games you haven't solved can result in
 *		giving away passwords, spoiling surprises, and just maybe
 *		relieving the frustration of Dave Lebling's "what verb does
 *		this stupid game want me to use?" puzzles.
 *
 * Questions? Comments? Send Internet e-mail to lionheart@mad.scientist.com.
 *
 * For compilers, interpreters, specifications, and more Z-machine games
 * than you can wave a scepter at, go to the interactive fiction archive
 * at ftp.gmd.de.
 *
 * This code should be portable ANSI C, and work with data files for any
 * machine, big- or little-endian. For the record, I've compiled this under
 * Borland C++ in MS-DOS 5.0, and run it on every game produced by Infocom,
 * and several that weren't.
 */
/* Disclaimer:
 *
 * I'm not an employee of Infocom or any of the companies that bought its name.
 * Nobody within Infocom told me any technical details, even when I asked.
 * All the information here comes from painstaking reverse engineering by
 * dedicated hackers. Activision's not complaining. They're even writing
 * an Inform game to include as a bonus with Zork: Grand Inquisitor!
 *
 * Alas, Infocom. I wore a black armband when you went under. It was my dream
 * to write adventure games with you.
 *
 * (Boring saga: The closest I came was maybe writing one function for them.
 *  I hated using the SETUP.EXE program, which asks you your screen size and
 *  saves it to a 3-byte data file, when I changed my VGA into 50-line mode.
 *  I noticed the Beyond Zork MS-DOS interpreter was in MSC, so I mailed them
 *  a MSC function to get the screen size from the BIOS automatically, and
 *  released the simple function to them.
 *
 *  Some time later, a new text game was released, with VisiClues. I started
 *  it in 50-line mode, but the screen was reset to 25-line CGA color mode...
 *  and the text ran off the bottom of the screen. The screen didn't scroll
 *  until the text reached virtual line 50. Apparently, in the new display
 *  code, the mode change came after the screen length setting. Oops.
 *
 *  Too late to fix it now. Oh well, it'd've been great to tell them in a job
 *  interview that I'd already debugged one of their games. Sigh.)
 */
/* Some things to use this program for:
 *    read every last Encyclopedia Frobizzica entry,
 *    be sure you've learned every spell in the Enchanter series,
 *    know all the Wizard of Frobozz's wand's F-words,
 *    discover obscure synonyms (like Glamdring for the Zorkian elvish sword)
 *    learn trivia about internal operations (like the `intnum' noun, used
 *	 when you type a number)
 *    play with debugging commands in some games (Stu Galley's especially)
 */

#define cols()		79
#define BUFSIZE 	512

/* #define DEBUG */

#if !defined(__STDC__) && !defined(__GNUC__) && !defined(__TURBOC__) && !defined(LINT)
#error This is an ANSI-complaint.  It looks like you are not ANSI-compliant.
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h> /* if you don't got it, you probably ain't ANSI-standard */
#include <string.h>
#include <ctype.h>

#ifndef SEEK_SET
#define SEEK_SET	0
#endif

#define S_BLANK 	1
#define S_MACRO1	2
#define S_MACRO2	3
#define S_MACRO3	4
#define S_CAPS		5
#define S_PUNC		6
#define S_FILLER	6
#define S_OFF		7

/* The text compression scheme uses 2 bytes to hold 3 5-bit codes and an
 * end-of-text flag bit.  Each 5-bit codes is usually a space, one of 5
 * control codes, or one of 26 characters in the alphabet.
 *
 *    code   meaning
 *    ----   -------
 *       0   space
 *     1-3   next character is an index in macro table 1-3
 *	 4   next character is in alphabet 2 (A-Z)
 *       5   next character is in alphabet 3 (special, NL, 0-9, punctuation)
 *    6-31   character in alphabet 1 (a-z), except after codes 4-5
 *
 * When in alphabet 2 or 3, switching to the other alternate alphabet actually
 * returns you to alphabet 1.
 *
 * The first character in alphabet 3 (the punctuation table) is a special code,
 * which means to compose an 8-bit ASCII character from the last eight bits of
 * the next two 5-bit characters, so any ASCII character left out of the tables
 * (for example, "$") can be represented.  Code 5 is also used as padding at
 * the end of the string.
 */

/* These are the default alphabets used by the Z-code interpreter. Alternate
 * alphabets may be compiled into later games.
 */
const char default_alfabets[3][26] = {
	{
	'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
	'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z'
	},
	{
	'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
	'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z'
	},
	{
	'\0' /* ASCII literal */, '\n',
	'0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
	'.', ',', '!', '?', '_', '#', '\'', '\"', '/', '\\', '-', ':', '(', ')'
	}
};

#define ERR_CHAR '?'

char alfabets[3][26];

char *macros[3][32] = { { NULL } };

typedef unsigned char	byte;
typedef unsigned short	word;
typedef unsigned short	z_word;

struct info_header {
	byte	z_version;
		/* Z-code version:
		 * 3 = Standard Series (ZIP)
		 * 4 = Plus Series (EZIP/LZIP)
                 * 5 = Advanced Series (XZIP)
		 * 6 = Graphic Interactive Fiction (YZIP)
                 * To get around size limitations, Graham Nelson's Inform
                 * introduced two more versions:
                 * 7 = non-Infocom (version 5 with version 6 address ranges)
                 * 8 = non-Infocom (version 5 with larger address ranges)
		 */
	byte	flags;
		/* Except for the status line flag, these are set by
                 * the interpreter.
		 *
		 * x01 = v3: Header initialization error (?)
                 *	 v5+: Colors available
		 * x02 = v3: "Time:" status line (otherwise "Score:")
                 *	 v4: Inverse text available
                 *	 v6+: Graphics available
                 * x04 = v3: Unknown, always set
                 *	 v4+: Boldface available
		 * x08 = v3: Licensed to Tandy Corp.
		 *	    Causes VERSION command to add a licensing message
		 *	    on games which have been licensed to Tandy.  Also:
		 *		Zork I: Final text doesn't mention Zork II/III
		 *		Witness: censors some possibly offensive prose
		 *	 v4+: Emphasis (italic/underline) available
                 *	      (otherwise CAPITALIZE)
		 * x10 = v3: Use alternate prompt
		 *	    Displays status info in prompts in a few games for
		 *	    interpreters with no status lines.
		 *		Suspended: prompt includes linked robot name
		 *		mysteries: "keep waiting" prompt includes time
                 *	 v4+: Fixed width font available
		 * x20 = v3: Status window available
		 *	    Seastalker uses this for a fixed automatic
		 *	    sonarscope display under the status line.
                 *       v6: Sound available
                 * x40 = v3: Font is variable width
                 *	 v4+: Non-Infocom: Timed input available
                 * x80 = unused
                 */
	z_word	release;		/* release number */
	z_word	resident_size;		/* file offset where Z code begins */
	z_word	start_offset;		/* where to begin running Z code */
	z_word	vocab_offset;		/* file offset to vocabulary table */
	z_word	object_offset;		/* file offset to object table */
	z_word	variable_offset;	/* file offset to variables */
	z_word	static_offset;	 	/* file offset of static memory,
        				 * thus, offset where save files end
                                         */
	z_word	flags2;
		/*  x01 = script on (set by game)
		 *  x02 = fixed-width font (set by game)
		 *	  Usually set before a table or map is printed.
		 *  x04 = v6: redraw status line (set by game)
                 * The following are set by the interpreter to indicate
                 * whether the game can use these features.
		 *  x08 = v5+: graphics available
		 *  x10 = v3: sound available (Lurking Horror),
                 *        v5+: undo available
		 *  x20 = v5+: mouse available
		 *  x40 = v5+: colors available
		 *  x80 = v5+: sound available
                 * x100 = v5+: menus available
                 * x400 = script write (printer) error (?)
		 */
	char	serial[6];		/* serial number in ASCII, yymmdd */
	z_word	macro_offset;		/* file offset to macro table */
	z_word	verify_length;		/* how far $verify checks, in words */
	z_word	verify_checksum;
	/* v4+ status (set by interpreter) */
	char	interp_type;
		/*  1=DEC-20			 7=Commodore 128
		 *  2=Apple //e 		 8=Commodore 64
		 *  3=Macintosh 		 9=Apple //c
		 *  4=Amiga			10=Apple //gs
		 *  5=Atari ST			11=Tandy Color Computer
		 *  6=IBM/MS-DOS
                 * Beyond Zork uses this byte to determine what character
                 * graphics to use.
		 */
	char	interp_version;
        	/* typically an ASCII v3-5: letter v6: number */
	byte	screen_rows;		/* in rows, 255=infinite */
	byte	screen_cols; 		/* in columns */
	/* v5+ status (set by interpreter) */
	z_word	screen_width;		/* in units */
	z_word	screen_height;   	/* in units */
        byte	font_size1;
        	/* v5: font width (width of "0" in units) */
        	/* v6: font height */
        byte	font_size2;
        	/* v5: font height */
        	/* v6: font width (width of "0" in units) */
        /* v6 data */
	z_word	routine_offset;		/* in words */
	z_word	static_string_offset;	/* in words */
	/* v5+ status (set by interpreter) */
        byte	bg_color;	/* default background color */
        byte	fg_color;     	/* default foreground color */
        	/* 2=black	6=blue
                 * 3=red	7=magenta
                 * 4=green	8=cyan
                 * 5=yellow	9=white
                 */
	/* v5+ data */
	z_word	term_ch_offset;		/* offset to input terminator list */
	z_word	stream3_width;		/* v6: total width of stream 3 output */
        byte	std_major;        	/* non-Infocom: version of Z-Machine standard */
	byte	std_minor;              /*     supported by interpreter */
	z_word	alfabet_offset; 	/* text-compression alphabet table, if any */
	z_word	ext_offset;		/* extension table */
	char	username[8];		/* internal: username on Infocom's DEC-20 */
        	/* Inform 6 uses the last 4 bytes of the login name
                 * field for an ASCII representation of the compiler
                 * version #. Perhaps Graham Nelson foresees a time
                 * when an interpreter will change its behavior based on
                 * what version of Inform compiled a given game. Ugh.
                 */
};

/* The header is usually followed by several tables whose file offsets (bytes
 * from the start of the file) are recorded in the file header.  They are,
 * not necessarily in this order:
 *
 * - the macro table
 *
 *   All the macro compressed text is followed by an index of 3 x 32 file
 *   offsets where each macro begins.  In this case, the header contains the
 *   file offset of the index.
 *
 * - the object table
 *
 *   After the global property table (53 bytes in version 3 interpreter games,
 *   112 bytes in later ones) follows a table of data for each object in the
 *   game (up to 255 objects in version 3 games, no limit in later ones).  The
 *   data consists of attributes, parent/sibling/child in the object tree, and
 *   a pointer to that object's property list.  Object 0 is a null object and
 *   the top node in the object tree.
 *
 *   Each object's property list (up to 32 properties/object in version 3
 *   games, 64 in later ones) follow this table.  The property list starts
 *   with the object name (if it's not variable) and continues with a list
 *   of properties in descending numerical order (ex. descriptions, weight
 *   and capacity, action routines).
 *
 * - the variables table
 *
 *   Variables used by the game.  The static data offset in the header usually
 *   ends in this area, before the constants at the end.
 *
 *	variable
 *	0		current stack entry
 *	1 - 15		local variables (stacked on Z-code routine calls)
 *	16 ->		global variables
 *	16		current location
 *	17		score or hour
 *	18		moves or minutes
 *
 *   The global variables are followed by the input routine's word buffer
 *   and character buffer.  The first byte in each buffer gives its length,
 *   including that byte.
 *
 * - the vocabulary table
 *
 *   A list of characters which the parser recognizes as punctuation.  Then,
 *   the length of each vocabulary entry (which varies) and the total number
 *   of entries, followed by the table itself.  Each vocabulary entry has
 *   the word's text and a few data bytes used by the game.
 *
 * - the extension table
 *
 *   Additional header fields. The first word is the number of words of
 *   data that follows. For Infocom games with mouse support, the first
 *   two entries of this table are set to the mouse X and Y coordinates upon
 *   a mouse click.
 *
 * - the compiled Z code
 *
 *   The game code itself, followed by constant data (mostly string constants).
 */

FILE	*infile;
short	column = 0,
	columns = 0;
char	show_flags = 0,
	numbers = 0,
#ifdef DEBUG
	debug_info = 0,
#endif
	did_file = 0;
char *	prog_name = "zorkword";

word
unzword ( z_word z )
/* change most-significant-byte-first 2-byte int to whatever 2-byte int we use */
{
	byte *zp = (byte *)&z;
	return (zp[0] << 8) + zp[1];
}

void
newline ( void )
{
	putchar('\n');
	column = 0;
}

short
end_of_text ( z_word chars )
/* is the end of text bit set on this compressed text word? */
{
	return chars >> 15;
}

char *
expand ( word chars )
/* break a compressed text word into three characters */
{
	static char buf[4] = {0, 0, 0, 0};

	buf[2] = (chars & 0x1F) + 1;
	chars >>= 5;
	buf[1] = (chars & 0x1F) + 1;
	chars >>= 5;
	buf[0] = (chars & 0x1F) + 1;

	return buf;
}

char *
decode ( char *s )
/* translate the 5-bit characters into text */
{
	int		len = strlen(s);
	static char	new[BUFSIZE];
	unsigned	newlen = 0;

	/* trim off code 5's used as padding at the end of the string */
	while (s[len-1] == S_FILLER)
		s[--len] = '\0';

	while (*s) {
		switch (*s) {
			case S_MACRO1:
			case S_MACRO2:
			case S_MACRO3:
				/* next character is index in macro table */
				/* shouldn't appear in vocabulary list */
				{
					char *tmp = macros[(*s)-2][(*(s+1))-1];
					s++;
					if (tmp) {
						strcpy(&new [ newlen ], tmp);
						newlen += strlen(tmp);
					}
				}
				break;
			case S_CAPS:
				/* next character is in capitals alphabet */
				if (*(s+1) >= S_OFF)
					new [ newlen++ ] = alfabets[1][*++s - S_OFF];
				else
					new [ newlen++ ] = ERR_CHAR;
				break;
			case S_PUNC:
				/* next character is in punctuation alphabet */
				if (*(s+1) >= S_OFF)
					if (*++s == S_OFF) {
						/* combine next 2 codes into ASCII character */
						new [ newlen ] = ((*++s - 1) & 0x03) << 5;
						new [ newlen++ ] += *++s - 1;
					} else
						new [ newlen++ ] = alfabets[2][*s - S_OFF];
				else
					new [ newlen++ ] = ERR_CHAR;
				break;
			case S_BLANK:
				new [ newlen++ ] = ' ';
				break;
			default:
				new [ newlen++ ] = alfabets[0][*s - S_OFF];
		}
		s++;
	}

	new [ newlen ] = '\0';

	return new;
}

void
disp_ch ( char x )
{
	putchar(x);
	column++;
}

void
disp_str ( char *fmt, ... )
{
	va_list 	argptr;
	static char	buf[16];

	va_start(argptr, fmt);
	vsprintf(buf, fmt, argptr);
	va_end(argptr);

	column += printf(buf);
}

void
disp_bits ( char c )
/* display character in binary */
{
	unsigned b;

	disp_ch(' ');
	for (b = 0x80; b; b >>= 1)
		disp_ch(c & b ? '1' : '0');
}

void
error ( char *fmt, ... )
{
	va_list 	argptr;

	fprintf(stderr, "\nError: ");

	va_start(argptr, fmt);
	vfprintf(stderr, fmt, argptr);
	va_end(argptr);

	exit(1);
}

void
read_error ( void )
{
	error("Can't read file at offset %04lX.\n", ftell(infile));
}

void
seek_pos ( unsigned long pos )
{
	if (fseek(infile, pos, SEEK_SET) != 0)
		error("Can't seek offset %04lX.\n", pos);
}

/* At the object table offset recorded in the file header is a table of
 * data for each object in the game, preceded by the global property
 * table (53 bytes in version 3 games, 112 bytes in later ones).
 *
 * In version 3 games:
 *	byte	attributes[4];
 *	byte	location,	[parent in object tree]
 *		next,		[sibling in object tree]
 *		contents;	[child in object tree]
 *	z_word	data_offset;	[file offset for rest of data]
 *
 * In later games:
 *	byte	attributes[6];
 *	z_word	location,
 *		next,
 *		contents;
 *	z_word	data_offset;
 *
 * At the data offset, the first bytes are:
 *	1 byte		number of compressed text words to follow
 *	n words 	compressed text name of the object
 */

/* At the vocabulary table offset recorded in the file header is this data:
 *	1 byte		number of punctuation characters to follow
 *	n characters	characters recognized by the parser as punctuation
 *	1 byte		length of each vocabulary entry (usually 7 or 9)
 *	1 word		number of vocabulary entries in the table
 *
 * Then the vocabulary entries themselves.  The first 2 or 3 words of each
 * entry are the compressed text, followed by a few data bytes associated
 * with each entry.
 *
 * Type 3 games use an entry length of 7, for 2 compressed text words
 * encoding 6 letters, and 3 data bytes.  Later games have larger entry
 * lengths, for 3 compressed text words encoding 9 letters, and 2 or more
 * data bytes (but usually 3).
 */
void
dump_vocab ( unsigned long pos )
{
	register unsigned	count = 0, i;
	word			words;
	byte			letters_per_word,
				zwords_per_word,
				flag_bytes;
	short			vocab_entry_size,
				entry_width,
				entries_per_line;
	char			format[sizeof("%%-%ds")];
	int			punct;
	char *			buf;
	byte *			flags;

#ifdef DEBUG
	printf("Vocabulary table at offset %04lX\n", pos);
#endif

	seek_pos(pos);

	/* display the list of punctuation characters before the table */
	if ((punct = getc(infile)) == EOF)
		read_error();
	if (punct) {
		printf("Punctuation: ");
		for (; punct; punct--)
			putchar(getc(infile));
		putchar('\n');
	}

	/* size of each vocabulary entry */
	if ((vocab_entry_size = getc(infile)) == EOF)
		read_error();

	/* number of entries */
	if (fread(&words, sizeof(words), 1, infile) < 1)
		read_error();
	words = unzword(words);
	if (!numbers && columns != 1)
		printf("%u vocabulary entries\n", words);

	/* determine where each word ends and the data begins */
	zwords_per_word = vocab_entry_size <= 7 ? 2 : 3;
	letters_per_word = zwords_per_word * 3;
	flag_bytes = vocab_entry_size - zwords_per_word * sizeof(z_word);

	/* figure out how wide the columns are */
	entry_width = letters_per_word + 2;
	if (numbers)
		entry_width += 5;
	if (show_flags)
		entry_width += flag_bytes * (columns == 1 ? 9 : 3) + 1 + 2;
	entries_per_line = (columns ? columns : cols() + 2) / entry_width;

	/* set up buffers */
	buf = malloc(letters_per_word + 1);
	flags = malloc(flag_bytes);
	sprintf(format, "%%-%ds", letters_per_word);

	while ( count < words ) {
		++count;
		if (numbers)
			disp_str("%04d ", count);

		/* decode the text for this vocabulary entry */
		for (i = 0; i < zwords_per_word; i++) {
			word	z;

			if (fread(&z, sizeof(z), 1, infile) < 1)
				read_error();
			z = unzword(z);
			if (i)
				strcat(buf, expand(z));
			else
				strcpy(buf, expand(z));

			/* note: you can't type words without this bit set */
			if (end_of_text(z))
				break;
		}

		/* in show_flags mode, bracket words with no end-of-text bit */
		if (show_flags)
			disp_ch(i == zwords_per_word ? '[' : ' ');
		disp_str(format, decode(buf));
		if (show_flags)
			disp_ch(i == zwords_per_word ? ']' : ' ');

		/* get the additional data bytes for this entry */
		if (fread(flags, sizeof(char), flag_bytes, infile) < flag_bytes)
			read_error();

		if (show_flags) {
			register short n;

			disp_ch(' ');
			for (n = 1; n <= flag_bytes; n++)
				if (columns == 1)
					disp_bits(flags[n]);
				else
					disp_str(" %02x", flags[n]);
		}

		/* if another column won't fit, go to the next line */
		if (entries_per_line > 1 && (count % entries_per_line)) {
                        disp_ch(' ');
                        disp_ch(' ');
                } else {
			newline();
		}
	}

	free(buf);
	free(flags);

	if (column)
		newline();
}

/* At the macro offset recorded in the file header is an index for macro
 * tables 1-3, containg the file offsets in 3 tables of 32 words each.
 */
#ifdef DEBUG
void
read_macros ( unsigned long pos )
{
	register short	table, elem;
	z_word		macro_offsets[3][32];

	seek_pos(pos);

	fread(macro_offsets, sizeof(z_word), 3 * 32, infile);

	for (table = 0; table < 3; table++)
		for (elem = 0; elem < 32; elem++) {
			macros[table][elem] = NULL;

			/* display macro trivia */
			if (debug_info)
				printf("Macro %d-%02d: ", table, elem);

			if ((pos = unzword(macro_offsets[table][elem])) != 0) {
				char	macro[BUFSIZE];
				z_word	z;
				char	*text;

				seek_pos(pos *= sizeof(z_word));

				macro[0] = '\0';
				do {
					if (fread(&z, sizeof(z), 1, infile) < 1)
						read_error();
					z = unzword(z);
					if (macro[0])
						strcat(macro, expand(z));
					else
						strcpy(macro, expand(z));
				} while (!end_of_text(z));

				text = decode(macro);
				macros[table][elem] = malloc(strlen(text) + 1);
				strcpy(macros[table][elem], text);

				if (debug_info)
					printf("\"%s\"\n", text);
			}
		}
}
#endif

/* At the alphabet offset recorded in the file header are 3 tables of 26
 * ASCII characters used for text compression alphabets 1-3.
 */
void
read_alfabets ( unsigned long pos )
{
	register short	table, elem;

	/* if included in the file, read it */
	if (pos) {
		seek_pos(pos);
		fread(alfabets, 1, 3 * 26, infile);

#ifdef DEBUG
		if (debug_info)
			for (table = 0; table < 3; table++) {
				printf("Alphabet %d: ", table+1);
				for (elem = 0; elem < 26; elem++)
					putchar(isprint(alfabets[table][elem]) ? alfabets[table][elem] : ERR_CHAR);
				putchar('\n');
			}
#endif

		return;
	}

	/* load default alfabets */
	for (table = 0; table < 3; table++)
		for (elem = 0; elem < 26; elem++)
			alfabets[table][elem] = default_alfabets[table][elem];

}

const char *extensions[] = {
	".data", ".z8", ".z7", ".z6", ".z5", ".z4", ".z3", NULL
};

void
frob_file ( const char *filename )
{
	struct info_header	header;
	int			i;

	if((infile = fopen(filename, "rb")) == NULL) {
               	char *add_ext = malloc(strlen(filename) + 6);

		for (i = 0; extensions[i] != NULL; i++) {

			/* add extension to file name */
			strcpy(add_ext, filename);
			strcat(add_ext, extensions[i]);

			/* try again */
			if((infile = fopen(add_ext, "rb")) != NULL)
                        	break;
		}

		/* if none of them worked */
                if (extensions[i] == NULL)
			error("Can't open file \"%s\".\n", filename);

		printf("%s:\n", add_ext);

		free(add_ext);
	} else
		printf("%s:\n", filename);

	if (fread(&header, sizeof(header), 1, infile) < 1)
		read_error();

	/* Sanity check - Assume it's not a data file if revision date
	 * is not printable, since running on other files can be messy.
	 * Don't use isdigit() because some warez flatheads patch them
         * out with "LOST  " and the like. Also, Paul David Doherty has
         * reported on v2 versions of Zork I and II with a serial number
         * of "UG3AU5". I have not tested this program with any v1 or
         * v2 files.
         */
	for (i = 0; i < 6; i++)
		if (!isprint(header.serial[i]))
			break;

	if (header.z_version >= 2 && header.z_version <= 12 && i == 6) {
#ifdef DEBUG
		/* display Z machine needed to run game */
		if (debug_info) {
			printf("Z machine version %d\n", (int)header.z_version);

			/* display Inform compiler version */
                	if (isprint(header.username[4]))
                		printf("Compiler version %.4s\n", &header.username[4]);
                }
#endif

		/* display release number and date from the file header */
		printf("Release %u / Serial number %.6s\n", unzword(header.release), header.serial);

		/* load alphabets and macros */
		read_alfabets(unzword(header.alfabet_offset));

#ifdef DEBUG
		if (debug_info)
			read_macros(unzword(header.macro_offset));
#endif

		dump_vocab(unzword(header.vocab_offset));
	}
	/* catch pix data files */
	else if (header.z_version == 1) {
		z_word	release;

		seek_pos(0x0D);
		fread(&release, sizeof(z_word), 1, infile);

		printf("Pix release %u\n", unzword(release));

		fclose(infile);
		return;
	}

	fclose(infile);
}

#ifndef LINT
const char sccsid[] = "@(#) zorkword.c by Mike Threepoint compiled " __DATE__;
#endif

void
info ( void )
{
	printf(INFO, prog_name);
	exit(0);
}

void
parse_opt ( char p )
{
	switch (p) {
		case 'w':
			columns = 0;
			break;
		case '#':
		case 'n':
			numbers = 1;
			break;
		case 'f':
		case 'b':
			show_flags = 1;
			break;
#ifdef DEBUG
		case 'd':
			debug_info = 1;
			break;
#endif
		case 'h':
		case '?':
			info();
			break;
		default:
			/* if it's a digit, use that many columns */
			if (isdigit(p))
				columns = (p - '0');
	}
}

void
parse ( char *parm )
{
	switch (*parm) {
		case '/':
			parse_opt(*++parm);
			break;
		case '-':
			while (*++parm) parse_opt(*parm);
			break;
		default:
			if (did_file) newline();
			frob_file(parm);
			did_file = 1;
	}
}

int
main ( const unsigned argc, char *argv[] )
{
	if (argv[0]) prog_name = argv[0];

	if (argc > 1) {
		register count;

		if (strcmp(argv[1], "?") == 0)
			info();

		for (count = 1; count < argc; count++)
			parse(argv[count]);

		if (did_file) return 0;
	}

	info();
	return 1;
}
