// zeal - A portable Glk-based Z-code interpreter
// Copyright (C) 2000 Jeremy Condit <jcondit@eecs.harvard.edu>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

// =======================================================================
//  error.h:
//
//  this module defines some simple functions designed for use with the
//  assertion mechanism.  since we want assertions to be printed to the
//  game console, we need to use the io global, but we don't want to
//  access that within macros, since that's just plain messy.
//
//  in the header file, we also have macros that use these functions.
//  other modules should use the macros, not the functions themselves!
// =======================================================================

#ifndef _error_h_
#define _error_h_

void
zeal_assert(char* msg, char* file, int line);

void
zeal_warn(char* msg, bool* warned);

// ASSERT_MSG
//
// if cond is false, print message msg and exit.
//
// for use with fatal error conditions that the user would want to hear
// about.  e.g. this could be used when checking for a valid input file,
// where the user might want to know what went wrong when we bail.

#define ASSERT_MSG(cond, msg) \
    if (!(cond)) { \
        zeal_assert((msg), __FILE__, __LINE__); \
    }

// ASSERT
//
// if cond is false, print generic message and exit.
//
// for use with internal assertions--that is, sanity checks for things that
// really shouldn't happen and that the user can't do anything about if they
// fail.  e.g. an invariant in some code is violated--a true bug.

#define ASSERT(cond) \
    ASSERT_MSG(cond, "Assertion failed.")

// FATAL
//
// we've hit a fatal error.  bail immediately, issuing the message supplied.

#define FATAL(msg) \
    ASSERT_MSG(false, msg)

// WARN
//
// issues a warning to the user depending on the current strictz level.  note
// that we introduce an extra static variable to keep track of whether the
// user has been warned by this code before.

#define WARN(msg) \
    { \
        static bool zzz_warned = false; \
        zeal_warn((msg), &zzz_warned); \
    }

// MEMCHECK
//
// used on newly allocated chunks of memory to make sure allocation succeeded.

#define MEMCHECK(var) \
    ASSERT_MSG((var) != NULL, "Out of memory.")

#endif // _error_h_
