/*
	TC.C: Converts text files between PC and UNIX format.
	Written in ANSI-C.
	(c) Fredrik Ramsberg 1996

	Known bugs:
	None.

*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define UNKNOWN 0
#define UNIX 1
#define PC 2

#define MIXED 10
#define NOTUNIX 11
#define NOTPC 12

int type, only, force, err, errln;
unsigned long nofc, nofnl, k;

void printsyntax()
{
	printf("\ntc: A utility that converts textfiles between PC and UNIX format.\n");
	printf("(c) Copyright Fredrik Ramsberg 1996.\n\n");
	printf("Usage: tc [-puf] infile [outfile]\n\n");
	printf("When tc is used without switches, it detects the file type automatically.\n\n");
	printf("If no outfile is given,");
	printf(" a temporary file is used and then copied onto infile.\n\n");
	printf("Switches:\n");
	printf(" -p Only convert the file if it is currently in PC format.\n");
	printf(" -u Only convert the file if it is currently in UNIX format.\n");
	printf(" -pf Force conversion of all line-endings that are currently in PC format.\n");
	printf(" -uf Force conversion of all line-endings that are currently in UNIX format.\n");
}

void newtype(ty)
int ty;
{
	if(ty!=type)
		if(type==UNKNOWN)
			type=ty;
		else {
			if(force)
				type=ty;
			else {
				err=MIXED;
				errln=nofnl+1;
			}
		}

}


int main(argc, argv)
int argc;
char *argv[];

{
int i, c, lastc, syntax, thisarg;
char outfile[255];
char infile[255];
char tempname[255];
FILE *in, *out;

tmpnam(tempname);
type=UNKNOWN;
err=0;
errln=-1;
nofc=0;
nofnl=0;

thisarg=0;
only=0;
force=0;
while(++thisarg<argc && err==0) {
	if(argv[thisarg][0]=='-') {
		for(i=1;i<(int)strlen(argv[thisarg]);i++) {
			switch(argv[thisarg][i]) {
			case 'p':
				/* Only convert PC -> UNIX */
				only=PC;
				break;
			case 'u':
				/* Only convert UNIX -> PC */
				only=UNIX;
				break;
			case 'f':
				/* Enforce conversion */
				force=1;
				break;
			default:
				printf("Illegal switch: -%c\n", argv[thisarg][i]);
				printsyntax();
				exit(1);
			}
		}
	}
	else
		err=1; /* No more switches */
}
if(only==0)
	force=0;
if(err)
	thisarg--;
err=0;

switch (argc-thisarg) {
case 1:
	strcpy(outfile, tempname);
	syntax=1;
	break;
case 2:
	strcpy(outfile, argv[thisarg+1]);
	syntax=2;
	break;
default:
	printsyntax();
	exit(1);
}

strcpy(infile,argv[thisarg]);

if ((in=fopen(outfile,"r"))!=NULL) {
	printf("The outfile %s already exists.",outfile);
	fclose(in);
	exit(1);
}

if ((out=fopen(outfile,"wb"))!=NULL) {
	if ((in=fopen(infile,"rb"))!=NULL) {
		lastc=0;
		while ((c=getc(in))!=EOF && err==0) {

			switch(c) {

			case 13:
				if(only==UNIX) {
					if(force==0) {
						err=NOTUNIX;
						errln=nofnl+1;
					}
					putc(c,out);
				}
				if(!err)
					newtype(PC);
				break;

			case 10:
				if(lastc!=13) {
					if(only==PC) {
						if(!force) {
							err=NOTPC;
							errln=nofnl+1;
						}
					}
					else
						putc(13,out);
					if(!err)
						newtype(UNIX);
				}
				nofnl++;
				putc(c,out);
				break;

			default:
				putc(c,out);
				nofc++;
			}
			lastc=c;

		} /* while */
		fclose(in);
	}
	else {
		printf("Could not open infile: %s\n", infile);
		fclose(out);
		remove(outfile);
		exit(1);
	}
	fclose(out);
}
else {
	printf("Could not open outfile: %s\n",outfile);
	remove(outfile);
	exit(1);
}

if(err) {
	switch(err) {
	case MIXED:
		printf("ERROR: Mixed file type discovered on line %d of file %s",
						errln, infile);
		break;
	case NOTUNIX:
		printf("NOTE: File %s is not in UNIX format (discovered on line %d).\n",
						infile, errln);
		remove(outfile);
		exit(0);
	case NOTPC:
		printf("NOTE: File %s is not in PC format (discovered on line %d).\n",
						infile, errln);
		remove(outfile);
		exit(0);
	default:
		if(errln>=0)
			printf("ERROR: General error #%d on line %d of file %s",
							err, errln, infile);
		else
			printf("ERROR: General error #%d in file %s", err, infile);
	}
	remove(outfile);
	exit(err);
}


if(syntax==1) {
	k=1L;
	if(nofnl>0)
		k=nofc/nofnl;
	if(k>79) {
		printf("%s has an average of %d characters/line.\n", infile, k);
		printf("Thus, it doesn't seem to be a normal textfile.\n\n");
		printf("Convert it anyway (Y/N) ? ");
		if((c=getchar())!='y' && c!='Y') {
			printf("No!\n");
			if(remove(outfile)!=0)
				printf("Removal of tempfile failed!\n");
			exit(1);
		}
		else {
			printf("Yes!\n");
			k=1;
		}
	}
	if(k<=79) {
		if(fopen(outfile,"rb")!=NULL) {
			if ((in=fopen(infile,"wb"))!=NULL) {
				while((c=getc(out))!=EOF && !ferror(in) && !ferror(out))
					putc(c,in);
				if(ferror(in)) {
					printf("Error while creating or writing new version of file %s\n",infile);
					printf("Contents stored in file %s\n", outfile);
					fclose(in);
					fclose(out);
					exit(1);
				}
				else if(ferror(out)) {
					printf("Error while reading tempfile %s\n",outfile);
					printf("File %s destroyed. New contents stored in file %s\n", outfile);
					fclose(in);
					fclose(out);
					exit(1);
				}
				fclose(in);
				fclose(out);
				remove(outfile);
			}
			else {
				printf("Error while creating new version of file %s\n",infile);
				printf("Contents stored in file %s\n", outfile);
				fclose(out);
				exit(1);
			}
		}
		else {
			printf("Error while opening tempfile %s for reading\n",outfile);
			printf("Old file %s left unchanged.\n", infile);
			exit(1);
		}
	}
}


if(force)
	if(only==PC)
		printf("Forced conversion from PC to UNIX format complete.\n");
	else
		printf("Forced conversion from UNIX to PC format complete.\n");
else
	if(type==PC)
		printf("Conversion from PC to UNIX format complete.\n");
	else
		printf("Conversion from UNIX to PC format complete.\n");

return 0;
}
