/* fgsubs.c - underlying subroutines used from fgdat
 *
 * All access to the data file should go through this module, so that
 * we can properly change over to another data file when needed.
 *
 * Basically, this file contains routines which do not specifically
 * name the fields, whereas routines in fgdat do specifically name fields.
 *
 * 26.Oct.87  jimmc  Extracted from fgdat
 *  8.Jan.88  jimmc  Lint cleanup
 * 19.Jan.88  jimmc  Add fgenum, fginit
 */

/* Functions which return an allocated string are:
 * fgstr(n,s)	the basic string routine which returns field s
 *
 * Functions which return an integer are:
 * fgnum(n,s)	returns the integer value of field s
 * fgblist(n,s,av)	get a list; return valus is ac, fills array av
 * fglist(n,s,avp)	get a list; return value is ac, fills pointer avp
 *
 * Other functions:
 * fgbstr(n,s,b)	like fgstr, but returns string in buffer b
 * fgchar(n)		returns single character (first char of data word)
 * fgenum(funcp)	enumerates all entries in the data file
 */

#include <stdio.h>
#include <strings.h>
#include "dataman.h"
#include "geneal.h"
#include "xalloc.h"

extern struct dpoint *initDataFile();

extern char *dataErrMsg;

static struct dpoint *gendp;		/* data file handle */
static char *datfilename;
static char *opendatfilename;

/*..........*/

fgopenfn(fn)
char *fn;
{
	fgsetfn(fn);
	fgopen();
}

fgsetfn(fn)
char *fn;
{
	if (datfilename) freestr(datfilename);
	datfilename = strsav(fn);
}

fgopen()
{
	if (!datfilename)
		fatalerr("No data filename specified");
	gendp = initDataFile(datfilename);
	if (!gendp)
		fatalerr("error opening data file %s: %s",
			datfilename,dataErrMsg);
	if (opendatfilename) freestr(opendatfilename);
	opendatfilename = datfilename;
	datfilename = 0;
}

fgclose()
{
	if (gendp) {
		closeDataFile(gendp);
		gendp = 0;
		if (opendatfilename) freestr(opendatfilename);
	}
}

/*..........*/

static
checkdp()
{
	if (gendp==0) {
		fgopen();
		if (gendp==0) fatalerr("no data file open");
	}
	/* return and continue */
}

/*..........*/

int
fgenum(funcp)
int (*funcp)();
{
	int t;

	checkdp();
	t = enumIndex(gendp->xx,funcp);
	return t;
}

/*..........*/

int
fgnum(n,s)		/* get an item number from the data file */
int n;			/* person to get data item for */
char *s;		/* name of data item */
{
	int tt, dd;
	char *str;

	checkdp();
	if (n<=0) return -1;
	str = getData(gendp,n,s);	/* get pointer to data item */
	if (str==0) return -1;		/* -1 if no such item */
	tt = sscanf(str,"%d", &dd);	/* get the number */
	if (tt!=1) return -1;		/* if no succesful scan */
	return dd;			/* return the number we found */
}

/*..........*/

int
fgchar(n,s)		/* get the first character from a field */
int n;			/* record to get data from */
char *s;		/* name of data item */
{
	char *word;

	checkdp();
	if (n<=0) return 0;
	word = getData(gendp,n,s);
	if (!word) return 0;
	return word[0];
}

/*..........*/

char *
fgstr(n,s)		/* get an item string from the data file */
int n;			/* person to get data item for */
char *s;		/* name of data item */
{
	char *str;

	checkdp();
	if (n<=0) return "";
	str = getData(gendp,n,s);	/* get pointer to data item */
	if (str==0) return "";		/* null string if no such item */
	return strsav(str);
}

/*..........*/

int			/* returns 1 if found, 0 if not */
fgbstr(n,s,b)		/* get an item string from the data file */
int n;			/* person to get data item for */
char *s;		/* name of data item */
char *b;		/* the buffer to put it into */
{
	char *str;

	checkdp();
	if (n<=0) { 
		*b=0; 
		return 0; 
	}	/* make string null */
	str = getData(gendp,n,s);	/* get pointer to data item */
	if (str==0) { 
		*b=0; 
		return 0; 
	}	/* null string if no such item */
	strcpy(b,str);		/* copy the string to his buffer */
	return 1;
}

/*..........*/

int			/* returns count */
fgblist(n,t,av)
int n;			/* id of family */
char *t;		/* the keyword tag for the field to read */
int *av;		/* array to put data into */
{
	char cstr[200];
	char *cstrp;
	int *clist;
	int i, ac, tt, dd;

	fgbstr(n,t,cstr);	/* get list of kids */
	for (ac=0, cstrp=cstr; cstrp!=NULL && *cstrp!='\0'; ac++)
		cstrp=index(cstrp+1,',');	/* count separators */
	if (ac==0) return 0;
	clist = av;			/* put data into his array */
	for(cstrp=cstr, i=0; cstrp!=NULL && *cstrp!='\0'; i++) {
		if (i>=ac) fatalerr("loop too far on %s list in fgblist", t);
		tt = sscanf(cstrp,"%d",&dd);	/* read number */
		if (tt==1) clist[i]=dd;
		else {
			warning("bad %s list format in family %d", t, n);
			clist[i] = dd = -1;
		}
		cstrp = index(cstrp,',');
		if (cstrp!=NULL) cstrp++;
	}
	if (i!=ac) {
		warning("bad %s list format in family %d", t, n);
		for (; i<ac; i++) clist[i] = -1;	/* fill with -1 */
	}
	return ac;		/* return count */
}

/*..........*/

int			/* returns count */
fglist(n,t,avp)
int n;			/* id of family */
char *t;		/* the keyword of the list to get */
int **avp;		/* ADDRESS of where to put the av we return */
{
	int *clist;
	int cbuf[1000];
	int i, ac;

	ac = fgblist(n,t,cbuf);
	i = (ac==0)? 1 : ac;
	clist = XALLOCM(int, i, "fglist");
	for (i=0; i<ac; i++) clist[i] = cbuf[i];	/* copy the list */
	*avp = clist;	/* fill in his pointer */
	return ac;		/* return count */
}

/* end */
