/* fgdat.c - routines to get pieces of info records
 * Written by Jim McBeath (jimmc) at SCI
 *
 * Revision history:
 * 27-Jan-85	Jim McBeath	convert fgclist to fg*list functions
 *				add addn[cp]list functions
 * 29-Jan-85	Jim McBeath	add family index in names if index>1
 *				convert to 7-char-distinct names
 *  2-Feb-85	Jim McBeath	Convert to LN.A instead of LNM.
 * 13-Feb-85	(Ian Darwin)	use NULL and \0 instead of 0 for cstr
 * 17.Aug.87  jimmc  Add fglname
 * 18.Aug.87  jimmc  Add fgtype, fgspouse
 * 24.Aug.87  jimmc  Add fgburied
 * 25.Aug.87  jimmc  Add fgnmarriage
 * 18.Sep.87  jimmc  Add #include xalloc.h; add fgaddrphn
 * 26.Oct.87  jimmc  Remove basic routines into fgdatsubs.c
 * 27.Oct.87  jimmc  Add fggen
 *  1.Jan.88  jimmc  Make fgtnote and fggen also pick up TGEN; add fglpname
 *  4.Jan.88  jimmc  Add fglhname
 *  8.Jan.88  jimmc  Lint cleanup
 */

/* These routines all package up various pieces of information
 * from specified records.  They each take as arguments a record
 * ID number; some take a field name as well.  Most return a pointer
 * to a string allocated from dynamic memory, which can be freed
 * by passing the pointer to the freestr() function.
 * The functions in this file in general refer to and format specific
 * fields; the functions in fgdatsubs are more general, without references
 * to specific fields.
 * In the arguments, i is an ID for an individual's record; f is an ID for
 *  a family record, and n is an ID for either (or sometimes any record kind).
 *  Functions which return an allocated string are:
 * fgsname(i)   just the nickname; else first name; else middle name; else "?"
 * fgtname(i)	all of name but last name
 * fglname(f)   last name of a family only
 * fglpname(f)  last name of a family plus names of parents
 * fgbname(i)	full born (maiden) name
 * fgfname(i)	full name (including maiden and married last names)
 * fgbirth(i)	birth date and place
 * fgdeath(i)	death date and place
 * fgburied(i)	burial location
 * fgbrtdeath(i)	birth and death dates
 * fgmarriage(f)	marriage date and place
 * fgnmarriage(f)	marriage date and place, and number if +n switch
 * fgtnote(n)		TNOTE plus TGEN
 * fggen(n)		GEN plus TGEN
 *  Functions which return an integer are:
 * fgspouse(f_m,i_n)	returns the spouse for marriage f_m who is not i_n
 * fgbclist(f,av)	get child list; return valus is ac, fills array av
 * fgclist(f,avp)	get child list; return value is ac, fills pointer avp
 * fgbslist(i,av)	get spouse list; return valus is ac, fills array av
 * fgslist(i,avp)	get spouse list; return value is ac, fills pointer avp
 *  Other functions:
 * fgtype(n)		returns single character which is type code (e.g. 'I')
 */

#include <stdio.h>
#include <strings.h>
#include "geneal.h"
#include "xalloc.h"

extern char *strcrep();

/* for forward references */
char *fgdateplace();

/*..........*/

int
fgtype(n)		/* get a type character from the file */
int n;
{
	return fgchar(n,"T");
}

/*..........*/

int
fgspouse(m,n)
int m;			/* id of the marriage */
int n;			/* id of the spouse we DONT want */
{
	int t;

	t = fgnum(m,"H");
	if (t>0 && t!=n) return t;
	t = fgnum(m,"W");
	if (t>0 && t!=n) return t;
	return -1;
}

/*..........*/

char *
fgsname(n)		/* short name (first name or whatever) */
int n;
{
	char xname[200];
	int l;

	xname[0] = 0;
	addindex(xname,n);
	addncstr(xname,n,"PN");
	l = strlen(xname);
	addncstr(xname,n,"NN");	/* use nickname if there */
	if (!xname[l]) addncstr(xname,n,"FN");
	if (!xname[l]) addncstr(xname,n,"MN");
	if (!xname[l]) addcstr(xname,"??");
	addncstr(xname,n,"SN");
	return strsav(xname);
}

/*..........*/

char *
fgtname(n)		/* get the full name (except last name) */
int n;			/* person to get data item for */
{
	char xname[200];

	xname[0] = 0;		/* start with null */
	addindex(xname,n);	/* put in id number if requested */
	addncstr(xname,n,"PN");	/* prefix name */
	addncstr(xname,n,"FN");	/* add first name */
	addncstr(xname,n,"MN");	/* add middle name */
	addnpstr(xname,n,"NN");	/* add nickname in parens */
	addncstr(xname,n,"SN");	/* suffix name */
	return strsav(xname);
}

/*..........*/

char *
fglname(f)
int f;			/* family id number */
{
	char xname[200];

	xname[0] = 0;		/* start with null */
	addindex(xname,f);	/* put in id number if requested */
	addncstr(xname,f,"N");	/* the name */
	return strsav(xname);
}

/*..........*/

char *
fglpname(f)
int f;			/* family id number */
{
	char xname[200];
	int h,w;
	char *fn, *hln, *wln, *hfn, *wfn, *fpn;

	xname[0] = 0;
	addindex(xname,f);
	fn = fgstr(f,"N");	/* get family last name */
	h = fgnum(f,"H");	/* husband and wife ID numbers */
	w = fgnum(f,"W");
	if (h<=0 && w<=0) {
		addcstr(xname,fn);
		freestr(fn);
		return strsav(xname);
	}
	hln = fgstr(h,"LN");
	wln = fgstr(w,"LN.A");	/*** sex-biased assumptions about names... */
	hfn = fgsname(h);
	wfn = fgsname(w);
	if (strcmp(fn,hln)==0 && strcmp(fn,wln)==0) {
		fpn = tprintf("%s and %s %s", hfn, wfn, fn);
	}
	else if (strcmp(fn,hln)==0) {
		if (wln && wln[0])
			fpn = tprintf("%s %s and %s %s", hfn, hln, wfn, wln);
		else
			fpn = tprintf("%s %s and %s", hfn, hln, wfn);
	}
	else {
		fpn = tprintf("%s: %s %s and %s %s", fn, hfn, hln, wfn, wln);
	}
	addcstr(xname,fpn);
	freestr(hln);
	freestr(wln);
	freestr(hfn);
	freestr(wfn);
	freestr(fn);
	freestr(fpn);
	return strsav(xname);
}

/*..........*/

char *
fglhname(f)
int f;			/* family id number */
{
	char xname[200];
	int h,w;
	char *hpn, *wpn, *fn;

	xname[0] = 0;
	addindex(xname,f);
	h = fgnum(f,"H");	/* husband and wife ID numbers */
	w = fgnum(f,"W");
	hpn = fgstr(h,"PN");
	if (!hpn || !hpn[0]) hpn="Mr.";
	wpn = fgstr(w,"PN");
	if (!wpn || !wpn[0]) wpn="Mrs.";
	addcstr(xname,hpn);
	addcstr(xname,"and");
	addcstr(xname,wpn);
	freestr(hpn);
	freestr(wpn);
	if (h>0) {
		addncstr(xname,h,"FN");
		addncstr(xname,h,"MN");
		addnpstr(xname,h,"NN");
		addncstr(xname,h,"LN");
		addncstr(xname,h,"SN");
	}
	else {
		fn = fgstr(f,"N");	/* get family last name */
		addcstr(xname,fn);
		freestr(fn);
	}
	return strsav(xname);
}

/*..........*/

char *
fgbname(n)		/* get the full born name */
int n;			/* person to get data item for */
{
	char xname[200];

	xname[0] = 0;		/* start with null */
	addindex(xname,n);		/* put in id number if requested */
	addncstr(xname,n,"PN");
	addncstr(xname,n,"FN");
	addncstr(xname,n,"MN");
	addnpstr(xname,n,"NN");
	addncstr(xname,n,"LN");
	addncstr(xname,n,"SN");
	return strsav(xname);
}

/*..........*/

char *
fgfname(n)		/* get the full name */
int n;			/* person to get data item for */
{
	char xname[200];
	char lnaname[200];

	fgbstr(n,"LN.A",lnaname);
	strcrep(lnaname, ';', ' ');	/* convert semicolons to space */
	xname[0] = 0;		/* start with null */
	addindex(xname,n);		/* put in id number if requested */
	addncstr(xname,n,"PN");
	addncstr(xname,n,"FN");
	addncstr(xname,n,"MN");
	addnpstr(xname,n,"NN");
	addncstr(xname,n,"LN");
	addcstr(xname,lnaname);
	addncstr(xname,n,"SN");
	return strsav(xname);
}

/*..........*/

char *
fgbirth(n)		/* get birth date info */
int n;			/* person to get data item for */
{
	return fgdateplace(n,"B","BP","b");
}

/*..........*/

char *
fgdeath(n)		/* get death date info */
int n;			/* person to get data item for */
{
	return fgdateplace(n,"D","DP","d");
}

/*..........*/

char *
fgburied(n)		/* get burail place */
int n;			/* person to get data item for */
{
	char b[1000];

	fgbstr(n,"BUR",b);
	if (*b) return tprintf("bur: %s",b);
	else return strsav("");
}

/*..........*/

char *
fgbrtdeath(n)		/* get birth/death date info */
int n;			/* person to get data item for */
{
	char bdate[100], ddate[100], dates[200];

	fgbstr(n,"B",bdate);	/* get birth date */
	fgbstr(n,"D",ddate);	/* get death date */
	if (*bdate==0 && *ddate==0) return "";
	if (*ddate==0) sprintf(dates," ( b: %s )", bdate);
	else if (*bdate==0) sprintf(dates," ( d: %s )", ddate);
	else sprintf(dates," ( b: %s, d: %s )", bdate, ddate);
	return strsav(dates);
}

/*..........*/

char *
fgmarriage(n)		/* get marriage date info */
int n;			/* person to get data item for */
{
	return fgdateplace(n,"M","MP","m");
}

/*..........*/

char *
fgnmarriage(n)		/* get marriage date info and optionally number */
int n;			/* person to get data item for */
{
	char *s, *t;

	s = fgdateplace(n,"M","MP","m");
	if (!Gflag['n']) return s;
	t = tprintf("%s [%d]",s,n);
	freestr(s);
	return t;
}

/*..........*/

char *fgt2gen(n,item)
int n;
char *item;
{
char *tgen;
char *istr;
char *rstr;

	tgen = fgstr(n,"TGEN");
	istr = fgstr(n,item);
	if (tgen && tgen[0] && istr && istr[0]) {
		rstr = tprintf("%s\n%s",tgen,istr);
		freestr(tgen);
		freestr(istr);
		return rstr;
	}
	if (tgen && tgen[0]) return tgen;
	return istr;
}

char * fgtnote(n) int n; { return fgt2gen(n,"TNOTE"); }
char * fggen(n) int n; { return fgt2gen(n,"GEN"); }

/*..........*/

char *
fgaddrphn(n)
int n;
{
	char addr[1000];
	char phone[1000];
	char *t;

	fgbstr(n,"ADDR",addr);
	fgbstr(n,"PHONE",phone);
	if (*addr && *phone)
		t = tprintf("%s\n%s",addr,phone);
	else if (*addr)
		t = strsav(addr);
	else if (*phone)
		t = strsav(phone);
	else
		t = strsav("");
	return t;
}

/*..........*/

int
fgclist(n,avp)		/* get child list into allocated array */
int n;			/* family to get list for */
int **avp;		/* where we should put the pointer to the array */
{
	return fglist(n,"C",avp);
}

/*..........*/

int
fgbclist(n,av)		/* get child list into specified array */
int n;			/* family to get list for */
int *av;		/* where we should put the data */
{
	return fgblist(n,"C",av);
}

/*..........*/

int
fgslist(n,avp)		/* get spouse list into allocated array */
int n;			/* person to get list for */
int **avp;		/* where we should put the pointer to the array */
{
	return fglist(n,"S",avp);
}

/*..........*/

int
fgbslist(n,av)		/* get spouse list into specified array */
int n;			/* person to get list for */
int *av;		/* where we should put the data */
{
	return fgblist(n,"S",av);
}

/*..........*/

char *
fgdateplace(n,dk,pk,cc)	/* get date/place info */
int n;			/* person to get data item for */
char *dk;		/* date keyword */
char *pk;		/* place keyword */
char *cc;		/* string to use as output key */
{
	char date[100], place[100];

	fgbstr(n,dk,date);	/* get date */
	fgbstr(n,pk,place);	/* get place */
	if (*date==0 && *place==0) return "";
	if (*date && *place) return tprintf("%s: %s, %s", cc, date, place);
	if (*date) return tprintf("%s: %s", cc, date);
	return tprintf("%s: %s", cc, place);
}

/*..........*/

addncstr(dd,n,ff)	/* add a field value onto a string */
char *dd;		/* the string being built */
int n;			/* record number */
char *ff;		/* name of field to get from that record */
{
	char buf[1000];

	fgbstr(n,ff,buf);
	addcstr(dd,buf);
}

/*..........*/

addnpstr(dd,n,ff)	/* add a field value onto a string in parens */
char *dd;		/* the string being built */
int n;			/* record number */
char *ff;		/* name of field to get from that record */
{
	char buf[1000];

	fgbstr(n,ff,buf);
	addpstr(dd,buf);
}

/*..........*/

addcstr(dd,ss)		/* add a string to another */
char *dd;		/* the string being built */
char *ss;		/* the string to add if not null */
{
	if (ss && *ss) {
		if (dd[0]) strcat(dd," ");
		strcat(dd,ss);
	}
}

/*..........*/

addpstr(dd,ss)		/* add a string in parens */
char *dd;		/* the string to add to */
char *ss;		/* the string to add in parens if not null */
{
	if (ss[0]) {
		if (dd[0]) strcat(dd," ");
		strcat(dd,"(");
		strcat(dd,ss);		/* add the string in parens */
		strcat(dd,")");
	}
}

/*..........*/

addindex(ss,n)
char *ss;			/* the string to add to */
int n;				/* id number of person */
{
	if (Gflag['N'])
		sprintf(ss,"[%d/%d]", n, fgnum(n,"P"));
		/* put in both individual and family if requested */
	else if (Gflag['n'])
		sprintf(ss, "[%d]", n);  /* put in the id number */
	/* if neither flag set, put in nothing */
}

/* end */
