/*
** Program fdb - family tree database generation and manipulation
**
** Copyright (C) 1994 Andy Burrows 
**
**            email: cadellin@corum.me.man.ac.uk (130.88.29.14)
**
** This program is free software; you can redistribute it and/or modify it
** under the terms of the GNU General Public Licence as published by the Free
** Software Foundation; either version 1, or any later version.
** 
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
** or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
** more details. It should be included in this distribution in a file 
** called COPYING
**
*/

/*
   this file contains the callback functions for the scrolling lists
*/

/*#define DEBUG */

/* standard headers */

#include <stdio.h>
#include <unistd.h>
#include <sys/times.h>
#ifndef SYSV
#include <sys/timeb.h>
#endif

/* XView headers */

#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/font.h>
#include <xview/rect.h>

/* fdb database definitions */

#include "definitions.h"

/* load file from selection callback */

void load_file_select(item, string, client_data, op, event)
Panel_item      item;
char           *string;
caddr_t         client_data;
Panel_list_op   op;
Event          *event;
{
Xv_object             return_code;
char                  footer_string[255];
static int            initialise = 1;
static double         last_time = 0.0;
double                this_time;
double                elapsed_secs;
#ifdef SYSV
struct tms            tms_buf;
#else
static struct timeb   timeb_old_buf;
struct timeb          timeb_buf;
#endif

/* if this is a deselection, just return */

if(op == PANEL_LIST_OP_DESELECT)
    {
#ifdef DEBUG
    printf("load file scrolling list callback DESELECTed file **%s**\n",string);
#endif
    return;
    }

/* 
   this code added after a report of mouse button click bounces
   causing selection of two people consecutively
   so if the last selection was too recent, just return 
*/

#ifdef SYSV  /* if system V then use gethrtime function... */

/* find the time with the gethrtime function (returns value in nanoseconds) */
this_time = (double) gethrtime();
elapsed_secs = ((this_time - last_time))/1000000000.0;
last_time = this_time;

#else /* ...otherwise assumes BSD and uses ftime */

/* find the time with the ftime function (returns a timeb struct) */
if(ftime(&timeb_buf) == -1)
    printf("ftime error!\n"); 
if(!initialise)
    {
    elapsed_secs = ((double) (timeb_buf.time - timeb_old_buf.time)) + 
                   (((double) timeb_buf.millitm) - 
                    ((double) timeb_old_buf.millitm))/1000.;
    }
else
    {
    elapsed_secs = 2.0;
    initialise = 0;
    }
timeb_old_buf = timeb_buf;
#endif

#ifdef DEBUG
printf("elapsed value: %f\n",elapsed_secs);
#endif

/* if the last selection was too recent, unset the selection and return */

if(elapsed_secs < FILE_MULTICLICK_TOLERANCE)
    {
#ifdef DEBUG
    printf("apparent double click ignored\n");
#endif
    return;
    }

strcpy(new_file_name, string);

/* if the last character in the string is a slash, lose it */

if(new_file_name[strlen(new_file_name) - 1] == '/')
    new_file_name[strlen(new_file_name) - 1] = '\0';

#ifdef DEBUG
printf("file name **%s** selected\n",new_file_name);
#endif

/* 
   check if the selected entry is a directory by trying to move into it!
   if O.K. rerun the scrolling list creation using the new directory
*/

if(chdir(new_file_name) == 0)
    {
    /* need to rerun the file selection setup routine */
    file_menu_selection(file_menu, load_file_item);
    return;
    }

/* unshow the "Load From" window */

if((return_code = xv_set(load_frame, XV_SHOW, FALSE, NULL)) != XV_OK)
    {
    printf("unable to unshow the 'Load From' window\n");
    }

/* call the database load function */

initialise_database();

/* update the displays */

update_panel_items();

/* create the new footer display string and set it */

sprintf(footer_string,"Current Database File: %s",file_name);
xv_set(fdb_frame, FRAME_LEFT_FOOTER, footer_string, NULL);

/* set the icon label to the new file name */

xv_set(main_icon, ICON_LABEL, file_name, NULL);

}

/* save file as selection callback */

void save_file_select(item, string, client_data, op, event)
Panel_item      item;
char           *string;
caddr_t         client_data;
Panel_list_op   op;
Event          *event;
{
Xv_object             return_code;
char                  footer_string[255];
static int            initialise = 1;
static double         last_time = 0.0;
double                this_time;
double                elapsed_secs;
#ifdef SYSV
struct tms            tms_buf;
#else
static struct timeb   timeb_old_buf;
struct timeb          timeb_buf;
#endif


/* if this is a deselection, just return */

if(op == PANEL_LIST_OP_DESELECT)
    {
#ifdef DEBUG
    printf("\nsave file scrolling list callback DESELECTed file **%s**\n",string);
#endif
    return;
    }

/* 
   this code added after a report of mouse button click bounces
   causing selection of two people consecutively
   so if the last selection was too recent, just return 
*/

#ifdef SYSV  /* if system V then use gethrtime function... */

/* find the time with the gethrtime function (returns value in nanoseconds) */
this_time = (double) gethrtime();
elapsed_secs = ((this_time - last_time))/1000000000.0;
last_time = this_time;

#else /* ...otherwise assumes BSD and uses ftime */

/* find the time with the ftime function (returns a timeb struct) */
if(ftime(&timeb_buf) == -1)
    printf("ftime error!\n"); 
if(!initialise)
    {
    elapsed_secs = ((double) (timeb_buf.time - timeb_old_buf.time)) + 
                   (((double) timeb_buf.millitm) - 
                    ((double) timeb_old_buf.millitm))/1000.;
    }
else
    {
    elapsed_secs = 2.0;
    initialise = 0;
    }
timeb_old_buf = timeb_buf;
#endif

#ifdef DEBUG
printf("elapsed value: %f\n",elapsed_secs);
#endif

/* if the last selection was too recent, unset the selection and return */

if(elapsed_secs < FILE_MULTICLICK_TOLERANCE)
    {
#ifdef DEBUG
    printf("apparent double click ignored\n");
#endif
    return;
    }

strcpy(new_file_name, string);

/* if the last character in the string is a slash, lose it */

if(new_file_name[strlen(new_file_name) - 1] == '/')
    new_file_name[strlen(new_file_name) - 1] = '\0';

#ifdef DEBUG
printf("file name **%s** selected\n",new_file_name);
#endif

/* 
   check if the selected entry is a directory by trying to move into it!
   if O.K. rerun the scrolling list creation using the new directory
*/

if(chdir(new_file_name) == 0)
    {
    /* need to rerun the file selection setup routine */
    file_menu_selection(file_menu, save_file_item);
    return;
    }

/* unshow the "Save as" window */

if((return_code = xv_set(save_frame, XV_SHOW, FALSE, NULL)) != XV_OK)
    {
    printf("unable to unshow the 'Save as' window\n");
    }


#ifdef DEBUG
printf("file name **%s** selected\n",new_file_name);
#endif

/* call the database save function */

save_database();

/* create the new footer display string and set it */

sprintf(footer_string,"Current Database File: %s",file_name);
xv_set(fdb_frame, FRAME_LEFT_FOOTER, footer_string, NULL);

}

/*
   display the information of the selected spouse, making them the
   current person
*/

void display_spouse(item, string, client_data, op, event)
Panel_item      item;
char           *string;
caddr_t         client_data;
Panel_list_op   op;
Event          *event;
{
int                   i, selection;
static int            initialise = 1;
static double         last_time = 0.0;
double                this_time;
double                elapsed_secs;
#ifdef SYSV
struct tms            tms_buf;
#else
static struct timeb   timeb_old_buf;
struct timeb          timeb_buf;
#endif

/* update the text items before a change of current person */

update_text();

/* if this is a deselection, just return */

if(op == PANEL_LIST_OP_DESELECT)
    {
#ifdef DEBUG
    printf("display spouse callback DESELECTed **%s**\n",string);
#endif
    return;
    }

/* 
   this code added after a report of mouse button click bounces
   causing selection of two people consecutively
   so if the last selection was too recent, just return 
*/

#ifdef SYSV  /* if system V then use gethrtime function... */

/* find the time with the gethrtime function (returns value in nanoseconds) */
this_time = (double) gethrtime();
elapsed_secs = ((this_time - last_time))/1000000000.0;
last_time = this_time;

#else /* ...otherwise assumes BSD and uses ftime */

/* find the time with the ftime function (returns a timeb struct) */
if(ftime(&timeb_buf) == -1)
    printf("ftime error!\n"); 
if(!initialise)
    {
    elapsed_secs = ((double) (timeb_buf.time - timeb_old_buf.time)) + 
                   (((double) timeb_buf.millitm) - 
                    ((double) timeb_old_buf.millitm))/1000.;
    }
else
    {
    elapsed_secs = 1.0;
    initialise = 0;
    }
timeb_old_buf = timeb_buf;
#endif

#ifdef DEBUG
printf("elapsed value: %f\n",elapsed_secs);
#endif

/* determine the number of the list item selected */

for(i=0;i<entry[current_index]->number_of_spouses;i++)
    {
    if(xv_get(spouse_list, PANEL_LIST_SELECTED, i))
        selection = i;
    }

/* if the last selection was too recent, unset the selection and return */

if(elapsed_secs < MULTICLICK_TOLERANCE)
    {
#ifdef DEBUG
    printf("apparent double click ignored\n");
#endif
    xv_set(spouse_list, PANEL_LIST_SELECT, selection, FALSE, NULL);
    return;
    }

#ifdef DEBUG
    printf("display spouse callback SELECTed **%s** (%d)\n",string, selection);
#endif

/* make the selected spouse the current person */

current_index = entry[current_index]->spouse[selection];

#ifdef DEBUG
    printf("current index set to %d\n",current_index);
#endif

/* set the default file menu selection to save since something has changed */

xv_set(file_menu, MENU_DEFAULT, 3, NULL);

/* update the display */

update_panel_items();

}

/*
   display the information of the selected child, making them the
   current person
*/

void display_child(item, string, client_data, op, event)
Panel_item      item;
char           *string;
caddr_t         client_data;
Panel_list_op   op;
Event          *event;
{
int                   i, selection;
static int            initialise = 1;
static double         last_time = 0.0;
double                this_time;
double                elapsed_secs;
#ifdef SYSV
struct tms            tms_buf;
#else
static struct timeb   timeb_old_buf;
struct timeb          timeb_buf;
#endif

/* update the text items before a change of current person */

update_text();

/* if this is a deselection, just return */

if(op == PANEL_LIST_OP_DESELECT)
    {
#ifdef DEBUG
    printf("display child callback DESELECTed **%s**\n",string);
#endif
    return;
    }

/* if the last selection was too recent, just return */

#ifdef SYSV  /* if system V then use gethrtime function... */

/* find the time with the gethrtime function (returns value in nanoseconds) */
this_time = (double) gethrtime();
elapsed_secs = ((this_time - last_time))/1000000000.0;
last_time = this_time;

#else /* ...otherwise assumes BSD and uses ftime */

/* find the time with the ftime function (returns a timeb struct) */
if(ftime(&timeb_buf) == -1)
    printf("ftime error!\n"); 
if(!initialise)
    {
    elapsed_secs = ((double) (timeb_buf.time - timeb_old_buf.time)) + 
                   (((double) timeb_buf.millitm) - 
                    ((double) timeb_old_buf.millitm))/1000.;
    }
else
    {
    elapsed_secs = 1.0;
    initialise = 0;
    }
timeb_old_buf = timeb_buf;
#endif
#ifdef DEBUG
printf("elapsed value: %f\n",elapsed_secs);
#endif

/* determine the number of the list item selected */

for(i=0;i<entry[current_index]->number_of_children;i++)
    {
    if(xv_get(child_list, PANEL_LIST_SELECTED, i))
        selection = i;
    }

/* if the last selection was too recent, unset the selection and return */

if(elapsed_secs < MULTICLICK_TOLERANCE)
    {
#ifdef DEBUG
    printf("apparent double click ignored\n");
#endif
    xv_set(child_list, PANEL_LIST_SELECT, selection, FALSE, NULL);
    return;
    }

#ifdef DEBUG
    printf("display child callback SELECTed **%s** (%d)\n",string, selection);
#endif

/* make the selected spouse the current person */

current_index = entry[current_index]->child[selection];

#ifdef DEBUG
    printf("current index set to %d\n",current_index);
#endif

/* set the default file menu selection to save since something has changed */

xv_set(file_menu, MENU_DEFAULT, 3, NULL);

/* update the display */

update_panel_items();

}

/*
   a database entry has been selected from the entries scrolling list
*/

void entry_select(item, string, client_data, op, event)
Panel_item      item;
char           *string;
caddr_t         client_data;
Panel_list_op   op;
Event          *event;
{
Xv_object   return_code;
int         i, j, selection, index_chosen, num_choices;
int         old_mother_index, old_father_index, spouse_index, child_index;

/* update the text items before a change of current person */

update_text();

/* if this is a deselection, just return */

if(op == PANEL_LIST_OP_DESELECT)
    {
#ifdef DEBUG
    printf("select entry scrolling list callback DESELECTed file **%s**\n",string);
#endif
    return;
    }

#ifdef DEBUG
printf("Arrived in entry_select with selection nature %d, type %d\n",
                 selection_nature, selection_type);
#endif

/*
   unshow the select entry scrolling list 
*/

if((return_code = xv_set(entries_frame, XV_SHOW, FALSE, NULL)) != XV_OK)
    {
    printf("unable to unshow the 'Select Entry' window\n");
    }

/* 
   find the index of the selected entry
*/

/* first, find out how many elements currently in the entries scrolling list... */

num_choices = (int) xv_get(entries_list, PANEL_LIST_NROWS);

/* ...then determine the number of the list item selected... */

for(i=0;i<num_choices;i++)
    {
    if(xv_get(entries_list, PANEL_LIST_SELECTED, i))
        selection = i;
    }

/* ...and finally, locate the index of that list item */

index_chosen = entries_index[selection];

/* amend the appropriate records, depending on the selection nature and type */

if(selection_nature == SELECTION)
    {
    switch(selection_type)
        {
        case MOTHER:
#ifdef DEBUG
            printf("Mother selected - index: %d - name: %s %s\n",index_chosen,
                entry[index_chosen]->forenames, entry[index_chosen]->surname);
#endif

            /* if the current person already has a mother, deselect her */
            if(entry[current_index]->mother != -1)
                {
                /* 
                   loop through the old mothers children to find the position of the
                   current person
                */
                old_mother_index = entry[current_index]->mother;
                selection = 0;
                for(i=0;i<entry[old_mother_index]->number_of_children;i++)
                    if(entry[old_mother_index]->child[i] == current_index)
                        selection = i;
                /* check that the current person actually was a child of the old mother! */
                if(selection == entry[old_mother_index]->number_of_children)
                    {
                    /* indicate the problem */
                    sprintf(error_message,"Database Inconsistency - Current Person Was Not a Child of its Mother!");
                    show_error();
                    goto amend_m_records;
                    }
                 
                /* 
                   remove the current person from the list of its old mothers children 
                   shunting the remaining children down in the array as necessary
                */
                if(selection < entry[old_mother_index]->number_of_children - 1)
                    {
                    for(i=selection;i<entry[old_mother_index]->number_of_children - 1;i++)
                        entry[old_mother_index]->child[i] = entry[old_mother_index]->child[i+1];
                    } 
                entry[old_mother_index]->number_of_children -= 1;
                }
            amend_m_records:

            /* amend the current persons record */
            entry[current_index]->mother = index_chosen;

            /* amend the new mothers record */
            entry[index_chosen]->child[entry[index_chosen]->number_of_children] = 
                    current_index;
            entry[index_chosen]->number_of_children += 1;
            break;
        case FATHER:
#ifdef DEBUG
            printf("Father selected - index: %d - name: %s %s\n",index_chosen,
                entry[index_chosen]->forenames, entry[index_chosen]->surname);
#endif

            /* if the current person already has a father, deselect him */
            if(entry[current_index]->father != -1)
                {
                /* 
                   loop through the old fathers children to find the position of the
                   current person
                */
                old_father_index = entry[current_index]->father;
                selection = 0;
                for(i=0;i<entry[old_father_index]->number_of_children;i++)
                    if(entry[old_father_index]->child[i] == current_index)
                        selection = i;
                /* check that the current person actually was a child of the old father! */
                if(selection == entry[old_father_index]->number_of_children)
                    {
                    /* indicate the problem */
                    sprintf(error_message,"Database Inconsistency - Current Person Was Not a Child of its Father!");
                    show_error();
                    goto amend_f_records;
                    }
                 
                /* 
                   remove the current person from the list of its old fathers children 
                   shunting the remaining children down in the array as necessary
                */
                if(selection < entry[old_father_index]->number_of_children - 1)
                    {
                    for(i=selection;i<entry[old_father_index]->number_of_children - 1;i++)
                        entry[old_father_index]->child[i] = entry[old_father_index]->child[i+1];
                    } 
                entry[old_father_index]->number_of_children -= 1;
                }
            amend_f_records:

            /* amend the current persons record */
            entry[current_index]->father = index_chosen;

            /* amend the new fathers record */
            entry[index_chosen]->child[entry[index_chosen]->number_of_children] = 
                    current_index;
            entry[index_chosen]->number_of_children += 1;
            break;
        case SPOUSE:
            /* 
                increment the number of spouses of the current and selected persons 
                and make the appropriate entries in the spouse arrays
            */
            if(((entry[index_chosen]->number_of_spouses + 1) < MAX_SPOUSES) &&
                  ((entry[current_index]->number_of_spouses + 1) < MAX_SPOUSES))
                {
                entry[index_chosen]->spouse[entry[index_chosen]->number_of_spouses] = 
                                                                      current_index;
                entry[index_chosen]->number_of_spouses++;
                entry[current_index]->spouse[entry[current_index]->number_of_spouses] = 
                                                                      index_chosen;
                entry[current_index]->number_of_spouses++;
                }
            else
                {
                sprintf(error_message,"Too Many Spouses! - Increase MAX_SPOUSES in definitions.h and Recompile");
                show_error();
                }
            break;
        case CHILD:
            /* 
               increment the number of children for the current person, enter the
               chosen person in the child array and set the childs appropriate parent 
               entry
               two provisos: MAX_CHILDREN is not exceeded for the parent and the
               child must not have the appropriate parent set
            */
            if(entry[current_index]->number_of_children < MAX_CHILDREN) 
                {
                if(entry[current_index]->gender == MALE)  /* current person is father */
                    {
                    if(entry[index_chosen]->father != -1)
                        {
                        sprintf(error_message,"Selected Child Already has a Father! - Deselect Current Father First");
                        show_error();
                        break;
                        }
                    else
                        {
                        entry[current_index]->child[entry[current_index]->number_of_children] =
                                                                           index_chosen;
                        entry[current_index]->number_of_children++;
                        entry[index_chosen]->father = current_index;
                        }
                    }
                else
                    {
                    if(entry[index_chosen]->mother != -1)
                        {
                        sprintf(error_message,"Selected Child Already has a Mother! - Deselect Current Mother First");
                        show_error();
                        break;
                        }
                    else
                        {
                        entry[current_index]->child[entry[current_index]->number_of_children] =
                                                                           index_chosen;
                        entry[current_index]->number_of_children++;
                        entry[index_chosen]->mother = current_index;
                        }
                    }
                }
            else
                {
                sprintf(error_message,"Too Many Children! - Increase MAX_CHILDREN in definitions.h and Recompile");
                show_error();
                }
            break;
        case ANYBODY:
            /* make the selected person the current person */

            current_index = index_chosen;

            break;
        default:
            /* something has gone badly wrong if this code ever runs! */
            strcpy(error_message,"Major Software Embarrassment - Please Report This Error to the Author");
            show_error();
        }
    }
else /* if not a SELECTION then it must be a DESELECTION */
    {
    switch(selection_type)
        {
        case SPOUSE:
            /* deselect the specified spouse */
            /* 
               loop through the spouses spouses (sic) to find the position of the
               current person
            */
            spouse_index = index_chosen;
            selection = 0;
            for(j=0;j<entry[spouse_index]->number_of_spouses;j++)
                if(entry[spouse_index]->spouse[j] == current_index)
                    selection = j;
            /* check that the current person actually was a spouse of its spouse! */
            if(selection == entry[spouse_index]->number_of_spouses)
                {
                /* indicate the problem */
                sprintf(error_message,"Database Inconsistency - Current Person Was Not a Spouse of its Spouse!");
                show_error();
                }
            else
                {
                /* 
                   remove the current person from the list of its spouses spouses 
                   shunting the remaining spouses down in the array as necessary
                */

                if(selection < entry[spouse_index]->number_of_spouses - 1)
                    {
                    for(i=selection;i<entry[spouse_index]->number_of_spouses - 1;i++)
                        entry[spouse_index]->spouse[i] = entry[spouse_index]->spouse[i+1];
                    } 
                entry[spouse_index]->number_of_spouses -= 1;
                }
            /* 
               now remove the specified spouse from the list of the 
               current persons spouses
            */
            selection = 0;
            for(j=0;j<entry[current_index]->number_of_spouses;j++)
                if(entry[current_index]->spouse[j] == spouse_index)
                    selection = j;
            /* 
               check that the selected spouse actually was a spouse of 
               the current person! 
            */
            if(selection == entry[current_index]->number_of_spouses)
                {
                /* indicate the problem */
                sprintf(error_message,"Database Inconsistency - Specified Spouse Was Not a Spouse of the Current Person!");
                show_error();
                }
            else
                {
                /* 
                   remove the specified spouse from the list of the current persons 
                   spouses shunting the remaining spouses down in the array as necessary
                */

                if(selection < entry[current_index]->number_of_spouses - 1)
                    {
                    for(i=selection;i<entry[current_index]->number_of_spouses - 1;i++)
                        entry[current_index]->spouse[i] = entry[current_index]->spouse[i+1];
                    } 
                entry[current_index]->number_of_spouses -= 1;
                }
            break;
        case CHILD:

            child_index = index_chosen;

            /* check that the current person actually was the parent of its child! */

            if(((entry[current_index]->gender == MALE) && 
                  (entry[child_index]->father != current_index)) || 
                ((entry[current_index]->gender == FEMALE) && 
                  (entry[child_index]->mother != current_index)))
                {
                /* indicate the problem */
                sprintf(error_message,"Database Inconsistency - Parent of Specified Person Was Not Current Person!");
                show_error();
                }
            else
                {
                /* 
                   remove the current person from the appropriate parent 
                   entry of the child
                */
                if(entry[current_index]->gender == MALE)
                    entry[child_index]->father = -1;
                else
                    entry[child_index]->mother = -1;
                }

            /* 
               now remove the specified child from the list of the 
               current persons children
            */
            selection = 0;
            for(j=0;j<entry[current_index]->number_of_children;j++)
                if(entry[current_index]->child[j] == child_index)
                    selection = j;
            /* 
               check that the selected child actually was a child of 
               the current person! 
            */
            if(selection == entry[current_index]->number_of_children)
                {
                /* indicate the problem */
                sprintf(error_message,"Database Inconsistency - Specified Child Was Not a Child of the Current Person!");
                show_error();
                }
            else
                {
                /* 
                   remove the specified child from the list of the current persons 
                   children shunting the remaining children down in the array as necessary
                */

                if(selection < entry[current_index]->number_of_children - 1)
                    {
                    for(i=selection;i<entry[current_index]->number_of_children - 1;i++)
                        entry[current_index]->child[i] = entry[current_index]->child[i+1];
                    } 
                entry[current_index]->number_of_children -= 1;
                }
            break;
        default:
            /* something has gone badly wrong if this code ever runs! */
            strcpy(error_message,"Major Software Embarrassment - Please Report This Error to the Author");
            show_error();
        }
    }

/* set the default file menu selection to save since something has changed */

xv_set(file_menu, MENU_DEFAULT, 3, NULL);

/* update the display elements */

update_panel_items();

}
