/*=============================================================
 * show.c -- Curses version of display functions
 * Copyright(c) 1992-94 by T.T. Wetmore IV; all rights reserved
 *   2.3.4 - 24 Jun 93    2.3.5 - 24 Aug 93
 *   3.0.0 - 14 Sep 94
 *===========================================================*/

#include "standard.h"
#include "screen.h"
#include "table.h"
#include "gedcom.h"
#include "indiseq.h"

STRING person_display();

#define MAXOTHERS 70
typedef char LINESTRING[80];

static LINESTRING show_pers;
static LINESTRING show_birt;
static LINESTRING show_deat;
static LINESTRING show_fath;
static LINESTRING show_moth;

static LINESTRING show_husb;
static LINESTRING show_husb_birt;
static LINESTRING show_husb_deat;
static LINESTRING show_wife;
static LINESTRING show_wife_birt;
static LINESTRING show_wife_deat;
static LINESTRING show_marr;

static INT show_olen = 0;
static INT show_otop = 0;
static LINESTRING show_others[MAXOTHERS];

/*===============================================
 * init_display_indi -- Initialize display person
 *=============================================*/
init_display_indi (pers)
NODE pers;	/* person */
{
	INT nsp, nch, num, nm;
	STRING s;
	NODE fth = indi_to_fath(pers);
	NODE mth = indi_to_moth(pers);
	BOOLEAN one;

	ASSERT(pers);
	s = indi_to_name(pers, 60);
	sprintf(show_pers, "person: %s (%s)", s, rmvat(nxref(pers))+1);

	s = indi_to_event(pers, "BIRT", "  born: ", 77, FALSE);
	if (!s) s = indi_to_event(pers, "CHR", "  bapt: ", 77, FALSE);
	if (s) sprintf(show_birt, s);
	else sprintf(show_birt, "  born:");

	s = indi_to_event(pers, "DEAT", "  died: ", 77, FALSE);
	if (!s) s = indi_to_event(pers, "BURI", "  buri: ", 77, FALSE);
	if (s) sprintf(show_deat, s);
	else sprintf(show_deat, "  died:");

	s = person_display(fth, NULL, 67);
	if (s) sprintf(show_fath, "  father: %s", s);
	else sprintf(show_fath, "  father:");

	s = person_display(mth, NULL, 67);
	if (s) sprintf(show_moth, "  mother: %s", s);
	else sprintf(show_moth, "  mother:");

	show_olen = 0;
	one = num_spouses(pers) == 1;
	nsp = nch = 0;
	FORFAMILIES(pers, fam, sp, num)
		if (sp) add_spouse_line(++nsp, sp, fam, one);
		FORCHILDREN(fam, chld, nm)
			add_child_line(++nch, chld);
		ENDCHILDREN
	ENDFAMILIES
}
/*==============================
 * show_person -- Display person
 *============================*/
show_person (pers, win, row, hgt)
NODE pers;	/* person */
WINDOW *win;	/* window */
INT row;	/* start row */
INT hgt;	/* avail rows */
{
	INT i;
	init_display_indi(pers);
	for (i = 0; i < hgt; i++) {
		wmove(win, row+i, 1);
		wclrtoeol(win);
#ifndef BSD
		mvwaddch(win, row+i, 79, ACS_VLINE);
#endif
	}
	mvwaddstr(win, row+0, 1, show_pers);
	mvwaddstr(win, row+1, 1, show_birt);
	mvwaddstr(win, row+2, 1, show_deat);
	mvwaddstr(win, row+3, 1, show_fath);
	mvwaddstr(win, row+4, 1, show_moth);
	show_otop = 0;
	for (i = 0; i < show_olen && i < hgt-5; i++)
		mvwaddstr(win, row+5+i, 1, show_others[i]);
}
/*=============================================
 * add_spouse_line -- Add spouse line to others
 *===========================================*/
add_spouse_line (num, indi, fam, one)
INT num;
NODE indi, fam;
BOOLEAN one;
{
	STRING line;
	if (show_olen >= MAXOTHERS) return;
	line = person_display(indi, fam, one?67:66);
#if 0
	if (one)
#endif
		sprintf(show_others[show_olen], "  spouse: %s", line);
#if 0
	else
		sprintf(show_others[show_olen], "  spouse%d: %s", num, line);
#endif
	show_others[show_olen++][78] = 0;
}
/*===========================================
 * add_child_line -- Add child line to others
 *=========================================*/
add_child_line (num, indi)
INT num;
NODE indi;
{
	STRING line;
	if (show_olen >= MAXOTHERS) return;
	line = person_display(indi, NULL, 65);
#if 0
sprintf(show_others[show_olen], "   child%d: %s", num, line);
#endif
	sprintf(show_others[show_olen], "    child: %s", line);
	show_others[show_olen++][78] = 0;
}
/*==============================================
 * init_display_fam -- Initialize display family
 *============================================*/
init_display_fam (fam)
NODE fam;	/* family */
{
	NODE husb = fam_to_husb(fam);
	NODE wife = fam_to_wife(fam);
	STRING s, ik, fk;
	INT len, nch, nm;

	ASSERT(fam);
	if (husb) {
		ik = rmvat(nxref(husb)) + 1;
		fk = rmvat(nxref(fam))  + 1;
		len = 64 - (strlen(ik) + strlen(fk));
		s = indi_to_name(husb, len);
		sprintf(show_husb, "father: %s (%s) (%s)", s, ik, fk);
	} else
		sprintf(show_husb, "father:");

	s = indi_to_event(husb, "BIRT", "  born: ", 77, FALSE);
	if (!s) s = indi_to_event(husb, "CHR", "  bapt: ", 77, FALSE);
	if (s) sprintf(show_husb_birt, s);
	else sprintf(show_husb_birt, " born:");

	s = indi_to_event(husb, "DEAT", "  died: ", 77, FALSE);
	if (!s) s = indi_to_event(husb, "BURI", "  buri: ", 77, FALSE);
	if (s) sprintf(show_husb_deat, s);
	else sprintf(show_husb_deat, "  died:");

	if (wife) {
		ik = rmvat(nxref(wife)) + 1;
		len = 67 - strlen(ik);
		s = indi_to_name(wife, len);
		sprintf(show_wife, "mother: %s (%s)", s, ik);
	} else
		sprintf(show_wife, "mother:");

	s = indi_to_event(wife, "BIRT", "  born: ", 77, FALSE);
	if (!s) s = indi_to_event(wife, "CHR", " bapt: ", 77, FALSE);
	if (s) sprintf(show_wife_birt, s);
	else sprintf(show_wife_birt, "  born:");

	s = indi_to_event(wife, "DEAT", "  died: ", 77, FALSE);
	if (!s) s = indi_to_event(wife, "BURI", " buri: ", 77, FALSE);
	if (s) sprintf(show_wife_deat, s);
	else sprintf(show_wife_deat, "  died:");

	s = indi_to_event(fam, "MARR", "married: ", 78, FALSE);
	if (s) sprintf(show_marr, s);
	else sprintf(show_marr, "married:");

	show_olen = 0;
	nch = 0;
	FORCHILDREN(fam, chld, nm)
		add_child_line(++nch, chld);
	ENDCHILDREN
}
/*===================================
 * show_long_family -- Display family
 *=================================*/
show_long_family (fam, win, row, hgt)
NODE fam;
WINDOW *win;
INT row;
INT hgt;
{
	INT i;
	init_display_fam(fam);
	for (i = 0; i < hgt; i++) {
		wmove(win, row+i, 1);
		wclrtoeol(win);
#ifndef BSD
		mvwaddch(win, row+i, COLS-1, ACS_VLINE);
#endif
	}
	mvwaddstr(win, row+0, 1, show_husb);
	mvwaddstr(win, row+1, 1, show_husb_birt);
	mvwaddstr(win, row+2, 1, show_husb_deat);
	mvwaddstr(win, row+3, 1, show_wife);
	mvwaddstr(win, row+4, 1, show_wife_birt);
	mvwaddstr(win, row+5, 1, show_wife_deat);
	mvwaddstr(win, row+6, 1, show_marr);
	show_otop = 0;
	for (i = 0; i < show_olen && i < hgt-7; i++)
		mvwaddstr(win, row+7+i, 1, show_others[i]+1);
}
/*=====================================
 * show_short_family -- Display family.
 *===================================*/
show_short_family (fam, win, row, hgt)
NODE fam;
WINDOW *win;
INT row;
INT hgt;
{
	INT i;
	init_display_fam(fam);
	for (i = 0; i < hgt; i++) {
		wmove(win, row+i, 1);
		wclrtoeol(win);
#ifndef BSD
		mvwaddch(win, row+i, COLS-1, ACS_VLINE);
#endif
	}
	mvwaddstr(win, row+0, 1, show_husb);
	if (strlen(show_husb_birt) > 5)
		mvwaddstr(win, row+1, 1, show_husb_birt);
	else if (strlen(show_husb_deat) > 5)
		mvwaddstr(win, row+1, 1, show_husb_deat);
	else
		mvwaddstr(win, row+1, 1, show_husb_birt);

	mvwaddstr(win, row+2, 1, show_wife);
	if (strlen(show_wife_birt) > 5)
		mvwaddstr(win, row+3, 1, show_wife_birt);
	else if (strlen(show_wife_deat) > 5)
		mvwaddstr(win, row+3, 1, show_wife_deat);
	else
		mvwaddstr(win, row+3, 1, show_wife_birt);

	mvwaddstr(win, row+4, 1, show_marr);
	show_otop = 0;
	for (i = 0; i < show_olen && i < hgt-5; i++)
		mvwaddstr(win, row+5+i, 1, show_others[i]+1);
}
/*=================================================
 * show_pedigree -- Show person in pedigree format.
 *===============================================*/
show_pedigree (indi)
NODE indi;
{
	NODE f = indi_to_fath(indi);	/* 2nd generation */
	NODE m = indi_to_moth(indi);
	NODE ff = indi_to_fath(f);	/* 3rd generation */
	NODE fm = indi_to_moth(f);
	NODE mf = indi_to_fath(m);
	NODE mm = indi_to_moth(m);
	NODE fff = indi_to_fath(ff);	/* 4th generation */
	NODE ffm = indi_to_moth(ff);
	NODE fmf = indi_to_fath(fm);
	NODE fmm = indi_to_moth(fm);
	NODE mff = indi_to_fath(mf);
	NODE mfm = indi_to_moth(mf);
	NODE mmf = indi_to_fath(mm);
	NODE mmm = indi_to_moth(mm);
	STRING indi_to_ped_fix();
	WINDOW *w = main_win;
	INT i;

	for (i = 1; i <= 15; i++) {
		wmove(w, i, 1);
		wclrtoeol(w);
#ifndef BSD
		mvwaddch(w, i, COLS-1, ACS_VLINE);
#endif
	}
	mvwprintw(w, 1, 2, "                  %s", indi_to_ped_fix(fff, 52));
	mvwprintw(w, 2, 2, "            %s", indi_to_ped_fix(ff, 60));
	mvwprintw(w, 3, 2, "                  %s", indi_to_ped_fix(ffm, 52));
	mvwprintw(w, 4, 2, "      %s", indi_to_ped_fix(f, 68));
	mvwprintw(w, 5, 2, "                  %s", indi_to_ped_fix(fmf, 52));
	mvwprintw(w, 6, 2, "            %s", indi_to_ped_fix(fm, 60));
	mvwprintw(w, 7, 2, "                  %s", indi_to_ped_fix(fmm, 52));
 	mvwprintw(w, 8, 2, "%s",indi_to_ped_fix(indi, 68));
	mvwprintw(w, 9, 2, "                  %s", indi_to_ped_fix(mff, 52));
	mvwprintw(w, 10, 2, "            %s", indi_to_ped_fix(mf, 60));
	mvwprintw(w, 11, 2, "                  %s", indi_to_ped_fix(mfm, 52));
	mvwprintw(w, 12, 2, "      %s", indi_to_ped_fix(m, 68));
	mvwprintw(w, 13, 2, "                  %s", indi_to_ped_fix(mmf, 52));
	mvwprintw(w, 14, 2, "            %s", indi_to_ped_fix(mm, 60));
	mvwprintw(w, 15, 2, "                  %s", indi_to_ped_fix(mmm, 52));
}
/*================================================================
 * indi_to_ped_fix -- Construct person STRING for pedigree screen.
 *==============================================================*/
STRING indi_to_ped_fix (indi, len)
NODE indi;
INT len;
{
	STRING bevt, devt, name, key;
	static char scratch[100];

	if (!indi) return "------------";
	bevt = event_to_date(BIRT(indi), TRUE);
	if (!bevt) bevt = event_to_date(BAPT(indi), TRUE);
	if (!bevt) bevt = "";
	devt = event_to_date(DEAT(indi), TRUE);
	if (!devt) devt = event_to_date(BURI(indi), TRUE);
	if (!devt) devt = "";
	name = indi_to_name(indi, len);
	if (keyflag) key = rmvat(nxref(indi)) + 1;
	if (keyflag)
		sprintf(scratch, "%s [%s-%s] (%s)", name, bevt, devt, key);
	else
		sprintf(scratch, "%s (%s-%s)", name, bevt, devt);
	return scratch;
}
/*=============================================
 * person_display -- Create person display line
 *===========================================*/
STRING person_display (indi, fam, len)
NODE indi;
NODE fam;
INT len;
{
	static char scratch[100];
	STRING p = scratch, evt = NULL;
	NODE chld;

	if (!indi) return NULL;
	strcpy(p, indi_to_name(indi, 40));
	p += strlen(p);
	if (fam) {
		evt = fam_to_event(fam, "MARR", "m. ", 35, TRUE);
		if (!evt) {
			if (chld = fam_to_first_chil(fam)) {
				evt = indi_to_event(chld, "BIRT", "cb. ",
				    35, TRUE);
				if (!evt) evt = indi_to_event(chld, "CHR",
					    "cb. ", 35, TRUE);
			}
		}
		if (!evt) evt = indi_to_event(indi, "BIRT", "b. ", 35, TRUE);
		if (!evt) evt = indi_to_event(indi, "CHR", "bap. ", 35, TRUE);
		if (!evt) evt = indi_to_event(indi, "DEAT", "d. ", 35, TRUE);
		if (!evt) evt = indi_to_event(indi, "BURI", "bur. ", 35, TRUE);
	} else {
		evt = indi_to_event(indi, "BIRT", "b. ", 20, TRUE);
		if (!evt) evt = indi_to_event(indi, "CHR", "bap. ", 20, TRUE);
		if (evt) {
			sprintf(p, ", %s", evt);
			p += strlen(p);
		}
		evt = indi_to_event(indi, "DEAT", "d. ", 20, TRUE);
		if (!evt) evt = indi_to_event(indi, "BURI", "bur. ", 20, TRUE);
	}
	if (evt) {
		sprintf(p, ", %s", evt);
		p += strlen(p);
	}
	sprintf(p, " (%s)", rmvat(nxref(indi)) + 1);
	scratch[66] = 0;
	return scratch;
}

static STRING empstr = "                                                 ";
/*==========================================
 * show_list - Show name list in list screen
 *========================================*/
#define VIEWABLE 13
show_list (seq, top, cur, mark)
INDISEQ seq;
INT top, cur, mark;
{
	WINDOW *win = main_win;
	INT i, j, row, len = length_indiseq(seq);
	STRING key, name;
	NODE indi;
	char scratch[200], *p;

	for (i = 8; i <= 20; i++)
		mvwaddstr(win, i, 1, empstr);
	row = 8;
	for (i = top, j = 0; j < VIEWABLE && i < len; i++, j++) {
		element_indiseq(seq, i, &key, &name);
		indi = key_to_indi(key);
		if (i == mark) mvwaddch(win, row, 2, 'x');
		if (i == cur) {
			mvwaddch(win, row, 3, '>');
			show_person(indi, win, 1, 6);
		}
		name = manip_name(name, TRUE, TRUE, 40);
		strcpy(scratch, name);
		p = scratch + strlen(scratch);
		*p++ = ' ';
		sprintf(p, "(%s)", &key[1]);
		mvwaddstr(win, row, 4, scratch);
		row++;
	}
}
