/*================================================================
 * intrpseq.c -- Programming interface to the INDISEQ data type.
 * Copyright(c) 1992 by Thomas T. Wetmore IV; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 21 Aug 93 - modified
 *================================================================
 */
#include <stdio.h>
#include "standard.h"
#include "table.h"
#include "gedcom.h"
#include "interp.h"
#include "indiseq.h"

LIST keysets;
INDISEQ union_indiseq(), intersect_indiseq(), child_indiseq();
INDISEQ parent_indiseq(), spouse_indiseq(), ancestor_indiseq();
INDISEQ descendent_indiseq(), difference_indiseq();
INDISEQ sibling_indiseq();

/*========================================================
 * initset -- Initialize list that holds created INDISEQs.
 *======================================================*/
initset ()
{
	keysets = create_list();
}
/*=========================================
 * _indiset -- Declare an INDISEQ variable.
 *   usage: indiset(VARB) -> VOID
 *=======================================*/
WORD _indiset (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq;
	INTERP var = (INTERP) ielist(node);
	*eflg = TRUE;
	if (!iistype(var, IIDENT)) return NULL;
	*eflg = FALSE;
	seq = create_indiseq();
	assign_iden(stab, iident(var), seq);
	push_list(keysets, seq);
	return NULL;
}
/*==========================================
 * _addtoset -- Add person to INDISEQ.
 *   usage: addtoset(SET, INDI, ANY) -> VOID
 *========================================*/
WORD _addtoset (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg1 = (INTERP) ielist(node), arg2 = inext(arg1),
	    arg3 = inext(arg2);
	INDISEQ seq = (INDISEQ) evaluate(arg1, stab, eflg);
	NODE indi;
	STRING key;
	WORD any;
	if (*eflg || !seq) return NULL;
	indi = (NODE) eval_indi(arg2, stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	*eflg = TRUE;
	key = strsave(rmvat(nxref(indi)));
	if (!seq || !key) return NULL;
	any = (WORD) evaluate(arg3, stab, eflg);
	if (*eflg) return NULL;
	append_indiseq(seq, key, NULL, any, FALSE, TRUE);
	return NULL;
}
/*=========================================
 * _lengthset -- Find length of an INDISEQ.
 *   usage: lengthset(SET) -> INT
 *=======================================*/
WORD _lengthset (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	if (*eflg || !seq) return NULL;
	return (WORD) length_indiseq(seq);
}
/*================================================
 * _deletefromset -- Remove person from INDISEQ.
 *   usage: deletefromset(SET, INDI, BOOL) -> VOID
 *==============================================*/
WORD _deletefromset (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg1 = (INTERP) ielist(node), arg2 = inext(arg1),
	    arg3 = inext(arg2);
	INDISEQ seq = (INDISEQ) evaluate(arg1, stab, eflg);
	NODE indi;
	STRING key;
	BOOLEAN all, rc;
	if (*eflg) return NULL;
	indi = (NODE) eval_indi(arg2, stab, eflg, NULL);
	if (*eflg) return NULL;
	*eflg = TRUE;
	key = rmvat(nxref(indi));
	if (!seq || !key) return NULL;
	all = (BOOLEAN) evaluate(arg3, stab, eflg);
	if (*eflg) return NULL;
	do {
		rc = delete_indiseq(seq, key, NULL, 0);
	} while (rc && all);
	return NULL;
}
/*===================================
 * _namesort -- Sort INDISEQ by name.
 *   usage: namesort(SET) -> VOID
 *=================================*/
WORD _namesort (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	if (*eflg || !seq) return NULL;
	namesort_indiseq(seq);
	return NULL;
}
/*=================================
 * _keysort -- Sort INDISEQ by key.
 *   usage: keysort(SET) -> VOID
 *===============================*/
WORD _keysort (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	if (*eflg || !seq) return NULL;
	keysort_indiseq(seq);
	return NULL;
}
/*=====================================
 * _valuesort -- Sort INDISEQ by value.
 *   usage: valuesort(SET) -> VOID
 *===================================*/
WORD _valuesort (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	if (*eflg || !seq) return NULL;
	valuesort_indiseq(seq);
	return NULL;
}
/*===========================================
 * _uniqueset -- Eliminate dupes from INDISEQ.
 *   usage: uniqueset(SET) -> VOID
 *=========================================*/
WORD _uniqueset (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	if (*eflg || !seq) return NULL;
	return (WORD) unique_indiseq(seq);
}
/*========================================
 * _union -- Create union of two INDISEQs.
 *   usage: union(SET, SET) -> SET
 *======================================*/
WORD _union (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg1 = (INTERP) ielist(node), arg2 = inext(arg1);
	INDISEQ op2, op1 = (INDISEQ) evaluate(arg1, stab, eflg);
	if (*eflg) return NULL;
	op2 = (INDISEQ) evaluate(arg2, stab, eflg);
	if (*eflg || !op1 || !op2) return NULL;
	op2 = union_indiseq(op1, op2);
	push_list(keysets, op2);
	return (WORD) op2;
}
/*===================================================
 * _intersect -- Create intersection of two INDISEQs.
 *   usage: intersect(SET, SET) -> SET
 *=================================================*/
WORD _intersect (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg1 = (INTERP) ielist(node), arg2 = inext(arg1);
	INDISEQ op2, op1 = (INDISEQ) evaluate(arg1, stab, eflg);
	if (*eflg) return NULL;
	op2 = (INDISEQ) evaluate(arg2, stab, eflg);
	if (*eflg || !op1 || !op2) return NULL;
	op2 = intersect_indiseq(op1, op2);
	push_list(keysets, op2);
	return (WORD) op2;
}
/*==================================================
 * _difference -- Create difference of two INDISEQs.
 *   usage: difference(SET, SET) -> SET
 *================================================*/
WORD _difference (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg1 = (INTERP) ielist(node), arg2 = inext(arg1);
	INDISEQ op2, op1 = (INDISEQ) evaluate(arg1, stab, eflg);
	if (*eflg) return NULL;
	op2 = (INDISEQ) evaluate(arg2, stab, eflg);
	if (*eflg || !op1 || !op2) return NULL;
	op2 = difference_indiseq(op1, op2);
	push_list(keysets, op2);
	return (WORD) op2;
}
/*============================================
 * _parentset -- Create parent set of INDISEQ.
 *   usage: parentset(SET) -> SET
 *==========================================*/
WORD _parentset (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	if (*eflg || !seq) return NULL;
	seq = parent_indiseq(seq);
	push_list(keysets, seq);
	return (WORD) seq;
}
/*=============================================
 * _childset -- Create child set of an INDISEQ.
 *   usage: childset(SET) -> SET
 *===========================================*/
WORD _childset (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	if (*eflg || !seq) return NULL;
	seq = child_indiseq(seq);
	push_list(keysets, seq);
	return (WORD) seq;
}
/*=================================================
 * _siblingset -- Create sibling set of an INDISEQ.
 *   usage: siblingset(SET) -> SET
 *===============================================*/
WORD _siblingset (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	if (*eflg || !seq) return NULL;
	seq = sibling_indiseq(seq, TRUE);
	push_list(keysets, seq);
	return (WORD) seq;
}
/*===============================================
 * _spouseset -- Create spouse set of an INDISEQ.
 *   usage: spouseset(SET) -> SET
 *=============================================*/
WORD _spouseset (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	if (*eflg || !seq) return NULL;
	seq = spouse_indiseq(seq);
	push_list(keysets, seq);
	return (WORD) seq;
}
/*===================================================
 * _ancestorset -- Create ancestor set of an INDISEQ.
 *   usage: ancestorset(SET) -> SET
 *=================================================*/
WORD _ancestorset (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	if (*eflg || !seq) return NULL;
	seq = ancestor_indiseq(seq);
	push_list(keysets, seq);
	return (WORD) seq;
}
/*=======================================================
 * _descendentset -- Create descendent set of an INDISEQ.
 *   usage: descendentset(SET) -> SET
 *   usage: descendantset(SET) -> SET
 *=====================================================*/
WORD _descendentset (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	if (*eflg || !seq) return NULL;
	seq = descendent_indiseq(seq);
	push_list(keysets, seq);
	return (WORD) seq;
}
/*======================================================
 * _gengedcom -- Generate GEDCOM output from an INDISEQ.
 *   usage: gengedcom(SET) -> VOID
 *====================================================*/
WORD _gengedcom (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INDISEQ seq = (INDISEQ) evaluate(ielist(node), stab, eflg);
	if (*eflg || !seq) return NULL;
	gen_gedcom(seq);
	return NULL;
}
