/*=============================================================
 * builtin.c -- Many interpreter builtin functions
 * Copyright(c) 1992-94 by T.T. Wetmore IV; all rights reserved
 *   2.3.4 - 24 Jun 93    2.3.5 - 07 Sep 93
 *   3.0.0 - 07 May 94
 *===========================================================*/

#include "standard.h"
#include "table.h"
#include "gedcom.h"
#include "cache.h"
#include "interp.h"
#include "indiseq.h"

extern STRING llprograms;

/*==========================================
 * _getint -- Have user provide integer
 *   usage: getint(IDEN [,STRING]) --> VOID
 *   usage: getintmsg(IDEN, STRING) --> VOID
 *========================================*/
WORD _getint (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	INT val;
	STRING ttl = "Enter integer for report program";
	*eflg = TRUE;
	if (!iistype(arg, IIDENT)) return NULL;
	*eflg = FALSE;
	if (inext(arg)) ttl = (STRING) evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	val = ask_for_int(ttl);
	assign_iden(stab, iident(arg), (WORD) val);
	return NULL;
}
/*===========================================
 * _getstr -- Have user provide string
 *   usage: getstr(IDEN [,STRING]) --> VOID
 *   usage: getstrmsg(IDEN, STRING]) --> VOID
 *=========================================*/
WORD _getstr (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	STRING val;
	static STRING ttl = "Enter string for report program";
	*eflg = TRUE;
	if (!iistype(arg, IIDENT)) return NULL;
	*eflg = FALSE;
	if (inext(arg)) ttl = (STRING) evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	val = (STRING) ask_for_string(ttl, "enter string: ");
	assign_iden(stab, iident(arg), (WORD) val);
	return NULL;
}
/*===========================================
 * _getindi -- Have user identify person
 *   usage: getindi(IDEN [,STRING]) --> VOID
 *   usage: getindimsg(IDEN, STRING) --> VOID
 *=========================================*/
WORD _getindi (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	CACHEEL cel;
	STRING ttl = "Identify person for report program:";
	*eflg = TRUE;
	if (!iistype(arg, IIDENT)) return NULL;
	*eflg = FALSE;
	if (inext(arg)) ttl = (STRING) evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	cel = key_to_indi_cacheel(ask_for_indi_key(ttl, FALSE, TRUE));
	assign_iden(stab, iident(arg), (WORD) cel);
	return NULL;
}
/*=========================================
 * _getfam -- Have user identify family
 *   usage: getfam(IDEN [,STRING]) --> VOID
 *=======================================*/
static STRING chfamily = "Choose family by selecting spouse:";
WORD _getfam (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	CACHEEL cel = NULL;
	NODE fam;
	STRING ttl = "Identify family for report program:";
	*eflg = TRUE;
	if (!iistype(arg, IIDENT)) return NULL;
	*eflg = FALSE;
	if (inext(arg)) ttl = (STRING) evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	fam = ask_for_fam("Enter a spouse from family.",
	    "Enter a sibling from family.");
	if (fam) cel = fam_to_cacheel(fam);
	assign_iden(stab, iident(arg), (WORD) cel);
	return NULL;
}
/*=================================================
 * _getindiset -- Have user identify set of persons
 *   usage: getindiset(IDEN [,STRING]) --> VOID
 *===============================================*/
WORD _getindiset (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	INDISEQ seq;
	STRING ttl = "Identify person list for report program:";
	*eflg = TRUE;
	if (!iistype(arg, IIDENT)) return NULL;
	*eflg = FALSE;
	if (inext(arg)) ttl = (STRING) evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	seq = (INDISEQ) ask_for_indi_list(ttl, TRUE);
	assign_iden(stab, iident(arg), (WORD) seq);
	return NULL;
}
/*==================================
 * _gettoday -- Create today's event
 *   usage: gettoday() --> EVENT
 *================================*/
WORD _gettoday (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE prnt = create_node(NULL, "EVEN", NULL, NULL);
	NODE chil = create_node(NULL, "DATE", get_date(), prnt);
	nchild(prnt) = chil;
	*eflg = FALSE;
	return (WORD) prnt;
}
/*======================================
 * _name -- Find person's name
 *   usage: name(INDI [,BOOL]) -> STRING
 *====================================*/
WORD _name (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	NODE indi = eval_indi(arg, stab, eflg, NULL);
	BOOLEAN caps = TRUE;
	if (*eflg || !indi) return NULL;
	if (inext(arg))
		caps = (BOOLEAN) evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	indi = find_tag(nchild(indi), "NAME");
	return manip_name(nval(indi), caps, TRUE, 68);
}
/*===================================================
 * _fullname -- Process person's name
 *   usage: fullname(INDI, BOOL, BOOL, INT) -> STRING
 *=================================================*/
WORD _fullname (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	NODE name, indi = eval_indi(arg, stab, eflg, NULL);
	BOOLEAN caps, reg;
	INT len;
	if (*eflg || !indi) return NULL;
	*eflg = FALSE;
	caps = (BOOLEAN) evaluate(arg = inext(arg), stab, eflg);
	if (*eflg) return NULL;
	reg = (BOOLEAN) evaluate(arg = inext(arg), stab, eflg);
	if (*eflg) return NULL;
	len = (INT) evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	if (!(name = NAME(indi)) || !nval(name)) {
		*eflg = TRUE;
		return NULL;
	}
	return manip_name(nval(name), caps, reg, len);
}
/*==================================
 * _surname -- Find person's surname
 *   usage: surname(INDI) -> STRING
 *================================*/
WORD _surname (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE this = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !this) return NULL;
	if (!(this = NAME(this)) || !nval(this)) return "ERROR";
	return (WORD) getsurname(nval(this));
}
/*========================================
 * _soundex -- SOUNDEX function on persons
 *   usage: soundex(INDI) -> STRING
 *======================================*/
WORD _soundex (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE this = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !this) return NULL;
	if (!(this = NAME(this)) || !nval(this)) return "ERROR";
	return (WORD) soundex(getsurname(nval(this)));
}
/*===========================================
 * _strsoundex -- SOUNDEX function on strings
 *   usage: strsoundex(STRING) -> STRING
 *=========================================*/
WORD _strsoundex (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	STRING str = (STRING) evaluate(ielist(node), stab, eflg);
	if (*eflg || !str) return NULL;
	return (WORD) soundex(str);
}
/*================================
 * _givens -- Find given names
 *   usage: givens(INDI) -> STRING
 *==============================*/
WORD _givens (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE this = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !this) return NULL;
	if (!(this = NAME(this)) || !nval(this)) return "ERROR";
	return (WORD) givens(nval(this));
}
/*================================
 * _set -- Assignment operation
 *   usage: set(IDEN, ANY) -> VOID
 *==============================*/
WORD _set (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP var = (INTERP) ielist(node);
	INTERP expr = inext(var);
	WORD value;
	*eflg = TRUE;
	if (!iistype(var, IIDENT)) return NULL;
	value = evaluate(expr, stab, eflg);
	if (*eflg) return NULL;
	assign_iden(stab, iident(var), value);
	return NULL;
}
/*===================================
 * _husband -- Find husband of family
 *   usage: husband(FAM) -> INDI
 *=================================*/
WORD _husband (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE fam = eval_fam(ielist(node), stab, eflg, NULL);
	if (*eflg || !fam) return NULL;
	return (WORD) indi_to_cacheel(fam_to_husb(fam));
}
/*=============================
 * _wife -- Find wife of family
 *   usage: wife(FAM) -> INDI
 *===========================*/
WORD _wife (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE fam = eval_fam(ielist(node), stab, eflg, NULL);
	if (*eflg || !fam) return NULL;
	return (WORD) indi_to_cacheel(fam_to_wife(fam));
}
/*==========================================
 * _firstchild -- Find first child of family
 *   usage: firstchild(FAM) -> INDI
 *========================================*/
WORD _firstchild (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE fam = eval_fam(ielist(node), stab, eflg, NULL);
	if (*eflg || !fam) return NULL;
	return (WORD) indi_to_cacheel(fam_to_first_chil(fam));
}
/*========================================
 * _lastchild -- Find last child of family
 *   usage: lastchild(FAM) -> INDI
 *======================================*/
WORD _lastchild (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE fam = eval_fam(ielist(node), stab, eflg, NULL);
	if (*eflg || !fam) return NULL;
	return (WORD) indi_to_cacheel(fam_to_last_chil(fam));
}
/*=================================
 * _marr -- Find marriage of family
 *   usage: marriage(FAM) -> EVENT
 *===============================*/
WORD _marr (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE fam = eval_fam(ielist(node), stab, eflg, NULL);
	if (*eflg || !fam) return NULL;
	return (WORD) MARR(fam);
}
/*==============================
 * _birt -- Find birth of person
 *   usage: birth(INDI) -> EVENT
 *============================*/
WORD _birt (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	return (WORD) BIRT(indi);
}
/*==============================
 * _deat -- Find death of person
 *   usage: death(INDI) -> EVENT
 *============================*/
WORD _deat (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	return (WORD) DEAT(indi);
}
/*================================
 * _bapt -- Find baptism of person
 *   usage: baptism(INDI) -> EVENT
 *==============================*/
WORD _bapt (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	return (WORD) BAPT(indi);
}
/*===============================
 * _buri -- Find burial of person
 *   usage: burial(INDI) -> EVENT
 *=============================*/
WORD _buri (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	return (WORD) BURI(indi);
}
/*===============================
 * _titl -- Find title of person
 *   usage: title(INDI) -> STRING
 *=============================*/
WORD _titl (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	indi = find_tag(nchild(indi), "TITL");
	return (WORD) (indi ? nval(indi) : NULL);
}
/*===================================
 * _long -- Return long form of event
 *   usage: long(EVENT) -> STRING
 *=================================*/
WORD _long (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE event = (NODE) evaluate(ielist(node), stab, eflg);
	if (*eflg || !event) return NULL;
	return (WORD) event_to_string(event, FALSE);
}
/*=====================================
 * _short -- Return short form of event
 *   usage: short(EVENT) -> STRING
 *===================================*/
WORD _short (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE event = (NODE) evaluate(ielist(node), stab, eflg);
	if (*eflg || !event) return NULL;
	return (WORD) event_to_string(event, TRUE);
}
/*================================
 * _fath -- Find father of person
 *   usage: father(INDI) -> INDI
 *==============================*/
WORD _fath (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	return (WORD) indi_to_cacheel(indi_to_fath(indi));
}
/*===============================
 * _moth -- Find mother of person
 *   usage: mother(INDI) -> INDI
 *=============================*/
WORD _moth (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	return (WORD) indi_to_cacheel(indi_to_moth(indi));
}
/*===========================================
 * _parents -- Find parents' family of person
 *   usage: parents(INDI) -> FAM
 *=========================================*/
WORD _parents (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	return (WORD) fam_to_cacheel(indi_to_famc(indi));
}
/*==========================================
 * _nextsib -- Find person's younger sibling
 *   usage: nextsib(INDI) -> INDI
 *========================================*/
WORD _nextsib (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	return (WORD) indi_to_cacheel(indi_to_next_sib(indi));
}
/*========================================
 * _prevsib -- Find person's older sibling
 *   usage: prevsib(INDI) -> INDI
 *======================================*/
WORD _prevsib (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	return (WORD) indi_to_cacheel(indi_to_prev_sib(indi));
}
/*========================================
 * _d -- Return cardinal integer as string
 *   usage: d(INT) -> STRING
 *======================================*/
WORD _d (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	static char scratch[20];
	WORD value = evaluate(ielist(node), stab, eflg);
	if (*eflg) return "";
	sprintf(scratch, "%d", value);
	return scratch;
}
/*==========================================
 * _alpha -- Convert small integer to letter
 *   usage: alpha(INT) -> STRING
 *========================================*/
WORD _alpha (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	static char scratch[2];
	INT value = (INT) evaluate(ielist(node), stab, eflg);
	if (*eflg) return NULL;
	if (value < 1 || value > 26) return "XX";
	sprintf(scratch, "%c", 'a' + value - 1);
	return scratch;
}
/*================================================
 * _ord -- Convert small integer to ordinal string
 *   usage: ord(INT) -> STRING
 *==============================================*/
static STRING ordinals[] = {
	"first", "second", "third", "fourth", "fifth",
	"sixth", "seventh", "eighth", "ninth", "tenth",
	"eleventh", "twelfth"
};
WORD _ord (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	static char scratch[12];
	INT value = (INT) evaluate(ielist(node), stab, eflg);
	if (*eflg || value < 1) return NULL;
	if (value > 12) {
		sprintf(scratch, "%dth", value);
		return (WORD) scratch;
	}
	return (WORD) ordinals[value - 1];
}
/*==================================================
 * _card -- Convert small integer to cardinal string
 *   usage: card(INT) -> STRING
 *================================================*/
static STRING cardinals[] = {
	"zero", "one", "two", "three", "four", "five",
	"six", "seven", "eight", "nine", "ten",
	"eleven", "twelve"
};
WORD _card (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	static char scratch[8];
	INT value = (INT) evaluate(ielist(node), stab, eflg);
	if (*eflg) return NULL;
	if (value < 0 || value > 12) {
		sprintf(scratch, "%d", value);
		return (WORD) scratch;
	}
	return (WORD) cardinals[value];
}
/*===========================================
 * _roman -- Convert integer to Roman numeral
 *   usage: roman(INT) -> STRING
 *=========================================*/
static STRING rodigits[] = {
	"", "i", "ii", "iii", "iv", "v", "vi", "vii", "viii", "ix"
};
static STRING rotens[] = {
	"", "x", "xx", "xxx", "xl", "l", "lx", "lxx", "lxxx", "xc"
};
WORD _roman (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	static char scratch[10];
	INT value = (INT) evaluate(ielist(node), stab, eflg);
	if (*eflg) return NULL;
	if (value < 1 || value >= 99) return _d(node, stab, eflg);
	sprintf(scratch, "%s%s", rotens[value/10], rodigits[value%10]);
	return scratch;
}
/*================================================
 * _nchildren -- Find number of children in family
 *   usage: nchildren(FAM) -> INT
 *==============================================*/
WORD _nchildren (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE fam = eval_fam(ielist(node), stab, eflg, NULL);
	if (*eflg || !fam) return NULL;
	return (WORD) node_list_length(CHIL(fam));
}
/*===================================================
 * _nfamilies -- Find number of families person is in
 *   usage: nfamilies(INDI) -> INT
 *=================================================*/
WORD _nfamilies (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	return (WORD) node_list_length(FAMS(indi));
}
/*===============================================
 * _nspouses -- Find number of spouses person has
 *   usage: nspouses(INDI) -> INT
 *=============================================*/
WORD _nspouses (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	INT nspouses;
	if (*eflg || !indi) return NULL;
	FORSPOUSES(indi,spouse,fam,nspouses) ENDSPOUSES
	return (WORD) nspouses;
}
/*==============================
 * _eq -- Equal operation
 *   usage: eq(ANY, ANY) -> BOOL
 *============================*/
WORD _eq (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INT val1, val2;
	eval_binary(node, &val1, &val2, stab, eflg);
	return (WORD) (val1 == val2);
}
/*==============================
 * _ne -- Not equal operation
 *   usage: ne(ANY, ANY) -> BOOL
 *============================*/
WORD _ne (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INT val1, val2;
	eval_binary(node, &val1, &val2, stab, eflg);
	return (WORD) (val1 != val2);
}
/*===============================
 * _le -- Less or equal operation
 *   usage: le(ANY, ANY) -> BOOL
 *=============================*/
WORD _le (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INT val1, val2;
	eval_binary(node, &val1, &val2, stab, eflg);
	return (WORD) (val1 <= val2);
}
/*==================================
 * _ge -- Greater or equal operation
 *   usage: ge(ANY, ANY) -> BOOL
 *================================*/
WORD _ge (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INT val1, val2;
	eval_binary(node, &val1, &val2, stab, eflg);
	return (WORD) (val1 >= val2);
}
/*=============================
 * _lt -- Less than operation
 *   usage: lt(ANY,ANY) -> BOOL
 *===========================*/
WORD _lt (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INT val1, val2;
	eval_binary(node, &val1, &val2, stab, eflg);
	return (WORD) (val1 < val2);
}
/*==============================
 * _gt -- Greater than operation
 *   usage: gt(ANY, ANY) -> BOOL
 *============================*/
WORD _gt (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INT val1, val2;
	eval_binary(node, &val1, &val2, stab, eflg);
	return (WORD) (val1 > val2);
}
/*==================================
 * _and -- And operation
 *   usage: and(ANY [,ANY]+) -> INT
 *================================*/
WORD _and (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	INT val = (INT) evaluate(arg, stab, eflg);
	if (*eflg) return NULL;
	if (!val) return (WORD) FALSE;
	while (arg = inext(arg)) {
		val = val && (INT) evaluate(arg, stab, eflg);
		if (*eflg) return NULL;
		if (!val) return (WORD) FALSE;
	}
	return (WORD) val;
}
/*================================
 * _or -- Or operation
 *   usage: or(ANY [,ANY]+) -> INT
 *==============================*/
WORD _or (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	INT val = (INT) evaluate(arg, stab, eflg);
	if (*eflg) return NULL;
	if (val) return (WORD) TRUE;
	while (arg = inext(arg)) {
		val = val || (INT) evaluate(arg, stab, eflg);
		if (*eflg) return NULL;
		if (val) return (WORD) TRUE;
	}
	return (WORD) val;
}
/*==================================
 * _add -- Add operation
 *   usage: add(INT [,INT]+) -> INT
 *================================*/
WORD _add (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	INT val = (INT) evaluate(arg, stab, eflg);
	if (*eflg) return NULL;
	while (arg = inext(arg)) {
		val += (INT) evaluate(arg, stab, eflg);
		if (*eflg) return NULL;
	}
	return (WORD) val;
}
/*==============================
 * _sub -- Subtract operation
 *   usage: sub(INT, INT) -> INT
 *============================*/
WORD _sub (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INT val1, val2;
	eval_binary(node, &val1, &val2, stab, eflg);
	return (WORD) (val1 - val2);
}
/*=================================
 * _mul -- Multiply operation
 *   usage: mul(INT [,INT]+) -> INT
 *===============================*/
WORD _mul (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	INT val = (INT) evaluate(arg, stab, eflg);
	if (*eflg) return NULL;
	while (arg = inext(arg)) {
		val *= (INT) evaluate(arg, stab, eflg);
		if (*eflg) return NULL;
	}
	return (WORD) val;
}
/*==============================
 * _div -- Divide operation
 *   usage: div(INT, INT) -> INT
 *============================*/
WORD _div (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INT val1, val2;
	eval_binary(node, &val1, &val2, stab, eflg);
	if (*eflg || val2 == 0) {
		*eflg = TRUE;
		return NULL;
	}
	return (WORD) (val1/val2);
}
/*==============================
 * _mod -- Modulus operation
 *   usage: mod(INT, INT) -> INT
 *============================*/
WORD _mod (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INT val1, val2;
	eval_binary(node, &val1, &val2, stab, eflg);
	if (*eflg || val2 == 0) {
		*eflg = TRUE;
		return NULL;
	}
	return (WORD) (val1%val2);
}
/*=================================
 * _exp -- Exponentiation operation
 *   usage: exp(INT, INT) -> INT
 *===============================*/
WORD _exp (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INT i, val1, val2, val3;
	eval_binary(node, &val1, &val2, stab, eflg);
	if (*eflg) return NULL;
	if (val2 < 0) {
		*eflg = TRUE;
		return NULL;
	}
	if (*eflg || val2 == 0) {
		*eflg = TRUE;
		return NULL;
	}
	val3 = 1;
	for (i = 0; i < val2; i++)
		val3 *= val1;
	return (WORD) val3;
}
/*===========================
 * _neg -- Negation operation
 *   usage: neg(INT) -> INT
 *=========================*/
WORD _neg (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	WORD value = evaluate(ielist(node), stab, eflg);
	if (*eflg) return NULL;
	return (WORD) - (INT) value;
}
/*============================
 * _incr -- Increment variable
 *   usage: incr(VARB) -> VOID
 *==========================*/
WORD _incr (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP var = (INTERP) ielist(node);
	INT val;
	*eflg = TRUE;
	if (!iistype(var, IIDENT)) return NULL;
	val = (INT) evaluate(var, stab, eflg);
	if (*eflg) return NULL;
	assign_iden(stab, iident(var), val + 1);
	return NULL;
}
/*============================
 * _decr -- Decrement variable
 *   usage: decr(VARB) -> VOID
 *==========================*/
WORD _decr (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP var = (INTERP) ielist(node);
	INT val;
	*eflg = TRUE;
	if (!iistype(var, IIDENT)) return NULL;
	val = (INT) evaluate(var, stab, eflg);
	if (*eflg) return NULL;
	assign_iden(stab, iident(var), val - 1);
	return NULL;
}
/*=======================================
 * _strcmp -- Compare two strings
 *   usage: strcmp(STRING, STRING) -> INT
 *=====================================*/
WORD _strcmp (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	STRING str2, str1 = (STRING) evaluate(arg, stab, eflg);
	STRING emp = "";
	if (*eflg) return NULL;
	str2 = (STRING) evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	if (!str1) str1 = emp;
	if (!str2) str2 = emp;
	return (WORD) strcmp(str1, str2);
}
/*=======================================
 * _strtoint -- Convert string to integer
 *  usage: strtoint(STRING) -> INT
 *=====================================*/
WORD _strtoint (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	STRING str = (STRING) evaluate(ielist(node), stab, eflg);
	if (*eflg || !str || *str == 0) return NULL;
	return (WORD) atoi(str);
}
/*=============================
 * _list -- Create list
 *   usage: list(IDENT) -> VOID
 *===========================*/
WORD _list (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	LIST list;
	INTERP var = (INTERP) ielist(node);
	*eflg = TRUE;
	if (!iistype(var, IIDENT)) return NULL;
	*eflg = FALSE;
	list = create_list();
	assign_iden(stab, iident(var), (WORD) list);
	return NULL;
}
/*=======================================
 * _push -- Push element on front of list
 *   usage: push(LIST, ANY) -> VOID
 *   usage: enqueue(LIST, ANY) -> VOID
 *=====================================*/
WORD _push (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	LIST list = (LIST) evaluate(arg, stab, eflg);
	WORD el;
	if (*eflg || !list) return NULL;
	el = evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	push_list(list, el);
	return NULL;
}
/*========================================
 * _requeue -- Add element to back of list
 *   usage: requeue(LIST, ANY) -> VOID
 *======================================*/
WORD _requeue (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	LIST list = (LIST) evaluate(arg, stab, eflg);
	WORD el;
	if (*eflg || !list) return NULL;
	el = evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	back_list(list, el);
	return NULL;
}
/*=======================================
 * _pop -- Pop element from front of list
 *   usage: pop(LIST) -> ANY
 *=====================================*/
WORD _pop (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	LIST list = (LIST) evaluate(ielist(node), stab, eflg);
	if (*eflg || !list) return NULL;
	return pop_list(list);
}
/*=============================================
 * _dequeue -- Remove element from back of list
 *   usage dequeue(LIST) -> ANY
 *===========================================*/
WORD _dequeue (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	LIST list = (LIST) evaluate(ielist(node), stab, eflg);
	if (*eflg || !list) return NULL;
	return dequeue_list(list);
}
/*=================================
 * _empty -- Check if list is empty
 *   usage: empty(LIST) -> BOOL
 *===============================*/
WORD _empty (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	LIST list = (LIST) evaluate(ielist(node), stab, eflg);
	if (*eflg || !list) return NULL;
	return (WORD) empty_list(list);
}
/*==================================
 * _getel -- Get nth value from list
 *   usage: getel(LIST, INT) -> ANY
 *================================*/
WORD _getel (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	LIST list = (LIST) evaluate(arg, stab, eflg);
	INT ind;
	if (*eflg || !list) return NULL;
	ind = (INT) evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	return get_list_element(list, ind);
}
/*========================================
 * _setel -- Set nth value in list
 *   usage: setel(LIST, INT, ANY) -> VOID
 *======================================*/
WORD _setel (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	LIST list = (LIST) evaluate(arg, stab, eflg);
	INT ind;
	WORD val;
	if (*eflg || !list) return NULL;
	arg = inext(arg);
	ind = (INT) evaluate(arg, stab, eflg);
	if (*eflg || ind < 1) return NULL;
	val = evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	set_list_element(list, ind, val);
	return NULL;
}
/*===============================
 * _length -- Find length of list
 *   usage: length(LIST) -> INT
 *=============================*/
WORD _length (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	LIST list = (LIST) evaluate(ielist(node), stab, eflg);
	if (*eflg || !list) return NULL;
	return (WORD) length_list(list);
}
/*=========================
 * _not -- Not operation
 *   usage: not(INT) -> INT
 *=======================*/
WORD _not (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	WORD value = evaluate(ielist(node), stab, eflg);
	if (*eflg) return NULL;
	return (WORD) !value;
}
/*================================
 * _save -- Copy string
 *   usage: save(STRING) -> STRING
 *==============================*/
WORD _save (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	WORD value = evaluate(ielist(node), stab, eflg);
	if (*eflg) return NULL;
	return value? (WORD) strsave(value) : NULL;
}
/*=================================
 * _strlen -- Find length of string
 *   usage: strlen(STRING) -> INT
 *===============================*/
WORD _strlen (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	STRING str = (STRING) evaluate(ielist(node), stab, eflg);
	if (*eflg || !str) return NULL;
	return (WORD) strlen(str);
}
/*==============================================
 * _concat -- Catenate strings
 *   usage: concat(STRING [, STRING]+) -> STRING
 *============================================*/
WORD _concat (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
        INTERP arg = (INTERP) ielist(node);
        INT len = 0, i, nstrs = 0;
        STRING hold[32];
        STRING p, new, str;

        while (arg) {
                str = (STRING) evaluate(arg, stab, eflg);
                if (*eflg) return NULL;
                if (str) {
                        len += strlen(str);
                        hold[nstrs++] = strsave(str);
                } else
                        hold[nstrs++] = NULL;
                arg = inext(arg);
        }
        p = new = (STRING) stdalloc(len + 1);
        for (i = 0; i < nstrs; i++) {
                str = hold[i];
                if (str) {
                        strcpy(p, str);
                        p += strlen(p);
                        stdfree(str);
                }
        }
        return (WORD) new;
}
/*=======================================
 * _lower -- Convert string to lower case
 *   usage: lower(STRING) -> STRING
 *=====================================*/
WORD _lower (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	WORD val = evaluate(ielist(node), stab, eflg);
	if (*eflg || !val) return NULL;
        return (WORD) lower(val);
}
/*=======================================
 * _upper -- Convert string to upper case
 *   usage: upper(STRING) -> STRING
 *=====================================*/
WORD _upper (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	WORD val = evaluate(ielist(node), stab, eflg);
	if (*eflg || !val) return NULL;
	return (WORD) upper(val);
}
/*======================================
 * _capitalize -- Capitalize string
 *   usage: capitalize(STRING) -> STRING
 *====================================*/
WORD _capitalize (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	WORD val = evaluate(ielist(node), stab, eflg);
	if (*eflg || !val) return NULL;
	return (WORD) capitalize(val);
}
/*=================================
 * _pn -- Generate pronoun
 *   usage: pn(INDI, INT) -> STRING
 *===============================*/
static STRING mpns[] = {  "He",  "he", "His", "his", "him" };
static STRING fpns[] = { "She", "she", "Her", "her", "her" };
WORD _pn (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	NODE indi;
	INT typ;
	indi = eval_indi(arg, stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	typ = (INT) evaluate(inext(arg), stab, eflg);
	if (*eflg || typ < 0 || typ > 4) return NULL;
	if (SEX(indi) == SEX_FEMALE)
		return (WORD) fpns[typ];
	else
		return (WORD) mpns[typ];
}
/*===================================
 * _print -- Print to stdout window
 *   usage: print([STRING]+,) -> VOID
 *=================================*/
WORD _print (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	WORD val;
	while (arg) {
		val = evaluate(arg, stab, eflg);
		if (*eflg) return NULL;
		if (val) wprintf("%s", (STRING) val);
		arg = inext(arg);
	}
	return NULL;
}
/*=================================================
 * _sex -- Find sex, as string M, F or U, of person
 *   usage: sex(INDI) -> STRING
 *===============================================*/
WORD _sex (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	INT sex;
	if (*eflg || !indi) return NULL;
	if ((sex = SEX(indi)) == SEX_MALE) return "M";
	if (sex == SEX_FEMALE) return "F";
	return "U";
}
/*=================================
 * _male -- Check if person is male
 *   usage: male(INDI) -> BOOL
 *===============================*/
WORD _male (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	return (WORD) (SEX(indi) == SEX_MALE);
}
/*=====================================
 * _female -- Check if person is female
 *   usage: female(INDI) -> BOOL
 *===================================*/
WORD _female (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE indi = eval_indi(ielist(node), stab, eflg, NULL);
	if (*eflg || !indi) return NULL;
	return (WORD) (SEX(indi) == SEX_FEMALE);
}
/*=========================================
 * _key -- Return person or family key
 *   usage: key(INDI|FAM [,BOOL]) -> STRING
 *=======================================*/
WORD _key (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	CACHEEL cel = (CACHEEL) evaluate(arg, stab, eflg);
	BOOLEAN strip = FALSE;
	STRING key;
	if (*eflg || !cel) return NULL;
	if (inext(arg)) strip = (BOOLEAN) evaluate(inext(arg), stab, eflg);
	key = (STRING) ckey(cel);
	return (WORD) (strip ? key + 1 : key);
}
/*================================
 * _inode -- Return root of person
 *   usage: inode(INDI) -> NODE
 *==============================*/
WORD _inode (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	return (WORD) eval_indi(ielist(node), stab, eflg, NULL);
}
/*================================
 * _fnode -- Return root of family
 *   usage: fnode(FAM) -> NODE
 *==============================*/
WORD _fnode (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	return (WORD) eval_fam(ielist(node), stab, eflg, NULL);
}
/*==============================
 * _table -- Create table
 *   usage: table(IDENT) -> VOID
 *============================*/
WORD _table (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	TABLE tab;
	INTERP var = (INTERP) ielist(node);
	*eflg = TRUE;
	if (!iistype(var, IIDENT)) return NULL;
	*eflg = FALSE;
	tab = create_table();
	assign_iden(stab, iident(var), (WORD) tab);
	return NULL;
}
/*==========================================
 * _insert -- Add element to table
 *   usage: insert(TAB, STRING, ANY) -> VOID
 *========================================*/
WORD _insert (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	TABLE tab = (TABLE) evaluate(arg, stab, eflg);
	STRING str;
	WORD val;
	if (*eflg || !tab) return NULL;
	arg = inext(arg);
	str = (STRING) evaluate(arg, stab, eflg);
	if (*eflg) return NULL;
	val = (WORD) evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	insert_table(tab, str, val);
	return NULL;
}
/*====================================
 * _lookup -- Look up element in table
 *   usage: lookup(TAB, STRING) -> ANY
 *==================================*/
WORD _lookup (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	TABLE tab = (TABLE) evaluate(arg, stab, eflg);
	STRING str;
	if (*eflg || !tab) return NULL;
	str = (STRING) evaluate(inext(arg), stab, eflg);
	if (*eflg || !str) return NULL;
	return (WORD) valueof(tab, str);
}
/*=====================================
 * _trim -- Trim string if too long
 *   usage: trim(STRING, INT) -> STRING
 *===================================*/
WORD _trim (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	STRING str = (STRING) evaluate(arg, stab, eflg);
	INT len;
	if (*eflg) return NULL;
	len = (INT) evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	return (STRING) trim(str, len);
}
/*=======================================
 * _trimname -- Trim name if too long
 *   usage: trimname(INDI, INT) -> STRING
 *=====================================*/
WORD _trimname (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	INT len;
	NODE indi = eval_indi(arg, stab, eflg, (CACHEEL *) NULL);
	if (*eflg || !indi) return NULL;
	*eflg = TRUE;
	if (!(indi = NAME(indi)) || !nval(indi)) return NULL;
	*eflg = FALSE;
	len = (INT) evaluate(inext(arg), stab, eflg);
	if (*eflg) return NULL;
	return (STRING) name_string(trim_name(nval(indi), len));
}
/*===============================
 * _date -- Return date of event
 *   usage: date(EVENT) -> STRING
 *=============================*/
WORD _date (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE evnt = (NODE) evaluate(ielist(node), stab, eflg);
	if (*eflg || !evnt) return NULL;
	return event_to_date(evnt, FALSE);
}
/*=====================================================
 * _extractdate -- Extract date from EVENT or DATE NODE
 *   usage: extractdate(NODE, VARB, VARB, VARB) -> VOID
 *===================================================*/
WORD _extractdate (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INT da = 0, mo = 0, yr = 0;
	STRING str;
	INTERP arg = (INTERP) ielist(node);
	INTERP dvar = inext(arg);
	INTERP mvar = inext(dvar);
	INTERP yvar = inext(mvar);
	NODE lin = (NODE) evaluate(arg, stab, eflg);
	if (*eflg) return NULL;
	*eflg = TRUE;
	if (!lin) return NULL;
	if (!iistype(dvar, IIDENT)) return NULL;
	if (!iistype(mvar, IIDENT)) return NULL;
	if (!iistype(yvar, IIDENT)) return NULL;
	if (strcmp("DATE", ntag(lin)))
		str = event_to_date(lin, FALSE);
	else
		str = nval(lin);
	extract_date(str, &da, &mo, &yr);
	assign_iden(stab, iident(dvar), (WORD) da);
	assign_iden(stab, iident(mvar), (WORD) mo);
	assign_iden(stab, iident(yvar), (WORD) yr);
	*eflg = FALSE;
	return NULL;
}
/*=================================================
 * _stddate -- Return standard date format of event
 *   usage: stddate(EVENT) -> STRING
 *===============================================*/
static INT daycode = 0;
static INT monthcode = 3;
static INT datecode = 0;
WORD _stddate (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	extern STRING format_date();
	NODE evnt = (NODE) evaluate(ielist(node), stab, eflg);
	if (*eflg || !evnt) return NULL;
	return format_date(event_to_date(evnt, FALSE), daycode,
	    monthcode, 1, datecode);
}
/*===============================================
 * _dayformat -- Set day format for standard date
 *   usage: dayformat(INT) -> NULL
 *=============================================*/
WORD _dayformat (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	INT value = (INT) evaluate(arg, stab, eflg);
	if (*eflg || value < 0) value = 0;
	if (value > 2) value = 2;
	daycode = value;
	return NULL;
}
/*===============================================
 * _monthformat -- Set month format standard date
 *   usage: dayformat(INT) -> NULL
 *=============================================*/
WORD _monthformat (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	INT value = (INT) evaluate(arg, stab, eflg);
	if (*eflg || value < 0) value = 0;
	if (value > 6) value = 6;
	monthcode = value;
	return NULL;
}
/*=================================================
 * _dateformat -- Set date format for standard date
 *   usage: dateformat(INT) -> NULL
 *===============================================*/
WORD _dateformat (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
        INTERP arg = (INTERP) ielist(node);
        INT value = (INT) evaluate(arg, stab, eflg);
        if (*eflg || value < 0) value = 0;
        if (value > 11) value = 11;
        datecode = value;
        return NULL;
}
/*===============================
 * _year -- Return year of event
 *   usage: year(EVENT) -> STRING
 *=============================*/
WORD _year (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE evnt = (NODE) evaluate(ielist(node), stab, eflg);
	if (*eflg || !evnt) return NULL;
	return event_to_date(evnt, TRUE);
}
/*=================================
 * _place -- Return place of event
 *   usage: place(EVENT) -> STRING
 *==============================*/
WORD _place (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE evnt = (NODE) evaluate(ielist(node), stab, eflg);
	if (*eflg || !evnt) return NULL;
	return event_to_plac(evnt, FALSE);
}
/*=============================
 * _tag -- Return tag of node
 *   usage: tag(NODE) -> STRING
 *===========================*/
WORD _tag (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE ged = (NODE) evaluate(ielist(node), stab, eflg);
	if (*eflg || !ged) return NULL;
	return ntag(ged);
}
/*===============================
 * _value -- Return value of node
 *   usage: value(NODE) -> STRING
 *=============================*/
WORD _value (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE ged = (NODE) evaluate(ielist(node), stab, eflg);
	if (*eflg || !ged) return NULL;
	return nval(ged);
}
/*==============================
 * _xref -- Return xref of node
 *   usage: xref(NODE) -> STRING
 *============================*/
WORD _xref (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE ged = (NODE) evaluate(ielist(node), stab, eflg);
	if (*eflg || !ged) return NULL;
	return nxref(ged);
}
/*===============================
 * _child -- Return child of node
 *   usage: child(NODE) -> NODE
 *=============================*/
WORD _child (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE ged = (NODE) evaluate(ielist(node), stab, eflg);
	if (*eflg || !ged) return NULL;
	return (WORD) nchild(ged);
}
/*=================================
 * _parent -- Return parent of node
 *   usage: parent(NODE) -> NODE
 *===============================*/
WORD _parent (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE ged = (NODE) evaluate(ielist(node), stab, eflg);
	if (*eflg || !ged) return NULL;
	return (WORD) nparent(ged);
}
/*========================================
 * _sibling -- Return next sibling of node
 *   usage: sibling(NODE) -> NODE
 *======================================*/
WORD _sibling (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	NODE ged = (NODE) evaluate(ielist(node), stab, eflg);
	if (*eflg || !ged) return NULL;
	return (WORD) nsibling(ged);
}
/*==================================
 * _copyfile -- Copy file to output
 *   usage: copyfile(STRING) -> VOID
 *================================*/
WORD _copyfile (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	FILE *cfp, *fopenpath();
	int c;
	STRING fname = (STRING) evaluate(ielist(node), stab, eflg);
	char buffer[1024];
	if (*eflg || !fname) return NULL;
	if (!(cfp = fopenpath(fname, "r", llprograms))) {
		*eflg = TRUE;
		return NULL;
	}
	while (fgets(buffer, 1024, cfp))
		poutput(buffer);
	fclose(cfp);
	return NULL;
}
/*========================
 * _nl -- Newline function
 *   usage: nl() -> STRING
 *======================*/
WORD _nl (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	*eflg = FALSE;
	return "\n";
}
/*=========================
 * _space -- Space function
 *   usage: sp() -> STRING
 *=======================*/
WORD _space (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	*eflg = FALSE;
	return " ";
}
/*=============================
 * _qt -- Double quote function
 *   usage: qt() -> STRING
 *===========================*/
WORD _qt (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	*eflg = FALSE;
	return "\"";
}
/*===============================
 * _indi -- Convert key to INDI
 *   usage: indi(STRING) -> INDI
 *============================*/
WORD _indi (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	STRING str = (STRING) evaluate(ielist(node), stab, eflg);
	STRING rec;
	char scratch[200], *p = str, *q = scratch;
	INT c, len;
	if (*eflg || !str) return NULL;
	while ((c = *p++) && chartype(c) != DIGIT)
		;
	if (c == 0) return NULL;
	*q++ = 'I';
	*q++ = c;
	while (chartype(c = *p++) == DIGIT)
		*q++ = c;
	*q = 0;
	if (strlen(scratch) == 1) return NULL;
	if (rec = (STRING) retrieve_record(scratch, &len)) {
		stdfree(rec);
		return (WORD) key_to_indi_cacheel(scratch);
	}
	return NULL;
}
/*=============================
 * _fam -- Convert key to FAM
 *   usage: fam(STRING) -> FAM
 *==========================*/
WORD _fam (node, stab, eflg)
INTERP node; TABLE stab; BOOLEAN *eflg;
{
	STRING str = (STRING) evaluate(ielist(node), stab, eflg);
	STRING rec;
	char scratch[200], *p = str, *q = scratch;
	INT c, len;
	if (*eflg || !str) return NULL;
	while ((c = *p++) && chartype(c) != DIGIT)
		;
	if (c == 0) return NULL;
	*q++ = 'F';
	*q++ = c;
	while (chartype(c = *p++) == DIGIT)
		*q++ = c;
	*q = 0;
	if (strlen(scratch) == 1) return NULL;
	if (rec = (STRING) retrieve_record(scratch, &len)) {
		stdfree(rec);
		return (WORD) key_to_fam_cacheel(scratch);
	}
	return NULL;
}
/*==========================================
 * eval_binary -- Evaluate binary expression
 *========================================*/
eval_binary (node, pval1, pval2, stab, eflg)
INTERP node; TABLE stab; INT *pval1, *pval2; BOOLEAN *eflg;
{
	INTERP arg = (INTERP) ielist(node);
	*pval1 = (INT) evaluate(arg, stab, eflg);
	if (*eflg) return;
	*pval2 = (INT) evaluate(inext(arg), stab, eflg);
}
/*========================================
 * eval_indi -- Evaluate person expression
 *======================================*/
NODE eval_indi (expr, stab, eflg, pcel)
INTERP expr; TABLE stab; BOOLEAN *eflg; CACHEEL *pcel;
{
	NODE indi;
	CACHEEL cel = (CACHEEL) evaluate(expr, stab, eflg);
	if (*eflg || !cel) return NULL;
	if (!cnode(cel)) cel = key_to_indi_cacheel(ckey(cel));
	indi = cnode(cel);
	if (strcmp("INDI", ntag(indi))) return NULL;
	if (pcel) *pcel = cel;
	return indi;
}
/*=======================================
 * eval_fam -- Evaluate family expression
 *=====================================*/
NODE eval_fam (expr, stab, eflg, pcel)
INTERP expr; TABLE stab; BOOLEAN *eflg; CACHEEL *pcel;
{
	NODE fam;
	CACHEEL cel = (CACHEEL) evaluate(expr, stab, eflg);
	if (*eflg || !cel) return NULL;
	if (!cnode(cel)) cel = key_to_fam_cacheel(ckey(cel));
	fam = cnode(cel);
	if (strcmp("FAM", ntag(fam))) return NULL;
	if (pcel) *pcel = cel;
	return fam;
}
/*===================================================
 * indi_to_cacheel -- Convert person to cache element
 *=================================================*/
CACHEEL indi_to_cacheel (indi)
NODE indi;
{
	CACHEEL cel;
	if (!indi) return NULL;
	cel = key_to_indi_cacheel(rmvat(nxref(indi)));
	ASSERT(cel);
	return cel;
}
/*==================================================
 * fam_to_cacheel -- Convert family to cache element
 *================================================*/
CACHEEL fam_to_cacheel (fam)
NODE fam;
{
	CACHEEL cel;
	if (!fam) return NULL;
	cel = key_to_fam_cacheel(rmvat(nxref(fam)));
	ASSERT(cel);
	return cel;
}
