/*=============================================================
 * keytonod.c -- Convert between keys and node trees
 * Copyright(c) 1992-94 by T.T. Wetmore IV; all rights reserved
 *   2.3.4 - 24 Jun 93    2.3.5 - 01 Sep 93
 *   3.0.0 - 08 May 94
 *===========================================================*/

#include "standard.h"
#include "table.h"
#include "gedcom.h"
#include "cache.h"

static CACHE create_cache();
static NODE key_to_node();
static CACHEEL key_to_cacheel();
static dereference();
static CACHE indicache, famcache;

/*=====================================
 * key_to_indi -- Convert key to person
 *===================================*/
NODE key_to_indi (key)
STRING key;
{
	NODE node;
	if (!key) return NULL;
	if (!(node = (NODE) key_to_node(indicache, key))) return NULL;
	ASSERT(!strcmp("INDI", ntag(node)));
	return node;
}
/*==============================================
 * key_to_indi_cacheel -- Convert key to CACHEEL
 *=============================================*/
CACHEEL key_to_indi_cacheel (key)
STRING key;
{
	CACHEEL cel;
	if (!key) return NULL;
	if (!(cel = (CACHEEL) key_to_cacheel(indicache, key))) return NULL;
	ASSERT(!strcmp("INDI", ntag(cnode(cel))));
	return cel;
}
/*====================================
 * key_to_fam -- Convert key to family
 *==================================*/
NODE key_to_fam (key)
STRING key;
{
	NODE node;
	if (!key) return NULL;
	if (!(node = (NODE) key_to_node(famcache, key))) return NULL;
	ASSERT(!strcmp("FAM", ntag(node)));
	return node;
}
/*=============================================
 * key_to_fam_cacheel -- Convert key to CACHEEL
 *===========================================*/
CACHEEL key_to_fam_cacheel (key)
STRING key;
{
	CACHEEL cel;
	if (!key) return NULL;
	if (!(cel = (CACHEEL) key_to_cacheel(famcache, key))) return NULL;
	ASSERT(!strcmp("FAM", ntag(cnode(cel))));
	return cel;
}
/*==========================================
 * init_caches -- Create and init two caches
 *========================================*/
init_caches ()
{
	indicache = create_cache((INT)300, (INT)3000);
	famcache = create_cache((INT)200, (INT)2000);
}
/*=============================
 * create_cache -- Create cache
 *===========================*/
static CACHE create_cache (dirsize, indsize)
INT dirsize, indsize;
{
	CACHE cache;
	if (dirsize < 1) dirsize = 1;
	if (indsize < 1) indsize = 1;
	cache = (CACHE) stdalloc(sizeof(*cache));
	cdata(cache) = create_table();
	cfirstdir(cache) = clastdir(cache) = NULL;
	cfirstind(cache) = clastind(cache) = NULL;
	csizedir(cache) = csizeind(cache) = 0;
	cmaxdir(cache) = dirsize;
	cmaxind(cache) = indsize;
	return cache;
}
/*=================================================
 * remove_direct -- Unlink CACHEEL from direct list
 *===============================================*/
static remove_direct (cache, cel)
CACHE cache;
CACHEEL cel;
{
	CACHEEL prev = cprev(cel);
	CACHEEL next = cnext(cel);
	ASSERT(cache && cel);
	if (prev) cnext(prev) = next;
	if (next) cprev(next) = prev;
	if (!prev) cfirstdir(cache) = next;
	if (!next) clastdir(cache) = prev;
	csizedir(cache)--;
}
/*=====================================================
 * remove_indirect -- Unlink CACHEEL from indirect list
 *===================================================*/
static remove_indirect (cache, cel)
CACHE cache;
CACHEEL cel;
{
	CACHEEL prev = cprev(cel);
	CACHEEL next = cnext(cel);
	ASSERT(cache && cel);
	if (prev) cnext(prev) = next;
	if (next) cprev(next) = prev;
	if (!prev) cfirstind(cache) = next;
	if (!next) clastind(cache) = prev;
	csizeind(cache)--;
}
/*===========================================================
 * first_direct -- Make unlinked CACHEEL first in direct list
 *=========================================================*/
static first_direct (cache, cel)
CACHE cache;
CACHEEL cel;
{
	CACHEEL frst = cfirstdir(cache);
	ASSERT(cache && cel);
	csizedir(cache)++;
	cprev(cel) = NULL;
	cnext(cel) = frst;
	if (frst) cprev(frst) = cel;
	if (!frst) clastdir(cache) = cel;
	cfirstdir(cache) = cel;
}
/*===============================================================
 * first_indirect -- Make unlinked CACHEEL first in indirect list
 *=============================================================*/
static first_indirect (cache, cel)
CACHE cache;
CACHEEL cel;
{
	CACHEEL frst = cfirstind(cache);
	ASSERT(cache && cel);
	csizeind(cache)++;
	cprev(cel) = NULL;
	cnext(cel) = frst;
	if (frst) cprev(frst) = cel;
	if (!frst) clastind(cache) = cel;
	cfirstind(cache) = cel;
}
/*=======================================================
 * remove_last -- Remove last indirect element from cache
 *=====================================================*/
static remove_last (cache)
CACHE cache;
{
	CACHEEL cel = clastind(cache);
	STRING key;
	ASSERT(cel);
	remove_indirect(cache, cel);
	key = ckey(cel);
	stdfree(cel);
	delete_table(cdata(cache), key);
	stdfree(key);
}
/*============================================================
 * direct_to_first -- Make direct CACHEEL first in direct list
 *==========================================================*/
static direct_to_first (cache, cel)
CACHE cache;
CACHEEL cel;
{
	ASSERT(cache && cel);
	if (cel == cfirstdir(cache)) return;
	remove_direct(cache, cel);
	first_direct(cache, cel);
}
/*==================================================================
 * indirect_to_first -- Make indirect CACHEEL first in indirect list
 *================================================================*/
static indirect_to_first (cache, cel)
CACHE cache;
CACHEEL cel;
{
	ASSERT(cache && cel);
	remove_indirect(cache, cel);
	dereference(cel);
	first_direct(cache, cel);
}
/*==============================================================
 * direct_to_indirect -- Make last direct CACHEEL first indirect
 *============================================================*/
static direct_to_indirect (che)
CACHE che;
{
	CACHEEL cel = clastdir(che);
	for (cel = clastdir(che); cel && clock(cel); cel = cprev(cel))
		;
	ASSERT(cel);
	remove_direct(che, cel);
	free_nodes(cnode(cel));
	cnode(cel) = NULL;
	first_indirect(che, cel);
}
/*=====================================================
 * dereference -- Dereference cel by reading its record
 *===================================================*/
static dereference (cel)
CACHEEL cel;
{
	STRING record;
	INT len;
	NODE node;
	ASSERT(cel);
	ASSERT(record = retrieve_record(ckey(cel), &len));
	ASSERT(node = string_to_node(record));
	cnode(cel) = node;
	stdfree(record);
}
/*========================================================
 * add_to_direct -- Add new CACHEL to direct part of cache
 *======================================================*/
static CACHEEL add_to_direct (cache, key)
CACHE cache;
STRING key;
{
	STRING record;
	INT len;
	CACHEEL cel;
	NODE node;
	ASSERT(cache && key);
	ASSERT(csizedir(cache) < cmaxdir(cache));
	cel = (CACHEEL) stdalloc(sizeof(*cel));
	insert_table(cdata(cache), key = strsave(key), cel);
	ASSERT(record = retrieve_record(key, &len));
	ASSERT(node = string_to_node(record));
	cnode(cel) = node;
	ckey(cel) = key;
	clock(cel) = FALSE;
	first_direct(cache, cel);
	stdfree(record);
	return cel;
}
/*======================================================
 * key_to_cacheel -- Return CACHEEL corresponding to key
 *====================================================*/
static CACHEEL key_to_cacheel (cache, key)
CACHE cache;
STRING key;
{
	CACHEEL cel;
	if (cel = (CACHEEL) valueof(cdata(cache), key)) {
		if (cnode(cel))
			direct_to_first(cache, cel);
		else {
			if (csizedir(cache) >= cmaxdir(cache))
				direct_to_indirect(cache);
			indirect_to_first(cache, cel);
		}
		return cel;
	}
	if (csizedir(cache) >= cmaxdir(cache)) {
		if (csizeind(cache) >= cmaxind(cache))
			remove_last(cache);
		direct_to_indirect(cache);
	}
	return add_to_direct(cache, key);
}
/*===============================================================
 * key_to_node -- Return tree from key; add to cache if not there
 *=============================================================*/
static NODE key_to_node (cache, key)
CACHE cache;
STRING key;
{
	CACHEEL cel;
	ASSERT(cache && key);
	if (!(cel = key_to_cacheel(cache, key))) return NULL;
	return cnode(cel);
}
/*======================================
 * lock_cache -- Lock CACHEEL into cache
 *====================================*/
lock_cache (cel)
CACHEEL cel;
{
	ASSERT(cnode(cel));
	clock(cel)++;
}
/*==========================================
 * unlock_cache -- Unlock CACHEEL from cache
 *========================================*/
unlock_cache (cel)
CACHEEL cel;
{
	ASSERT(cnode(cel));
	clock(cel)--;
}
/*================================
 * cache_stats -- Show cache stats
 *==============================*/
cache_stats ()
{
	CACHE c = indicache;
	CACHE f = famcache;
	INT n = 0;
	CACHEEL cel;
	for (cel = cfirstdir(c); cel; cel = cnext(cel)) {
		if (clock(cel)) n++;
	}
	mprintf("Cache contents -- I: %dD  %dI  %dL;  F: %dD  %dI",
	    csizedir(c), csizeind(c), n, csizedir(f), csizeind(f));
}
/*============================================
 * indi_to_cache -- Add person to person cache
 *==========================================*/
indi_to_cache (node)
NODE node;
{
	node_to_cache(indicache, node);
}
/*===========================================
 * fam_to_cache -- Add family to family cache
 *=========================================*/
fam_to_cache (node)
NODE node;
{
	node_to_cache(famcache, node);
}
/*========================================
 * node_to_cache -- Add node tree to cache
 *======================================*/
node_to_cache (cache, node)
CACHE cache;
NODE node;
{
	STRING key = rmvat(nxref(node));
	ASSERT(cache && node);
	ASSERT(!valueof(cdata(cache), key));
	if (csizedir(cache) >= cmaxdir(cache)) {
		if (csizeind(cache) >= cmaxind(cache)) {
			remove_last(cache);
			direct_to_indirect(cache);
			add_node_to_direct(cache, node);
		} else {
			direct_to_indirect(cache);
			add_node_to_direct(cache, node);
		}
	} else {
		add_node_to_direct(cache, node);
	}
}
/*=======================================================
 * add_node_to_direct -- Add node to direct part of cache
 *=====================================================*/
add_node_to_direct(cache, node)
CACHE cache;
NODE node;
{
	CACHEEL cel;
	STRING key;
	ASSERT(cache && node);
	ASSERT(csizedir(cache) < cmaxdir(cache));
	cel = (CACHEEL) stdalloc(sizeof(*cel));
	insert_table(cdata(cache), key = strsave(rmvat(nxref(node))), cel);
	cnode(cel) = node;
	ckey(cel) = key;
	clock(cel) = FALSE;
	first_direct(cache, cel);
}
/*==============================================
 * remove_indi_cache -- Remove person from cache
 *============================================*/
remove_indi_cache (key)
STRING key;
{
	remove_from_cache(indicache, key);
}
/*=============================================
 * remove_fam_cache -- Remove family from cache
 *===========================================*/
remove_fam_cache (key)
STRING key;
{
	remove_from_cache(famcache, key);
}
/*=============================================
 * remove_from_cache -- Remove entry from cache
 *===========================================*/
remove_from_cache (cache, key)
CACHE cache;
STRING key;
{
	CACHEEL cel;
	if (!key || *key == 0 || !cache) return;
	if (!(cel = (CACHEEL) valueof(cdata(cache), key))) return;
	if (cnode(cel))
		remove_direct(cache, cel);
	else
		remove_indirect(cache, cel);
	stdfree(cel);
	delete_table(cdata(cache), key);
}
