/*================================================================
 * gedcom.c -- Read and convert GEDCOM records to LifeLines format
 * Copyright(c) 1992-94 by T.T. Wetmore IV; all rights reserved
 *   2.3.4 - 24 Jun 93    2.3.5 - 01 Sep 93
 *   3.0.0 - 08 May 94
 *==============================================================*/

#include "standard.h"
#include "table.h"
#include "gedcom.h"
#include "sequence.h"
#include "gedcheck.h"

static NODE indi_to_indi();
static NODE fam_to_fam();
static NODE even_to_even();
static NODE sour_to_sour();
static NODE othr_to_othr();

STRING exfamc = "Extra FAMC line in INDI record; line ignored.\n";
STRING misnam = "Missing NAME line in INDI record; record ignored.\n";
STRING noiref = "FAM record has no INDI references; record ignored.\n";

/*========================================================
 * node_to_node -- Convert GEDCOM record to LifeLines form
 *======================================================*/
NODE node_to_node (node, ptype)
NODE node;
INT *ptype;
{
	*ptype = 0;
	if (!strcmp("HEAD", ntag(node)) ||
	    !strcmp("TRLR", ntag(node))) return NULL;
	if (!strcmp("INDI", ntag(node)))      *ptype = INDI_REC;
	else if (!strcmp("FAM",  ntag(node))) *ptype = FAM_REC;
	else if (!strcmp("EVEN", ntag(node))) *ptype = EVEN_REC;
	else if (!strcmp("SOUR", ntag(node))) *ptype = SOUR_REC;
	else                                  *ptype = OTHR_REC;
	switch (*ptype) {
	case INDI_REC: return indi_to_indi(node);
	case FAM_REC:  return fam_to_fam(node);
	case EVEN_REC: return even_to_even(node);
	case SOUR_REC: return sour_to_sour(node);
	case OTHR_REC: return othr_to_othr(node);
	default: FATAL();
	}
}
/*=======================================================
 * indi_to_indi - Convert person record to LifeLines form
 *=====================================================*/
static NODE indi_to_indi (indi)
NODE indi;
{
	NODE node, last, name, lfms, famc, fams, first, prev;
	last = name = famc = fams = first = NULL;
	node = nchild(indi);
	while (node) {
		STRING tag = ntag(node);
		if (!strcmp("NAME", tag) && !name)
			name = node;
		else if (!strcmp("FAMS", tag)) {
			if (!fams)
				lfms = fams = node;
			else
				lfms = nsibling(lfms) = node;
		} else if (!strcmp("FAMC", tag)) {
			if (famc)
				wprintf(exfamc);
			else
				famc = node;
		} else {
			if (!first)
				first = last = node;
			else
				last = nsibling(last) = node;
		}
		prev = node;
		node = nsibling(node);
		nsibling(prev) = NULL;
	}
	if (!name) {
		wprintf(misnam);
		return NULL;
	}
	nchild(indi) = node = name;
	if (first) {
		nsibling(node) = first;
		node = last;
	}
	if (famc) node = nsibling(node) = famc;
	nsibling(node) = fams;
	return indi;
}
/*======================================================
 * fam_to_fam -- Convert family record to LifeLines form
 *====================================================*/
static NODE fam_to_fam (fam)
NODE fam;
{
	NODE node, first, last, husb, wife, chil, lchl, prev;
	first = last = husb = wife = chil = NULL;
	node = nchild(fam);
	while (node) {
		STRING tag = ntag(node);
		if (!strcmp("HUSB", tag) && !husb)
			husb = node;
		else if (!strcmp("WIFE", tag) && !wife)
			wife = node;
		else if (!strcmp("CHIL", tag)) {
			if (!chil)
				chil = lchl = node;
			else
				lchl = nsibling(lchl) = node;
		} else {
			if (!first)
				last = first = node;
			else
				last = nsibling(last) = node;
		}
		prev = node;
		node = nsibling(node);
		nsibling(prev) = NULL;
	}
	node = NULL;
	if (!husb && !wife && !chil) {
		wprintf(noiref);
		return NULL;
	}
	if (husb) nchild(fam) = node = husb;
	if (wife) {
		if (!node)
			nchild(fam) = node = wife;
		else
			node = nsibling(node) = wife;
	}
	if (first) {
		if (!node)
			nchild(fam) = first;
		else
			nsibling(node) = first;
		node = last;
	}
	if (chil) {
		if (!node)
			nchild(fam) = chil;
		else
			nsibling(node) = chil;
	}
	return fam;
}
/*=======================================================
 * even_to_even -- Convert event record to LifeLines form
 *=====================================================*/
static NODE even_to_even (evn)
NODE evn;
{
	return evn;
}
/*========================================================
 * sour_to_sour -- Convert source record to LifeLines form
 *======================================================*/
static NODE sour_to_sour (src)
NODE src;
{
	return src;
}
/*=======================================================
 * othr_to_othr -- Convert other record to LifeLines form
 *=====================================================*/
static NODE othr_to_othr (oth)
NODE oth;
{
	return oth;
}
