/*================================================================
 * screen.c -- ETI user interface code.
 * Copyright(c) 1992, by Thomas T. Wetmore IV; all rights reserved
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 29 Sep 93 - modified
 *   Version 2.3.6 - 01 Jan 04 - modified
 *   Version 3.0.0 - 01 Jan 04 - modified
 *================================================================
 */
#include "standard.h"
#include "screen.h"
#include "table.h"
#include "gedcom.h"
#include "indiseq.h"
#include "varargs.h"

#define VIEWABLE 10

STRING get_answer();
WINDOW *choose_win();
extern BOOLEAN alldone;
extern STRING version, empstr, empstr65, empstr71;
STRING mtitle = "LifeLines %s - Genealogical Database and Report Generator";
STRING cright = "Copyright(c) 1991, 1992, 1993, 1994 by Thomas T. Wetmore IV";

extern INDISEQ indiseq_list_interact();

INT cur_screen;
WINDOW *current_window = NULL;

WINDOW *std_out_win;
WINDOW *std_out_box_win;
PANEL *std_out_pnl;

WINDOW *ask_window;
PANEL *ask_panel;
WINDOW *ask_msg_window;
PANEL *ask_msg_panel;
WINDOW *choose_from_list_win;
PANEL *choose_from_list_pnl;
WINDOW *add_menu_win;
PANEL *add_menu_pnl;
WINDOW *del_menu_win;
PANEL *del_menu_pnl;
WINDOW *utils_menu_win;
PANEL *utils_menu_pnl;

/*======================================
 * init_smanager -- Init screen manager.
 *====================================*/
init_smanager ()
{
	create_main_screen();
	create_one_person_screen();
	create_one_family_screen();
	create_two_person_screen();
	create_two_family_screen();
	create_pedigree_screen();
	create_list_screen();
	rest_create();
	init_all_panels();
	curs_set(0);
}
/*===================================================
 * create_main_screen -- Create and init main screen.
 *=================================================*/
WINDOW *main_window;
PANEL *main_panel;
static create_main_screen ()
{
	INT row = (LINES - 15)/2;
	INT col = (COLS - 79)/2;
	WINDOW *win = main_window = newwin(15, 79, row, col);
	main_panel = new_panel(win);
	paint_main_screen();
}
/*========================================
 * paint_main_screen -- Paint main screen.
 *======================================*/
paint_main_screen()
{
	WINDOW *win = main_window;
	INT row;
	werase(win);
	box(win, 0, 0);
	show_horz_line(win, 4, 0, 79);
	show_horz_line(win, 12, 0, 79);
	wmove(win, 1, 2);
	wprintw(win, mtitle, version);
	mvwaddstr(win, 2, 4, cright);
	row = 5;
	mvwaddstr(win, row++, 2, "Please choose an operation:");
	mvwaddstr(win, row++, 4, "b  Browse the persons in the database");
	mvwaddstr(win, row++, 4, "a  Add information to the database");
	mvwaddstr(win, row++, 4, "d  Delete information from the database");
	mvwaddstr(win, row++, 4, "r  Generate reports from the database");
	mvwaddstr(win, row++, 4, "u  Miscellaneous utilities");
	mvwaddstr(win, row++, 4, "q  Quit and return to UNIX");
	hide_panel(main_panel);
}
/*===============================================================
 * create_one_person_screen -- Create and init one person screen.
 *=============================================================*/
WINDOW *one_person_window;
PANEL *one_person_panel;
static create_one_person_screen ()
{
	INT row = (LINES - 19)/2;
	INT col = (COLS - 79)/2;
	WINDOW *win = one_person_window = newwin(19, 79, row, col);
	one_person_panel = new_panel(win);
}
/*====================================================
 * paint_one_person_screen -- Paint one person screen.
 *==================================================*/
paint_one_person_screen ()
{
	WINDOW *win = one_person_window;
	INT row, col;
	werase(win);
	box(win, 0, 0);
	show_horz_line(win, 7,  0, 79);
	show_horz_line(win, 16, 0, 79);
	mvwaddstr(win, 8, 2, "Please choose a person browse operation:");
	row = 9; col = 3;
	mvwaddstr(win, row++, col, "e  Edit the person");
	mvwaddstr(win, row++, col, "f  Browse to father");
	mvwaddstr(win, row++, col, "m  Browse to mother");
	mvwaddstr(win, row++, col, "s  Browse to spouse/s");
	mvwaddstr(win, row++, col, "c  Browse to children");
	mvwaddstr(win, row++, col, "o  Browse to older sib");
	mvwaddstr(win, row++, col, "y  Browse to younger sib");
	row = 9; col = 28;
	mvwaddstr(win, row++, col, "g  Browse to family");
	mvwaddstr(win, row++, col, "u  Browse to parents");
	mvwaddstr(win, row++, col, "b  Browse to persons");
	mvwaddstr(win, row++, col, "h  Add as spouse");
	mvwaddstr(win, row++, col, "i  Add as child");
	mvwaddstr(win, row++, col, "r  Remove as spouse");
	mvwaddstr(win, row++, col, "d  Remove as child");
	row = 9; col = 53;
	mvwaddstr(win, row++, col, "p  Show pedigree");
	mvwaddstr(win, row++, col, "n  Create new person");
	mvwaddstr(win, row++, col, "a  Create new family");
	mvwaddstr(win, row++, col, "x  Swap two families");
	mvwaddstr(win, row++, col, "t  Enter tandem mode");
	mvwaddstr(win, row++, col, "z  Browse to person");
	mvwaddstr(win, row++, col, "q  Return to main menu");
	hide_panel(one_person_panel);
}
/*===============================================================
 * create_one_family_screen -- Create and init one family screen.
 *=============================================================*/
WINDOW *one_family_window;
PANEL *one_family_panel;
static create_one_family_screen ()
{
	INT row = (LINES - 19)/2;
	INT col = (COLS - 79)/2;
	WINDOW *win = one_family_window = newwin(19, 79, row, col);
	one_family_panel = new_panel(win);
	paint_one_family_screen();
}
/*====================================================
 * paint_one_family_screen -- Paint one family screen.
 *==================================================*/
paint_one_family_screen ()
{
	WINDOW *win = one_family_window;
	INT row, col;
	werase(win);
	box(win, 0, 0);
	show_horz_line(win, 9, 0, 79);
	show_horz_line(win, 16, 0, 79);
	mvwaddstr(win, 10, 2, "Please choose a family browse operation:");
	row = 11; col = 3;
	mvwaddstr(win, row++, col, "e  Edit the family");
	mvwaddstr(win, row++, col, "h  Browse to husband");
	mvwaddstr(win, row++, col, "w  Browse to wife");
	mvwaddstr(win, row++, col, "c  Browse to children");
	mvwaddstr(win, row++, col, "n  Create new person");
	row = 11; col = 28;
	mvwaddstr(win, row++, col, "s  Add spouse to family");
	mvwaddstr(win, row++, col, "a  Add child to family");
	mvwaddstr(win, row++, col, "r  Remove spouse from");
	mvwaddstr(win, row++, col, "d  Remove child from");
	mvwaddstr(win, row++, col, "x  Swap two children");
	row = 11; col = 53;
	mvwaddstr(win, row++, col, "t  Enter family tandem");
	mvwaddstr(win, row++, col, "b  Browse to persons");
	mvwaddstr(win, row++, col, "z  Browse to person");
	mvwaddstr(win, row++, col, "q  Return to main menu");
	hide_panel(one_family_panel);
}
/*===============================================================
 * create_two_person_screen -- Create and init two person screen.
 *=============================================================*/
WINDOW *two_person_window;
PANEL *two_person_panel;
static create_two_person_screen ()
{
	INT row = (LINES - 23)/2;
	INT col = (COLS - 79)/2;
	WINDOW *win = two_person_window = newwin(23, 79, row, col);
	two_person_panel = new_panel(win);
	paint_two_person_screen();
}
/*====================================================
 * paint_two_person_screen -- Paint two person screen.
 *==================================================*/
paint_two_person_screen ()
{
	WINDOW *win = two_person_window;
	INT row, col;
	werase(win);
	box(win, 0, 0);
	show_horz_line(win, 7, 0, 79);
	show_horz_line(win, 14, 0, 79);
	show_horz_line(win, 20, 0, 79);
	mvwaddstr(win, 15, 2, "Please choose a tandem browse operation:");
	row = 16; col = 3;
	mvwaddstr(win, row++, col, "e  Edit top person");
	mvwaddstr(win, row++, col, "t  Browse to top");
	mvwaddstr(win, row++, col, "f  Browse top father");
	mvwaddstr(win, row++, col, "m  Browse top mother");
	row = 16; col = 28;
	mvwaddstr(win, row++, col, "s  Browse top spouse/s");
	mvwaddstr(win, row++, col, "c  Browse top children");
	mvwaddstr(win, row++, col, "b  Browse to persons");
	mvwaddstr(win, row++, col, "d  Copy top to bottom");
	row = 16; col = 53;
	mvwaddstr(win, row++, col, "a  Add family");
	mvwaddstr(win, row++, col, "j  Merge bottom to top");
	mvwaddstr(win, row++, col, "x  Switch top/bottom");
	mvwaddstr(win, row++, col, "q  Return to main menu");
	hide_panel(two_person_panel);
}
/*===============================================================
 * create_two_family_screen -- Create and init two family screen.
 *=============================================================*/
WINDOW *two_family_window;
PANEL *two_family_panel;
static create_two_family_screen ()
{
	INT row = (LINES - 22)/2;
	INT col = (COLS - 79)/2;
	WINDOW *win = two_family_window = newwin(22, 79, row, col);
	two_family_panel = new_panel(win);
	paint_two_family_screen();
}
/*====================================================
 * paint_two_family_screen -- Paint two family screen.
 *==================================================*/
paint_two_family_screen ()
{
	WINDOW *win = two_family_window;
	INT row, col;
	werase(win);
	box(win, 0, 0);
	show_horz_line(win, 7, 0, 79);
	show_horz_line(win, 14, 0, 79);
	show_horz_line(win, 19, 0, 79);
	mvwaddstr(win, 15, 2, "Please choose a two family browse operation:");
	row = 16; col = 3;
	mvwaddstr(win, row++, col, "e  Edit top family");
	mvwaddstr(win, row++, col, "t  Browse to top");
	mvwaddstr(win, row++, col, "b  Browse to bottom");
	row = 16; col = 28;
	mvwaddstr(win, row++, col, "h  Browse to husbands");
	mvwaddstr(win, row++, col, "w  Browse to wives");
	mvwaddstr(win, row++, col, "x  Switch top/bottom");
	row = 16; col = 53;
	mvwaddstr(win, row++, col, "j  Merge bottom to top");
	mvwaddstr(win, row++, col, "q  Return to main menu");
	hide_panel(two_family_panel);
}
/*===========================================================
 * create_pedigree_screen -- Create and init pedigree screen.
 *=========================================================*/
WINDOW *pedigree_window;
PANEL *pedigree_panel;
static create_pedigree_screen ()
{
	INT row = (LINES - 23)/2;
	INT col = (COLS - 79)/2;
	WINDOW *win = pedigree_window = newwin(23, 79, row, col);
	pedigree_panel = new_panel(win);
	paint_pedigree_screen();
}
/*================================================
 * paint_pedigree_screen -- Paint pedigree screen.
 *==============================================*/
paint_pedigree_screen ()
{
	WINDOW *win = pedigree_window;
	INT row, col;
	werase(win);
	box(win, 0, 0);
	show_horz_line(win, 16, 0, 79);
	show_horz_line(win, 20, 0, 79);
	mvwaddstr(win, 17, 2, "Please choose a pedigree browse operation:");
	row = 18; col = 3;
	mvwaddstr(win, row++, col, "i  Browse to person");
	mvwaddstr(win, row++, col, "f  Browse to father");
	row = 18; col = 28;
	mvwaddstr(win, row++, col, "m  Browse to mother");
	mvwaddstr(win, row++, col, "c  Browse to children");
	row = 18; col = 53;
	mvwaddstr(win, row++, col, "g  Browse to family");
	mvwaddstr(win, row++, col, "q  Return to main menu");
	hide_panel(pedigree_panel);
}
/*==========================================================
 * create_list_screen -- Create and init list browse screen.
 *========================================================*/
WINDOW *list_window;
PANEL *list_panel;
static create_list_screen ()
{
	INT row = (LINES - 23)/2;
	INT col = (COLS - 79)/2;
	WINDOW *win = list_window = newwin(23, 79, row, col);
	list_panel = new_panel(win);
	paint_list_screen();
}
/*===============================================
 * paint_list_screen -- Paint list browse screen.
 *=============================================*/
paint_list_screen ()
{
	WINDOW *win = list_window;
	INT row, col;
	werase(win);
	box(win, 0, 0);
	show_vert_line(win, 7, 52, 16);
	show_horz_line(win, 7, 0, 79);
	show_horz_line(win, 20, 0, 53);
	mvwaddch(win, 7, 52, ACS_TTEE);
	mvwaddstr(win, 8, 54, "Choose an operation:");
	row = 9; col = 55;
	mvwaddstr(win, row++, col, "j  Move down list");
	mvwaddstr(win, row++, col, "k  Move up list");
	mvwaddstr(win, row++, col, "e  Edit this person");
	mvwaddstr(win, row++, col, "i  Browse this person");
	mvwaddstr(win, row++, col, "m  Mark this person");
	mvwaddstr(win, row++, col, "d  Delete from list");
	mvwaddstr(win, row++, col, "t  Enter tandem mode");
	mvwaddstr(win, row++, col, "n  Name this list");
	mvwaddstr(win, row++, col, "b  Browse new persons");
	mvwaddstr(win, row++, col, "a  Add to this list");
	mvwaddstr(win, row++, col, "x  Swap mark/current");
	mvwaddstr(win, row++, col, "q  Return to main menu");
	hide_panel(list_panel);
}
/*===========================================================
 * rest_create -- Create and init rest of windows and panels.
 *=========================================================*/
rest_create ()
{
	INT col = (COLS - 73)/2, row = (LINES - 19)/2;
	std_out_box_win = newwin(19, 73, row, col);
	std_out_win = subwin(std_out_box_win, 17, 71, row + 1, col + 1);
	std_out_pnl = new_panel(std_out_box_win);
	hide_panel(std_out_pnl);
	wsetscrreg(std_out_win, 1, 17);

	row = (LINES - 8)/2; col = (COLS - 66)/2;
	add_menu_win = newwin(8, 66, row, col);
	add_menu_pnl = new_panel(add_menu_win);
	box(add_menu_win, 0, 0);
	hide_panel(add_menu_pnl);

	row = (LINES - 7)/2; col = (COLS - 66)/2;
	del_menu_win = newwin(7, 66, row, col);
	del_menu_pnl = new_panel(del_menu_win);
	box(del_menu_win, 0, 0);
	hide_panel(del_menu_pnl);

	row = (LINES - 11)/2; col = (COLS - 66)/2;
	utils_menu_win = newwin(11, 66, row, col);
	utils_menu_pnl = new_panel(utils_menu_win);
	box(utils_menu_win, 0, 0);
	hide_panel(utils_menu_pnl);

	row = (LINES - 4)/2; col = (COLS - 73)/2;
	ask_window = newwin(4, 73, row, col);
	ask_panel = new_panel(ask_window);
	box(ask_window, 0, 0);
	hide_panel(ask_panel);

	row = (LINES - 5)/2; col = (COLS - 73)/2;
	ask_msg_window = newwin(5, 73, row, col);
	ask_msg_panel = new_panel(ask_msg_window);
	box(ask_msg_window, 0, 0);
	hide_panel(ask_msg_panel);

	row = (LINES - 15)/2; col = (COLS - 73)/2;
	choose_from_list_win = newwin(15, 73, row, col);
	choose_from_list_pnl = new_panel(choose_from_list_win);
	hide_panel(choose_from_list_pnl);
	win_list_init();
}
/*================================================
 * init_all_panels -- Init all WINDOWs and PANELs.
 *==============================================*/
init_all_panels ()
{
	WINDOW *win;
	INT row;

	win = add_menu_win;
	row = 1;
	mvwaddstr(win, row++, 2, "What do you want to add?");
	mvwaddstr(win, row++, 4, "p  Person - add new person to the database");
	mvwaddstr(win, row++, 4, "f  Family - create family record from one or two spouses");
	mvwaddstr(win, row++, 4, "c  Child - add a child to an existing family");
	mvwaddstr(win, row++, 4, "s  Spouse - add a spouse to an existing family");
	mvwaddstr(win, row++, 4, "q  Quit - return to the previous menu");
	win = del_menu_win;
	row = 1;
	mvwaddstr(win, row++, 2, "What do you want to remove?");
	mvwaddstr(win, row++, 4, "c  Child - remove a child from his/her family");
	mvwaddstr(win, row++, 4, "s  Spouse - remove a spouse from a family");
	mvwaddstr(win, row++, 4, "i  Individual - remove a person completely");
	mvwaddstr(win, row++, 4, "q  Quit - return to the previous menu");
	win = utils_menu_win;
	row = 1;
	mvwaddstr(win, row++, 2, "What utility do you want to perform?");
	mvwaddstr(win, row++, 4, "s  Save the database in a GEDCOM file");
	mvwaddstr(win, row++, 4, "r  Read in data from a GEDCOM file");
	mvwaddstr(win, row++, 4, "k  Find a persons's key value");
	mvwaddstr(win, row++, 4, "i  Identify a person from key value");
	mvwaddstr(win, row++, 4, "d  Show database statistics");
	mvwaddstr(win, row++, 4, "m  Show memory statistics");
	mvwaddstr(win, row++, 4, "e  Edit the place abbreviation file");
	mvwaddstr(win, row++, 4, "q  Return to the main menu");
}
/*==================================================================
 * display_screen -- Display screen as function of screen parameter.
 *================================================================*/
#define HIDEALL() HIDE(main_panel); HIDE(one_person_panel);\
		HIDE(two_person_panel); HIDE(one_family_panel);\
		HIDE(two_family_panel); HIDE(pedigree_panel);\
		HIDE(list_panel);

display_screen (new_screen)
INT new_screen;
{
	BOOLEAN do_message = (new_screen != cur_screen);
	INT c;

	if (!panel_hidden(std_out_pnl)) {
		wprintf("\nStrike any key to continue.\n");
		update_panels();
		doupdate();
		c = wgetch(std_out_win);
		hide_panel(std_out_pnl);
		update_panels();
		doupdate();
	}
	switch (cur_screen = new_screen) {
	case MAIN_SCREEN:
		current_window = main_window;
		HIDEALL(); SHOW(main_panel);
		break;
	case ONE_PERSON_SCREEN:
		current_window = one_person_window;
		HIDEALL(); SHOW(one_person_panel);
		break;
	case ONE_FAMILY_SCREEN:
		current_window = one_family_window;
		HIDEALL(); SHOW(one_family_panel);
		break;
	case TWO_PERSON_SCREEN:
		current_window = two_person_window;
		HIDEALL(); SHOW(two_person_panel);
		break;
	case TWO_FAMILY_SCREEN:
		current_window = two_family_window;
		HIDEALL(); SHOW(two_family_panel);
		break;
	case PEDIGREE_SCREEN:
		current_window = pedigree_window;
		HIDEALL(); SHOW(pedigree_panel);
		break;
	case LIST_SCREEN:
		current_window = list_window;
		HIDEALL(); SHOW(list_panel);
		break;
	}
	place_cursor();
	if (do_message) place_standard_message();
	update_panels();
	doupdate();
}
/*======================================
 * main_menu -- Handle main_menu screen.
 *====================================*/
main_menu ()
{
	display_screen(MAIN_SCREEN);
	switch (interact(main_window, "badruq")) {
	case 'b': browse(NULL); break;
	case 'a': add_menu(); break;
	case 'd': del_menu(); break;
	case 'r': interp_report(); break;
	case 'u': utils_menu(); break;
	case 'q': alldone = TRUE; break;
	}
}
/*==========================================
 * indi_browse -- Handle indi_browse screen.
 *========================================*/
indi_browse (indi)
NODE indi;
{
	if (cur_screen != ONE_PERSON_SCREEN) paint_one_person_screen();
	show_person(indi, one_person_window, 1);
	display_screen(ONE_PERSON_SCREEN);
	return interact(one_person_window, "efmscoygubhirdpnaxtzq");
}
/*========================================
 * fam_browse -- Handle fam_browse screen.
 *======================================*/
fam_browse (fam)
NODE fam;
{
	if (cur_screen != ONE_FAMILY_SCREEN) paint_one_family_screen();
	show_long_family(fam, one_family_window, 1);
	display_screen(ONE_FAMILY_SCREEN);
	return interact(one_family_window, "ehwcnsardxtbzq");
}
/*==============================================
 * tandem_browse -- Handle tandem_browse screen.
 *============================================*/
tandem_browse (indi1, indi2)
NODE indi1, indi2;
{
	if (cur_screen != TWO_PERSON_SCREEN) paint_two_person_screen();
	show_person(indi1, two_person_window, 1);
	show_person(indi2, two_person_window, 8);
	display_screen(TWO_PERSON_SCREEN);
	return interact(two_person_window, "etfmscbdajxq");
}
/*==============================================
 * twofam_browse -- Handle twofam_browse screen.
 *============================================*/
twofam_browse (fam1, fam2)
NODE fam1, fam2;
{
	if (cur_screen != TWO_FAMILY_SCREEN) paint_two_family_screen();
	show_short_family(fam1, two_family_window, 1);
	show_short_family(fam2, two_family_window, 8);
	display_screen(TWO_FAMILY_SCREEN);
	return interact(two_family_window, "etbhwxjq");
}
/*========================================
 * ped_browse -- Handle ped_browse screen.
 *======================================*/
ped_browse (indi)
NODE indi;
{
	if (cur_screen != PEDIGREE_SCREEN) paint_pedigree_screen();
	show_pedigree(indi);
	display_screen(PEDIGREE_SCREEN);
	return interact(pedigree_window, "ifmcgq");
}
/*==========================================
 * list_browse -- Handle list_browse screen.
 *========================================*/
list_browse (seq, top, cur, mark)
INDISEQ seq;
INT top, cur, mark;
{
	if (cur_screen != LIST_SCREEN) paint_list_screen();
	show_list(seq, top, cur, mark);
	display_screen(LIST_SCREEN);
	return interact(list_window, "jkeimdtbanxq");
}
/*=======================================
 * ask_for_string -- Ask user for string.
 *=====================================*/
STRING ask_for_string (ttl, prmpt)
STRING ttl, prmpt;
{
	WINDOW *win = ask_window;
	STRING rv, p;
	SHOW(ask_panel);
	werase(win);
	box(win, 0, 0);
	mvwaddstr(win, 1, 1, ttl);
	mvwaddstr(win, 2, 1, prmpt);
	update_panels();
	doupdate();
	rv = get_answer(win, prmpt);
	hide_panel(ask_panel);
	update_panels();
	if (!rv) return "";
	p = rv;
	while (chartype(*p) == WHITE)
		p++;
	striptrail(p);
	return p;
}
/*========================================
 * ask_for_char -- Ask user for character.
 *======================================*/
INT ask_for_char (ttl, prmpt, ptrn)
STRING ttl, prmpt, ptrn;
{
	WINDOW *win = ask_window;
	INT rv;
	SHOW(ask_panel);
	werase(win);
	box(win, 0, 0);
	mvwaddstr(win, 1, 2, ttl);
	mvwaddstr(win, 2, 2, prmpt);
	update_panels();
	doupdate();
	rv = interact(win, ptrn);
	hide_panel(ask_panel);
	update_panels();
	return rv;
}
/*============================================
 * ask_for_char_msg -- Ask user for character.
 *==========================================*/
INT ask_for_char_msg (msg, ttl, prmpt, ptrn)
STRING msg, ttl, prmpt, ptrn;
{
	WINDOW *win = ask_msg_window;
	INT rv;
	SHOW(ask_msg_panel);
	werase(win);
	box(win, 0, 0);
	mvwaddstr(win, 1, 2, msg);
	mvwaddstr(win, 2, 2, ttl);
	mvwaddstr(win, 3, 2, prmpt);
	update_panels();
	doupdate();
	rv = interact(win, ptrn);
	hide_panel(ask_msg_panel);
	update_panels();
	return rv;
}
/*=============================================
 * choose_from_list -- Choose from string list.
 *===========================================*/
INT choose_from_list (ttl, no, pstrngs)
STRING ttl;
INT no;
STRING *pstrngs;
{
	WINDOW *win;
	INT rv;
	SHOW(choose_from_list_pnl);
	win = choose_win(no);
	werase(win);
	box(win, 0, 0);
	update_panels();
	doupdate();
	rv = list_interact(win, ttl, no, pstrngs);
	HIDE(choose_from_list_pnl);
	update_panels();
	return rv;
}
/*==================================================================
 * choose_one_from_indiseq -- Have user choose person from sequence.
 *================================================================*/
INT choose_one_from_indiseq (ttl, seq)
STRING ttl;
INDISEQ seq;
{
	WINDOW *win;
	INT rv, len;
	ASSERT(seq);
	len = length_indiseq(seq);
	SHOW(choose_from_list_pnl);
	win = choose_win(len);
	werase(win);
	box(win, 0, 0);
	update_panels();
	doupdate();
	rv = indiseq_interact(win, ttl, seq);
	HIDE(choose_from_list_pnl);
	update_panels();
	return rv;
}
/*==============================================================
 * choose_list_from_indiseq -- Have user choose subsequence from
 *   person sequence.
 *=============================================================*/
INDISEQ choose_list_from_indiseq (ttl, seq)
STRING ttl;
INDISEQ seq;
{
	WINDOW *win;
	INT len;
	ASSERT(seq);
	len = length_indiseq(seq);
	SHOW(choose_from_list_pnl);
	win = choose_win(len);
	werase(win);
	box(win, 0, 0);
	update_panels();
	doupdate();
	seq = indiseq_list_interact(win, ttl, seq);
	HIDE(choose_from_list_pnl);
	update_panels();
	return seq;
}
/*=============================
 * add_menu -- Handle add menu.
 *===========================*/
add_menu ()
{
	NODE node;
	SHOW(add_menu_pnl);
	update_panels();
	doupdate();
	HIDE(add_menu_pnl);
	switch (interact(add_menu_win, "pfcsq")) {
	case 'p':
		node = add_indi_by_edit();
		if (node) browse(node);
		break;
	case 'f':
		add_family(NULL, NULL, NULL); break;
	case 'c':
		add_child(NULL, NULL); break;
	case 's':
		add_spouse(NULL, NULL, TRUE); break;
	case 'q':
		break;
	}
	HIDE(add_menu_pnl);
	update_panels();
}
/*================================
 * del_menu -- Handle delete menu.
 *==============================*/
del_menu ()
{
	SHOW(del_menu_pnl);
	update_panels();
	doupdate();
	HIDE(del_menu_pnl);
	switch (interact(del_menu_win, "csiq")) {
	case 'c':
		remove_child(NULL, FALSE); break;
	case 's':
		remove_spouse(NULL, NULL, FALSE); break;
	case 'i':
		delete_indi(NULL, TRUE); break;
	case 'q':
		break;
	}
	HIDE(del_menu_pnl);
	update_panels();
}
/*=====================================
 * utils_menu -- Handle utilities menu.
 *===================================*/
utils_menu ()
{
	SHOW(utils_menu_pnl);
	update_panels();
	doupdate();
	HIDE(utils_menu_pnl);
	switch (interact(utils_menu_win, "srkidmeq")) {
	case 's': archive_in_file(); break;
	case 'r': restore_from_file(); break;
	case 'k': key_util(); break;
	case 'i': who_is_he_she(); break;
	case 'd': show_database_stats(); break;
	case 'm': show_all_stats(); break;
	case 'e': edit_plac_abbvs(); break;
	case 'q': break;
	}
	HIDE(utils_menu_pnl);
	update_panels();
}
/*================================
 * interact -- Interact with user.
 *==============================*/
interact (win, str)
WINDOW *win;
STRING str;
{
	INT c, i, n = strlen(str);
	while (TRUE) {
		c = wgetch(win);
		place_standard_message();
		for (i = 0; i < n; i++) {
			if (c == str[i]) return c;
		}
		beep();
	}
}
/*=======================================================
 * get_answer -- Have user respond with string in WINDOW.
 *=====================================================*/
STRING get_answer (win, prmpt)
WINDOW *win;
STRING prmpt;
{
	static char lcl[100];
	echo();
	mvwgetstr(win, 2, strlen(prmpt) + 2, lcl);
	noecho();
	return lcl;
}
/*============================================================
 * win_list_init -- Create list of WINDOWs of increasing size.
 *==========================================================*/
static WINDOW *list_wins[VIEWABLE];
win_list_init ()
{
	INT i, row, col = (COLS - 73)/2;
	for (i = 0; i < VIEWABLE; i++) {
		row = (LINES - i - 6)/2;
		list_wins[i] = newwin(i+6, 73, row, col);
	}
}
/*==========================================
 * choose_win -- Choose window to hold list.
 *========================================*/
WINDOW *choose_win (len)
INT len;	/* list length */
{
	WINDOW *win = list_wins[VIEWABLE-1];
	if (len <= VIEWABLE) win = list_wins[len-1];
	replace_panel(choose_from_list_pnl, win);
	return win;
}
/*======================================================
 * indiseq_interact -- Interact with user over sequence.
 *====================================================*/
INT indiseq_interact (win, ttl, seq)
WINDOW *win;
STRING ttl;
INDISEQ seq;
{
	INT c, top, cur, len, row;
	top = cur = 0;
	len = length_indiseq(seq);
	werase(win);
	box(win, 0, 0);
	mvwaddstr(win, 1, 1, ttl);
	row = len > VIEWABLE ? VIEWABLE + 2 : len + 2;
	show_horz_line(win, row++, 0, 73);
	mvwaddstr(win, row, 2, "Commands:   j Move down     k Move up    i Select     q Quit");
	while (TRUE) {
		shw_list(win, seq, len, top, cur);
		wmove(win, row, 11);
		switch (c = interact(win, "jkiq")) {
		case 'j':
			if (cur >= len - 1) break;
			cur++;
			if (cur >= top + VIEWABLE) top++;
			break;
		case 'k':
			if (cur <= 0) break;
			cur--;
			if (cur + 1 == top) top--;
			break;
		case 'i':
			return cur;
		case 'q':
		default:
			return -1;
		}
	}
}
INDISEQ indiseq_list_interact (win, ttl, seq)
WINDOW *win;
STRING ttl;
INDISEQ seq;
{
	INT c, top, cur, len, len0, row;

	top = cur = 0;
	len = len0 = length_indiseq(seq);
	werase(win);
	box(win, 0, 0);
	mvwaddstr(win, 1, 1, ttl);
	row = len0 > VIEWABLE ? VIEWABLE + 2 : len0 + 2;
	show_horz_line(win, row++, 0, 73);
	mvwaddstr(win, row, 2, "Commands:  j Move down   k Move up  d Delete   i Select   q Quit");
	while (TRUE) {
		shw_list(win, seq, len0, top, cur);
		wmove(win, row, 11);
		update_panels();
		doupdate();
		switch (c = interact(win, "jkdiq")) {
		case 'j':
			if (cur >= len - 1) break;
			cur++;
			if (cur >= top + VIEWABLE) top++;
			break;
		case 'k':
			if (cur <= 0) break;
			cur--;
			if (cur + 1 == top) top--;
			break;
		case 'd':
			delete_indiseq(seq, NULL, NULL, cur);
			if (--len == 0) {
				remove_indiseq(seq, FALSE);
				return NULL;
			}
			if (cur == len) cur--;
			if (cur < top) top = cur;
			break;
		case 'i':
			return seq;
		case 'q':
		default:
			return NULL;
		}
	}
}
shw_list (win, seq, len0, top, cur)
WINDOW *win;
INDISEQ seq;
INT top, cur, len0;
{
	INT i, j, row, len;
	INT x, y;
	STRING key, name;
	NODE indi;
	char scratch[200], *p;

	len = length_indiseq(seq);
	row = 2;
	j = len0 > VIEWABLE ? VIEWABLE : len0;
	for (i = 0; i < j; i++)
		mvwaddstr(win, row++, 1, empstr71);
	row = 2;
	for (i = top, j = 0; j < VIEWABLE && i < len; i++, j++) {
		if (i == cur) mvwaddch(win, row, 3, '>');
		mvwaddstr(win, row, 4, sprn(IData(seq)[i]));
		row++;
	}
}
shw_list_of_strings (win, strings, len, top, cur)
WINDOW *win;
STRING *strings;
INT len, top, cur;
{
	INT i, j, row;
	STRING key, name;
	NODE indi;
	char scratch[200], *p;

	row = len > VIEWABLE ? VIEWABLE + 1 : len + 1;
	for (i = 2; i <= row; i++)
		mvwaddstr(win, i, 1, empstr71);
	row = 2;
	for (i = top, j = 0; j < VIEWABLE && i < len; i++, j++) {
		if (i == cur) mvwaddch(win, row, 3, '>');
		mvwaddstr(win, row, 4, strings[i]);
		row++;
	}
}
/*===============================================
 * list_interact -- Interact with user over list.
 *=============================================*/
INT list_interact(win, ttl, len, strings)
WINDOW *win;	/* interaction WINDOW */
STRING ttl;	/* title */
INT len;	/* list length */
STRING *strings;/* string list */
{
	INT c, top = 0, cur = 0, row;
	while (TRUE) {
		werase(win);
		box(win, 0, 0);
		mvwaddstr(win, 1, 1, ttl);
		row = len > VIEWABLE ? VIEWABLE + 2 : len + 2;
		show_horz_line(win, row++, 0, 73);
		mvwaddstr(win, row, 2, "Commands:   j Move down     k Move up    i Select     q Quit");
		shw_list_of_strings(win, strings, len, top, cur);
		switch (c = interact(win, "jkiq")) {
		case 'j':
			if (cur >= len - 1) break;
			cur++;
			if (cur >= top + VIEWABLE) top++;
			break;
		case 'k':
			if (cur <= 0) break;
			cur--;
			if (cur + 1 == top) top--;
			break;
		case 'i':
			return cur;
		case 'q':
		default:
			return -1;
		}
	}
}
/*=======================================
 * message - Simple interface to mprintf.
 *=====================================*/
message (s)
char *s;
{
	mprintf("%s", s);
}
/*======================================================
 * mprintf -- Called as mprintf(fmt, arg, arg, arg, ...)
 *=====================================================*/
mprintf (va_alist)
va_dcl
{
	va_list args;
	char *fmt;
	INT row;
	va_start(args);
	fmt = va_arg(args, char *);
	wmove(current_window, row = message_row(), 2);
	if (cur_screen != LIST_SCREEN) {
		wclrtoeol(current_window);
		mvwaddch(current_window, row, 78, ACS_VLINE);
	} else
		mvwaddstr(current_window, row, 2, empstr);
	wmove(current_window, row, 2);
	vwprintw(current_window, fmt, args);
	place_cursor();
	update_panels();
}
/*====================================================
 * message_string -- Return background message string.
 *==================================================*/
STRING message_string ()
{
	switch (cur_screen) {
	case MAIN_SCREEN:
		return "LifeLines -- Main Menu";
	case ONE_PERSON_SCREEN:
		return "LifeLines -- Person Browse Mode";
	case ONE_FAMILY_SCREEN:
		return "LifeLines -- Family Browse Mode";
	case PEDIGREE_SCREEN:
		return "LifeLines -- Pedigree Browse Mode";
	case TWO_PERSON_SCREEN:
		return "LifeLines -- Tandem Browse Mode";
	case TWO_FAMILY_SCREEN:
		return "LifeLines -- Two Family Browse Mode";
	case LIST_SCREEN:
		return "LifeLines -- List Browse Mode";
	default:
		return NULL;
	}
}
/*=============================================
 * message_row -- Return row of message string.
 *===========================================*/
INT message_row ()
{
	switch (cur_screen) {
	case MAIN_SCREEN: return 13;
	case ONE_PERSON_SCREEN: return 17;
	case ONE_FAMILY_SCREEN: return 17;
	case PEDIGREE_SCREEN: return 21;
	case TWO_PERSON_SCREEN: return 21;
	case TWO_FAMILY_SCREEN: return 20;
	case LIST_SCREEN: return 21;
	default: return 23;
	}
}
place_standard_message ()
{
	STRING str = message_string();
	if (str) message(str);
}
/*=======================================================
 * wprintf -- Called as wprintf(fmt, arg, arg, arg, ...).
 *=====================================================*/
/*VARARGS*/
wprintf (va_alist)
va_dcl
{
	va_list args;
	char *fmt;
	if (panel_hidden(std_out_pnl)) {
		scrollok(std_out_win, TRUE);
		werase(std_out_win);
		show_panel(std_out_pnl);
		box(std_out_box_win, 0, 0);
		wmove(std_out_win, 0, 0);
	}
	va_start(args);
	fmt = va_arg(args, char *);
	vwprintw(std_out_win, fmt, args);
	touchwin(std_out_box_win);
	update_panels();
	doupdate();
}
wputs (str)
STRING str;
{
	if (panel_hidden(std_out_pnl)) {
		scrollok(std_out_win, TRUE);
		werase(std_out_win);
		show_panel(std_out_pnl);
		box(std_out_box_win, 0, 0);
		wmove(std_out_win, 0, 0);
	}
	waddstr(std_out_win, str);
	touchwin(std_out_box_win);
	update_panels();
	doupdate();
}
/*========================================
 * show_horz_line -- Draw horizontal line.
 *======================================*/
show_horz_line (win, row, col, len)
WINDOW *win;
INT row, col, len;
{
	INT i;
	mvwaddch(win, row, col, ACS_LTEE);
	for (i = 0; i < len-2; i++)
		waddch(win, ACS_HLINE);
	waddch(win, ACS_RTEE);
}
/*======================================
 * show_vert_line -- Draw vertical line.
 *====================================*/
show_vert_line (win, row, col, len)
WINDOW *win;
INT row, col, len;
{
	INT i;
	mvwaddch(win, row++, col, ACS_TTEE);
	for (i = 0; i < len-2; i++)
		mvwaddch(win, row++, col, ACS_VLINE);
	mvwaddch(win, row, col, ACS_BTEE);
}
/*==============================================
 * place_cursor -- Move to idle cursor location.
 *============================================*/
place_cursor ()
{
	INT row, col;
	switch (cur_screen) {
	case MAIN_SCREEN: row = 5; col = 30; break;
	case ONE_PERSON_SCREEN: row = 8; col = 43; break;
	case ONE_FAMILY_SCREEN: row = 10; col = 43; break;
	case PEDIGREE_SCREEN: row = 17; col = 45; break;
	case TWO_PERSON_SCREEN: row = 15; col = 43; break;
	case TWO_FAMILY_SCREEN: row = 15; col = 47; break;
	case LIST_SCREEN: row = 8; col = 75; break;
	default: row = 1; col = 1; break;
	}
	wmove(current_window, row, col);
}
