/*===========================================================
 * lex.c -- Low level lexing code of the report interpreter.
 * Copyright(c) by Thomas T. Wetmore IV; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 07 Sep 93 - modified
 *===========================================================
 */
#include <stdio.h>
#include "standard.h"
#include "stdio.h"
#include "table.h"
#include "gedcom.h"
#include "interp.h"
#undef FORLIST
#include "y.tab.h"

extern FILE *ifp;
extern INT rplineno;
extern INTERP yylval;
static char scratch[200];

/*========================================================
 * yylex -- Main lexer entry point for report interpreter.
 *======================================================*/
int yylex()
{
	INT lex = lowyylex();
	return lex;
}
/*============================
 * lowyylex -- Lexer function.
 *==========================*/
int lowyylex ()
{
	INT c, t, retval, ivalue;
	STRING p = scratch;
	while (TRUE) {
		while ((t = chartype(c = inchar())) == WHITE)
			;
		if (c != '/') break;
		if ((c = inchar()) != '*') {
			unchar(c);
			return '/';
		}
		while (TRUE) {
			while ((c = inchar()) != '*' && c != EOF)
				;
			if (c == EOF) return 0;
			while ((c = inchar()) == '*')
				;
			if (c == '/') break;
			if (c == EOF) return 0;
		}
	}
	if (t == LETTER) {
		p = scratch;
		while (t == LETTER || t == DIGIT || c == '_') {
			*p++ = c;
			t = chartype(c = inchar());
		}
		*p = 0;
		unchar(c);
		if (reserved(scratch, &retval)) return retval;
		yylval = (INTERP) strsave(scratch);
		return IDEN;
	}
	if (t == DIGIT) {
		ivalue = 0;
		while (t == DIGIT) {
			ivalue = ivalue*10 + c - '0';
			t = chartype(c = inchar());
		}
		unchar(c);
		yylval = (INTERP) ivalue;
		return ICONS;
	}
	if (c == '"') {
		p = scratch;
		while (TRUE) {
			while ((c = inchar()) != EOF && c != '"' && c != '\\')
				*p++ = c;
			if (c == 0 || c == '"') {
				*p = 0;
				yylval = literal_node(scratch);
				return LITERAL;
			}
			switch (c = inchar()) {
			case 'n':
				*p++ = '\n'; break;
			case 't':
				*p++ = '\t'; break;
			case 'v':
				*p++ = '\v'; break;
			case 'r':
				*p++ = '\r'; break;
			case 'b':
				*p++ = '\b'; break;
			case 'f':
				*p++ = '\f'; break;
			case '"':
				*p++ = '"'; break;
			case '\\':
				*p++ = '\\'; break;
			case EOF:
				*p = 0;
				yylval = literal_node(scratch);
				return LITERAL;
			default:
				*p++ = c; break;
			}
		}
	}
	if (c == EOF) return 0;
	return c;
}

static struct {
	STRING rword;
	INT val;
} rwordtable[] = {
	"call",		CALL,
	"children",	CHILDREN,
	"else",		ELSE,
	"elsif",	ELSIF,
	"families",	FAMILIES,
	"forfam",	FORFAM,
	"forindiset",	FORINDISET,
	"forindi",	FORINDI,
	"forlist",	FORLIST,
	"fornodes",	FORNODES,
	"fornotes",	FORNOTES,
	"if",		IF,
	"proc",		PROC,
	"spouses",	SPOUSES,
	"traverse",	TRAVERSE,
	"while",	WHILE,
};
static nrwords = 16;

/*==============================================
 * reserved -- Check if string is reserved word.
 *============================================*/
BOOLEAN reserved (word, pval)
STRING word;
INT *pval;
{
	INT i;
	for (i = 0; i < nrwords; i++) {
		if (!strcmp(word, rwordtable[i].rword)) {
			*pval = rwordtable[i].val;
			return TRUE;
		}
	}
	return FALSE;
}
/*================================================================
 * inchar -- Read char from input file; keep track of line number.
 *==============================================================*/
inchar ()
{
	INT c = getc(ifp);
	if (c == '\n') rplineno++;
	return c;
}
/*==================================================================
 * unchar -- Put char back on input file; keep track of line number.
 *================================================================*/
unchar (c)
INT c;
{
	ungetc(c, ifp);
	if (c == '\n') rplineno--;
}
