/*================================================================
 * names.c -- Handle name indexing.
 * Copyright(c) 1992 by Thomas T. Wetmore IV; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 02 Sep 93 - modified
 *================================================================
 */
#include "standard.h"
#include "table.h"
#include "btree.h"
#include "gedcom.h"

extern BTREE BTR;

RKEY name2rkey();
RKEY namerkey, *namerkeys;
INT namelen, namereclen, *nameoffs;
INT maxnamelen = 0;
STRING *namestrs;
static STRING inrec = NULL;
static int old = 0;
static INT codeof();
static STRING parts_to_name();
static squeeze();

/*=====================================================
 * getnamerec -- Read name record and store in globals.
 *===================================================*/
BOOLEAN getnamerec (name)
STRING name;
{
	STRING p;
	INT i;

/* Convert name to key and read name record */
	namerkey = name2rkey(name);
	if (inrec) stdfree(inrec);
	p = inrec = (STRING) getrecord(BTR, namerkey, &namereclen);
	if (!inrec) {
		namelen = 0;
		if (maxnamelen == 0) {
			maxnamelen = 10;
			namerkeys= (RKEY *) stdalloc(10*sizeof(RKEY));
			nameoffs= (INT *) stdalloc(10*sizeof(INT));
			namestrs= (STRING *) stdalloc(10*sizeof(STRING));
		}
		return FALSE;
	}

/* Store name record in data structures */
	memcpy (&namelen, p, sizeof(INT));
	p += sizeof(INT);
	if (namelen >= maxnamelen - 1) {
		if (maxnamelen != 0) {
			stdfree(namerkeys);
			stdfree(nameoffs);
			stdfree(namestrs);
		}
		maxnamelen = namelen + 10;
		namerkeys= (RKEY *) stdalloc((maxnamelen)*sizeof(RKEY));
		nameoffs= (INT *) stdalloc((maxnamelen)*sizeof(INT));
		namestrs= (STRING *) stdalloc((maxnamelen)*sizeof(STRING));
	}
	for (i = 0; i < namelen; i++) {
		memcpy(&namerkeys[i], p, sizeof(RKEY));
		p += sizeof(RKEY);
	}
	for (i = 0; i < namelen; i++) {
		memcpy(&nameoffs[i], p, sizeof(INT));
		p += sizeof(INT);
	}
	for (i = 0; i < namelen; i++)
		namestrs[i] = p + nameoffs[i];
	return TRUE;
}
/*=============================================
 * name2rkey - Convert name to name record key.
 *===========================================*/
RKEY name2rkey (name)
STRING name;
{
	RKEY rkey;
	STRING sdex = soundex(getsurname(name));
	char finitial = getfinitial(name);
	rkey.r_rkey[0] = rkey.r_rkey[1] = ' ';
	rkey.r_rkey[2] = 'N';
	rkey.r_rkey[3] = finitial;
	rkey.r_rkey[4] = *sdex++;
	rkey.r_rkey[5] = *sdex++;
	rkey.r_rkey[6] = *sdex++;
	rkey.r_rkey[7] = *sdex;
	return rkey;
}
/*================================================================
 * getsurname -- Return surname of name; if name has no surname or
 *   it starts with non-letter, return ____.
 *==============================================================*/
STRING getsurname (name)
STRING name;
{
	INT c;
	static char buffer[3][MAXLINELEN+1];
	static INT dex = 0;
	STRING p, surname;
	if (++dex > 2) dex = 0;
	p = surname = buffer[dex];
	while ((c = *name++) && c != '/')
		;
	if (c == 0) return "____";
	while (iswhite(c = *name++))
		;
	if (c == 0 || c == '/'|| !isletter(c)) return "____";
	*p++ = c;
	while ((c = *name++) && c != '/') {
		if (c == '\\') c = '/';
		*p++ = c;
	}
	*p = 0;
	return surname;
}
/*==========================================================
 * getfinitial -- Return first initial of name, capitalized;
 *   if none return $.
 *========================================================*/
INT getfinitial (name)
STRING name;
{
	INT c;
	while (TRUE) {
		while (iswhite(c = *name++))
			;
		if (isletter(c)) return toupper(c);
		if (c == 0) return '$';
		if (c != '/') return '$';
		while ((c = *name++) != '/' && c != 0)
			;
		if (c == 0) return '$';
	}
}
/*==================================================================
 * soundex -- Return SOUNDEX code of name; any case; return Z999 for
 *   problem names.
 *================================================================*/
STRING soundex (name)
STRING name;	/* surname */
{
	static char scratch[MAXNAMELEN+2];
	STRING p = name, q = scratch;
	INT c, i, j;
	if (!name || strlen(name) > MAXNAMELEN || !strcmp(name, "____"))
		return "Z999";
	p = name;
	q = scratch;
	while (c = *p++)
		*q++ = toupper(c);
	*q = 0;
	p = q = &scratch[1];
	i = 1;
	old = 0;
	while ((c = *p++) && i < 4) {
		if ((j = codeof(c)) == 0) continue;
		*q++ = j;
		i++;
	}
	while (i < 4) {
		*q++ = '0';
		i++;
	}
	*q = 0;
	return scratch;
}
/*========================================
 * codeof -- Return letter's SOUNDEX code.
 *======================================*/
static INT codeof (letter)
int letter;
{
	int new = 0;
	switch (letter) {
	case 'B': case 'P': case 'F': case 'V':
		new = '1'; break;
	case 'C': case 'S': case 'K': case 'G':
	case 'J': case 'Q': case 'X': case 'Z':
		new = '2'; break;
	case 'D': case 'T':
		new = '3'; break;
	case 'L':
		new = '4'; break;
	case 'M': case 'N':
		new = '5'; break;
	case 'R':
		new = '6'; break;
	default:	/* new stays zero */
		break;
	}
	if (new == 0) {
		old = 0;
		return 0;
	}
	if (new == old) return 0;
	old = new;
	return new;
}
/*===========================================
 * add_name -- Add new entry to name records.
 *=========================================*/
BOOLEAN add_name (name, key)
STRING name;	/* name of person */
STRING key;	/* key of person */
{
	STRING rec, p;
	INT i, len, off;
	RKEY rkey;
	rkey = str2rkey(key);
	(void) getnamerec(name);
	for (i = 0; i < namelen; i++) {
		if (!strncmp(rkey.r_rkey, namerkeys[i].r_rkey, 8) &&
		    !strcmp(name, namestrs[i]))
			return TRUE;
	}
	namerkeys[namelen] = rkey;
	namestrs[namelen] = name;
	namelen++;
	p = rec = (STRING) stdalloc(namereclen + sizeof(RKEY) +
	    sizeof(INT) + strlen(name) + 10);
	len = 0;
	memcpy(p, &namelen, sizeof(INT));
	p += sizeof(INT);
	len += sizeof(INT);
	for (i = 0; i < namelen; i++) {
		memcpy(p, &namerkeys[i], sizeof(RKEY));
		p += sizeof(RKEY);
		len += sizeof(RKEY);
	}
	off = 0;
	for (i = 0; i < namelen; i++) {
		memcpy(p, &off, sizeof(INT));
		p += sizeof(INT);
		len += sizeof(INT);
		off += strlen(namestrs[i]) + 1;
	}
	for (i = 0; i < namelen; i++) {
		memcpy(p, namestrs[i], strlen(namestrs[i]) + 1);
		p += strlen(namestrs[i]) + 1;
		len += strlen(namestrs[i]) + 1;
	}
	addrecord(BTR, namerkey, rec, len);
	stdfree(rec);
	return TRUE;
}
/*===============================================
 * remove_name -- Remove entry from name records.
 *=============================================*/
BOOLEAN remove_name (name, key)
STRING name;	/* name of person */
STRING key;	/* key of person */
{
	STRING rec, p;
	INT i, len, off;
	BOOLEAN found;
	RKEY rkey;
	rkey = str2rkey(key);
	(void) getnamerec(name);
	found = FALSE;
	for (i = 0; i < namelen; i++) {
		if (!strncmp(rkey.r_rkey, namerkeys[i].r_rkey, 8) &&
		    !strcmp(name, namestrs[i])) {
			found = TRUE;
			break;
		}
	}
	if (!found) return FALSE;
	namelen--;
	for ( ; i < namelen; i++) {
		namerkeys[i] = namerkeys[i+1];
		namestrs[i] = namestrs[i+1];
	}
	p = rec = (STRING) stdalloc(namereclen);
	len = 0;
	memcpy(p, &namelen, sizeof(INT));
	p += sizeof(INT);
	len += sizeof(INT);
	for (i = 0; i < namelen; i++) {
		memcpy(p, &namerkeys[i], sizeof(RKEY));
		p += sizeof(RKEY);
		len += sizeof(RKEY);
	}
	off = 0;
	for (i = 0; i < namelen; i++) {
		memcpy(p, &off, sizeof(INT));
		p += sizeof(INT);
		len += sizeof(INT);
		off += strlen(namestrs[i]) + 1;
	}
	for (i = 0; i < namelen; i++) {
		memcpy(p, namestrs[i], strlen(namestrs[i]) + 1);
		p += strlen(namestrs[i]) + 1;
		len += strlen(namestrs[i]) + 1;
	}
	addrecord(BTR, namerkey, rec, len);
	stdfree(rec);
	return TRUE;
}
/*===============================================
 * replace_name -- Replace entry in name records.
 *=============================================*/
BOOLEAN replace_name (old, new, key)
STRING old;	/* old name */
STRING new;	/* new name */
STRING key;	/* person key */
{
	remove_name(old, key);
	add_name(new, key);
	return TRUE;
}
/*===============================================================
 * exactmatch -- Check if names match; first must be contained in
 *   second.
 *=============================================================*/
BOOLEAN exactmatch (partial, complete)
STRING partial;		/* name from user */
STRING complete;	/* GEDCOM name */
{
	char part[MAXNAMELEN+2], comp[MAXNAMELEN+2], *p, *q;
	BOOLEAN okay;

	if (strlen(partial) > MAXNAMELEN || strlen(complete) > MAXNAMELEN)
		return FALSE;
	squeeze(partial, part);
	squeeze(complete, comp);
	q = comp;
	for (p = part; *p; p += strlen(p) + 1) {
		okay = FALSE;
		for (; !okay && *q; q += strlen(q) + 1) {
			if (piecematch(p, q)) okay = TRUE;
		}
		if (!okay) return FALSE;
	}
	return TRUE;
}
/*================================================================
 * piecematch -- Match partial word with complete; must begin with
 *   same letter; letters in partial must be in same order as in
 *   complete; case insensitive.
 *==============================================================*/
BOOLEAN piecematch (part, comp)
STRING part, comp;
{
	if (*part++ != *comp++) return FALSE;
	while (*part && *comp) {
		if (*part == *comp++) part++;
	}
	return *part == 0;
}
/*===============================================================
 * squeeze -- Squeeze string to superstring, string of uppercase,
 *   0-terminated words, ending with another 0; non-letters not
 *   copied; eg., `Anna /Van Cott/' maps to `ANNA\0VANCOTT\0\0'.
 *=============================================================*/
static squeeze (in, out)
STRING in;	/* string of words */
STRING out;	/* superstring of words */
{
	INT c;
	STRING out0 = out;
	while ((c = *in++) && chartype(c) != LETTER)
		;
	if (c == 0) {
		*out++ = 0; *out = 0;
		return;
	}
	while (TRUE) {
		*out++ = toupper(c);
		while ((c = *in++) && c != '/' && chartype(c) != WHITE) {
			if (chartype(c) == LETTER) *out++ = toupper(c);
		}
		if (c == 0) {
			*out++ = 0; *out = 0;
			return;
		}
		*out++ = 0;
		while ((c = *in++) && chartype(c) != LETTER)
			;
		if (c == 0) {
			*out++ = 0; *out = 0;
			return;
		}
	}
}

static STRING *localkeys = NULL;
static INT nlocalkeys = 0;
static INT maxlocalkeys = 0;
/*=====================================================
 * get_names -- Find all persons who match name or key.
 *===================================================*/
STRING *get_names (name, pnum, pkeys, exact)
STRING name;
INT *pnum;
STRING **pkeys;
BOOLEAN exact;	/* unused! */
{
	INT i, nmatch;
	STRING *strs, *id_by_key();
	if (strs = id_by_key(name, pkeys)) {
		*pnum = 1;
		return strs;
	}
	if (nlocalkeys) {
		for (i = 0; i < nlocalkeys; i++)
			stdfree(localkeys[i]);
	}
	nlocalkeys = 0;
	if (!getnamerec(name)) {
		*pnum = 0;
		return NULL;
	}
	nmatch = 0;
	for (i = 0; i < namelen; i++) {
		if (exactmatch(name, namestrs[i])) {
			if (i != nmatch) {
				namestrs[nmatch] = namestrs[i];
				namerkeys[nmatch] = namerkeys[i];
			}
			nmatch++;
		}
	}
	*pnum = namelen = nmatch;
	if (namelen > maxlocalkeys) {
		if (maxlocalkeys) stdfree(localkeys);
		localkeys=(STRING *) stdalloc((namelen + 5)*sizeof(STRING));
		maxlocalkeys = namelen + 5;
	}
	for (i = 0; i < namelen; i++)
		localkeys[i] = strsave(rkey2str(namerkeys[i]));
	*pkeys = localkeys;
	return namestrs;
}

static STRING nextpiece();

/*=====================================
 * namecmp -- Compare two GEDCOM names.
 *===================================*/
int namecmp (name1, name2)
STRING name1, name2;
{
	char sqz1[MAXNAMELEN], sqz2[MAXNAMELEN];
	STRING p1 = sqz1, p2 = sqz2;
	INT r = strcmp(getsurname(name1), getsurname(name2));
	if (r) return r;
	r = getfinitial(name1) - getfinitial(name2);
	if (r) return r;
	cmpsqueeze(name1, p1);
	cmpsqueeze(name2, p2);
	while (*p1 && *p2) {
		r = strcmp(p1, p2);
		if (r) return r;
		p1 += strlen(p1) + 1;
		p2 += strlen(p2) + 1;
	}
	if (*p1) return 1;
	if (*p2) return -1;
	return 0;
}
/*============================================================
 * cmpsqueeze -- Squeeze GEDCOM name to superstring of givens.
 *==========================================================*/
cmpsqueeze (in, out)
STRING in, out;
{
	INT c;
	while (in = nextpiece(in)) {
		while (TRUE) {
			c = *in++;
			if (iswhite(c) || c == '/' || c == 0) {
				*out++ = 0;
				--in;
				break;
			}
			*out++ = c;
		}
	}
	*out = 0;
}
/*======================================
 * givens -- Return given names of name.
 *====================================*/
STRING givens (name)
STRING name;
{
	INT c;
	static char scratch[MAXNAMELEN+1];
	STRING out = scratch;
	while (name = nextpiece(name)) {
		while (TRUE) {
			if ((c = *name++) == 0) {
				if (*(out-1) == ' ') --out;
				*out = 0;
				return scratch;
			}
			if (iswhite(c) || c == '/') {
				*out++ = ' ';
				--name;
				break;
			}
			*out++ = c;
		}
	}
	if (*(out-1) == ' ') --out;
	*out = 0;
	return scratch;
}
/*=========================================
 * nextpiece -- Return next word in string.
 *=======================================*/
static STRING nextpiece (in)
STRING in;
{
	int c;
	while (TRUE) {
		while (iswhite(c = *in++))
			;
		if (c == 0) return NULL;
		if (c != '/') return --in;
		while ((c = *in++) && c != '/')
			;
		if (c == 0) return NULL;
	}
}
/*=====================================================================
 * trim_name -- Trim GEDCOM name to another that is less or equal to
 *   given length (but not shorter than the first initial and surname).
 *===================================================================*/
#define MAXPARTS 100
STRING trim_name (name, len)
STRING name;
INT len;
{
	STRING parts[MAXPARTS];
	INT i, sdex = -1, nparts;
	name_to_parts(name, parts);
	name = parts_to_name(parts);
	if (strlen(name) <= len + 2) return name;
	for (i = 0; i < MAXPARTS; i++) {
		if (!parts[i]) break;
		if (*parts[i] == '/') sdex = i;
	}
	nparts = i;
	if (sdex == -1) FATAL();
	for (i = sdex-1; i >= 0; --i) {
		*(parts[i] + 1) = 0;
		name = parts_to_name(parts);
		if (strlen(name) <= len + 2) return name;
	}
	for (i = sdex-1; i >= 1; --i) {
		parts[i] = NULL;
		name = parts_to_name(parts);
		if (strlen(name) <= len + 2) return name;
	}
	for (i = nparts-1; i > sdex; --i) {
		parts[i] = NULL;
		name = parts_to_name(parts);
		if (strlen(name) <= len + 2) return name;
	}
	return name;
}
/*==================================================================
 * name_to_parts -- Convert GEDCOM name to parts list; slashes kept.
 *================================================================*/
static name_to_parts (name, parts)
STRING name;
STRING *parts;
{
	static char scratch[MAXNAMELEN+1];
	STRING p = scratch;
	INT c, i = 0;

	if (strlen(name) > MAXNAMELEN) FATAL();
	for (i = 0; i < MAXPARTS; i++)
		parts[i] = NULL;
	i = 0;
	while (TRUE) {
		while (iswhite(c = *name++))
			;
		if (c == 0) return;
		if (i >= MAXPARTS) FATAL();
		parts[i++] = p;
		*p++ = c;
		if (c == '/') {
			while ((c = *name++) && c != '/')
				*p++ = c;
			if (c == 0) FATAL();
			*p++ = c;
			*p++ = 0;
		} else {
			while ((c = *name++) && !iswhite(c) && c != '/')
				*p++ = c;
			*p++ = 0;
			if (c == 0) return;
			if (c == '/') name--;
		}
	}
}
/*=======================================================
 * parts_to_name -- Convert list of parts back to string.
 *=====================================================*/
static STRING parts_to_name (parts)
STRING *parts;
{
	INT i;
	static char scratch[MAXNAMELEN+1];
	STRING p = scratch;
	for (i = 0; i < MAXPARTS; i++) {
		if (!parts[i]) continue;
		strcpy(p, parts[i]);
		p += strlen(parts[i]);
		*p++ = ' ';
	}
	if (*(p - 1) == ' ')
		*(p - 1) = 0;
	else
		*p = 0;
	return scratch;
}
/*===========================================================
 * upsurname -- Convert surname of GEDCOM name to upper case.
 *=========================================================*/
STRING upsurname (name)
STRING name;
{
	static char scratch[MAXNAMELEN+1];
	STRING p = scratch;
	INT c;
	while ((c = *p++ = *name++) && c != '/')
		;
	if (c == 0) return scratch;
	while ((c = *name++) && c != '/')
		*p++ = toupper(c);
	*p++ = c;
	if (c == 0) return scratch;
	while (c = *p++ = *name++)
		;
	return scratch;
}
/*================================================================
 * manip_name - Converts GEDCOM names to various formats; second
 *   parameter specifies surname in upper case; third specifies
 *   surname first in string and separated from givens by a comma;
 *   fourth parameter specifies max length the output string.
 *==============================================================*/
STRING manip_name (name, caps, reg, len)
STRING name;	/* name */
BOOLEAN caps;	/* surname in caps? */
BOOLEAN reg;	/* regular order? (not surname first) */
INT len;	/* max name length */
{
	if (!name || *name == 0) return NULL;
	if (caps) name = upsurname(name);
	name = trim_name(name, reg ? len: len-1);
	if (reg) return trim(name_string(name), len);
	return trim(name_surfirst(name), len);
}
/*=================================================================
 * name_string -- Removes slashes from GEDCOM name; convert \ to /.
 *===============================================================*/
STRING name_string (name)
STRING name;
{
	static char scratch[MAXNAMELEN+1];
	STRING p = scratch;
	ASSERT(strlen(name) <= MAXNAMELEN);
	while (*name) {
		if (*name == '\\') *p++ = '/';
		else if (*name != '/') *p++ = *name;
		name++;
	}
	*p-- = 0;
	if (*p == ' ') *p = 0;
	return scratch;
}
/*===========================================================
 * name_surfirst - Convert GEDCOM name to surname first form.
 *=========================================================*/
STRING name_surfirst (name)
STRING name;
{
	static char scratch[MAXNAMELEN+1];
	STRING p = scratch;
	if (strlen(name) > MAXNAMELEN) FATAL();
	strcpy(p, getsurname(name));
	p += strlen(p);
	strcpy(p, ", ");
	p += strlen(p);
	strcpy(p, givens(name));
	return scratch;
}
/*==================================
 * id_by_key -- Finds name from key.
 *================================*/
STRING *id_by_key (name, pkeys)
STRING name;
STRING **pkeys;
{
	STRING rec, str, p = name;
	static char kbuf[MAXNAMELEN];
	static char nbuf[MAXNAMELEN];
	static STRING kaddr, naddr;
	INT i = 0, c, len;
	NODE indi;
	while ((c = *p++) && chartype(c) == WHITE)
		;
	if (c == 0) return NULL;
	if (c != 'I' && chartype(c) != DIGIT) return NULL;
	kbuf[i++] = 'I';
	if (c != 'I') kbuf[i++] = c;
	while ((c = *p++) && chartype(c) == DIGIT)
		kbuf[i++] = c;
	if (c != 0) return NULL;
	kbuf[i] = 0;
	kaddr = kbuf;
	*pkeys = &kaddr;
	if (!(rec = getrecord(BTR, str2rkey(kbuf), &len))) return NULL;
	if (!(indi = string_to_node(rec))) {
		stdfree(rec);
		return NULL;
	}
	if (!(str = nval(NAME(indi))) || *str == 0) {
		stdfree(rec);
		return NULL;
	}
	strcpy(nbuf, str);
	naddr = nbuf;
	return &naddr;
}
/*=============================================
 * name_to_list -- Convert name to string list.
 *===========================================*/
BOOLEAN name_to_list (name, list, plen, psind)
STRING name;	/* GEDCOM name */
LIST list;	/* list (must exist) */
INT *plen;	/* returned length */
INT *psind;	/* index (rel 1) of surname in list */
{
	INT i;
	STRING str;
	STRING parts[MAXPARTS];
	if (!name || *name == 0 || !list) return FALSE;
	make_list_empty(list);
	set_list_type(list, LISTDOFREE);
	*psind = 0;
	name_to_parts(name, parts);
	for (i = 0; i < MAXPARTS; i++) {
		if (!parts[i]) break;
		if (*parts[i] == '/') {
			*psind = i + 1;
			str = strsave(parts[i] + 1);
			str[strlen(str) - 1] = 0;
		} else
			str = strsave(parts[i]);
		set_list_element(list, i + 1, str);
	}
	*plen = i;
	ASSERT(*psind);
	return TRUE;
}
