/*===================================================================
 * index.c -- This file contains the code for handling BTREE indices.
 * Copyright(c) 1991 by Thomas Trask Wetmore IV; all rights reserved.
 * Version 2.3.4 - controlled 24 June 1993
 *===================================================================
 */
#include "standard.h"
#include "btree.h"

/*===========================================
 * crtindex - Create a new index for a btree.
 *=========================================*/
INDEX crtindex (btree)
BTREE btree;  /*btree handle*/
{
	INDEX index = (INDEX) stdalloc(BUFLEN);
	nkeys(index) = 0;
	iparent(index) = NULL;
	itype(index) = BTINDEXTYPE;
	iself(index) = btree->b_kfile.k_fkey;
	nextfkey(btree);
	rewind(bkfp(btree));
	if (fwrite(&bkfile(btree), sizeof(KEYFILE), 1, bkfp(btree)) != 1)
		FATAL();
	writeindex(bbasedir(btree), index);
	return index;
}
/*=================================================
 * readindex - Read an index from an external file.
 *===============================================*/
INDEX readindex (basedir, ikey)
STRING basedir;  /*basedir of btree*/
FKEY ikey;       /*index file key*/
{
	FILE *fp;
	INDEX index;
	char scratch[200];

/*printf("[READ: %s: ", fkey2path(ikey));/*DEBUG*/
	sprintf(scratch, "%s/%s", basedir, fkey2path(ikey));
	if ((fp = fopen(scratch, "r")) == NULL) {
		bterrno = BTERROPNIDX;
		return NULL;
	}
	index = (INDEX) stdalloc(BUFLEN);
	if (fread(index, BUFLEN, 1, fp) != 1)
		FATAL();
	fclose(fp);
/*printf("read: iself == %s]\n", fkey2path(iself(index)));/*DEBUG*/
	return index;
}
/*============================================
 * writeindex - Writes out an index to a file.
 *==========================================*/
writeindex (basedir, index)
STRING basedir; /*base directory of btree*/
INDEX index;    /*index block*/
{
	FILE *fp;
	char scratch[200];
	
/*printf("{WRITEINDEX: %s ", fkey2path(iself(index)));/*DEBUG*/
/*printf("contents\n"); prtindex(index);/*DEBUG*/
	sprintf(scratch, "%s/%s", basedir, fkey2path(iself(index)));
/*printf("writing file %s}\n", scratch);/*DEBUG*/
	if ((fp = fopen(scratch, "w")) == NULL)
		FATAL();
	if (fwrite(index, BUFLEN, 1, fp) != 1)
		FATAL();
	fclose(fp);
}
/*=====================================================
 * initcache -- Initialize the INDEX cache for a BTREE.
 *====================================================*/
initcache (btree, n)
BTREE btree; /* btree handle */
INT n;       /* number of cache blocks to allow */
{
	INT i;

	n = (n < 5) ? 5 : n;
	bncache(btree) = n;
	bcache(btree) = (INDEX *) stdalloc(n*sizeof(INDEX));
	for (i = 0;  i < n;  i++)
		bcache(btree)[i] = NULL;
}
/*====================================================
 * cacheindex -- Place an INDEX or BLOCK in the cache.
 *===================================================*/
BOOLEAN cacheindex (btree, index)
BTREE btree; /* btree handle */
INDEX index; /* INDEX or BLOCK */
{
	INT n = bncache(btree);
	INDEX *indices = bcache(btree);
	INT j = incache(btree, iself(index));

/*printf("<<CACHEINDEX: %s: ", fkey2path(iself(index)));/*DEBUG*/
	if (j == -1) {	/* index is not in the cache */
		if (indices[n-1])
			stdfree(indices[n-1]);
		for (j = n - 1; j > 0; --j)
			indices[j] = indices[j - 1];
		indices[0] = index;
	} else {	/* index is already in the cache */
		for (; j > 0; --j)
			indices[j] = indices[j - 1];
		indices[0] = index;
	}
}
/*======================================
 * getindex - Get an index from a btree.
 *====================================*/
INDEX getindex (btree, fkey)
BTREE btree;
FKEY fkey;
{
	INT j;
	INDEX index;

	if (fkey == iself(bmaster(btree)))
		return bmaster(btree);
	if ((j = incache(btree, fkey)) == -1) {	/* not in cache */
		if ((index = readindex(bbasedir(btree), fkey)) == NULL)
			return NULL;
		cacheindex(btree, index);
		return index;
	}
	return (bcache(btree))[j];
}
/*===================================================================
 * putindex -- put out an index - cache it
 *===================================================================*/
putindex (btree, index)
BTREE btree;
INDEX index;
{
	writeindex(bbasedir(btree), index);
	if (iself(index) == iself(bmaster(btree)))
		return;
	cacheindex(btree, index);
}
/*===================================================================
 * putheader -- cache a block header - but don't write it
 *===================================================================
 */
putheader (btree, block)
BTREE btree;
BLOCK block;
{
/*printf("(PUTHEADER: %s ", fkey2path(iself(block)));/*DEBUG*/
	cacheindex(btree, block); /*PRETTY SIMPLE, HUH?*/
}
/*===================================================================
 * incache -- checks whether an INDEX is in the cache - if it is then
 *   its cache index is retured - otherwise -1 is returned.
 *===================================================================*/
INT incache (btree, fkey)
BTREE btree;
FKEY fkey;
{
	INT i, n = bncache(btree);
	INDEX index, *indices = bcache(btree);

	for (i = 0; i < n; i++) {
		if ((index = *indices++) == NULL)
			return -1;
		if (iself(index) == fkey)
			return i;
	}
	return -1;
}
