/*=============================================================
 * add.c -- Add new person or family to database; add child to
 *   family; add spouse to family
 * Copyright(c) 1992-94 by T.T. Wetmore IV; all rights reserved
 *   2.3.4 - 24 Jun 93    2.3.5 - 01 Sep 93
 *   2.3.6 - 29 Oct 93    3.0.0 - 23 Sep 94
 *===========================================================*/

#include "standard.h"
#include "table.h"
#include "gedcom.h"
#include "translat.h"

extern STRING idcfam, fredit, cffadd, idprnt, unksex, idsbln, mklast;
extern STRING idsadd, idsinf, kchild, iscinf, notopp, idsps1, idsps2;
extern STRING nosex,  hashsb, haswif, idchld, gdfadd, cfcadd, iredit;
extern STRING cfpadd, cfsadd, gdpadd, gdcadd, gdsadd, ronlya, ronlye;

extern TRANTABLE tran_tables[];

/*==========================================================
 * add_indi_by_edit -- Add new person to database by editing
 *========================================================*/
NODE add_indi_by_edit ()
{
	FILE *fp;
	NODE node;
	STRING str, msg;
	BOOLEAN emp;
	TRANTABLE tti = tran_tables[MEDIN];
	if (readonly) {
		message(ronlya);
		return NULL;
	}

/* Create person template for user to edit */
	if (!(fp = fopen(editfile, "w"))) return NULL;
	if (str = (STRING) valueof(useropts, "INDIREC"))
		fprintf(fp, "%s\n", str);
	else {
		fprintf(fp, "0 INDI\n1 NAME Fname /Surname/\n1 SEX MF\n");
		fprintf(fp, "1 BIRT\n  2 DATE\n  2 PLAC\n  2 SOUR\n");
		fprintf(fp, "1 DEAT\n  2 DATE\n  2 PLAC\n  2 SOUR\n");
	}

/* Have user edit new person record */
	fclose(fp);
	do_edit();
	while (TRUE) {
		node = file_to_node(editfile, tti, &msg, &emp);
		if (!node) {
			if (ask_yes_or_no_msg(msg, iredit)) {
				do_edit();
				continue;
			} 
			break;
		}
		if (!valid_indi_tree(node, &msg, SEX_UNKNOWN, NULL,
		    NULL, NULL)) {
			if (ask_yes_or_no_msg(msg, iredit)) {
				do_edit();
				continue;
			}
			free_nodes(node);
			node = NULL;
			break;
		}
		break;
	}
	if (!node || !ask_yes_or_no(cfpadd)) {
		if (node) free_nodes(node);
		return NULL;
	}
	return add_unlinked_indi(node);
}
/*==========================================================
 * add_unlinked_indi -- Add person with no links to database
 *========================================================*/
NODE add_unlinked_indi (indi)
NODE indi;
{
	NODE name, sex, body, dumb, this;
	STRING key;
	split_indi(indi, &name, &sex, &body, &dumb, &dumb);
	ASSERT(name);
	nxref(indi) = strsave(getixref());
	key = rmvat(nxref(indi));
	for (this = name; this; this = nsibling(this))
		add_name(nval(this), key);
	join_indi(indi, name, sex, body, NULL, NULL);
	indi_to_dbase(indi);
	indi_to_cache(indi);
	mprintf(gdpadd, indi_to_name(indi, 35));
	return indi;
}
/*================================================================
 * add_linked_indi -- Add linked person to database; links assumed
 *   correct
 *==============================================================*/
BOOLEAN add_linked_indi (indi)
NODE indi;
{
	NODE node, name, sex, body, famc, fams;
	STRING str, key;
	split_indi(indi, &name, &sex, &body, &famc, &fams);
	ASSERT(name);
	key = rmvat(nxref(indi));
	for (node = name; node; node = nsibling(node))
		add_name(nval(node), key);
	join_indi(indi, name, sex, body, famc, fams);
	ASSERT(str = node_to_string(indi));
	ASSERT(store_record(key, str, strlen(str)));
	stdfree(str);
	return TRUE;
}
/*==================================
 * add_child --  Add child to family
 *================================*/
NODE add_child (child, fam)
NODE child, fam;
{
	NODE node, last, new, name, sex, body, famc, fams;
	NODE husb, wife, chil, rest;
	INT i, j, nchildren;
	STRING *childstrings, *childkeys;

	if (readonly) {
		message(ronlye);
		return NULL;
	}

/* Identify child if caller did not */
	if (!child) child = ask_for_indi(idchld, FALSE, TRUE);
	if (!child) return NULL;
	if (famc = FAMC(child)) {
		message(iscinf);
		return NULL;
	}

/* Identify family if caller did not */
	if (!fam) fam = ask_for_fam(idprnt, idsbln);
	if (!fam) return NULL;

/* If first child in family, confirm and add */
	childstrings = get_child_strings(fam, &nchildren, &childkeys);
	if (nchildren == 0) {
		if (!ask_yes_or_no(cfcadd)) return NULL;
		split_fam(fam, &husb, &wife, &chil, &rest);
		chil = create_node(NULL, "CHIL", nxref(child), fam);
		join_fam(fam, husb, wife, chil, rest);

/* If not first, find where child belongs */
	} else {
		childstrings[nchildren] = mklast;
		i = choose_from_list(idcfam, nchildren+1, childstrings);
		if (i == -1) return NULL;
		split_fam(fam, &husb, &wife, &chil, &rest);
		last = NULL;
		node = chil;
		j = 0;
		while (j++ < i) {
			last = node;
			node = nsibling(node);
		}
		new = create_node(NULL, "CHIL", nxref(child), fam);
		nsibling(new) = node;
		if (last)
			nsibling(last) = new;
		else
			chil = new;
		join_fam(fam, husb, wife, chil, rest);
	}

/* Add FAMC node to child */
	split_indi(child, &name, &sex, &body, &famc, &fams);
	famc = create_node(NULL, "FAMC", nxref(fam), child);
	join_indi(child, name, sex, body, famc, fams);

/* Write updated records to database */
	fam_to_dbase(fam);
	indi_to_dbase(child);
	mprintf(gdcadd, indi_to_name(child, 35));
	return fam;
}
/*===================================
 * add_spouse -- Add spouse to family
 *=================================*/
BOOLEAN add_spouse (spouse, fam, conf)
NODE spouse, fam;
BOOLEAN conf;
{
	INT sex;
	NODE husb, wife, chil, rest, fams, last, node;

	if (readonly) {
		message(ronlye);
		return NULL;
	}

/* Identify spouse to add to family */
	if (!spouse) spouse = ask_for_indi(idsadd, FALSE, TRUE);
	if (!spouse) return FALSE;
	if ((sex = SEX(spouse)) == SEX_UNKNOWN) {
		message(nosex);
		return FALSE;
	}

/* Identify family to add spouse to */
	if (!fam) fam = ask_for_fam(idsinf, kchild);
	if (!fam) return FALSE;

/* Check that spouse can be added */
	split_fam(fam, &husb, &wife, &chil, &rest);
	join_fam(fam, husb, wife, chil, rest);
	if (sex == SEX_MALE && husb) {
		message(hashsb);
		return FALSE;
	}
	if (sex == SEX_FEMALE && wife) {
		message(haswif);
		return FALSE;
	}
	if (conf && !ask_yes_or_no(cfsadd)) return FALSE;

/* Add HUSB or WIFE node to family */
	split_fam(fam, &husb, &wife, &chil, &rest);
	if (sex == SEX_MALE)
		husb = create_node(NULL, "HUSB", nxref(spouse), fam);
	else
		wife = create_node(NULL, "WIFE", nxref(spouse), fam);
	join_fam(fam, husb, wife, chil, rest);

/* Add FAMS node to spouse */
	fams = create_node(NULL, "FAMS", nxref(fam), spouse);
	last = NULL;
	node = nchild(spouse);
	while (node) {
		last = node;
		node = nsibling(node);
	}
	ASSERT(last);
	nsibling(last) = fams;

/* Write updated records to database */
	fam_to_dbase(fam);
	indi_to_dbase(spouse);
	mprintf(gdsadd, indi_to_name(spouse, 35));
	return TRUE;
}
/*=========================================
 * add_family -- Add new family to database
 *=======================================*/
NODE add_family (spouse1, spouse2, child)
NODE spouse1, spouse2, child;
{
	INT sex1, sex2;
	NODE fam1, fam2, husb1, wife1, chil1, node, last, new;
	TRANTABLE tti = tran_tables[MEDIN], tto = tran_tables[MINED];
	STRING xref, msg;
	BOOLEAN emp;
	FILE *fp;

	if (readonly) {
		message(ronlya);
		return NULL;
	}

/* Handle case where child is known */
	if (child)  {
		if (FAMC(child)) {
			message(iscinf);
			return NULL;
		}
		spouse1 = spouse2 = NULL;
		goto editfam;
	}

/* Identify first spouse */
	if (!spouse1) spouse1 = ask_for_indi(idsps1, FALSE, FALSE);
	if (!spouse1) return NULL;
	if ((sex1 = SEX(spouse1)) == SEX_UNKNOWN) {
		message(unksex);
		return NULL;
	}

/* Identify optional spouse */
	if (!spouse2) spouse2 = ask_for_indi(idsps2, FALSE, TRUE);
	if (spouse2) {
		if ((sex2 = SEX(spouse2)) == SEX_UNKNOWN || sex1 == sex2) {
			message(notopp);
			return NULL;
		}
	}

/* Create new family */
editfam:
	fam1 = create_node(NULL, "FAM", NULL, NULL);
	husb1 = wife1 = chil1 = NULL;
	if (spouse1) {
		if (sex1 == SEX_MALE)
			husb1 = create_node(NULL, "HUSB", nxref(spouse1), fam1);
		else
			wife1 = create_node(NULL, "WIFE", nxref(spouse1), fam1);
	}
	if (spouse2) {
		if (sex2 == SEX_MALE)
			husb1 = create_node(NULL, "HUSB", nxref(spouse2), fam1);
		else
			wife1 = create_node(NULL, "WIFE", nxref(spouse2), fam1);
	}
	if (child)
		chil1 = create_node(NULL, "CHIL", nxref(child), fam1);

/* Have user edit marriage info */
	ASSERT(fp = fopen(editfile, "w"));
	write_nodes(0, fp, tto, fam1, TRUE, TRUE, TRUE);
	write_nodes(1, fp, tto, husb1, TRUE, TRUE, TRUE);
	write_nodes(1, fp, tto, wife1, TRUE, TRUE, TRUE);
	fprintf(fp, "1 MARR\n  2 DATE\n  2 PLAC\n  2 SOUR\n");
	write_nodes(1, fp, tto, chil1, TRUE, TRUE, TRUE);
	fclose(fp);
	do_edit();
	while (TRUE) {
		fam2 = file_to_node(editfile, tti, &msg, &emp);
		if (!fam2) {
			if (ask_yes_or_no_msg(msg, fredit)) {
				do_edit();
				continue;
			}
			break;
		}
		if (!valid_fam_tree(fam2, &msg, fam1, husb1, wife1, chil1)) {
			if (ask_yes_or_no_msg(msg, fredit)) {
				do_edit();
				continue;
			}
			free_nodes(fam2);
			fam2 = NULL;
			break;
		}
		break;
	}

/* Confirm family add operation */
	join_fam(fam1, husb1, wife1, chil1, NULL);
	free_nodes(fam1);
	if (!fam2 || !ask_yes_or_no(cffadd)) {
		free_nodes(fam2);
		return NULL;
	}
	nxref(fam2) = strsave(xref = getfxref());

/* Modify spouse/s and/or child */
	if (spouse1) {
		new = create_node(NULL, "FAMS", xref, spouse1);
		last = NULL;
		node = nchild(spouse1);
		while (node) {
			last = node;
			node = nsibling(node);
		}
		ASSERT(last);
		nsibling(last) = new;
	}
	if (spouse2) {
		new = create_node(NULL, "FAMS", xref, spouse2);
		last = NULL;
		node = nchild(spouse2);
		while (node) {
			last = node;
			node = nsibling(node);
		}
		ASSERT(last);
		nsibling(last) = new;
	}
	if (child) {
		NODE name, sex, body, famc, fams;
		split_indi(child, &name, &sex, &body, &famc, &fams);
		ASSERT(!famc);
		new = create_node(NULL, "FAMC", xref, child);
		join_indi(child, name, sex, body, new, fams);
	}

/* Write updated records to database */
	fam_to_dbase(fam2);
	fam_to_cache(fam2);
	if (spouse1) indi_to_dbase(spouse1);
	if (spouse2) indi_to_dbase(spouse2);
	if (child) indi_to_dbase(child);
	message(gdfadd);
	return fam2;
}
