#include <stdio.h>
#include "machdep.h"
#include "timer.h"
#include "mbuf.h"
#include "netuser.h"
#include "iface.h"
#include "internet.h"
#include "ip.h"
#include "tcp.h"
#include "udp.h"
#include "icmp.h"
#include "ec.h"
#include "arp.h"
#include "ax25.h"

/* Comment this out if your library already has this function */
#define	isprint(c)	((c) >= ' ' && (c) < 0x7f)		/* Assumes ASCII */

/* TCP connection states */
char *tcpstates[] = {
	"Closed",
	"Listen",
	"SYN sent",
	"SYN received",
	"Established",
	"FIN wait 1",
	"FIN wait 2",
	"Close wait",
	"Closing",
	"Last ACK",
	"Time wait"
};

/* TCP closing reasons */
char *reasons[] = {
	"Normal",
	"Reset",
	"Timeout",
	"ICMP"
};
/* ICMP message types */
char *icmptypes[] = {
	"Echo Reply",
	NULLCHAR,
	NULLCHAR,
	"Unreachable",
	"Source Quench",
	"Redirect",
	NULLCHAR,
	NULLCHAR,
	"Echo Request",
	NULLCHAR,
	NULLCHAR,
	"Time Exceeded",
	"Parameter Problem",
	"Timestamp",
	"Timestamp Reply",
	"Information Request",
	"Information Reply"
};

/* ICMP unreachable messages */
char *unreach[] = {
	"Network",
	"Host",
	"Protocol",
	"Port",
	"Fragmentation",
	"Source route"
};
/* ICMP Time exceeded messages */
char *exceed[] = {
	"Time-to-live",
	"Fragment reassembly"
};

/* ICMP redirect messages */
char *redirect[] = {
	"Network",
	"Host",
	"TOS & Network",
	"TOS & Host"
};
/* TCP segment header flags */
char *tcpflags[] = {
	"FIN",	/* 0x01 */
	"SYN",	/* 0x02 */
	"RST",	/* 0x04 */
	"PSH",	/* 0x08 */
	"ACK",	/* 0x10 */
	"URG"	/* 0x20 */
};
char *arptypes[] = {
	NULLCHAR,
	"Ethernet",
	"Exp Ethernet",
	"AX.25",
	"Pronet",
	"Chaos"
};

/* Convert an internet address (in host byte order) to a dotted decimal ascii
 * string, e.g., 255.255.255.255\0
 */
char *
inet_ntoa(a)
int32 a;
{
	static char buf[16];

	sprintf(buf,"%u.%u.%u.%u",
		hibyte(hiword(a)),
		lobyte(hiword(a)),
		hibyte(loword(a)),
		lobyte(loword(a)) );
	return buf;
}
/* Convert a socket (address + port) to an ascii string of the form
 * aaa.aaa.aaa.aaa:ppppp
 */
char *
psocket(s)
struct socket *s;
{
	static char buf[30];

	sprintf(buf,"%s:%u",inet_ntoa(s->address),s->port);
	return buf;
}

/* Dump an mbuf in hex */
void
hexdump(bp)
struct mbuf *bp;
{
	register struct mbuf *tbp;
	int16 n;
	int16 address;
	void fmtline();

	if(bp == NULLBUF)
		return;
	tbp = copy_p(bp,len_mbuf(bp));
	address = 0;
	while(tbp->cnt != 0){
		n = min(tbp->cnt,16);
		fmtline(address,tbp->data,n);
		address += n;
		tbp->data += n;
		tbp->cnt -= n;
	}
	free_p(tbp);
}
/* Print a buffer up to 16 bytes long in formatted hex with ascii
 * translation, e.g.,
 * 0000: 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f  0123456789:;<=>?
 */
static void
fmtline(addr,buf,len)
int16 addr;
char *buf;
int16 len;
{
	char line[80];
	char *aptr,*cptr;
	int16 c;
	void ctohex();

	for(cptr = line;cptr < &line[80];cptr++)
		*cptr = ' ';
	ctohex(line,(int16)hibyte(addr));
	ctohex(line+2,(int16)lobyte(addr));
	aptr = &line[6];
	cptr = &line[55];
	while(len-- != 0){
		c = *buf++ & 0xff;
		ctohex(aptr,c);
		aptr += 3;
		if(isprint(c)){
			*cptr++ = c;
		} else {
			*cptr++ = '.';
		}
	}
	*cptr++ = '\r';
	*cptr++ = '\n';
	fwrite(line,1,(unsigned)(cptr-line),stdout);
	fflush(stdout);
}
/* Dump a TCP control block */
void
state_tcp(tcb)
struct tcb *tcb;
{
	if(tcb == NULLTCB)
		return;
	printf("TCP Control Block at 0x%x:\r\n",(int)tcb);
	printf("Local: %s",psocket(&tcb->conn.local));
	printf(" Remote: %s",psocket(&tcb->conn.remote));
	printf(" State: %s\r\n",tcpstates[tcb->state]);
	printf("        Init seq      Unack       Next  Wind   Urg        WL1        WL2 Queue\r\n");
	printf("Send:");
	printf("%11lu",tcb->iss);
	printf("%11lu",tcb->snd.una);
	printf("%11lu",tcb->snd.nxt);
	printf("%6u",tcb->snd.wnd);
	printf("%6u",tcb->snd.up);
	printf("%11lu",tcb->snd.wl1);
	printf("%11lu",tcb->snd.wl2);
	printf("%6u\r\n",tcb->sndcnt);

	printf("Recv:");
	printf("%11lu",tcb->irs);
	printf("           ");
	printf("%11lu",tcb->rcv.nxt);
	printf("%6u",tcb->rcv.wnd);
	printf("%6u",tcb->rcv.up);
	printf("           ");
	printf("           ");
	printf("%6u\r\n",tcb->rcvcnt);

	if(tcb->reseq != (struct reseq *)NULL){
		register struct reseq *rp;

		printf("Reassembly queue:\r\n");
		for(rp = tcb->reseq;rp != (struct reseq *)NULL; rp = rp->next){
			printf("  seq %lu %u bytes\r\n",rp->seg.seq,rp->length);
		}
	}
	printf("Retry: %u Backoff: %u",tcb->retry,tcb->backoff);
	switch(tcb->timer.state){
	case TIMER_STOP:
		printf(" Timer stopped");
		break;
	case TIMER_RUN:
		printf(" Timer running (%ld mS)",
		 (long)MSPTICK * (tcb->timer.start - tcb->timer.count));
		break;
	case TIMER_EXPIRE:
		printf(" Timer expired");
	}
	printf(" Smoothed round trip time: %ld mS\r\n",tcb->srtt);
	fflush(stdout);
}

/* Dump a TCP segment header. Assumed to be in network byte order */
void
dump_tcp(bp,source,dest)
struct mbuf *bp;
int32 source,dest;
{
	int hdr_len,i;
	register struct tcp_header *tcph;
	struct pseudo_header ph;

	if(bp == NULLBUF)
		return;
	tcph = (struct tcp_header *)bp->data;
	hdr_len = hinibble(tcph->offset) * sizeof(int32);
	printf("TCP: %u->%u Seq %lu",
		ntohs(tcph->source),ntohs(tcph->dest),
		ntohl(tcph->seq),ntohl(tcph->ack));
	
	if(tcph->flags & ACK)
		printf(" Ack %lu",ntohl(tcph->ack));
	for(i=0;i<6;i++){
		if(tcph->flags & 1 << i){
			printf(" %s",tcpflags[i]);
		}
	}
	printf(" Wnd %u",ntohs(tcph->wnd));
	if(tcph->flags & URG)
		printf(" UP %u",ntohs(tcph->up));

	if(hdr_len > sizeof(struct tcp_header)){
		struct mss *mssp;
	
		mssp = (struct mss *)(tcph + 1);
		if(mssp->kind == MSS_KIND && mssp->length == MSS_LENGTH){
			printf(" MSS %u",ntohs(mssp->mss));
		}
	}
	/* Verify checksum */
	ph.source = source;
	ph.dest = dest;
	ph.protocol = TCP_PTCL;
	ph.length = len_mbuf(bp);
	if((i = cksum(&ph,bp,ph.length)) != 0)
		printf(" CHECKSUM ERROR (%u)",i);
	printf("\r\n");
}

/* Dump a UDP header */
void
dump_udp(bp,source,dest)
struct mbuf *bp;
int32 source,dest;
{
	register struct udp_header *udph;
	struct pseudo_header ph;
	int i;

	if(bp == NULLBUF)
		return;
	udph = (struct udp_header *)bp->data;
	printf("UDP:");
	printf(" %u->%u",ntohs(udph->source),
		ntohs(udph->dest));
	printf(" len %u",ntohs(udph->length));
	
	/* Verify checksum */
	ph.source = source;
	ph.dest = dest;
	ph.zero = 0;
	ph.protocol = UDP_PTCL;
	ph.length = len_mbuf(bp);
	if((i = cksum(&ph,bp,ph.length)) != 0)
		printf(" CHECKSUM ERROR (%u)",i);
	printf("\r\n");
}

/* Dump an ICMP header */
void
dump_icmp(bp,source,dest)
struct mbuf *bp;
int32 source,dest;
{
	register struct icmp *icmp;
	char *codemsg;
	struct mbuf *ibp;
	int i;

	if(bp == NULLBUF)
		return;
	codemsg = NULLCHAR;
	icmp = (struct icmp *)bp->data;
	if(icmp->type <= 16 && icmptypes[icmp->type] != NULLCHAR)
		printf("ICMP: %s",icmptypes[icmp->type]);
	else
		printf("ICMP: type %u",icmp->type);

	switch(icmp->type){
	case DEST_UNREACH:
		if(icmp->code <= 5)
			codemsg = unreach[icmp->code];
		break;
	case REDIRECT:
		if(icmp->code <= 3)
			codemsg = redirect[icmp->code];
		break;
	case TIME_EXCEED:
		if(icmp->code <= 1)
			codemsg = exceed[icmp->code];
		break;
	}	
	if(codemsg != NULLCHAR)
		printf(" %s",codemsg);
	else
		printf(" code %u",icmp->code);

	/* Special case for parameter problem message */
	if(icmp->type == PARAM_PROB)
		printf(" pointer = 0x%x",icmp->args.pointer);

	/* Verify checksum */
	if((i = cksum((struct pseudo_header *)NULL,bp,len_mbuf(bp))) != 0)
		printf(" CHECKSUM ERROR (%u)",i);
	printf("\r\n");

	/* Dump the offending IP header, if any */
	switch(icmp->type){
	case DEST_UNREACH:
	case TIME_EXCEED:
	case PARAM_PROB:
	case QUENCH:
	case REDIRECT:
		printf("Returned ");
		ibp = copy_p(bp,len_mbuf(bp));
		pullup(&ibp,NULLCHAR,sizeof(struct icmp));
		dump_ip(ibp,1);
		free_p(ibp);
	}
}

/* Dump an IP datagram header. If it's the first fragment, also dump
 * the next layer header (if known)
 * Level 0: no dump
 * Level 1: IP header only
 * Level 2: All headers
 * Level 3: All headers + hex dump
 */
void
dump_ip(bp,level)
struct mbuf *bp;
int level;
{
	register struct ip_header *ip;
	int32 source,dest;
	int16 ip_len;
	int16 length;
	struct mbuf *tbp;
	int16 offset;
	int i;

	if(level < 1 || bp == NULLBUF)
		return;	
	ip = (struct ip_header *)bp->data;
	ip_len = lonibble(ip->v_ihl) * sizeof(int32);
	length = ntohs(ip->length);
	offset = (ntohs(ip->fl_offs) & F_OFFSET) << 3 ;
	source = ntohl(ip->source);
	dest = ntohl(ip->dest);
	printf("IP: %s",inet_ntoa(source));
	printf("->%s len %u ihl %u ttl %u prot %u",
		inet_ntoa(dest),length,ip_len,ip->ttl & 0xff,
		ip->protocol & 0xff);

	if(ip->tos != 0)
		printf(" tos %u",ip->tos);
	if(offset != 0 || (ntohs(ip->fl_offs) & MF))
		printf(" id %u offs %u",ntohs(ip->id),offset);

	if(ntohs(ip->fl_offs) & DF)
		printf(" DF");
	if(ntohs(ip->fl_offs) & MF)
		printf(" MF");
	if((i = cksum((struct pseudo_header *)NULL,bp,ip_len)) != 0)
		printf(" CHECKSUM ERROR (%u)",i);
	printf("\r\n");

	if(level > 1){
		if(offset == 0){
			dup_p(&tbp,bp,ip_len,length - ip_len);
			switch(ip->protocol & 0xff){
			case TCP_PTCL:
				dump_tcp(tbp,source,dest);
				break;
			case UDP_PTCL:
				dump_udp(tbp,source,dest);
				break;
			case ICMP_PTCL:
				dump_icmp(tbp,source,dest);
				break;
			}
			free_p(tbp);
		}
		if(level > 2){
			hexdump(bp);
			printf("\r\n");
		}
	}
	fflush(stdout);
}

/* Convert byte to two ascii-hex characters */
static
void
ctohex(buf,c)
char *buf;
int16 c;
{
	static char hex[] = "0123456789abcdef";

	buf[0] = hex[hinibble(c)];
	buf[1] = hex[lonibble(c)];
}
/* Return 1 if arg is a valid TCB, 0 otherwise */
int
tcpval(tcb)
struct tcb *tcb;
{
	register int i;
	register struct tcb *tcb1;

	for(i=0;i<NTCB;i++){
		for(tcb1=tcbs[i];tcb1 != NULLTCB;tcb1 = tcb1->next){
			if(tcb1 == tcb)
				return 1;
		}
	}
	return 0;
}
/* Dump UDP statistics and control blocks */
doudpstat()
{
	extern struct udp_stat udp_stat;
	char *psocket();
	register struct udp_cb *udp;
	register int i;

	printf("sent %u rcvd %u cksum err %u unknown socket %u\r\n",
	udp_stat.sent,udp_stat.rcvd,udp_stat.cksum,udp_stat.unknown);
	printf("&UCB Rcv-Q  Local socket\r\n");
	for(i=0;i<NUDP;i++){
		for(udp = udps[i];udp != NULLUDP; udp = udp->next){
			printf("%x%6u  %s\r\n",(int)udp,udp->rcvcnt,
			 psocket(&udp->socket));
		}
	}
}
/* Dump TCP stats and summary of all TCBs
/* &TCB Rcv-Q Snd-Q  Local socket           Remote socket          State
 * 1234     0     0  xxx.xxx.xxx.xxx:xxxxx  xxx.xxx.xxx.xxx:xxxxx  Established
 */
int
tcpstat()
{
	register int i;
	register struct tcb *tcb;
	char *psocket();

	printf("conout %u conin %u reset out %u runt %u chksum err %u\r\n",
		tcp_stat.conout,tcp_stat.conin,tcp_stat.resets,tcp_stat.runt,
		tcp_stat.checksum);
	printf("&TCB Rcv-Q Snd-Q  Local socket           Remote socket          State\r\n");
	for(i=0;i<NTCB;i++){
		for(tcb=tcbs[i];tcb != NULLTCB;tcb = tcb->next){
			printf("%4x%6u%6u  ",(int)tcb,tcb->rcvcnt,tcb->sndcnt);
			printf("%-23s",psocket(&tcb->conn.local));
			printf("%-23s",psocket(&tcb->conn.remote));
			printf("%-s\r\n",tcpstates[tcb->state]);
		}
	}
	fflush(stdout);
	return 0;
}
#ifdef	ETHER
dump_ether(bp)
struct mbuf *bp;
{
	register struct ether *ep;
	char s[20],d[20];

	ep = (struct ether *)bp->data;
	pether(s,ep->source);
	pether(d,ep->dest);
	printf("Ether: %s->%s type 0x%x len %u\r\n",s,d,ntohs(ep->type),
		len_mbuf(bp));
	fflush(stdout);
}
static
pether(out,addr)
char *out,*addr;
{
	sprintf(out,"%02x:%02x:%02x:%02x:%02x:%02x",
		addr[0] & 0xff,
		addr[1] & 0xff,
		addr[2] & 0xff,
		addr[3] & 0xff,
		addr[4] & 0xff,
		addr[5] & 0xff,
		addr[6] & 0xff);
}
int
doetherstat(argc,argv)
int argc;
char *argv[];
{
	struct ec *ecp;
	char *cp;

	for(ecp = ec;ecp < &ec[nec]; ecp++){
		printf("Controller %u address: ",ecp-ec);
		for(cp = ecp->addr;cp < &ecp->addr[6];cp++){
			if(cp != ecp->addr)
				printf(":");
			printf("%02x",*cp & 0xff);
		}
		printf("\r\n");

		printf("intrpt    spur      shorts    dribble   fcs       overf     any\r\n");
		printf("%-10ld%-10u%-10u%-10u%-10u%-10u%-10ld\r\n",
		 ecp->estats.ints,ecp->estats.spur,ecp->estats.shorts,
		 ecp->estats.dribble,ecp->estats.fcs,ecp->estats.over,ecp->estats.any);

		printf("stale     badsize   output    timeout   badtype   drop      nomem\r\n");
		printf("%-10u%-10u%-10ld%-10u%-10u%-10u%-10u\r\n",
		 ecp->estats.stale,ecp->estats.badsize,ecp->estats.out,
		 ecp->estats.timeout,ecp->estats.badtype,ecp->estats.drop,
		 ecp->estats.nomem);
	}
	return 0;
}
#endif
int
doipstat(argc,argv)
int argc;
char *argv[];
{
	extern struct ip_stats ip_stats;
	extern struct reasm *reasmq;
	register struct reasm *rp;
	register struct frag *fp;
	char *inet_ntoa();

	printf("total %ld runt %u len err %u vers err %u",
		ip_stats.total,ip_stats.runt,ip_stats.length,ip_stats.version);
	printf(" chksum err %u nofwd %u badproto %u\r\n",
		ip_stats.checksum,ip_stats.noforward,ip_stats.badproto);

	if(reasmq != NULLREASM)
		printf("Reassembly fragments:\r\n");
	for(rp = reasmq;rp != NULLREASM;rp = rp->next){
		printf("src %s",inet_ntoa(rp->source));
		printf(" dest %s",inet_ntoa(rp->dest));
		printf(" id %u pctl %u time %u len %u\r\n",
			rp->id,rp->protocol,rp->timer.count,rp->length);
		for(fp = rp->fraglist;fp != NULLFRAG;fp = fp->next){
			printf(" offset %u last %u\r\n",fp->offset,fp->last);
		}
	}
	return 0;
}
int
doicmpstat(argc,argv)
int argc;
char *argv[];
{
	extern struct icmp_errors icmp_errors;
	extern struct icmp_stats icmp_stats;
	register int i;

	printf("chksum err %u no space %u icmp %u\r\n",
	 icmp_errors.checksum,icmp_errors.nospace,icmp_errors.noloop);
	printf("type  rcvd  sent\r\n");
	for(i=0;i<ICMP_TYPES;i++){
		if(icmp_stats.input[i] == 0 && icmp_stats.output[i] == 0)
			continue;
		printf("%-6u%-6u%-6u",i,icmp_stats.input[i],
			icmp_stats.output[i]);
		if(icmptypes[i] != NULL)
			printf("  %s",icmptypes[i]);
		printf("\r\n");
	}
	return 0;
}
int
doarp(argc,argv)
int argc;
char *argv[];
{
	register int i;
	extern struct arp_stat arp_stat;
	register struct arp_tab *ap;
	char e[20];

	printf("received %u badtype %u reqst in %u replies %u reqst out %u\r\n",
	 arp_stat.recv,arp_stat.badtype,arp_stat.inreq,
	 arp_stat.replies,arp_stat.outreq);

	printf("IP addr         Type            Addr                Time Queue\r\n");
	for(i=0;i<ARPSIZE;i++){
		for(ap = arp_tab[i];ap != NULL;ap = ap->next){
			printf("%-16s",inet_ntoa(ap->ip_addr));
			printf("%-16s",arptypes[ap->hardware]);
			if(ap->state == ARP_VALID){
				switch(ap->hardware){
#ifdef	ETHER
				case ARP_ETHER:
					pether(e,ap->hw_addr);
					break;
#endif
#ifdef	AX25
				case ARP_AX25:
					pax25(e,ap->hw_addr);
					break;
#endif
				default:
					e[0] = '\0';
					break;
				}
				printf("%-20s",e);
			} else {
				printf("[unknown]           ");
			}
			printf("%-5ld",ap->timer.count*(long)MSPTICK/1000);
			if(ap->state == ARP_PENDING)
				printf("%-5u",len_q(ap->pending));
			printf("\r\n");
		}
	}
	return 0;
}
#ifdef	AX25
/* Dump an AX.25 packet header */
dump_ax25(bp)
struct mbuf *bp;
{
	struct mbuf *copy_p();
	struct ax25 *ax25;
	char tmp[20],*cp;
	char copyflag,control,cmdrsp;
	int16 len,type,ftype();

	len = len_mbuf(bp);
	if(bp->next != NULLBUF){
		/* Copy to contiguous buffer */
		bp = copy_p(bp,len);
		copyflag = 1;
	} else
		copyflag = 0;

	ax25 = (struct ax25 *)bp->data;
	pax25(tmp,ax25->src);
	printf("AX25: %s",tmp);
	pax25(tmp,ax25->dest);
	printf(" -> %s ",tmp);

	/* Find last byte of address field */
	cp = bp->data;
	while(len--){
		if(*cp++ & 1)
			break;
	}
	if(len == 0)
		return;

	if((ax25->dest[6] & C) != (ax25->src[6] & C)){
		if(ax25->dest[6] & C)
			cmdrsp = COMMAND;
		else
			cmdrsp = RESPONSE;
	} else 
		cmdrsp = UNKNOWN;

	control = *cp++;
	len--;

	type = ftype(control);
	switch(type){
	case I:
		printf("I");
		break;
	case SABM:
		printf("SABM");
		break;
	case DISC:
		printf("DISC");
		break;
	case DM:
		printf("DM");
		break;
	case UA:
		printf("UA");
		break;
	case RR:
		printf("RR");
		break;
	case RNR:
		printf("RNR");
		break;
	case REJ:
		printf("REJ");
		break;
	case FRMR:
		printf("FRMR");
		break;
	case UI:
		printf("UI");
		break;
	default:
		printf("[invalid]");
	}
	/* Dump poll/final bit */
	if(control & PF){
		switch(cmdrsp){
		case COMMAND:
			printf("(P)");
			break;
		case RESPONSE:
			printf("(F)");
			break;
		default:
			printf("(P/F)");
			break;
		}
	}
	/* Dump sequence numbers */
	if((type & 0x3) != U)	/* I or S frame? */
		printf(" NR=%d",(control>>5)&7);
	if((type & 0x1) == I)	/* I frame? */
		printf(" NS=%d",(control>>1)&7);

	if(len != 0)
		printf(" pid 0x%x",*cp++ & 0xff);

	printf("\r\n");
	hexdump(bp);
	if(copyflag)
		free_p(bp);
	fflush(stdout);
}
/* Figure out the frame type from the control field
 * This is done by masking out any sequence numbers and the
 * poll/final bit after determining the general class (I/S/U) of the frame
 */
static
int16
ftype(control)
register char control;
{
	if((control & 1) == 0)	/* An I-frame is an I-frame... */
		return I;
	if(control & 2)			/* U-frames use all except P/F bit for type */
		return(control & ~PF);
	else					/* S-frames use low order 4 bits for type */
		return(control & 0xf);
}
#endif
