/* Vestigial AX.25 link layer, understands only UI frames */

#include "machdep.h"
#include "mbuf.h"
#include "iface.h"
#include "timer.h"
#include "arp.h"
#include "slip.h"
#include "ax25.h"
#include <ctype.h>

#ifdef	TRACE
extern int16 trace;
#include <stdio.h>
#endif

/* AX.25 broadcast address: "QST-0" in shifted ascii */
struct addr ax25_bdcst = {
	'Q'<<1, 'S'<<1, 'T'<<1, ' '<<1, ' '<<1, ' '<<1,
	('0'<<1) | E,
};
struct addr mycall;

/* Send IP datagrams in AX.25 UI frames using ARP */
int
ax_send(bp,interface,gateway,precedence,delay,throughput,reliability)
struct mbuf *bp;
struct interface *interface;
int32 gateway;
char precedence;
char delay;
char throughput;
char reliability;
{
	char *hw_addr,*res_arp();

	hw_addr = res_arp(interface,ARP_AX25,gateway,bp);
	if(hw_addr != NULLCHAR)
		(*interface->output)(interface,(struct addr *)hw_addr,&mycall,PID_IP,bp);
}
/* Send a raw packet to the KISS TNC using AX.25 link header.
 * Note that the calling order here must match ec_output
 * since ARP also uses it.
 */
kiss_output(interface,dest,src,pid,bp)
struct interface *interface;
struct addr *dest;	/* Destination AX.25 address (7 bytes, shifted) */
struct addr *src;	/* Source AX.25 address (7 bytes, shifted) */
char pid;		/* Protocol ID */
struct mbuf *bp;	/* Data field (follows PID) */
{
	register struct mbuf *hbp;
	struct mbuf *ax_encode();
	struct slip *sp;

	if((bp = ax_encode(dest,src,pid,bp)) == NULLBUF)
		return;
#ifdef	TRACE
	if(trace){
		printf("%s sent:\r\n",interface->name);
		if((trace & 0xf) > 1)
			ax25_dump(bp);
		if(trace & 0x10)
			hexdump(bp);
		fflush(stdout);
	}
#endif
	/* Put type field for KISS TNC on front */
	if((hbp = alloc_mbuf(1)) == NULLBUF){
		free_p(bp);
		return;
	}
	*hbp->data = 0;
	hbp->cnt = 1;
	hbp->next = bp;
	slipq(interface->dev,hbp);
}

/* Put a AX.25 header on the front of a packet */
struct mbuf *
ax_encode(dest,src,pid,bp)
struct addr *dest;	/* Destination AX.25 address (7 bytes, shifted) */
struct addr *src;	/* Source AX.25 address (7 bytes, shifted) */
char pid;		/* Protocol ID */
struct mbuf *bp;	/* Data field (follows PID) */
{
	register struct addr *ap;
	struct mbuf *abp;
	char *cp;
	int ndigi;
	int16 hdr_len;

	/* Determine length of dest addr */
	for(ndigi = 0,ap = dest; (ap->ssid & E) == 0; ap++)
		ndigi++;

	/* Computer header length:
	 * 2 AX.25 address fields for source and dest +
	 * "ndigi" AX.25 address field(s) for digipeaters +
	 * 2 bytes for control and PID fields
	 */

	hdr_len = (2 + ndigi)*sizeof(struct addr) + 2;

	/* Create AX.25 link level header */
	if((abp = alloc_mbuf(hdr_len)) == NULLBUF){
		free_p(bp);
		return NULLBUF;
	}
	abp->cnt = hdr_len;

	/* Now fill it in */

	ap = (struct addr *)(abp->data);
	bcopy((char *)dest++,(char *)ap++,sizeof(struct addr));
	(ap - 1)->ssid &= ~E;
	bcopy((char *)src,(char *)ap++,sizeof(struct addr));
	(ap - 1)->ssid &= ~E;
	while(ndigi-- != 0)
		bcopy((char *)dest++,(char *)ap++,sizeof(struct addr));

	(ap - 1)->ssid |= E;	/* Mark end of address field */	

	cp = (char *)ap;	/* Point to first byte past address field */
	*cp++ = UI;
	*cp++ = pid;

	abp->next = bp;		/* Link in data field */
	return abp;
}
/* Process incoming KISS TNC frame */
kiss_recv(interface,bp)
struct interface *interface;
struct mbuf *bp;
{
	pullup(&bp,NULLCHAR,1);	/* remove KISS TNC type field */
	ax_recv(interface,bp);
}
/* Process an incoming AX.25 packet */
int
ax_recv(interface,bp)
struct interface *interface;
struct mbuf *bp;
{
	void arp_input(),ip_route();
	struct addr addr;
	char pid,type,multicast;

#ifdef	TRACE
	if(trace){
		printf("%s recv:\r\n",interface->name);
		if((trace & 0xf) > 1)
			ax25_dump(bp);
		if(trace & 0x10)
			hexdump(bp);
		fflush(stdout);
	}
#endif
	/* Examine AX.25 destination address */
	if(pullup(&bp,(char *)&addr,sizeof(struct addr)) != sizeof(struct addr))
		return;
	if(addreq(&addr,&ax25_bdcst)){
		multicast = 1;	/* Broadcast packet */
	} else if(addreq(&addr,&mycall)){
		multicast = 0;	/* Packet directed at us */
	} else {
		/* Not for us */
		free_p(bp);
		return;
	}
	/* Now find the last address entry */
	while(pullup(&bp,(char *)&addr,sizeof(addr)) == sizeof(addr)){
		if(addr.ssid & E)
			break;	/* Found it */
	}
	if(pullup(&bp,&type,1) != 1)
		return;
	if(pullup(&bp,&pid,1) != 1)
		return;
	switch(pid){
	case PID_ARP:
		arp_input(interface,bp);
		break;
	case PID_IP:
		ip_route(bp,multicast);
		break;
	default:
		free_p(bp);
		break;
	}
}
/* Display or change our AX.25 address */
domycall(argc,argv)
int argc;
char *argv[];
{
	char buf[15];

	if(argc < 2){
		pax25(buf,&mycall);
		printf("%s\r\n",buf);
		return 0;
	}
	if(setcall(&mycall,argv[1]) == -1)
		return -1;
	mycall.ssid |= E;	/* Indicate end-of-address */
}
/*
 * setcall - convert callsign plus substation ID of the form
 * "KA9Q-0" to AX.25 (shifted) address format
 *   Address extension bit is left clear
 *   Return -1 on error, 0 if OK
 */
int
setcall(out,call)
struct addr *out;
char *call;
{
	int csize;
	unsigned ssid;
	register int i;
	register char *cp,*dp;
	char c,*index();

	if(out == (struct addr *)NULL || call == NULLCHAR || *call == '\0'){
		return -1;
	}
	/* Find dash, if any, separating callsign from ssid
	 * Then compute length of callsign field and make sure
	 * it isn't excessive
	 */
	dp = index(call,'-');
	if(dp == NULLCHAR)
		csize = strlen(call);
	else
		csize = dp - call;
	if(csize > 6)
		return -1;
	/* Now find and convert ssid, if any */
	if(dp != NULLCHAR){
		dp++;	/* skip dash */
		ssid = atoi(dp);
		if(ssid > 15)
			return -1;
	} else
		ssid = 0;
	/* Copy upper-case callsign, left shifted one bit */
	cp = out->call;
	for(i=0;i<csize;i++){
		c = *call++;
		if(islower(c))
			c = toupper(c);
		*cp++ = c << 1;
	}
	/* Pad with shifted spaces if necessary */
	for(;i<6;i++)
		*cp++ = ' ' << 1;
	
	/* Insert substation ID field and set reserved bits */
	out->ssid = 0x60 | (ssid << 1);
	return 0;
}
static
addreq(a,b)
register struct addr *a,*b;
{
	if(bcmp(a->call,b->call,ALEN) == 0 && (a->ssid & SSID) == (b->ssid & SSID))
		return 1;
	else
		return 0;
}
/* Convert encoded AX.25 address to printable string */
pax25(e,addr)
char *e;
struct addr *addr;
{
	register int i;
	char c,*cp;

	cp = addr->call;
	for(i=6;i != 0;i--){
		c = (*cp++ >> 1) & 0x7f;
		if(c == ' ')
			break;
		*e++ = c;
	}
	sprintf(e,"-%d",(addr->ssid >> 1) & 0xf);	/* ssid */
}
#ifdef	TRACE

/* Dump an AX.25 packet header */
ax25_dump(abp)
struct mbuf *abp;
{
	struct mbuf *bp;
	struct addr src,dest,addr;
	char tmp[20],*cp,control,cmdrsp,pid;
	int16 len,type,ftype();

	dup_p(&bp,abp,0,len_mbuf(abp));

	/* Read and print the destination and source addresses */
	if(pullup(&bp,(char *)&dest,sizeof(dest)) != sizeof(dest))
		goto quit;
	if(pullup(&bp,(char *)&src,sizeof(src)) != sizeof(src))
		goto quit;

	pax25(tmp,&src);
	printf("AX25: %s",tmp);
	pax25(tmp,&dest);
	printf("->%s",tmp);

	if((src.ssid & E) == 0){
		/* Find the last address entry in the digi string */
		printf(" v");
		while(pullup(&bp,(char *)&addr,sizeof(addr)) == sizeof(addr)){
			pax25(tmp,&addr);
			printf(" %s%s",tmp,(addr.ssid & REPEATED) ? "H":"");
			if(addr.ssid & E)
				break;	/* Found it */
		}
	}
	if(pullup(&bp,&control,1) != 1)
		goto quit;

	putchar(' ');
	type = ftype(control);
	switch(type){
	case I:
		printf("I");
		break;
	case SABM:
		printf("SABM");
		break;
	case DISC:
		printf("DISC");
		break;
	case DM:
		printf("DM");
		break;
	case UA:
		printf("UA");
		break;
	case RR:
		printf("RR");
		break;
	case RNR:
		printf("RNR");
		break;
	case REJ:
		printf("REJ");
		break;
	case FRMR:
		printf("FRMR");
		break;
	case UI:
		printf("UI");
		break;
	default:
		printf("[invalid]");
	}

	if((dest.ssid & C) != (src.ssid & C)){
		if(dest.ssid & C)
			cmdrsp = COMMAND;
		else
			cmdrsp = RESPONSE;
	} else 
		cmdrsp = UNKNOWN;
	/* Dump poll/final bit */
	if(control & PF){
		switch(cmdrsp){
		case COMMAND:
			printf("(P)");
			break;
		case RESPONSE:
			printf("(F)");
			break;
		default:
			printf("(P/F)");
			break;
		}
	}
	/* Dump sequence numbers */
	if((type & 0x3) != U)	/* I or S frame? */
		printf(" NR=%d",(control>>5)&7);
	if((type & 0x1) == I)	/* I frame? */
		printf(" NS=%d",(control>>1)&7);

	if(pullup(&bp,&pid,1) != 1)
		goto quit;
	printf(" pid 0x%x\r\n",pid & 0xff);
	if((trace & 0xf) > 2){
		switch(pid & 0xff){
		case PID_ARP:
			arp_dump(bp);
			break;
		case PID_IP:
			ip_dump(bp);
			break;
		}
	}
	free_p(bp);
	fflush(stdout);
	return;
quit:	/* I hate go-to's, but sometimes there's no real choice */
	free_p(bp);
	printf("\r\n");
	fflush(stdout);
}
/* Figure out the frame type from the control field
 * This is done by masking out any sequence numbers and the
 * poll/final bit after determining the general class (I/S/U) of the frame
 */
static
int16
ftype(control)
register char control;
{
	if((control & 1) == 0)	/* An I-frame is an I-frame... */
		return I;
	if(control & 2)			/* U-frames use all except P/F bit for type */
		return(control & ~PF);
	else					/* S-frames use low order 4 bits for type */
		return(control & 0xf);
}
#endif
