/*
	Boot Stage 2 Stack construction

These routines will construct an initial stack for ther Kernel, Memory Manager,
File System and Init programs.

*/
#include <minix/const.h>
#include <minix/type.h>
#include <kernel/const.h>
#include <kernel/type.h>
#include <kernel/proc.h>
#include <a.out.h>
#include <errno.h>
#include "type.h"

extern char **environ;			/* environment pointer */

#define	PTRSIZE	sizeof(char *)

/*#define DEBUG	/**/

/*
 * Build a stack from a set of arguments.
 *
 * This function is derived from Minix's lib/exec.c(execve) function.
 */
int buildstack(argv, argsize, envp, envsize, stack)
char *argv[];			/* pointer to argument array */
int argsize;			/* size of argument array elements (fixed)*/
char *envp[];			/* pointer to environment */
int envsize;			/* size of environment array elements (fixed)*/
char *stack;			/* memory to build the stack in */
{
/*  char stack[MAX_ISTACK_BYTES];*/
  char **argorg, **envorg, *hp, **ap, *p;
  int i, nargs, nenvps, stackbytes, offset;
  extern errno;

  /* Count the argument pointers and environment pointers. */
  nargs = 0;
  nenvps = 0;
  argorg = argv;
  envorg = envp;
  while (*argorg++ != NIL_PTR) nargs++;
  while (*envorg++ != NIL_PTR) nenvps++;

  /* Prepare to set up the initial stack. */
  hp = &stack[(nargs + nenvps + 3) * PTRSIZE];
#ifdef DEBUG
printf("nargs=%d nenvps=%d hp=0x%x\n",nargs,nenvps,hp);
#endif
  if (hp + nargs + nenvps >= &stack[MAX_ISTACK_BYTES]) {
	errno = E2BIG;
	return(-1);
  }
  ap = (char **) stack;
  *ap++ = (char *) nargs;

  /* Prepare the argument pointers and strings. */
  for (i = 0; i < nargs; i++) {
	offset = hp - stack;
	*ap++ = (char *) offset;
	p = *argv++;
	copy(hp, p, argsize);
	hp += argsize;
	if (hp >= &stack[MAX_ISTACK_BYTES]) {
		errno = E2BIG;
		return(-1);
	}
  }
  *ap++ = NIL_PTR;

  /* Prepare the environment pointers and strings. */
  for (i = 0; i < nenvps; i++) {
	offset = hp - stack;
	*ap++ = (char *) offset;
	p = *envp++;

	hp += envsize;
	if (hp >= &stack[MAX_ISTACK_BYTES]) {
		errno = E2BIG;
		return(-1);
	}
  }
  *ap++ = NIL_PTR;
  stackbytes = ( ( (int)(hp - stack) + PTRSIZE - 1)/PTRSIZE) * PTRSIZE;
  return(stackbytes);
}

/*===========================================================================*
 *				patch_ptr				     *
 *===========================================================================*/
/* taken from Minix's mm/exec.c */
PRIVATE patch_ptr(stack, base)
char stack[MAX_ISTACK_BYTES];	/* pointer to stack image within MM */
vir_bytes base;			/* virtual address of stack base inside user */
{
/* When doing an exec(name, argv, envp) call, the user builds up a stack
 * image with arg and env pointers relative to the start of the stack.  Now
 * these pointers must be relocated, since the stack is not positioned at
 * address 0 in the user's address space.
 */

  char **ap, flag;
  vir_bytes v;

  flag = 0;			/* counts number of 0-pointers seen */
  ap = (char **) stack;		/* points initially to 'nargs' */
  ap++;				/* now points to argv[0] */
  while (flag < 2) {
	if (ap >= (char **) &stack[MAX_ISTACK_BYTES]) return;	/* too bad */
	if (*ap != NIL_PTR) {
		v = (vir_bytes) *ap;	/* v is relative pointer */
		v += base;		/* relocate it */
		*ap = (char *) v;	/* put it back */
	} else {
		flag++;
	}
	ap++;
  }
}

/*
 * ProcStack - give a process a stack
 *
 * A stack is built up from the arguments (argv, envp) given, and copied
 * into the process' stack space.  The address that the stack pointer should
 * be assigned is returned.
 */
procstack(argv, sizeargv, envp, sizeenvp, proc)
char **argv;
int sizeargv;
char **envp;
int sizeenvp;
AOUT proc;
{	static char stack[MAX_ISTACK_BYTES];
	int	size,		/* size of process' stack */
		seg, offset;	/* segment and offset of stack dest. */

#ifdef DEBUG
	printf("Procstack...\n");
#endif
	size=buildstack(argv, sizeargv, envp, sizeenvp, stack);
	seg = (int) proc->a_dbase;
	offset = (int) proc->a_total - size;
	patch_ptr(stack, offset);
#ifdef DEBUG
	printf("seg= 0x%x offset=0x%x length=0x%x\n", seg, offset, size);
	printf("trsize=0x%x tbase=0x%x drsize=0x%x dbase=0x%x total=0x%x\n",
		(int) proc->a_trsize, (int) proc->a_tbase,
		(int) proc->a_drsize, (int) proc->a_dbase,
		(int) proc->a_total);
#endif
	lcopy(seg, offset, ds(), stack, size);
	return(offset);
}

/*
 *	Kernel Stack
 *
 *  The kernel stack consists of a "proc" array as argv and a device
 * configuration array as envp.  The processes present in the proc
 * array start with PID 0 (MM) and end with the first user process (INIT).
 *
 * Initially the device configuration is nonexistant.
 */
kstack(procinfo, sps)
AOUT *procinfo;
int *sps;
{	static struct proc proc[4], 	/* Kernel, MM, FS, INIT */
		*pproc[] = { proc, proc+1, proc+2, proc+3, 0 };
	static int *kenv[] = { 0, 0 };

	struct proc *p;
	int *s, sp;

	/* Build a process table for the kernel */
	p = proc; s = sps; 

	kproc(p++, s++, procinfo[0]);	/* kernel*/
	kproc(p++, s++, procinfo[1]);	/* MM	*/
	kproc(p++, s++, procinfo[2]);	/* FS	*/
	kproc(p++, s++, procinfo[3]);	/* init	*/

	dmpproc(proc);
	sp = procstack(pproc, sizeof(struct proc), kenv, sizeof(int), procinfo[0]);
	return(sp);	/* initial stack */
}

/*
 *	Set Up Kernel Process Table
 */
kproc(p, s, procinfo)
struct proc *p;
int *s;
AOUT procinfo;
{	p->p_map[T].mem_vir  = 0;
	p->p_map[T].mem_phys = (int) procinfo->a_tbase;
	p->p_map[T].mem_len  = clicks(procinfo->a_trsize);

	p->p_map[D].mem_vir  = (procinfo->a_trsize == 0) ?
				 (int) procinfo->a_trsize : 0;
	p->p_map[D].mem_phys = (int) procinfo->a_dbase;
	p->p_map[D].mem_len  = clicks(procinfo->a_total);

	p->p_map[S].mem_vir  = p->p_map[D].mem_vir + clicks(procinfo->a_total);
	p->p_map[S].mem_phys = p->p_map[D].mem_phys +clicks(procinfo->a_total);
	p->p_map[S].mem_len  = 0;

	p->p_sp = *s;
}

dmpproc(proc)
struct proc *proc;
{
  register struct proc *rp;
  vir_bytes base, limit, first, last;
  phys_bytes ltmp;

  printf("\nPROC   -----TEXT-----  -----DATA-----  ----STACK-----  BASE SIZE SP\r\n");
  for (rp = proc; rp < proc+4; rp++)  {
	if (rp->p_flags & P_SLOT_FREE) continue;
	first = rp->p_map[T].mem_phys;
	last = rp->p_map[S].mem_phys + rp->p_map[S].mem_len;
	ltmp = ((long) first << 4) + 512L;
	base = (vir_bytes) (ltmp/1024L);
	ltmp = (((long) (last-first) << 4) + 512L);
	limit = (vir_bytes) (ltmp/1024L);
	printf("%4d", rp - proc);

	printf(" %4x %4x %4x  %4x %4x %4x  %4x %4x %4x  %3dK %3dK %4x\r\n", 
	    rp->p_map[T].mem_vir, rp->p_map[T].mem_phys, rp->p_map[T].mem_len,
	    rp->p_map[D].mem_vir, rp->p_map[D].mem_phys, rp->p_map[D].mem_len,
	    rp->p_map[S].mem_vir, rp->p_map[S].mem_phys, rp->p_map[S].mem_len,
	    base, limit, rp->p_sp);
  }
}

/*
 *	Memory Manager Stack
 *
 *  The memory manager's stack consists of only the memory size of the system.
 */
mmstack(mmproc, sp)
AOUT mmproc;
int *sp;
{	static int mminfo[] = { 0x55AA },
	    *mmptr[] = { mminfo, 0 };
	static int *mmenv[] = { 0 };

#ifdef DEBUG
	printf("MM ");
#endif
	*sp = procstack(mmptr, sizeof(int), mmenv, sizeof(int), mmproc);
}

/*
 *	File System Stack
 *
 *  The File System's stack consists of five integers in "argv":
 *	[0] Device to load ram disk image (0 if root doesn't need loading)
 *	[1] Device to mount as root
 *	[2] Size of Local RAM disk
 *	[3] Size of Extended Memory (AT) RAM disk
 *	[4] Size of EMS RAM disk
 */
#define RAM_IMAGE (dev_nr)0x303	/* major-minor dev where root image is kept */
fsstack(fsproc, sp)
AOUT fsproc;
int *sp;
{	static int fsinfo[] = { 0, RAM_IMAGE, 0x1234, 0x5678, 0x9abc, 0 },
	    *fsptr[] = { fsinfo, fsinfo+1, fsinfo+2, fsinfo+3, fsinfo+4, 0 };
	static int *fsenv[] = { 0, 0 };

#ifdef DEBUG
	printf("FS ");
#endif
	*sp=procstack(fsptr, sizeof(int), fsenv, sizeof(int), fsproc);
#ifdef DEBUG
  printf("FS: 0x%x 0x%x 0x%x 0x%x 0x%x\n", *fsptr[0], *fsptr[1], *fsptr[2], *fsptr[3], *fsptr[4]);
#endif
}
