;\c	    Copyright (C) 1990 Pertti Kellomaki
;\c	 
;\c	 This file is part of Taurus, a parser generator producing Scheme
;\c	 
;\c	 Taurus is free software; you can redistribute it and/or modify
;\c	 it under the terms of the GNU General Public License as published by
;\c	 the Free Software Foundation; either version 1, or (at your option)
;\c	 any later version.
;\c	 
;\c	 Taurus is distributed in the hope that it will be useful,
;\c	 but WITHOUT ANY WARRANTY; without even the implied warranty of
;\c	 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;\c	 GNU General Public License for more details.
;\c	 
;\c	 You should have received a copy of the GNU General Public License
;\c	 along with Taurus; see the file COPYING.  If not, write to
;\c	 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;
;\node Fixed Parts, , , 
;\comment  Node-Name,  Next,  Previous,  Up
;\chapter{Fixed Parts}
;
;The generated parser contains a number of procedures that must be
;present. They are the same for every parser. The procedures deal  with
;reading of tokens and implement the error recovery. Many of the
;procedures change their behavior during error recovery, when they are used
;for collecting recovery information.
;
;Because the result of code generation is a list representation of the
;parser, the fixed procedures are given here in list form, each bound to a
;symbol named \var{procedure}\code{-source}. The fixed parts are
;generated with a call to \code{fixed-code}.
;
;These defines form the module
;
(module tfixed)

;\findex{fixed-code}
(define (fixed-code)
  (list tg-make-token-source
	tg-token-name-source
	tg-token-value-source
	tg-token-line-source
	tg-token-file-source
	tg-get-token-source
	tg-expect-source
	tg-next-source
	tg-check-for-source
	tg-invoke-error-recovery-source
	tg-add-synchronizing-token-source
	tg-add-anchor-tokens-source
	tg-push-back-source
	tg-recover-source
	tg-insert-tokens-source
	tg-delete-tokens-source
	tg-error-message-header-source
	tg-issue-severe-error-source
	tg-display-tokens-source
	tg-issue-error-message-source))


;\node Handling Tokens, , , 
;\comment  Node-Name,  Next,  Previous,  Up
;\section{Handling Tokens}
;
;The tokens that the scanner returns are made with the procedure
;\code{tg-make-token} that
;the parser returns in response to the message \code{get-token-maker}. The
;procedure takes four arguments: the name of the token, the semantic
;value of the token and the current file name and line number. 
;The following procedures implement the type token.
;
;\findex{tg-make-token-source}
(define tg-make-token-source
  '(define (tg-make-token name value line file)
     (list name value line file)))
;\findex{tg-token-name-source}
(define tg-token-name-source '(define tg-token-name car))
;\findex{tg-token-value-source}
(define tg-token-value-source '(define tg-token-value cadr))
;\findex{tg-token-line-source}
(define tg-token-line-source '(define tg-token-line caddr))
;\findex{tg-token-file-source}
(define tg-token-file-source '(define tg-token-file cadddr))


;\node Reading Tokens, , , 
;\comment  Node-Name,  Next,  Previous,  Up
;\section{Reading Tokens}
;
;Most of the time, the parser reads tokens that come directly from the
;lexical analyzer. After syntax errors, however, the error recovery
;scheme may generate tokens that should be consumed before
;continuing with the actual input. Therefore, \code{tg-get-token} maintains
;\code{*pushback-tokens*}, a list of pushed back tokens.
;
;The symbols \code{*lookahead*} and \code{*semantic-lookahead*} are
;\code{set!} to the next token and its semantic value, respectively.
;
;\findex{tg-get-token-source}
(define tg-get-token-source    
  '(define (tg-get-token)
     (let ((token
	    (if *pushback-tokens*
		(let ((temp (car *pushback-tokens*)))
		  (set! *pushback-tokens*
			(cdr *pushback-tokens*))
		  temp)
		(scan))))
       (set! *lookahead* (tg-token-name token))
       (set! *semantic-lookahead* (tg-token-value token))
       (cond ((tg-token-line token)
	      (set! *line-number* (tg-token-line token))))
       (cond ((tg-token-file token)
	      (set! *input-file* (tg-token-file token)))))))


;When the parser expects to see a certain token in the input stream, it
;calls the procedure \code{tg-expect}. \code{tg-expect} checks that the
;lookahead token is the required one.  If it is, it is consumed with
;\code{tg-get-token} and its semantic value is returned.
;If the token is not the desired one, error recovery is invoked using the
;procedure \code{tg-invoke-error-recovery}.
;
;From the caller's point of view, \code{tg-expect} just checks that the next
;token is legal, consumes it and returns its semantic value.
;\code{tg-expect} does do a bit more behind the scenes, though. If the next
;token is not the desired one, error recovery deletes and inserts tokens
;so that the next token always is the desired one. Thus, the
;other parts of the parser do not need to worry about erroneous
;tokens, because they never see them.
;
;During error recovery, the role of \code{tg-expect} is different. It is
;then used for collecting recovery information. The generated parser
;calls \code{tg-expect} for every token that \emph{must} be present in the
;input stream. They are collected and used when new tokens are generated. These
;tokens form the shortest way to end the parse, provided that the grammar
;is correctly constructed.
;
;\findex{tg-expect-source}
(define tg-expect-source    
  '(define (tg-expect token calling-nonterminal)
     (tg-check-for (list token) calling-nonterminal)
     (cond
      ((eq? *mode* 'collecting)
       (tg-add-synchronizing-token token))
      (else
       (let ((return-value *semantic-lookahead*))
	 (tg-get-token)
	 return-value)))))


;\node Checking For Tokens That May Be Present, , , 
;\comment  Node-Name,  Next,  Previous,  Up
;\section{Checking For Tokens That May Be Present}
;
;The generated parser uses the procedure \code{tg-next} to check if the
;lookahead token contains one of the tokens in a lookahead set. During
;error recovery, these tokens are collected. They are used for
;determining when to stop discarding tokens from the input stream. During
;error recovery, \code{tg-next} always returns \code{#f}.
;
;\findex{tg-next-source}
(define tg-next-source
    '(define (tg-next tokens)
      (cond
       ((eq? *mode* 'parsing)
	(member *lookahead* tokens))
       ((eq? *mode* 'collecting)
	(tg-add-anchor-tokens tokens)
	#f))))


;\node Checking For Tokens That Must Be Present, , , 
;\comment  Node-Name,  Next,  Previous,  Up
;\section{Checking For Tokens That Must Be Present}
;
;In order to prevent error recovery from missing possible synchronizing
;tokens, a call to \code{tg-check-for} is placed before some \code{cond}
;forms generated from alternative expressions (\pxref{Code Generated for
;Alternatives} for more information). This call makes sure that one
;branch of the \code{cond} will be selected, and invokes error recovery
;if it is not the case. During
;error recovery this procedure does nothing.
;
;\findex{tg-check-for-source}
(define tg-check-for-source
  '(define (tg-check-for tokens calling-nonterminal)
       (if (and (eq? *mode* 'parsing)
		(not (member *lookahead* tokens)))
	   (begin
	     (set! *error-level* calling-nonterminal)
	     (tg-invoke-error-recovery))
	    #f)))


;\node Implementation Of Error Recovery, , , 
;\comment  Node-Name,  Next,  Previous,  Up
;\section{Implementation Of Error Recovery}
;
;When a syntacticlly erroneous token is encountered, the procedure
;\code{tg-invoke-error-recovery} is called. This procedure calls a small
;\code{lambda} form using \code{call-with-current-continuation}.
;The \code{lambda} form stores the current continuation in
;\code{*continuation*} and returns \code{#t}. This way, the first branch
;of the \code{if} form is selected, and the variable
;\code{*mode*} is updated.
;If a synchronizing token is given (ie. the syntax error was
;encountered in \code{tg-expect}, \code{*anchor-set*} and \code{*recovery-info*}
;are initialized with it.
;
;After error recovery has taken place, the continuation that was stored
;in \code{*continuation*} is invoked with the argument \code{#f}
;{}(\pxref{Overall Structure of the Parser}), and thus the second
;branch is selected. If value of the variable \code{*mode*} is
;\code{collecting}, it is \code{set!} to \code{recovering}.
;
;\code{*recovery-info*} and \code{*anchor-set*} contain information
;about the tokens that may follow the current state of the parser.
;The anchor set of the current parser state is collected in
;\code{*anchor-set*}. \code{*recovery-info*} contains entries of the form
;\code{(allowable} \var{token}\code{)} and \code{(synchronizing} \var{token}\code{)}. These
;correspond to tokens that may appear  and tokens that must appear
;in the input stream in order for the parse to succeed, respectively. The
;procedure \code{tg-recover} uses this information for inserting corrective
;tokens to the input stream.
;
;\findex{tg-invoke-error-recovery-source}
(define tg-invoke-error-recovery-source
  '(define (tg-invoke-error-recovery)
     (if (call-with-current-continuation
	  (lambda (cont) (set! *continuation* cont) #t))
	 (begin
	   (set! *anchor-set* '())
	   (set! *recovery-info* '())
	   (set! *mode* 'collecting))
	 (if (eq? *mode* 'collecting)
	     (set! *mode* 'recovering)))))

;\node Adding Tokens To The Recovery Information, , , 
;\comment  Node-Name,  Next,  Previous,  Up
;\subsection{Adding Tokens To The Recovery Information}
;
;Tokens that must be present in the input stream are added to the
;recovery information using \code{tg-add-synchronizing-token}.
;
;\findex{tg-add-synchronizing-token-source}
(define tg-add-synchronizing-token-source
  '(define (tg-add-synchronizing-token token)
     (set! *recovery-info*
	   `((synhronizing ,token) ,@*recovery-info*))
     (set! *anchor-set* (cons token *anchor-set*))))

;Tokens that may be present, are added using \code{tg-add-anchor-tokens}.
;
;\findex{tg-add-anchor-tokens-source}
(define tg-add-anchor-tokens-source
  '(define (tg-add-anchor-tokens tokens)
     (let loop ((tokens tokens))
       (cond ((not (null? tokens))
	      (set! *recovery-info*
		    `((allowable ,(car tokens))
		      ,@*recovery-info*))
	      (loop (cdr tokens)))))
     (set! *anchor-set* (append tokens *anchor-set*))))

;\node Recovering, , , 
;\comment  Node-Name,  Next,  Previous,  Up
;\subsection{Recovering}
;
;When error recovery information has been collected, the procedure
;\code{tg-recover} is called. It deletes and inserts tokens based on the
;information in \code{*anchor-set*} and \code{*recovery-info*}.
;    
;\findex{tg-recover-source}
(define tg-recover-source
  '(define (tg-recover)
     (tg-delete-tokens *anchor-set*)
     (tg-insert-tokens (reverse *recovery-info*) *lookahead*)))
    

;\subsubsection{Deleting Tokens}
;
;Erroneous tokens are deleted using \code{tg-delete-tokens}. Tokens are read
;from the input stream and discarded until a token that appears in
;\code{follow-set} is found. That token is left in \code{*lookahead*}.
;    
;\findex{tg-delete-tokens-source}
(define tg-delete-tokens-source
  '(define (tg-delete-tokens follow-set)
     (set! *deleted-tokens* '())
     (let loop ((deleted-tokens '()))
       (if (and (not (member *lookahead* follow-set))
		(not (eof-object? *lookahead*)))
	   (let ((token (tg-make-token *lookahead*
				       *semantic-lookahead*
				       *line-number*
				       *input-file*)))
	     (tg-get-token)
	     (loop (cons token deleted-tokens)))
	   (set! *deleted-tokens* (reverse deleted-tokens))))))


;\subsubsection{Inserting tokens}
;
;Tokens are inserted to the input stream based on information in
;\code{recovery-list}.  \code{recovery-list} is scanned until the same token as
;\code{synch-token} is found, ie. the parser can synchronize. During the
;scan, all tokens in \code{recovery-list} that are marked as \code{synhronizing}
;tokens, are pushed back to the input stream with a semantic value
;\code{undefined}.
;    
;\findex{tg-insert-tokens-source}
(define tg-insert-tokens-source
  '(define (tg-insert-tokens recovery-list synch-token)
     (set! *inserted-tokens* '())
     (let loop ((recovery-list recovery-list)
		(inserted-tokens '()))
       (cond ((or (null? recovery-list)
		  (eq? (cadar recovery-list) synch-token))
	      (set! *inserted-tokens*
		    (reverse inserted-tokens))
	      (tg-push-back *inserted-tokens*))
	     ((eq? (caar recovery-list) 'synhronizing)
	      (loop (cdr recovery-list)
		    (cons (tg-make-token (cadar recovery-list)
					 'undefined
					 #f
					 #f)
			  inserted-tokens)))
	     (else (loop (cdr recovery-list)
			 inserted-tokens))))))
	  

;\code{tg-push-back} pushes \code{tokens} back to the input stream so that
;\code{tg-get-token} will read them. 
;    
;\findex{tg-push-back-source}
(define tg-push-back-source
  '(define (tg-push-back tokens)
     (cond (tokens
	    (set! *pushback-tokens*
		  `(,@(cdr tokens)
		    ,(tg-make-token *lookahead*
				 *semantic-lookahead*
				 *line-number*
				 *input-file*)
		    ,@*pushback-tokens*))
	    (set! *lookahead* (caar tokens))
	    (set! *semantic-lookahead* (cdar tokens))))))


;\node Issuing Error Messages
;
;For deletion of tokens and insertion of purely syntactic tokens, the
;parser simply issues an error message and continues parsing.
;
;\findex{tg-error-message-header-source}
(define tg-error-message-header-source
  '(define (tg-error-message-header line-number filename)
   (display filename stderr-port)
   (display ":" stderr-port)
   (display line-number stderr-port)
   (display ":" stderr-port)))

;\findex{tg-display-tokens-source}
(define tg-display-tokens-source
  '(define (tg-display-tokens tokens)
     (let
	 loop
       ((tokens tokens))
       (cond (tokens
	      (display (if (eq? (tg-token-value (car tokens))
				'undefined)
			   (tg-token-name (car tokens))
			   (tg-token-value (car tokens)))
		       stderr-port)
	      (display " " stderr-port)
	      (loop (cdr tokens)))))))


;\findex{tg-issue-severe-error-source}
(define tg-issue-severe-error-source
  '(define (tg-issue-severe-error line-number
				  filename
				  nonterminal-name
				  lookahead
				  semantic-lookahead
				  pushback-tokens
				  deleted-tokens
				  inserted-tokens)
     (tg-error-message-header line-number filename)
     (display nonterminal-name stderr-port)
     (display " expected." stderr-port)
     (newline stderr-port)
     (cond (deleted-tokens
	    (tg-error-message-header line-number filename)
	    (display "Extra " stderr-port)
	    (tg-display-tokens deleted-tokens)
	    (newline stderr-port)))
     (cond ((eq? semantic-lookahead 'undefined)
	    (let loop
		((expected-tokens
		  (tg-make-token lookahead
				 semantic-lookahead
				 line-number
				 filename))
		 (pushback-tokens pushback-tokens))
	      (cond ((eq? (tg-token-value
			   (car pushback-tokens))
			  'undefined)
		     (loop (cons (car pushback-tokens)
				 expected-tokens)
			   (cdr pushback-tokens)))
		    (else
		     (tg-display-tokens expected-tokens)
		     (display "expected." stderr-port)
		     (newline stderr-port))))))))

;\findex{tg-issue-error-message-source}
(define tg-issue-error-message-source
  `(define (tg-issue-error-message)

     (define (pure-syntactic-sugar? tokens)
       (let loop ((tokens tokens))
	 (cond ((null? tokens))
	       ((not (member (caar tokens) *sugar*))
		#f)
	       (else
		(loop (cdr tokens))))))
     (cond ((null? *inserted-tokens*)
	    (tg-error-message-header *line-number*
				     *input-file*)
	    (display "Extra " stderr-port)
	    (tg-display-tokens *deleted-tokens*)
	    (newline stderr-port)
	    (set! *mode* 'parsing))
	   ((and (null? *deleted-tokens*)
		 (pure-syntactic-sugar? *inserted-tokens*))
	    (tg-error-message-header *line-number*
				     *input-file*)
	    (tg-display-tokens *inserted-tokens*)
	    (display "expected." stderr-port)
	    (newline stderr-port)
	    (set! *mode* 'parsing)))))
	    

