;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;; SCIX display object.

;;; $Id: dpy-obj.sc,v 1.6 91/09/15 01:13:15 johani Exp $

(module scixdpy); (top-level xdisplay make-display))

(include "../include/lw-objs.sch")
(include "../include/lowlevel.sch")
(include "../include/requests.sch")
(include "../include/msgutil.sch")
(include "../include/util.sch")
(include "../include/masks.sch")
(include "../include/types.sch")

(define-external screen scixobj)
(define-external visual scixobj)
(define-external make-hash-vector hash)

(include "../macros/extsyntax.sc")
(include "../macros/oos.sc")

(include "atoms.sc")
(include "formats.sc")

;;; Display object -- Most of the functionality of the display should be
;;;                   exported to the screen objects.
(define-class (xdisplay . args)
  (locals
   (fd #f) (allocator #f) (screens '()) (ptr #f) (kbd #f) (atombox #f) 
   ;; request-buffer, current-request and max-buffer are changed in the
   ;; init part, after the servers limits are known.
   (request-buffer (make-string 1024)) (current-request (make-string 1024))
   (max-buffer 1024) (tot-buffer-size 0) (synchronous-mode #f)
   (queried-extensions `()) (initial-reply #f)
   (extension-major-opcodes '())
   (extension-first-events '()) 
   (extension-first-errors '())
   (extension-objects '())
   (id-vector (make-hash-vector (lambda (x) x)
				(lambda (o) (o 'id))
				assv
				101))
   (event-formats (make-vector 256 '()))
   (error-names (make-vector 256 '()))
   (history-list '((-1 history-list)))
   (error-handler (lambda (error)
		    (scix-msg
     "XDPY: error-handler: Got a ~a-error. Opcode: ~a. Error object: ~a~%"
		     ((car error) 'error-name)
		     ((car error) 'major-opcode)
		     (if (cadr error) (cadr error) 'no))
		    (set! *error-queue* (append *error-queue*
						(list (car error)))) )) )
  (methods
   (scix-socket (lambda () fd))
   ;;
   ;; Buffering operations
   ;;
   ;; Note1: It is difficult to have the check for overflow in C, as it is
   ;;        impossible to have the current position there all the time
   ;;        (because of several dpy's sharing the same C routines).
   ;; Note2: Currently we are using a hybrid assembler: some requests 
   ;;        generate their assembly as two vectors, one with type-functions
   ;;        and one with data, other requests still emit a list of pairs.
   ;;        The old requests now send this list as the third argument (datav)
   ;;        to the 'scix-xas method. Args #1 and #2 are guaranteed to be #f.
   (scix-xas
    (let ((current-request-size 0) (errobj #f) (seq-nr -1))
      (lambda (len fmt dta rest)
	(set! current-request-size 0)
	(zero-buffer-position)
;;; We do no long support the old encoding method.
;;;	(if fmt			      ; New encoding method?
;;;	    (begin
	      (let loop ((index 0))
		((vector-ref fmt index) (vector-ref dta index)
					current-request
					me)
		(if (< index len)
		    (loop (+ index 1)) ))
;;;	      )
;;;	    (begin
;;;	      (display
;;;	       (format
;"DISPLAY: Hmm... This request (~a) looks like the old format. Using plan B.~%"
;;;                (cdar dta)))
;;;	      (for-each (lambda (p)     ; Nope.
;;;			  ((car p) (cdr p) current-request me) )
;;;			dta)))
	
	(set! current-request-size (current-buffer-position))
	(if synchronous-mode
	    (begin
	      (writesock fd current-request current-request-size)
	      (set! tot-buffer-size 0) )
	    (if (> (+ tot-buffer-size current-request-size)
		   max-buffer)
		(begin		  ;; Need to flush the buffer
		  (writesock fd request-buffer tot-buffer-size)
		  (set! tot-buffer-size
			(string-append! request-buffer current-request 0
					current-request-size)) )
		(begin
		  (set! tot-buffer-size
			(string-append! request-buffer current-request
					tot-buffer-size
					current-request-size)))))
	(set! seq-nr (+ 1 seq-nr))
	(set! errobj (strip-object rest)) ; Any special error object?
	(if (not (null? errobj))
	    (set-cdr! (last-pair history-list) (list (cons seq-nr errobj)) ))
	seq-nr)))
    
   (synchronize (lambda (onoff)
		  (if (not synchronous-mode)
		      (me 'flush!) )
		  (set! synchronous-mode onoff) ))
   (flush! (lambda ()
	     (if (not (zero? tot-buffer-size))
		 (begin
		   (writesock fd request-buffer tot-buffer-size)
		   (set! tot-buffer-size 0)
		   #t)		; #t indicates stuff found to flush
		 #f)))		; #f           nothing to flush
   

   (sync! (lambda (discard)
	    (msg-handler 'sync! me discard) ))

   ;;
   ;; Some internal stuff.
   ;;
   (scix-buffer-size (lambda () max-buffer))
   (scix-history-list (lambda () history-list)) ; Debug
   (scix-initial-reply (lambda ()
			 initial-reply))
   (scix-id-vector (lambda () id-vector))
   (scix-event-formats (lambda () event-formats))
   (scix-error-names (lambda () error-names))
   (scix-unique-id (lambda ()		; Return a new unused X id.
		     (allocator) ))
   (disconnect (lambda ()
		 (disconnectdisplay fd)
		 (define-system-file-task fd #f #f) ; Scheme->C specific:
		                                    ; Remove us from watched
		 'disconnected))	            ; sockets
   (atombox (lambda ()
	      (if atombox
		  atombox
		  (begin
		    (set! atombox (make-atombox x-protocol-atoms me))
		    atombox))))		    
   (pointer (lambda ()		; Return the pointer object
	      (if ptr
		  ptr
		  (begin
		    (set! ptr (pointer 'make me))
		    ptr))))
   (keyboard (lambda ()		; Return the keyboard object
	       (if kbd
		   kbd
		   (begin
		     (set! kbd (keyboard 'make me))
		     kbd))))
   (defaultscreen (lambda ()
		    (car screens) ))
   (screens (lambda () screens))	; A list of screen objects
   
   ;;
   ;; Methods directly mapped on X protocol core requests
   ;;
   (setselectionowner (lambda (owner sel time . rest)  ; #22: SetSelectionOwner
			(send-setselectionowner owner sel time me rest) ))
   ;; Not really the purist's cup of tea.
   (getselectionowner (lambda (sel . rest)             ; #23: GetSelectionOwner
 			(let ((r (send-getselectionowner sel me rest)))
 			  (if (x-reply? r)
 			      (r 'owner)
 			      r)) ))
   (convertselection (lambda (requestor sel target      ; #24: ConvertSelection
					prop time . rest) 
		       (send-convertselection requestor sel target
					      prop time me rest) ))
   (sendevent (lambda (propagate w ev-mask              ; #25: SendEvent
				 ev-name ev-data . rest) 
		(send-sendevent propagate w ev-mask
				ev-name ev-data me rest) ))
   (allowevents (lambda (mode time . rest)              ; #35: AllowEvents
		  (send-allowevents mode time me rest) ))
   (grabserver (lambda rest		                ; #36: GrabServer
		 (send-grabserver me rest) ))
   (ungrabserver (lambda rest		                ; #37: UngrabServer
		   (send-ungrabserver me rest) ))
   (setinputfocus (lambda (revert-to focus time . rest) ; #42: SetInputFocus
		    (send-setinputfocus revert-to focus time me rest) ))
   (getinputfocus (lambda rest		                ; #43: GetInputFocus
		    (send-getinputfocus me rest) ))
   (listfonts (lambda (maxnames pattern . rest)         ; #49: ListFonts
		(send-listfonts maxnames pattern me rest) ))
   (listfontswithinfo (lambda (maxnames pattern . rest) ;#50: ListFontsWithInfo
			(send-listfontswithinfo maxnames pattern me rest) ))
   (setfontpath (lambda (list-of-str . rest)	        ; #51: SetFontPath
		  (send-setfontpath list-of-str me rest) ))
   (getfontpath (lambda rest		                ; #52: GetFontPath
		  (send-getfontpath me rest) ))
   (queryextension (lambda (name . rest)                ; #98: QueryExtension
		     (send-queryextension name me rest) ))
   (listextensions (lambda rest                         ; #99: ListExtensions
		     (send-listextensions me rest) ))
   (bell (lambda (percent . rest)	                ; #104: Bell
	   (send-bell percent me rest) ))
   (setscreensaver (lambda (timeout interval            ; #107: SetScreenSaver
				    prefer-blanking allow-exposures . rest)
		     (send-setscreensaver timeout interval prefer-blanking
					  allow-exposures me rest) ))
   (getscreensaver (lambda rest                         ; #108: GetScreenSaver
		     (send-getscreensaver me rest) ))
   (changehosts (lambda (mode family addr . rest)       ; #109: ChangeHosts
		  (send-changehosts mode family addr me rest) ))
   (listhosts (lambda rest                              ; #110: ListHosts
		(send-listhosts me rest) ))
   (setaccesscontrol (lambda (mode . rest)             ; #111: SetAccessControl
		       (send-setaccesscontrol mode me rest) ))
   (setclosedownmode (lambda (mode . rest)             ; #112: SetCloseDownMode
		       (send-setclosedownmode mode me rest) ))
   (killclient (lambda (resource . rest)                ; #113: KillClient
		 (send-killclient resource me rest) ))
   (forcescreensaver (lambda (mode . rest)             ; #115: ForceScreenSaver
		       (send-forcescreensaver mode me rest) ))
   (nooperation (lambda rest                            ; #127: NoOperation
		  (send-nooperation me rest) ))
   ;;
   ;; Error handling.
   ;;
   ;; Note: The structure of an error-object as recieved by an error-handler
   ;;       is (<error> <object>), where <error> is the error-object supplied
   ;;       by the disassembler and <object> is the unparsed object that can
   ;;       be supplied in the assembler. It can typically be a continuation.
   ;;       The history-list is a list of pairs of seq-nr and <object>.
   (install-error-handler (lambda (handler)
			    (let ((old-handler error-handler))
			      (set! error-handler handler)
			      old-handler)))
   (scix-error-dispatch (lambda (error)
			  (let ((error-seq-nr (error 'seq-nr)))
			    (let loop ((hist-l history-list))
			      (cond ((null? hist-l)
				     (error-handler (list error #f)) )
				    ((= error-seq-nr (caar hist-l))
				     (set-cdr! history-list (cdr hist-l))
				     (error-handler (list error
							  (cdar hist-l) )))
				    (else
				     (loop (cdr hist-l)) ))))))
   
   ;; Extension hook. Not yet finished!
   (activate-extension (lambda (name events errors object)
			 (if (not (assoc name extension-major-opcodes))
			     (let ((r (me 'queryextension name)))
			       (if (r 'present)
				   (begin
				     (set! extension-major-opcodes
					   (cons (cons name (r 'major-opcode))
						 extension-major-opcodes))
				     (set! extension-first-events
					   (cons (cons name (r 'first-event))
						 extension-first-events))
				     (set! extension-first-errors
					   (cons (cons name (r 'first-error))
						 extension-first-errors))
				     (set! extension-objects
					   (cons (cons name object)
						 extension-objects))
				     (let loop ((i (r 'first-event))
						(new-events events) )
				       (if (not (null? new-events))
					   (begin
					     (vector-set! event-formats
							  i
							  (car new-events) )
					     (loop (+ 1 i)
						   (cdr new-events) ))))
				     (let loop ((i (r 'first-error))
						(new-errors errors) )
				       (if (not (null? new-errors))
					   (begin
					     (vector-set! error-names
							  i
							  (car new-errors) )
					     (loop (+ 1 i)
						   (cdr new-errors) ))))
				     #t) ; Extension existed
				   #f))	 ; Extension did not exist
			     #t)))	 ; Extension already activated
   
   ;; The three selectors for extension offset information have NO safety
   ;; checks. It is the responsibility of the application to ensure that the
   ;; extensions exist and are properly activated.
   (extension-major-opcode (lambda (name)
			     (cdr (assoc name extension-major-opcodes)) ))
   (extension-first-event (lambda (name)
			    (cdr (assoc name extension-first-events)) ))
   (extension-first-error (lambda (name)
			    (cdr (assoc name extension-first-errors)) ))
   (extension-object (lambda (name)
		       (cdr (assoc name extension-objects)) ))
   (set-extension-object! (lambda (name object)
			    (let ((p (assoc name extension-objects)))
			      (if p (set-cdr! p object)) )))
   )
  
  ;; The init routine for a display object attempts to connect to the server
  ;; and on success sets up various local variables.
  ;;
  (init
   (set! fd (connectdisplay (if (or (null? args)
				    (zero? (string-length (car args))) )
				0
				(car args) )))
   (cond ((negative? fd)
	  (scix-msg "xdisplay: Error when connecting to server~%")
	  (set! me #f) )		; Junk display object
	 (else
	  (let ((reply (send-initconnection me)))
	    (cond ((pair? reply)
		   ;; 
		   ;; Connection refused.
		   ;; 
		   (scix-msg "xdisplay: Connection refused. Reason: ~a~%"
			     ((car reply) 'reason) )
		   (set! me #f) )       ; Junk display object
		  (else
		   ;;
		   ;; Connection accepted.
		   ;;
		   (set! initial-reply reply)
		   (set! allocator (make-xid-allocator initial-reply))
		   (blocking-io fd)
		   (set! screens (map (lambda (data)
					(screen 'make data me))
				      (initial-reply 'roots) ))
		   (set! max-buffer (initial-reply 'max-request-len) )
		   (set! request-buffer (make-string max-buffer))
		   (set! current-request (make-string max-buffer))
		   ;; copy the event formats...
		   (let loop ((i (- (vector-length *event-formats*) 1)))
		     (vector-set! event-formats
				  i
				  (tree-copy (vector-ref *event-formats* i)))
		     (if (positive? i)
			 (loop (- i 1)) ))
		   ;; copy the error formats...
		   (let loop ((i (- (vector-length *error-names*) 1)))
		     (vector-set! error-names
				  i
				  (tree-copy (vector-ref *error-names* i)))
		     (if (positive? i)
			 (loop (- i 1)) ))
		   ))))))
  ) ;; End of define-class

;;; make-xid-allocator is converted to both display and initial-reply as
;;; objects
(define (make-xid-allocator initial-reply)
  (let ((id 0)
	(id-mask (initial-reply 'resource-id-mask))
	(id-base (initial-reply 'resource-id-base)) )
    (lambda ()
      (set! id (+ 1 id))
      (bit-or (bit-and id id-mask) id-base) )))
