;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Author: Johan Ihren, KTH

;;;
;;; Some random thoughts on how to incorporate extensions into SCIX as
;;; applied to the sample SHAPE extension available in X11R4.
;;;
;;; $Id: shape.sc,v 1.1 91/08/19 14:35:50 johani Exp $

(module scixshapeext)

(include "../include/types.sch")
(include "../include/util.sch")
(include "../include/msg-util.sch")
(include "../include/lowlevel.sch")

(define-external create-event scixce)
;(define-external x-protocol-atoms scixglobal) ; Note: Wrong place.

;;; Note1: It is obvious that the current format for the request encoding won't
;;;        work for extensions. It has to be conform for all fields in the
;;;        request so the current special handling of the opcode must go away.
;;; Note2: The 'info-extension method of the display/screen is thought to
;;;        cache the values major-opcode, first-event and first-error, in that
;;;        order, when it is first used. So (car (...)) should return the
;;;        major-opcode.

(define (a-pixmap-or-none p buffer dpy)
  (if (eq? p 'None)
      (a-card32 0 buffer dpy)
      (a-pixmap p buffer dpy) ))

;;; shape-window is a suggestion for how the Shape extension might be used.
;;; It is not thoroughly tested.

(define-class (shape-window . args)
  (locals
   (scr (match-arg 'screen 'no-default args)) )
  (inherit (window args))
  (methods
   (shaperectangles (lambda (op kind order xoff yoff rec-list . rest)
		      (send-ShapeRectangles op kind order me
					    xoff yoff rec-list scr rest) )) 
   (shapemask (lambda (op kind xoff yoff mask . rest)
		(send-ShapeMask op kind me xoff yoff mask scr rest) ))
   (shapecombine (lambda () 'foo))
   (shapeoffset (lambda (dest destkind xoff yoff . rest)
		  (send-ShapeOffset dest xoff yoff scr rest) ))
   (shapequeryextents (lambda () 'foo))
   (shapeselectinput (lambda (enable . rest)
		       (send-ShapeSelectInput me enable scr rest) ))
   (shapeinputselected (lambda rest
			 (send-ShapeInputSelected me scr rest) ))
   (shapegetrectangles (lambda (kind . rest)
			 (send-ShapeGetRectangles me kind scr rest) ))
   ))

;;; Request Shape-#0: ShapeQueryVersion
(define send-ShapeQueryVersion
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector #f 0 1)) )
    (lambda (scr . rest)
      (vector-set! dta 0 (scr 'extension-major-opcode "SHAPE"))
      (msg-handler 'wait-for-reply
		   (scr 'scix-xas 2 fmt dta rest)
		   `((reply-name    . ,(d-identity 'ShapeQueryVersion))
		     (unused        . ,d-card8)
		     (seq-nr        . ,d-card16)
		     (unused        . ,d-card32) ; Reply length not used
		     (major-version . ,d-card16)
		     (minor-version . ,d-card16)
		     (unused        . ,(d-unused 20)) )
		   scr) )))

;;; Request Shape-1: ShapeRectangles
(define send-ShapeRectangles
  (let ((fmt (vector a-request a-card8 a-card16 a-card8 a-card8 a-card8
		     a-card8 a-window a-int16 a-int16 (a-list a-rectangle)))
	(dta (vector #f 1 #f #f #f #f 0 #f #f #f #f)) )
    (lambda (operation dstkind
		       ordering dstwin xoff yoff rectangle-list scr . rest)
      (vector-set! dta 0 (scr 'extension-major-opcode "SHAPE"))
      (vector-set! dta 2 (+ 4 (* 2 (length rectangle-list))))
      (vector-set! dta 3 (lookup-constant operation '((Set       . 0)
						      (Union     . 1)
						      (Intersect . 2)
						      (Subtract  . 3)
						      (Invert    . 4) )))
      (vector-set! dta 4 (lookup-constant dstkind '((Bounding . 0)(Clip . 1))))
      (vector-set! dta 5 (lookup-constant ordering '((UnSorted . 0)
						     (YSorted  . 1)
						     (YXSorted . 2)
						     (YXBanded . 3) )))
      (vector-set! dta 7 dstwin)
      (vector-set! dta 8 xoff)
      (vector-set! dta 9 yoff)
      (vector-set! dta 10 rectangle-list)
      (scr 'scix-xas 10 fmt dta rest) )))

;;; Request Shape #2: ShapeMask
(define send-ShapeMask
  (let ((fmt (vector a-request a-card8 a-card16 a-card8 a-card8
		     a-card16 a-window a-int16 a-int16 a-pixmap-or-none))
	(dta (vector #f 2 5 #f #f 0 #f #f #f #f)) )
    (lambda (operation dstkind dstwin xoff yoff srcbmap scr . rest)
      (vector-set! dta 0 (scr 'extension-major-opcode "SHAPE"))
      (vector-set! dta 3 (lookup-constant operation '((Set . 0) (Union . 1)
						      (Intersect . 2)
						      (Subtract  . 3)
						      (Invert    . 4) )))
      (vector-set! dta 4 (lookup-constant dstkind '((Bounding . 0)(Clip . 1))))
      (vector-set! dta 6 dstwin)
      (vector-set! dta 7 xoff)
      (vector-set! dta 8 yoff)
      (vector-set! dta 9 srcbmap)
      (scr 'scix-xas 9 fmt dta rest) )))

;;; Request Shape-3: ShapeCombine
(define send-ShapeCombine
  (let ((fmt (vector a-request a-card8 a-card16 a-card8 a-card8
		     a-card8 a-card8 a-window a-int16 a-int16 a-window))
	(dta (vector #f 3 5 #f #f #f 0 #f #f #f #f)) )
    (lambda (operation dstkind srckind dstwin xoff yoff srcwin scr . rest)
      (vector-set! dta 0 (scr 'extension-major-opcode "SHAPE"))
      (vector-set! dta 3 (lookup-constant operation '((Set . 0) (Union . 1)
						      (Intersect . 2)
						      (Subtract  . 3)
						      (Invert    . 4) )))
      (vector-set! dta 4 (lookup-constant dstkind '((Bounding . 0)(Clip . 1))))
      (vector-set! dta 5 (lookup-constant srckind '((Bounding . 0)(Clip . 1))))
      (vector-set! dta 7 dstwin)
      (vector-set! dta 8 xoff)
      (vector-set! dta 9 yoff)
      (vector-set! dta 10 srcwin)
      (scr 'scix-xas 10 fmt dta rest) )))

;;; Request Shape-4: ShapeOffset
(define send-ShapeOffset
  (let ((fmt (vector a-request a-card8 a-card16
		     a-card8 a-card8 a-card16 a-window a-int16 a-int16))
	(dta (vector #f 4 4 #f 0 0 #f #f #f)) )
    (lambda (dstkind dstwin xoff yoff scr . rest)
      (vector-set! dta 0 (scr 'extension-major-opcode "SHAPE"))
      (vector-set! dta 3 (lookup-constant dstkind '((Bounding . 0)(Clip . 1))))
      (vector-set! dta 6 dstwin)
      (vector-set! dta 7 xoff)
      (vector-set! dta 8 yoff)
      (scr 'scix-xas 8 fmt dta rest) )))

;;; Request Shape-5: ShapeQueryExtents
(define send-ShapeQueryExtents
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector #f 5 2 #f)) )
    (lambda (win scr . rest)
      (vector-set! dta 0 (scr 'extension-major-opcode "SHAPE"))
      (vector-set! dta 3 win)
      (msg-handler 'wait-for-reply
		   (scr 'scix-xas 3 fmt dta rest)
		   `((reply-name . ,(d-identity 'ShapeQueryExtents))
		     (unused                        . ,card8)
		     (seq-nr                        . ,d-card16)
		     (unused                        . ,d-card32)
		     (bounding-shaped               . ,d-bool)
		     (clip-shaped                   . ,d-bool)
		     (unused                        . ,d-card16)
		     (bounding-shape-extents-x      . ,d-int16)
		     (bounding-shape-extents-y      . ,d-int16)
		     (bounding-shape-extents-width  . ,d-card16)
		     (bounding-shape-extents-height . ,d-card16)
		     (clip-shape-extents-x          . ,d-int16)
		     (clip-shape-extents-y          . ,d-int16)
		     (clip-shape-extents-width      . ,d-card16)
		     (clip-shape-extents-height     . ,d-card16)
		     (unused                        . ,d-card32) )
		   scr) )))

;;; Request Shape-6: ShapeSelectInput
(define send-ShapeSelectInput
  (let ((fmt (vector a-request
		     a-card8 a-card16 a-window a-bool a-card8 a-card16))
	(dta (vector #f 6 3 #f #f 0 0)) )
    (lambda (dstwin enable scr . rest)
      (vector-set! dta 0 (scr 'extension-major-opcode "SHAPE"))
      (vector-set! dta 3 dstwin)
      (vector-set! dta 4 enable)
      (scr 'scix-xas 6 fmt dta rest) )))

;;; Request Shape-7: ShapeInputSelected
(define send-ShapeInputSelected
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector #f 7 2 #f)) )
    (lambda (dstwin scr . rest)
      (vector-set! dta 0 (scr 'extension-major-opcode "SHAPE"))
      (vector-set! dta 3 dstwin)
      (msg-handler 'wait-for-reply 
		   (scr 'scix-xas 3 fmt dta rest)
		   `((reply-name . ,(d-identity 'ShapeInputSelected))
		     (enabled    . ,d-bool)
		     (seq-nr     . ,d-card16)
		     (unused     . ,(d-unused 24)) )
		   scr) )))
    
;;; Request Shape-8: ShapeGetRectangles
(define send-ShapeGetRectangles
  (let ((fmt (vector a-request
		     a-card8 a-card16 a-window a-card8 a-card8 a-card16))
	(dta (vector #f 8 3 #f #f 0 0)) )
    (lambda (win srckind scr . rest)
      (vector-set! dta 0 (scr 'extension-major-opcode "SHAPE"))
      (vector-set! dta 3 win)
      (vector-set! dta 4 (lookup-constant srckind '((Bounding . 0)(Clip . 1))))
      (msg-handler 'wait-for-reply
		   (scr 'scix-xas 6 fmt dta rest)
		   `((reply-name   . ,(d-identity 'ShapeGetRectangles))
		     (ordering     . ,(d-const #t d-card8 '((0 . UnSorted)
							    (1 . YSorted)
							    (2 . YXSorted)
							    (3 . YXBanded) )))
		     (seq-nr       . ,d-card16)
		     (reply-length . ,d-card16)
		     (nrects       . ,d-card32)
		     (unused       . ,(d-unused 20))
		     (rectangles   . ,(lambda (str dpy)
					((d-list d-rectangle
						 (backtrack 'nrects))
					 str dpy))) )
		   scr) )))

(define (activate-shape-extension dpy)
  (if (not (dpy 'activate-extension
		"SHAPE"
		`(			; Event formats
;;; Event Shape #0: ShapeNotify
		  ((event-name . ,(d-identity 'ShapeNotify))
		   (shape-kind . ,(d-const #f d-card8 '((0 . Bounding)
							(1 . Clip) )))
		   (seq-nr             . ,d-card16)
		   (window             . ,d-window)
		   (x-value-of-extents . ,d-int16)
		   (y-value-of-extents . ,d-int16)
		   (width-of-extents   . ,d-card16)
		   (height-of-extents  . ,d-card16)
		   (server-time        . ,d-timestamp)
		   (shaped             . ,d-bool)
		   (unused             . ,(d-unused 11)) ))
		'()			; No new errors in the shape extension
		#f			; Currently no special extension object
		))
      (error 'activate-shape-extension
	     "SHAPE extension not present in server") ))