;;;
;;;              Copyright 1991 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;; This code bears no resemblance to the original pointer-tracker with a
;;; quit-button that it once was. Now it seems to be a game of fifteen.
;;; Johan Ihren, KTH

;;; $Id: fifteen.sc,v 2.3 91/09/15 01:02:02 johani Exp $

(module demofifteen (top-level demo-fifteen))

(include "../include/stox.sch")
(include "../include/scix.sch")

(include "../macros/extsyntax.sc")
(include "../macros/oos.sc")

;;; Note: A possible improvement would be to be able to move an entire
;;;       row of squares with one click the same way that it is done
;;;       with a real game. This shouldn't be hard to implement.

(define-class (fifteen-button . args)
  (locals
   (xpos (match-arg 'matrix-x 0 args))
   (ypos (match-arg 'matrix-y 0 args))
   (matrix (match-arg 'matrix 'no-default args))
   (my-number (string->number (match-arg 'label 0 args)))
   (s (match-arg 'button-space 0 args)) )
  (inherit (button (cons* 'width 32 'height 32 args)))
  (methods
   (xpos (lambda () xpos))
   (ypos (lambda () ypos))
   (new-position! (lambda (newx newy)
		    (set! xpos newx)
		    (set! ypos newy)
		    (me 'stox-fake-event 'leavenotify)
		    (me 'move (+ 1 (* s (- xpos 1))) (+ 1 (* s (- ypos 1))))
		    ))
   (at-home? (lambda (cols)
	       (= my-number (+ xpos (* cols (- ypos 1)))) )))
  (init
   (matrix-set! matrix xpos ypos inner-me)
   (me 'stox-activate) ))

(define-class (hole . args)
  (locals
   (scr ((match-arg 'parent 'no-default args) 'screen))
   (random (make-random-generator (vector-ref (my-rusage) 6))) ;; Why not...
   (matrix (match-arg 'matrix #f args))
   (scrambles (* 2 (vector-length matrix)
		 (vector-length (vector-ref matrix 1)) ))		 
   (xpos (match-arg 'matrix-x 0 args))
   (ypos (match-arg 'matrix-y 0 args))
   (s (match-arg 'button-space 0 args)) )
  (inherit (widget args))
  (methods
   (xpos (lambda () xpos))
   (ypos (lambda () ypos))
   (at-home? (lambda (cols) #t))
   (scramble (lambda ()
	       (let loop ((n scrambles) (last-neighbour #f))
		 (let* ((nb (find-neighbours matrix last-neighbour xpos ypos))
			(choice (list-ref nb
					  (floor (* (length nb) (random))) )))
		   (me 'new-position! (choice 'xpos) (choice 'ypos))
		   (if (positive? n) (loop (- n 1) choice))))))		       
   (new-position! (lambda (newx newy)
		    (if (procedure? (matrix-ref matrix newx newy))
			((matrix-ref matrix newx newy) 'new-position!
						       xpos ypos))
		    (matrix-set! matrix xpos ypos
				 (matrix-ref matrix newx newy))
		    (matrix-set! matrix newx newy me)
		    (set! xpos newx)
		    (set! ypos newy)
		    (me 'move (+ 1 (* s (- xpos 1))) (+ 1 (* s (- ypos 1))))
		    (scr 'flush!) ))))

(define (demo-fifteen cols rows scr)
  (let ((w (container 'make 'parent (scr 'root) 'title "SCIX Fifteen"))
	(gcache (graphics-cache 'make 'screen scr 'fontname "variable"))
	(matrix (list->vector (let loop ((rows rows) (ls '()))
				(if (= rows -1)
				    ls
				    (loop (- rows 1)
					  (cons (make-vector (+ cols 1) #t)
						ls)))))))
    (let* ((cleanup (lambda (e)
		      (set! matrix #f)
		      (w 'destroywindow)
		      (gcache 'free)
		      (scr 'flush!) ))
	   (btn-size 32) (btn-skip 2)
	   (btn-space (+ 2 btn-size btn-skip)) ; 2 is for border-width
	   (the-hole (hole 'make 'matrix matrix 'parent w 'width btn-size
			   'button-space btn-space
			   'height btn-size 'matrix-x cols 'matrix-y rows)) )
      (let* ((scramb (button 'make 'parent w 'label "Scramble"
			     'on-action (lambda (e) (the-hole 'scramble) )))
	     (quit (button 'make 'parent w 'label "Quit"
			   'on-action cleanup 'width (scramb 'width)))
	     (solve (button 'make 'parent w
			    'label "Solve*" 'width (scramb 'width)))
	     (action (lambda (e)
		       (let ((b (e 'event)))
			 (if (get-nearby-hole b the-hole)
			     (let ((xpos (b 'xpos)) (ypos (b 'ypos)))
			       (the-hole 'new-position! xpos ypos)
			       (if (and (= cols xpos)
					(= rows ypos)
					(found-solution? matrix) )
				   (display (format "Congratulations!~%")) ))
			     (scr 'bell 40) )
			 (scr 'flush!) ))))
	;; The following code is written this way to avoid the need for named
	;; bindings for every button.
	(let yloop ((y rows))
	  (let xloop ((x cols))
	    (let ((b (if (and (= x cols) (= y rows))
			 the-hole
			 (fifteen-button 'make 'parent w 'matrix matrix
					 'width btn-size 'height btn-size
					 'matrix-x x 'matrix-y y
					 'button-space btn-space
					 'label (number->string
						 (+ x (* (- y 1) cols)))
					 'on-action action  'gcache gcache))))
	      (b 'new-position! x y)
	      (cond ((> x 1) (xloop (- x 1)))
		    ((> y 1) (yloop (- y 1)))
		    (else
		     (let ((vsp (vskip btn-skip)))
		       (w 'set-box!
			  (hbox (vbox-to (* rows btn-space)
					 (vfill)  ; A box large enough to fit
					 (hbox-to ; all game-buttons.
					  (* cols btn-space)
					  (hfill)))
				(hfill) (hskip 1)
				(vbox (vskip 1)
				      quit vsp scramb vsp solve vsp (vss))
				(hskip 1)))))))))
	(map (lambda (o) (o 'stox-activate)) (list quit scramb solve the-hole))
	;;(the-hole 'scramble)
	(w 'mapwindow)
	(scr 'flush!)
	(scix-process-events scr) ))))

(define (matrix-ref m x y)
  (vector-ref (vector-ref m y) x) )

(define (matrix-set! m x y v)
  (vector-set! (vector-ref m y) x v) )

(define (get-nearby-hole b h)
  (if (or (and (= (abs (- (b 'xpos) (h 'xpos))) 1)
	       (= (abs (- (b 'ypos) (h 'ypos))) 0))
	  (and (= (abs (- (b 'xpos) (h 'xpos))) 0)
	       (= (abs (- (b 'ypos) (h 'ypos))) 1)) )
      (list (h 'xpos) (h 'ypos))
      #f))

(define (found-solution? m)
  (matrix-andmap (lambda (e)
		   (if (procedure? e)
		       (e 'at-home? (- (vector-length (vector-ref m 1)) 1))
		       #t))
		 m))

(define (matrix-andmap p m)
  (let ((ysize (- (vector-length m) 1))
	(xsize (- (vector-length (vector-ref m 1)) 1)) )
    (let loop ((r #t) (x xsize) (y ysize))
      (set! r (and r (p (matrix-ref m x y))))
      (cond ((not r) #f)
	    ((positive? x) (loop r (- x 1) y))
	    ((positive? y) (loop r xsize (- y 1)))
	    (else r)))))

(define (find-neighbours m ln x y)
  (let ((xsize (- (vector-length (vector-ref m 1)) 1))
	(ysize (- (vector-length m) 1)) )
    (remq ln
	  (remq #f (list (if (= x 1) #f (matrix-ref m (- x 1) y))
			 (if (= y 1) #f (matrix-ref m x (- y 1)))
			 (if (= x xsize) #f (matrix-ref m (+ x 1) y))
			 (if (= y ysize) #f (matrix-ref m x (+ y 1)))) ))))
		
;;; A simple pseudo-random number generator generator.
;;; Returns a psuedo-random number generator, using the given initial
;;; seed. The result is a number in [0, 1].
;;;
;;; The method used is Lehmer's method:
;;;    seed <- (5^(2p+1) * seed) mod 2^n.
;;; In this implementation, p is 5 and n is 26.

(define (make-random-generator seed)

  ;; Multiply p and q avoiding overflow, retaining only the lower
  ;; 26 bits of the result.
  (define (mult-low-bits p q)
    (let ((p1 (quotient p 8192))
	  (p0 (modulo p 8192))
	  (q1 (quotient q 8192))
	  (q0 (modulo q 8192)) )
      (modulo (+ (* (modulo (+ (* p0 q1) (* p1 q0)) 8192) 8192)
		 (* p0 q0) )
	      67108864)))

  (lambda ()
    (set! seed (mult-low-bits seed 48828125))
    (/ seed 67108863)))

