package Tsk::Iterator::Fs;
use 5.012005;
use strict;
use warnings;
use Tsk;
use Tsk::Stack;
use Tsk::Fs::File;
use Tsk::Fs::Dir;
use Tsk::Fs::Info ;
use Tsk::Img::Info ;
use Tsk::Vs::Info;
use Carp;
our $VERSION = '0.05';
=head1 NAME

Tsk::Iterator::Fs - Filesystem iterator

=head1 DESCRIPTION

This module finds all offsets of partitions in the image

=cut

sub new {
    my ($class,$image,$flags) = @_;
    $flags = $flags || ~($TSK_VS_PART_FLAG_META | $TSK_VS_PART_FLAG_UNALLOC);
    my $h = bless {
        tsk_fs_iterator => undef,
        volume_offsets  => undef,
        image_path      => $image,
        iter_idx        => 0,
        flags           => $flags,
    },$class;
    croak "[ERR] image path invalid" if !-f $image;
    $h->{volume_offsets} = $h->find_volume_offsets;
    return $h;
}

sub reset {
    my ($self) = @_;
    $self->{iter_idx} = 0;
}

sub next_offset {
    my ($self) = @_;
    my $volume_count = scalar(@{$self->{volume_offsets}});
    if($self->{iter_idx} < $volume_count){
        my $idx       = $self->{iter_idx};
        my $offset    = $self->{volume_offsets}->[$idx];
        $self->{iter_idx}++;
        return $offset;
    };
    return undef;
}

=head1 next_offset()

Return the next volume offset.

=head1 find_volumes_offsets()

Finds volume offsets inside the disk image.

=cut

sub find_volume_offsets {
    my ($self) = @_;
    my $img_info = Tsk::Img::Info->new();
    $img_info->open($self->{image_path},$TSK_IMG_TYPE_DETECT,0);

    my $vs_info = Tsk::Vs::Info->new();
    my @volume_offsets = ();
    if( ($vs_info->open($img_info,0, $TSK_VS_TYPE_DETECT)) != 1) {
        my $partCount = $vs_info->getPartCount();
        for(my $i = 0; $i < $partCount ; $i++) {
            my $vs_part = $vs_info->getPart($i);
            my $flags = $vs_part->getFlags();
            next unless $flags & $self->{flags};
            my $offset = $vs_part->getStart() * $vs_info->getBlockSize();
            push @volume_offsets, $offset;
        };
    } else {
        # it's not a multi-volume image, so assume there's just one volume
        # starting at offset 0 
        @volume_offsets = (0);
    };
    return \@volume_offsets;
}


=head1 BUGS

L<https://rt.cpan.org/Public/Bug/Report.html?Queue=Tsk>

L<https://github.com/wsdookadr/Tsk-XS/issues>

=head1 AUTHOR

Stefan Petrea, C<< <stefan at garage-coding.com> >>

=cut

1;
