# Makefile.PL for PGP::Sign module  -*- perl -*-
# $Id: Makefile.PL,v 0.8 1999/02/10 08:37:21 eagle Exp $
use ExtUtils::MakeMaker;

# The build process for PGP::Sign normally will attempt to find programs to
# use for signature generation and verification and will set PGPSTYLE based
# on what programs are found.  We want to allow people to override that
# earlier and without requiring interaction, though, so we allow four
# additional command line options.  We have to pull them out of @ARGV before
# handing things over to WriteMakefile(), since otherwise MakeMaker will
# complain about them.
#
# PGP sets both PGPS and PGPV to the same value; PGPS and PGPV can also be
# set seperately for separate signing and verification programs.
my ($pgps, $pgpv, $pgpstyle);
@ARGV = map {
    if    (/^PGP=(.*)/)      { $pgps = $pgpv = $1; () }
    elsif (/^PGPS=(.*)/)     { $pgps = $1;         () }
    elsif (/^PGPV=(.*)/)     { $pgpv = $1;         () }
    elsif (/^PGPSTYLE=(.*)/) { $pgpstyle = $1;     () }
    else                     {                     $_ }
} @ARGV;
if ($pgps && !$pgpv) {
    die "Must set PGPV if PGPS is set\n";
} elsif ($pgpv && !$pgps) {
    die "Must set PGPS if PGPV is set\n";
}

# If either PGP or PGPS and PGPV are set, and PGPSTYLE isn't, we try to
# figure out PGPSTYLE from the program names.  We just base this on the name
# of the executable used for signing, which should be reasonably reliable.
# If this check fails, people can always set it themselves.
if ($pgps && !$pgpstyle) {
    if ($pgps =~ m%pgps[^/]*$%)   { $pgpstyle = 'PGP5' }
    elsif ($pgps =~ m%gpg[^/]*$%) { $pgpstyle = 'GPG'  }
    else                          { $pgpstyle = 'PGP2' }
}

# Paranoia.
for ($pgps, $pgpv, $pgpstyle) { s/\\/\\\\/g; s/\'/\\\'/g }

# If any of these are set, write out the results to paths for makepm.PL to
# pick up.
if ($pgps) {
    open (PATHS, '> paths') or die "Can't create file 'paths': $!\n";
    print PATHS "# Automatically generated PGP::Sign configuration\n";
    print PATHS "\$PGPS = '$pgps';\n";
    print PATHS "\$PGPV = '$pgpv';\n";
    print PATHS "\$PGPSTYLE = '$pgpstyle';\n";
    close PATHS;
}

# Now actually build the Makefile.
WriteMakefile (
    NAME         => 'PGP::Sign',
    DISTNAME     => 'PGP-Sign',
    ($] >= 5.005 ?
     (ABSTRACT   => 'Create and check PGP signatures, securely',
      AUTHOR     => 'Russ Allbery (rra@stanford.edu)') : ()),
    MAN3PODS     => { 'PGP/Sign.pm' => '$(INST_MAN3DIR)/PGP::Sign.3' },
    PL_FILES     => { 'makepm.PL' => 'PGP/Sign.pm' },
    PM           => { 'PGP/Sign.pm' => '$(INST_LIBDIR)/Sign.pm' },
    VERSION_FROM => 'VERSION.pm',
    dist         => { COMPRESS => 'gzip', SUFFIX => 'gz' },
    realclean    => { FILES => 'PGP/Sign.pm data/randseed.bin paths' }
);
