package PDF::Builder::Resource::Glyphs;

use strict;
use warnings;

our $VERSION = '3.012'; # VERSION
my $LAST_UPDATE = '3.010'; # manually update whenever code is changed

=head1 NAME

PDF::Builder::Resource::Glyphs - preparsed uniglyph.txt file information

=cut

# The data in this file is generated from uniglyph.txt.  In earlier
# versions of PDF::API2, this file was parsed and sorted every time
# the module was loaded, which was inefficient.
#
# There's probably a way to remove this information from the module
# altogether, but in the meantime, the file now only needs to be
# parsed when it changes.
#
# If uniglyph.txt changes, use _generate to create new hashes, and
# paste the output into this file.

sub _generate {
    my $uniglyph_file = shift;
    die 'uniglyph.txt not found' unless -f $uniglyph_file;

    my $fh;
    open $fh, '<', $uniglyph_file;
    my $uuu = {};
    while (my $line = <$fh>) {
        next if $line =~ m|^#|;
        chomp $line;
        $line =~ s|\s+\#.+$||go;
        my ($uni, $name, $prio) = split(/\s+;\s+/, $line);
        $uni = hex($uni);
        $uuu->{'u'}->{$uni} ||= [];
        $uuu->{'g'}->{$name} ||= [];
        push @{$uuu->{'u'}->{$uni}},  { uni => $uni, name => $name, prio => $prio };
        push @{$uuu->{'g'}->{$name}}, { uni => $uni, name => $name, prio => $prio };
    }
    close($fh);

    my %u2n;
    foreach my $k (keys %{$uuu->{'u'}}) {
        $u2n{$k} = $uuu->{'u'}->{$k}->[0]->{'name'};
    }
    print "our \$u2n = {\n";
    foreach my $key (sort { $a <=> $b } keys %u2n) {
        print "    '$key' => '$u2n{$key}',\n";
    }
    print "};\n\n";

    my %n2u;
    foreach my $k (keys %{$uuu->{'g'}}) {
        my ($r) = sort { $a->{'prio'} <=> $b->{'prio'}} @{$uuu->{'g'}->{$k}};
        $n2u{$k} = $r->{'uni'};
    }
    print "our \$n2u = {\n";
    foreach my $key (sort { $a cmp $b } keys %n2u) {
        print "    '$key' => '$n2u{$key}',\n";
    }
    print "};\n\n";
    return;
}

# pasted in whenever update needed. approximately 9716 entries.
our $u2n = {
    '0' => '.notdef',
    '1' => 'acute',
    '2' => 'caron',
    '3' => 'circumflex',
    '4' => 'dieresis',
    '5' => 'grave',
    '6' => 'macron',
    '7' => 'ring',
    '8' => 'tilde',
    '9' => 'breve',
    '10' => 'ogonek',
    '11' => 'dotaccent',
    '12' => 'hungarumlaut',
    '13' => 'cedilla',
    '14' => 'dblgrave',
    '15' => 'controlSI',
    '16' => 'controlDLE',
    '17' => 'controlDC1',
    '18' => 'controlDC2',
    '19' => 'controlDC3',
    '20' => 'controlDC4',
    '21' => 'controlNAK',
    '22' => 'controlSYN',
    '23' => 'controlETB',
    '24' => 'controlCAN',
    '25' => 'controlEM',
    '26' => 'controlSUB',
    '27' => 'controlESC',
    '28' => 'controlFS',
    '29' => 'controlGS',
    '30' => 'dotlessi',
    '31' => 'dotlessj',
    '32' => 'space',
    '33' => 'exclam',
    '34' => 'quotedbl',
    '35' => 'numbersign',
    '36' => 'dollar',
    '37' => 'percent',
    '38' => 'ampersand',
    '39' => 'quotesingle',
    '40' => 'parenleft',
    '41' => 'parenright',
    '42' => 'asterisk',
    '43' => 'plus',
    '44' => 'comma',
    '45' => 'hyphen',
    '46' => 'period',
    '47' => 'slash',
    '48' => 'zero',
    '49' => 'one',
    '50' => 'two',
    '51' => 'three',
    '52' => 'four',
    '53' => 'five',
    '54' => 'six',
    '55' => 'seven',
    '56' => 'eight',
    '57' => 'nine',
    '58' => 'colon',
    '59' => 'semicolon',
    '60' => 'less',
    '61' => 'equal',
    '62' => 'greater',
    '63' => 'question',
    '64' => 'at',
    '65' => 'A',
    '66' => 'B',
    '67' => 'C',
    '68' => 'D',
    '69' => 'E',
    '70' => 'F',
    '71' => 'G',
    '72' => 'H',
    '73' => 'I',
    '74' => 'J',
    '75' => 'K',
    '76' => 'L',
    '77' => 'M',
    '78' => 'N',
    '79' => 'O',
    '80' => 'P',
    '81' => 'Q',
    '82' => 'R',
    '83' => 'S',
    '84' => 'T',
    '85' => 'U',
    '86' => 'V',
    '87' => 'W',
    '88' => 'X',
    '89' => 'Y',
    '90' => 'Z',
    '91' => 'bracketleft',
    '92' => 'backslash',
    '93' => 'bracketright',
    '94' => 'asciicircum',
    '95' => 'underscore',
    '96' => 'grave',
    '97' => 'a',
    '98' => 'b',
    '99' => 'c',
    '100' => 'd',
    '101' => 'e',
    '102' => 'f',
    '103' => 'g',
    '104' => 'h',
    '105' => 'i',
    '106' => 'j',
    '107' => 'k',
    '108' => 'l',
    '109' => 'm',
    '110' => 'n',
    '111' => 'o',
    '112' => 'p',
    '113' => 'q',
    '114' => 'r',
    '115' => 's',
    '116' => 't',
    '117' => 'u',
    '118' => 'v',
    '119' => 'w',
    '120' => 'x',
    '121' => 'y',
    '122' => 'z',
    '123' => 'braceleft',
    '124' => 'bar',
    '125' => 'braceright',
    '126' => 'asciitilde',
    '127' => 'controlDEL',
    '128' => 'Euro',
    '129' => 'bullet',
    '130' => 'quotesinglbase',
    '131' => 'florin',
    '132' => 'quotedblbase',
    '133' => 'ellipsis',
    '134' => 'dagger',
    '135' => 'daggerdbl',
    '136' => 'circumflex',
    '137' => 'perthousand',
    '138' => 'Scaron',
    '139' => 'guilsinglleft',
    '140' => 'OE',
    '141' => 'bullet',
    '142' => 'Zcaron',
    '143' => 'bullet',
    '144' => 'bullet',
    '145' => 'quoteleft',
    '146' => 'quoteright',
    '147' => 'quotedblleft',
    '148' => 'quotedblright',
    '149' => 'bullet',
    '150' => 'endash',
    '151' => 'emdash',
    '152' => 'tilde',
    '153' => 'trademark',
    '154' => 'scaron',
    '155' => 'guilsinglright',
    '156' => 'oe',
    '157' => 'bullet',
    '158' => 'zcaron',
    '159' => 'Ydieresis',
    '160' => 'space',
    '161' => 'exclamdown',
    '162' => 'cent',
    '163' => 'sterling',
    '164' => 'currency',
    '165' => 'yen',
    '166' => 'brokenbar',
    '167' => 'section',
    '168' => 'dieresis',
    '169' => 'copyright',
    '170' => 'ordfeminine',
    '171' => 'guillemotleft',
    '172' => 'logicalnot',
    '173' => 'hyphen',
    '174' => 'registered',
    '175' => 'macron',
    '176' => 'degree',
    '177' => 'plusminus',
    '178' => 'twosuperior',
    '179' => 'threesuperior',
    '180' => 'acute',
    '181' => 'mu',
    '182' => 'paragraph',
    '183' => 'periodcentered',
    '184' => 'cedilla',
    '185' => 'onesuperior',
    '186' => 'ordmasculine',
    '187' => 'guillemotright',
    '188' => 'onequarter',
    '189' => 'onehalf',
    '190' => 'threequarters',
    '191' => 'questiondown',
    '192' => 'Agrave',
    '193' => 'Aacute',
    '194' => 'Acircumflex',
    '195' => 'Atilde',
    '196' => 'Adieresis',
    '197' => 'Aring',
    '198' => 'AE',
    '199' => 'Ccedilla',
    '200' => 'Egrave',
    '201' => 'Eacute',
    '202' => 'Ecircumflex',
    '203' => 'Edieresis',
    '204' => 'Igrave',
    '205' => 'Iacute',
    '206' => 'Icircumflex',
    '207' => 'Idieresis',
    '208' => 'Eth',
    '209' => 'Ntilde',
    '210' => 'Ograve',
    '211' => 'Oacute',
    '212' => 'Ocircumflex',
    '213' => 'Otilde',
    '214' => 'Odieresis',
    '215' => 'multiply',
    '216' => 'Oslash',
    '217' => 'Ugrave',
    '218' => 'Uacute',
    '219' => 'Ucircumflex',
    '220' => 'Udieresis',
    '221' => 'Yacute',
    '222' => 'Thorn',
    '223' => 'germandbls',
    '224' => 'agrave',
    '225' => 'aacute',
    '226' => 'acircumflex',
    '227' => 'atilde',
    '228' => 'adieresis',
    '229' => 'aring',
    '230' => 'ae',
    '231' => 'ccedilla',
    '232' => 'egrave',
    '233' => 'eacute',
    '234' => 'ecircumflex',
    '235' => 'edieresis',
    '236' => 'igrave',
    '237' => 'iacute',
    '238' => 'icircumflex',
    '239' => 'idieresis',
    '240' => 'eth',
    '241' => 'ntilde',
    '242' => 'ograve',
    '243' => 'oacute',
    '244' => 'ocircumflex',
    '245' => 'otilde',
    '246' => 'odieresis',
    '247' => 'divide',
    '248' => 'oslash',
    '249' => 'ugrave',
    '250' => 'uacute',
    '251' => 'ucircumflex',
    '252' => 'udieresis',
    '253' => 'yacute',
    '254' => 'thorn',
    '255' => 'ydieresis',
    '256' => 'Amacron',
    '257' => 'amacron',
    '258' => 'Abreve',
    '259' => 'abreve',
    '260' => 'Aogonek',
    '261' => 'aogonek',
    '262' => 'Cacute',
    '263' => 'cacute',
    '264' => 'Ccircumflex',
    '265' => 'ccircumflex',
    '266' => 'Cdotaccent',
    '267' => 'cdotaccent',
    '268' => 'Ccaron',
    '269' => 'ccaron',
    '270' => 'Dcaron',
    '271' => 'dcaron',
    '272' => 'Dcroat',
    '273' => 'dcroat',
    '274' => 'Emacron',
    '275' => 'emacron',
    '276' => 'Ebreve',
    '277' => 'ebreve',
    '278' => 'Edotaccent',
    '279' => 'edotaccent',
    '280' => 'Eogonek',
    '281' => 'eogonek',
    '282' => 'Ecaron',
    '283' => 'ecaron',
    '284' => 'Gcircumflex',
    '285' => 'gcircumflex',
    '286' => 'Gbreve',
    '287' => 'gbreve',
    '288' => 'Gdotaccent',
    '289' => 'gdotaccent',
    '290' => 'Gcommaaccent',
    '291' => 'gcommaaccent',
    '292' => 'Hcircumflex',
    '293' => 'hcircumflex',
    '294' => 'Hbar',
    '295' => 'hbar',
    '296' => 'Itilde',
    '297' => 'itilde',
    '298' => 'Imacron',
    '299' => 'imacron',
    '300' => 'Ibreve',
    '301' => 'ibreve',
    '302' => 'Iogonek',
    '303' => 'iogonek',
    '304' => 'Idotaccent',
    '305' => 'dotlessi',
    '306' => 'IJ',
    '307' => 'ij',
    '308' => 'Jcircumflex',
    '309' => 'jcircumflex',
    '310' => 'Kcommaaccent',
    '311' => 'kcommaaccent',
    '312' => 'kgreenlandic',
    '313' => 'Lacute',
    '314' => 'lacute',
    '315' => 'Lcommaaccent',
    '316' => 'lcommaaccent',
    '317' => 'Lcaron',
    '318' => 'lcaron',
    '319' => 'Ldot',
    '320' => 'ldot',
    '321' => 'Lslash',
    '322' => 'lslash',
    '323' => 'Nacute',
    '324' => 'nacute',
    '325' => 'Ncommaaccent',
    '326' => 'ncommaaccent',
    '327' => 'Ncaron',
    '328' => 'ncaron',
    '329' => 'napostrophe',
    '330' => 'Eng',
    '331' => 'eng',
    '332' => 'Omacron',
    '333' => 'omacron',
    '334' => 'Obreve',
    '335' => 'obreve',
    '336' => 'Ohungarumlaut',
    '337' => 'ohungarumlaut',
    '338' => 'OE',
    '339' => 'oe',
    '340' => 'Racute',
    '341' => 'racute',
    '342' => 'Rcommaaccent',
    '343' => 'rcommaaccent',
    '344' => 'Rcaron',
    '345' => 'rcaron',
    '346' => 'Sacute',
    '347' => 'sacute',
    '348' => 'Scircumflex',
    '349' => 'scircumflex',
    '350' => 'Scedilla',
    '351' => 'scedilla',
    '352' => 'Scaron',
    '353' => 'scaron',
    '354' => 'Tcedilla',
    '355' => 'tcedilla',
    '356' => 'Tcaron',
    '357' => 'tcaron',
    '358' => 'Tbar',
    '359' => 'tbar',
    '360' => 'Utilde',
    '361' => 'utilde',
    '362' => 'Umacron',
    '363' => 'umacron',
    '364' => 'Ubreve',
    '365' => 'ubreve',
    '366' => 'Uring',
    '367' => 'uring',
    '368' => 'Uhungarumlaut',
    '369' => 'uhungarumlaut',
    '370' => 'Uogonek',
    '371' => 'uogonek',
    '372' => 'Wcircumflex',
    '373' => 'wcircumflex',
    '374' => 'Ycircumflex',
    '375' => 'ycircumflex',
    '376' => 'Ydieresis',
    '377' => 'Zacute',
    '378' => 'zacute',
    '379' => 'Zdotaccent',
    '380' => 'zdotaccent',
    '381' => 'Zcaron',
    '382' => 'zcaron',
    '383' => 'longs',
    '384' => 'bstroke',
    '385' => 'Bhook',
    '386' => 'Btopbar',
    '387' => 'btopbar',
    '388' => 'Tonesix',
    '389' => 'tonesix',
    '390' => 'Oopen',
    '391' => 'Chook',
    '392' => 'chook',
    '393' => 'Dafrican',
    '394' => 'Dhook',
    '395' => 'Dtopbar',
    '396' => 'dtopbar',
    '397' => 'deltaturned',
    '398' => 'Ereversed',
    '399' => 'Schwa',
    '400' => 'Eopen',
    '401' => 'Fhook',
    '402' => 'florin',
    '403' => 'Ghook',
    '404' => 'Gammaafrican',
    '405' => 'hv',
    '406' => 'Iotaafrican',
    '407' => 'Istroke',
    '408' => 'Khook',
    '409' => 'khook',
    '410' => 'lbar',
    '411' => 'lambdastroke',
    '412' => 'Mturned',
    '413' => 'Nhookleft',
    '414' => 'nlegrightlong',
    '415' => 'Ocenteredtilde',
    '416' => 'Ohorn',
    '417' => 'ohorn',
    '418' => 'Oi',
    '419' => 'oi',
    '420' => 'Phook',
    '421' => 'phook',
    '422' => 'yr',
    '423' => 'Tonetwo',
    '424' => 'tonetwo',
    '425' => 'Esh',
    '426' => 'eshreversedloop',
    '427' => 'tpalatalhook',
    '428' => 'Thook',
    '429' => 'thook',
    '430' => 'Tretroflexhook',
    '431' => 'Uhorn',
    '432' => 'uhorn',
    '433' => 'Upsilonafrican',
    '434' => 'Vhook',
    '435' => 'Yhook',
    '436' => 'yhook',
    '437' => 'Zstroke',
    '438' => 'zstroke',
    '439' => 'Ezh',
    '440' => 'Ezhreversed',
    '441' => 'ezhreversed',
    '442' => 'ezhtail',
    '443' => 'twostroke',
    '444' => 'Tonefive',
    '445' => 'tonefive',
    '446' => 'glottalinvertedstroke',
    '447' => 'wynn',
    '448' => 'clickdental',
    '449' => 'clicklateral',
    '450' => 'clickalveolar',
    '451' => 'clickretroflex',
    '452' => 'DZcaron',
    '453' => 'Dzcaron',
    '454' => 'dzcaron',
    '455' => 'LJ',
    '456' => 'Lj',
    '457' => 'lj',
    '458' => 'NJ',
    '459' => 'Nj',
    '460' => 'nj',
    '461' => 'Acaron',
    '462' => 'acaron',
    '463' => 'Icaron',
    '464' => 'icaron',
    '465' => 'Ocaron',
    '466' => 'ocaron',
    '467' => 'Ucaron',
    '468' => 'ucaron',
    '469' => 'Udieresismacron',
    '470' => 'udieresismacron',
    '471' => 'Udieresisacute',
    '472' => 'udieresisacute',
    '473' => 'Udieresiscaron',
    '474' => 'udieresiscaron',
    '475' => 'Udieresisgrave',
    '476' => 'udieresisgrave',
    '477' => 'eturned',
    '478' => 'Adieresismacron',
    '479' => 'adieresismacron',
    '480' => 'Adotmacron',
    '481' => 'adotmacron',
    '482' => 'AEmacron',
    '483' => 'aemacron',
    '484' => 'Gstroke',
    '485' => 'gstroke',
    '486' => 'Gcaron',
    '487' => 'gcaron',
    '488' => 'Kcaron',
    '489' => 'kcaron',
    '490' => 'Oogonek',
    '491' => 'oogonek',
    '492' => 'Oogonekmacron',
    '493' => 'oogonekmacron',
    '494' => 'Ezhcaron',
    '495' => 'ezhcaron',
    '496' => 'jcaron',
    '497' => 'DZ',
    '498' => 'Dz',
    '499' => 'dz',
    '500' => 'Gacute',
    '501' => 'gacute',
    '506' => 'Aringacute',
    '507' => 'aringacute',
    '508' => 'AEacute',
    '509' => 'aeacute',
    '510' => 'Oslashacute',
    '511' => 'oslashacute',
    '512' => 'Adblgrave',
    '513' => 'adblgrave',
    '514' => 'Ainvertedbreve',
    '515' => 'ainvertedbreve',
    '516' => 'Edblgrave',
    '517' => 'edblgrave',
    '518' => 'Einvertedbreve',
    '519' => 'einvertedbreve',
    '520' => 'Idblgrave',
    '521' => 'idblgrave',
    '522' => 'Iinvertedbreve',
    '523' => 'iinvertedbreve',
    '524' => 'Odblgrave',
    '525' => 'odblgrave',
    '526' => 'Oinvertedbreve',
    '527' => 'oinvertedbreve',
    '528' => 'Rdblgrave',
    '529' => 'rdblgrave',
    '530' => 'Rinvertedbreve',
    '531' => 'rinvertedbreve',
    '532' => 'Udblgrave',
    '533' => 'udblgrave',
    '534' => 'Uinvertedbreve',
    '535' => 'uinvertedbreve',
    '536' => 'Scommaaccent',
    '537' => 'scommaaccent',
    '538' => 'Tcommaaccent',
    '539' => 'tcommaaccent',
    '592' => 'aturned',
    '593' => 'ascript',
    '594' => 'ascriptturned',
    '595' => 'bhook',
    '596' => 'oopen',
    '597' => 'ccurl',
    '598' => 'dtail',
    '599' => 'dhook',
    '600' => 'ereversed',
    '601' => 'schwa',
    '602' => 'schwahook',
    '603' => 'eopen',
    '604' => 'eopenreversed',
    '605' => 'eopenreversedhook',
    '606' => 'eopenreversedclosed',
    '607' => 'jdotlessstroke',
    '608' => 'ghook',
    '609' => 'gscript',
    '611' => 'gammalatinsmall',
    '612' => 'ramshorn',
    '613' => 'hturned',
    '614' => 'hhook',
    '615' => 'henghook',
    '616' => 'istroke',
    '617' => 'iotalatin',
    '619' => 'lmiddletilde',
    '620' => 'lbelt',
    '621' => 'lhookretroflex',
    '622' => 'lezh',
    '623' => 'mturned',
    '624' => 'mlonglegturned',
    '625' => 'mhook',
    '626' => 'nhookleft',
    '627' => 'nhookretroflex',
    '629' => 'obarred',
    '631' => 'omegalatinclosed',
    '632' => 'philatin',
    '633' => 'rturned',
    '634' => 'rlonglegturned',
    '635' => 'rhookturned',
    '636' => 'rlongleg',
    '637' => 'rhook',
    '638' => 'rfishhook',
    '639' => 'rfishhookreversed',
    '641' => 'Rsmallinverted',
    '642' => 'shook',
    '643' => 'esh',
    '644' => 'dotlessjstrokehook',
    '645' => 'eshsquatreversed',
    '646' => 'eshcurl',
    '647' => 'tturned',
    '648' => 'tretroflexhook',
    '649' => 'ubar',
    '650' => 'upsilonlatin',
    '651' => 'vhook',
    '652' => 'vturned',
    '653' => 'wturned',
    '654' => 'yturned',
    '656' => 'zretroflexhook',
    '657' => 'zcurl',
    '658' => 'ezh',
    '659' => 'ezhcurl',
    '660' => 'glottalstop',
    '661' => 'glottalstopreversed',
    '662' => 'glottalstopinverted',
    '663' => 'cstretched',
    '664' => 'bilabialclick',
    '666' => 'eopenclosed',
    '667' => 'Gsmallhook',
    '669' => 'jcrossedtail',
    '670' => 'kturned',
    '672' => 'qhook',
    '673' => 'glottalstopstroke',
    '674' => 'glottalstopstrokereversed',
    '675' => 'dzaltone',
    '676' => 'dezh',
    '677' => 'dzcurl',
    '678' => 'ts',
    '679' => 'tesh',
    '680' => 'tccurl',
    '688' => 'hsuperior',
    '689' => 'hhooksuperior',
    '690' => 'jsuperior',
    '692' => 'rturnedsuperior',
    '693' => 'rhookturnedsuperior',
    '694' => 'Rsmallinvertedsuperior',
    '695' => 'wsuperior',
    '696' => 'ysuperior',
    '697' => 'primemod',
    '698' => 'dblprimemod',
    '699' => 'commaturnedmod',
    '700' => 'afii57929',
    '701' => 'afii64937',
    '702' => 'ringhalfright',
    '703' => 'ringhalfleft',
    '704' => 'glottalstopmod',
    '705' => 'glottalstopreversedmod',
    '706' => 'arrowheadleftmod',
    '707' => 'arrowheadrightmod',
    '708' => 'arrowheadupmod',
    '709' => 'arrowheaddownmod',
    '710' => 'circumflex',
    '711' => 'caron',
    '712' => 'verticallinemod',
    '713' => 'macron',
    '714' => 'secondtonechinese',
    '715' => 'fourthtonechinese',
    '716' => 'verticallinelowmod',
    '717' => 'macronlowmod',
    '718' => 'gravelowmod',
    '719' => 'acutelowmod',
    '720' => 'colontriangularmod',
    '721' => 'colontriangularhalfmod',
    '722' => 'ringhalfrightcentered',
    '723' => 'ringhalfleftcentered',
    '724' => 'uptackmod',
    '725' => 'downtackmod',
    '726' => 'plusmod',
    '727' => 'minusmod',
    '728' => 'breve',
    '729' => 'dotaccent',
    '730' => 'ring',
    '731' => 'ogonek',
    '732' => 'tilde',
    '733' => 'hungarumlaut',
    '734' => 'rhotichookmod',
    '736' => 'gammasuperior',
    '739' => 'xsuperior',
    '740' => 'glottalstopreversedsuperior',
    '741' => 'tonebarextrahighmod',
    '742' => 'tonebarhighmod',
    '743' => 'tonebarmidmod',
    '744' => 'tonebarlowmod',
    '745' => 'tonebarextralowmod',
    '768' => 'gravecomb',
    '769' => 'acutecomb',
    '770' => 'circumflexcmb',
    '771' => 'tildecomb',
    '772' => 'macroncmb',
    '773' => 'overlinecmb',
    '774' => 'brevecmb',
    '775' => 'dotaccentcmb',
    '776' => 'dieresiscmb',
    '777' => 'hookabovecomb',
    '778' => 'ringcmb',
    '779' => 'hungarumlautcmb',
    '780' => 'caroncmb',
    '781' => 'verticallineabovecmb',
    '782' => 'dblverticallineabovecmb',
    '783' => 'dblgravecmb',
    '784' => 'candrabinducmb',
    '785' => 'breveinvertedcmb',
    '786' => 'commaturnedabovecmb',
    '787' => 'commaabovecmb',
    '788' => 'commareversedabovecmb',
    '789' => 'commaaboverightcmb',
    '790' => 'gravebelowcmb',
    '791' => 'acutebelowcmb',
    '792' => 'lefttackbelowcmb',
    '793' => 'righttackbelowcmb',
    '794' => 'leftangleabovecmb',
    '795' => 'horncmb',
    '796' => 'ringhalfleftbelowcmb',
    '797' => 'uptackbelowcmb',
    '798' => 'downtackbelowcmb',
    '799' => 'plusbelowcmb',
    '800' => 'minusbelowcmb',
    '801' => 'hookpalatalizedbelowcmb',
    '802' => 'hookretroflexbelowcmb',
    '803' => 'dotbelowcomb',
    '804' => 'dieresisbelowcmb',
    '805' => 'ringbelowcmb',
    '807' => 'cedillacmb',
    '808' => 'ogonekcmb',
    '809' => 'verticallinebelowcmb',
    '810' => 'bridgebelowcmb',
    '811' => 'dblarchinvertedbelowcmb',
    '812' => 'caronbelowcmb',
    '813' => 'circumflexbelowcmb',
    '814' => 'brevebelowcmb',
    '815' => 'breveinvertedbelowcmb',
    '816' => 'tildebelowcmb',
    '817' => 'macronbelowcmb',
    '818' => 'lowlinecmb',
    '819' => 'dbllowlinecmb',
    '820' => 'tildeoverlaycmb',
    '821' => 'strokeshortoverlaycmb',
    '822' => 'strokelongoverlaycmb',
    '823' => 'solidusshortoverlaycmb',
    '824' => 'soliduslongoverlaycmb',
    '825' => 'ringhalfrightbelowcmb',
    '826' => 'bridgeinvertedbelowcmb',
    '827' => 'squarebelowcmb',
    '828' => 'seagullbelowcmb',
    '829' => 'xabovecmb',
    '830' => 'tildeverticalcmb',
    '831' => 'dbloverlinecmb',
    '832' => 'gravetonecmb',
    '833' => 'acutetonecmb',
    '834' => 'perispomenigreekcmb',
    '835' => 'koroniscmb',
    '836' => 'dialytikatonoscmb',
    '837' => 'ypogegrammenigreekcmb',
    '864' => 'tildedoublecmb',
    '865' => 'breveinverteddoublecmb',
    '884' => 'numeralsigngreek',
    '885' => 'numeralsignlowergreek',
    '890' => 'ypogegrammeni',
    '894' => 'questiongreek',
    '900' => 'tonos',
    '901' => 'dieresistonos',
    '902' => 'Alphatonos',
    '903' => 'anoteleia',
    '904' => 'Epsilontonos',
    '905' => 'Etatonos',
    '906' => 'Iotatonos',
    '908' => 'Omicrontonos',
    '910' => 'Upsilontonos',
    '911' => 'Omegatonos',
    '912' => 'iotadieresistonos',
    '913' => 'Alpha',
    '914' => 'Beta',
    '915' => 'Gamma',
    '916' => 'Delta',
    '917' => 'Epsilon',
    '918' => 'Zeta',
    '919' => 'Eta',
    '920' => 'Theta',
    '921' => 'Iota',
    '922' => 'Kappa',
    '923' => 'Lambda',
    '924' => 'Mu',
    '925' => 'Nu',
    '926' => 'Xi',
    '927' => 'Omicron',
    '928' => 'Pi',
    '929' => 'Rho',
    '931' => 'Sigma',
    '932' => 'Tau',
    '933' => 'Upsilon',
    '934' => 'Phi',
    '935' => 'Chi',
    '936' => 'Psi',
    '937' => 'Omega',
    '938' => 'Iotadieresis',
    '939' => 'Upsilondieresis',
    '940' => 'alphatonos',
    '941' => 'epsilontonos',
    '942' => 'etatonos',
    '943' => 'iotatonos',
    '944' => 'upsilondieresistonos',
    '945' => 'alpha',
    '946' => 'beta',
    '947' => 'gamma',
    '948' => 'delta',
    '949' => 'epsilon',
    '950' => 'zeta',
    '951' => 'eta',
    '952' => 'theta',
    '953' => 'iota',
    '954' => 'kappa',
    '955' => 'lambda',
    '956' => 'mu',
    '957' => 'nu',
    '958' => 'xi',
    '959' => 'omicron',
    '960' => 'pi',
    '961' => 'rho',
    '962' => 'sigma1',
    '963' => 'sigma',
    '964' => 'tau',
    '965' => 'upsilon',
    '966' => 'phi',
    '967' => 'chi',
    '968' => 'psi',
    '969' => 'omega',
    '970' => 'iotadieresis',
    '971' => 'upsilondieresis',
    '972' => 'omicrontonos',
    '973' => 'upsilontonos',
    '974' => 'omegatonos',
    '976' => 'betasymbolgreek',
    '977' => 'theta1',
    '978' => 'Upsilon1',
    '979' => 'Upsilonacutehooksymbolgreek',
    '980' => 'Upsilondieresishooksymbolgreek',
    '981' => 'phi1',
    '982' => 'omega1',
    '986' => 'Stigmagreek',
    '988' => 'Digammagreek',
    '989' => 'gammad',
    '990' => 'Koppagreek',
    '992' => 'Sampigreek',
    '994' => 'Sheicoptic',
    '995' => 'sheicoptic',
    '996' => 'Feicoptic',
    '997' => 'feicoptic',
    '998' => 'Kheicoptic',
    '999' => 'kheicoptic',
    '1000' => 'Horicoptic',
    '1001' => 'horicoptic',
    '1002' => 'Gangiacoptic',
    '1003' => 'gangiacoptic',
    '1004' => 'Shimacoptic',
    '1005' => 'shimacoptic',
    '1006' => 'Deicoptic',
    '1007' => 'deicoptic',
    '1008' => 'kappasymbolgreek',
    '1009' => 'rhosymbolgreek',
    '1010' => 'sigmalunatesymbolgreek',
    '1011' => 'yotgreek',
    '1025' => 'afii10023',
    '1026' => 'afii10051',
    '1027' => 'afii10052',
    '1028' => 'afii10053',
    '1029' => 'afii10054',
    '1030' => 'afii10055',
    '1031' => 'afii10056',
    '1032' => 'afii10057',
    '1033' => 'afii10058',
    '1034' => 'afii10059',
    '1035' => 'afii10060',
    '1036' => 'afii10061',
    '1038' => 'afii10062',
    '1039' => 'afii10145',
    '1040' => 'afii10017',
    '1041' => 'afii10018',
    '1042' => 'afii10019',
    '1043' => 'afii10020',
    '1044' => 'afii10021',
    '1045' => 'afii10022',
    '1046' => 'afii10024',
    '1047' => 'afii10025',
    '1048' => 'afii10026',
    '1049' => 'afii10027',
    '1050' => 'afii10028',
    '1051' => 'afii10029',
    '1052' => 'afii10030',
    '1053' => 'afii10031',
    '1054' => 'afii10032',
    '1055' => 'afii10033',
    '1056' => 'afii10034',
    '1057' => 'afii10035',
    '1058' => 'afii10036',
    '1059' => 'afii10037',
    '1060' => 'afii10038',
    '1061' => 'afii10039',
    '1062' => 'afii10040',
    '1063' => 'afii10041',
    '1064' => 'afii10042',
    '1065' => 'afii10043',
    '1066' => 'afii10044',
    '1067' => 'afii10045',
    '1068' => 'afii10046',
    '1069' => 'afii10047',
    '1070' => 'afii10048',
    '1071' => 'afii10049',
    '1072' => 'afii10065',
    '1073' => 'afii10066',
    '1074' => 'afii10067',
    '1075' => 'afii10068',
    '1076' => 'afii10069',
    '1077' => 'afii10070',
    '1078' => 'afii10072',
    '1079' => 'afii10073',
    '1080' => 'afii10074',
    '1081' => 'afii10075',
    '1082' => 'afii10076',
    '1083' => 'afii10077',
    '1084' => 'afii10078',
    '1085' => 'afii10079',
    '1086' => 'afii10080',
    '1087' => 'afii10081',
    '1088' => 'afii10082',
    '1089' => 'afii10083',
    '1090' => 'afii10084',
    '1091' => 'afii10085',
    '1092' => 'afii10086',
    '1093' => 'afii10087',
    '1094' => 'afii10088',
    '1095' => 'afii10089',
    '1096' => 'afii10090',
    '1097' => 'afii10091',
    '1098' => 'afii10092',
    '1099' => 'afii10093',
    '1100' => 'afii10094',
    '1101' => 'afii10095',
    '1102' => 'afii10096',
    '1103' => 'afii10097',
    '1105' => 'afii10071',
    '1106' => 'afii10099',
    '1107' => 'afii10100',
    '1108' => 'afii10101',
    '1109' => 'afii10102',
    '1110' => 'afii10103',
    '1111' => 'afii10104',
    '1112' => 'afii10105',
    '1113' => 'afii10106',
    '1114' => 'afii10107',
    '1115' => 'afii10108',
    '1116' => 'afii10109',
    '1118' => 'afii10110',
    '1119' => 'afii10193',
    '1120' => 'Omegacyrillic',
    '1121' => 'omegacyrillic',
    '1122' => 'afii10146',
    '1123' => 'afii10194',
    '1124' => 'Eiotifiedcyrillic',
    '1125' => 'eiotifiedcyrillic',
    '1126' => 'Yuslittlecyrillic',
    '1127' => 'yuslittlecyrillic',
    '1128' => 'Yuslittleiotifiedcyrillic',
    '1129' => 'yuslittleiotifiedcyrillic',
    '1130' => 'Yusbigcyrillic',
    '1131' => 'yusbigcyrillic',
    '1132' => 'Yusbigiotifiedcyrillic',
    '1133' => 'yusbigiotifiedcyrillic',
    '1134' => 'Ksicyrillic',
    '1135' => 'ksicyrillic',
    '1136' => 'Psicyrillic',
    '1137' => 'psicyrillic',
    '1138' => 'afii10147',
    '1139' => 'afii10195',
    '1140' => 'afii10148',
    '1141' => 'afii10196',
    '1142' => 'Izhitsadblgravecyrillic',
    '1143' => 'izhitsadblgravecyrillic',
    '1144' => 'Ukcyrillic',
    '1145' => 'ukcyrillic',
    '1146' => 'Omegaroundcyrillic',
    '1147' => 'omegaroundcyrillic',
    '1148' => 'Omegatitlocyrillic',
    '1149' => 'omegatitlocyrillic',
    '1150' => 'Otcyrillic',
    '1151' => 'otcyrillic',
    '1152' => 'Koppacyrillic',
    '1153' => 'koppacyrillic',
    '1154' => 'thousandcyrillic',
    '1155' => 'titlocyrilliccmb',
    '1156' => 'palatalizationcyrilliccmb',
    '1157' => 'dasiapneumatacyrilliccmb',
    '1158' => 'psilipneumatacyrilliccmb',
    '1168' => 'afii10050',
    '1169' => 'afii10098',
    '1170' => 'Ghestrokecyrillic',
    '1171' => 'ghestrokecyrillic',
    '1172' => 'Ghemiddlehookcyrillic',
    '1173' => 'ghemiddlehookcyrillic',
    '1174' => 'Zhedescendercyrillic',
    '1175' => 'zhedescendercyrillic',
    '1176' => 'Zedescendercyrillic',
    '1177' => 'zedescendercyrillic',
    '1178' => 'Kadescendercyrillic',
    '1179' => 'kadescendercyrillic',
    '1180' => 'Kaverticalstrokecyrillic',
    '1181' => 'kaverticalstrokecyrillic',
    '1182' => 'Kastrokecyrillic',
    '1183' => 'kastrokecyrillic',
    '1184' => 'Kabashkircyrillic',
    '1185' => 'kabashkircyrillic',
    '1186' => 'Endescendercyrillic',
    '1187' => 'endescendercyrillic',
    '1188' => 'Enghecyrillic',
    '1189' => 'enghecyrillic',
    '1190' => 'Pemiddlehookcyrillic',
    '1191' => 'pemiddlehookcyrillic',
    '1192' => 'Haabkhasiancyrillic',
    '1193' => 'haabkhasiancyrillic',
    '1194' => 'Esdescendercyrillic',
    '1195' => 'esdescendercyrillic',
    '1196' => 'Tedescendercyrillic',
    '1197' => 'tedescendercyrillic',
    '1198' => 'Ustraightcyrillic',
    '1199' => 'ustraightcyrillic',
    '1200' => 'Ustraightstrokecyrillic',
    '1201' => 'ustraightstrokecyrillic',
    '1202' => 'Hadescendercyrillic',
    '1203' => 'hadescendercyrillic',
    '1204' => 'Tetsecyrillic',
    '1205' => 'tetsecyrillic',
    '1206' => 'Chedescendercyrillic',
    '1207' => 'chedescendercyrillic',
    '1208' => 'Cheverticalstrokecyrillic',
    '1209' => 'cheverticalstrokecyrillic',
    '1210' => 'Shhacyrillic',
    '1211' => 'shhacyrillic',
    '1212' => 'Cheabkhasiancyrillic',
    '1213' => 'cheabkhasiancyrillic',
    '1214' => 'Chedescenderabkhasiancyrillic',
    '1215' => 'chedescenderabkhasiancyrillic',
    '1216' => 'palochkacyrillic',
    '1217' => 'Zhebrevecyrillic',
    '1218' => 'zhebrevecyrillic',
    '1219' => 'Kahookcyrillic',
    '1220' => 'kahookcyrillic',
    '1223' => 'Enhookcyrillic',
    '1224' => 'enhookcyrillic',
    '1227' => 'Chekhakassiancyrillic',
    '1228' => 'chekhakassiancyrillic',
    '1232' => 'Abrevecyrillic',
    '1233' => 'abrevecyrillic',
    '1234' => 'Adieresiscyrillic',
    '1235' => 'adieresiscyrillic',
    '1236' => 'Aiecyrillic',
    '1237' => 'aiecyrillic',
    '1238' => 'Iebrevecyrillic',
    '1239' => 'iebrevecyrillic',
    '1240' => 'Schwacyrillic',
    '1241' => 'afii10846',
    '1242' => 'Schwadieresiscyrillic',
    '1243' => 'schwadieresiscyrillic',
    '1244' => 'Zhedieresiscyrillic',
    '1245' => 'zhedieresiscyrillic',
    '1246' => 'Zedieresiscyrillic',
    '1247' => 'zedieresiscyrillic',
    '1248' => 'Dzeabkhasiancyrillic',
    '1249' => 'dzeabkhasiancyrillic',
    '1250' => 'Imacroncyrillic',
    '1251' => 'imacroncyrillic',
    '1252' => 'Idieresiscyrillic',
    '1253' => 'idieresiscyrillic',
    '1254' => 'Odieresiscyrillic',
    '1255' => 'odieresiscyrillic',
    '1256' => 'Obarredcyrillic',
    '1257' => 'obarredcyrillic',
    '1258' => 'Obarreddieresiscyrillic',
    '1259' => 'obarreddieresiscyrillic',
    '1262' => 'Umacroncyrillic',
    '1263' => 'umacroncyrillic',
    '1264' => 'Udieresiscyrillic',
    '1265' => 'udieresiscyrillic',
    '1266' => 'Uhungarumlautcyrillic',
    '1267' => 'uhungarumlautcyrillic',
    '1268' => 'Chedieresiscyrillic',
    '1269' => 'chedieresiscyrillic',
    '1272' => 'Yerudieresiscyrillic',
    '1273' => 'yerudieresiscyrillic',
    '1329' => 'Aybarmenian',
    '1330' => 'Benarmenian',
    '1331' => 'Gimarmenian',
    '1332' => 'Daarmenian',
    '1333' => 'Echarmenian',
    '1334' => 'Zaarmenian',
    '1335' => 'Eharmenian',
    '1336' => 'Etarmenian',
    '1337' => 'Toarmenian',
    '1338' => 'Zhearmenian',
    '1339' => 'Iniarmenian',
    '1340' => 'Liwnarmenian',
    '1341' => 'Xeharmenian',
    '1342' => 'Caarmenian',
    '1343' => 'Kenarmenian',
    '1344' => 'Hoarmenian',
    '1345' => 'Jaarmenian',
    '1346' => 'Ghadarmenian',
    '1347' => 'Cheharmenian',
    '1348' => 'Menarmenian',
    '1349' => 'Yiarmenian',
    '1350' => 'Nowarmenian',
    '1351' => 'Shaarmenian',
    '1352' => 'Voarmenian',
    '1353' => 'Chaarmenian',
    '1354' => 'Peharmenian',
    '1355' => 'Jheharmenian',
    '1356' => 'Raarmenian',
    '1357' => 'Seharmenian',
    '1358' => 'Vewarmenian',
    '1359' => 'Tiwnarmenian',
    '1360' => 'Reharmenian',
    '1361' => 'Coarmenian',
    '1362' => 'Yiwnarmenian',
    '1363' => 'Piwrarmenian',
    '1364' => 'Keharmenian',
    '1365' => 'Oharmenian',
    '1366' => 'Feharmenian',
    '1369' => 'ringhalfleftarmenian',
    '1370' => 'apostrophearmenian',
    '1371' => 'emphasismarkarmenian',
    '1372' => 'exclamarmenian',
    '1373' => 'commaarmenian',
    '1374' => 'questionarmenian',
    '1375' => 'abbreviationmarkarmenian',
    '1377' => 'aybarmenian',
    '1378' => 'benarmenian',
    '1379' => 'gimarmenian',
    '1380' => 'daarmenian',
    '1381' => 'echarmenian',
    '1382' => 'zaarmenian',
    '1383' => 'eharmenian',
    '1384' => 'etarmenian',
    '1385' => 'toarmenian',
    '1386' => 'zhearmenian',
    '1387' => 'iniarmenian',
    '1388' => 'liwnarmenian',
    '1389' => 'xeharmenian',
    '1390' => 'caarmenian',
    '1391' => 'kenarmenian',
    '1392' => 'hoarmenian',
    '1393' => 'jaarmenian',
    '1394' => 'ghadarmenian',
    '1395' => 'cheharmenian',
    '1396' => 'menarmenian',
    '1397' => 'yiarmenian',
    '1398' => 'nowarmenian',
    '1399' => 'shaarmenian',
    '1400' => 'voarmenian',
    '1401' => 'chaarmenian',
    '1402' => 'peharmenian',
    '1403' => 'jheharmenian',
    '1404' => 'raarmenian',
    '1405' => 'seharmenian',
    '1406' => 'vewarmenian',
    '1407' => 'tiwnarmenian',
    '1408' => 'reharmenian',
    '1409' => 'coarmenian',
    '1410' => 'yiwnarmenian',
    '1411' => 'piwrarmenian',
    '1412' => 'keharmenian',
    '1413' => 'oharmenian',
    '1414' => 'feharmenian',
    '1415' => 'echyiwnarmenian',
    '1417' => 'periodarmenian',
    '1425' => 'etnahtafoukhhebrew',
    '1426' => 'segoltahebrew',
    '1427' => 'shalshelethebrew',
    '1428' => 'zaqefqatanhebrew',
    '1429' => 'zaqefgadolhebrew',
    '1430' => 'tipehahebrew',
    '1431' => 'reviahebrew',
    '1432' => 'zarqahebrew',
    '1433' => 'pashtahebrew',
    '1434' => 'yetivhebrew',
    '1435' => 'tevirhebrew',
    '1436' => 'gereshaccenthebrew',
    '1437' => 'gereshmuqdamhebrew',
    '1438' => 'gershayimaccenthebrew',
    '1439' => 'qarneyparahebrew',
    '1440' => 'telishagedolahebrew',
    '1441' => 'pazerhebrew',
    '1443' => 'munahhebrew',
    '1444' => 'mahapakhhebrew',
    '1445' => 'merkhahebrew',
    '1446' => 'merkhakefulahebrew',
    '1447' => 'dargahebrew',
    '1448' => 'qadmahebrew',
    '1449' => 'telishaqetanahebrew',
    '1450' => 'yerahbenyomohebrew',
    '1451' => 'olehebrew',
    '1452' => 'iluyhebrew',
    '1453' => 'dehihebrew',
    '1454' => 'zinorhebrew',
    '1455' => 'masoracirclehebrew',
    '1456' => 'afii57799',
    '1457' => 'afii57801',
    '1458' => 'afii57800',
    '1459' => 'afii57802',
    '1460' => 'afii57793',
    '1461' => 'afii57794',
    '1462' => 'afii57795',
    '1463' => 'afii57798',
    '1464' => 'afii57797',
    '1465' => 'afii57806',
    '1467' => 'afii57796',
    '1468' => 'afii57807',
    '1469' => 'afii57839',
    '1470' => 'afii57645',
    '1471' => 'afii57841',
    '1472' => 'afii57842',
    '1473' => 'afii57804',
    '1474' => 'afii57803',
    '1475' => 'afii57658',
    '1476' => 'upperdothebrew',
    '1488' => 'afii57664',
    '1489' => 'afii57665',
    '1490' => 'afii57666',
    '1491' => 'afii57667',
    '1492' => 'afii57668',
    '1493' => 'afii57669',
    '1494' => 'afii57670',
    '1495' => 'afii57671',
    '1496' => 'afii57672',
    '1497' => 'afii57673',
    '1498' => 'afii57674',
    '1499' => 'afii57675',
    '1500' => 'afii57676',
    '1501' => 'afii57677',
    '1502' => 'afii57678',
    '1503' => 'afii57679',
    '1504' => 'afii57680',
    '1505' => 'afii57681',
    '1506' => 'afii57682',
    '1507' => 'afii57683',
    '1508' => 'afii57684',
    '1509' => 'afii57685',
    '1510' => 'afii57686',
    '1511' => 'afii57687',
    '1512' => 'afii57688',
    '1513' => 'afii57689',
    '1514' => 'afii57690',
    '1520' => 'afii57716',
    '1521' => 'afii57717',
    '1522' => 'afii57718',
    '1523' => 'gereshhebrew',
    '1524' => 'gershayimhebrew',
    '1548' => 'afii57388',
    '1563' => 'afii57403',
    '1567' => 'afii57407',
    '1569' => 'afii57409',
    '1570' => 'afii57410',
    '1571' => 'afii57411',
    '1572' => 'afii57412',
    '1573' => 'afii57413',
    '1574' => 'afii57414',
    '1575' => 'afii57415',
    '1576' => 'afii57416',
    '1577' => 'afii57417',
    '1578' => 'afii57418',
    '1579' => 'afii57419',
    '1580' => 'afii57420',
    '1581' => 'afii57421',
    '1582' => 'afii57422',
    '1583' => 'afii57423',
    '1584' => 'afii57424',
    '1585' => 'afii57425',
    '1586' => 'afii57426',
    '1587' => 'afii57427',
    '1588' => 'afii57428',
    '1589' => 'afii57429',
    '1590' => 'afii57430',
    '1591' => 'afii57431',
    '1592' => 'afii57432',
    '1593' => 'afii57433',
    '1594' => 'afii57434',
    '1600' => 'afii57440',
    '1601' => 'afii57441',
    '1602' => 'afii57442',
    '1603' => 'afii57443',
    '1604' => 'afii57444',
    '1605' => 'afii57445',
    '1606' => 'afii57446',
    '1607' => 'afii57470',
    '1608' => 'afii57448',
    '1609' => 'afii57449',
    '1610' => 'afii57450',
    '1611' => 'afii57451',
    '1612' => 'afii57452',
    '1613' => 'afii57453',
    '1614' => 'afii57454',
    '1615' => 'afii57455',
    '1616' => 'afii57456',
    '1617' => 'afii57457',
    '1618' => 'afii57458',
    '1632' => 'afii57392',
    '1633' => 'afii57393',
    '1634' => 'afii57394',
    '1635' => 'afii57395',
    '1636' => 'afii57396',
    '1637' => 'afii57397',
    '1638' => 'afii57398',
    '1639' => 'afii57399',
    '1640' => 'afii57400',
    '1641' => 'afii57401',
    '1642' => 'afii57381',
    '1643' => 'decimalseparatorarabic',
    '1644' => 'thousandsseparatorarabic',
    '1645' => 'afii63167',
    '1657' => 'afii57511',
    '1662' => 'afii57506',
    '1670' => 'afii57507',
    '1672' => 'afii57512',
    '1681' => 'afii57513',
    '1688' => 'afii57508',
    '1700' => 'afii57505',
    '1711' => 'afii57509',
    '1722' => 'afii57514',
    '1729' => 'haaltonearabic',
    '1745' => 'yehthreedotsbelowarabic',
    '1746' => 'afii57519',
    '1749' => 'afii57534',
    '1776' => 'zeropersian',
    '1777' => 'onepersian',
    '1778' => 'twopersian',
    '1779' => 'threepersian',
    '1780' => 'fourpersian',
    '1781' => 'fivepersian',
    '1782' => 'sixpersian',
    '1783' => 'sevenpersian',
    '1784' => 'eightpersian',
    '1785' => 'ninepersian',
    '2305' => 'candrabindudeva',
    '2306' => 'anusvaradeva',
    '2307' => 'visargadeva',
    '2309' => 'adeva',
    '2310' => 'aadeva',
    '2311' => 'ideva',
    '2312' => 'iideva',
    '2313' => 'udeva',
    '2314' => 'uudeva',
    '2315' => 'rvocalicdeva',
    '2316' => 'lvocalicdeva',
    '2317' => 'ecandradeva',
    '2318' => 'eshortdeva',
    '2319' => 'edeva',
    '2320' => 'aideva',
    '2321' => 'ocandradeva',
    '2322' => 'oshortdeva',
    '2323' => 'odeva',
    '2324' => 'audeva',
    '2325' => 'kadeva',
    '2326' => 'khadeva',
    '2327' => 'gadeva',
    '2328' => 'ghadeva',
    '2329' => 'ngadeva',
    '2330' => 'cadeva',
    '2331' => 'chadeva',
    '2332' => 'jadeva',
    '2333' => 'jhadeva',
    '2334' => 'nyadeva',
    '2335' => 'ttadeva',
    '2336' => 'tthadeva',
    '2337' => 'ddadeva',
    '2338' => 'ddhadeva',
    '2339' => 'nnadeva',
    '2340' => 'tadeva',
    '2341' => 'thadeva',
    '2342' => 'dadeva',
    '2343' => 'dhadeva',
    '2344' => 'nadeva',
    '2345' => 'nnnadeva',
    '2346' => 'padeva',
    '2347' => 'phadeva',
    '2348' => 'badeva',
    '2349' => 'bhadeva',
    '2350' => 'madeva',
    '2351' => 'yadeva',
    '2352' => 'radeva',
    '2353' => 'rradeva',
    '2354' => 'ladeva',
    '2355' => 'lladeva',
    '2356' => 'llladeva',
    '2357' => 'vadeva',
    '2358' => 'shadeva',
    '2359' => 'ssadeva',
    '2360' => 'sadeva',
    '2361' => 'hadeva',
    '2364' => 'nuktadeva',
    '2365' => 'avagrahadeva',
    '2366' => 'aavowelsigndeva',
    '2367' => 'ivowelsigndeva',
    '2368' => 'iivowelsigndeva',
    '2369' => 'uvowelsigndeva',
    '2370' => 'uuvowelsigndeva',
    '2371' => 'rvocalicvowelsigndeva',
    '2372' => 'rrvocalicvowelsigndeva',
    '2373' => 'ecandravowelsigndeva',
    '2374' => 'eshortvowelsigndeva',
    '2375' => 'evowelsigndeva',
    '2376' => 'aivowelsigndeva',
    '2377' => 'ocandravowelsigndeva',
    '2378' => 'oshortvowelsigndeva',
    '2379' => 'ovowelsigndeva',
    '2380' => 'auvowelsigndeva',
    '2381' => 'viramadeva',
    '2384' => 'omdeva',
    '2385' => 'udattadeva',
    '2386' => 'anudattadeva',
    '2387' => 'gravedeva',
    '2388' => 'acutedeva',
    '2392' => 'qadeva',
    '2393' => 'khhadeva',
    '2394' => 'ghhadeva',
    '2395' => 'zadeva',
    '2396' => 'dddhadeva',
    '2397' => 'rhadeva',
    '2398' => 'fadeva',
    '2399' => 'yyadeva',
    '2400' => 'rrvocalicdeva',
    '2401' => 'llvocalicdeva',
    '2402' => 'lvocalicvowelsigndeva',
    '2403' => 'llvocalicvowelsigndeva',
    '2404' => 'danda',
    '2405' => 'dbldanda',
    '2406' => 'zerodeva',
    '2407' => 'onedeva',
    '2408' => 'twodeva',
    '2409' => 'threedeva',
    '2410' => 'fourdeva',
    '2411' => 'fivedeva',
    '2412' => 'sixdeva',
    '2413' => 'sevendeva',
    '2414' => 'eightdeva',
    '2415' => 'ninedeva',
    '2416' => 'abbreviationsigndeva',
    '2433' => 'candrabindubengali',
    '2434' => 'anusvarabengali',
    '2435' => 'visargabengali',
    '2437' => 'abengali',
    '2438' => 'aabengali',
    '2439' => 'ibengali',
    '2440' => 'iibengali',
    '2441' => 'ubengali',
    '2442' => 'uubengali',
    '2443' => 'rvocalicbengali',
    '2444' => 'lvocalicbengali',
    '2447' => 'ebengali',
    '2448' => 'aibengali',
    '2451' => 'obengali',
    '2452' => 'aubengali',
    '2453' => 'kabengali',
    '2454' => 'khabengali',
    '2455' => 'gabengali',
    '2456' => 'ghabengali',
    '2457' => 'ngabengali',
    '2458' => 'cabengali',
    '2459' => 'chabengali',
    '2460' => 'jabengali',
    '2461' => 'jhabengali',
    '2462' => 'nyabengali',
    '2463' => 'ttabengali',
    '2464' => 'tthabengali',
    '2465' => 'ddabengali',
    '2466' => 'ddhabengali',
    '2467' => 'nnabengali',
    '2468' => 'tabengali',
    '2469' => 'thabengali',
    '2470' => 'dabengali',
    '2471' => 'dhabengali',
    '2472' => 'nabengali',
    '2474' => 'pabengali',
    '2475' => 'phabengali',
    '2476' => 'babengali',
    '2477' => 'bhabengali',
    '2478' => 'mabengali',
    '2479' => 'yabengali',
    '2480' => 'rabengali',
    '2482' => 'labengali',
    '2486' => 'shabengali',
    '2487' => 'ssabengali',
    '2488' => 'sabengali',
    '2489' => 'habengali',
    '2492' => 'nuktabengali',
    '2494' => 'aavowelsignbengali',
    '2495' => 'ivowelsignbengali',
    '2496' => 'iivowelsignbengali',
    '2497' => 'uvowelsignbengali',
    '2498' => 'uuvowelsignbengali',
    '2499' => 'rvocalicvowelsignbengali',
    '2500' => 'rrvocalicvowelsignbengali',
    '2503' => 'evowelsignbengali',
    '2504' => 'aivowelsignbengali',
    '2507' => 'ovowelsignbengali',
    '2508' => 'auvowelsignbengali',
    '2509' => 'viramabengali',
    '2519' => 'aulengthmarkbengali',
    '2524' => 'rrabengali',
    '2525' => 'rhabengali',
    '2527' => 'yyabengali',
    '2528' => 'rrvocalicbengali',
    '2529' => 'llvocalicbengali',
    '2530' => 'lvocalicvowelsignbengali',
    '2531' => 'llvocalicvowelsignbengali',
    '2534' => 'zerobengali',
    '2535' => 'onebengali',
    '2536' => 'twobengali',
    '2537' => 'threebengali',
    '2538' => 'fourbengali',
    '2539' => 'fivebengali',
    '2540' => 'sixbengali',
    '2541' => 'sevenbengali',
    '2542' => 'eightbengali',
    '2543' => 'ninebengali',
    '2544' => 'ramiddlediagonalbengali',
    '2545' => 'ralowerdiagonalbengali',
    '2546' => 'rupeemarkbengali',
    '2547' => 'rupeesignbengali',
    '2548' => 'onenumeratorbengali',
    '2549' => 'twonumeratorbengali',
    '2550' => 'threenumeratorbengali',
    '2551' => 'fournumeratorbengali',
    '2552' => 'denominatorminusonenumeratorbengali',
    '2553' => 'sixteencurrencydenominatorbengali',
    '2554' => 'issharbengali',
    '2562' => 'bindigurmukhi',
    '2565' => 'agurmukhi',
    '2566' => 'aagurmukhi',
    '2567' => 'igurmukhi',
    '2568' => 'iigurmukhi',
    '2569' => 'ugurmukhi',
    '2570' => 'uugurmukhi',
    '2575' => 'eegurmukhi',
    '2576' => 'aigurmukhi',
    '2579' => 'oogurmukhi',
    '2580' => 'augurmukhi',
    '2581' => 'kagurmukhi',
    '2582' => 'khagurmukhi',
    '2583' => 'gagurmukhi',
    '2584' => 'ghagurmukhi',
    '2585' => 'ngagurmukhi',
    '2586' => 'cagurmukhi',
    '2587' => 'chagurmukhi',
    '2588' => 'jagurmukhi',
    '2589' => 'jhagurmukhi',
    '2590' => 'nyagurmukhi',
    '2591' => 'ttagurmukhi',
    '2592' => 'tthagurmukhi',
    '2593' => 'ddagurmukhi',
    '2594' => 'ddhagurmukhi',
    '2595' => 'nnagurmukhi',
    '2596' => 'tagurmukhi',
    '2597' => 'thagurmukhi',
    '2598' => 'dagurmukhi',
    '2599' => 'dhagurmukhi',
    '2600' => 'nagurmukhi',
    '2602' => 'pagurmukhi',
    '2603' => 'phagurmukhi',
    '2604' => 'bagurmukhi',
    '2605' => 'bhagurmukhi',
    '2606' => 'magurmukhi',
    '2607' => 'yagurmukhi',
    '2608' => 'ragurmukhi',
    '2610' => 'lagurmukhi',
    '2613' => 'vagurmukhi',
    '2614' => 'shagurmukhi',
    '2616' => 'sagurmukhi',
    '2617' => 'hagurmukhi',
    '2620' => 'nuktagurmukhi',
    '2622' => 'aamatragurmukhi',
    '2623' => 'imatragurmukhi',
    '2624' => 'iimatragurmukhi',
    '2625' => 'umatragurmukhi',
    '2626' => 'uumatragurmukhi',
    '2631' => 'eematragurmukhi',
    '2632' => 'aimatragurmukhi',
    '2635' => 'oomatragurmukhi',
    '2636' => 'aumatragurmukhi',
    '2637' => 'halantgurmukhi',
    '2649' => 'khhagurmukhi',
    '2650' => 'ghhagurmukhi',
    '2651' => 'zagurmukhi',
    '2652' => 'rragurmukhi',
    '2654' => 'fagurmukhi',
    '2662' => 'zerogurmukhi',
    '2663' => 'onegurmukhi',
    '2664' => 'twogurmukhi',
    '2665' => 'threegurmukhi',
    '2666' => 'fourgurmukhi',
    '2667' => 'fivegurmukhi',
    '2668' => 'sixgurmukhi',
    '2669' => 'sevengurmukhi',
    '2670' => 'eightgurmukhi',
    '2671' => 'ninegurmukhi',
    '2672' => 'tippigurmukhi',
    '2673' => 'addakgurmukhi',
    '2674' => 'irigurmukhi',
    '2675' => 'uragurmukhi',
    '2676' => 'ekonkargurmukhi',
    '2689' => 'candrabindugujarati',
    '2690' => 'anusvaragujarati',
    '2691' => 'visargagujarati',
    '2693' => 'agujarati',
    '2694' => 'aagujarati',
    '2695' => 'igujarati',
    '2696' => 'iigujarati',
    '2697' => 'ugujarati',
    '2698' => 'uugujarati',
    '2699' => 'rvocalicgujarati',
    '2701' => 'ecandragujarati',
    '2703' => 'egujarati',
    '2704' => 'aigujarati',
    '2705' => 'ocandragujarati',
    '2707' => 'ogujarati',
    '2708' => 'augujarati',
    '2709' => 'kagujarati',
    '2710' => 'khagujarati',
    '2711' => 'gagujarati',
    '2712' => 'ghagujarati',
    '2713' => 'ngagujarati',
    '2714' => 'cagujarati',
    '2715' => 'chagujarati',
    '2716' => 'jagujarati',
    '2717' => 'jhagujarati',
    '2718' => 'nyagujarati',
    '2719' => 'ttagujarati',
    '2720' => 'tthagujarati',
    '2721' => 'ddagujarati',
    '2722' => 'ddhagujarati',
    '2723' => 'nnagujarati',
    '2724' => 'tagujarati',
    '2725' => 'thagujarati',
    '2726' => 'dagujarati',
    '2727' => 'dhagujarati',
    '2728' => 'nagujarati',
    '2730' => 'pagujarati',
    '2731' => 'phagujarati',
    '2732' => 'bagujarati',
    '2733' => 'bhagujarati',
    '2734' => 'magujarati',
    '2735' => 'yagujarati',
    '2736' => 'ragujarati',
    '2738' => 'lagujarati',
    '2739' => 'llagujarati',
    '2741' => 'vagujarati',
    '2742' => 'shagujarati',
    '2743' => 'ssagujarati',
    '2744' => 'sagujarati',
    '2745' => 'hagujarati',
    '2748' => 'nuktagujarati',
    '2750' => 'aavowelsigngujarati',
    '2751' => 'ivowelsigngujarati',
    '2752' => 'iivowelsigngujarati',
    '2753' => 'uvowelsigngujarati',
    '2754' => 'uuvowelsigngujarati',
    '2755' => 'rvocalicvowelsigngujarati',
    '2756' => 'rrvocalicvowelsigngujarati',
    '2757' => 'ecandravowelsigngujarati',
    '2759' => 'evowelsigngujarati',
    '2760' => 'aivowelsigngujarati',
    '2761' => 'ocandravowelsigngujarati',
    '2763' => 'ovowelsigngujarati',
    '2764' => 'auvowelsigngujarati',
    '2765' => 'viramagujarati',
    '2768' => 'omgujarati',
    '2784' => 'rrvocalicgujarati',
    '2790' => 'zerogujarati',
    '2791' => 'onegujarati',
    '2792' => 'twogujarati',
    '2793' => 'threegujarati',
    '2794' => 'fourgujarati',
    '2795' => 'fivegujarati',
    '2796' => 'sixgujarati',
    '2797' => 'sevengujarati',
    '2798' => 'eightgujarati',
    '2799' => 'ninegujarati',
    '3585' => 'kokaithai',
    '3586' => 'khokhaithai',
    '3587' => 'khokhuatthai',
    '3588' => 'khokhwaithai',
    '3589' => 'khokhonthai',
    '3590' => 'khorakhangthai',
    '3591' => 'ngonguthai',
    '3592' => 'chochanthai',
    '3593' => 'chochingthai',
    '3594' => 'chochangthai',
    '3595' => 'sosothai',
    '3596' => 'chochoethai',
    '3597' => 'yoyingthai',
    '3598' => 'dochadathai',
    '3599' => 'topatakthai',
    '3600' => 'thothanthai',
    '3601' => 'thonangmonthothai',
    '3602' => 'thophuthaothai',
    '3603' => 'nonenthai',
    '3604' => 'dodekthai',
    '3605' => 'totaothai',
    '3606' => 'thothungthai',
    '3607' => 'thothahanthai',
    '3608' => 'thothongthai',
    '3609' => 'nonuthai',
    '3610' => 'bobaimaithai',
    '3611' => 'poplathai',
    '3612' => 'phophungthai',
    '3613' => 'fofathai',
    '3614' => 'phophanthai',
    '3615' => 'fofanthai',
    '3616' => 'phosamphaothai',
    '3617' => 'momathai',
    '3618' => 'yoyakthai',
    '3619' => 'roruathai',
    '3620' => 'ruthai',
    '3621' => 'lolingthai',
    '3622' => 'luthai',
    '3623' => 'wowaenthai',
    '3624' => 'sosalathai',
    '3625' => 'sorusithai',
    '3626' => 'sosuathai',
    '3627' => 'hohipthai',
    '3628' => 'lochulathai',
    '3629' => 'oangthai',
    '3630' => 'honokhukthai',
    '3631' => 'paiyannoithai',
    '3632' => 'saraathai',
    '3633' => 'maihanakatthai',
    '3634' => 'saraaathai',
    '3635' => 'saraamthai',
    '3636' => 'saraithai',
    '3637' => 'saraiithai',
    '3638' => 'sarauethai',
    '3639' => 'saraueethai',
    '3640' => 'sarauthai',
    '3641' => 'sarauuthai',
    '3642' => 'phinthuthai',
    '3647' => 'bahtthai',
    '3648' => 'saraethai',
    '3649' => 'saraaethai',
    '3650' => 'saraothai',
    '3651' => 'saraaimaimuanthai',
    '3652' => 'saraaimaimalaithai',
    '3653' => 'lakkhangyaothai',
    '3654' => 'maiyamokthai',
    '3655' => 'maitaikhuthai',
    '3656' => 'maiekthai',
    '3657' => 'maithothai',
    '3658' => 'maitrithai',
    '3659' => 'maichattawathai',
    '3660' => 'thanthakhatthai',
    '3661' => 'nikhahitthai',
    '3662' => 'yamakkanthai',
    '3663' => 'fongmanthai',
    '3664' => 'zerothai',
    '3665' => 'onethai',
    '3666' => 'twothai',
    '3667' => 'threethai',
    '3668' => 'fourthai',
    '3669' => 'fivethai',
    '3670' => 'sixthai',
    '3671' => 'seventhai',
    '3672' => 'eightthai',
    '3673' => 'ninethai',
    '3674' => 'angkhankhuthai',
    '3675' => 'khomutthai',
    '7680' => 'Aringbelow',
    '7681' => 'aringbelow',
    '7682' => 'Bdotaccent',
    '7683' => 'bdotaccent',
    '7684' => 'Bdotbelow',
    '7685' => 'bdotbelow',
    '7686' => 'Blinebelow',
    '7687' => 'blinebelow',
    '7688' => 'Ccedillaacute',
    '7689' => 'ccedillaacute',
    '7690' => 'Ddotaccent',
    '7691' => 'ddotaccent',
    '7692' => 'Ddotbelow',
    '7693' => 'ddotbelow',
    '7694' => 'Dlinebelow',
    '7695' => 'dlinebelow',
    '7696' => 'Dcedilla',
    '7697' => 'dcedilla',
    '7698' => 'Dcircumflexbelow',
    '7699' => 'dcircumflexbelow',
    '7700' => 'Emacrongrave',
    '7701' => 'emacrongrave',
    '7702' => 'Emacronacute',
    '7703' => 'emacronacute',
    '7704' => 'Ecircumflexbelow',
    '7705' => 'ecircumflexbelow',
    '7706' => 'Etildebelow',
    '7707' => 'etildebelow',
    '7708' => 'Ecedillabreve',
    '7709' => 'ecedillabreve',
    '7710' => 'Fdotaccent',
    '7711' => 'fdotaccent',
    '7712' => 'Gmacron',
    '7713' => 'gmacron',
    '7714' => 'Hdotaccent',
    '7715' => 'hdotaccent',
    '7716' => 'Hdotbelow',
    '7717' => 'hdotbelow',
    '7718' => 'Hdieresis',
    '7719' => 'hdieresis',
    '7720' => 'Hcedilla',
    '7721' => 'hcedilla',
    '7722' => 'Hbrevebelow',
    '7723' => 'hbrevebelow',
    '7724' => 'Itildebelow',
    '7725' => 'itildebelow',
    '7726' => 'Idieresisacute',
    '7727' => 'idieresisacute',
    '7728' => 'Kacute',
    '7729' => 'kacute',
    '7730' => 'Kdotbelow',
    '7731' => 'kdotbelow',
    '7732' => 'Klinebelow',
    '7733' => 'klinebelow',
    '7734' => 'Ldotbelow',
    '7735' => 'ldotbelow',
    '7736' => 'Ldotbelowmacron',
    '7737' => 'ldotbelowmacron',
    '7738' => 'Llinebelow',
    '7739' => 'llinebelow',
    '7740' => 'Lcircumflexbelow',
    '7741' => 'lcircumflexbelow',
    '7742' => 'Macute',
    '7743' => 'macute',
    '7744' => 'Mdotaccent',
    '7745' => 'mdotaccent',
    '7746' => 'Mdotbelow',
    '7747' => 'mdotbelow',
    '7748' => 'Ndotaccent',
    '7749' => 'ndotaccent',
    '7750' => 'Ndotbelow',
    '7751' => 'ndotbelow',
    '7752' => 'Nlinebelow',
    '7753' => 'nlinebelow',
    '7754' => 'Ncircumflexbelow',
    '7755' => 'ncircumflexbelow',
    '7756' => 'Otildeacute',
    '7757' => 'otildeacute',
    '7758' => 'Otildedieresis',
    '7759' => 'otildedieresis',
    '7760' => 'Omacrongrave',
    '7761' => 'omacrongrave',
    '7762' => 'Omacronacute',
    '7763' => 'omacronacute',
    '7764' => 'Pacute',
    '7765' => 'pacute',
    '7766' => 'Pdotaccent',
    '7767' => 'pdotaccent',
    '7768' => 'Rdotaccent',
    '7769' => 'rdotaccent',
    '7770' => 'Rdotbelow',
    '7771' => 'rdotbelow',
    '7772' => 'Rdotbelowmacron',
    '7773' => 'rdotbelowmacron',
    '7774' => 'Rlinebelow',
    '7775' => 'rlinebelow',
    '7776' => 'Sdotaccent',
    '7777' => 'sdotaccent',
    '7778' => 'Sdotbelow',
    '7779' => 'sdotbelow',
    '7780' => 'Sacutedotaccent',
    '7781' => 'sacutedotaccent',
    '7782' => 'Scarondotaccent',
    '7783' => 'scarondotaccent',
    '7784' => 'Sdotbelowdotaccent',
    '7785' => 'sdotbelowdotaccent',
    '7786' => 'Tdotaccent',
    '7787' => 'tdotaccent',
    '7788' => 'Tdotbelow',
    '7789' => 'tdotbelow',
    '7790' => 'Tlinebelow',
    '7791' => 'tlinebelow',
    '7792' => 'Tcircumflexbelow',
    '7793' => 'tcircumflexbelow',
    '7794' => 'Udieresisbelow',
    '7795' => 'udieresisbelow',
    '7796' => 'Utildebelow',
    '7797' => 'utildebelow',
    '7798' => 'Ucircumflexbelow',
    '7799' => 'ucircumflexbelow',
    '7800' => 'Utildeacute',
    '7801' => 'utildeacute',
    '7802' => 'Umacrondieresis',
    '7803' => 'umacrondieresis',
    '7804' => 'Vtilde',
    '7805' => 'vtilde',
    '7806' => 'Vdotbelow',
    '7807' => 'vdotbelow',
    '7808' => 'Wgrave',
    '7809' => 'wgrave',
    '7810' => 'Wacute',
    '7811' => 'wacute',
    '7812' => 'Wdieresis',
    '7813' => 'wdieresis',
    '7814' => 'Wdotaccent',
    '7815' => 'wdotaccent',
    '7816' => 'Wdotbelow',
    '7817' => 'wdotbelow',
    '7818' => 'Xdotaccent',
    '7819' => 'xdotaccent',
    '7820' => 'Xdieresis',
    '7821' => 'xdieresis',
    '7822' => 'Ydotaccent',
    '7823' => 'ydotaccent',
    '7824' => 'Zcircumflex',
    '7825' => 'zcircumflex',
    '7826' => 'Zdotbelow',
    '7827' => 'zdotbelow',
    '7828' => 'Zlinebelow',
    '7829' => 'zlinebelow',
    '7830' => 'hlinebelow',
    '7831' => 'tdieresis',
    '7832' => 'wring',
    '7833' => 'yring',
    '7834' => 'arighthalfring',
    '7835' => 'slongdotaccent',
    '7840' => 'Adotbelow',
    '7841' => 'adotbelow',
    '7842' => 'Ahookabove',
    '7843' => 'ahookabove',
    '7844' => 'Acircumflexacute',
    '7845' => 'acircumflexacute',
    '7846' => 'Acircumflexgrave',
    '7847' => 'acircumflexgrave',
    '7848' => 'Acircumflexhookabove',
    '7849' => 'acircumflexhookabove',
    '7850' => 'Acircumflextilde',
    '7851' => 'acircumflextilde',
    '7852' => 'Acircumflexdotbelow',
    '7853' => 'acircumflexdotbelow',
    '7854' => 'Abreveacute',
    '7855' => 'abreveacute',
    '7856' => 'Abrevegrave',
    '7857' => 'abrevegrave',
    '7858' => 'Abrevehookabove',
    '7859' => 'abrevehookabove',
    '7860' => 'Abrevetilde',
    '7861' => 'abrevetilde',
    '7862' => 'Abrevedotbelow',
    '7863' => 'abrevedotbelow',
    '7864' => 'Edotbelow',
    '7865' => 'edotbelow',
    '7866' => 'Ehookabove',
    '7867' => 'ehookabove',
    '7868' => 'Etilde',
    '7869' => 'etilde',
    '7870' => 'Ecircumflexacute',
    '7871' => 'ecircumflexacute',
    '7872' => 'Ecircumflexgrave',
    '7873' => 'ecircumflexgrave',
    '7874' => 'Ecircumflexhookabove',
    '7875' => 'ecircumflexhookabove',
    '7876' => 'Ecircumflextilde',
    '7877' => 'ecircumflextilde',
    '7878' => 'Ecircumflexdotbelow',
    '7879' => 'ecircumflexdotbelow',
    '7880' => 'Ihookabove',
    '7881' => 'ihookabove',
    '7882' => 'Idotbelow',
    '7883' => 'idotbelow',
    '7884' => 'Odotbelow',
    '7885' => 'odotbelow',
    '7886' => 'Ohookabove',
    '7887' => 'ohookabove',
    '7888' => 'Ocircumflexacute',
    '7889' => 'ocircumflexacute',
    '7890' => 'Ocircumflexgrave',
    '7891' => 'ocircumflexgrave',
    '7892' => 'Ocircumflexhookabove',
    '7893' => 'ocircumflexhookabove',
    '7894' => 'Ocircumflextilde',
    '7895' => 'ocircumflextilde',
    '7896' => 'Ocircumflexdotbelow',
    '7897' => 'ocircumflexdotbelow',
    '7898' => 'Ohornacute',
    '7899' => 'ohornacute',
    '7900' => 'Ohorngrave',
    '7901' => 'ohorngrave',
    '7902' => 'Ohornhookabove',
    '7903' => 'ohornhookabove',
    '7904' => 'Ohorntilde',
    '7905' => 'ohorntilde',
    '7906' => 'Ohorndotbelow',
    '7907' => 'ohorndotbelow',
    '7908' => 'Udotbelow',
    '7909' => 'udotbelow',
    '7910' => 'Uhookabove',
    '7911' => 'uhookabove',
    '7912' => 'Uhornacute',
    '7913' => 'uhornacute',
    '7914' => 'Uhorngrave',
    '7915' => 'uhorngrave',
    '7916' => 'Uhornhookabove',
    '7917' => 'uhornhookabove',
    '7918' => 'Uhorntilde',
    '7919' => 'uhorntilde',
    '7920' => 'Uhorndotbelow',
    '7921' => 'uhorndotbelow',
    '7922' => 'Ygrave',
    '7923' => 'ygrave',
    '7924' => 'Ydotbelow',
    '7925' => 'ydotbelow',
    '7926' => 'Yhookabove',
    '7927' => 'yhookabove',
    '7928' => 'Ytilde',
    '7929' => 'ytilde',
    '8194' => 'enspace',
    '8195' => 'emsp',
    '8196' => 'emsp13',
    '8197' => 'emsp14',
    '8199' => 'numsp',
    '8200' => 'puncsp',
    '8201' => 'thinsp',
    '8202' => 'hairsp',
    '8203' => 'zerowidthspace',
    '8204' => 'afii61664',
    '8205' => 'afii301',
    '8206' => 'afii299',
    '8207' => 'afii300',
    '8208' => 'hyphentwo',
    '8210' => 'figuredash',
    '8211' => 'endash',
    '8212' => 'emdash',
    '8213' => 'afii00208',
    '8214' => 'dblverticalbar',
    '8215' => 'underscoredbl',
    '8216' => 'quoteleft',
    '8217' => 'quoteright',
    '8218' => 'quotesinglbase',
    '8219' => 'quotereversed',
    '8220' => 'quotedblleft',
    '8221' => 'quotedblright',
    '8222' => 'quotedblbase',
    '8224' => 'dagger',
    '8225' => 'daggerdbl',
    '8226' => 'bullet',
    '8228' => 'onedotenleader',
    '8229' => 'twodotenleader',
    '8230' => 'ellipsis',
    '8236' => 'afii61573',
    '8237' => 'afii61574',
    '8238' => 'afii61575',
    '8240' => 'perthousand',
    '8242' => 'minute',
    '8243' => 'second',
    '8244' => 'tprime',
    '8245' => 'primereversed',
    '8249' => 'guilsinglleft',
    '8250' => 'guilsinglright',
    '8251' => 'referencemark',
    '8252' => 'exclamdbl',
    '8254' => 'overline',
    '8257' => 'caret',
    '8258' => 'asterism',
    '8259' => 'hybull',
    '8260' => 'fraction',
    '8304' => 'zerosuperior',
    '8308' => 'foursuperior',
    '8309' => 'fivesuperior',
    '8310' => 'sixsuperior',
    '8311' => 'sevensuperior',
    '8312' => 'eightsuperior',
    '8313' => 'ninesuperior',
    '8314' => 'plussuperior',
    '8316' => 'equalsuperior',
    '8317' => 'parenleftsuperior',
    '8318' => 'parenrightsuperior',
    '8319' => 'nsuperior',
    '8320' => 'zeroinferior',
    '8321' => 'oneinferior',
    '8322' => 'twoinferior',
    '8323' => 'threeinferior',
    '8324' => 'fourinferior',
    '8325' => 'fiveinferior',
    '8326' => 'sixinferior',
    '8327' => 'seveninferior',
    '8328' => 'eightinferior',
    '8329' => 'nineinferior',
    '8333' => 'parenleftinferior',
    '8334' => 'parenrightinferior',
    '8353' => 'colonmonetary',
    '8354' => 'cruzeiro',
    '8355' => 'franc',
    '8356' => 'lira',
    '8359' => 'peseta',
    '8361' => 'won',
    '8362' => 'afii57636',
    '8363' => 'dong',
    '8364' => 'Euro',
    '8411' => 'tdot',
    '8412' => 'DotDot',
    '8451' => 'centigrade',
    '8453' => 'afii61248',
    '8457' => 'fahrenheit',
    '8459' => 'hamilt',
    '8463' => 'planck',
    '8465' => 'Ifraktur',
    '8466' => 'lagran',
    '8467' => 'afii61289',
    '8470' => 'afii61352',
    '8471' => 'copysr',
    '8472' => 'weierstrass',
    '8476' => 'Rfraktur',
    '8478' => 'prescription',
    '8481' => 'telephone',
    '8482' => 'trademark',
    '8486' => 'Omega',
    '8491' => 'angstrom',
    '8492' => 'bernou',
    '8494' => 'estimated',
    '8499' => 'phmmat',
    '8500' => 'order',
    '8501' => 'aleph',
    '8502' => 'beth',
    '8503' => 'gimel',
    '8504' => 'daleth',
    '8531' => 'onethird',
    '8532' => 'twothirds',
    '8533' => 'frac15',
    '8534' => 'frac25',
    '8535' => 'frac35',
    '8536' => 'frac45',
    '8537' => 'frac16',
    '8538' => 'frac56',
    '8539' => 'oneeighth',
    '8540' => 'threeeighths',
    '8541' => 'fiveeighths',
    '8542' => 'seveneighths',
    '8544' => 'Oneroman',
    '8545' => 'Tworoman',
    '8546' => 'Threeroman',
    '8547' => 'Fourroman',
    '8548' => 'Fiveroman',
    '8549' => 'Sixroman',
    '8550' => 'Sevenroman',
    '8551' => 'Eightroman',
    '8552' => 'Nineroman',
    '8553' => 'Tenroman',
    '8554' => 'Elevenroman',
    '8555' => 'Twelveroman',
    '8560' => 'oneroman',
    '8561' => 'tworoman',
    '8562' => 'threeroman',
    '8563' => 'fourroman',
    '8564' => 'fiveroman',
    '8565' => 'sixroman',
    '8566' => 'sevenroman',
    '8567' => 'eightroman',
    '8568' => 'nineroman',
    '8569' => 'tenroman',
    '8570' => 'elevenroman',
    '8571' => 'twelveroman',
    '8592' => 'arrowleft',
    '8593' => 'arrowup',
    '8594' => 'arrowright',
    '8595' => 'arrowdown',
    '8596' => 'arrowboth',
    '8597' => 'arrowupdn',
    '8598' => 'arrowupleft',
    '8599' => 'arrowupright',
    '8600' => 'arrowdownright',
    '8601' => 'arrowdownleft',
    '8602' => 'nlarr',
    '8603' => 'nrarr',
    '8605' => 'rarrw',
    '8606' => 'Larr',
    '8608' => 'Rarr',
    '8610' => 'larrtl',
    '8611' => 'rarrtl',
    '8614' => 'map',
    '8616' => 'arrowupdnbse',
    '8617' => 'larrhk',
    '8618' => 'rarrhk',
    '8619' => 'larrlp',
    '8620' => 'rarrlp',
    '8621' => 'harrw',
    '8622' => 'nharr',
    '8624' => 'lsh',
    '8625' => 'rsh',
    '8629' => 'carriagereturn',
    '8630' => 'cularr',
    '8631' => 'curarr',
    '8634' => 'olarr',
    '8635' => 'orarr',
    '8636' => 'harpoonleftbarbup',
    '8637' => 'lhard',
    '8638' => 'uharr',
    '8639' => 'uharl',
    '8640' => 'harpoonrightbarbup',
    '8641' => 'rhard',
    '8642' => 'dharr',
    '8643' => 'dharl',
    '8644' => 'arrowrightoverleft',
    '8645' => 'arrowupleftofdown',
    '8646' => 'arrowleftoverright',
    '8647' => 'larr2',
    '8648' => 'uarr2',
    '8649' => 'rarr2',
    '8650' => 'darr2',
    '8651' => 'lrhar2',
    '8652' => 'rlhar2',
    '8653' => 'arrowleftdblstroke',
    '8654' => 'nhArr',
    '8655' => 'arrowrightdblstroke',
    '8656' => 'arrowdblleft',
    '8657' => 'arrowdblup',
    '8658' => 'arrowdblright',
    '8659' => 'arrowdbldown',
    '8660' => 'arrowdblboth',
    '8661' => 'vArr',
    '8666' => 'lAarr',
    '8667' => 'rAarr',
    '8670' => 'pageup',
    '8671' => 'pagedown',
    '8672' => 'arrowdashleft',
    '8673' => 'arrowdashup',
    '8674' => 'arrowdashright',
    '8675' => 'arrowdashdown',
    '8676' => 'arrowtableft',
    '8677' => 'arrowtabright',
    '8678' => 'arrowleftwhite',
    '8679' => 'arrowupwhite',
    '8680' => 'arrowrightwhite',
    '8681' => 'arrowdownwhite',
    '8682' => 'capslock',
    '8704' => 'universal',
    '8705' => 'comp',
    '8706' => 'partialdiff',
    '8707' => 'existential',
    '8708' => 'nexist',
    '8709' => 'emptyset',
    '8710' => 'Delta',
    '8711' => 'gradient',
    '8712' => 'element',
    '8713' => 'notelement',
    '8714' => 'epsis',
    '8715' => 'suchthat',
    '8716' => 'notcontains',
    '8717' => 'bepsi',
    '8719' => 'product',
    '8720' => 'samalg',
    '8721' => 'summation',
    '8722' => 'minus',
    '8723' => 'minusplus',
    '8724' => 'plusdo',
    '8725' => 'fraction',
    '8726' => 'setmn',
    '8727' => 'asteriskmath',
    '8728' => 'compfn',
    '8729' => 'periodcentered',
    '8730' => 'radical',
    '8733' => 'proportional',
    '8734' => 'infinity',
    '8735' => 'orthogonal',
    '8736' => 'angle',
    '8737' => 'angmsd',
    '8738' => 'angsph',
    '8739' => 'divides',
    '8740' => 'nmid',
    '8741' => 'parallel',
    '8742' => 'notparallel',
    '8743' => 'logicaland',
    '8744' => 'logicalor',
    '8745' => 'intersection',
    '8746' => 'union',
    '8747' => 'integral',
    '8748' => 'dblintegral',
    '8750' => 'contourintegral',
    '8756' => 'therefore',
    '8757' => 'because',
    '8758' => 'ratio',
    '8759' => 'proportion',
    '8764' => 'similar',
    '8765' => 'reversedtilde',
    '8768' => 'wreath',
    '8769' => 'nsim',
    '8771' => 'asymptoticallyequal',
    '8772' => 'nsime',
    '8773' => 'congruent',
    '8775' => 'ncong',
    '8776' => 'approxequal',
    '8777' => 'nap',
    '8778' => 'ape',
    '8780' => 'allequal',
    '8782' => 'bump',
    '8783' => 'bumpe',
    '8784' => 'approaches',
    '8785' => 'geometricallyequal',
    '8786' => 'approxequalorimage',
    '8787' => 'imageorapproximatelyequal',
    '8788' => 'colone',
    '8789' => 'ecolon',
    '8790' => 'ecir',
    '8791' => 'cire',
    '8793' => 'wedgeq',
    '8796' => 'trie',
    '8800' => 'notequal',
    '8801' => 'equivalence',
    '8802' => 'notidentical',
    '8804' => 'lessequal',
    '8805' => 'greaterequal',
    '8806' => 'lessoverequal',
    '8807' => 'greateroverequal',
    '8808' => 'lvnE',
    '8809' => 'gvnE',
    '8810' => 'muchless',
    '8811' => 'muchgreater',
    '8812' => 'twixt',
    '8814' => 'notless',
    '8815' => 'notgreater',
    '8816' => 'notlessnorequal',
    '8817' => 'notgreaternorequal',
    '8818' => 'lessorequivalent',
    '8819' => 'greaterorequivalent',
    '8822' => 'lessorgreater',
    '8823' => 'greaterorless',
    '8825' => 'notgreaternorless',
    '8826' => 'precedes',
    '8827' => 'succeeds',
    '8828' => 'cupre',
    '8829' => 'sccue',
    '8830' => 'prsim',
    '8831' => 'scsim',
    '8832' => 'notprecedes',
    '8833' => 'notsucceeds',
    '8834' => 'propersubset',
    '8835' => 'propersuperset',
    '8836' => 'notsubset',
    '8837' => 'notsuperset',
    '8838' => 'reflexsubset',
    '8839' => 'reflexsuperset',
    '8840' => 'nsube',
    '8841' => 'nsupe',
    '8842' => 'subsetnotequal',
    '8843' => 'supersetnotequal',
    '8846' => 'uplus',
    '8847' => 'sqsub',
    '8848' => 'sqsup',
    '8849' => 'sqsube',
    '8850' => 'sqsupe',
    '8851' => 'sqcap',
    '8852' => 'sqcup',
    '8853' => 'circleplus',
    '8854' => 'minuscircle',
    '8855' => 'circlemultiply',
    '8856' => 'osol',
    '8857' => 'circleot',
    '8858' => 'ocir',
    '8859' => 'oast',
    '8861' => 'odash',
    '8862' => 'plusb',
    '8863' => 'minusb',
    '8864' => 'timesb',
    '8865' => 'sdotb',
    '8866' => 'vdash',
    '8867' => 'tackleft',
    '8868' => 'tackdown',
    '8869' => 'perpendicular',
    '8871' => 'models',
    '8872' => 'vDash',
    '8873' => 'Vdash',
    '8874' => 'Vvdash',
    '8876' => 'nvdash',
    '8877' => 'nvDash',
    '8878' => 'nVdash',
    '8879' => 'nVDash',
    '8882' => 'vltri',
    '8883' => 'vrtri',
    '8884' => 'ltrie',
    '8885' => 'rtrie',
    '8888' => 'mumap',
    '8890' => 'intcal',
    '8891' => 'veebar',
    '8892' => 'barwed',
    '8895' => 'righttriangle',
    '8900' => 'diam',
    '8901' => 'dotmath',
    '8902' => 'sstarf',
    '8903' => 'divonx',
    '8904' => 'bowtie',
    '8905' => 'ltimes',
    '8906' => 'rtimes',
    '8907' => 'lthree',
    '8908' => 'rthree',
    '8909' => 'bsime',
    '8910' => 'curlyor',
    '8911' => 'curlyand',
    '8912' => 'Sub',
    '8913' => 'Sup',
    '8914' => 'Cap',
    '8915' => 'Cup',
    '8916' => 'fork',
    '8919' => 'gsdot',
    '8920' => 'Ll',
    '8921' => 'Gg',
    '8922' => 'lessequalorgreater',
    '8923' => 'greaterequalorless',
    '8924' => 'els',
    '8925' => 'egs',
    '8926' => 'cuepr',
    '8927' => 'cuesc',
    '8928' => 'npre',
    '8929' => 'nsce',
    '8934' => 'lnsim',
    '8935' => 'gnsim',
    '8936' => 'prnsim',
    '8937' => 'scnsim',
    '8938' => 'nltri',
    '8939' => 'nrtri',
    '8940' => 'nltrie',
    '8941' => 'nrtrie',
    '8942' => 'ellipsisvertical',
    '8962' => 'house',
    '8963' => 'control',
    '8965' => 'projective',
    '8966' => 'Barwed',
    '8968' => 'lceil',
    '8969' => 'rceil',
    '8970' => 'lfloor',
    '8971' => 'rfloor',
    '8972' => 'drcrop',
    '8973' => 'dlcrop',
    '8974' => 'urcrop',
    '8975' => 'ulcrop',
    '8976' => 'revlogicalnot',
    '8978' => 'arc',
    '8981' => 'telrec',
    '8984' => 'propellor',
    '8988' => 'ulcorn',
    '8989' => 'urcorn',
    '8990' => 'dlcorn',
    '8991' => 'drcorn',
    '8992' => 'integraltp',
    '8993' => 'integralbt',
    '8994' => 'frown',
    '8995' => 'smile',
    '8997' => 'option',
    '8998' => 'deleteright',
    '8999' => 'clear',
    '9001' => 'angleleft',
    '9002' => 'angleright',
    '9003' => 'deleteleft',
    '9251' => 'blank',
    '9312' => 'onecircle',
    '9313' => 'twocircle',
    '9314' => 'threecircle',
    '9315' => 'fourcircle',
    '9316' => 'fivecircle',
    '9317' => 'sixcircle',
    '9318' => 'sevencircle',
    '9319' => 'eightcircle',
    '9320' => 'ninecircle',
    '9321' => 'tencircle',
    '9322' => 'elevencircle',
    '9323' => 'twelvecircle',
    '9324' => 'thirteencircle',
    '9325' => 'fourteencircle',
    '9326' => 'fifteencircle',
    '9327' => 'sixteencircle',
    '9328' => 'seventeencircle',
    '9329' => 'eighteencircle',
    '9330' => 'nineteencircle',
    '9331' => 'twentycircle',
    '9332' => 'oneparen',
    '9333' => 'twoparen',
    '9334' => 'threeparen',
    '9335' => 'fourparen',
    '9336' => 'fiveparen',
    '9337' => 'sixparen',
    '9338' => 'sevenparen',
    '9339' => 'eightparen',
    '9340' => 'nineparen',
    '9341' => 'tenparen',
    '9342' => 'elevenparen',
    '9343' => 'twelveparen',
    '9344' => 'thirteenparen',
    '9345' => 'fourteenparen',
    '9346' => 'fifteenparen',
    '9347' => 'sixteenparen',
    '9348' => 'seventeenparen',
    '9349' => 'eighteenparen',
    '9350' => 'nineteenparen',
    '9351' => 'twentyparen',
    '9352' => 'oneperiod',
    '9353' => 'twoperiod',
    '9354' => 'threeperiod',
    '9355' => 'fourperiod',
    '9356' => 'fiveperiod',
    '9357' => 'sixperiod',
    '9358' => 'sevenperiod',
    '9359' => 'eightperiod',
    '9360' => 'nineperiod',
    '9361' => 'tenperiod',
    '9362' => 'elevenperiod',
    '9363' => 'twelveperiod',
    '9364' => 'thirteenperiod',
    '9365' => 'fourteenperiod',
    '9366' => 'fifteenperiod',
    '9367' => 'sixteenperiod',
    '9368' => 'seventeenperiod',
    '9369' => 'eighteenperiod',
    '9370' => 'nineteenperiod',
    '9371' => 'twentyperiod',
    '9372' => 'aparen',
    '9373' => 'bparen',
    '9374' => 'cparen',
    '9375' => 'dparen',
    '9376' => 'eparen',
    '9377' => 'fparen',
    '9378' => 'gparen',
    '9379' => 'hparen',
    '9380' => 'iparen',
    '9381' => 'jparen',
    '9382' => 'kparen',
    '9383' => 'lparen',
    '9384' => 'mparen',
    '9385' => 'nparen',
    '9386' => 'oparen',
    '9387' => 'pparen',
    '9388' => 'qparen',
    '9389' => 'rparen',
    '9390' => 'sparen',
    '9391' => 'tparen',
    '9392' => 'uparen',
    '9393' => 'vparen',
    '9394' => 'wparen',
    '9395' => 'xparen',
    '9396' => 'yparen',
    '9397' => 'zparen',
    '9398' => 'Acircle',
    '9399' => 'Bcircle',
    '9400' => 'Ccircle',
    '9401' => 'Dcircle',
    '9402' => 'Ecircle',
    '9403' => 'Fcircle',
    '9404' => 'Gcircle',
    '9405' => 'Hcircle',
    '9406' => 'Icircle',
    '9407' => 'Jcircle',
    '9408' => 'Kcircle',
    '9409' => 'Lcircle',
    '9410' => 'Mcircle',
    '9411' => 'Ncircle',
    '9412' => 'Ocircle',
    '9413' => 'Pcircle',
    '9414' => 'Qcircle',
    '9415' => 'Rcircle',
    '9416' => 'Scircle',
    '9417' => 'Tcircle',
    '9418' => 'Ucircle',
    '9419' => 'Vcircle',
    '9420' => 'Wcircle',
    '9421' => 'Xcircle',
    '9422' => 'Ycircle',
    '9423' => 'Zcircle',
    '9424' => 'acircle',
    '9425' => 'bcircle',
    '9426' => 'ccircle',
    '9427' => 'dcircle',
    '9428' => 'ecircle',
    '9429' => 'fcircle',
    '9430' => 'gcircle',
    '9431' => 'hcircle',
    '9432' => 'icircle',
    '9433' => 'jcircle',
    '9434' => 'kcircle',
    '9435' => 'lcircle',
    '9436' => 'mcircle',
    '9437' => 'ncircle',
    '9438' => 'ocircle',
    '9439' => 'pcircle',
    '9440' => 'qcircle',
    '9441' => 'rcircle',
    '9442' => 'scircle',
    '9443' => 'tcircle',
    '9444' => 'ucircle',
    '9445' => 'vcircle',
    '9446' => 'wcircle',
    '9447' => 'xcircle',
    '9448' => 'ycircle',
    '9449' => 'zcircle',
    '9472' => 'SF100000',
    '9474' => 'SF110000',
    '9484' => 'SF010000',
    '9488' => 'SF030000',
    '9492' => 'SF020000',
    '9496' => 'SF040000',
    '9500' => 'SF080000',
    '9508' => 'SF090000',
    '9516' => 'SF060000',
    '9524' => 'SF070000',
    '9532' => 'SF050000',
    '9552' => 'SF430000',
    '9553' => 'SF240000',
    '9554' => 'SF510000',
    '9555' => 'SF520000',
    '9556' => 'SF390000',
    '9557' => 'SF220000',
    '9558' => 'SF210000',
    '9559' => 'SF250000',
    '9560' => 'SF500000',
    '9561' => 'SF490000',
    '9562' => 'SF380000',
    '9563' => 'SF280000',
    '9564' => 'SF270000',
    '9565' => 'SF260000',
    '9566' => 'SF360000',
    '9567' => 'SF370000',
    '9568' => 'SF420000',
    '9569' => 'SF190000',
    '9570' => 'SF200000',
    '9571' => 'SF230000',
    '9572' => 'SF470000',
    '9573' => 'SF480000',
    '9574' => 'SF410000',
    '9575' => 'SF450000',
    '9576' => 'SF460000',
    '9577' => 'SF400000',
    '9578' => 'SF540000',
    '9579' => 'SF530000',
    '9580' => 'SF440000',
    '9600' => 'upblock',
    '9604' => 'dnblock',
    '9608' => 'block',
    '9612' => 'lfblock',
    '9616' => 'rtblock',
    '9617' => 'ltshade',
    '9618' => 'shade',
    '9619' => 'dkshade',
    '9632' => 'filledbox',
    '9633' => 'H22073',
    '9635' => 'squarewhitewithsmallblack',
    '9636' => 'squarehorizontalfill',
    '9637' => 'squareverticalfill',
    '9638' => 'squareorthogonalcrosshatchfill',
    '9639' => 'squareupperlefttolowerrightfill',
    '9640' => 'squareupperrighttolowerleftfill',
    '9641' => 'squarediagonalcrosshatchfill',
    '9642' => 'H18543',
    '9643' => 'H18551',
    '9644' => 'filledrect',
    '9645' => 'rect',
    '9646' => 'marker',
    '9650' => 'triagup',
    '9651' => 'whiteuppointingtriangle',
    '9652' => 'blackuppointingsmalltriangle',
    '9653' => 'whiteuppointingsmalltriangle',
    '9654' => 'blackrightpointingtriangle',
    '9655' => 'whiterightpointingtriangle',
    '9656' => 'rtrif',
    '9657' => 'whiterightpointingsmalltriangle',
    '9658' => 'triagrt',
    '9660' => 'triagdn',
    '9661' => 'whitedownpointingtriangle',
    '9662' => 'dtrif',
    '9663' => 'whitedownpointingsmalltriangle',
    '9664' => 'blackleftpointingtriangle',
    '9665' => 'whiteleftpointingtriangle',
    '9666' => 'ltrif',
    '9667' => 'whiteleftpointingsmalltriangle',
    '9668' => 'triaglf',
    '9670' => 'blackdiamond',
    '9671' => 'whitediamond',
    '9672' => 'whitediamondcontainingblacksmalldiamond',
    '9673' => 'fisheye',
    '9674' => 'lozenge',
    '9675' => 'circle',
    '9676' => 'dottedcircle',
    '9678' => 'bullseye',
    '9679' => 'H18533',
    '9680' => 'circlewithlefthalfblack',
    '9681' => 'circlewithrighthalfblack',
    '9687' => 'a81',
    '9688' => 'invbullet',
    '9689' => 'invcircle',
    '9698' => 'blacklowerrighttriangle',
    '9699' => 'blacklowerlefttriangle',
    '9700' => 'blackupperlefttriangle',
    '9701' => 'blackupperrighttriangle',
    '9702' => 'openbullet',
    '9711' => 'largecircle',
    '9733' => 'blackstar',
    '9734' => 'whitestar',
    '9742' => 'telephoneblack',
    '9743' => 'whitetelephone',
    '9755' => 'a11',
    '9756' => 'pointingindexleftwhite',
    '9757' => 'pointingindexupwhite',
    '9758' => 'pointingindexrightwhite',
    '9759' => 'pointingindexdownwhite',
    '9775' => 'yinyang',
    '9786' => 'smileface',
    '9787' => 'invsmileface',
    '9788' => 'sun',
    '9792' => 'female',
    '9793' => 'earth',
    '9794' => 'male',
    '9824' => 'spade',
    '9825' => 'heartsuitwhite',
    '9826' => 'diamondsuitwhite',
    '9827' => 'club',
    '9828' => 'spadesuitwhite',
    '9829' => 'heart',
    '9830' => 'diamond',
    '9831' => 'clubsuitwhite',
    '9832' => 'hotsprings',
    '9833' => 'quarternote',
    '9834' => 'musicalnote',
    '9835' => 'musicalnotedbl',
    '9836' => 'beamedsixteenthnotes',
    '9837' => 'musicflatsign',
    '9838' => 'natur',
    '9839' => 'musicsharpsign',
    '9985' => 'a1',
    '9986' => 'a2',
    '9987' => 'a202',
    '9988' => 'a3',
    '9990' => 'a5',
    '9991' => 'a119',
    '9992' => 'a118',
    '9993' => 'a117',
    '9996' => 'a13',
    '9997' => 'a14',
    '9998' => 'a15',
    '9999' => 'a16',
    '10000' => 'a105',
    '10001' => 'a17',
    '10002' => 'a18',
    '10003' => 'checkmark',
    '10004' => 'a20',
    '10005' => 'a21',
    '10006' => 'a22',
    '10007' => 'a23',
    '10008' => 'a24',
    '10009' => 'a25',
    '10010' => 'a26',
    '10011' => 'a27',
    '10012' => 'a28',
    '10013' => 'a6',
    '10014' => 'a7',
    '10015' => 'a8',
    '10016' => 'a9',
    '10017' => 'a10',
    '10018' => 'a29',
    '10019' => 'a30',
    '10020' => 'a31',
    '10021' => 'a32',
    '10022' => 'a33',
    '10023' => 'a34',
    '10025' => 'a36',
    '10026' => 'a37',
    '10027' => 'a38',
    '10028' => 'a39',
    '10029' => 'a40',
    '10030' => 'a41',
    '10031' => 'a42',
    '10032' => 'a43',
    '10033' => 'a44',
    '10034' => 'a45',
    '10035' => 'a46',
    '10036' => 'a47',
    '10037' => 'a48',
    '10038' => 'a49',
    '10039' => 'a50',
    '10040' => 'a51',
    '10041' => 'a52',
    '10042' => 'a53',
    '10043' => 'a54',
    '10044' => 'a55',
    '10045' => 'a56',
    '10046' => 'a57',
    '10047' => 'a58',
    '10048' => 'a59',
    '10049' => 'a60',
    '10050' => 'a61',
    '10051' => 'a62',
    '10052' => 'a63',
    '10053' => 'a64',
    '10054' => 'a65',
    '10055' => 'a66',
    '10056' => 'a67',
    '10057' => 'a68',
    '10058' => 'a69',
    '10059' => 'a70',
    '10061' => 'a72',
    '10063' => 'a74',
    '10064' => 'a203',
    '10065' => 'a75',
    '10066' => 'a204',
    '10070' => 'a79',
    '10072' => 'a82',
    '10073' => 'a83',
    '10074' => 'a84',
    '10075' => 'a97',
    '10076' => 'a98',
    '10077' => 'a99',
    '10078' => 'a100',
    '10081' => 'a101',
    '10082' => 'a102',
    '10083' => 'a103',
    '10084' => 'a104',
    '10085' => 'a106',
    '10086' => 'a107',
    '10087' => 'a108',
    '10102' => 'a130',
    '10103' => 'a131',
    '10104' => 'a132',
    '10105' => 'a133',
    '10106' => 'a134',
    '10107' => 'a135',
    '10108' => 'a136',
    '10109' => 'a137',
    '10110' => 'a138',
    '10111' => 'a139',
    '10112' => 'a140',
    '10113' => 'a141',
    '10114' => 'a142',
    '10115' => 'a143',
    '10116' => 'a144',
    '10117' => 'a145',
    '10118' => 'a146',
    '10119' => 'a147',
    '10120' => 'a148',
    '10121' => 'a149',
    '10122' => 'onecircleinversesansserif',
    '10123' => 'twocircleinversesansserif',
    '10124' => 'threecircleinversesansserif',
    '10125' => 'fourcircleinversesansserif',
    '10126' => 'fivecircleinversesansserif',
    '10127' => 'sixcircleinversesansserif',
    '10128' => 'sevencircleinversesansserif',
    '10129' => 'eightcircleinversesansserif',
    '10130' => 'ninecircleinversesansserif',
    '10131' => 'a159',
    '10132' => 'a160',
    '10136' => 'a196',
    '10137' => 'a165',
    '10138' => 'a192',
    '10139' => 'a166',
    '10140' => 'a167',
    '10141' => 'a168',
    '10142' => 'arrowrightheavy',
    '10143' => 'a170',
    '10144' => 'a171',
    '10145' => 'a172',
    '10146' => 'a173',
    '10147' => 'a162',
    '10148' => 'a174',
    '10149' => 'a175',
    '10150' => 'a176',
    '10151' => 'a177',
    '10152' => 'a178',
    '10153' => 'a179',
    '10154' => 'a193',
    '10155' => 'a180',
    '10156' => 'a199',
    '10157' => 'a181',
    '10158' => 'a200',
    '10159' => 'a182',
    '10161' => 'a201',
    '10162' => 'a183',
    '10163' => 'a184',
    '10164' => 'a197',
    '10165' => 'a185',
    '10166' => 'a194',
    '10167' => 'a198',
    '10168' => 'a186',
    '10169' => 'a195',
    '10170' => 'a187',
    '10171' => 'a188',
    '10172' => 'a189',
    '10173' => 'a190',
    '10174' => 'a191',
    '12288' => 'ideographicspace',
    '12289' => 'ideographiccomma',
    '12290' => 'ideographicperiod',
    '12291' => 'dittomark',
    '12292' => 'jis',
    '12293' => 'ideographiciterationmark',
    '12294' => 'ideographicclose',
    '12295' => 'ideographiczero',
    '12296' => 'anglebracketleft',
    '12297' => 'anglebracketright',
    '12298' => 'dblanglebracketleft',
    '12299' => 'dblanglebracketright',
    '12300' => 'cornerbracketleft',
    '12301' => 'cornerbracketright',
    '12302' => 'whitecornerbracketleft',
    '12303' => 'whitecornerbracketright',
    '12304' => 'blacklenticularbracketleft',
    '12305' => 'blacklenticularbracketright',
    '12306' => 'postalmark',
    '12307' => 'getamark',
    '12308' => 'tortoiseshellbracketleft',
    '12309' => 'tortoiseshellbracketright',
    '12310' => 'whitelenticularbracketleft',
    '12311' => 'whitelenticularbracketright',
    '12312' => 'whitetortoiseshellbracketleft',
    '12313' => 'whitetortoiseshellbracketright',
    '12316' => 'wavedash',
    '12317' => 'quotedblprimereversed',
    '12318' => 'quotedblprime',
    '12320' => 'postalmarkface',
    '12321' => 'onehangzhou',
    '12322' => 'twohangzhou',
    '12323' => 'threehangzhou',
    '12324' => 'fourhangzhou',
    '12325' => 'fivehangzhou',
    '12326' => 'sixhangzhou',
    '12327' => 'sevenhangzhou',
    '12328' => 'eighthangzhou',
    '12329' => 'ninehangzhou',
    '12342' => 'circlepostalmark',
    '12353' => 'asmallhiragana',
    '12354' => 'ahiragana',
    '12355' => 'ismallhiragana',
    '12356' => 'ihiragana',
    '12357' => 'usmallhiragana',
    '12358' => 'uhiragana',
    '12359' => 'esmallhiragana',
    '12360' => 'ehiragana',
    '12361' => 'osmallhiragana',
    '12362' => 'ohiragana',
    '12363' => 'kahiragana',
    '12364' => 'gahiragana',
    '12365' => 'kihiragana',
    '12366' => 'gihiragana',
    '12367' => 'kuhiragana',
    '12368' => 'guhiragana',
    '12369' => 'kehiragana',
    '12370' => 'gehiragana',
    '12371' => 'kohiragana',
    '12372' => 'gohiragana',
    '12373' => 'sahiragana',
    '12374' => 'zahiragana',
    '12375' => 'sihiragana',
    '12376' => 'zihiragana',
    '12377' => 'suhiragana',
    '12378' => 'zuhiragana',
    '12379' => 'sehiragana',
    '12380' => 'zehiragana',
    '12381' => 'sohiragana',
    '12382' => 'zohiragana',
    '12383' => 'tahiragana',
    '12384' => 'dahiragana',
    '12385' => 'tihiragana',
    '12386' => 'dihiragana',
    '12387' => 'tusmallhiragana',
    '12388' => 'tuhiragana',
    '12389' => 'duhiragana',
    '12390' => 'tehiragana',
    '12391' => 'dehiragana',
    '12392' => 'tohiragana',
    '12393' => 'dohiragana',
    '12394' => 'nahiragana',
    '12395' => 'nihiragana',
    '12396' => 'nuhiragana',
    '12397' => 'nehiragana',
    '12398' => 'nohiragana',
    '12399' => 'hahiragana',
    '12400' => 'bahiragana',
    '12401' => 'pahiragana',
    '12402' => 'hihiragana',
    '12403' => 'bihiragana',
    '12404' => 'pihiragana',
    '12405' => 'huhiragana',
    '12406' => 'buhiragana',
    '12407' => 'puhiragana',
    '12408' => 'hehiragana',
    '12409' => 'behiragana',
    '12410' => 'pehiragana',
    '12411' => 'hohiragana',
    '12412' => 'bohiragana',
    '12413' => 'pohiragana',
    '12414' => 'mahiragana',
    '12415' => 'mihiragana',
    '12416' => 'muhiragana',
    '12417' => 'mehiragana',
    '12418' => 'mohiragana',
    '12419' => 'yasmallhiragana',
    '12420' => 'yahiragana',
    '12421' => 'yusmallhiragana',
    '12422' => 'yuhiragana',
    '12423' => 'yosmallhiragana',
    '12424' => 'yohiragana',
    '12425' => 'rahiragana',
    '12426' => 'rihiragana',
    '12427' => 'ruhiragana',
    '12428' => 'rehiragana',
    '12429' => 'rohiragana',
    '12430' => 'wasmallhiragana',
    '12431' => 'wahiragana',
    '12432' => 'wihiragana',
    '12433' => 'wehiragana',
    '12434' => 'wohiragana',
    '12435' => 'nhiragana',
    '12436' => 'vuhiragana',
    '12443' => 'voicedmarkkana',
    '12444' => 'semivoicedmarkkana',
    '12445' => 'iterationhiragana',
    '12446' => 'voicediterationhiragana',
    '12449' => 'asmallkatakana',
    '12450' => 'akatakana',
    '12451' => 'ismallkatakana',
    '12452' => 'ikatakana',
    '12453' => 'usmallkatakana',
    '12454' => 'ukatakana',
    '12455' => 'esmallkatakana',
    '12456' => 'ekatakana',
    '12457' => 'osmallkatakana',
    '12458' => 'okatakana',
    '12459' => 'kakatakana',
    '12460' => 'gakatakana',
    '12461' => 'kikatakana',
    '12462' => 'gikatakana',
    '12463' => 'kukatakana',
    '12464' => 'gukatakana',
    '12465' => 'kekatakana',
    '12466' => 'gekatakana',
    '12467' => 'kokatakana',
    '12468' => 'gokatakana',
    '12469' => 'sakatakana',
    '12470' => 'zakatakana',
    '12471' => 'sikatakana',
    '12472' => 'zikatakana',
    '12473' => 'sukatakana',
    '12474' => 'zukatakana',
    '12475' => 'sekatakana',
    '12476' => 'zekatakana',
    '12477' => 'sokatakana',
    '12478' => 'zokatakana',
    '12479' => 'takatakana',
    '12480' => 'dakatakana',
    '12481' => 'tikatakana',
    '12482' => 'dikatakana',
    '12483' => 'tusmallkatakana',
    '12484' => 'tukatakana',
    '12485' => 'dukatakana',
    '12486' => 'tekatakana',
    '12487' => 'dekatakana',
    '12488' => 'tokatakana',
    '12489' => 'dokatakana',
    '12490' => 'nakatakana',
    '12491' => 'nikatakana',
    '12492' => 'nukatakana',
    '12493' => 'nekatakana',
    '12494' => 'nokatakana',
    '12495' => 'hakatakana',
    '12496' => 'bakatakana',
    '12497' => 'pakatakana',
    '12498' => 'hikatakana',
    '12499' => 'bikatakana',
    '12500' => 'pikatakana',
    '12501' => 'hukatakana',
    '12502' => 'bukatakana',
    '12503' => 'pukatakana',
    '12504' => 'hekatakana',
    '12505' => 'bekatakana',
    '12506' => 'pekatakana',
    '12507' => 'hokatakana',
    '12508' => 'bokatakana',
    '12509' => 'pokatakana',
    '12510' => 'makatakana',
    '12511' => 'mikatakana',
    '12512' => 'mukatakana',
    '12513' => 'mekatakana',
    '12514' => 'mokatakana',
    '12515' => 'yasmallkatakana',
    '12516' => 'yakatakana',
    '12517' => 'yusmallkatakana',
    '12518' => 'yukatakana',
    '12519' => 'yosmallkatakana',
    '12520' => 'yokatakana',
    '12521' => 'rakatakana',
    '12522' => 'rikatakana',
    '12523' => 'rukatakana',
    '12524' => 'rekatakana',
    '12525' => 'rokatakana',
    '12526' => 'wasmallkatakana',
    '12527' => 'wakatakana',
    '12528' => 'wikatakana',
    '12529' => 'wekatakana',
    '12530' => 'wokatakana',
    '12531' => 'nkatakana',
    '12532' => 'vukatakana',
    '12533' => 'kasmallkatakana',
    '12534' => 'kesmallkatakana',
    '12535' => 'vakatakana',
    '12536' => 'vikatakana',
    '12537' => 'vekatakana',
    '12538' => 'vokatakana',
    '12539' => 'dotkatakana',
    '12540' => 'prolongedkana',
    '12541' => 'iterationkatakana',
    '12542' => 'voicediterationkatakana',
    '12549' => 'bbopomofo',
    '12550' => 'pbopomofo',
    '12551' => 'mbopomofo',
    '12552' => 'fbopomofo',
    '12553' => 'dbopomofo',
    '12554' => 'tbopomofo',
    '12555' => 'nbopomofo',
    '12556' => 'lbopomofo',
    '12557' => 'gbopomofo',
    '12558' => 'kbopomofo',
    '12559' => 'hbopomofo',
    '12560' => 'jbopomofo',
    '12561' => 'qbopomofo',
    '12562' => 'xbopomofo',
    '12563' => 'zhbopomofo',
    '12564' => 'chbopomofo',
    '12565' => 'shbopomofo',
    '12566' => 'rbopomofo',
    '12567' => 'zbopomofo',
    '12568' => 'cbopomofo',
    '12569' => 'sbopomofo',
    '12570' => 'abopomofo',
    '12571' => 'obopomofo',
    '12572' => 'ebopomofo',
    '12573' => 'ehbopomofo',
    '12574' => 'aibopomofo',
    '12575' => 'eibopomofo',
    '12576' => 'aubopomofo',
    '12577' => 'oubopomofo',
    '12578' => 'anbopomofo',
    '12579' => 'enbopomofo',
    '12580' => 'angbopomofo',
    '12581' => 'engbopomofo',
    '12582' => 'erbopomofo',
    '12583' => 'ibopomofo',
    '12584' => 'ubopomofo',
    '12585' => 'iubopomofo',
    '12593' => 'kiyeokkorean',
    '12594' => 'ssangkiyeokkorean',
    '12595' => 'kiyeoksioskorean',
    '12596' => 'nieunkorean',
    '12597' => 'nieuncieuckorean',
    '12598' => 'nieunhieuhkorean',
    '12599' => 'tikeutkorean',
    '12600' => 'ssangtikeutkorean',
    '12601' => 'rieulkorean',
    '12602' => 'rieulkiyeokkorean',
    '12603' => 'rieulmieumkorean',
    '12604' => 'rieulpieupkorean',
    '12605' => 'rieulsioskorean',
    '12606' => 'rieulthieuthkorean',
    '12607' => 'rieulphieuphkorean',
    '12608' => 'rieulhieuhkorean',
    '12609' => 'mieumkorean',
    '12610' => 'pieupkorean',
    '12611' => 'ssangpieupkorean',
    '12612' => 'pieupsioskorean',
    '12613' => 'sioskorean',
    '12614' => 'ssangsioskorean',
    '12615' => 'ieungkorean',
    '12616' => 'cieuckorean',
    '12617' => 'ssangcieuckorean',
    '12618' => 'chieuchkorean',
    '12619' => 'khieukhkorean',
    '12620' => 'thieuthkorean',
    '12621' => 'phieuphkorean',
    '12622' => 'hieuhkorean',
    '12623' => 'akorean',
    '12624' => 'aekorean',
    '12625' => 'yakorean',
    '12626' => 'yaekorean',
    '12627' => 'eokorean',
    '12628' => 'ekorean',
    '12629' => 'yeokorean',
    '12630' => 'yekorean',
    '12631' => 'okorean',
    '12632' => 'wakorean',
    '12633' => 'waekorean',
    '12634' => 'oekorean',
    '12635' => 'yokorean',
    '12636' => 'ukorean',
    '12637' => 'weokorean',
    '12638' => 'wekorean',
    '12639' => 'wikorean',
    '12640' => 'yukorean',
    '12641' => 'eukorean',
    '12642' => 'yikorean',
    '12643' => 'ikorean',
    '12644' => 'hangulfiller',
    '12645' => 'ssangnieunkorean',
    '12646' => 'nieuntikeutkorean',
    '12647' => 'nieunsioskorean',
    '12648' => 'nieunpansioskorean',
    '12649' => 'rieulkiyeoksioskorean',
    '12650' => 'rieultikeutkorean',
    '12651' => 'rieulpieupsioskorean',
    '12652' => 'rieulpansioskorean',
    '12653' => 'rieulyeorinhieuhkorean',
    '12654' => 'mieumpieupkorean',
    '12655' => 'mieumsioskorean',
    '12656' => 'mieumpansioskorean',
    '12657' => 'kapyeounmieumkorean',
    '12658' => 'pieupkiyeokkorean',
    '12659' => 'pieuptikeutkorean',
    '12660' => 'pieupsioskiyeokkorean',
    '12661' => 'pieupsiostikeutkorean',
    '12662' => 'pieupcieuckorean',
    '12663' => 'pieupthieuthkorean',
    '12664' => 'kapyeounpieupkorean',
    '12665' => 'kapyeounssangpieupkorean',
    '12666' => 'sioskiyeokkorean',
    '12667' => 'siosnieunkorean',
    '12668' => 'siostikeutkorean',
    '12669' => 'siospieupkorean',
    '12670' => 'sioscieuckorean',
    '12671' => 'pansioskorean',
    '12672' => 'ssangieungkorean',
    '12673' => 'yesieungkorean',
    '12674' => 'yesieungsioskorean',
    '12675' => 'yesieungpansioskorean',
    '12676' => 'kapyeounphieuphkorean',
    '12677' => 'ssanghieuhkorean',
    '12678' => 'yeorinhieuhkorean',
    '12679' => 'yoyakorean',
    '12680' => 'yoyaekorean',
    '12681' => 'yoikorean',
    '12682' => 'yuyeokorean',
    '12683' => 'yuyekorean',
    '12684' => 'yuikorean',
    '12685' => 'araeakorean',
    '12686' => 'araeaekorean',
    '12800' => 'kiyeokparenkorean',
    '12801' => 'nieunparenkorean',
    '12802' => 'tikeutparenkorean',
    '12803' => 'rieulparenkorean',
    '12804' => 'mieumparenkorean',
    '12805' => 'pieupparenkorean',
    '12806' => 'siosparenkorean',
    '12807' => 'ieungparenkorean',
    '12808' => 'cieucparenkorean',
    '12809' => 'chieuchparenkorean',
    '12810' => 'khieukhparenkorean',
    '12811' => 'thieuthparenkorean',
    '12812' => 'phieuphparenkorean',
    '12813' => 'hieuhparenkorean',
    '12814' => 'kiyeokaparenkorean',
    '12815' => 'nieunaparenkorean',
    '12816' => 'tikeutaparenkorean',
    '12817' => 'rieulaparenkorean',
    '12818' => 'mieumaparenkorean',
    '12819' => 'pieupaparenkorean',
    '12820' => 'siosaparenkorean',
    '12821' => 'ieungaparenkorean',
    '12822' => 'cieucaparenkorean',
    '12823' => 'chieuchaparenkorean',
    '12824' => 'khieukhaparenkorean',
    '12825' => 'thieuthaparenkorean',
    '12826' => 'phieuphaparenkorean',
    '12827' => 'hieuhaparenkorean',
    '12828' => 'cieucuparenkorean',
    '12832' => 'oneideographicparen',
    '12833' => 'twoideographicparen',
    '12834' => 'threeideographicparen',
    '12835' => 'fourideographicparen',
    '12836' => 'fiveideographicparen',
    '12837' => 'sixideographicparen',
    '12838' => 'sevenideographicparen',
    '12839' => 'eightideographicparen',
    '12840' => 'nineideographicparen',
    '12841' => 'tenideographicparen',
    '12842' => 'ideographicmoonparen',
    '12843' => 'ideographicfireparen',
    '12844' => 'ideographicwaterparen',
    '12845' => 'ideographicwoodparen',
    '12846' => 'ideographicmetalparen',
    '12847' => 'ideographicearthparen',
    '12848' => 'ideographicsunparen',
    '12849' => 'ideographicstockparen',
    '12850' => 'ideographichaveparen',
    '12851' => 'ideographicsocietyparen',
    '12852' => 'ideographicnameparen',
    '12853' => 'ideographicspecialparen',
    '12854' => 'ideographicfinancialparen',
    '12855' => 'ideographiccongratulationparen',
    '12856' => 'ideographiclaborparen',
    '12857' => 'ideographicrepresentparen',
    '12858' => 'ideographiccallparen',
    '12859' => 'ideographicstudyparen',
    '12860' => 'ideographicsuperviseparen',
    '12861' => 'ideographicenterpriseparen',
    '12862' => 'ideographicresourceparen',
    '12863' => 'ideographicallianceparen',
    '12864' => 'ideographicfestivalparen',
    '12866' => 'ideographicselfparen',
    '12867' => 'ideographicreachparen',
    '12896' => 'kiyeokcirclekorean',
    '12897' => 'nieuncirclekorean',
    '12898' => 'tikeutcirclekorean',
    '12899' => 'rieulcirclekorean',
    '12900' => 'mieumcirclekorean',
    '12901' => 'pieupcirclekorean',
    '12902' => 'sioscirclekorean',
    '12903' => 'ieungcirclekorean',
    '12904' => 'cieuccirclekorean',
    '12905' => 'chieuchcirclekorean',
    '12906' => 'khieukhcirclekorean',
    '12907' => 'thieuthcirclekorean',
    '12908' => 'phieuphcirclekorean',
    '12909' => 'hieuhcirclekorean',
    '12910' => 'kiyeokacirclekorean',
    '12911' => 'nieunacirclekorean',
    '12912' => 'tikeutacirclekorean',
    '12913' => 'rieulacirclekorean',
    '12914' => 'mieumacirclekorean',
    '12915' => 'pieupacirclekorean',
    '12916' => 'siosacirclekorean',
    '12917' => 'ieungacirclekorean',
    '12918' => 'cieucacirclekorean',
    '12919' => 'chieuchacirclekorean',
    '12920' => 'khieukhacirclekorean',
    '12921' => 'thieuthacirclekorean',
    '12922' => 'phieuphacirclekorean',
    '12923' => 'hieuhacirclekorean',
    '12927' => 'koreanstandardsymbol',
    '12938' => 'ideographmooncircle',
    '12939' => 'ideographfirecircle',
    '12940' => 'ideographwatercircle',
    '12941' => 'ideographwoodcircle',
    '12942' => 'ideographmetalcircle',
    '12943' => 'ideographearthcircle',
    '12944' => 'ideographsuncircle',
    '12948' => 'ideographnamecircle',
    '12950' => 'ideographicfinancialcircle',
    '12952' => 'ideographiclaborcircle',
    '12953' => 'ideographicsecretcircle',
    '12957' => 'ideographicexcellentcircle',
    '12958' => 'ideographicprintcircle',
    '12963' => 'ideographiccorrectcircle',
    '12964' => 'ideographichighcircle',
    '12965' => 'ideographiccentrecircle',
    '12966' => 'ideographiclowcircle',
    '12967' => 'ideographicleftcircle',
    '12968' => 'ideographicrightcircle',
    '12969' => 'ideographicmedicinecircle',
    '13056' => 'apaatosquare',
    '13059' => 'aarusquare',
    '13061' => 'intisquare',
    '13069' => 'karoriisquare',
    '13076' => 'kirosquare',
    '13077' => 'kiroguramusquare',
    '13078' => 'kiromeetorusquare',
    '13080' => 'guramusquare',
    '13086' => 'kooposquare',
    '13090' => 'sentisquare',
    '13091' => 'sentosquare',
    '13094' => 'dorusquare',
    '13095' => 'tonsquare',
    '13098' => 'haitusquare',
    '13099' => 'paasentosquare',
    '13105' => 'birusquare',
    '13107' => 'huiitosquare',
    '13110' => 'hekutaarusquare',
    '13113' => 'herutusquare',
    '13115' => 'peezisquare',
    '13122' => 'hoonsquare',
    '13127' => 'mansyonsquare',
    '13129' => 'mirisquare',
    '13130' => 'miribaarusquare',
    '13133' => 'meetorusquare',
    '13134' => 'yaadosquare',
    '13137' => 'rittorusquare',
    '13143' => 'wattosquare',
    '13179' => 'heiseierasquare',
    '13180' => 'syouwaerasquare',
    '13181' => 'taisyouerasquare',
    '13182' => 'meizierasquare',
    '13183' => 'corporationsquare',
    '13184' => 'paampssquare',
    '13185' => 'nasquare',
    '13186' => 'muasquare',
    '13187' => 'masquare',
    '13188' => 'kasquare',
    '13189' => 'KBsquare',
    '13190' => 'MBsquare',
    '13191' => 'GBsquare',
    '13192' => 'calsquare',
    '13193' => 'kcalsquare',
    '13194' => 'pfsquare',
    '13195' => 'nfsquare',
    '13196' => 'mufsquare',
    '13197' => 'mugsquare',
    '13198' => 'squaremg',
    '13199' => 'squarekg',
    '13200' => 'Hzsquare',
    '13201' => 'khzsquare',
    '13202' => 'mhzsquare',
    '13203' => 'ghzsquare',
    '13204' => 'thzsquare',
    '13205' => 'mulsquare',
    '13206' => 'mlsquare',
    '13207' => 'dlsquare',
    '13208' => 'klsquare',
    '13209' => 'fmsquare',
    '13210' => 'nmsquare',
    '13211' => 'mumsquare',
    '13212' => 'squaremm',
    '13213' => 'squarecm',
    '13214' => 'squarekm',
    '13215' => 'mmsquaredsquare',
    '13216' => 'cmsquaredsquare',
    '13217' => 'squaremsquared',
    '13218' => 'kmsquaredsquare',
    '13219' => 'mmcubedsquare',
    '13220' => 'cmcubedsquare',
    '13221' => 'mcubedsquare',
    '13222' => 'kmcubedsquare',
    '13223' => 'moverssquare',
    '13224' => 'moverssquaredsquare',
    '13225' => 'pasquare',
    '13226' => 'kpasquare',
    '13227' => 'mpasquare',
    '13228' => 'gpasquare',
    '13229' => 'radsquare',
    '13230' => 'radoverssquare',
    '13231' => 'radoverssquaredsquare',
    '13232' => 'pssquare',
    '13233' => 'nssquare',
    '13234' => 'mussquare',
    '13235' => 'mssquare',
    '13236' => 'pvsquare',
    '13237' => 'nvsquare',
    '13238' => 'muvsquare',
    '13239' => 'mvsquare',
    '13240' => 'kvsquare',
    '13241' => 'mvmegasquare',
    '13242' => 'pwsquare',
    '13243' => 'nwsquare',
    '13244' => 'muwsquare',
    '13245' => 'mwsquare',
    '13246' => 'kwsquare',
    '13247' => 'mwmegasquare',
    '13248' => 'kohmsquare',
    '13249' => 'mohmsquare',
    '13250' => 'amsquare',
    '13251' => 'bqsquare',
    '13252' => 'squarecc',
    '13253' => 'cdsquare',
    '13254' => 'coverkgsquare',
    '13255' => 'cosquare',
    '13256' => 'dbsquare',
    '13257' => 'gysquare',
    '13258' => 'hasquare',
    '13259' => 'HPsquare',
    '13261' => 'KKsquare',
    '13262' => 'squarekmcapital',
    '13263' => 'ktsquare',
    '13264' => 'lmsquare',
    '13265' => 'squareln',
    '13266' => 'squarelog',
    '13267' => 'lxsquare',
    '13268' => 'mbsquare',
    '13269' => 'squaremil',
    '13270' => 'molsquare',
    '13272' => 'pmsquare',
    '13275' => 'srsquare',
    '13276' => 'svsquare',
    '13277' => 'wbsquare',
    '21316' => 'twentyhangzhou',
    '61441' => 'fi',
    '61442' => 'fl',
    '61472' => 'space',
    '61473' => 'pencil',
    '61474' => 'scissors',
    '61475' => 'scissorscutting',
    '61476' => 'readingglasses',
    '61477' => 'bell',
    '61478' => 'book',
    '61479' => 'candle',
    '61480' => 'telephonesolid',
    '61481' => 'telhandsetcirc',
    '61482' => 'envelopeback',
    '61483' => 'envelopefront',
    '61484' => 'mailboxflagdwn',
    '61485' => 'mailboxflagup',
    '61486' => 'mailbxopnflgup',
    '61487' => 'mailbxopnflgdwn',
    '61488' => 'folder',
    '61489' => 'folderopen',
    '61490' => 'filetalltext1',
    '61491' => 'filetalltext',
    '61492' => 'filetalltext3',
    '61493' => 'filecabinet',
    '61494' => 'hourglass',
    '61495' => 'keyboard',
    '61496' => 'mouse2button',
    '61497' => 'ballpoint',
    '61498' => 'pc',
    '61499' => 'harddisk',
    '61500' => 'floppy3',
    '61501' => 'floppy5',
    '61502' => 'tapereel',
    '61503' => 'handwrite',
    '61504' => 'handwriteleft',
    '61505' => 'handv',
    '61506' => 'handok',
    '61507' => 'thumbup',
    '61508' => 'thumbdown',
    '61509' => 'handptleft',
    '61510' => 'handptright',
    '61511' => 'handptup',
    '61512' => 'handptdwn',
    '61513' => 'handhalt',
    '61514' => 'smileface',
    '61515' => 'neutralface',
    '61516' => 'frownface',
    '61517' => 'bomb',
    '61518' => 'skullcrossbones',
    '61519' => 'flag',
    '61520' => 'pennant',
    '61521' => 'airplane',
    '61522' => 'sunshine',
    '61523' => 'droplet',
    '61524' => 'snowflake',
    '61525' => 'crossoutline',
    '61526' => 'crossshadow',
    '61527' => 'crossceltic',
    '61528' => 'crossmaltese',
    '61529' => 'starofdavid',
    '61530' => 'crescentstar',
    '61531' => 'yinyang',
    '61532' => 'om',
    '61533' => 'wheel',
    '61534' => 'aries',
    '61535' => 'taurus',
    '61536' => 'gemini',
    '61537' => 'cancer',
    '61538' => 'leo',
    '61539' => 'virgo',
    '61540' => 'libra',
    '61541' => 'scorpio',
    '61542' => 'saggitarius',
    '61543' => 'capricorn',
    '61544' => 'aquarius',
    '61545' => 'pisces',
    '61546' => 'ampersanditlc',
    '61547' => 'ampersandit',
    '61548' => 'circle6',
    '61549' => 'circleshadowdwn',
    '61550' => 'square6',
    '61551' => 'box3',
    '61552' => 'box4',
    '61553' => 'boxshadowdwn',
    '61554' => 'boxshadowup',
    '61555' => 'lozenge4',
    '61556' => 'lozenge6',
    '61557' => 'rhombus6',
    '61558' => 'xrhombus',
    '61559' => 'rhombus4',
    '61560' => 'clear',
    '61561' => 'escape',
    '61562' => 'command',
    '61563' => 'rosette',
    '61564' => 'rosettesolid',
    '61565' => 'quotedbllftbld',
    '61566' => 'quotedblrtbld',
    '61568' => 'zerosans',
    '61569' => 'onesans',
    '61570' => 'twosans',
    '61571' => 'threesans',
    '61572' => 'foursans',
    '61573' => 'fivesans',
    '61574' => 'sixsans',
    '61575' => 'sevensans',
    '61576' => 'eightsans',
    '61577' => 'ninesans',
    '61578' => 'tensans',
    '61579' => 'zerosansinv',
    '61580' => 'onesansinv',
    '61581' => 'twosansinv',
    '61582' => 'threesansinv',
    '61583' => 'foursansinv',
    '61584' => 'fivesansinv',
    '61585' => 'sixsansinv',
    '61586' => 'sevensansinv',
    '61587' => 'eightsansinv',
    '61588' => 'ninesansinv',
    '61589' => 'tensansinv',
    '61590' => 'budleafne',
    '61591' => 'budleafnw',
    '61592' => 'budleafsw',
    '61593' => 'budleafse',
    '61594' => 'vineleafboldne',
    '61595' => 'vineleafboldnw',
    '61596' => 'vineleafboldsw',
    '61597' => 'vineleafboldse',
    '61598' => 'circle2',
    '61599' => 'circle4',
    '61600' => 'square2',
    '61601' => 'ring2',
    '61602' => 'ring4',
    '61603' => 'ring6',
    '61604' => 'ringbutton2',
    '61605' => 'target',
    '61606' => 'circleshadowup',
    '61607' => 'square4',
    '61608' => 'box2',
    '61609' => 'tristar2',
    '61610' => 'crosstar2',
    '61611' => 'pentastar2',
    '61612' => 'hexstar2',
    '61613' => 'octastar2',
    '61614' => 'dodecastar3',
    '61615' => 'octastar4',
    '61616' => 'registersquare',
    '61617' => 'registercircle',
    '61618' => 'cuspopen',
    '61619' => 'cuspopen1',
    '61620' => 'query',
    '61621' => 'circlestar',
    '61622' => 'starshadow',
    '61623' => 'oneoclock',
    '61624' => 'twooclock',
    '61625' => 'threeoclock',
    '61626' => 'fouroclock',
    '61627' => 'fiveoclock',
    '61628' => 'sixoclock',
    '61629' => 'sevenoclock',
    '61630' => 'eightoclock',
    '61631' => 'nineoclock',
    '61632' => 'tenoclock',
    '61633' => 'elevenoclock',
    '61634' => 'twelveoclock',
    '61635' => 'arrowdwnleft1',
    '61636' => 'arrowdwnrt1',
    '61637' => 'arrowupleft1',
    '61638' => 'arrowuprt1',
    '61639' => 'arrowleftup1',
    '61640' => 'arrowrtup1',
    '61641' => 'arrowleftdwn1',
    '61642' => 'arrowrtdwn1',
    '61643' => 'quiltsquare2',
    '61644' => 'quiltsquare2inv',
    '61645' => 'leafccwsw',
    '61646' => 'leafccwnw',
    '61647' => 'leafccwse',
    '61648' => 'leafccwne',
    '61649' => 'leafnw',
    '61650' => 'leafsw',
    '61651' => 'leafne',
    '61652' => 'leafse',
    '61653' => 'deleteleft',
    '61654' => 'deleteright',
    '61655' => 'head2left',
    '61656' => 'head2right',
    '61657' => 'head2up',
    '61658' => 'head2down',
    '61659' => 'circleleft',
    '61660' => 'circleright',
    '61661' => 'circleup',
    '61662' => 'circledown',
    '61663' => 'barb2left',
    '61664' => 'barb2right',
    '61665' => 'barb2up',
    '61666' => 'barb2down',
    '61667' => 'barb2nw',
    '61668' => 'barb2ne',
    '61669' => 'barb2sw',
    '61670' => 'barb2se',
    '61671' => 'barb4left',
    '61672' => 'barb4right',
    '61673' => 'barb4up',
    '61674' => 'barb4down',
    '61675' => 'barb4nw',
    '61676' => 'barb4ne',
    '61677' => 'barb4sw',
    '61678' => 'barb4se',
    '61679' => 'bleft',
    '61680' => 'bright',
    '61681' => 'bup',
    '61682' => 'bdown',
    '61683' => 'bleftright',
    '61684' => 'bupdown',
    '61685' => 'bnw',
    '61686' => 'bne',
    '61687' => 'bsw',
    '61688' => 'bse',
    '61689' => 'bdash1',
    '61690' => 'bdash2',
    '61691' => 'xmarkbld',
    '61692' => 'checkbld',
    '61693' => 'boxxmarkbld',
    '61694' => 'boxcheckbld',
    '61695' => 'windowslogo',
    '63166' => 'dotlessj',
    '63167' => 'LL',
    '63168' => 'll',
    '63169' => 'Scedilla',
    '63170' => 'scedilla',
    '63171' => 'commaaccent',
    '63172' => 'afii10063',
    '63173' => 'afii10064',
    '63174' => 'afii10192',
    '63175' => 'afii10831',
    '63176' => 'afii10832',
    '63177' => 'Acute',
    '63178' => 'Caron',
    '63179' => 'Dieresis',
    '63180' => 'DieresisAcute',
    '63181' => 'DieresisGrave',
    '63182' => 'Grave',
    '63183' => 'Hungarumlaut',
    '63184' => 'Macron',
    '63185' => 'cyrBreve',
    '63186' => 'cyrFlex',
    '63187' => 'dblGrave',
    '63188' => 'cyrbreve',
    '63189' => 'cyrflex',
    '63190' => 'dblgrave',
    '63191' => 'dieresisacute',
    '63192' => 'dieresisgrave',
    '63193' => 'copyrightserif',
    '63194' => 'registerserif',
    '63195' => 'trademarkserif',
    '63196' => 'onefitted',
    '63197' => 'rupiah',
    '63198' => 'threequartersemdash',
    '63199' => 'centinferior',
    '63200' => 'centsuperior',
    '63201' => 'commainferior',
    '63202' => 'commasuperior',
    '63203' => 'dollarinferior',
    '63204' => 'dollarsuperior',
    '63205' => 'hypheninferior',
    '63206' => 'hyphensuperior',
    '63207' => 'periodinferior',
    '63208' => 'periodsuperior',
    '63209' => 'asuperior',
    '63210' => 'bsuperior',
    '63211' => 'dsuperior',
    '63212' => 'esuperior',
    '63213' => 'isuperior',
    '63214' => 'lsuperior',
    '63215' => 'msuperior',
    '63216' => 'osuperior',
    '63217' => 'rsuperior',
    '63218' => 'ssuperior',
    '63219' => 'tsuperior',
    '63220' => 'Brevesmall',
    '63221' => 'Caronsmall',
    '63222' => 'Circumflexsmall',
    '63223' => 'Dotaccentsmall',
    '63224' => 'Hungarumlautsmall',
    '63225' => 'Lslashsmall',
    '63226' => 'OEsmall',
    '63227' => 'Ogoneksmall',
    '63228' => 'Ringsmall',
    '63229' => 'Scaronsmall',
    '63230' => 'Tildesmall',
    '63231' => 'Zcaronsmall',
    '63265' => 'exclamsmall',
    '63268' => 'dollaroldstyle',
    '63270' => 'ampersandsmall',
    '63280' => 'zerooldstyle',
    '63281' => 'oneoldstyle',
    '63282' => 'twooldstyle',
    '63283' => 'threeoldstyle',
    '63284' => 'fouroldstyle',
    '63285' => 'fiveoldstyle',
    '63286' => 'sixoldstyle',
    '63287' => 'sevenoldstyle',
    '63288' => 'eightoldstyle',
    '63289' => 'nineoldstyle',
    '63295' => 'questionsmall',
    '63328' => 'Gravesmall',
    '63329' => 'Asmall',
    '63330' => 'Bsmall',
    '63331' => 'Csmall',
    '63332' => 'Dsmall',
    '63333' => 'Esmall',
    '63334' => 'Fsmall',
    '63335' => 'Gsmall',
    '63336' => 'Hsmall',
    '63337' => 'Ismall',
    '63338' => 'Jsmall',
    '63339' => 'Ksmall',
    '63340' => 'Lsmall',
    '63341' => 'Msmall',
    '63342' => 'Nsmall',
    '63343' => 'Osmall',
    '63344' => 'Psmall',
    '63345' => 'Qsmall',
    '63346' => 'Rsmall',
    '63347' => 'Ssmall',
    '63348' => 'Tsmall',
    '63349' => 'Usmall',
    '63350' => 'Vsmall',
    '63351' => 'Wsmall',
    '63352' => 'Xsmall',
    '63353' => 'Ysmall',
    '63354' => 'Zsmall',
    '63393' => 'exclamdownsmall',
    '63394' => 'centoldstyle',
    '63400' => 'Dieresissmall',
    '63407' => 'Macronsmall',
    '63412' => 'Acutesmall',
    '63416' => 'Cedillasmall',
    '63423' => 'questiondownsmall',
    '63456' => 'Agravesmall',
    '63457' => 'Aacutesmall',
    '63458' => 'Acircumflexsmall',
    '63459' => 'Atildesmall',
    '63460' => 'Adieresissmall',
    '63461' => 'Aringsmall',
    '63462' => 'AEsmall',
    '63463' => 'Ccedillasmall',
    '63464' => 'Egravesmall',
    '63465' => 'Eacutesmall',
    '63466' => 'Ecircumflexsmall',
    '63467' => 'Edieresissmall',
    '63468' => 'Igravesmall',
    '63469' => 'Iacutesmall',
    '63470' => 'Icircumflexsmall',
    '63471' => 'Idieresissmall',
    '63472' => 'Ethsmall',
    '63473' => 'Ntildesmall',
    '63474' => 'Ogravesmall',
    '63475' => 'Oacutesmall',
    '63476' => 'Ocircumflexsmall',
    '63477' => 'Otildesmall',
    '63478' => 'Odieresissmall',
    '63480' => 'Oslashsmall',
    '63481' => 'Ugravesmall',
    '63482' => 'Uacutesmall',
    '63483' => 'Ucircumflexsmall',
    '63484' => 'Udieresissmall',
    '63485' => 'Yacutesmall',
    '63486' => 'Thornsmall',
    '63487' => 'Ydieresissmall',
    '63620' => 'maihanakatleftthai',
    '63621' => 'saraileftthai',
    '63622' => 'saraiileftthai',
    '63623' => 'saraueleftthai',
    '63624' => 'saraueeleftthai',
    '63625' => 'maitaikhuleftthai',
    '63626' => 'maiekupperleftthai',
    '63627' => 'maieklowrightthai',
    '63628' => 'maieklowleftthai',
    '63629' => 'maithoupperleftthai',
    '63630' => 'maitholowrightthai',
    '63631' => 'maitholowleftthai',
    '63632' => 'maitriupperleftthai',
    '63633' => 'maitrilowrightthai',
    '63634' => 'maitrilowleftthai',
    '63635' => 'maichattawaupperleftthai',
    '63636' => 'maichattawalowrightthai',
    '63637' => 'maichattawalowleftthai',
    '63638' => 'thanthakhatupperleftthai',
    '63639' => 'thanthakhatlowrightthai',
    '63640' => 'thanthakhatlowleftthai',
    '63641' => 'nikhahitleftthai',
    '63703' => 'a89',
    '63704' => 'a90',
    '63705' => 'a93',
    '63706' => 'a94',
    '63707' => 'a91',
    '63708' => 'a92',
    '63709' => 'a205',
    '63710' => 'a85',
    '63711' => 'a206',
    '63712' => 'a86',
    '63713' => 'a87',
    '63714' => 'a88',
    '63715' => 'a95',
    '63716' => 'a96',
    '63717' => 'radicalex',
    '63718' => 'arrowvertex',
    '63719' => 'arrowhorizex',
    '63720' => 'registersans',
    '63721' => 'copyrightsans',
    '63722' => 'trademarksans',
    '63723' => 'parenlefttp',
    '63724' => 'parenleftex',
    '63725' => 'parenleftbt',
    '63726' => 'bracketlefttp',
    '63727' => 'bracketleftex',
    '63728' => 'bracketleftbt',
    '63729' => 'bracelefttp',
    '63730' => 'braceleftmid',
    '63731' => 'braceleftbt',
    '63732' => 'braceex',
    '63733' => 'integralex',
    '63734' => 'parenrighttp',
    '63735' => 'parenrightex',
    '63736' => 'parenrightbt',
    '63737' => 'bracketrighttp',
    '63738' => 'bracketrightex',
    '63739' => 'bracketrightbt',
    '63740' => 'bracerighttp',
    '63741' => 'bracerightmid',
    '63742' => 'bracerightbt',
    '63743' => 'apple',
    '64256' => 'ff',
    '64257' => 'fi',
    '64258' => 'fl',
    '64259' => 'ffi',
    '64260' => 'ffl',
    '64287' => 'afii57705',
    '64288' => 'ayinaltonehebrew',
    '64298' => 'afii57694',
    '64299' => 'afii57695',
    '64300' => 'shindageshshindot',
    '64301' => 'shindageshsindot',
    '64302' => 'alefpatahhebrew',
    '64303' => 'alefqamatshebrew',
    '64304' => 'alefdageshhebrew',
    '64305' => 'betdagesh',
    '64306' => 'gimeldagesh',
    '64307' => 'daletdagesh',
    '64308' => 'hedagesh',
    '64309' => 'afii57723',
    '64310' => 'zayindagesh',
    '64312' => 'tetdagesh',
    '64313' => 'yoddagesh',
    '64314' => 'finalkafdagesh',
    '64315' => 'kafdagesh',
    '64316' => 'lameddagesh',
    '64318' => 'memdagesh',
    '64320' => 'nundagesh',
    '64321' => 'samekhdagesh',
    '64323' => 'pefinaldageshhebrew',
    '64324' => 'pedagesh',
    '64326' => 'tsadidagesh',
    '64327' => 'qofdagesh',
    '64328' => 'reshdageshhebrew',
    '64329' => 'shindagesh',
    '64330' => 'tavdages',
    '64331' => 'afii57700',
    '64332' => 'betrafehebrew',
    '64333' => 'kafrafehebrew',
    '64334' => 'perafehebrew',
    '64335' => 'aleflamedhebrew',
    '64343' => 'pehfinalarabic',
    '64344' => 'pehinitialarabic',
    '64345' => 'pehmedialarabic',
    '64359' => 'ttehfinalarabic',
    '64360' => 'ttehinitialarabic',
    '64361' => 'ttehmedialarabic',
    '64363' => 'vehfinalarabic',
    '64364' => 'vehinitialarabic',
    '64365' => 'vehmedialarabic',
    '64379' => 'tchehfinalarabic',
    '64380' => 'tchehinitialarabic',
    '64381' => 'tchehmedialarabic',
    '64393' => 'ddalfinalarabic',
    '64395' => 'jehfinalarabic',
    '64397' => 'rrehfinalarabic',
    '64403' => 'gaffinalarabic',
    '64404' => 'gafinitialarabic',
    '64405' => 'gafmedialarabic',
    '64415' => 'noonghunnafinalarabic',
    '64420' => 'hehhamzaaboveisolatedarabic',
    '64421' => 'hehhamzaabovefinalarabic',
    '64423' => 'hehfinalaltonearabic',
    '64424' => 'hehinitialaltonearabic',
    '64425' => 'hehmedialaltonearabic',
    '64431' => 'yehbarreefinalarabic',
    '64520' => 'behmeemisolatedarabic',
    '64523' => 'tehjeemisolatedarabic',
    '64524' => 'tehhahisolatedarabic',
    '64526' => 'tehmeemisolatedarabic',
    '64584' => 'meemmeemisolatedarabic',
    '64587' => 'noonjeemisolatedarabic',
    '64590' => 'noonmeemisolatedarabic',
    '64600' => 'yehmeemisolatedarabic',
    '64606' => 'shaddadammatanarabic',
    '64607' => 'shaddakasratanarabic',
    '64608' => 'shaddafathaarabic',
    '64609' => 'shaddadammaarabic',
    '64610' => 'shaddakasraarabic',
    '64621' => 'behnoonfinalarabic',
    '64627' => 'tehnoonfinalarabic',
    '64653' => 'noonnoonfinalarabic',
    '64660' => 'yehnoonfinalarabic',
    '64671' => 'behmeeminitialarabic',
    '64673' => 'tehjeeminitialarabic',
    '64674' => 'tehhahinitialarabic',
    '64676' => 'tehmeeminitialarabic',
    '64713' => 'lamjeeminitialarabic',
    '64714' => 'lamhahinitialarabic',
    '64715' => 'lamkhahinitialarabic',
    '64716' => 'lammeeminitialarabic',
    '64721' => 'meemmeeminitialarabic',
    '64722' => 'noonjeeminitialarabic',
    '64725' => 'noonmeeminitialarabic',
    '64733' => 'yehmeeminitialarabic',
    '64830' => 'parenleftaltonearabic',
    '64831' => 'parenrightaltonearabic',
    '64904' => 'lammeemhahinitialarabic',
    '65010' => 'lamlamhehisolatedarabic',
    '65018' => 'sallallahoualayhewasallamarabic',
    '65072' => 'twodotleadervertical',
    '65073' => 'emdashvertical',
    '65074' => 'endashvertical',
    '65075' => 'underscorevertical',
    '65076' => 'wavyunderscorevertical',
    '65077' => 'parenleftvertical',
    '65078' => 'parenrightvertical',
    '65079' => 'braceleftvertical',
    '65080' => 'bracerightvertical',
    '65081' => 'tortoiseshellbracketleftvertical',
    '65082' => 'tortoiseshellbracketrightvertical',
    '65083' => 'blacklenticularbracketleftvertical',
    '65084' => 'blacklenticularbracketrightvertical',
    '65085' => 'dblanglebracketleftvertical',
    '65086' => 'dblanglebracketrightvertical',
    '65087' => 'anglebracketleftvertical',
    '65088' => 'anglebracketrightvertical',
    '65089' => 'cornerbracketleftvertical',
    '65090' => 'cornerbracketrightvertical',
    '65091' => 'whitecornerbracketleftvertical',
    '65092' => 'whitecornerbracketrightvertical',
    '65097' => 'overlinedashed',
    '65098' => 'overlinecenterline',
    '65099' => 'overlinewavy',
    '65100' => 'overlinedblwavy',
    '65101' => 'lowlinedashed',
    '65102' => 'lowlinecenterline',
    '65103' => 'underscorewavy',
    '65104' => 'commasmall',
    '65106' => 'periodsmall',
    '65108' => 'semicolonsmall',
    '65109' => 'colonsmall',
    '65113' => 'parenleftsmall',
    '65114' => 'parenrightsmall',
    '65115' => 'braceleftsmall',
    '65116' => 'bracerightsmall',
    '65117' => 'tortoiseshellbracketleftsmall',
    '65118' => 'tortoiseshellbracketrightsmall',
    '65119' => 'numbersignsmall',
    '65121' => 'asterisksmall',
    '65122' => 'plussmall',
    '65123' => 'hyphensmall',
    '65124' => 'lesssmall',
    '65125' => 'greatersmall',
    '65126' => 'equalsmall',
    '65129' => 'dollarsmall',
    '65130' => 'percentsmall',
    '65131' => 'atsmall',
    '65154' => 'alefmaddaabovefinalarabic',
    '65156' => 'alefhamzaabovefinalarabic',
    '65158' => 'wawhamzaabovefinalarabic',
    '65160' => 'alefhamzabelowfinalarabic',
    '65162' => 'yehhamzaabovefinalarabic',
    '65163' => 'yehhamzaaboveinitialarabic',
    '65164' => 'yehhamzaabovemedialarabic',
    '65166' => 'aleffinalarabic',
    '65168' => 'behfinalarabic',
    '65169' => 'behinitialarabic',
    '65170' => 'behmedialarabic',
    '65172' => 'tehmarbutafinalarabic',
    '65174' => 'tehfinalarabic',
    '65175' => 'tehinitialarabic',
    '65176' => 'tehmedialarabic',
    '65178' => 'thehfinalarabic',
    '65179' => 'thehinitialarabic',
    '65180' => 'thehmedialarabic',
    '65182' => 'jeemfinalarabic',
    '65183' => 'jeeminitialarabic',
    '65184' => 'jeemmedialarabic',
    '65186' => 'hahfinalarabic',
    '65187' => 'hahinitialarabic',
    '65188' => 'hahmedialarabic',
    '65190' => 'khahfinalarabic',
    '65191' => 'khahinitialarabic',
    '65192' => 'khahmedialarabic',
    '65194' => 'dalfinalarabic',
    '65196' => 'thalfinalarabic',
    '65198' => 'rehfinalarabic',
    '65200' => 'zainfinalarabic',
    '65202' => 'seenfinalarabic',
    '65203' => 'seeninitialarabic',
    '65204' => 'seenmedialarabic',
    '65206' => 'sheenfinalarabic',
    '65207' => 'sheeninitialarabic',
    '65208' => 'sheenmedialarabic',
    '65210' => 'sadfinalarabic',
    '65211' => 'sadinitialarabic',
    '65212' => 'sadmedialarabic',
    '65214' => 'dadfinalarabic',
    '65215' => 'dadinitialarabic',
    '65216' => 'dadmedialarabic',
    '65218' => 'tahfinalarabic',
    '65219' => 'tahinitialarabic',
    '65220' => 'tahmedialarabic',
    '65222' => 'zahfinalarabic',
    '65223' => 'zahinitialarabic',
    '65224' => 'zahmedialarabic',
    '65226' => 'ainfinalarabic',
    '65227' => 'aininitialarabic',
    '65228' => 'ainmedialarabic',
    '65230' => 'ghainfinalarabic',
    '65231' => 'ghaininitialarabic',
    '65232' => 'ghainmedialarabic',
    '65234' => 'fehfinalarabic',
    '65235' => 'fehinitialarabic',
    '65236' => 'fehmedialarabic',
    '65238' => 'qaffinalarabic',
    '65239' => 'qafinitialarabic',
    '65240' => 'qafmedialarabic',
    '65242' => 'kaffinalarabic',
    '65243' => 'kafinitialarabic',
    '65244' => 'kafmedialarabic',
    '65246' => 'lamfinalarabic',
    '65247' => 'laminitialarabic',
    '65248' => 'lammedialarabic',
    '65250' => 'meemfinalarabic',
    '65251' => 'meeminitialarabic',
    '65252' => 'meemmedialarabic',
    '65254' => 'noonfinalarabic',
    '65255' => 'noonhehinitialarabic',
    '65256' => 'noonmedialarabic',
    '65258' => 'hehfinalalttwoarabic',
    '65259' => 'hehinitialarabic',
    '65260' => 'hehmedialarabic',
    '65262' => 'wawfinalarabic',
    '65264' => 'alefmaksurafinalarabic',
    '65266' => 'yehfinalarabic',
    '65267' => 'alefmaksurainitialarabic',
    '65268' => 'alefmaksuramedialarabic',
    '65269' => 'lamalefmaddaaboveisolatedarabic',
    '65270' => 'lamalefmaddaabovefinalarabic',
    '65271' => 'lamalefhamzaaboveisolatedarabic',
    '65272' => 'lamalefhamzaabovefinalarabic',
    '65273' => 'lamalefhamzabelowisolatedarabic',
    '65274' => 'lamalefhamzabelowfinalarabic',
    '65275' => 'lamalefisolatedarabic',
    '65276' => 'lamaleffinalarabic',
    '65279' => 'zerowidthjoiner',
    '65281' => 'exclammonospace',
    '65282' => 'quotedblmonospace',
    '65283' => 'numbersignmonospace',
    '65284' => 'dollarmonospace',
    '65285' => 'percentmonospace',
    '65286' => 'ampersandmonospace',
    '65287' => 'quotesinglemonospace',
    '65288' => 'parenleftmonospace',
    '65289' => 'parenrightmonospace',
    '65290' => 'asteriskmonospace',
    '65291' => 'plusmonospace',
    '65292' => 'commamonospace',
    '65293' => 'hyphenmonospace',
    '65294' => 'periodmonospace',
    '65295' => 'slashmonospace',
    '65296' => 'zeromonospace',
    '65297' => 'onemonospace',
    '65298' => 'twomonospace',
    '65299' => 'threemonospace',
    '65300' => 'fourmonospace',
    '65301' => 'fivemonospace',
    '65302' => 'sixmonospace',
    '65303' => 'sevenmonospace',
    '65304' => 'eightmonospace',
    '65305' => 'ninemonospace',
    '65306' => 'colonmonospace',
    '65307' => 'semicolonmonospace',
    '65308' => 'lessmonospace',
    '65309' => 'equalmonospace',
    '65310' => 'greatermonospace',
    '65311' => 'questionmonospace',
    '65312' => 'atmonospace',
    '65313' => 'Amonospace',
    '65314' => 'Bmonospace',
    '65315' => 'Cmonospace',
    '65316' => 'Dmonospace',
    '65317' => 'Emonospace',
    '65318' => 'Fmonospace',
    '65319' => 'Gmonospace',
    '65320' => 'Hmonospace',
    '65321' => 'Imonospace',
    '65322' => 'Jmonospace',
    '65323' => 'Kmonospace',
    '65324' => 'Lmonospace',
    '65325' => 'Mmonospace',
    '65326' => 'Nmonospace',
    '65327' => 'Omonospace',
    '65328' => 'Pmonospace',
    '65329' => 'Qmonospace',
    '65330' => 'Rmonospace',
    '65331' => 'Smonospace',
    '65332' => 'Tmonospace',
    '65333' => 'Umonospace',
    '65334' => 'Vmonospace',
    '65335' => 'Wmonospace',
    '65336' => 'Xmonospace',
    '65337' => 'Ymonospace',
    '65338' => 'Zmonospace',
    '65339' => 'bracketleftmonospace',
    '65340' => 'backslashmonospace',
    '65341' => 'bracketrightmonospace',
    '65342' => 'asciicircummonospace',
    '65343' => 'underscoremonospace',
    '65344' => 'gravemonospace',
    '65345' => 'amonospace',
    '65346' => 'bmonospace',
    '65347' => 'cmonospace',
    '65348' => 'dmonospace',
    '65349' => 'emonospace',
    '65350' => 'fmonospace',
    '65351' => 'gmonospace',
    '65352' => 'hmonospace',
    '65353' => 'imonospace',
    '65354' => 'jmonospace',
    '65355' => 'kmonospace',
    '65356' => 'lmonospace',
    '65357' => 'mmonospace',
    '65358' => 'nmonospace',
    '65359' => 'omonospace',
    '65360' => 'pmonospace',
    '65361' => 'qmonospace',
    '65362' => 'rmonospace',
    '65363' => 'smonospace',
    '65364' => 'tmonospace',
    '65365' => 'umonospace',
    '65366' => 'vmonospace',
    '65367' => 'wmonospace',
    '65368' => 'xmonospace',
    '65369' => 'ymonospace',
    '65370' => 'zmonospace',
    '65371' => 'braceleftmonospace',
    '65372' => 'barmonospace',
    '65373' => 'bracerightmonospace',
    '65374' => 'asciitildemonospace',
    '65377' => 'periodhalfwidth',
    '65378' => 'cornerbracketlefthalfwidth',
    '65379' => 'cornerbracketrighthalfwidth',
    '65380' => 'ideographiccommaleft',
    '65381' => 'middledotkatakanahalfwidth',
    '65382' => 'wokatakanahalfwidth',
    '65383' => 'asmallkatakanahalfwidth',
    '65384' => 'ismallkatakanahalfwidth',
    '65385' => 'usmallkatakanahalfwidth',
    '65386' => 'esmallkatakanahalfwidth',
    '65387' => 'osmallkatakanahalfwidth',
    '65388' => 'yasmallkatakanahalfwidth',
    '65389' => 'yusmallkatakanahalfwidth',
    '65390' => 'yosmallkatakanahalfwidth',
    '65391' => 'tusmallkatakanahalfwidth',
    '65392' => 'katahiraprolongmarkhalfwidth',
    '65393' => 'akatakanahalfwidth',
    '65394' => 'ikatakanahalfwidth',
    '65395' => 'ukatakanahalfwidth',
    '65396' => 'ekatakanahalfwidth',
    '65397' => 'okatakanahalfwidth',
    '65398' => 'kakatakanahalfwidth',
    '65399' => 'kikatakanahalfwidth',
    '65400' => 'kukatakanahalfwidth',
    '65401' => 'kekatakanahalfwidth',
    '65402' => 'kokatakanahalfwidth',
    '65403' => 'sakatakanahalfwidth',
    '65404' => 'sikatakanahalfwidth',
    '65405' => 'sukatakanahalfwidth',
    '65406' => 'sekatakanahalfwidth',
    '65407' => 'sokatakanahalfwidth',
    '65408' => 'takatakanahalfwidth',
    '65409' => 'tikatakanahalfwidth',
    '65410' => 'tukatakanahalfwidth',
    '65411' => 'tekatakanahalfwidth',
    '65412' => 'tokatakanahalfwidth',
    '65413' => 'nakatakanahalfwidth',
    '65414' => 'nikatakanahalfwidth',
    '65415' => 'nukatakanahalfwidth',
    '65416' => 'nekatakanahalfwidth',
    '65417' => 'nokatakanahalfwidth',
    '65418' => 'hakatakanahalfwidth',
    '65419' => 'hikatakanahalfwidth',
    '65420' => 'hukatakanahalfwidth',
    '65421' => 'hekatakanahalfwidth',
    '65422' => 'hokatakanahalfwidth',
    '65423' => 'makatakanahalfwidth',
    '65424' => 'mikatakanahalfwidth',
    '65425' => 'mukatakanahalfwidth',
    '65426' => 'mekatakanahalfwidth',
    '65427' => 'mokatakanahalfwidth',
    '65428' => 'yakatakanahalfwidth',
    '65429' => 'yukatakanahalfwidth',
    '65430' => 'yokatakanahalfwidth',
    '65431' => 'rakatakanahalfwidth',
    '65432' => 'rikatakanahalfwidth',
    '65433' => 'rukatakanahalfwidth',
    '65434' => 'rekatakanahalfwidth',
    '65435' => 'rokatakanahalfwidth',
    '65436' => 'wakatakanahalfwidth',
    '65437' => 'nkatakanahalfwidth',
    '65438' => 'voicedmarkkanahalfwidth',
    '65439' => 'semivoicedmarkkanahalfwidth',
    '65504' => 'centmonospace',
    '65505' => 'sterlingmonospace',
    '65507' => 'macronmonospace',
    '65509' => 'yenmonospace',
    '65510' => 'wonmonospace',
};

our $n2u = {
    '.notdef' => '0',
    'A' => '65',
    'AE' => '198',
    'AEacute' => '508',
    'AElig' => '198',
    'AEmacron' => '482',
    'AEsmall' => '63462',
    'Aacgr' => '902',
    'Aacute' => '193',
    'Aacutesmall' => '63457',
    'Abreve' => '258',
    'Abreveacute' => '7854',
    'Abrevecyrillic' => '1232',
    'Abrevedotbelow' => '7862',
    'Abrevegrave' => '7856',
    'Abrevehookabove' => '7858',
    'Abrevetilde' => '7860',
    'Acaron' => '461',
    'Acirc' => '194',
    'Acircle' => '9398',
    'Acircumflex' => '194',
    'Acircumflexacute' => '7844',
    'Acircumflexdotbelow' => '7852',
    'Acircumflexgrave' => '7846',
    'Acircumflexhookabove' => '7848',
    'Acircumflexsmall' => '63458',
    'Acircumflextilde' => '7850',
    'Acute' => '63177',
    'Acutesmall' => '63412',
    'Acy' => '1040',
    'Acyrillic' => '1040',
    'Adblgrave' => '512',
    'Adieresis' => '196',
    'Adieresiscyrillic' => '1234',
    'Adieresismacron' => '478',
    'Adieresissmall' => '63460',
    'Adotbelow' => '7840',
    'Adotmacron' => '480',
    'Agr' => '913',
    'Agrave' => '192',
    'Agravesmall' => '63456',
    'Ahookabove' => '7842',
    'Aiecyrillic' => '1236',
    'Ainvertedbreve' => '514',
    'Alpha' => '913',
    'Alphatonos' => '902',
    'Amacr' => '256',
    'Amacron' => '256',
    'Amonospace' => '65313',
    'Aogon' => '260',
    'Aogonek' => '260',
    'Aring' => '197',
    'Aringacute' => '506',
    'Aringbelow' => '7680',
    'Aringsmall' => '63461',
    'Asmall' => '63329',
    'Atilde' => '195',
    'Atildesmall' => '63459',
    'Auml' => '196',
    'Aybarmenian' => '1329',
    'B' => '66',
    'Barwed' => '8966',
    'Bcircle' => '9399',
    'Bcy' => '1041',
    'Bdotaccent' => '7682',
    'Bdotbelow' => '7684',
    'Becyrillic' => '1041',
    'Benarmenian' => '1330',
    'Beta' => '914',
    'Bgr' => '914',
    'Bhook' => '385',
    'Blinebelow' => '7686',
    'Bmonospace' => '65314',
    'Brevesmall' => '63220',
    'Bsmall' => '63330',
    'Btopbar' => '386',
    'C' => '67',
    'CHcy' => '1063',
    'Caarmenian' => '1342',
    'Cacute' => '262',
    'Cap' => '8914',
    'Caron' => '63178',
    'Caronsmall' => '63221',
    'Ccaron' => '268',
    'Ccedil' => '199',
    'Ccedilla' => '199',
    'Ccedillaacute' => '7688',
    'Ccedillasmall' => '63463',
    'Ccirc' => '264',
    'Ccircle' => '9400',
    'Ccircumflex' => '264',
    'Cdot' => '266',
    'Cdotaccent' => '266',
    'Cedillasmall' => '63416',
    'Chaarmenian' => '1353',
    'Cheabkhasiancyrillic' => '1212',
    'Checyrillic' => '1063',
    'Chedescenderabkhasiancyrillic' => '1214',
    'Chedescendercyrillic' => '1206',
    'Chedieresiscyrillic' => '1268',
    'Cheharmenian' => '1347',
    'Chekhakassiancyrillic' => '1227',
    'Cheverticalstrokecyrillic' => '1208',
    'Chi' => '935',
    'Chook' => '391',
    'Circumflexsmall' => '63222',
    'Cmonospace' => '65315',
    'Coarmenian' => '1361',
    'Csmall' => '63331',
    'Cup' => '8915',
    'D' => '68',
    'DJcy' => '1026',
    'DScy' => '1029',
    'DZ' => '497',
    'DZcaron' => '452',
    'DZcy' => '1039',
    'Daarmenian' => '1332',
    'Dafrican' => '393',
    'Dagger' => '8225',
    'Dcaron' => '270',
    'Dcedilla' => '7696',
    'Dcircle' => '9401',
    'Dcircumflexbelow' => '7698',
    'Dcroat' => '272',
    'Ddotaccent' => '7690',
    'Ddotbelow' => '7692',
    'Decyrillic' => '1044',
    'Deicoptic' => '1006',
    'Delta' => '916',
    'Deltagreek' => '916',
    'Dgr' => '916',
    'Dhook' => '394',
    'Dieresis' => '63179',
    'DieresisAcute' => '63180',
    'DieresisGrave' => '63181',
    'Dieresissmall' => '63400',
    'Digammagreek' => '988',
    'Djecyrillic' => '1026',
    'Dlinebelow' => '7694',
    'Dmonospace' => '65316',
    'DotDot' => '8412',
    'Dotaccentsmall' => '63223',
    'Dslash' => '272',
    'Dsmall' => '63332',
    'Dstrok' => '272',
    'Dtopbar' => '395',
    'Dz' => '498',
    'Dzcaron' => '453',
    'Dzeabkhasiancyrillic' => '1248',
    'Dzecyrillic' => '1029',
    'Dzhecyrillic' => '1039',
    'E' => '69',
    'EEacgr' => '905',
    'EEgr' => '919',
    'ENG' => '330',
    'ETH' => '208',
    'Eacgr' => '904',
    'Eacute' => '201',
    'Eacutesmall' => '63465',
    'Ebreve' => '276',
    'Ecaron' => '282',
    'Ecedillabreve' => '7708',
    'Echarmenian' => '1333',
    'Ecirc' => '202',
    'Ecircle' => '9402',
    'Ecircumflex' => '202',
    'Ecircumflexacute' => '7870',
    'Ecircumflexbelow' => '7704',
    'Ecircumflexdotbelow' => '7878',
    'Ecircumflexgrave' => '7872',
    'Ecircumflexhookabove' => '7874',
    'Ecircumflexsmall' => '63466',
    'Ecircumflextilde' => '7876',
    'Ecy' => '1069',
    'Ecyrillic' => '1028',
    'Edblgrave' => '516',
    'Edieresis' => '203',
    'Edieresissmall' => '63467',
    'Edot' => '278',
    'Edotaccent' => '278',
    'Edotbelow' => '7864',
    'Efcyrillic' => '1060',
    'Egr' => '917',
    'Egrave' => '200',
    'Egravesmall' => '63464',
    'Eharmenian' => '1335',
    'Ehookabove' => '7866',
    'Eightroman' => '8551',
    'Einvertedbreve' => '518',
    'Eiotifiedcyrillic' => '1124',
    'Elcyrillic' => '1051',
    'Elevenroman' => '8554',
    'Emacr' => '274',
    'Emacron' => '274',
    'Emacronacute' => '7702',
    'Emacrongrave' => '7700',
    'Emcyrillic' => '1052',
    'Emonospace' => '65317',
    'Encyrillic' => '1053',
    'Endescendercyrillic' => '1186',
    'Eng' => '330',
    'Enghecyrillic' => '1188',
    'Enhookcyrillic' => '1223',
    'Eogon' => '280',
    'Eogonek' => '280',
    'Eopen' => '400',
    'Epsilon' => '917',
    'Epsilontonos' => '904',
    'Ercyrillic' => '1056',
    'Ereversed' => '398',
    'Ereversedcyrillic' => '1069',
    'Escyrillic' => '1057',
    'Esdescendercyrillic' => '1194',
    'Esh' => '425',
    'Esmall' => '63333',
    'Eta' => '919',
    'Etarmenian' => '1336',
    'Etatonos' => '905',
    'Eth' => '208',
    'Ethsmall' => '63472',
    'Etilde' => '7868',
    'Etildebelow' => '7706',
    'Euml' => '203',
    'Euro' => '8364',
    'Ezh' => '439',
    'Ezhcaron' => '494',
    'Ezhreversed' => '440',
    'F' => '70',
    'Fcircle' => '9403',
    'Fcy' => '1060',
    'Fdotaccent' => '7710',
    'Feharmenian' => '1366',
    'Feicoptic' => '996',
    'Fhook' => '401',
    'Fitacyrillic' => '1138',
    'Fiveroman' => '8548',
    'Fmonospace' => '65318',
    'Fourroman' => '8547',
    'Fsmall' => '63334',
    'G' => '71',
    'GBsquare' => '13191',
    'GJcy' => '1027',
    'Gacute' => '500',
    'Gamma' => '915',
    'Gammaafrican' => '404',
    'Gangiacoptic' => '1002',
    'Gbreve' => '286',
    'Gcaron' => '486',
    'Gcedil' => '290',
    'Gcedilla' => '290',
    'Gcirc' => '284',
    'Gcircle' => '9404',
    'Gcircumflex' => '284',
    'Gcommaaccent' => '290',
    'Gcy' => '1043',
    'Gdot' => '288',
    'Gdotaccent' => '288',
    'Gecyrillic' => '1043',
    'Gg' => '8921',
    'Ggr' => '915',
    'Ghadarmenian' => '1346',
    'Ghemiddlehookcyrillic' => '1172',
    'Ghestrokecyrillic' => '1170',
    'Gheupturncyrillic' => '1168',
    'Ghook' => '403',
    'Gimarmenian' => '1331',
    'Gjecyrillic' => '1027',
    'Gmacron' => '7712',
    'Gmonospace' => '65319',
    'Grave' => '63182',
    'Gravesmall' => '63328',
    'Gsmall' => '63335',
    'Gsmallhook' => '667',
    'Gstroke' => '484',
    'Gt' => '8811',
    'H' => '72',
    'H18533' => '9679',
    'H18543' => '9642',
    'H18551' => '9643',
    'H22073' => '9633',
    'HARDcy' => '1066',
    'HPsquare' => '13259',
    'Haabkhasiancyrillic' => '1192',
    'Hadescendercyrillic' => '1202',
    'Hardsigncyrillic' => '1066',
    'Hbar' => '294',
    'Hbrevebelow' => '7722',
    'Hcedilla' => '7720',
    'Hcirc' => '292',
    'Hcircle' => '9405',
    'Hcircumflex' => '292',
    'Hdieresis' => '7718',
    'Hdotaccent' => '7714',
    'Hdotbelow' => '7716',
    'Hmonospace' => '65320',
    'Hoarmenian' => '1344',
    'Horicoptic' => '1000',
    'Hsmall' => '63336',
    'Hstrok' => '294',
    'Hungarumlaut' => '63183',
    'Hungarumlautsmall' => '63224',
    'Hzsquare' => '13200',
    'I' => '73',
    'IAcyrillic' => '1071',
    'IEcy' => '1045',
    'IJ' => '306',
    'IJlig' => '306',
    'IOcy' => '1025',
    'IUcyrillic' => '1070',
    'Iacgr' => '906',
    'Iacute' => '205',
    'Iacutesmall' => '63469',
    'Ibreve' => '300',
    'Icaron' => '463',
    'Icirc' => '206',
    'Icircle' => '9406',
    'Icircumflex' => '206',
    'Icircumflexsmall' => '63470',
    'Icy' => '1048',
    'Icyrillic' => '1030',
    'Idblgrave' => '520',
    'Idieresis' => '207',
    'Idieresisacute' => '7726',
    'Idieresiscyrillic' => '1252',
    'Idieresissmall' => '63471',
    'Idigr' => '938',
    'Idot' => '304',
    'Idotaccent' => '304',
    'Idotbelow' => '7882',
    'Iebrevecyrillic' => '1238',
    'Iecyrillic' => '1045',
    'Ifraktur' => '8465',
    'Igr' => '921',
    'Igrave' => '204',
    'Igravesmall' => '63468',
    'Ihookabove' => '7880',
    'Iicyrillic' => '1048',
    'Iinvertedbreve' => '522',
    'Iishortcyrillic' => '1049',
    'Imacr' => '298',
    'Imacron' => '298',
    'Imacroncyrillic' => '1250',
    'Imonospace' => '65321',
    'Iniarmenian' => '1339',
    'Iocyrillic' => '1025',
    'Iogon' => '302',
    'Iogonek' => '302',
    'Iota' => '921',
    'Iotaafrican' => '406',
    'Iotadieresis' => '938',
    'Iotatonos' => '906',
    'Ismall' => '63337',
    'Istroke' => '407',
    'Itilde' => '296',
    'Itildebelow' => '7724',
    'Iukcy' => '1030',
    'Iuml' => '207',
    'Izhitsacyrillic' => '1140',
    'Izhitsadblgravecyrillic' => '1142',
    'J' => '74',
    'Jaarmenian' => '1345',
    'Jcirc' => '308',
    'Jcircle' => '9407',
    'Jcircumflex' => '308',
    'Jcy' => '1049',
    'Jecyrillic' => '1032',
    'Jheharmenian' => '1355',
    'Jmonospace' => '65322',
    'Jsercy' => '1032',
    'Jsmall' => '63338',
    'Jukcy' => '1028',
    'K' => '75',
    'KBsquare' => '13189',
    'KHcy' => '1061',
    'KHgr' => '935',
    'KJcy' => '1036',
    'KKsquare' => '13261',
    'Kabashkircyrillic' => '1184',
    'Kacute' => '7728',
    'Kacyrillic' => '1050',
    'Kadescendercyrillic' => '1178',
    'Kahookcyrillic' => '1219',
    'Kappa' => '922',
    'Kastrokecyrillic' => '1182',
    'Kaverticalstrokecyrillic' => '1180',
    'Kcaron' => '488',
    'Kcedil' => '310',
    'Kcedilla' => '310',
    'Kcircle' => '9408',
    'Kcommaaccent' => '310',
    'Kcy' => '1050',
    'Kdotbelow' => '7730',
    'Keharmenian' => '1364',
    'Kenarmenian' => '1343',
    'Kgr' => '922',
    'Khacyrillic' => '1061',
    'Kheicoptic' => '998',
    'Khook' => '408',
    'Kjecyrillic' => '1036',
    'Klinebelow' => '7732',
    'Kmonospace' => '65323',
    'Koppacyrillic' => '1152',
    'Koppagreek' => '990',
    'Ksicyrillic' => '1134',
    'Ksmall' => '63339',
    'L' => '76',
    'LJ' => '455',
    'LJcy' => '1033',
    'LL' => '63167',
    'Lacute' => '313',
    'Lambda' => '923',
    'Larr' => '8606',
    'Lcaron' => '317',
    'Lcedil' => '315',
    'Lcedilla' => '315',
    'Lcircle' => '9409',
    'Lcircumflexbelow' => '7740',
    'Lcommaaccent' => '315',
    'Lcy' => '1051',
    'Ldot' => '319',
    'Ldotaccent' => '319',
    'Ldotbelow' => '7734',
    'Ldotbelowmacron' => '7736',
    'Lgr' => '923',
    'Liwnarmenian' => '1340',
    'Lj' => '456',
    'Ljecyrillic' => '1033',
    'Ll' => '8920',
    'Llinebelow' => '7738',
    'Lmidot' => '319',
    'Lmonospace' => '65324',
    'Lslash' => '321',
    'Lslashsmall' => '63225',
    'Lsmall' => '63340',
    'Lstrok' => '321',
    'Lt' => '8810',
    'M' => '77',
    'MBsquare' => '13190',
    'Macron' => '63184',
    'Macronsmall' => '63407',
    'Macute' => '7742',
    'Mcircle' => '9410',
    'Mcy' => '1052',
    'Mdotaccent' => '7744',
    'Mdotbelow' => '7746',
    'Menarmenian' => '1348',
    'Mgr' => '924',
    'Mmonospace' => '65325',
    'Msmall' => '63341',
    'Mturned' => '412',
    'Mu' => '924',
    'N' => '78',
    'NJ' => '458',
    'NJcy' => '1034',
    'Nacute' => '323',
    'Ncaron' => '327',
    'Ncedil' => '325',
    'Ncedilla' => '325',
    'Ncircle' => '9411',
    'Ncircumflexbelow' => '7754',
    'Ncommaaccent' => '325',
    'Ncy' => '1053',
    'Ndotaccent' => '7748',
    'Ndotbelow' => '7750',
    'Ngr' => '925',
    'Nhookleft' => '413',
    'Nineroman' => '8552',
    'Nj' => '459',
    'Njecyrillic' => '1034',
    'Nlinebelow' => '7752',
    'Nmonospace' => '65326',
    'Nowarmenian' => '1350',
    'Nsmall' => '63342',
    'Ntilde' => '209',
    'Ntildesmall' => '63473',
    'Nu' => '925',
    'O' => '79',
    'OE' => '338',
    'OElig' => '338',
    'OEsmall' => '63226',
    'OHacgr' => '911',
    'OHgr' => '937',
    'Oacgr' => '908',
    'Oacute' => '211',
    'Oacutesmall' => '63475',
    'Obarredcyrillic' => '1256',
    'Obarreddieresiscyrillic' => '1258',
    'Obreve' => '334',
    'Ocaron' => '465',
    'Ocenteredtilde' => '415',
    'Ocirc' => '212',
    'Ocircle' => '9412',
    'Ocircumflex' => '212',
    'Ocircumflexacute' => '7888',
    'Ocircumflexdotbelow' => '7896',
    'Ocircumflexgrave' => '7890',
    'Ocircumflexhookabove' => '7892',
    'Ocircumflexsmall' => '63476',
    'Ocircumflextilde' => '7894',
    'Ocy' => '1054',
    'Ocyrillic' => '1054',
    'Odblac' => '336',
    'Odblacute' => '336',
    'Odblgrave' => '524',
    'Odieresis' => '214',
    'Odieresiscyrillic' => '1254',
    'Odieresissmall' => '63478',
    'Odotbelow' => '7884',
    'Ogoneksmall' => '63227',
    'Ogr' => '927',
    'Ograve' => '210',
    'Ogravesmall' => '63474',
    'Oharmenian' => '1365',
    'Ohm' => '8486',
    'Ohookabove' => '7886',
    'Ohorn' => '416',
    'Ohornacute' => '7898',
    'Ohorndotbelow' => '7906',
    'Ohorngrave' => '7900',
    'Ohornhookabove' => '7902',
    'Ohorntilde' => '7904',
    'Ohungarumlaut' => '336',
    'Oi' => '418',
    'Oinvertedbreve' => '526',
    'Omacr' => '332',
    'Omacron' => '332',
    'Omacronacute' => '7762',
    'Omacrongrave' => '7760',
    'Omega' => '937',
    'Omegacyrillic' => '1120',
    'Omegagreek' => '937',
    'Omegaroundcyrillic' => '1146',
    'Omegatitlocyrillic' => '1148',
    'Omegatonos' => '911',
    'Omicron' => '927',
    'Omicrontonos' => '908',
    'Omonospace' => '65327',
    'Oneroman' => '8544',
    'Oogonek' => '490',
    'Oogonekmacron' => '492',
    'Oopen' => '390',
    'Oslash' => '216',
    'Oslashacute' => '510',
    'Oslashsmall' => '63480',
    'Osmall' => '63343',
    'Ostrokeacute' => '510',
    'Otcyrillic' => '1150',
    'Otilde' => '213',
    'Otildeacute' => '7756',
    'Otildedieresis' => '7758',
    'Otildesmall' => '63477',
    'Ouml' => '214',
    'P' => '80',
    'PHgr' => '934',
    'PSgr' => '936',
    'Pacute' => '7764',
    'Pcircle' => '9413',
    'Pcy' => '1055',
    'Pdotaccent' => '7766',
    'Pecyrillic' => '1055',
    'Peharmenian' => '1354',
    'Pemiddlehookcyrillic' => '1190',
    'Pgr' => '928',
    'Phi' => '934',
    'Phook' => '420',
    'Pi' => '928',
    'Piwrarmenian' => '1363',
    'Pmonospace' => '65328',
    'Prime' => '8243',
    'Psi' => '936',
    'Psicyrillic' => '1136',
    'Psmall' => '63344',
    'Q' => '81',
    'Qcircle' => '9414',
    'Qmonospace' => '65329',
    'Qsmall' => '63345',
    'R' => '82',
    'Raarmenian' => '1356',
    'Racute' => '340',
    'Rarr' => '8608',
    'Rcaron' => '344',
    'Rcedil' => '342',
    'Rcedilla' => '342',
    'Rcircle' => '9415',
    'Rcommaaccent' => '342',
    'Rcy' => '1056',
    'Rdblgrave' => '528',
    'Rdotaccent' => '7768',
    'Rdotbelow' => '7770',
    'Rdotbelowmacron' => '7772',
    'Reharmenian' => '1360',
    'Rfraktur' => '8476',
    'Rgr' => '929',
    'Rho' => '929',
    'Ringsmall' => '63228',
    'Rinvertedbreve' => '530',
    'Rlinebelow' => '7774',
    'Rmonospace' => '65330',
    'Rsmall' => '63346',
    'Rsmallinverted' => '641',
    'Rsmallinvertedsuperior' => '694',
    'S' => '83',
    'SF010000' => '9484',
    'SF020000' => '9492',
    'SF030000' => '9488',
    'SF040000' => '9496',
    'SF050000' => '9532',
    'SF060000' => '9516',
    'SF070000' => '9524',
    'SF080000' => '9500',
    'SF090000' => '9508',
    'SF100000' => '9472',
    'SF110000' => '9474',
    'SF190000' => '9569',
    'SF200000' => '9570',
    'SF210000' => '9558',
    'SF220000' => '9557',
    'SF230000' => '9571',
    'SF240000' => '9553',
    'SF250000' => '9559',
    'SF260000' => '9565',
    'SF270000' => '9564',
    'SF280000' => '9563',
    'SF360000' => '9566',
    'SF370000' => '9567',
    'SF380000' => '9562',
    'SF390000' => '9556',
    'SF400000' => '9577',
    'SF410000' => '9574',
    'SF420000' => '9568',
    'SF430000' => '9552',
    'SF440000' => '9580',
    'SF450000' => '9575',
    'SF460000' => '9576',
    'SF470000' => '9572',
    'SF480000' => '9573',
    'SF490000' => '9561',
    'SF500000' => '9560',
    'SF510000' => '9554',
    'SF520000' => '9555',
    'SF530000' => '9579',
    'SF540000' => '9578',
    'SHCHcy' => '1065',
    'SHcy' => '1064',
    'SOFTcy' => '1068',
    'Sacute' => '346',
    'Sacutedotaccent' => '7780',
    'Sampigreek' => '992',
    'Scaron' => '352',
    'Scarondotaccent' => '7782',
    'Scaronsmall' => '63229',
    'Scedil' => '350',
    'Scedilla' => '350',
    'Schwa' => '399',
    'Schwacyrillic' => '1240',
    'Schwadieresiscyrillic' => '1242',
    'Scirc' => '348',
    'Scircle' => '9416',
    'Scircumflex' => '348',
    'Scommaaccent' => '536',
    'Scy' => '1057',
    'Sdotaccent' => '7776',
    'Sdotbelow' => '7778',
    'Sdotbelowdotaccent' => '7784',
    'Seharmenian' => '1357',
    'Sevenroman' => '8550',
    'Sgr' => '931',
    'Shaarmenian' => '1351',
    'Shacyrillic' => '1064',
    'Shchacyrillic' => '1065',
    'Sheicoptic' => '994',
    'Shhacyrillic' => '1210',
    'Shimacoptic' => '1004',
    'Sigma' => '931',
    'Sixroman' => '8549',
    'Smonospace' => '65331',
    'Softsigncyrillic' => '1068',
    'Ssmall' => '63347',
    'Stigmagreek' => '986',
    'Sub' => '8912',
    'Sup' => '8913',
    'T' => '84',
    'THORN' => '222',
    'THgr' => '920',
    'TSHcy' => '1035',
    'TScy' => '1062',
    'Tau' => '932',
    'Tbar' => '358',
    'Tcaron' => '356',
    'Tcedil' => '354',
    'Tcedilla' => '354',
    'Tcircle' => '9417',
    'Tcircumflexbelow' => '7792',
    'Tcommaaccent' => '538',
    'Tcy' => '1058',
    'Tdotaccent' => '7786',
    'Tdotbelow' => '7788',
    'Tecyrillic' => '1058',
    'Tedescendercyrillic' => '1196',
    'Tenroman' => '8553',
    'Tetsecyrillic' => '1204',
    'Tgr' => '932',
    'Theta' => '920',
    'Thook' => '428',
    'Thorn' => '222',
    'Thornsmall' => '63486',
    'Threeroman' => '8546',
    'Tildesmall' => '63230',
    'Tiwnarmenian' => '1359',
    'Tlinebelow' => '7790',
    'Tmonospace' => '65332',
    'Toarmenian' => '1337',
    'Tonefive' => '444',
    'Tonesix' => '388',
    'Tonetwo' => '423',
    'Tretroflexhook' => '430',
    'Tsecyrillic' => '1062',
    'Tshecyrillic' => '1035',
    'Tsmall' => '63348',
    'Tstrok' => '358',
    'Twelveroman' => '8555',
    'Tworoman' => '8545',
    'U' => '85',
    'Uacgr' => '910',
    'Uacute' => '218',
    'Uacutesmall' => '63482',
    'Ubrcy' => '1038',
    'Ubreve' => '364',
    'Ucaron' => '467',
    'Ucirc' => '219',
    'Ucircle' => '9418',
    'Ucircumflex' => '219',
    'Ucircumflexbelow' => '7798',
    'Ucircumflexsmall' => '63483',
    'Ucy' => '1059',
    'Ucyrillic' => '1059',
    'Udblac' => '368',
    'Udblacute' => '368',
    'Udblgrave' => '532',
    'Udieresis' => '220',
    'Udieresisacute' => '471',
    'Udieresisbelow' => '7794',
    'Udieresiscaron' => '473',
    'Udieresiscyrillic' => '1264',
    'Udieresisgrave' => '475',
    'Udieresismacron' => '469',
    'Udieresissmall' => '63484',
    'Udigr' => '939',
    'Udotbelow' => '7908',
    'Ugrave' => '217',
    'Ugravesmall' => '63481',
    'Uhookabove' => '7910',
    'Uhorn' => '431',
    'Uhornacute' => '7912',
    'Uhorndotbelow' => '7920',
    'Uhorngrave' => '7914',
    'Uhornhookabove' => '7916',
    'Uhorntilde' => '7918',
    'Uhungarumlaut' => '368',
    'Uhungarumlautcyrillic' => '1266',
    'Uinvertedbreve' => '534',
    'Ukcyrillic' => '1144',
    'Umacr' => '362',
    'Umacron' => '362',
    'Umacroncyrillic' => '1262',
    'Umacrondieresis' => '7802',
    'Umonospace' => '65333',
    'Uogon' => '370',
    'Uogonek' => '370',
    'Upsi' => '933',
    'Upsilon' => '933',
    'Upsilon1' => '978',
    'Upsilonacutehooksymbolgreek' => '979',
    'Upsilonafrican' => '433',
    'Upsilondieresis' => '939',
    'Upsilondieresishooksymbolgreek' => '980',
    'Upsilonhooksymbol' => '978',
    'Upsilontonos' => '910',
    'Uring' => '366',
    'Ushortcyrillic' => '1038',
    'Usmall' => '63349',
    'Ustraightcyrillic' => '1198',
    'Ustraightstrokecyrillic' => '1200',
    'Utilde' => '360',
    'Utildeacute' => '7800',
    'Utildebelow' => '7796',
    'Uuml' => '220',
    'V' => '86',
    'Vcircle' => '9419',
    'Vcy' => '1042',
    'Vdash' => '8873',
    'Vdotbelow' => '7806',
    'Vecyrillic' => '1042',
    'Verbar' => '8214',
    'Vewarmenian' => '1358',
    'Vhook' => '434',
    'Vmonospace' => '65334',
    'Voarmenian' => '1352',
    'Vsmall' => '63350',
    'Vtilde' => '7804',
    'Vvdash' => '8874',
    'W' => '87',
    'Wacute' => '7810',
    'Wcirc' => '372',
    'Wcircle' => '9420',
    'Wcircumflex' => '372',
    'Wdieresis' => '7812',
    'Wdotaccent' => '7814',
    'Wdotbelow' => '7816',
    'Wgrave' => '7808',
    'Wmonospace' => '65335',
    'Wsmall' => '63351',
    'X' => '88',
    'Xcircle' => '9421',
    'Xdieresis' => '7820',
    'Xdotaccent' => '7818',
    'Xeharmenian' => '1341',
    'Xgr' => '926',
    'Xi' => '926',
    'Xmonospace' => '65336',
    'Xsmall' => '63352',
    'Y' => '89',
    'YAcy' => '1071',
    'YIcy' => '1031',
    'YUcy' => '1070',
    'Yacute' => '221',
    'Yacutesmall' => '63485',
    'Yatcyrillic' => '1122',
    'Ycirc' => '374',
    'Ycircle' => '9422',
    'Ycircumflex' => '374',
    'Ycy' => '1067',
    'Ydieresis' => '376',
    'Ydieresissmall' => '63487',
    'Ydotaccent' => '7822',
    'Ydotbelow' => '7924',
    'Yericyrillic' => '1067',
    'Yerudieresiscyrillic' => '1272',
    'Ygrave' => '7922',
    'Yhook' => '435',
    'Yhookabove' => '7926',
    'Yiarmenian' => '1349',
    'Yicyrillic' => '1031',
    'Yiwnarmenian' => '1362',
    'Ymonospace' => '65337',
    'Ysmall' => '63353',
    'Ytilde' => '7928',
    'Yuml' => '376',
    'Yusbigcyrillic' => '1130',
    'Yusbigiotifiedcyrillic' => '1132',
    'Yuslittlecyrillic' => '1126',
    'Yuslittleiotifiedcyrillic' => '1128',
    'Z' => '90',
    'ZHcy' => '1046',
    'Zaarmenian' => '1334',
    'Zacute' => '377',
    'Zcaron' => '381',
    'Zcaronsmall' => '63231',
    'Zcircle' => '9423',
    'Zcircumflex' => '7824',
    'Zcy' => '1047',
    'Zdot' => '379',
    'Zdotaccent' => '379',
    'Zdotbelow' => '7826',
    'Zecyrillic' => '1047',
    'Zedescendercyrillic' => '1176',
    'Zedieresiscyrillic' => '1246',
    'Zeta' => '918',
    'Zgr' => '918',
    'Zhearmenian' => '1338',
    'Zhebrevecyrillic' => '1217',
    'Zhecyrillic' => '1046',
    'Zhedescendercyrillic' => '1174',
    'Zhedieresiscyrillic' => '1244',
    'Zlinebelow' => '7828',
    'Zmonospace' => '65338',
    'Zsmall' => '63354',
    'Zstroke' => '437',
    'a' => '97',
    'a1' => '9985',
    'a10' => '10017',
    'a100' => '10078',
    'a101' => '10081',
    'a102' => '10082',
    'a103' => '10083',
    'a104' => '10084',
    'a105' => '10000',
    'a106' => '10085',
    'a107' => '10086',
    'a108' => '10087',
    'a11' => '9755',
    'a117' => '9993',
    'a118' => '9992',
    'a119' => '9991',
    'a12' => '9758',
    'a120' => '9312',
    'a121' => '9313',
    'a122' => '9314',
    'a123' => '9315',
    'a124' => '9316',
    'a125' => '9317',
    'a126' => '9318',
    'a127' => '9319',
    'a128' => '9320',
    'a129' => '9321',
    'a13' => '9996',
    'a130' => '10102',
    'a131' => '10103',
    'a132' => '10104',
    'a133' => '10105',
    'a134' => '10106',
    'a135' => '10107',
    'a136' => '10108',
    'a137' => '10109',
    'a138' => '10110',
    'a139' => '10111',
    'a14' => '9997',
    'a140' => '10112',
    'a141' => '10113',
    'a142' => '10114',
    'a143' => '10115',
    'a144' => '10116',
    'a145' => '10117',
    'a146' => '10118',
    'a147' => '10119',
    'a148' => '10120',
    'a149' => '10121',
    'a15' => '9998',
    'a150' => '10122',
    'a151' => '10123',
    'a152' => '10124',
    'a153' => '10125',
    'a154' => '10126',
    'a155' => '10127',
    'a156' => '10128',
    'a157' => '10129',
    'a158' => '10130',
    'a159' => '10131',
    'a16' => '9999',
    'a160' => '10132',
    'a162' => '10147',
    'a165' => '10137',
    'a166' => '10139',
    'a167' => '10140',
    'a168' => '10141',
    'a169' => '10142',
    'a17' => '10001',
    'a170' => '10143',
    'a171' => '10144',
    'a172' => '10145',
    'a173' => '10146',
    'a174' => '10148',
    'a175' => '10149',
    'a176' => '10150',
    'a177' => '10151',
    'a178' => '10152',
    'a179' => '10153',
    'a18' => '10002',
    'a180' => '10155',
    'a181' => '10157',
    'a182' => '10159',
    'a183' => '10162',
    'a184' => '10163',
    'a185' => '10165',
    'a186' => '10168',
    'a187' => '10170',
    'a188' => '10171',
    'a189' => '10172',
    'a19' => '10003',
    'a190' => '10173',
    'a191' => '10174',
    'a192' => '10138',
    'a193' => '10154',
    'a194' => '10166',
    'a195' => '10169',
    'a196' => '10136',
    'a197' => '10164',
    'a198' => '10167',
    'a199' => '10156',
    'a2' => '9986',
    'a20' => '10004',
    'a200' => '10158',
    'a201' => '10161',
    'a202' => '9987',
    'a203' => '10064',
    'a204' => '10066',
    'a205' => '63709',
    'a206' => '63711',
    'a21' => '10005',
    'a22' => '10006',
    'a23' => '10007',
    'a24' => '10008',
    'a25' => '10009',
    'a26' => '10010',
    'a27' => '10011',
    'a28' => '10012',
    'a29' => '10018',
    'a3' => '9988',
    'a30' => '10019',
    'a31' => '10020',
    'a32' => '10021',
    'a33' => '10022',
    'a34' => '10023',
    'a35' => '9733',
    'a36' => '10025',
    'a37' => '10026',
    'a38' => '10027',
    'a39' => '10028',
    'a4' => '9742',
    'a40' => '10029',
    'a41' => '10030',
    'a42' => '10031',
    'a43' => '10032',
    'a44' => '10033',
    'a45' => '10034',
    'a46' => '10035',
    'a47' => '10036',
    'a48' => '10037',
    'a49' => '10038',
    'a5' => '9990',
    'a50' => '10039',
    'a51' => '10040',
    'a52' => '10041',
    'a53' => '10042',
    'a54' => '10043',
    'a55' => '10044',
    'a56' => '10045',
    'a57' => '10046',
    'a58' => '10047',
    'a59' => '10048',
    'a6' => '10013',
    'a60' => '10049',
    'a61' => '10050',
    'a62' => '10051',
    'a63' => '10052',
    'a64' => '10053',
    'a65' => '10054',
    'a66' => '10055',
    'a67' => '10056',
    'a68' => '10057',
    'a69' => '10058',
    'a7' => '10014',
    'a70' => '10059',
    'a72' => '10061',
    'a74' => '10063',
    'a75' => '10065',
    'a78' => '9670',
    'a79' => '10070',
    'a8' => '10015',
    'a81' => '9687',
    'a82' => '10072',
    'a83' => '10073',
    'a84' => '10074',
    'a85' => '63710',
    'a86' => '63712',
    'a87' => '63713',
    'a88' => '63714',
    'a89' => '63703',
    'a9' => '10016',
    'a90' => '63704',
    'a91' => '63707',
    'a92' => '63708',
    'a93' => '63705',
    'a94' => '63706',
    'a95' => '63715',
    'a96' => '63716',
    'a97' => '10075',
    'a98' => '10076',
    'a99' => '10077',
    'aabengali' => '2438',
    'aacgr' => '940',
    'aacute' => '225',
    'aadeva' => '2310',
    'aagujarati' => '2694',
    'aagurmukhi' => '2566',
    'aamatragurmukhi' => '2622',
    'aarusquare' => '13059',
    'aavowelsignbengali' => '2494',
    'aavowelsigndeva' => '2366',
    'aavowelsigngujarati' => '2750',
    'abbreviationmarkarmenian' => '1375',
    'abbreviationsigndeva' => '2416',
    'abengali' => '2437',
    'abopomofo' => '12570',
    'abreve' => '259',
    'abreveacute' => '7855',
    'abrevecyrillic' => '1233',
    'abrevedotbelow' => '7863',
    'abrevegrave' => '7857',
    'abrevehookabove' => '7859',
    'abrevetilde' => '7861',
    'acaron' => '462',
    'acirc' => '226',
    'acircle' => '9424',
    'acircumflex' => '226',
    'acircumflexacute' => '7845',
    'acircumflexdotbelow' => '7853',
    'acircumflexgrave' => '7847',
    'acircumflexhookabove' => '7849',
    'acircumflextilde' => '7851',
    'acute' => '180',
    'acutebelowcmb' => '791',
    'acutecmb' => '769',
    'acutecomb' => '769',
    'acutedeva' => '2388',
    'acutelowmod' => '719',
    'acutetonecmb' => '833',
    'acy' => '1072',
    'acyrillic' => '1072',
    'adblgrave' => '513',
    'addakgurmukhi' => '2673',
    'adeva' => '2309',
    'adieresis' => '228',
    'adieresiscyrillic' => '1235',
    'adieresismacron' => '479',
    'adotbelow' => '7841',
    'adotmacron' => '481',
    'ae' => '230',
    'aeacute' => '509',
    'aekorean' => '12624',
    'aelig' => '230',
    'aemacron' => '483',
    'afii00208' => '8213',
    'afii08941' => '8356',
    'afii10017' => '1040',
    'afii10018' => '1041',
    'afii10019' => '1042',
    'afii10020' => '1043',
    'afii10021' => '1044',
    'afii10022' => '1045',
    'afii10023' => '1025',
    'afii10024' => '1046',
    'afii10025' => '1047',
    'afii10026' => '1048',
    'afii10027' => '1049',
    'afii10028' => '1050',
    'afii10029' => '1051',
    'afii10030' => '1052',
    'afii10031' => '1053',
    'afii10032' => '1054',
    'afii10033' => '1055',
    'afii10034' => '1056',
    'afii10035' => '1057',
    'afii10036' => '1058',
    'afii10037' => '1059',
    'afii10038' => '1060',
    'afii10039' => '1061',
    'afii10040' => '1062',
    'afii10041' => '1063',
    'afii10042' => '1064',
    'afii10043' => '1065',
    'afii10044' => '1066',
    'afii10045' => '1067',
    'afii10046' => '1068',
    'afii10047' => '1069',
    'afii10048' => '1070',
    'afii10049' => '1071',
    'afii10050' => '1168',
    'afii10051' => '1026',
    'afii10052' => '1027',
    'afii10053' => '1028',
    'afii10054' => '1029',
    'afii10055' => '1030',
    'afii10056' => '1031',
    'afii10057' => '1032',
    'afii10058' => '1033',
    'afii10059' => '1034',
    'afii10060' => '1035',
    'afii10061' => '1036',
    'afii10062' => '1038',
    'afii10063' => '63172',
    'afii10064' => '63173',
    'afii10065' => '1072',
    'afii10066' => '1073',
    'afii10067' => '1074',
    'afii10068' => '1075',
    'afii10069' => '1076',
    'afii10070' => '1077',
    'afii10071' => '1105',
    'afii10072' => '1078',
    'afii10073' => '1079',
    'afii10074' => '1080',
    'afii10075' => '1081',
    'afii10076' => '1082',
    'afii10077' => '1083',
    'afii10078' => '1084',
    'afii10079' => '1085',
    'afii10080' => '1086',
    'afii10081' => '1087',
    'afii10082' => '1088',
    'afii10083' => '1089',
    'afii10084' => '1090',
    'afii10085' => '1091',
    'afii10086' => '1092',
    'afii10087' => '1093',
    'afii10088' => '1094',
    'afii10089' => '1095',
    'afii10090' => '1096',
    'afii10091' => '1097',
    'afii10092' => '1098',
    'afii10093' => '1099',
    'afii10094' => '1100',
    'afii10095' => '1101',
    'afii10096' => '1102',
    'afii10097' => '1103',
    'afii10098' => '1169',
    'afii10099' => '1106',
    'afii10100' => '1107',
    'afii10101' => '1108',
    'afii10102' => '1109',
    'afii10103' => '1110',
    'afii10104' => '1111',
    'afii10105' => '1112',
    'afii10106' => '1113',
    'afii10107' => '1114',
    'afii10108' => '1115',
    'afii10109' => '1116',
    'afii10110' => '1118',
    'afii10145' => '1039',
    'afii10146' => '1122',
    'afii10147' => '1138',
    'afii10148' => '1140',
    'afii10192' => '63174',
    'afii10193' => '1119',
    'afii10194' => '1123',
    'afii10195' => '1139',
    'afii10196' => '1141',
    'afii10831' => '63175',
    'afii10832' => '63176',
    'afii10846' => '1241',
    'afii299' => '8206',
    'afii300' => '8207',
    'afii301' => '8205',
    'afii57381' => '1642',
    'afii57388' => '1548',
    'afii57392' => '1632',
    'afii57393' => '1633',
    'afii57394' => '1634',
    'afii57395' => '1635',
    'afii57396' => '1636',
    'afii57397' => '1637',
    'afii57398' => '1638',
    'afii57399' => '1639',
    'afii57400' => '1640',
    'afii57401' => '1641',
    'afii57403' => '1563',
    'afii57407' => '1567',
    'afii57409' => '1569',
    'afii57410' => '1570',
    'afii57411' => '1571',
    'afii57412' => '1572',
    'afii57413' => '1573',
    'afii57414' => '1574',
    'afii57415' => '1575',
    'afii57416' => '1576',
    'afii57417' => '1577',
    'afii57418' => '1578',
    'afii57419' => '1579',
    'afii57420' => '1580',
    'afii57421' => '1581',
    'afii57422' => '1582',
    'afii57423' => '1583',
    'afii57424' => '1584',
    'afii57425' => '1585',
    'afii57426' => '1586',
    'afii57427' => '1587',
    'afii57428' => '1588',
    'afii57429' => '1589',
    'afii57430' => '1590',
    'afii57431' => '1591',
    'afii57432' => '1592',
    'afii57433' => '1593',
    'afii57434' => '1594',
    'afii57440' => '1600',
    'afii57441' => '1601',
    'afii57442' => '1602',
    'afii57443' => '1603',
    'afii57444' => '1604',
    'afii57445' => '1605',
    'afii57446' => '1606',
    'afii57448' => '1608',
    'afii57449' => '1609',
    'afii57450' => '1610',
    'afii57451' => '1611',
    'afii57452' => '1612',
    'afii57453' => '1613',
    'afii57454' => '1614',
    'afii57455' => '1615',
    'afii57456' => '1616',
    'afii57457' => '1617',
    'afii57458' => '1618',
    'afii57470' => '1607',
    'afii57505' => '1700',
    'afii57506' => '1662',
    'afii57507' => '1670',
    'afii57508' => '1688',
    'afii57509' => '1711',
    'afii57511' => '1657',
    'afii57512' => '1672',
    'afii57513' => '1681',
    'afii57514' => '1722',
    'afii57519' => '1746',
    'afii57534' => '1749',
    'afii57636' => '8362',
    'afii57645' => '1470',
    'afii57658' => '1475',
    'afii57664' => '1488',
    'afii57665' => '1489',
    'afii57666' => '1490',
    'afii57667' => '1491',
    'afii57668' => '1492',
    'afii57669' => '1493',
    'afii57670' => '1494',
    'afii57671' => '1495',
    'afii57672' => '1496',
    'afii57673' => '1497',
    'afii57674' => '1498',
    'afii57675' => '1499',
    'afii57676' => '1500',
    'afii57677' => '1501',
    'afii57678' => '1502',
    'afii57679' => '1503',
    'afii57680' => '1504',
    'afii57681' => '1505',
    'afii57682' => '1506',
    'afii57683' => '1507',
    'afii57684' => '1508',
    'afii57685' => '1509',
    'afii57686' => '1510',
    'afii57687' => '1511',
    'afii57688' => '1512',
    'afii57689' => '1513',
    'afii57690' => '1514',
    'afii57694' => '64298',
    'afii57695' => '64299',
    'afii57700' => '64331',
    'afii57705' => '64287',
    'afii57716' => '1520',
    'afii57717' => '1521',
    'afii57718' => '1522',
    'afii57723' => '64309',
    'afii57793' => '1460',
    'afii57794' => '1461',
    'afii57795' => '1462',
    'afii57796' => '1467',
    'afii57797' => '1464',
    'afii57798' => '1463',
    'afii57799' => '1456',
    'afii57800' => '1458',
    'afii57801' => '1457',
    'afii57802' => '1459',
    'afii57803' => '1474',
    'afii57804' => '1473',
    'afii57806' => '1465',
    'afii57807' => '1468',
    'afii57839' => '1469',
    'afii57841' => '1471',
    'afii57842' => '1472',
    'afii57929' => '700',
    'afii61248' => '8453',
    'afii61289' => '8467',
    'afii61352' => '8470',
    'afii61573' => '8236',
    'afii61574' => '8237',
    'afii61575' => '8238',
    'afii61664' => '8204',
    'afii63167' => '1645',
    'afii64937' => '701',
    'agr' => '945',
    'agrave' => '224',
    'agujarati' => '2693',
    'agurmukhi' => '2565',
    'ahiragana' => '12354',
    'ahookabove' => '7843',
    'aibengali' => '2448',
    'aibopomofo' => '12574',
    'aideva' => '2320',
    'aiecyrillic' => '1237',
    'aigujarati' => '2704',
    'aigurmukhi' => '2576',
    'aimatragurmukhi' => '2632',
    'ainarabic' => '1593',
    'ainfinalarabic' => '65226',
    'aininitialarabic' => '65227',
    'ainmedialarabic' => '65228',
    'ainvertedbreve' => '515',
    'airplane' => '61521',
    'aivowelsignbengali' => '2504',
    'aivowelsigndeva' => '2376',
    'aivowelsigngujarati' => '2760',
    'akatakana' => '12450',
    'akatakanahalfwidth' => '65393',
    'akorean' => '12623',
    'alef' => '1488',
    'alefarabic' => '1575',
    'alefdageshhebrew' => '64304',
    'aleffinalarabic' => '65166',
    'alefhamzaabovearabic' => '1571',
    'alefhamzaabovefinalarabic' => '65156',
    'alefhamzabelowarabic' => '1573',
    'alefhamzabelowfinalarabic' => '65160',
    'alefhebrew' => '1488',
    'aleflamedhebrew' => '64335',
    'alefmaddaabovearabic' => '1570',
    'alefmaddaabovefinalarabic' => '65154',
    'alefmaksuraarabic' => '1609',
    'alefmaksurafinalarabic' => '65264',
    'alefmaksurainitialarabic' => '65267',
    'alefmaksuramedialarabic' => '65268',
    'alefpatahhebrew' => '64302',
    'alefqamatshebrew' => '64303',
    'alefsym' => '8501',
    'aleph' => '8501',
    'allequal' => '8780',
    'alpha' => '945',
    'alphatonos' => '940',
    'amacr' => '257',
    'amacron' => '257',
    'amonospace' => '65345',
    'amp' => '38',
    'ampersand' => '38',
    'ampersandit' => '61547',
    'ampersanditlc' => '61546',
    'ampersandmonospace' => '65286',
    'ampersandsmall' => '63270',
    'amsquare' => '13250',
    'anbopomofo' => '12578',
    'and' => '8743',
    'ang' => '8736',
    'ang90' => '8735',
    'angbopomofo' => '12580',
    'angkhankhuthai' => '3674',
    'angle' => '8736',
    'anglebracketleft' => '12296',
    'anglebracketleftvertical' => '65087',
    'anglebracketright' => '12297',
    'anglebracketrightvertical' => '65088',
    'angleleft' => '9001',
    'angleright' => '9002',
    'angmsd' => '8737',
    'angsph' => '8738',
    'angst' => '8491',
    'angstrom' => '8491',
    'anoteleia' => '903',
    'anudattadeva' => '2386',
    'anusvarabengali' => '2434',
    'anusvaradeva' => '2306',
    'anusvaragujarati' => '2690',
    'aogon' => '261',
    'aogonek' => '261',
    'apaatosquare' => '13056',
    'aparen' => '9372',
    'ape' => '8778',
    'apos' => '39',
    'apostrophearmenian' => '1370',
    'apostrophemod' => '700',
    'apple' => '63743',
    'approaches' => '8784',
    'approxequal' => '8776',
    'approxequalorimage' => '8786',
    'approximatelyequal' => '8773',
    'aquarius' => '61544',
    'araeaekorean' => '12686',
    'araeakorean' => '12685',
    'arc' => '8978',
    'aries' => '61534',
    'arighthalfring' => '7834',
    'aring' => '229',
    'aringacute' => '507',
    'aringbelow' => '7681',
    'arrowboth' => '8596',
    'arrowdashdown' => '8675',
    'arrowdashleft' => '8672',
    'arrowdashright' => '8674',
    'arrowdashup' => '8673',
    'arrowdblboth' => '8660',
    'arrowdbldown' => '8659',
    'arrowdblleft' => '8656',
    'arrowdblright' => '8658',
    'arrowdblup' => '8657',
    'arrowdown' => '8595',
    'arrowdownleft' => '8601',
    'arrowdownright' => '8600',
    'arrowdownwhite' => '8681',
    'arrowdwnleft1' => '61635',
    'arrowdwnrt1' => '61636',
    'arrowheaddownmod' => '709',
    'arrowheadleftmod' => '706',
    'arrowheadrightmod' => '707',
    'arrowheadupmod' => '708',
    'arrowhorizex' => '63719',
    'arrowleft' => '8592',
    'arrowleftdbl' => '8656',
    'arrowleftdblstroke' => '8653',
    'arrowleftdwn1' => '61641',
    'arrowleftoverright' => '8646',
    'arrowleftup1' => '61639',
    'arrowleftwhite' => '8678',
    'arrowright' => '8594',
    'arrowrightdblstroke' => '8655',
    'arrowrightheavy' => '10142',
    'arrowrightoverleft' => '8644',
    'arrowrightwhite' => '8680',
    'arrowrtdwn1' => '61642',
    'arrowrtup1' => '61640',
    'arrowtableft' => '8676',
    'arrowtabright' => '8677',
    'arrowup' => '8593',
    'arrowupdn' => '8597',
    'arrowupdnbse' => '8616',
    'arrowupdownbase' => '8616',
    'arrowupleft' => '8598',
    'arrowupleft1' => '61637',
    'arrowupleftofdown' => '8645',
    'arrowupright' => '8599',
    'arrowuprt1' => '61638',
    'arrowupwhite' => '8679',
    'arrowvertex' => '63718',
    'asciicircum' => '94',
    'asciicircummonospace' => '65342',
    'asciitilde' => '126',
    'asciitildemonospace' => '65374',
    'ascript' => '593',
    'ascriptturned' => '594',
    'asmallhiragana' => '12353',
    'asmallkatakana' => '12449',
    'asmallkatakanahalfwidth' => '65383',
    'ast' => '42',
    'asterisk' => '42',
    'asteriskaltonearabic' => '1645',
    'asteriskarabic' => '1645',
    'asteriskmath' => '8727',
    'asteriskmonospace' => '65290',
    'asterisksmall' => '65121',
    'asterism' => '8258',
    'asuperior' => '63209',
    'asymp' => '8776',
    'asymptoticallyequal' => '8771',
    'at' => '64',
    'atilde' => '227',
    'atmonospace' => '65312',
    'atsmall' => '65131',
    'aturned' => '592',
    'aubengali' => '2452',
    'aubopomofo' => '12576',
    'audeva' => '2324',
    'augujarati' => '2708',
    'augurmukhi' => '2580',
    'aulengthmarkbengali' => '2519',
    'aumatragurmukhi' => '2636',
    'auml' => '228',
    'auvowelsignbengali' => '2508',
    'auvowelsigndeva' => '2380',
    'auvowelsigngujarati' => '2764',
    'avagrahadeva' => '2365',
    'aybarmenian' => '1377',
    'ayin' => '1506',
    'ayinaltonehebrew' => '64288',
    'ayinhebrew' => '1506',
    'b' => '98',
    'babengali' => '2476',
    'backslash' => '92',
    'backslashmonospace' => '65340',
    'badeva' => '2348',
    'bagujarati' => '2732',
    'bagurmukhi' => '2604',
    'bahiragana' => '12400',
    'bahtthai' => '3647',
    'bakatakana' => '12496',
    'ballpoint' => '61497',
    'bar' => '124',
    'barb2down' => '61666',
    'barb2left' => '61663',
    'barb2ne' => '61668',
    'barb2nw' => '61667',
    'barb2right' => '61664',
    'barb2se' => '61670',
    'barb2sw' => '61669',
    'barb2up' => '61665',
    'barb4down' => '61674',
    'barb4left' => '61671',
    'barb4ne' => '61676',
    'barb4nw' => '61675',
    'barb4right' => '61672',
    'barb4se' => '61678',
    'barb4sw' => '61677',
    'barb4up' => '61673',
    'barmonospace' => '65372',
    'barwed' => '8892',
    'bbopomofo' => '12549',
    'bcircle' => '9425',
    'bcong' => '8780',
    'bcy' => '1073',
    'bdash1' => '61689',
    'bdash2' => '61690',
    'bdotaccent' => '7683',
    'bdotbelow' => '7685',
    'bdown' => '61682',
    'bdquo' => '8222',
    'beamedsixteenthnotes' => '9836',
    'becaus' => '8757',
    'because' => '8757',
    'becyrillic' => '1073',
    'beharabic' => '1576',
    'behfinalarabic' => '65168',
    'behinitialarabic' => '65169',
    'behiragana' => '12409',
    'behmedialarabic' => '65170',
    'behmeeminitialarabic' => '64671',
    'behmeemisolatedarabic' => '64520',
    'behnoonfinalarabic' => '64621',
    'bekatakana' => '12505',
    'bell' => '61477',
    'benarmenian' => '1378',
    'bepsi' => '8717',
    'bernou' => '8492',
    'bet' => '1489',
    'beta' => '946',
    'betasymbolgreek' => '976',
    'betdagesh' => '64305',
    'betdageshhebrew' => '64305',
    'beth' => '8502',
    'bethebrew' => '1489',
    'betrafehebrew' => '64332',
    'bgr' => '946',
    'bhabengali' => '2477',
    'bhadeva' => '2349',
    'bhagujarati' => '2733',
    'bhagurmukhi' => '2605',
    'bhook' => '595',
    'bihiragana' => '12403',
    'bikatakana' => '12499',
    'bilabialclick' => '664',
    'bindigurmukhi' => '2562',
    'birusquare' => '13105',
    'blackcircle' => '9679',
    'blackdiamond' => '9670',
    'blackdownpointingtriangle' => '9660',
    'blackleftpointingpointer' => '9668',
    'blackleftpointingtriangle' => '9664',
    'blacklenticularbracketleft' => '12304',
    'blacklenticularbracketleftvertical' => '65083',
    'blacklenticularbracketright' => '12305',
    'blacklenticularbracketrightvertical' => '65084',
    'blacklowerlefttriangle' => '9699',
    'blacklowerrighttriangle' => '9698',
    'blackrectangle' => '9644',
    'blackrightpointingpointer' => '9658',
    'blackrightpointingtriangle' => '9654',
    'blacksmallsquare' => '9642',
    'blacksmilingface' => '9787',
    'blacksquare' => '9632',
    'blackstar' => '9733',
    'blackupperlefttriangle' => '9700',
    'blackupperrighttriangle' => '9701',
    'blackuppointingsmalltriangle' => '9652',
    'blackuppointingtriangle' => '9650',
    'blank' => '9251',
    'bleft' => '61679',
    'bleftright' => '61683',
    'blinebelow' => '7687',
    'blk12' => '9618',
    'blk14' => '9617',
    'blk34' => '9619',
    'block' => '9608',
    'bmonospace' => '65346',
    'bne' => '61686',
    'bnw' => '61685',
    'bobaimaithai' => '3610',
    'bohiragana' => '12412',
    'bokatakana' => '12508',
    'bomb' => '61517',
    'book' => '61478',
    'bottom' => '8869',
    'bowtie' => '8904',
    'box2' => '61608',
    'box3' => '61551',
    'box4' => '61552',
    'boxDL' => '9559',
    'boxDR' => '9556',
    'boxDl' => '9558',
    'boxDr' => '9555',
    'boxH' => '9552',
    'boxHD' => '9574',
    'boxHU' => '9577',
    'boxHd' => '9572',
    'boxHu' => '9575',
    'boxUL' => '9565',
    'boxUR' => '9562',
    'boxUl' => '9564',
    'boxUr' => '9561',
    'boxV' => '9553',
    'boxVH' => '9580',
    'boxVL' => '9571',
    'boxVR' => '9568',
    'boxVh' => '9579',
    'boxVl' => '9570',
    'boxVr' => '9567',
    'boxcheckbld' => '61694',
    'boxdL' => '9557',
    'boxdR' => '9554',
    'boxdl' => '9488',
    'boxdr' => '9484',
    'boxh' => '9472',
    'boxhD' => '9573',
    'boxhU' => '9576',
    'boxhd' => '9516',
    'boxhu' => '9524',
    'boxshadowdwn' => '61553',
    'boxshadowup' => '61554',
    'boxuL' => '9563',
    'boxuR' => '9560',
    'boxul' => '9496',
    'boxur' => '9492',
    'boxv' => '9474',
    'boxvH' => '9578',
    'boxvL' => '9569',
    'boxvR' => '9566',
    'boxvh' => '9532',
    'boxvl' => '9508',
    'boxvr' => '9500',
    'boxxmarkbld' => '61693',
    'bparen' => '9373',
    'bprime' => '8245',
    'bqsquare' => '13251',
    'braceex' => '63732',
    'braceleft' => '123',
    'braceleftbt' => '63731',
    'braceleftmid' => '63730',
    'braceleftmonospace' => '65371',
    'braceleftsmall' => '65115',
    'bracelefttp' => '63729',
    'braceleftvertical' => '65079',
    'braceright' => '125',
    'bracerightbt' => '63742',
    'bracerightmid' => '63741',
    'bracerightmonospace' => '65373',
    'bracerightsmall' => '65116',
    'bracerighttp' => '63740',
    'bracerightvertical' => '65080',
    'bracketleft' => '91',
    'bracketleftbt' => '63728',
    'bracketleftex' => '63727',
    'bracketleftmonospace' => '65339',
    'bracketlefttp' => '63726',
    'bracketright' => '93',
    'bracketrightbt' => '63739',
    'bracketrightex' => '63738',
    'bracketrightmonospace' => '65341',
    'bracketrighttp' => '63737',
    'breve' => '728',
    'brevebelowcmb' => '814',
    'brevecmb' => '774',
    'breveinvertedbelowcmb' => '815',
    'breveinvertedcmb' => '785',
    'breveinverteddoublecmb' => '865',
    'bridgebelowcmb' => '810',
    'bridgeinvertedbelowcmb' => '826',
    'bright' => '61680',
    'brokenbar' => '166',
    'brvbar' => '166',
    'bse' => '61688',
    'bsim' => '8765',
    'bsime' => '8909',
    'bsol' => '92',
    'bstroke' => '384',
    'bsuperior' => '63210',
    'bsw' => '61687',
    'btopbar' => '387',
    'budleafne' => '61590',
    'budleafnw' => '61591',
    'budleafse' => '61593',
    'budleafsw' => '61592',
    'buhiragana' => '12406',
    'bukatakana' => '12502',
    'bull' => '8226',
    'bullet' => '8226',
    'bulletinverse' => '9688',
    'bulletoperator' => '8729',
    'bullseye' => '9678',
    'bump' => '8782',
    'bumpe' => '8783',
    'bup' => '61681',
    'bupdown' => '61684',
    'c' => '99',
    'caarmenian' => '1390',
    'cabengali' => '2458',
    'cacute' => '263',
    'cadeva' => '2330',
    'cagujarati' => '2714',
    'cagurmukhi' => '2586',
    'calsquare' => '13192',
    'cancer' => '61537',
    'candle' => '61479',
    'candrabindubengali' => '2433',
    'candrabinducmb' => '784',
    'candrabindudeva' => '2305',
    'candrabindugujarati' => '2689',
    'cap' => '8745',
    'capricorn' => '61543',
    'capslock' => '8682',
    'careof' => '8453',
    'caret' => '8257',
    'caron' => '711',
    'caronbelowcmb' => '812',
    'caroncmb' => '780',
    'carriagereturn' => '8629',
    'cbopomofo' => '12568',
    'ccaron' => '269',
    'ccedil' => '231',
    'ccedilla' => '231',
    'ccedillaacute' => '7689',
    'ccirc' => '265',
    'ccircle' => '9426',
    'ccircumflex' => '265',
    'ccurl' => '597',
    'cdot' => '267',
    'cdotaccent' => '267',
    'cdsquare' => '13253',
    'cedil' => '184',
    'cedilla' => '184',
    'cedillacmb' => '807',
    'cent' => '162',
    'centigrade' => '8451',
    'centinferior' => '63199',
    'centmonospace' => '65504',
    'centoldstyle' => '63394',
    'centsuperior' => '63200',
    'chaarmenian' => '1401',
    'chabengali' => '2459',
    'chadeva' => '2331',
    'chagujarati' => '2715',
    'chagurmukhi' => '2587',
    'chbopomofo' => '12564',
    'chcy' => '1095',
    'cheabkhasiancyrillic' => '1213',
    'check' => '10003',
    'checkbld' => '61692',
    'checkmark' => '10003',
    'checyrillic' => '1095',
    'chedescenderabkhasiancyrillic' => '1215',
    'chedescendercyrillic' => '1207',
    'chedieresiscyrillic' => '1269',
    'cheharmenian' => '1395',
    'chekhakassiancyrillic' => '1228',
    'cheverticalstrokecyrillic' => '1209',
    'chi' => '967',
    'chieuchacirclekorean' => '12919',
    'chieuchaparenkorean' => '12823',
    'chieuchcirclekorean' => '12905',
    'chieuchkorean' => '12618',
    'chieuchparenkorean' => '12809',
    'chochangthai' => '3594',
    'chochanthai' => '3592',
    'chochingthai' => '3593',
    'chochoethai' => '3596',
    'chook' => '392',
    'cieucacirclekorean' => '12918',
    'cieucaparenkorean' => '12822',
    'cieuccirclekorean' => '12904',
    'cieuckorean' => '12616',
    'cieucparenkorean' => '12808',
    'cieucuparenkorean' => '12828',
    'circ' => '710',
    'circle' => '9675',
    'circle2' => '61598',
    'circle4' => '61599',
    'circle6' => '61548',
    'circledown' => '61662',
    'circleleft' => '61659',
    'circlemultiply' => '8855',
    'circleot' => '8857',
    'circleplus' => '8853',
    'circlepostalmark' => '12342',
    'circleright' => '61660',
    'circleshadowdwn' => '61549',
    'circleshadowup' => '61606',
    'circlestar' => '61621',
    'circleup' => '61661',
    'circlewithlefthalfblack' => '9680',
    'circlewithrighthalfblack' => '9681',
    'circumflex' => '710',
    'circumflexbelowcmb' => '813',
    'circumflexcmb' => '770',
    'cire' => '8791',
    'clear' => '8999',
    'clickalveolar' => '450',
    'clickdental' => '448',
    'clicklateral' => '449',
    'clickretroflex' => '451',
    'club' => '9827',
    'clubs' => '9827',
    'clubsuitblack' => '9827',
    'clubsuitwhite' => '9831',
    'cmcubedsquare' => '13220',
    'cmonospace' => '65347',
    'cmsquaredsquare' => '13216',
    'coarmenian' => '1409',
    'colon' => '58',
    'colone' => '8788',
    'colonmonetary' => '8353',
    'colonmonospace' => '65306',
    'colonsign' => '8353',
    'colonsmall' => '65109',
    'colontriangularhalfmod' => '721',
    'colontriangularmod' => '720',
    'comma' => '44',
    'commaabovecmb' => '787',
    'commaaboverightcmb' => '789',
    'commaaccent' => '63171',
    'commaarabic' => '1548',
    'commaarmenian' => '1373',
    'commainferior' => '63201',
    'commamonospace' => '65292',
    'command' => '61562',
    'commareversedabovecmb' => '788',
    'commareversedmod' => '701',
    'commasmall' => '65104',
    'commasuperior' => '63202',
    'commat' => '64',
    'commaturnedabovecmb' => '786',
    'commaturnedmod' => '699',
    'comp' => '8705',
    'compass' => '9788',
    'compfn' => '8728',
    'cong' => '8773',
    'congruent' => '8773',
    'conint' => '8750',
    'contourintegral' => '8750',
    'control' => '8963',
    'controlACK' => '6',
    'controlBEL' => '7',
    'controlBS' => '8',
    'controlCAN' => '24',
    'controlCR' => '13',
    'controlDC1' => '17',
    'controlDC2' => '18',
    'controlDC3' => '19',
    'controlDC4' => '20',
    'controlDEL' => '127',
    'controlDLE' => '16',
    'controlEM' => '25',
    'controlENQ' => '5',
    'controlEOT' => '4',
    'controlESC' => '27',
    'controlETB' => '23',
    'controlETX' => '3',
    'controlFF' => '12',
    'controlFS' => '28',
    'controlGS' => '29',
    'controlHT' => '9',
    'controlLF' => '10',
    'controlNAK' => '21',
    'controlRS' => '30',
    'controlSI' => '15',
    'controlSO' => '14',
    'controlSOT' => '2',
    'controlSTX' => '1',
    'controlSUB' => '26',
    'controlSYN' => '22',
    'controlUS' => '31',
    'controlVT' => '11',
    'copy' => '169',
    'copyright' => '169',
    'copyrightsans' => '63721',
    'copyrightserif' => '63193',
    'copysr' => '8471',
    'cornerbracketleft' => '12300',
    'cornerbracketlefthalfwidth' => '65378',
    'cornerbracketleftvertical' => '65089',
    'cornerbracketright' => '12301',
    'cornerbracketrighthalfwidth' => '65379',
    'cornerbracketrightvertical' => '65090',
    'corporationsquare' => '13183',
    'cosquare' => '13255',
    'coverkgsquare' => '13254',
    'cparen' => '9374',
    'crarr' => '8629',
    'crescentstar' => '61530',
    'cross' => '10007',
    'crossceltic' => '61527',
    'crossmaltese' => '61528',
    'crossoutline' => '61525',
    'crossshadow' => '61526',
    'crosstar2' => '61610',
    'cruzeiro' => '8354',
    'cstretched' => '663',
    'cuepr' => '8926',
    'cuesc' => '8927',
    'cularr' => '8630',
    'cup' => '8746',
    'cupre' => '8828',
    'curarr' => '8631',
    'curlyand' => '8911',
    'curlyor' => '8910',
    'curren' => '164',
    'currency' => '164',
    'cuspopen' => '61618',
    'cuspopen1' => '61619',
    'cuvee' => '8910',
    'cuwed' => '8911',
    'cyrBreve' => '63185',
    'cyrFlex' => '63186',
    'cyrbreve' => '63188',
    'cyrflex' => '63189',
    'd' => '100',
    'dArr' => '8659',
    'daarmenian' => '1380',
    'dabengali' => '2470',
    'dadarabic' => '1590',
    'dadeva' => '2342',
    'dadfinalarabic' => '65214',
    'dadinitialarabic' => '65215',
    'dadmedialarabic' => '65216',
    'dagesh' => '1468',
    'dageshhebrew' => '1468',
    'dagger' => '8224',
    'daggerdbl' => '8225',
    'dagujarati' => '2726',
    'dagurmukhi' => '2598',
    'dahiragana' => '12384',
    'dakatakana' => '12480',
    'dalarabic' => '1583',
    'dalet' => '1491',
    'daletdagesh' => '64307',
    'daletdageshhebrew' => '64307',
    'daleth' => '8504',
    'dalethatafpatah' => '1491',
    'dalethatafpatahhebrew' => '1491',
    'dalethatafsegol' => '1491',
    'dalethatafsegolhebrew' => '1491',
    'dalethebrew' => '1491',
    'dalethiriq' => '1491',
    'dalethiriqhebrew' => '1491',
    'daletholam' => '1491',
    'daletholamhebrew' => '1491',
    'daletpatah' => '1491',
    'daletpatahhebrew' => '1491',
    'daletqamats' => '1491',
    'daletqamatshebrew' => '1491',
    'daletqubuts' => '1491',
    'daletqubutshebrew' => '1491',
    'daletsegol' => '1491',
    'daletsegolhebrew' => '1491',
    'daletsheva' => '1491',
    'daletshevahebrew' => '1491',
    'dalettsere' => '1491',
    'dalettserehebrew' => '1491',
    'dalfinalarabic' => '65194',
    'dammaarabic' => '1615',
    'dammalowarabic' => '1615',
    'dammatanaltonearabic' => '1612',
    'dammatanarabic' => '1612',
    'danda' => '2404',
    'dargahebrew' => '1447',
    'dargalefthebrew' => '1447',
    'darr' => '8595',
    'darr2' => '8650',
    'dash' => '8208',
    'dashv' => '8867',
    'dasiapneumatacyrilliccmb' => '1157',
    'dblGrave' => '63187',
    'dblac' => '733',
    'dblanglebracketleft' => '12298',
    'dblanglebracketleftvertical' => '65085',
    'dblanglebracketright' => '12299',
    'dblanglebracketrightvertical' => '65086',
    'dblarchinvertedbelowcmb' => '811',
    'dblarrowleft' => '8660',
    'dblarrowright' => '8658',
    'dbldanda' => '2405',
    'dblgrave' => '63190',
    'dblgravecmb' => '783',
    'dblintegral' => '8748',
    'dbllowline' => '8215',
    'dbllowlinecmb' => '819',
    'dbloverlinecmb' => '831',
    'dblprimemod' => '698',
    'dblverticalbar' => '8214',
    'dblverticallineabovecmb' => '782',
    'dbopomofo' => '12553',
    'dbsquare' => '13256',
    'dcaron' => '271',
    'dcedilla' => '7697',
    'dcircle' => '9427',
    'dcircumflexbelow' => '7699',
    'dcroat' => '273',
    'dcy' => '1044',
    'ddabengali' => '2465',
    'ddadeva' => '2337',
    'ddagujarati' => '2721',
    'ddagurmukhi' => '2593',
    'ddalarabic' => '1672',
    'ddalfinalarabic' => '64393',
    'dddhadeva' => '2396',
    'ddhabengali' => '2466',
    'ddhadeva' => '2338',
    'ddhagujarati' => '2722',
    'ddhagurmukhi' => '2594',
    'ddotaccent' => '7691',
    'ddotbelow' => '7693',
    'decimalseparatorarabic' => '1643',
    'decimalseparatorpersian' => '1643',
    'decyrillic' => '1076',
    'deg' => '176',
    'degree' => '176',
    'dehihebrew' => '1453',
    'dehiragana' => '12391',
    'deicoptic' => '1007',
    'dekatakana' => '12487',
    'deleteleft' => '9003',
    'deleteright' => '8998',
    'delta' => '948',
    'deltaturned' => '397',
    'denominatorminusonenumeratorbengali' => '2552',
    'dezh' => '676',
    'dgr' => '948',
    'dhabengali' => '2471',
    'dhadeva' => '2343',
    'dhagujarati' => '2727',
    'dhagurmukhi' => '2599',
    'dharl' => '8643',
    'dharr' => '8642',
    'dhook' => '599',
    'dialytikatonos' => '901',
    'dialytikatonoscmb' => '836',
    'diam' => '8900',
    'diamond' => '9830',
    'diamondsuitwhite' => '9826',
    'diams' => '9830',
    'die' => '168',
    'dieresis' => '168',
    'dieresisacute' => '63191',
    'dieresisbelowcmb' => '804',
    'dieresiscmb' => '776',
    'dieresisgrave' => '63192',
    'dieresistonos' => '901',
    'dihiragana' => '12386',
    'dikatakana' => '12482',
    'dittomark' => '12291',
    'divide' => '247',
    'divides' => '8739',
    'divisionslash' => '8725',
    'divonx' => '8903',
    'djcy' => '1106',
    'djecyrillic' => '1106',
    'dkshade' => '9619',
    'dlarr' => '8601',
    'dlcorn' => '8990',
    'dlcrop' => '8973',
    'dlinebelow' => '7695',
    'dlsquare' => '13207',
    'dmacron' => '273',
    'dmonospace' => '65348',
    'dnblock' => '9604',
    'dochadathai' => '3598',
    'dodecastar3' => '61614',
    'dodekthai' => '3604',
    'dohiragana' => '12393',
    'dokatakana' => '12489',
    'dollar' => '36',
    'dollarinferior' => '63203',
    'dollarmonospace' => '65284',
    'dollaroldstyle' => '63268',
    'dollarsmall' => '65129',
    'dollarsuperior' => '63204',
    'dong' => '8363',
    'dorusquare' => '13094',
    'dot' => '729',
    'dotaccent' => '729',
    'dotaccentcmb' => '775',
    'dotbelowcmb' => '803',
    'dotbelowcomb' => '803',
    'dotkatakana' => '12539',
    'dotlessi' => '305',
    'dotlessj' => '63166',
    'dotlessjstrokehook' => '644',
    'dotmath' => '8901',
    'dottedcircle' => '9676',
    'doubleyodpatah' => '64287',
    'doubleyodpatahhebrew' => '64287',
    'downtackbelowcmb' => '798',
    'downtackmod' => '725',
    'dparen' => '9375',
    'drarr' => '8600',
    'drcorn' => '8991',
    'drcrop' => '8972',
    'droplet' => '61523',
    'dscy' => '1109',
    'dstrok' => '273',
    'dsuperior' => '63211',
    'dtail' => '598',
    'dtopbar' => '396',
    'dtri' => '9663',
    'dtrif' => '9662',
    'duhiragana' => '12389',
    'dukatakana' => '12485',
    'dz' => '499',
    'dzaltone' => '675',
    'dzcaron' => '454',
    'dzcurl' => '677',
    'dzcy' => '1119',
    'dzeabkhasiancyrillic' => '1249',
    'dzecyrillic' => '1109',
    'dzhecyrillic' => '1119',
    'e' => '101',
    'eDot' => '8785',
    'eacgr' => '941',
    'eacute' => '233',
    'earth' => '9793',
    'ebengali' => '2447',
    'ebopomofo' => '12572',
    'ebreve' => '277',
    'ecandradeva' => '2317',
    'ecandragujarati' => '2701',
    'ecandravowelsigndeva' => '2373',
    'ecandravowelsigngujarati' => '2757',
    'ecaron' => '283',
    'ecedillabreve' => '7709',
    'echarmenian' => '1381',
    'echyiwnarmenian' => '1415',
    'ecir' => '8790',
    'ecirc' => '234',
    'ecircle' => '9428',
    'ecircumflex' => '234',
    'ecircumflexacute' => '7871',
    'ecircumflexbelow' => '7705',
    'ecircumflexdotbelow' => '7879',
    'ecircumflexgrave' => '7873',
    'ecircumflexhookabove' => '7875',
    'ecircumflextilde' => '7877',
    'ecolon' => '8789',
    'ecy' => '1101',
    'ecyrillic' => '1108',
    'edblgrave' => '517',
    'edeva' => '2319',
    'edieresis' => '235',
    'edot' => '279',
    'edotaccent' => '279',
    'edotbelow' => '7865',
    'eeacgr' => '942',
    'eegr' => '951',
    'eegurmukhi' => '2575',
    'eematragurmukhi' => '2631',
    'efDot' => '8786',
    'efcyrillic' => '1092',
    'egrave' => '232',
    'egs' => '8925',
    'egujarati' => '2703',
    'eharmenian' => '1383',
    'ehbopomofo' => '12573',
    'ehiragana' => '12360',
    'ehookabove' => '7867',
    'eibopomofo' => '12575',
    'eight' => '56',
    'eightarabic' => '1640',
    'eightbengali' => '2542',
    'eightcircle' => '9319',
    'eightcircleinversesansserif' => '10129',
    'eightdeva' => '2414',
    'eighteencircle' => '9329',
    'eighteenparen' => '9349',
    'eighteenperiod' => '9369',
    'eightgujarati' => '2798',
    'eightgurmukhi' => '2670',
    'eighthackarabic' => '1640',
    'eighthangzhou' => '12328',
    'eighthnotebeamed' => '9835',
    'eightideographicparen' => '12839',
    'eightinferior' => '8328',
    'eightmonospace' => '65304',
    'eightoclock' => '61630',
    'eightoldstyle' => '63288',
    'eightparen' => '9339',
    'eightperiod' => '9359',
    'eightpersian' => '1784',
    'eightroman' => '8567',
    'eightsans' => '61576',
    'eightsansinv' => '61587',
    'eightsuperior' => '8312',
    'eightthai' => '3672',
    'einvertedbreve' => '519',
    'eiotifiedcyrillic' => '1125',
    'ekatakana' => '12456',
    'ekatakanahalfwidth' => '65396',
    'ekonkargurmukhi' => '2676',
    'ekorean' => '12628',
    'elcyrillic' => '1083',
    'element' => '8712',
    'elevencircle' => '9322',
    'elevenoclock' => '61633',
    'elevenparen' => '9342',
    'elevenperiod' => '9362',
    'elevenroman' => '8570',
    'ell' => '8467',
    'ellipsis' => '8230',
    'ellipsisvertical' => '8942',
    'els' => '8924',
    'emacr' => '275',
    'emacron' => '275',
    'emacronacute' => '7703',
    'emacrongrave' => '7701',
    'emcyrillic' => '1084',
    'emdash' => '8212',
    'emdashvertical' => '65073',
    'emonospace' => '65349',
    'emphasismarkarmenian' => '1371',
    'empty' => '8709',
    'emptyset' => '8709',
    'emsp' => '8195',
    'emsp13' => '8196',
    'emsp14' => '8197',
    'enbopomofo' => '12579',
    'encyrillic' => '1085',
    'endash' => '8211',
    'endashvertical' => '65074',
    'endescendercyrillic' => '1187',
    'eng' => '331',
    'engbopomofo' => '12581',
    'enghecyrillic' => '1189',
    'enhookcyrillic' => '1224',
    'ensp' => '8194',
    'enspace' => '8194',
    'envelopeback' => '61482',
    'envelopefront' => '61483',
    'eogon' => '281',
    'eogonek' => '281',
    'eokorean' => '12627',
    'eopen' => '603',
    'eopenclosed' => '666',
    'eopenreversed' => '604',
    'eopenreversedclosed' => '606',
    'eopenreversedhook' => '605',
    'eparen' => '9376',
    'epsi' => '949',
    'epsilon' => '949',
    'epsilontonos' => '941',
    'epsis' => '8714',
    'equal' => '61',
    'equalmonospace' => '65309',
    'equals' => '61',
    'equalsmall' => '65126',
    'equalsuperior' => '8316',
    'equiv' => '8801',
    'equivalence' => '8801',
    'erDot' => '8787',
    'erbopomofo' => '12582',
    'ercyrillic' => '1088',
    'ereversed' => '600',
    'ereversedcyrillic' => '1101',
    'escape' => '61561',
    'escyrillic' => '1089',
    'esdescendercyrillic' => '1195',
    'esdot' => '8784',
    'esh' => '643',
    'eshcurl' => '646',
    'eshortdeva' => '2318',
    'eshortvowelsigndeva' => '2374',
    'eshreversedloop' => '426',
    'eshsquatreversed' => '645',
    'esmallhiragana' => '12359',
    'esmallkatakana' => '12455',
    'esmallkatakanahalfwidth' => '65386',
    'estimated' => '8494',
    'esuperior' => '63212',
    'eta' => '951',
    'etarmenian' => '1384',
    'etatonos' => '942',
    'eth' => '240',
    'etilde' => '7869',
    'etildebelow' => '7707',
    'etnahtafoukhhebrew' => '1425',
    'etnahtafoukhlefthebrew' => '1425',
    'etnahtahebrew' => '1425',
    'etnahtalefthebrew' => '1425',
    'eturned' => '477',
    'eukorean' => '12641',
    'euml' => '235',
    'euro' => '8364',
    'evowelsignbengali' => '2503',
    'evowelsigndeva' => '2375',
    'evowelsigngujarati' => '2759',
    'excl' => '33',
    'exclam' => '33',
    'exclamarmenian' => '1372',
    'exclamdbl' => '8252',
    'exclamdown' => '161',
    'exclamdownsmall' => '63393',
    'exclammonospace' => '65281',
    'exclamsmall' => '63265',
    'exist' => '8707',
    'existential' => '8707',
    'ezh' => '658',
    'ezhcaron' => '495',
    'ezhcurl' => '659',
    'ezhreversed' => '441',
    'ezhtail' => '442',
    'f' => '102',
    'fadeva' => '2398',
    'fagurmukhi' => '2654',
    'fahrenheit' => '8457',
    'fathaarabic' => '1614',
    'fathalowarabic' => '1614',
    'fathatanarabic' => '1611',
    'fbopomofo' => '12552',
    'fcircle' => '9429',
    'fcy' => '1092',
    'fdotaccent' => '7711',
    'feharabic' => '1601',
    'feharmenian' => '1414',
    'fehfinalarabic' => '65234',
    'fehinitialarabic' => '65235',
    'fehmedialarabic' => '65236',
    'feicoptic' => '997',
    'female' => '9792',
    'ff' => '64256',
    'ffi' => '64259',
    'ffl' => '64260',
    'fi' => '64257',
    'fifteencircle' => '9326',
    'fifteenparen' => '9346',
    'fifteenperiod' => '9366',
    'figuredash' => '8210',
    'filecabinet' => '61493',
    'filetalltext' => '61491',
    'filetalltext1' => '61490',
    'filetalltext3' => '61492',
    'filledbox' => '9632',
    'filledrect' => '9644',
    'finalkaf' => '1498',
    'finalkafdagesh' => '64314',
    'finalkafdageshhebrew' => '64314',
    'finalkafhebrew' => '1498',
    'finalkafqamats' => '1498',
    'finalkafqamatshebrew' => '1498',
    'finalkafsheva' => '1498',
    'finalkafshevahebrew' => '1498',
    'finalmem' => '1501',
    'finalmemhebrew' => '1501',
    'finalnun' => '1503',
    'finalnunhebrew' => '1503',
    'finalpe' => '1507',
    'finalpehebrew' => '1507',
    'finaltsadi' => '1509',
    'finaltsadihebrew' => '1509',
    'firsttonechinese' => '713',
    'fisheye' => '9673',
    'fitacyrillic' => '1139',
    'five' => '53',
    'fivearabic' => '1637',
    'fivebengali' => '2539',
    'fivecircle' => '9316',
    'fivecircleinversesansserif' => '10126',
    'fivedeva' => '2411',
    'fiveeighths' => '8541',
    'fivegujarati' => '2795',
    'fivegurmukhi' => '2667',
    'fivehackarabic' => '1637',
    'fivehangzhou' => '12325',
    'fiveideographicparen' => '12836',
    'fiveinferior' => '8325',
    'fivemonospace' => '65301',
    'fiveoclock' => '61627',
    'fiveoldstyle' => '63285',
    'fiveparen' => '9336',
    'fiveperiod' => '9356',
    'fivepersian' => '1781',
    'fiveroman' => '8564',
    'fivesans' => '61573',
    'fivesansinv' => '61584',
    'fivesuperior' => '8309',
    'fivethai' => '3669',
    'fl' => '64258',
    'flag' => '61519',
    'flat' => '9837',
    'floppy3' => '61500',
    'floppy5' => '61501',
    'florin' => '402',
    'fmonospace' => '65350',
    'fmsquare' => '13209',
    'fnof' => '402',
    'fofanthai' => '3615',
    'fofathai' => '3613',
    'folder' => '61488',
    'folderopen' => '61489',
    'fongmanthai' => '3663',
    'forall' => '8704',
    'fork' => '8916',
    'four' => '52',
    'fourarabic' => '1636',
    'fourbengali' => '2538',
    'fourcircle' => '9315',
    'fourcircleinversesansserif' => '10125',
    'fourdeva' => '2410',
    'fourgujarati' => '2794',
    'fourgurmukhi' => '2666',
    'fourhackarabic' => '1636',
    'fourhangzhou' => '12324',
    'fourideographicparen' => '12835',
    'fourinferior' => '8324',
    'fourmonospace' => '65300',
    'fournumeratorbengali' => '2551',
    'fouroclock' => '61626',
    'fouroldstyle' => '63284',
    'fourparen' => '9335',
    'fourperiod' => '9355',
    'fourpersian' => '1780',
    'fourroman' => '8563',
    'foursans' => '61572',
    'foursansinv' => '61583',
    'foursuperior' => '8308',
    'fourteencircle' => '9325',
    'fourteenparen' => '9345',
    'fourteenperiod' => '9365',
    'fourthai' => '3668',
    'fourthtonechinese' => '715',
    'fparen' => '9377',
    'frac12' => '189',
    'frac13' => '8531',
    'frac14' => '188',
    'frac15' => '8533',
    'frac16' => '8537',
    'frac18' => '8539',
    'frac23' => '8532',
    'frac25' => '8534',
    'frac34' => '190',
    'frac35' => '8535',
    'frac38' => '8540',
    'frac45' => '8536',
    'frac56' => '8538',
    'frac58' => '8541',
    'frac78' => '8542',
    'fraction' => '8260',
    'franc' => '8355',
    'frasl' => '8260',
    'frown' => '8994',
    'frownface' => '61516',
    'g' => '103',
    'gE' => '8807',
    'gabengali' => '2455',
    'gacute' => '501',
    'gadeva' => '2327',
    'gafarabic' => '1711',
    'gaffinalarabic' => '64403',
    'gafinitialarabic' => '64404',
    'gafmedialarabic' => '64405',
    'gagujarati' => '2711',
    'gagurmukhi' => '2583',
    'gahiragana' => '12364',
    'gakatakana' => '12460',
    'gamma' => '947',
    'gammad' => '989',
    'gammalatinsmall' => '611',
    'gammasuperior' => '736',
    'gangiacoptic' => '1003',
    'gbopomofo' => '12557',
    'gbreve' => '287',
    'gcaron' => '487',
    'gcedil' => '291',
    'gcedilla' => '291',
    'gcirc' => '285',
    'gcircle' => '9430',
    'gcircumflex' => '285',
    'gcommaaccent' => '291',
    'gcy' => '1075',
    'gdot' => '289',
    'gdotaccent' => '289',
    'ge' => '8805',
    'gecyrillic' => '1075',
    'gehiragana' => '12370',
    'gekatakana' => '12466',
    'gel' => '8923',
    'gemini' => '61536',
    'geometricallyequal' => '8785',
    'gereshaccenthebrew' => '1436',
    'gereshhebrew' => '1523',
    'gereshmuqdamhebrew' => '1437',
    'germandbls' => '223',
    'gershayimaccenthebrew' => '1438',
    'gershayimhebrew' => '1524',
    'ges' => '8805',
    'getamark' => '12307',
    'ggr' => '947',
    'ghabengali' => '2456',
    'ghadarmenian' => '1394',
    'ghadeva' => '2328',
    'ghagujarati' => '2712',
    'ghagurmukhi' => '2584',
    'ghainarabic' => '1594',
    'ghainfinalarabic' => '65230',
    'ghaininitialarabic' => '65231',
    'ghainmedialarabic' => '65232',
    'ghemiddlehookcyrillic' => '1173',
    'ghestrokecyrillic' => '1171',
    'gheupturncyrillic' => '1169',
    'ghhadeva' => '2394',
    'ghhagurmukhi' => '2650',
    'ghook' => '608',
    'ghzsquare' => '13203',
    'gihiragana' => '12366',
    'gikatakana' => '12462',
    'gimarmenian' => '1379',
    'gimel' => '8503',
    'gimeldagesh' => '64306',
    'gimeldageshhebrew' => '64306',
    'gimelhebrew' => '1490',
    'gjcy' => '1107',
    'gjecyrillic' => '1107',
    'gl' => '8823',
    'glottalinvertedstroke' => '446',
    'glottalstop' => '660',
    'glottalstopinverted' => '662',
    'glottalstopmod' => '704',
    'glottalstopreversed' => '661',
    'glottalstopreversedmod' => '705',
    'glottalstopreversedsuperior' => '740',
    'glottalstopstroke' => '673',
    'glottalstopstrokereversed' => '674',
    'gmacron' => '7713',
    'gmonospace' => '65351',
    'gnsim' => '8935',
    'gohiragana' => '12372',
    'gokatakana' => '12468',
    'gparen' => '9378',
    'gpasquare' => '13228',
    'gradient' => '8711',
    'grave' => '96',
    'gravebelowcmb' => '790',
    'gravecmb' => '768',
    'gravecomb' => '768',
    'gravedeva' => '2387',
    'gravelowmod' => '718',
    'gravemonospace' => '65344',
    'gravetonecmb' => '832',
    'greater' => '62',
    'greaterequal' => '8805',
    'greaterequalorless' => '8923',
    'greatermonospace' => '65310',
    'greaterorequivalent' => '8819',
    'greaterorless' => '8823',
    'greateroverequal' => '8807',
    'greatersmall' => '65125',
    'gscript' => '609',
    'gsdot' => '8919',
    'gsim' => '8819',
    'gstroke' => '485',
    'gt' => '62',
    'guhiragana' => '12368',
    'guillemotleft' => '171',
    'guillemotright' => '187',
    'guilsinglleft' => '8249',
    'guilsinglright' => '8250',
    'gukatakana' => '12464',
    'guramusquare' => '13080',
    'gvnE' => '8809',
    'gysquare' => '13257',
    'h' => '104',
    'hArr' => '8660',
    'haabkhasiancyrillic' => '1193',
    'haaltonearabic' => '1729',
    'habengali' => '2489',
    'hadescendercyrillic' => '1203',
    'hadeva' => '2361',
    'hagujarati' => '2745',
    'hagurmukhi' => '2617',
    'haharabic' => '1581',
    'hahfinalarabic' => '65186',
    'hahinitialarabic' => '65187',
    'hahiragana' => '12399',
    'hahmedialarabic' => '65188',
    'hairsp' => '8202',
    'haitusquare' => '13098',
    'hakatakana' => '12495',
    'hakatakanahalfwidth' => '65418',
    'halantgurmukhi' => '2637',
    'hamilt' => '8459',
    'hamzaarabic' => '1569',
    'hamzadammaarabic' => '1569',
    'hamzadammatanarabic' => '1569',
    'hamzafathaarabic' => '1569',
    'hamzafathatanarabic' => '1569',
    'hamzalowarabic' => '1569',
    'hamzalowkasraarabic' => '1569',
    'hamzalowkasratanarabic' => '1569',
    'hamzasukunarabic' => '1569',
    'handhalt' => '61513',
    'handok' => '61506',
    'handptdwn' => '61512',
    'handptleft' => '61509',
    'handptright' => '61510',
    'handptup' => '61511',
    'handv' => '61505',
    'handwrite' => '61503',
    'handwriteleft' => '61504',
    'hangulfiller' => '12644',
    'hardcy' => '1098',
    'harddisk' => '61499',
    'hardsigncyrillic' => '1098',
    'harpoonleftbarbup' => '8636',
    'harpoonrightbarbup' => '8640',
    'harr' => '8596',
    'harrw' => '8621',
    'hasquare' => '13258',
    'hatafpatah' => '1458',
    'hatafpatah16' => '1458',
    'hatafpatah23' => '1458',
    'hatafpatah2f' => '1458',
    'hatafpatahhebrew' => '1458',
    'hatafpatahnarrowhebrew' => '1458',
    'hatafpatahquarterhebrew' => '1458',
    'hatafpatahwidehebrew' => '1458',
    'hatafqamats' => '1459',
    'hatafqamats1b' => '1459',
    'hatafqamats28' => '1459',
    'hatafqamats34' => '1459',
    'hatafqamatshebrew' => '1459',
    'hatafqamatsnarrowhebrew' => '1459',
    'hatafqamatsquarterhebrew' => '1459',
    'hatafqamatswidehebrew' => '1459',
    'hatafsegol' => '1457',
    'hatafsegol17' => '1457',
    'hatafsegol24' => '1457',
    'hatafsegol30' => '1457',
    'hatafsegolhebrew' => '1457',
    'hatafsegolnarrowhebrew' => '1457',
    'hatafsegolquarterhebrew' => '1457',
    'hatafsegolwidehebrew' => '1457',
    'hbar' => '295',
    'hbopomofo' => '12559',
    'hbrevebelow' => '7723',
    'hcedilla' => '7721',
    'hcirc' => '293',
    'hcircle' => '9431',
    'hcircumflex' => '293',
    'hdieresis' => '7719',
    'hdotaccent' => '7715',
    'hdotbelow' => '7717',
    'he' => '1492',
    'head2down' => '61658',
    'head2left' => '61655',
    'head2right' => '61656',
    'head2up' => '61657',
    'heart' => '9829',
    'hearts' => '9829',
    'heartsuitblack' => '9829',
    'heartsuitwhite' => '9825',
    'hedagesh' => '64308',
    'hedageshhebrew' => '64308',
    'hehaltonearabic' => '1729',
    'heharabic' => '1607',
    'hehebrew' => '1492',
    'hehfinalaltonearabic' => '64423',
    'hehfinalalttwoarabic' => '65258',
    'hehfinalarabic' => '65258',
    'hehhamzaabovefinalarabic' => '64421',
    'hehhamzaaboveisolatedarabic' => '64420',
    'hehinitialaltonearabic' => '64424',
    'hehinitialarabic' => '65259',
    'hehiragana' => '12408',
    'hehmedialaltonearabic' => '64425',
    'hehmedialarabic' => '65260',
    'heiseierasquare' => '13179',
    'hekatakana' => '12504',
    'hekatakanahalfwidth' => '65421',
    'hekutaarusquare' => '13110',
    'hellip' => '8230',
    'henghook' => '615',
    'herutusquare' => '13113',
    'het' => '1495',
    'hethebrew' => '1495',
    'hexstar2' => '61612',
    'hhook' => '614',
    'hhooksuperior' => '689',
    'hieuhacirclekorean' => '12923',
    'hieuhaparenkorean' => '12827',
    'hieuhcirclekorean' => '12909',
    'hieuhkorean' => '12622',
    'hieuhparenkorean' => '12813',
    'hihiragana' => '12402',
    'hikatakana' => '12498',
    'hikatakanahalfwidth' => '65419',
    'hiriq' => '1460',
    'hiriq14' => '1460',
    'hiriq21' => '1460',
    'hiriq2d' => '1460',
    'hiriqhebrew' => '1460',
    'hiriqnarrowhebrew' => '1460',
    'hiriqquarterhebrew' => '1460',
    'hiriqwidehebrew' => '1460',
    'hlinebelow' => '7830',
    'hmonospace' => '65352',
    'hoarmenian' => '1392',
    'hohipthai' => '3627',
    'hohiragana' => '12411',
    'hokatakana' => '12507',
    'hokatakanahalfwidth' => '65422',
    'holam' => '1465',
    'holam19' => '1465',
    'holam26' => '1465',
    'holam32' => '1465',
    'holamhebrew' => '1465',
    'holamnarrowhebrew' => '1465',
    'holamquarterhebrew' => '1465',
    'holamwidehebrew' => '1465',
    'honokhukthai' => '3630',
    'hookabovecomb' => '777',
    'hookcmb' => '777',
    'hookpalatalizedbelowcmb' => '801',
    'hookretroflexbelowcmb' => '802',
    'hoonsquare' => '13122',
    'horbar' => '8213',
    'horicoptic' => '1001',
    'horizontalbar' => '8213',
    'horncmb' => '795',
    'hotsprings' => '9832',
    'hourglass' => '61494',
    'house' => '8962',
    'hparen' => '9379',
    'hstrok' => '295',
    'hsuperior' => '688',
    'hturned' => '613',
    'huhiragana' => '12405',
    'huiitosquare' => '13107',
    'hukatakana' => '12501',
    'hukatakanahalfwidth' => '65420',
    'hungarumlaut' => '733',
    'hungarumlautcmb' => '779',
    'hv' => '405',
    'hybull' => '8259',
    'hyphen' => '45',
    'hypheninferior' => '63205',
    'hyphenmonospace' => '65293',
    'hyphensmall' => '65123',
    'hyphensuperior' => '63206',
    'hyphentwo' => '8208',
    'i' => '105',
    'iacgr' => '943',
    'iacute' => '237',
    'iacyrillic' => '1103',
    'ibengali' => '2439',
    'ibopomofo' => '12583',
    'ibreve' => '301',
    'icaron' => '464',
    'icirc' => '238',
    'icircle' => '9432',
    'icircumflex' => '238',
    'icy' => '1080',
    'icyrillic' => '1110',
    'idblgrave' => '521',
    'ideographearthcircle' => '12943',
    'ideographfirecircle' => '12939',
    'ideographicallianceparen' => '12863',
    'ideographiccallparen' => '12858',
    'ideographiccentrecircle' => '12965',
    'ideographicclose' => '12294',
    'ideographiccomma' => '12289',
    'ideographiccommaleft' => '65380',
    'ideographiccongratulationparen' => '12855',
    'ideographiccorrectcircle' => '12963',
    'ideographicearthparen' => '12847',
    'ideographicenterpriseparen' => '12861',
    'ideographicexcellentcircle' => '12957',
    'ideographicfestivalparen' => '12864',
    'ideographicfinancialcircle' => '12950',
    'ideographicfinancialparen' => '12854',
    'ideographicfireparen' => '12843',
    'ideographichaveparen' => '12850',
    'ideographichighcircle' => '12964',
    'ideographiciterationmark' => '12293',
    'ideographiclaborcircle' => '12952',
    'ideographiclaborparen' => '12856',
    'ideographicleftcircle' => '12967',
    'ideographiclowcircle' => '12966',
    'ideographicmedicinecircle' => '12969',
    'ideographicmetalparen' => '12846',
    'ideographicmoonparen' => '12842',
    'ideographicnameparen' => '12852',
    'ideographicperiod' => '12290',
    'ideographicprintcircle' => '12958',
    'ideographicreachparen' => '12867',
    'ideographicrepresentparen' => '12857',
    'ideographicresourceparen' => '12862',
    'ideographicrightcircle' => '12968',
    'ideographicsecretcircle' => '12953',
    'ideographicselfparen' => '12866',
    'ideographicsocietyparen' => '12851',
    'ideographicspace' => '12288',
    'ideographicspecialparen' => '12853',
    'ideographicstockparen' => '12849',
    'ideographicstudyparen' => '12859',
    'ideographicsunparen' => '12848',
    'ideographicsuperviseparen' => '12860',
    'ideographicwaterparen' => '12844',
    'ideographicwoodparen' => '12845',
    'ideographiczero' => '12295',
    'ideographmetalcircle' => '12942',
    'ideographmooncircle' => '12938',
    'ideographnamecircle' => '12948',
    'ideographsuncircle' => '12944',
    'ideographwatercircle' => '12940',
    'ideographwoodcircle' => '12941',
    'ideva' => '2311',
    'idiagr' => '912',
    'idieresis' => '239',
    'idieresisacute' => '7727',
    'idieresiscyrillic' => '1253',
    'idigr' => '970',
    'idotbelow' => '7883',
    'iebrevecyrillic' => '1239',
    'iecy' => '1077',
    'iecyrillic' => '1077',
    'ieungacirclekorean' => '12917',
    'ieungaparenkorean' => '12821',
    'ieungcirclekorean' => '12903',
    'ieungkorean' => '12615',
    'ieungparenkorean' => '12807',
    'iexcl' => '161',
    'iff' => '8660',
    'igr' => '953',
    'igrave' => '236',
    'igujarati' => '2695',
    'igurmukhi' => '2567',
    'ihiragana' => '12356',
    'ihookabove' => '7881',
    'iibengali' => '2440',
    'iicyrillic' => '1080',
    'iideva' => '2312',
    'iigujarati' => '2696',
    'iigurmukhi' => '2568',
    'iimatragurmukhi' => '2624',
    'iinvertedbreve' => '523',
    'iishortcyrillic' => '1081',
    'iivowelsignbengali' => '2496',
    'iivowelsigndeva' => '2368',
    'iivowelsigngujarati' => '2752',
    'ij' => '307',
    'ijlig' => '307',
    'ikatakana' => '12452',
    'ikatakanahalfwidth' => '65394',
    'ikorean' => '12643',
    'ilde' => '732',
    'iluyhebrew' => '1452',
    'imacr' => '299',
    'imacron' => '299',
    'imacroncyrillic' => '1251',
    'image' => '8465',
    'imageorapproximatelyequal' => '8787',
    'imatragurmukhi' => '2623',
    'imonospace' => '65353',
    'incare' => '8453',
    'increment' => '8710',
    'infin' => '8734',
    'infinity' => '8734',
    'iniarmenian' => '1387',
    'inodot' => '305',
    'int' => '8747',
    'intcal' => '8890',
    'integral' => '8747',
    'integralbottom' => '8993',
    'integralbt' => '8993',
    'integralex' => '63733',
    'integraltop' => '8992',
    'integraltp' => '8992',
    'intersection' => '8745',
    'intisquare' => '13061',
    'invbullet' => '9688',
    'invcircle' => '9689',
    'invsmileface' => '9787',
    'iocy' => '1105',
    'iocyrillic' => '1105',
    'iogon' => '303',
    'iogonek' => '303',
    'iota' => '953',
    'iotadieresis' => '970',
    'iotadieresistonos' => '912',
    'iotalatin' => '617',
    'iotatonos' => '943',
    'iparen' => '9380',
    'iquest' => '191',
    'irigurmukhi' => '2674',
    'isin' => '8712',
    'ismallhiragana' => '12355',
    'ismallkatakana' => '12451',
    'ismallkatakanahalfwidth' => '65384',
    'issharbengali' => '2554',
    'istroke' => '616',
    'isuperior' => '63213',
    'iterationhiragana' => '12445',
    'iterationkatakana' => '12541',
    'itilde' => '297',
    'itildebelow' => '7725',
    'iubopomofo' => '12585',
    'iucyrillic' => '1102',
    'iukcy' => '1110',
    'iuml' => '239',
    'ivowelsignbengali' => '2495',
    'ivowelsigndeva' => '2367',
    'ivowelsigngujarati' => '2751',
    'izhitsacyrillic' => '1141',
    'izhitsadblgravecyrillic' => '1143',
    'j' => '106',
    'jaarmenian' => '1393',
    'jabengali' => '2460',
    'jadeva' => '2332',
    'jagujarati' => '2716',
    'jagurmukhi' => '2588',
    'jbopomofo' => '12560',
    'jcaron' => '496',
    'jcirc' => '309',
    'jcircle' => '9433',
    'jcircumflex' => '309',
    'jcrossedtail' => '669',
    'jcy' => '1081',
    'jdotlessstroke' => '607',
    'jecyrillic' => '1112',
    'jeemarabic' => '1580',
    'jeemfinalarabic' => '65182',
    'jeeminitialarabic' => '65183',
    'jeemmedialarabic' => '65184',
    'jeharabic' => '1688',
    'jehfinalarabic' => '64395',
    'jhabengali' => '2461',
    'jhadeva' => '2333',
    'jhagujarati' => '2717',
    'jhagurmukhi' => '2589',
    'jheharmenian' => '1403',
    'jis' => '12292',
    'jmonospace' => '65354',
    'jparen' => '9381',
    'jsercy' => '1112',
    'jsuperior' => '690',
    'jukcy' => '1108',
    'k' => '107',
    'kabashkircyrillic' => '1185',
    'kabengali' => '2453',
    'kacute' => '7729',
    'kacyrillic' => '1082',
    'kadescendercyrillic' => '1179',
    'kadeva' => '2325',
    'kaf' => '1499',
    'kafarabic' => '1603',
    'kafdagesh' => '64315',
    'kafdageshhebrew' => '64315',
    'kaffinalarabic' => '65242',
    'kafhebrew' => '1499',
    'kafinitialarabic' => '65243',
    'kafmedialarabic' => '65244',
    'kafrafehebrew' => '64333',
    'kagujarati' => '2709',
    'kagurmukhi' => '2581',
    'kahiragana' => '12363',
    'kahookcyrillic' => '1220',
    'kakatakana' => '12459',
    'kakatakanahalfwidth' => '65398',
    'kappa' => '954',
    'kappasymbolgreek' => '1008',
    'kappav' => '1008',
    'kapyeounmieumkorean' => '12657',
    'kapyeounphieuphkorean' => '12676',
    'kapyeounpieupkorean' => '12664',
    'kapyeounssangpieupkorean' => '12665',
    'karoriisquare' => '13069',
    'kashidaautoarabic' => '1600',
    'kashidaautonosidebearingarabic' => '1600',
    'kasmallkatakana' => '12533',
    'kasquare' => '13188',
    'kasraarabic' => '1616',
    'kasratanarabic' => '1613',
    'kastrokecyrillic' => '1183',
    'katahiraprolongmarkhalfwidth' => '65392',
    'kaverticalstrokecyrillic' => '1181',
    'kbopomofo' => '12558',
    'kcalsquare' => '13193',
    'kcaron' => '489',
    'kcedil' => '311',
    'kcedilla' => '311',
    'kcircle' => '9434',
    'kcommaaccent' => '311',
    'kcy' => '1082',
    'kdotbelow' => '7731',
    'keharmenian' => '1412',
    'kehiragana' => '12369',
    'kekatakana' => '12465',
    'kekatakanahalfwidth' => '65401',
    'kenarmenian' => '1391',
    'kesmallkatakana' => '12534',
    'keyboard' => '61495',
    'kgr' => '954',
    'kgreen' => '312',
    'kgreenlandic' => '312',
    'khabengali' => '2454',
    'khacyrillic' => '1093',
    'khadeva' => '2326',
    'khagujarati' => '2710',
    'khagurmukhi' => '2582',
    'khaharabic' => '1582',
    'khahfinalarabic' => '65190',
    'khahinitialarabic' => '65191',
    'khahmedialarabic' => '65192',
    'khcy' => '1093',
    'kheicoptic' => '999',
    'khgr' => '967',
    'khhadeva' => '2393',
    'khhagurmukhi' => '2649',
    'khieukhacirclekorean' => '12920',
    'khieukhaparenkorean' => '12824',
    'khieukhcirclekorean' => '12906',
    'khieukhkorean' => '12619',
    'khieukhparenkorean' => '12810',
    'khokhaithai' => '3586',
    'khokhonthai' => '3589',
    'khokhuatthai' => '3587',
    'khokhwaithai' => '3588',
    'khomutthai' => '3675',
    'khook' => '409',
    'khorakhangthai' => '3590',
    'khzsquare' => '13201',
    'kihiragana' => '12365',
    'kikatakana' => '12461',
    'kikatakanahalfwidth' => '65399',
    'kiroguramusquare' => '13077',
    'kiromeetorusquare' => '13078',
    'kirosquare' => '13076',
    'kiyeokacirclekorean' => '12910',
    'kiyeokaparenkorean' => '12814',
    'kiyeokcirclekorean' => '12896',
    'kiyeokkorean' => '12593',
    'kiyeokparenkorean' => '12800',
    'kiyeoksioskorean' => '12595',
    'kjcy' => '1116',
    'kjecyrillic' => '1116',
    'klinebelow' => '7733',
    'klsquare' => '13208',
    'kmcubedsquare' => '13222',
    'kmonospace' => '65355',
    'kmsquaredsquare' => '13218',
    'kohiragana' => '12371',
    'kohmsquare' => '13248',
    'kokaithai' => '3585',
    'kokatakana' => '12467',
    'kokatakanahalfwidth' => '65402',
    'kooposquare' => '13086',
    'koppacyrillic' => '1153',
    'koreanstandardsymbol' => '12927',
    'koroniscmb' => '835',
    'kparen' => '9382',
    'kpasquare' => '13226',
    'ksicyrillic' => '1135',
    'ktsquare' => '13263',
    'kturned' => '670',
    'kuhiragana' => '12367',
    'kukatakana' => '12463',
    'kukatakanahalfwidth' => '65400',
    'kvsquare' => '13240',
    'kwsquare' => '13246',
    'l' => '108',
    'lAarr' => '8666',
    'lArr' => '8656',
    'lE' => '8806',
    'labengali' => '2482',
    'lacute' => '314',
    'ladeva' => '2354',
    'lagran' => '8466',
    'lagujarati' => '2738',
    'lagurmukhi' => '2610',
    'lakkhangyaothai' => '3653',
    'lamaleffinalarabic' => '65276',
    'lamalefhamzaabovefinalarabic' => '65272',
    'lamalefhamzaaboveisolatedarabic' => '65271',
    'lamalefhamzabelowfinalarabic' => '65274',
    'lamalefhamzabelowisolatedarabic' => '65273',
    'lamalefisolatedarabic' => '65275',
    'lamalefmaddaabovefinalarabic' => '65270',
    'lamalefmaddaaboveisolatedarabic' => '65269',
    'lamarabic' => '1604',
    'lambda' => '955',
    'lambdastroke' => '411',
    'lamed' => '1500',
    'lameddagesh' => '64316',
    'lameddageshhebrew' => '64316',
    'lamedhebrew' => '1500',
    'lamedholam' => '1500',
    'lamedholamdagesh' => '1500',
    'lamedholamdageshhebrew' => '1500',
    'lamedholamhebrew' => '1500',
    'lamfinalarabic' => '65246',
    'lamhahinitialarabic' => '64714',
    'laminitialarabic' => '65247',
    'lamjeeminitialarabic' => '64713',
    'lamkhahinitialarabic' => '64715',
    'lamlamhehisolatedarabic' => '65010',
    'lammedialarabic' => '65248',
    'lammeemhahinitialarabic' => '64904',
    'lammeeminitialarabic' => '64716',
    'lammeemjeeminitialarabic' => '65247',
    'lammeemkhahinitialarabic' => '65247',
    'lang' => '9001',
    'laquo' => '171',
    'largecircle' => '9711',
    'larr' => '8592',
    'larr2' => '8647',
    'larrhk' => '8617',
    'larrlp' => '8619',
    'larrtl' => '8610',
    'lbar' => '410',
    'lbelt' => '620',
    'lbopomofo' => '12556',
    'lcaron' => '318',
    'lcedil' => '316',
    'lcedilla' => '316',
    'lceil' => '8968',
    'lcircle' => '9435',
    'lcircumflexbelow' => '7741',
    'lcommaaccent' => '316',
    'lcub' => '123',
    'lcy' => '1083',
    'ldot' => '320',
    'ldotaccent' => '320',
    'ldotbelow' => '7735',
    'ldotbelowmacron' => '7737',
    'ldquo' => '8220',
    'ldquor' => '8222',
    'le' => '8804',
    'leafccwne' => '61648',
    'leafccwnw' => '61646',
    'leafccwse' => '61647',
    'leafccwsw' => '61645',
    'leafne' => '61651',
    'leafnw' => '61649',
    'leafse' => '61652',
    'leafsw' => '61650',
    'leftangleabovecmb' => '794',
    'lefttackbelowcmb' => '792',
    'leg' => '8922',
    'leo' => '61538',
    'les' => '8804',
    'less' => '60',
    'lessequal' => '8804',
    'lessequalorgreater' => '8922',
    'lessmonospace' => '65308',
    'lessorequivalent' => '8818',
    'lessorgreater' => '8822',
    'lessoverequal' => '8806',
    'lesssmall' => '65124',
    'lezh' => '622',
    'lfblock' => '9612',
    'lfloor' => '8970',
    'lg' => '8822',
    'lgr' => '955',
    'lhard' => '8637',
    'lharu' => '8636',
    'lhblk' => '9604',
    'lhookretroflex' => '621',
    'libra' => '61540',
    'lira' => '8356',
    'liwnarmenian' => '1388',
    'lj' => '457',
    'ljcy' => '1113',
    'ljecyrillic' => '1113',
    'll' => '63168',
    'lladeva' => '2355',
    'llagujarati' => '2739',
    'llinebelow' => '7739',
    'llladeva' => '2356',
    'llvocalicbengali' => '2529',
    'llvocalicdeva' => '2401',
    'llvocalicvowelsignbengali' => '2531',
    'llvocalicvowelsigndeva' => '2403',
    'lmiddletilde' => '619',
    'lmidot' => '320',
    'lmonospace' => '65356',
    'lmsquare' => '13264',
    'lnsim' => '8934',
    'lochulathai' => '3628',
    'logicaland' => '8743',
    'logicalnot' => '172',
    'logicalnotreversed' => '8976',
    'logicalor' => '8744',
    'lolingthai' => '3621',
    'longs' => '383',
    'lowast' => '8727',
    'lowbar' => '95',
    'lowlinecenterline' => '65102',
    'lowlinecmb' => '818',
    'lowlinedashed' => '65101',
    'loz' => '9674',
    'lozenge' => '9674',
    'lozenge4' => '61555',
    'lozenge6' => '61556',
    'lozf' => '10022',
    'lpar' => '40',
    'lparen' => '9383',
    'lrarr2' => '8646',
    'lrhar2' => '8651',
    'lsaquo' => '8249',
    'lsh' => '8624',
    'lsim' => '8818',
    'lslash' => '322',
    'lsqb' => '91',
    'lsquare' => '8467',
    'lsquo' => '8216',
    'lsquor' => '8218',
    'lstrok' => '322',
    'lsuperior' => '63214',
    'lt' => '60',
    'lthree' => '8907',
    'ltimes' => '8905',
    'ltri' => '9667',
    'ltrie' => '8884',
    'ltrif' => '9666',
    'ltshade' => '9617',
    'luthai' => '3622',
    'lvnE' => '8808',
    'lvocalicbengali' => '2444',
    'lvocalicdeva' => '2316',
    'lvocalicvowelsignbengali' => '2530',
    'lvocalicvowelsigndeva' => '2402',
    'lxsquare' => '13267',
    'm' => '109',
    'mabengali' => '2478',
    'macr' => '175',
    'macron' => '175',
    'macronbelowcmb' => '817',
    'macroncmb' => '772',
    'macronlowmod' => '717',
    'macronmonospace' => '65507',
    'macute' => '7743',
    'madeva' => '2350',
    'magujarati' => '2734',
    'magurmukhi' => '2606',
    'mahapakhhebrew' => '1444',
    'mahapakhlefthebrew' => '1444',
    'mahiragana' => '12414',
    'maichattawalowleftthai' => '63637',
    'maichattawalowrightthai' => '63636',
    'maichattawathai' => '3659',
    'maichattawaupperleftthai' => '63635',
    'maieklowleftthai' => '63628',
    'maieklowrightthai' => '63627',
    'maiekthai' => '3656',
    'maiekupperleftthai' => '63626',
    'maihanakatleftthai' => '63620',
    'maihanakatthai' => '3633',
    'mailboxflagdwn' => '61484',
    'mailboxflagup' => '61485',
    'mailbxopnflgdwn' => '61487',
    'mailbxopnflgup' => '61486',
    'maitaikhuleftthai' => '63625',
    'maitaikhuthai' => '3655',
    'maitholowleftthai' => '63631',
    'maitholowrightthai' => '63630',
    'maithothai' => '3657',
    'maithoupperleftthai' => '63629',
    'maitrilowleftthai' => '63634',
    'maitrilowrightthai' => '63633',
    'maitrithai' => '3658',
    'maitriupperleftthai' => '63632',
    'maiyamokthai' => '3654',
    'makatakana' => '12510',
    'makatakanahalfwidth' => '65423',
    'male' => '9794',
    'malt' => '10016',
    'mansyonsquare' => '13127',
    'map' => '8614',
    'maqafhebrew' => '1470',
    'marker' => '9646',
    'mars' => '9794',
    'masoracirclehebrew' => '1455',
    'masquare' => '13187',
    'mbopomofo' => '12551',
    'mbsquare' => '13268',
    'mcircle' => '9436',
    'mcubedsquare' => '13221',
    'mcy' => '1084',
    'mdash' => '8212',
    'mdotaccent' => '7745',
    'mdotbelow' => '7747',
    'meemarabic' => '1605',
    'meemfinalarabic' => '65250',
    'meeminitialarabic' => '65251',
    'meemmedialarabic' => '65252',
    'meemmeeminitialarabic' => '64721',
    'meemmeemisolatedarabic' => '64584',
    'meetorusquare' => '13133',
    'mehiragana' => '12417',
    'meizierasquare' => '13182',
    'mekatakana' => '12513',
    'mekatakanahalfwidth' => '65426',
    'mem' => '1502',
    'memdagesh' => '64318',
    'memdageshhebrew' => '64318',
    'memhebrew' => '1502',
    'menarmenian' => '1396',
    'merkhahebrew' => '1445',
    'merkhakefulahebrew' => '1446',
    'merkhakefulalefthebrew' => '1446',
    'merkhalefthebrew' => '1445',
    'mgr' => '956',
    'mhook' => '625',
    'mhzsquare' => '13202',
    'micro' => '181',
    'mid' => '8739',
    'middledotkatakanahalfwidth' => '65381',
    'middot' => '183',
    'mieumacirclekorean' => '12914',
    'mieumaparenkorean' => '12818',
    'mieumcirclekorean' => '12900',
    'mieumkorean' => '12609',
    'mieumpansioskorean' => '12656',
    'mieumparenkorean' => '12804',
    'mieumpieupkorean' => '12654',
    'mieumsioskorean' => '12655',
    'mihiragana' => '12415',
    'mikatakana' => '12511',
    'mikatakanahalfwidth' => '65424',
    'minus' => '8722',
    'minusb' => '8863',
    'minusbelowcmb' => '800',
    'minuscircle' => '8854',
    'minusmod' => '727',
    'minusplus' => '8723',
    'minute' => '8242',
    'miribaarusquare' => '13130',
    'mirisquare' => '13129',
    'mldr' => '8230',
    'mlonglegturned' => '624',
    'mlsquare' => '13206',
    'mmcubedsquare' => '13219',
    'mmonospace' => '65357',
    'mmsquaredsquare' => '13215',
    'mnplus' => '8723',
    'models' => '8871',
    'mohiragana' => '12418',
    'mohmsquare' => '13249',
    'mokatakana' => '12514',
    'mokatakanahalfwidth' => '65427',
    'molsquare' => '13270',
    'momathai' => '3617',
    'mouse2button' => '61496',
    'moverssquare' => '13223',
    'moverssquaredsquare' => '13224',
    'mparen' => '9384',
    'mpasquare' => '13227',
    'mssquare' => '13235',
    'msuperior' => '63215',
    'mturned' => '623',
    'mu' => '181',
    'mu1' => '181',
    'muasquare' => '13186',
    'muchgreater' => '8811',
    'muchless' => '8810',
    'mufsquare' => '13196',
    'mugreek' => '956',
    'mugsquare' => '13197',
    'muhiragana' => '12416',
    'mukatakana' => '12512',
    'mukatakanahalfwidth' => '65425',
    'mulsquare' => '13205',
    'multiply' => '215',
    'mumap' => '8888',
    'mumsquare' => '13211',
    'munahhebrew' => '1443',
    'munahlefthebrew' => '1443',
    'musicalnote' => '9834',
    'musicalnotedbl' => '9835',
    'musicflatsign' => '9837',
    'musicsharpsign' => '9839',
    'mussquare' => '13234',
    'muvsquare' => '13238',
    'muwsquare' => '13244',
    'mvmegasquare' => '13241',
    'mvsquare' => '13239',
    'mwmegasquare' => '13247',
    'mwsquare' => '13245',
    'n' => '110',
    'nVDash' => '8879',
    'nVdash' => '8878',
    'nabengali' => '2472',
    'nabla' => '8711',
    'nacute' => '324',
    'nadeva' => '2344',
    'nagujarati' => '2728',
    'nagurmukhi' => '2600',
    'nahiragana' => '12394',
    'nakatakana' => '12490',
    'nakatakanahalfwidth' => '65413',
    'nap' => '8777',
    'napos' => '329',
    'napostrophe' => '329',
    'nasquare' => '13185',
    'natur' => '9838',
    'nbopomofo' => '12555',
    'nbsp' => '160',
    'nbspace' => '160',
    'ncaron' => '328',
    'ncedil' => '326',
    'ncedilla' => '326',
    'ncircle' => '9437',
    'ncircumflexbelow' => '7755',
    'ncommaaccent' => '326',
    'ncong' => '8775',
    'ncy' => '1085',
    'ndash' => '8211',
    'ndotaccent' => '7749',
    'ndotbelow' => '7751',
    'ne' => '8800',
    'nearr' => '8599',
    'nehiragana' => '12397',
    'nekatakana' => '12493',
    'nekatakanahalfwidth' => '65416',
    'nequiv' => '8802',
    'neutralface' => '61515',
    'newsheqelsign' => '8362',
    'nexist' => '8708',
    'nfsquare' => '13195',
    'ngabengali' => '2457',
    'ngadeva' => '2329',
    'ngagujarati' => '2713',
    'ngagurmukhi' => '2585',
    'nges' => '8817',
    'ngonguthai' => '3591',
    'ngr' => '957',
    'ngt' => '8815',
    'nhArr' => '8654',
    'nharr' => '8622',
    'nhiragana' => '12435',
    'nhookleft' => '626',
    'nhookretroflex' => '627',
    'ni' => '8715',
    'nieunacirclekorean' => '12911',
    'nieunaparenkorean' => '12815',
    'nieuncieuckorean' => '12597',
    'nieuncirclekorean' => '12897',
    'nieunhieuhkorean' => '12598',
    'nieunkorean' => '12596',
    'nieunpansioskorean' => '12648',
    'nieunparenkorean' => '12801',
    'nieunsioskorean' => '12647',
    'nieuntikeutkorean' => '12646',
    'nihiragana' => '12395',
    'nikatakana' => '12491',
    'nikatakanahalfwidth' => '65414',
    'nikhahitleftthai' => '63641',
    'nikhahitthai' => '3661',
    'nine' => '57',
    'ninearabic' => '1641',
    'ninebengali' => '2543',
    'ninecircle' => '9320',
    'ninecircleinversesansserif' => '10130',
    'ninedeva' => '2415',
    'ninegujarati' => '2799',
    'ninegurmukhi' => '2671',
    'ninehackarabic' => '1641',
    'ninehangzhou' => '12329',
    'nineideographicparen' => '12840',
    'nineinferior' => '8329',
    'ninemonospace' => '65305',
    'nineoclock' => '61631',
    'nineoldstyle' => '63289',
    'nineparen' => '9340',
    'nineperiod' => '9360',
    'ninepersian' => '1785',
    'nineroman' => '8568',
    'ninesans' => '61577',
    'ninesansinv' => '61588',
    'ninesuperior' => '8313',
    'nineteencircle' => '9330',
    'nineteenparen' => '9350',
    'nineteenperiod' => '9370',
    'ninethai' => '3673',
    'nj' => '460',
    'njcy' => '1114',
    'njecyrillic' => '1114',
    'nkatakana' => '12531',
    'nkatakanahalfwidth' => '65437',
    'nlArr' => '8653',
    'nlarr' => '8602',
    'nldr' => '8229',
    'nlegrightlong' => '414',
    'nles' => '8816',
    'nlinebelow' => '7753',
    'nlt' => '8814',
    'nltri' => '8938',
    'nltrie' => '8940',
    'nmid' => '8740',
    'nmonospace' => '65358',
    'nmsquare' => '13210',
    'nnabengali' => '2467',
    'nnadeva' => '2339',
    'nnagujarati' => '2723',
    'nnagurmukhi' => '2595',
    'nnnadeva' => '2345',
    'nohiragana' => '12398',
    'nokatakana' => '12494',
    'nokatakanahalfwidth' => '65417',
    'nonbreakingspace' => '160',
    'nonenthai' => '3603',
    'nonuthai' => '3609',
    'noonarabic' => '1606',
    'noonfinalarabic' => '65254',
    'noonghunnaarabic' => '1722',
    'noonghunnafinalarabic' => '64415',
    'noonhehinitialarabic' => '65255',
    'nooninitialarabic' => '65255',
    'noonjeeminitialarabic' => '64722',
    'noonjeemisolatedarabic' => '64587',
    'noonmedialarabic' => '65256',
    'noonmeeminitialarabic' => '64725',
    'noonmeemisolatedarabic' => '64590',
    'noonnoonfinalarabic' => '64653',
    'not' => '172',
    'notcontains' => '8716',
    'notelement' => '8713',
    'notelementof' => '8713',
    'notequal' => '8800',
    'notgreater' => '8815',
    'notgreaternorequal' => '8817',
    'notgreaternorless' => '8825',
    'notidentical' => '8802',
    'notin' => '8713',
    'notless' => '8814',
    'notlessnorequal' => '8816',
    'notparallel' => '8742',
    'notprecedes' => '8832',
    'notsubset' => '8836',
    'notsucceeds' => '8833',
    'notsuperset' => '8837',
    'nowarmenian' => '1398',
    'npar' => '8742',
    'nparen' => '9385',
    'npr' => '8832',
    'npre' => '8928',
    'nrArr' => '8655',
    'nrarr' => '8603',
    'nrtri' => '8939',
    'nrtrie' => '8941',
    'nsc' => '8833',
    'nsce' => '8929',
    'nsim' => '8769',
    'nsime' => '8772',
    'nssquare' => '13233',
    'nsub' => '8836',
    'nsube' => '8840',
    'nsup' => '8837',
    'nsupe' => '8841',
    'nsuperior' => '8319',
    'ntilde' => '241',
    'nu' => '957',
    'nuhiragana' => '12396',
    'nukatakana' => '12492',
    'nukatakanahalfwidth' => '65415',
    'nuktabengali' => '2492',
    'nuktadeva' => '2364',
    'nuktagujarati' => '2748',
    'nuktagurmukhi' => '2620',
    'num' => '35',
    'numbersign' => '35',
    'numbersignmonospace' => '65283',
    'numbersignsmall' => '65119',
    'numeralsigngreek' => '884',
    'numeralsignlowergreek' => '885',
    'numero' => '8470',
    'numsp' => '8199',
    'nun' => '1504',
    'nundagesh' => '64320',
    'nundageshhebrew' => '64320',
    'nunhebrew' => '1504',
    'nvDash' => '8877',
    'nvdash' => '8876',
    'nvsquare' => '13237',
    'nwarr' => '8598',
    'nwsquare' => '13243',
    'nyabengali' => '2462',
    'nyadeva' => '2334',
    'nyagujarati' => '2718',
    'nyagurmukhi' => '2590',
    'o' => '111',
    'oS' => '9416',
    'oacgr' => '972',
    'oacute' => '243',
    'oangthai' => '3629',
    'oast' => '8859',
    'obarred' => '629',
    'obarredcyrillic' => '1257',
    'obarreddieresiscyrillic' => '1259',
    'obengali' => '2451',
    'obopomofo' => '12571',
    'obreve' => '335',
    'ocandradeva' => '2321',
    'ocandragujarati' => '2705',
    'ocandravowelsigndeva' => '2377',
    'ocandravowelsigngujarati' => '2761',
    'ocaron' => '466',
    'ocir' => '8858',
    'ocirc' => '244',
    'ocircle' => '9438',
    'ocircumflex' => '244',
    'ocircumflexacute' => '7889',
    'ocircumflexdotbelow' => '7897',
    'ocircumflexgrave' => '7891',
    'ocircumflexhookabove' => '7893',
    'ocircumflextilde' => '7895',
    'octastar2' => '61613',
    'octastar4' => '61615',
    'ocy' => '1086',
    'ocyrillic' => '1086',
    'odash' => '8861',
    'odblac' => '337',
    'odblacute' => '337',
    'odblgrave' => '525',
    'odeva' => '2323',
    'odieresis' => '246',
    'odieresiscyrillic' => '1255',
    'odot' => '8857',
    'odotbelow' => '7885',
    'oe' => '339',
    'oekorean' => '12634',
    'oelig' => '339',
    'ogon' => '731',
    'ogonek' => '731',
    'ogonekcmb' => '808',
    'ogr' => '959',
    'ograve' => '242',
    'ogujarati' => '2707',
    'ohacgr' => '974',
    'oharmenian' => '1413',
    'ohgr' => '969',
    'ohiragana' => '12362',
    'ohm' => '8486',
    'ohookabove' => '7887',
    'ohorn' => '417',
    'ohornacute' => '7899',
    'ohorndotbelow' => '7907',
    'ohorngrave' => '7901',
    'ohornhookabove' => '7903',
    'ohorntilde' => '7905',
    'ohungarumlaut' => '337',
    'oi' => '419',
    'oinvertedbreve' => '527',
    'okatakana' => '12458',
    'okatakanahalfwidth' => '65397',
    'okorean' => '12631',
    'olarr' => '8634',
    'olehebrew' => '1451',
    'oline' => '8254',
    'om' => '61532',
    'omacr' => '333',
    'omacron' => '333',
    'omacronacute' => '7763',
    'omacrongrave' => '7761',
    'omdeva' => '2384',
    'omega' => '969',
    'omega1' => '982',
    'omegacyrillic' => '1121',
    'omegalatinclosed' => '631',
    'omegaroundcyrillic' => '1147',
    'omegatitlocyrillic' => '1149',
    'omegatonos' => '974',
    'omgujarati' => '2768',
    'omicron' => '959',
    'omicrontonos' => '972',
    'ominus' => '8854',
    'omonospace' => '65359',
    'one' => '49',
    'onearabic' => '1633',
    'onebengali' => '2535',
    'onecircle' => '9312',
    'onecircleinversesansserif' => '10122',
    'onedeva' => '2407',
    'onedotenleader' => '8228',
    'oneeighth' => '8539',
    'onefitted' => '63196',
    'onegujarati' => '2791',
    'onegurmukhi' => '2663',
    'onehackarabic' => '1633',
    'onehalf' => '189',
    'onehangzhou' => '12321',
    'oneideographicparen' => '12832',
    'oneinferior' => '8321',
    'onemonospace' => '65297',
    'onenumeratorbengali' => '2548',
    'oneoclock' => '61623',
    'oneoldstyle' => '63281',
    'oneparen' => '9332',
    'oneperiod' => '9352',
    'onepersian' => '1777',
    'onequarter' => '188',
    'oneroman' => '8560',
    'onesans' => '61569',
    'onesansinv' => '61580',
    'onesuperior' => '185',
    'onethai' => '3665',
    'onethird' => '8531',
    'oogonek' => '491',
    'oogonekmacron' => '493',
    'oogurmukhi' => '2579',
    'oomatragurmukhi' => '2635',
    'oopen' => '596',
    'oparen' => '9386',
    'openbullet' => '9702',
    'oplus' => '8853',
    'option' => '8997',
    'or' => '8744',
    'orarr' => '8635',
    'order' => '8500',
    'ordf' => '170',
    'ordfeminine' => '170',
    'ordm' => '186',
    'ordmasculine' => '186',
    'orthogonal' => '8735',
    'oshortdeva' => '2322',
    'oshortvowelsigndeva' => '2378',
    'oslash' => '248',
    'oslashacute' => '511',
    'osmallhiragana' => '12361',
    'osmallkatakana' => '12457',
    'osmallkatakanahalfwidth' => '65387',
    'osol' => '8856',
    'ostrokeacute' => '511',
    'osuperior' => '63216',
    'otcyrillic' => '1151',
    'otilde' => '245',
    'otildeacute' => '7757',
    'otildedieresis' => '7759',
    'otimes' => '8855',
    'oubopomofo' => '12577',
    'ouml' => '246',
    'overline' => '8254',
    'overlinecenterline' => '65098',
    'overlinecmb' => '773',
    'overlinedashed' => '65097',
    'overlinedblwavy' => '65100',
    'overlinewavy' => '65099',
    'overscore' => '175',
    'ovowelsignbengali' => '2507',
    'ovowelsigndeva' => '2379',
    'ovowelsigngujarati' => '2763',
    'p' => '112',
    'paampssquare' => '13184',
    'paasentosquare' => '13099',
    'pabengali' => '2474',
    'pacute' => '7765',
    'padeva' => '2346',
    'pagedown' => '8671',
    'pageup' => '8670',
    'pagujarati' => '2730',
    'pagurmukhi' => '2602',
    'pahiragana' => '12401',
    'paiyannoithai' => '3631',
    'pakatakana' => '12497',
    'palatalizationcyrilliccmb' => '1156',
    'palochkacyrillic' => '1216',
    'pansioskorean' => '12671',
    'par' => '8741',
    'para' => '182',
    'paragraph' => '182',
    'parallel' => '8741',
    'parenleft' => '40',
    'parenleftaltonearabic' => '64830',
    'parenleftbt' => '63725',
    'parenleftex' => '63724',
    'parenleftinferior' => '8333',
    'parenleftmonospace' => '65288',
    'parenleftsmall' => '65113',
    'parenleftsuperior' => '8317',
    'parenlefttp' => '63723',
    'parenleftvertical' => '65077',
    'parenright' => '41',
    'parenrightaltonearabic' => '64831',
    'parenrightbt' => '63736',
    'parenrightex' => '63735',
    'parenrightinferior' => '8334',
    'parenrightmonospace' => '65289',
    'parenrightsmall' => '65114',
    'parenrightsuperior' => '8318',
    'parenrighttp' => '63734',
    'parenrightvertical' => '65078',
    'part' => '8706',
    'partialdiff' => '8706',
    'paseqhebrew' => '1472',
    'pashtahebrew' => '1433',
    'pasquare' => '13225',
    'patah' => '1463',
    'patah11' => '1463',
    'patah1d' => '1463',
    'patah2a' => '1463',
    'patahhebrew' => '1463',
    'patahnarrowhebrew' => '1463',
    'patahquarterhebrew' => '1463',
    'patahwidehebrew' => '1463',
    'pazerhebrew' => '1441',
    'pbopomofo' => '12550',
    'pc' => '61498',
    'pcircle' => '9439',
    'pcy' => '1087',
    'pdotaccent' => '7767',
    'pe' => '1508',
    'pecyrillic' => '1087',
    'pedagesh' => '64324',
    'pedageshhebrew' => '64324',
    'peezisquare' => '13115',
    'pefinaldageshhebrew' => '64323',
    'peharabic' => '1662',
    'peharmenian' => '1402',
    'pehebrew' => '1508',
    'pehfinalarabic' => '64343',
    'pehinitialarabic' => '64344',
    'pehiragana' => '12410',
    'pehmedialarabic' => '64345',
    'pekatakana' => '12506',
    'pemiddlehookcyrillic' => '1191',
    'pencil' => '61473',
    'pennant' => '61520',
    'pentastar2' => '61611',
    'perafehebrew' => '64334',
    'percent' => '37',
    'percentarabic' => '1642',
    'percentmonospace' => '65285',
    'percentsmall' => '65130',
    'percnt' => '37',
    'period' => '46',
    'periodarmenian' => '1417',
    'periodcentered' => '183',
    'periodhalfwidth' => '65377',
    'periodinferior' => '63207',
    'periodmonospace' => '65294',
    'periodsmall' => '65106',
    'periodsuperior' => '63208',
    'perispomenigreekcmb' => '834',
    'permil' => '8240',
    'perp' => '8869',
    'perpendicular' => '8869',
    'perthousand' => '8240',
    'peseta' => '8359',
    'pfsquare' => '13194',
    'pgr' => '960',
    'phabengali' => '2475',
    'phadeva' => '2347',
    'phagujarati' => '2731',
    'phagurmukhi' => '2603',
    'phi' => '966',
    'phi1' => '981',
    'phieuphacirclekorean' => '12922',
    'phieuphaparenkorean' => '12826',
    'phieuphcirclekorean' => '12908',
    'phieuphkorean' => '12621',
    'phieuphparenkorean' => '12812',
    'philatin' => '632',
    'phinthuthai' => '3642',
    'phis' => '966',
    'phisymbolgreek' => '981',
    'phiv' => '981',
    'phmmat' => '8499',
    'phone' => '9742',
    'phook' => '421',
    'phophanthai' => '3614',
    'phophungthai' => '3612',
    'phosamphaothai' => '3616',
    'pi' => '960',
    'pieupacirclekorean' => '12915',
    'pieupaparenkorean' => '12819',
    'pieupcieuckorean' => '12662',
    'pieupcirclekorean' => '12901',
    'pieupkiyeokkorean' => '12658',
    'pieupkorean' => '12610',
    'pieupparenkorean' => '12805',
    'pieupsioskiyeokkorean' => '12660',
    'pieupsioskorean' => '12612',
    'pieupsiostikeutkorean' => '12661',
    'pieupthieuthkorean' => '12663',
    'pieuptikeutkorean' => '12659',
    'pihiragana' => '12404',
    'pikatakana' => '12500',
    'pisces' => '61545',
    'pisymbolgreek' => '982',
    'piv' => '982',
    'piwrarmenian' => '1411',
    'planck' => '8463',
    'plus' => '43',
    'plusb' => '8862',
    'plusbelowcmb' => '799',
    'pluscircle' => '8853',
    'plusdo' => '8724',
    'plusminus' => '177',
    'plusmn' => '177',
    'plusmod' => '726',
    'plusmonospace' => '65291',
    'plussmall' => '65122',
    'plussuperior' => '8314',
    'pmonospace' => '65360',
    'pmsquare' => '13272',
    'pohiragana' => '12413',
    'pointingindexdownwhite' => '9759',
    'pointingindexleftwhite' => '9756',
    'pointingindexrightwhite' => '9758',
    'pointingindexupwhite' => '9757',
    'pokatakana' => '12509',
    'poplathai' => '3611',
    'postalmark' => '12306',
    'postalmarkface' => '12320',
    'pound' => '163',
    'pparen' => '9387',
    'pr' => '8826',
    'precedes' => '8826',
    'prescription' => '8478',
    'prime' => '8242',
    'primemod' => '697',
    'primereversed' => '8245',
    'prnsim' => '8936',
    'prod' => '8719',
    'product' => '8719',
    'projective' => '8965',
    'prolongedkana' => '12540',
    'prop' => '8733',
    'propellor' => '8984',
    'propersubset' => '8834',
    'propersuperset' => '8835',
    'proportion' => '8759',
    'proportional' => '8733',
    'prsim' => '8830',
    'psgr' => '968',
    'psi' => '968',
    'psicyrillic' => '1137',
    'psilipneumatacyrilliccmb' => '1158',
    'pssquare' => '13232',
    'puhiragana' => '12407',
    'pukatakana' => '12503',
    'puncsp' => '8200',
    'pvsquare' => '13236',
    'pwsquare' => '13242',
    'q' => '113',
    'qadeva' => '2392',
    'qadmahebrew' => '1448',
    'qafarabic' => '1602',
    'qaffinalarabic' => '65238',
    'qafinitialarabic' => '65239',
    'qafmedialarabic' => '65240',
    'qamats' => '1464',
    'qamats10' => '1464',
    'qamats1a' => '1464',
    'qamats1c' => '1464',
    'qamats27' => '1464',
    'qamats29' => '1464',
    'qamats33' => '1464',
    'qamatsde' => '1464',
    'qamatshebrew' => '1464',
    'qamatsnarrowhebrew' => '1464',
    'qamatsqatanhebrew' => '1464',
    'qamatsqatannarrowhebrew' => '1464',
    'qamatsqatanquarterhebrew' => '1464',
    'qamatsqatanwidehebrew' => '1464',
    'qamatsquarterhebrew' => '1464',
    'qamatswidehebrew' => '1464',
    'qarneyparahebrew' => '1439',
    'qbopomofo' => '12561',
    'qcircle' => '9440',
    'qhook' => '672',
    'qmonospace' => '65361',
    'qof' => '1511',
    'qofdagesh' => '64327',
    'qofdageshhebrew' => '64327',
    'qofhatafpatah' => '1511',
    'qofhatafpatahhebrew' => '1511',
    'qofhatafsegol' => '1511',
    'qofhatafsegolhebrew' => '1511',
    'qofhebrew' => '1511',
    'qofhiriq' => '1511',
    'qofhiriqhebrew' => '1511',
    'qofholam' => '1511',
    'qofholamhebrew' => '1511',
    'qofpatah' => '1511',
    'qofpatahhebrew' => '1511',
    'qofqamats' => '1511',
    'qofqamatshebrew' => '1511',
    'qofqubuts' => '1511',
    'qofqubutshebrew' => '1511',
    'qofsegol' => '1511',
    'qofsegolhebrew' => '1511',
    'qofsheva' => '1511',
    'qofshevahebrew' => '1511',
    'qoftsere' => '1511',
    'qoftserehebrew' => '1511',
    'qparen' => '9388',
    'quarternote' => '9833',
    'qubuts' => '1467',
    'qubuts18' => '1467',
    'qubuts25' => '1467',
    'qubuts31' => '1467',
    'qubutshebrew' => '1467',
    'qubutsnarrowhebrew' => '1467',
    'qubutsquarterhebrew' => '1467',
    'qubutswidehebrew' => '1467',
    'query' => '61620',
    'quest' => '63',
    'question' => '63',
    'questionarabic' => '1567',
    'questionarmenian' => '1374',
    'questiondown' => '191',
    'questiondownsmall' => '63423',
    'questiongreek' => '894',
    'questionmonospace' => '65311',
    'questionsmall' => '63295',
    'quiltsquare2' => '61643',
    'quiltsquare2inv' => '61644',
    'quot' => '34',
    'quotedbl' => '34',
    'quotedblbase' => '8222',
    'quotedblleft' => '8220',
    'quotedbllftbld' => '61565',
    'quotedblmonospace' => '65282',
    'quotedblprime' => '12318',
    'quotedblprimereversed' => '12317',
    'quotedblright' => '8221',
    'quotedblrtbld' => '61566',
    'quoteleft' => '8216',
    'quoteleftreversed' => '8219',
    'quotereversed' => '8219',
    'quoteright' => '8217',
    'quoterightn' => '329',
    'quotesinglbase' => '8218',
    'quotesingle' => '39',
    'quotesinglemonospace' => '65287',
    'r' => '114',
    'rAarr' => '8667',
    'rArr' => '8658',
    'raarmenian' => '1404',
    'rabengali' => '2480',
    'racute' => '341',
    'radeva' => '2352',
    'radic' => '8730',
    'radical' => '8730',
    'radicalex' => '63717',
    'radoverssquare' => '13230',
    'radoverssquaredsquare' => '13231',
    'radsquare' => '13229',
    'rafe' => '1471',
    'rafehebrew' => '1471',
    'ragujarati' => '2736',
    'ragurmukhi' => '2608',
    'rahiragana' => '12425',
    'rakatakana' => '12521',
    'rakatakanahalfwidth' => '65431',
    'ralowerdiagonalbengali' => '2545',
    'ramiddlediagonalbengali' => '2544',
    'ramshorn' => '612',
    'rang' => '9002',
    'raquo' => '187',
    'rarr' => '8594',
    'rarr2' => '8649',
    'rarrhk' => '8618',
    'rarrlp' => '8620',
    'rarrtl' => '8611',
    'rarrw' => '8605',
    'ratio' => '8758',
    'rbopomofo' => '12566',
    'rcaron' => '345',
    'rcedil' => '343',
    'rcedilla' => '343',
    'rceil' => '8969',
    'rcircle' => '9441',
    'rcommaaccent' => '343',
    'rcub' => '125',
    'rcy' => '1088',
    'rdblgrave' => '529',
    'rdotaccent' => '7769',
    'rdotbelow' => '7771',
    'rdotbelowmacron' => '7773',
    'rdquo' => '8221',
    'rdquor' => '8220',
    'readingglasses' => '61476',
    'real' => '8476',
    'rect' => '9645',
    'referencemark' => '8251',
    'reflexsubset' => '8838',
    'reflexsuperset' => '8839',
    'reg' => '174',
    'registercircle' => '61617',
    'registered' => '174',
    'registersans' => '63720',
    'registerserif' => '63194',
    'registersquare' => '61616',
    'reharabic' => '1585',
    'reharmenian' => '1408',
    'rehfinalarabic' => '65198',
    'rehiragana' => '12428',
    'rehyehaleflamarabic' => '1585',
    'rekatakana' => '12524',
    'rekatakanahalfwidth' => '65434',
    'resh' => '1512',
    'reshdageshhebrew' => '64328',
    'reshhatafpatah' => '1512',
    'reshhatafpatahhebrew' => '1512',
    'reshhatafsegol' => '1512',
    'reshhatafsegolhebrew' => '1512',
    'reshhebrew' => '1512',
    'reshhiriq' => '1512',
    'reshhiriqhebrew' => '1512',
    'reshholam' => '1512',
    'reshholamhebrew' => '1512',
    'reshpatah' => '1512',
    'reshpatahhebrew' => '1512',
    'reshqamats' => '1512',
    'reshqamatshebrew' => '1512',
    'reshqubuts' => '1512',
    'reshqubutshebrew' => '1512',
    'reshsegol' => '1512',
    'reshsegolhebrew' => '1512',
    'reshsheva' => '1512',
    'reshshevahebrew' => '1512',
    'reshtsere' => '1512',
    'reshtserehebrew' => '1512',
    'reversedtilde' => '8765',
    'reviahebrew' => '1431',
    'reviamugrashhebrew' => '1431',
    'revlogicalnot' => '8976',
    'rfishhook' => '638',
    'rfishhookreversed' => '639',
    'rfloor' => '8971',
    'rgr' => '961',
    'rhabengali' => '2525',
    'rhadeva' => '2397',
    'rhard' => '8641',
    'rharu' => '8640',
    'rho' => '961',
    'rhombus4' => '61559',
    'rhombus6' => '61557',
    'rhook' => '637',
    'rhookturned' => '635',
    'rhookturnedsuperior' => '693',
    'rhosymbolgreek' => '1009',
    'rhotichookmod' => '734',
    'rhov' => '1009',
    'rieulacirclekorean' => '12913',
    'rieulaparenkorean' => '12817',
    'rieulcirclekorean' => '12899',
    'rieulhieuhkorean' => '12608',
    'rieulkiyeokkorean' => '12602',
    'rieulkiyeoksioskorean' => '12649',
    'rieulkorean' => '12601',
    'rieulmieumkorean' => '12603',
    'rieulpansioskorean' => '12652',
    'rieulparenkorean' => '12803',
    'rieulphieuphkorean' => '12607',
    'rieulpieupkorean' => '12604',
    'rieulpieupsioskorean' => '12651',
    'rieulsioskorean' => '12605',
    'rieulthieuthkorean' => '12606',
    'rieultikeutkorean' => '12650',
    'rieulyeorinhieuhkorean' => '12653',
    'rightangle' => '8735',
    'righttackbelowcmb' => '793',
    'righttriangle' => '8895',
    'rihiragana' => '12426',
    'rikatakana' => '12522',
    'rikatakanahalfwidth' => '65432',
    'ring' => '730',
    'ring2' => '61601',
    'ring4' => '61602',
    'ring6' => '61603',
    'ringbelowcmb' => '805',
    'ringbutton2' => '61604',
    'ringcmb' => '778',
    'ringhalfleft' => '703',
    'ringhalfleftarmenian' => '1369',
    'ringhalfleftbelowcmb' => '796',
    'ringhalfleftcentered' => '723',
    'ringhalfright' => '702',
    'ringhalfrightbelowcmb' => '825',
    'ringhalfrightcentered' => '722',
    'rinvertedbreve' => '531',
    'rittorusquare' => '13137',
    'rlarr2' => '8644',
    'rlhar2' => '8652',
    'rlinebelow' => '7775',
    'rlongleg' => '636',
    'rlonglegturned' => '634',
    'rmonospace' => '65362',
    'rohiragana' => '12429',
    'rokatakana' => '12525',
    'rokatakanahalfwidth' => '65435',
    'roruathai' => '3619',
    'rosette' => '61563',
    'rosettesolid' => '61564',
    'rpar' => '41',
    'rparen' => '9389',
    'rrabengali' => '2524',
    'rradeva' => '2353',
    'rragurmukhi' => '2652',
    'rreharabic' => '1681',
    'rrehfinalarabic' => '64397',
    'rrvocalicbengali' => '2528',
    'rrvocalicdeva' => '2400',
    'rrvocalicgujarati' => '2784',
    'rrvocalicvowelsignbengali' => '2500',
    'rrvocalicvowelsigndeva' => '2372',
    'rrvocalicvowelsigngujarati' => '2756',
    'rsaquo' => '8250',
    'rsh' => '8625',
    'rsqb' => '93',
    'rsquo' => '8217',
    'rsquor' => '8216',
    'rsuperior' => '63217',
    'rtblock' => '9616',
    'rthree' => '8908',
    'rtimes' => '8906',
    'rtri' => '9657',
    'rtrie' => '8885',
    'rtrif' => '9656',
    'rturned' => '633',
    'rturnedsuperior' => '692',
    'ruhiragana' => '12427',
    'rukatakana' => '12523',
    'rukatakanahalfwidth' => '65433',
    'rupeemarkbengali' => '2546',
    'rupeesignbengali' => '2547',
    'rupiah' => '63197',
    'ruthai' => '3620',
    'rvocalicbengali' => '2443',
    'rvocalicdeva' => '2315',
    'rvocalicgujarati' => '2699',
    'rvocalicvowelsignbengali' => '2499',
    'rvocalicvowelsigndeva' => '2371',
    'rvocalicvowelsigngujarati' => '2755',
    'rx' => '8478',
    's' => '115',
    'sabengali' => '2488',
    'sacute' => '347',
    'sacutedotaccent' => '7781',
    'sadarabic' => '1589',
    'sadeva' => '2360',
    'sadfinalarabic' => '65210',
    'sadinitialarabic' => '65211',
    'sadmedialarabic' => '65212',
    'saggitarius' => '61542',
    'sagujarati' => '2744',
    'sagurmukhi' => '2616',
    'sahiragana' => '12373',
    'sakatakana' => '12469',
    'sakatakanahalfwidth' => '65403',
    'sallallahoualayhewasallamarabic' => '65018',
    'samalg' => '8720',
    'samekh' => '1505',
    'samekhdagesh' => '64321',
    'samekhdageshhebrew' => '64321',
    'samekhhebrew' => '1505',
    'saraaathai' => '3634',
    'saraaethai' => '3649',
    'saraaimaimalaithai' => '3652',
    'saraaimaimuanthai' => '3651',
    'saraamthai' => '3635',
    'saraathai' => '3632',
    'saraethai' => '3648',
    'saraiileftthai' => '63622',
    'saraiithai' => '3637',
    'saraileftthai' => '63621',
    'saraithai' => '3636',
    'saraothai' => '3650',
    'saraueeleftthai' => '63624',
    'saraueethai' => '3639',
    'saraueleftthai' => '63623',
    'sarauethai' => '3638',
    'sarauthai' => '3640',
    'sarauuthai' => '3641',
    'sbopomofo' => '12569',
    'sbquo' => '8218',
    'sc' => '8827',
    'scaron' => '353',
    'scarondotaccent' => '7783',
    'sccue' => '8829',
    'scedil' => '351',
    'scedilla' => '351',
    'schwa' => '601',
    'schwacyrillic' => '1241',
    'schwadieresiscyrillic' => '1243',
    'schwahook' => '602',
    'scirc' => '349',
    'scircle' => '9442',
    'scircumflex' => '349',
    'scissors' => '61474',
    'scissorscutting' => '61475',
    'scnsim' => '8937',
    'scommaaccent' => '537',
    'scorpio' => '61541',
    'scsim' => '8831',
    'scy' => '1089',
    'sdot' => '8901',
    'sdotaccent' => '7777',
    'sdotb' => '8865',
    'sdotbelow' => '7779',
    'sdotbelowdotaccent' => '7785',
    'seagullbelowcmb' => '828',
    'second' => '8243',
    'secondtonechinese' => '714',
    'sect' => '167',
    'section' => '167',
    'seenarabic' => '1587',
    'seenfinalarabic' => '65202',
    'seeninitialarabic' => '65203',
    'seenmedialarabic' => '65204',
    'segol' => '1462',
    'segol13' => '1462',
    'segol1f' => '1462',
    'segol2c' => '1462',
    'segolhebrew' => '1462',
    'segolnarrowhebrew' => '1462',
    'segolquarterhebrew' => '1462',
    'segoltahebrew' => '1426',
    'segolwidehebrew' => '1462',
    'seharmenian' => '1405',
    'sehiragana' => '12379',
    'sekatakana' => '12475',
    'sekatakanahalfwidth' => '65406',
    'semi' => '59',
    'semicolon' => '59',
    'semicolonarabic' => '1563',
    'semicolonmonospace' => '65307',
    'semicolonsmall' => '65108',
    'semivoicedmarkkana' => '12444',
    'semivoicedmarkkanahalfwidth' => '65439',
    'sentisquare' => '13090',
    'sentosquare' => '13091',
    'setmn' => '8726',
    'seven' => '55',
    'sevenarabic' => '1639',
    'sevenbengali' => '2541',
    'sevencircle' => '9318',
    'sevencircleinversesansserif' => '10128',
    'sevendeva' => '2413',
    'seveneighths' => '8542',
    'sevengujarati' => '2797',
    'sevengurmukhi' => '2669',
    'sevenhackarabic' => '1639',
    'sevenhangzhou' => '12327',
    'sevenideographicparen' => '12838',
    'seveninferior' => '8327',
    'sevenmonospace' => '65303',
    'sevenoclock' => '61629',
    'sevenoldstyle' => '63287',
    'sevenparen' => '9338',
    'sevenperiod' => '9358',
    'sevenpersian' => '1783',
    'sevenroman' => '8566',
    'sevensans' => '61575',
    'sevensansinv' => '61586',
    'sevensuperior' => '8311',
    'seventeencircle' => '9328',
    'seventeenparen' => '9348',
    'seventeenperiod' => '9368',
    'seventhai' => '3671',
    'sextile' => '10038',
    'sfthyphen' => '173',
    'sgr' => '963',
    'shaarmenian' => '1399',
    'shabengali' => '2486',
    'shacyrillic' => '1096',
    'shaddaarabic' => '1617',
    'shaddadammaarabic' => '64609',
    'shaddadammatanarabic' => '64606',
    'shaddafathaarabic' => '64608',
    'shaddafathatanarabic' => '1617',
    'shaddakasraarabic' => '64610',
    'shaddakasratanarabic' => '64607',
    'shade' => '9618',
    'shadedark' => '9619',
    'shadelight' => '9617',
    'shademedium' => '9618',
    'shadeva' => '2358',
    'shagujarati' => '2742',
    'shagurmukhi' => '2614',
    'shalshelethebrew' => '1427',
    'sharp' => '9839',
    'shbopomofo' => '12565',
    'shchacyrillic' => '1097',
    'shchcy' => '1097',
    'shcy' => '1096',
    'sheenarabic' => '1588',
    'sheenfinalarabic' => '65206',
    'sheeninitialarabic' => '65207',
    'sheenmedialarabic' => '65208',
    'sheicoptic' => '995',
    'sheqel' => '8362',
    'sheqelhebrew' => '8362',
    'sheva' => '1456',
    'sheva115' => '1456',
    'sheva15' => '1456',
    'sheva22' => '1456',
    'sheva2e' => '1456',
    'shevahebrew' => '1456',
    'shevanarrowhebrew' => '1456',
    'shevaquarterhebrew' => '1456',
    'shevawidehebrew' => '1456',
    'shhacyrillic' => '1211',
    'shimacoptic' => '1005',
    'shin' => '1513',
    'shindagesh' => '64329',
    'shindageshhebrew' => '64329',
    'shindageshshindot' => '64300',
    'shindageshshindothebrew' => '64300',
    'shindageshsindot' => '64301',
    'shindageshsindothebrew' => '64301',
    'shindothebrew' => '1473',
    'shinhebrew' => '1513',
    'shinshindot' => '64298',
    'shinshindothebrew' => '64298',
    'shinsindot' => '64299',
    'shinsindothebrew' => '64299',
    'shook' => '642',
    'shy' => '173',
    'sigma' => '963',
    'sigma1' => '962',
    'sigmafinal' => '962',
    'sigmalunatesymbolgreek' => '1010',
    'sigmav' => '962',
    'sihiragana' => '12375',
    'sikatakana' => '12471',
    'sikatakanahalfwidth' => '65404',
    'siluqhebrew' => '1469',
    'siluqlefthebrew' => '1469',
    'sim' => '8764',
    'sime' => '8771',
    'similar' => '8764',
    'sindothebrew' => '1474',
    'siosacirclekorean' => '12916',
    'siosaparenkorean' => '12820',
    'sioscieuckorean' => '12670',
    'sioscirclekorean' => '12902',
    'sioskiyeokkorean' => '12666',
    'sioskorean' => '12613',
    'siosnieunkorean' => '12667',
    'siosparenkorean' => '12806',
    'siospieupkorean' => '12669',
    'siostikeutkorean' => '12668',
    'six' => '54',
    'sixarabic' => '1638',
    'sixbengali' => '2540',
    'sixcircle' => '9317',
    'sixcircleinversesansserif' => '10127',
    'sixdeva' => '2412',
    'sixgujarati' => '2796',
    'sixgurmukhi' => '2668',
    'sixhackarabic' => '1638',
    'sixhangzhou' => '12326',
    'sixideographicparen' => '12837',
    'sixinferior' => '8326',
    'sixmonospace' => '65302',
    'sixoclock' => '61628',
    'sixoldstyle' => '63286',
    'sixparen' => '9337',
    'sixperiod' => '9357',
    'sixpersian' => '1782',
    'sixroman' => '8565',
    'sixsans' => '61574',
    'sixsansinv' => '61585',
    'sixsuperior' => '8310',
    'sixteencircle' => '9327',
    'sixteencurrencydenominatorbengali' => '2553',
    'sixteenparen' => '9347',
    'sixteenperiod' => '9367',
    'sixthai' => '3670',
    'skullcrossbones' => '61518',
    'slash' => '47',
    'slashmonospace' => '65295',
    'slong' => '383',
    'slongdotaccent' => '7835',
    'smile' => '8995',
    'smileface' => '9786',
    'smonospace' => '65363',
    'snowflake' => '61524',
    'sofpasuqhebrew' => '1475',
    'softcy' => '1100',
    'softhyphen' => '173',
    'softsigncyrillic' => '1100',
    'sohiragana' => '12381',
    'sokatakana' => '12477',
    'sokatakanahalfwidth' => '65407',
    'sol' => '47',
    'soliduslongoverlaycmb' => '824',
    'solidusshortoverlaycmb' => '823',
    'sorusithai' => '3625',
    'sosalathai' => '3624',
    'sosothai' => '3595',
    'sosuathai' => '3626',
    'space' => '32',
    'spacehackarabic' => '32',
    'spade' => '9824',
    'spades' => '9824',
    'spadesuitblack' => '9824',
    'spadesuitwhite' => '9828',
    'sparen' => '9390',
    'sqcap' => '8851',
    'sqcup' => '8852',
    'sqsub' => '8847',
    'sqsube' => '8849',
    'sqsup' => '8848',
    'sqsupe' => '8850',
    'square' => '9633',
    'square2' => '61600',
    'square4' => '61607',
    'square6' => '61550',
    'squarebelowcmb' => '827',
    'squarecc' => '13252',
    'squarecm' => '13213',
    'squarediagonalcrosshatchfill' => '9641',
    'squarehorizontalfill' => '9636',
    'squarekg' => '13199',
    'squarekm' => '13214',
    'squarekmcapital' => '13262',
    'squareln' => '13265',
    'squarelog' => '13266',
    'squaremg' => '13198',
    'squaremil' => '13269',
    'squaremm' => '13212',
    'squaremsquared' => '13217',
    'squareorthogonalcrosshatchfill' => '9638',
    'squareupperlefttolowerrightfill' => '9639',
    'squareupperrighttolowerleftfill' => '9640',
    'squareverticalfill' => '9637',
    'squarewhitewithsmallblack' => '9635',
    'squf' => '9642',
    'srsquare' => '13275',
    'ssabengali' => '2487',
    'ssadeva' => '2359',
    'ssagujarati' => '2743',
    'ssangcieuckorean' => '12617',
    'ssanghieuhkorean' => '12677',
    'ssangieungkorean' => '12672',
    'ssangkiyeokkorean' => '12594',
    'ssangnieunkorean' => '12645',
    'ssangpieupkorean' => '12611',
    'ssangsioskorean' => '12614',
    'ssangtikeutkorean' => '12600',
    'sstarf' => '8902',
    'ssuperior' => '63218',
    'star' => '9734',
    'starf' => '9733',
    'starofdavid' => '61529',
    'starshadow' => '61622',
    'sterling' => '163',
    'sterlingmonospace' => '65505',
    'strokelongoverlaycmb' => '822',
    'strokeshortoverlaycmb' => '821',
    'sub' => '8834',
    'sube' => '8838',
    'subnE' => '8842',
    'subset' => '8834',
    'subsetnotequal' => '8842',
    'subsetorequal' => '8838',
    'succeeds' => '8827',
    'suchthat' => '8715',
    'suhiragana' => '12377',
    'sukatakana' => '12473',
    'sukatakanahalfwidth' => '65405',
    'sukunarabic' => '1618',
    'sum' => '8721',
    'summation' => '8721',
    'sun' => '9788',
    'sung' => '9834',
    'sunshine' => '61522',
    'sup' => '8835',
    'sup1' => '185',
    'sup2' => '178',
    'sup3' => '179',
    'supe' => '8839',
    'superset' => '8835',
    'supersetnotequal' => '8843',
    'supersetorequal' => '8839',
    'supnE' => '8843',
    'svsquare' => '13276',
    'syouwaerasquare' => '13180',
    'szlig' => '223',
    't' => '116',
    'tabengali' => '2468',
    'tackdown' => '8868',
    'tackleft' => '8867',
    'tadeva' => '2340',
    'tagujarati' => '2724',
    'tagurmukhi' => '2596',
    'taharabic' => '1591',
    'tahfinalarabic' => '65218',
    'tahinitialarabic' => '65219',
    'tahiragana' => '12383',
    'tahmedialarabic' => '65220',
    'taisyouerasquare' => '13181',
    'takatakana' => '12479',
    'takatakanahalfwidth' => '65408',
    'tapereel' => '61502',
    'target' => '61605',
    'tatweelarabic' => '1600',
    'tau' => '964',
    'taurus' => '61535',
    'tav' => '1514',
    'tavdages' => '64330',
    'tavdagesh' => '64330',
    'tavdageshhebrew' => '64330',
    'tavhebrew' => '1514',
    'tbar' => '359',
    'tbopomofo' => '12554',
    'tcaron' => '357',
    'tccurl' => '680',
    'tcedil' => '355',
    'tcedilla' => '355',
    'tcheharabic' => '1670',
    'tchehfinalarabic' => '64379',
    'tchehinitialarabic' => '64380',
    'tchehmedialarabic' => '64381',
    'tchehmeeminitialarabic' => '64380',
    'tcircle' => '9443',
    'tcircumflexbelow' => '7793',
    'tcommaaccent' => '539',
    'tcy' => '1090',
    'tdieresis' => '7831',
    'tdot' => '8411',
    'tdotaccent' => '7787',
    'tdotbelow' => '7789',
    'tecyrillic' => '1090',
    'tedescendercyrillic' => '1197',
    'teharabic' => '1578',
    'tehfinalarabic' => '65174',
    'tehhahinitialarabic' => '64674',
    'tehhahisolatedarabic' => '64524',
    'tehinitialarabic' => '65175',
    'tehiragana' => '12390',
    'tehjeeminitialarabic' => '64673',
    'tehjeemisolatedarabic' => '64523',
    'tehmarbutaarabic' => '1577',
    'tehmarbutafinalarabic' => '65172',
    'tehmedialarabic' => '65176',
    'tehmeeminitialarabic' => '64676',
    'tehmeemisolatedarabic' => '64526',
    'tehnoonfinalarabic' => '64627',
    'tekatakana' => '12486',
    'tekatakanahalfwidth' => '65411',
    'telephone' => '8481',
    'telephoneblack' => '9742',
    'telephonesolid' => '61480',
    'telhandsetcirc' => '61481',
    'telishagedolahebrew' => '1440',
    'telishaqetanahebrew' => '1449',
    'telrec' => '8981',
    'tencircle' => '9321',
    'tenideographicparen' => '12841',
    'tenoclock' => '61632',
    'tenparen' => '9341',
    'tenperiod' => '9361',
    'tenroman' => '8569',
    'tensans' => '61578',
    'tensansinv' => '61589',
    'tesh' => '679',
    'tet' => '1496',
    'tetdagesh' => '64312',
    'tetdageshhebrew' => '64312',
    'tethebrew' => '1496',
    'tetsecyrillic' => '1205',
    'tevirhebrew' => '1435',
    'tevirlefthebrew' => '1435',
    'tgr' => '964',
    'thabengali' => '2469',
    'thadeva' => '2341',
    'thagujarati' => '2725',
    'thagurmukhi' => '2597',
    'thalarabic' => '1584',
    'thalfinalarabic' => '65196',
    'thanthakhatlowleftthai' => '63640',
    'thanthakhatlowrightthai' => '63639',
    'thanthakhatthai' => '3660',
    'thanthakhatupperleftthai' => '63638',
    'theharabic' => '1579',
    'thehfinalarabic' => '65178',
    'thehinitialarabic' => '65179',
    'thehmedialarabic' => '65180',
    'there4' => '8756',
    'thereexists' => '8707',
    'therefore' => '8756',
    'theta' => '952',
    'theta1' => '977',
    'thetasym' => '977',
    'thetasymbolgreek' => '977',
    'thetav' => '977',
    'thgr' => '952',
    'thieuthacirclekorean' => '12921',
    'thieuthaparenkorean' => '12825',
    'thieuthcirclekorean' => '12907',
    'thieuthkorean' => '12620',
    'thieuthparenkorean' => '12811',
    'thinsp' => '8201',
    'thirteencircle' => '9324',
    'thirteenparen' => '9344',
    'thirteenperiod' => '9364',
    'thkap' => '8776',
    'thksim' => '8764',
    'thonangmonthothai' => '3601',
    'thook' => '429',
    'thophuthaothai' => '3602',
    'thorn' => '254',
    'thothahanthai' => '3607',
    'thothanthai' => '3600',
    'thothongthai' => '3608',
    'thothungthai' => '3606',
    'thousandcyrillic' => '1154',
    'thousandsseparatorarabic' => '1644',
    'thousandsseparatorpersian' => '1644',
    'three' => '51',
    'threearabic' => '1635',
    'threebengali' => '2537',
    'threecircle' => '9314',
    'threecircleinversesansserif' => '10124',
    'threedeva' => '2409',
    'threeeighths' => '8540',
    'threegujarati' => '2793',
    'threegurmukhi' => '2665',
    'threehackarabic' => '1635',
    'threehangzhou' => '12323',
    'threeideographicparen' => '12834',
    'threeinferior' => '8323',
    'threemonospace' => '65299',
    'threenumeratorbengali' => '2550',
    'threeoclock' => '61625',
    'threeoldstyle' => '63283',
    'threeparen' => '9334',
    'threeperiod' => '9354',
    'threepersian' => '1779',
    'threequarters' => '190',
    'threequartersemdash' => '63198',
    'threeroman' => '8562',
    'threesans' => '61571',
    'threesansinv' => '61582',
    'threesuperior' => '179',
    'threethai' => '3667',
    'thumbdown' => '61508',
    'thumbup' => '61507',
    'thzsquare' => '13204',
    'tihiragana' => '12385',
    'tikatakana' => '12481',
    'tikatakanahalfwidth' => '65409',
    'tikeutacirclekorean' => '12912',
    'tikeutaparenkorean' => '12816',
    'tikeutcirclekorean' => '12898',
    'tikeutkorean' => '12599',
    'tikeutparenkorean' => '12802',
    'tilde' => '732',
    'tildebelowcmb' => '816',
    'tildecmb' => '771',
    'tildecomb' => '771',
    'tildedoublecmb' => '864',
    'tildeoperator' => '8764',
    'tildeoverlaycmb' => '820',
    'tildeverticalcmb' => '830',
    'times' => '215',
    'timesb' => '8864',
    'timescircle' => '8855',
    'tipehahebrew' => '1430',
    'tipehalefthebrew' => '1430',
    'tippigurmukhi' => '2672',
    'titlocyrilliccmb' => '1155',
    'tiwnarmenian' => '1407',
    'tlinebelow' => '7791',
    'tmonospace' => '65364',
    'toarmenian' => '1385',
    'tohiragana' => '12392',
    'tokatakana' => '12488',
    'tokatakanahalfwidth' => '65412',
    'tonebarextrahighmod' => '741',
    'tonebarextralowmod' => '745',
    'tonebarhighmod' => '742',
    'tonebarlowmod' => '744',
    'tonebarmidmod' => '743',
    'tonefive' => '445',
    'tonesix' => '389',
    'tonetwo' => '424',
    'tonos' => '900',
    'tonsquare' => '13095',
    'top' => '8868',
    'topatakthai' => '3599',
    'tortoiseshellbracketleft' => '12308',
    'tortoiseshellbracketleftsmall' => '65117',
    'tortoiseshellbracketleftvertical' => '65081',
    'tortoiseshellbracketright' => '12309',
    'tortoiseshellbracketrightsmall' => '65118',
    'tortoiseshellbracketrightvertical' => '65082',
    'totaothai' => '3605',
    'tpalatalhook' => '427',
    'tparen' => '9391',
    'tprime' => '8244',
    'trade' => '8482',
    'trademark' => '8482',
    'trademarksans' => '63722',
    'trademarkserif' => '63195',
    'tretroflexhook' => '648',
    'triagdn' => '9660',
    'triaglf' => '9668',
    'triagrt' => '9658',
    'triagup' => '9650',
    'trie' => '8796',
    'tristar2' => '61609',
    'ts' => '678',
    'tsadi' => '1510',
    'tsadidagesh' => '64326',
    'tsadidageshhebrew' => '64326',
    'tsadihebrew' => '1510',
    'tscy' => '1094',
    'tsecyrillic' => '1094',
    'tsere' => '1461',
    'tsere12' => '1461',
    'tsere1e' => '1461',
    'tsere2b' => '1461',
    'tserehebrew' => '1461',
    'tserenarrowhebrew' => '1461',
    'tserequarterhebrew' => '1461',
    'tserewidehebrew' => '1461',
    'tshcy' => '1115',
    'tshecyrillic' => '1115',
    'tstrok' => '359',
    'tsuperior' => '63219',
    'ttabengali' => '2463',
    'ttadeva' => '2335',
    'ttagujarati' => '2719',
    'ttagurmukhi' => '2591',
    'tteharabic' => '1657',
    'ttehfinalarabic' => '64359',
    'ttehinitialarabic' => '64360',
    'ttehmedialarabic' => '64361',
    'tthabengali' => '2464',
    'tthadeva' => '2336',
    'tthagujarati' => '2720',
    'tthagurmukhi' => '2592',
    'tturned' => '647',
    'tuhiragana' => '12388',
    'tukatakana' => '12484',
    'tukatakanahalfwidth' => '65410',
    'tusmallhiragana' => '12387',
    'tusmallkatakana' => '12483',
    'tusmallkatakanahalfwidth' => '65391',
    'twelvecircle' => '9323',
    'twelveoclock' => '61634',
    'twelveparen' => '9343',
    'twelveperiod' => '9363',
    'twelveroman' => '8571',
    'twentycircle' => '9331',
    'twentyhangzhou' => '21316',
    'twentyparen' => '9351',
    'twentyperiod' => '9371',
    'twixt' => '8812',
    'two' => '50',
    'twoarabic' => '1634',
    'twobengali' => '2536',
    'twocircle' => '9313',
    'twocircleinversesansserif' => '10123',
    'twodeva' => '2408',
    'twodotenleader' => '8229',
    'twodotleader' => '8229',
    'twodotleadervertical' => '65072',
    'twogujarati' => '2792',
    'twogurmukhi' => '2664',
    'twohackarabic' => '1634',
    'twohangzhou' => '12322',
    'twoideographicparen' => '12833',
    'twoinferior' => '8322',
    'twomonospace' => '65298',
    'twonumeratorbengali' => '2549',
    'twooclock' => '61624',
    'twooldstyle' => '63282',
    'twoparen' => '9333',
    'twoperiod' => '9353',
    'twopersian' => '1778',
    'tworoman' => '8561',
    'twosans' => '61570',
    'twosansinv' => '61581',
    'twostroke' => '443',
    'twosuperior' => '178',
    'twothai' => '3666',
    'twothirds' => '8532',
    'u' => '117',
    'uArr' => '8657',
    'uacgr' => '973',
    'uacute' => '250',
    'uarr' => '8593',
    'uarr2' => '8648',
    'ubar' => '649',
    'ubengali' => '2441',
    'ubopomofo' => '12584',
    'ubrcy' => '1118',
    'ubreve' => '365',
    'ucaron' => '468',
    'ucirc' => '251',
    'ucircle' => '9444',
    'ucircumflex' => '251',
    'ucircumflexbelow' => '7799',
    'ucy' => '1091',
    'ucyrillic' => '1091',
    'udattadeva' => '2385',
    'udblac' => '369',
    'udblacute' => '369',
    'udblgrave' => '533',
    'udeva' => '2313',
    'udiagr' => '944',
    'udieresis' => '252',
    'udieresisacute' => '472',
    'udieresisbelow' => '7795',
    'udieresiscaron' => '474',
    'udieresiscyrillic' => '1265',
    'udieresisgrave' => '476',
    'udieresismacron' => '470',
    'udigr' => '971',
    'udotbelow' => '7909',
    'ugrave' => '249',
    'ugujarati' => '2697',
    'ugurmukhi' => '2569',
    'uharl' => '8639',
    'uharr' => '8638',
    'uhblk' => '9600',
    'uhiragana' => '12358',
    'uhookabove' => '7911',
    'uhorn' => '432',
    'uhornacute' => '7913',
    'uhorndotbelow' => '7921',
    'uhorngrave' => '7915',
    'uhornhookabove' => '7917',
    'uhorntilde' => '7919',
    'uhungarumlaut' => '369',
    'uhungarumlautcyrillic' => '1267',
    'uinvertedbreve' => '535',
    'ukatakana' => '12454',
    'ukatakanahalfwidth' => '65395',
    'ukcyrillic' => '1145',
    'ukorean' => '12636',
    'ulcorn' => '8988',
    'ulcrop' => '8975',
    'umacr' => '363',
    'umacron' => '363',
    'umacroncyrillic' => '1263',
    'umacrondieresis' => '7803',
    'umatragurmukhi' => '2625',
    'uml' => '168',
    'umonospace' => '65365',
    'underscore' => '95',
    'underscoredbl' => '8215',
    'underscoremonospace' => '65343',
    'underscorevertical' => '65075',
    'underscorewavy' => '65103',
    'union' => '8746',
    'universal' => '8704',
    'uogon' => '371',
    'uogonek' => '371',
    'uparen' => '9392',
    'upblock' => '9600',
    'uplus' => '8846',
    'upperdothebrew' => '1476',
    'upsi' => '965',
    'upsih' => '978',
    'upsilon' => '965',
    'upsilondieresis' => '971',
    'upsilondieresistonos' => '944',
    'upsilonlatin' => '650',
    'upsilontonos' => '973',
    'uptackbelowcmb' => '797',
    'uptackmod' => '724',
    'uragurmukhi' => '2675',
    'urcorn' => '8989',
    'urcrop' => '8974',
    'uring' => '367',
    'ushortcyrillic' => '1118',
    'usmallhiragana' => '12357',
    'usmallkatakana' => '12453',
    'usmallkatakanahalfwidth' => '65385',
    'ustraightcyrillic' => '1199',
    'ustraightstrokecyrillic' => '1201',
    'utilde' => '361',
    'utildeacute' => '7801',
    'utildebelow' => '7797',
    'utri' => '9653',
    'utrif' => '9652',
    'uubengali' => '2442',
    'uudeva' => '2314',
    'uugujarati' => '2698',
    'uugurmukhi' => '2570',
    'uumatragurmukhi' => '2626',
    'uuml' => '252',
    'uuvowelsignbengali' => '2498',
    'uuvowelsigndeva' => '2370',
    'uuvowelsigngujarati' => '2754',
    'uvowelsignbengali' => '2497',
    'uvowelsigndeva' => '2369',
    'uvowelsigngujarati' => '2753',
    'v' => '118',
    'vArr' => '8661',
    'vDash' => '8872',
    'vadeva' => '2357',
    'vagujarati' => '2741',
    'vagurmukhi' => '2613',
    'vakatakana' => '12535',
    'varr' => '8597',
    'vav' => '1493',
    'vavdagesh' => '64309',
    'vavdagesh65' => '64309',
    'vavdageshhebrew' => '64309',
    'vavhebrew' => '1493',
    'vavholam' => '64331',
    'vavholamhebrew' => '64331',
    'vavvavhebrew' => '1520',
    'vavyodhebrew' => '1521',
    'vcircle' => '9445',
    'vcy' => '1074',
    'vdash' => '8866',
    'vdotbelow' => '7807',
    'vecyrillic' => '1074',
    'veebar' => '8891',
    'veharabic' => '1700',
    'vehfinalarabic' => '64363',
    'vehinitialarabic' => '64364',
    'vehmedialarabic' => '64365',
    'vekatakana' => '12537',
    'vellip' => '8942',
    'venus' => '9792',
    'verbar' => '124',
    'verticalbar' => '124',
    'verticallineabovecmb' => '781',
    'verticallinebelowcmb' => '809',
    'verticallinelowmod' => '716',
    'verticallinemod' => '712',
    'vewarmenian' => '1406',
    'vhook' => '651',
    'vikatakana' => '12536',
    'vineleafboldne' => '61594',
    'vineleafboldnw' => '61595',
    'vineleafboldse' => '61597',
    'vineleafboldsw' => '61596',
    'viramabengali' => '2509',
    'viramadeva' => '2381',
    'viramagujarati' => '2765',
    'virgo' => '61539',
    'visargabengali' => '2435',
    'visargadeva' => '2307',
    'visargagujarati' => '2691',
    'vltri' => '8882',
    'vmonospace' => '65366',
    'voarmenian' => '1400',
    'voicediterationhiragana' => '12446',
    'voicediterationkatakana' => '12542',
    'voicedmarkkana' => '12443',
    'voicedmarkkanahalfwidth' => '65438',
    'vokatakana' => '12538',
    'vparen' => '9393',
    'vprime' => '8242',
    'vprop' => '8733',
    'vrtri' => '8883',
    'vtilde' => '7805',
    'vturned' => '652',
    'vuhiragana' => '12436',
    'vukatakana' => '12532',
    'w' => '119',
    'wacute' => '7811',
    'waekorean' => '12633',
    'wahiragana' => '12431',
    'wakatakana' => '12527',
    'wakatakanahalfwidth' => '65436',
    'wakorean' => '12632',
    'wasmallhiragana' => '12430',
    'wasmallkatakana' => '12526',
    'wattosquare' => '13143',
    'wavedash' => '12316',
    'wavyunderscorevertical' => '65076',
    'wawarabic' => '1608',
    'wawfinalarabic' => '65262',
    'wawhamzaabovearabic' => '1572',
    'wawhamzaabovefinalarabic' => '65158',
    'wbsquare' => '13277',
    'wcirc' => '373',
    'wcircle' => '9446',
    'wcircumflex' => '373',
    'wdieresis' => '7813',
    'wdotaccent' => '7815',
    'wdotbelow' => '7817',
    'wedgeq' => '8793',
    'wehiragana' => '12433',
    'weierp' => '8472',
    'weierstrass' => '8472',
    'wekatakana' => '12529',
    'wekorean' => '12638',
    'weokorean' => '12637',
    'wgrave' => '7809',
    'wheel' => '61533',
    'whitebullet' => '9702',
    'whitecircle' => '9675',
    'whitecircleinverse' => '9689',
    'whitecornerbracketleft' => '12302',
    'whitecornerbracketleftvertical' => '65091',
    'whitecornerbracketright' => '12303',
    'whitecornerbracketrightvertical' => '65092',
    'whitediamond' => '9671',
    'whitediamondcontainingblacksmalldiamond' => '9672',
    'whitedownpointingsmalltriangle' => '9663',
    'whitedownpointingtriangle' => '9661',
    'whiteleftpointingsmalltriangle' => '9667',
    'whiteleftpointingtriangle' => '9665',
    'whitelenticularbracketleft' => '12310',
    'whitelenticularbracketright' => '12311',
    'whiterightpointingsmalltriangle' => '9657',
    'whiterightpointingtriangle' => '9655',
    'whitesmallsquare' => '9643',
    'whitesmilingface' => '9786',
    'whitesquare' => '9633',
    'whitestar' => '9734',
    'whitetelephone' => '9743',
    'whitetortoiseshellbracketleft' => '12312',
    'whitetortoiseshellbracketright' => '12313',
    'whiteuppointingsmalltriangle' => '9653',
    'whiteuppointingtriangle' => '9651',
    'wihiragana' => '12432',
    'wikatakana' => '12528',
    'wikorean' => '12639',
    'windowslogo' => '61695',
    'wmonospace' => '65367',
    'wohiragana' => '12434',
    'wokatakana' => '12530',
    'wokatakanahalfwidth' => '65382',
    'won' => '8361',
    'wonmonospace' => '65510',
    'wowaenthai' => '3623',
    'wparen' => '9394',
    'wreath' => '8768',
    'wring' => '7832',
    'wsuperior' => '695',
    'wturned' => '653',
    'wynn' => '447',
    'x' => '120',
    'xabovecmb' => '829',
    'xbopomofo' => '12562',
    'xcirc' => '9675',
    'xcircle' => '9447',
    'xdieresis' => '7821',
    'xdotaccent' => '7819',
    'xdtri' => '9661',
    'xeharmenian' => '1389',
    'xgr' => '958',
    'xi' => '958',
    'xmarkbld' => '61691',
    'xmonospace' => '65368',
    'xparen' => '9395',
    'xrhombus' => '61558',
    'xsuperior' => '739',
    'xutri' => '9651',
    'y' => '121',
    'yaadosquare' => '13134',
    'yabengali' => '2479',
    'yacute' => '253',
    'yacy' => '1103',
    'yadeva' => '2351',
    'yaekorean' => '12626',
    'yagujarati' => '2735',
    'yagurmukhi' => '2607',
    'yahiragana' => '12420',
    'yakatakana' => '12516',
    'yakatakanahalfwidth' => '65428',
    'yakorean' => '12625',
    'yamakkanthai' => '3662',
    'yasmallhiragana' => '12419',
    'yasmallkatakana' => '12515',
    'yasmallkatakanahalfwidth' => '65388',
    'yatcyrillic' => '1123',
    'ycirc' => '375',
    'ycircle' => '9448',
    'ycircumflex' => '375',
    'ycy' => '1099',
    'ydieresis' => '255',
    'ydotaccent' => '7823',
    'ydotbelow' => '7925',
    'yeharabic' => '1610',
    'yehbarreearabic' => '1746',
    'yehbarreefinalarabic' => '64431',
    'yehfinalarabic' => '65266',
    'yehhamzaabovearabic' => '1574',
    'yehhamzaabovefinalarabic' => '65162',
    'yehhamzaaboveinitialarabic' => '65163',
    'yehhamzaabovemedialarabic' => '65164',
    'yehinitialarabic' => '65267',
    'yehmedialarabic' => '65268',
    'yehmeeminitialarabic' => '64733',
    'yehmeemisolatedarabic' => '64600',
    'yehnoonfinalarabic' => '64660',
    'yehthreedotsbelowarabic' => '1745',
    'yekorean' => '12630',
    'yen' => '165',
    'yenmonospace' => '65509',
    'yeokorean' => '12629',
    'yeorinhieuhkorean' => '12678',
    'yerahbenyomohebrew' => '1450',
    'yerahbenyomolefthebrew' => '1450',
    'yericyrillic' => '1099',
    'yerudieresiscyrillic' => '1273',
    'yesieungkorean' => '12673',
    'yesieungpansioskorean' => '12675',
    'yesieungsioskorean' => '12674',
    'yetivhebrew' => '1434',
    'ygrave' => '7923',
    'yhook' => '436',
    'yhookabove' => '7927',
    'yiarmenian' => '1397',
    'yicy' => '1111',
    'yicyrillic' => '1111',
    'yikorean' => '12642',
    'yinyang' => '9775',
    'yiwnarmenian' => '1410',
    'ymonospace' => '65369',
    'yod' => '1497',
    'yoddagesh' => '64313',
    'yoddageshhebrew' => '64313',
    'yodhebrew' => '1497',
    'yodyodhebrew' => '1522',
    'yodyodpatahhebrew' => '64287',
    'yohiragana' => '12424',
    'yoikorean' => '12681',
    'yokatakana' => '12520',
    'yokatakanahalfwidth' => '65430',
    'yokorean' => '12635',
    'yosmallhiragana' => '12423',
    'yosmallkatakana' => '12519',
    'yosmallkatakanahalfwidth' => '65390',
    'yotgreek' => '1011',
    'yoyaekorean' => '12680',
    'yoyakorean' => '12679',
    'yoyakthai' => '3618',
    'yoyingthai' => '3597',
    'yparen' => '9396',
    'ypogegrammeni' => '890',
    'ypogegrammenigreekcmb' => '837',
    'yr' => '422',
    'yring' => '7833',
    'ysuperior' => '696',
    'ytilde' => '7929',
    'yturned' => '654',
    'yucy' => '1102',
    'yuhiragana' => '12422',
    'yuikorean' => '12684',
    'yukatakana' => '12518',
    'yukatakanahalfwidth' => '65429',
    'yukorean' => '12640',
    'yuml' => '255',
    'yusbigcyrillic' => '1131',
    'yusbigiotifiedcyrillic' => '1133',
    'yuslittlecyrillic' => '1127',
    'yuslittleiotifiedcyrillic' => '1129',
    'yusmallhiragana' => '12421',
    'yusmallkatakana' => '12517',
    'yusmallkatakanahalfwidth' => '65389',
    'yuyekorean' => '12683',
    'yuyeokorean' => '12682',
    'yyabengali' => '2527',
    'yyadeva' => '2399',
    'z' => '122',
    'zaarmenian' => '1382',
    'zacute' => '378',
    'zadeva' => '2395',
    'zagurmukhi' => '2651',
    'zaharabic' => '1592',
    'zahfinalarabic' => '65222',
    'zahinitialarabic' => '65223',
    'zahiragana' => '12374',
    'zahmedialarabic' => '65224',
    'zainarabic' => '1586',
    'zainfinalarabic' => '65200',
    'zakatakana' => '12470',
    'zaqefgadolhebrew' => '1429',
    'zaqefqatanhebrew' => '1428',
    'zarqahebrew' => '1432',
    'zayin' => '1494',
    'zayindagesh' => '64310',
    'zayindageshhebrew' => '64310',
    'zayinhebrew' => '1494',
    'zbopomofo' => '12567',
    'zcaron' => '382',
    'zcircle' => '9449',
    'zcircumflex' => '7825',
    'zcurl' => '657',
    'zcy' => '1079',
    'zdot' => '380',
    'zdotaccent' => '380',
    'zdotbelow' => '7827',
    'zecyrillic' => '1079',
    'zedescendercyrillic' => '1177',
    'zedieresiscyrillic' => '1247',
    'zehiragana' => '12380',
    'zekatakana' => '12476',
    'zero' => '48',
    'zeroarabic' => '1632',
    'zerobengali' => '2534',
    'zerodeva' => '2406',
    'zerogujarati' => '2790',
    'zerogurmukhi' => '2662',
    'zerohackarabic' => '1632',
    'zeroinferior' => '8320',
    'zeromonospace' => '65296',
    'zerooldstyle' => '63280',
    'zeropersian' => '1776',
    'zerosans' => '61568',
    'zerosansinv' => '61579',
    'zerosuperior' => '8304',
    'zerothai' => '3664',
    'zerowidthjoiner' => '65279',
    'zerowidthnonjoiner' => '8204',
    'zerowidthspace' => '8203',
    'zeta' => '950',
    'zgr' => '950',
    'zhbopomofo' => '12563',
    'zhcy' => '1078',
    'zhearmenian' => '1386',
    'zhebrevecyrillic' => '1218',
    'zhecyrillic' => '1078',
    'zhedescendercyrillic' => '1175',
    'zhedieresiscyrillic' => '1245',
    'zihiragana' => '12376',
    'zikatakana' => '12472',
    'zinorhebrew' => '1454',
    'zlinebelow' => '7829',
    'zmonospace' => '65370',
    'zohiragana' => '12382',
    'zokatakana' => '12478',
    'zparen' => '9397',
    'zretroflexhook' => '656',
    'zstroke' => '438',
    'zuhiragana' => '12378',
    'zukatakana' => '12474',
    'zwnj' => '8204',
};

1;
