use strict;
use warnings FATAL => 'all';
use Config;
use File::Copy ();
use Symbol;

use constant DEVEL => getpwuid($>) =~ /^(?:stas|pgollucci)$/o;

my $VERSION = '1.05';

my %prereq = (
              "Apache::Test" => "1.25", # ipv6 fixes
              "Devel::Peek"  => 0.96,
             );

my %common_opts = (
    NAME         => "Apache::Peek",
    VERSION_FROM => "Peek.pm",
    XSPROTOARG   => '-noprototypes',
    DEFINE       => '-DMOD_PERL',
    PREREQ_PM    => \%prereq,
);

MAIN: {
  exit &main();
}

sub main {

  ## check PERL version

  my $perl_ver = &get_perl_ver();

  # currently we support only 5.8-perlio
  die "Perl $] must be built with PerlIO support in order to use this module." if $perl_ver eq '5.8' && !$Config{useperlio};

  ## which mp version ( 1.2x or 2.0.x)
  my ($mp_wanted, $mp_ver) = &get_mp_ver();

  print STDERR "Using Perl/$perl_ver and mod_perl/$mp_ver\n\n";

  &prepare_build($perl_ver, $mp_wanted);

  my @clean_files = &test_configure($mp_wanted);
  $common_opts{clean} = { FILES =>"@clean_files"};

  if ($mp_wanted == 2) {
    $prereq{mod_perl} = 1.99023;
    require ModPerl::MM;
    ModPerl::MM::WriteMakefile(%common_opts);
  }
  else {
    require Apache::src;
    my $inc = Apache::src->new->inc;
    die "Can't find mod_perl header files installed" unless $inc;

    require ExtUtils::MakeMaker;
    ExtUtils::MakeMaker::WriteMakefile(
        INC	         => $inc,
        LIBS         => [''],
        %common_opts
    );
  }

  return 0;
}

sub test_configure ($) {

  my $mp_gen = shift;
  my @clean_files = ();

  if (eval { require Apache::TestMM }) {
    Apache::TestMM->import(qw(test clean));
    my @scripts = qw(t/TEST);
    # accept the configs from command line
    Apache::TestMM::filter_args();

    Apache::TestMM::generate_script($_) for @scripts;

    push @clean_files, @scripts;

    my $httpd;
    # mp2 already knows its 
    if ($mp_gen == 1) {
      #        require Apache::test;
      #        # can't really use get_test_params, since it may pick the wrong httpd
      #        # must explicitly ask for the path to httpd
      #        my %params = Apache::test->get_test_params;
      #        $common_opts{macro}{APACHE} = $httpd;
    }
  } else {
    warn "***: You should install Apache::Test to do real testing\n";
    # META: soon on CPAN
    *MY::test = *MY::test = sub {
      return <<'EOF';
test : pure_all
	@echo \*** This test suite requires Apache::Test available from the
	@echo \*** mod_perl 2.0 sources or the httpd-test distribution.
EOF
    }
  }

  return @clean_files;
}


sub prepare_build ($$) {

  my ($perl_ver, $mp_wanted) = @_;

  my $xs = "Peek.xs.mp$mp_wanted.perl$perl_ver";
  my $pm = "Peek.pm.mp$mp_wanted.perl$perl_ver";

  unlink 'Peek.xs' if -e 'Peek.xs';

  if (DEVEL && eval { symlink("", ""); 1 }) {
    # so that it's easier to modify/rebuild w/o re-running Makefile.PL
    symlink $xs, 'Peek.xs' or die "Couldn't symlink $xs => 'Peek.xs': $!";
  }
  else {
    File::Copy::copy($xs, 'Peek.xs');
  }

  # adjust the version and glue the doc (open() for 5.005_03 support)
  unlink "Peek.pm" if "Peek.pm";

  my ($in, $out) = ($pm, 'Peek.pm');
  my ($ifh, $ofh) = (Symbol::gensym(), Symbol::gensym());
  open $ifh,  "<$in"  or die "can't open $in: $!";
  open $ofh, ">$out" or die "can't open $out: $!";
  print $ofh "# WARNING: DO NOT EDIT THIS FILE, EDIT $in instead\n\n";
  while (<$ifh>) {
    s/(.*\$VERSION\s*=).*/$1 $VERSION;/;
    print $ofh $_;
  }
  print $ofh <DATA>;
  close $ifh or die "can't close $in $!";
  close $ofh or die "can't close $out $!";

  return;
}

##############################################################################
##
##
##
##
##############################################################################
sub get_mp_ver() {

  my $flag = 0;
  my @args = ();

  while (my $arg = shift @ARGV) {
    if ($arg =~ /^MOD_PERL=([12])$/) {
      $flag = $1;
    }
    else {
      push @args, $arg;
    }
  }
  @ARGV = @args;

  # check %ENV
  my $env = exists $ENV{MOD_PERL} ? $ENV{MOD_PERL} : 0;

  # check for contradicting requirements
  if ($env && $flag && $flag != $env) {
    die <<EOF;
Can\'t decide which mod_perl version should be used, since you have
supplied contradicting requirements:
    enviroment variable MOD_PERL=$env
    Makefile.PL option  MOD_PERL=$flag
EOF
  }

  my $wanted = 2; ## default ot wanting mp2
  $wanted = 1 if $env == 1 || $flag == 1;

  my $mp_ver;

  if ($wanted == 2) {
    eval { require mod_perl2 };
    if ($mod_perl2::VERSION < 1.99023 || $@) {
      die "You don't seem to have mod_perl 2.0 installed";
    }
    else {
      $mp_ver = $mod_perl2::VERSION;
    }
  }
  else {
    eval { require mod_perl };
    no warnings qw(uninitialized);
    if ($mod_perl::VERSION > 1.99 || $@) {
      die "You don't seem to have mod_perl 1.0 installed";
    }
    else {
      $mp_ver = $mod_perl::VERSION;
    }
  }

  return ($wanted, $mp_ver);
}

sub get_perl_ver() {

  #eval { requie 5.10.0 }   ? "5.10" :
  eval { require 5.8.0 }    ? "5.8" :
  eval { require 5.6.0 }    ? "5.6" :
  eval { require 5.005_03 } ? "5.5" :
  die "Perl version $] is unsupported";
}

1;
__DATA__
__END__

=head1 NAME

Apache::Peek - A data debugging tool for the XS programmer (under mod_perl)

=head1 SYNOPSIS

        use Apache::Peek;
        Dump( $a );
        Dump( $a, 5 );
        DumpArray( 5, $a, $b, ... );

        # more functionality inherited from Devel::Peek

=head1 DESCRIPTION

C<Apache::Peek> is a sub-class of C<Devel::Peek>. The only difference
is that it overrides the stderr stream, to which C<Devel::Peek> sends
its output, and send the output to the client instead.

Note: The following table summarizes what parts of the
C<Devel::Peek>'s API are available, depending on the used Perl version
and configurations:

  5.005_0x                 all
  5.6.x                    only Dump()
  5.8.x (w/  -Duseperlio)  all
  5.8.x (w/o -Duseperlio)  none

Patches to complete the functionality under all configurations are
welcome.

C<Apache::Peek> works both with mod_perl 1.0 and 2.0.

See the C<Devel::Peek> manpage for a complete documentation.



=head1 Author

Hacking the mod_perl versions:

Philip M. Gollucci      pgollucci@p6m7g8.com

Previously: Doug MacEachern and Stas Bekman

Writing the original C<Devel::Peek>

Ilya Zakharevich	ilya@math.ohio-state.edu



=head1 Copyright

Copyright (c) 1995-98 Ilya Zakharevich. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

Author of this software makes no claim whatsoever about suitability,
reliability, edability, editability or usability of this product, and
should not be kept liable for any damage resulting from the use of
it. If you can use it, you are in luck, if not, I should not be kept
responsible. Keep a handy copy of your backup tape at hand.

=head1 SEE ALSO

L<perlguts>, and L<perlguts>, again.

=cut
