package Log::ger::Plugin::SingleLog;

our $DATE = '2017-06-28'; # DATE
our $VERSION = '0.001'; # VERSION

use strict;
use warnings;

use Log::ger::Util;

sub get_hooks {
    my %conf = @_;

    $conf{sub_name} //= "log";
    $conf{method_name} //= "log";

    return {
        before_install_routines => [
            __PACKAGE__, 99,

            sub {
                my %args = @_;

                my @logger_subs;
                for (@{ $args{routines} }) {
                    $logger_subs[$_->[2]]    = $_->[0] if $_->[3] & 1 && !($_->[3] & 2);
                }
                my @logger_methods;
                for (@{ $args{routines} }) {
                    $logger_methods[$_->[2]] = $_->[0] if $_->[3] & 1 &&  ($_->[3] & 2);
                }

                push @{ $args{routines} }, [
                    sub {
                        my $level = Log::ger::Util::numeric_level(shift);
                        goto &{$logger_subs[$level]};
                    },
                    $conf{sub_name},
                    -1,
                    1 | 0,
                ];
                push @{ $args{routines} }, [
                    sub {
                        shift;
                        my $level = Log::ger::Util::numeric_level(shift);
                        goto &{$logger_methods[$level]};
                    },
                    $conf{method_name},
                    -1,
                    1 | 2,
                ];

                [1];
            }],
    };
}

1;
# ABSTRACT: Create a log($LEVEL, ...) subroutine/method

__END__

=pod

=encoding UTF-8

=head1 NAME

Log::ger::Plugin::SingleLog - Create a log($LEVEL, ...) subroutine/method

=head1 VERSION

version 0.001

=head1 SYNOPSIS

 use Log::ger::Plugin SingleLog => (
     sub_name => 'log',    # optional
     method_name => 'log', # optional
 );
 use Log::ger;

=head1 DESCRIPTION

The L<Log::ger> way is to create separate C<log_XXX> routine for each level,
e.g. C<log_trace>, C<log_warn>, and so on. But sometimes you might want a single
log routine that takes $level as the first argument, e.g. instead of:

 log_warn('blah ...');

or:

 $log->debug('Blah: %s', $data);

you prefer:

 log('warn', 'blah ...');

or:

 $log->log('debug', 'Blah: %s', $data);

This plugin can create such log routine for you.

Caveats:

=over

=item * Slower

Using log($level, ...) is a bit slower (extra subroutine call, level parsing,
and per-level dispatching).

=item * The individual separate log_XXX subroutines/methods are still installed

=back

=for Pod::Coverage ^(.+)$

=head1 SEE ALSO

L<Log::ger>

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
