package Bencher::Scenario::DataCleansing::Object;

our $DATE = '2016-04-24'; # DATE
our $VERSION = '0.001'; # VERSION

use 5.010001;
use strict;
use warnings;

our $scenario = {
    summary => 'Benchmark data cleansing (unblessing object)',
    modules => {
        # for the Data::Rmap method
        'Acme::Damn' => {},
        'Scalar::Util' => {},
    },
    participants => [
        {
            name => 'Data::Clean-inplace',
            module => 'Data::Clean',
            code_template => 'state $cl = Data::Clean->new(-obj => ["unbless"]); $cl->clean_in_place(<data>)',
            tags => ['inplace'],
        },
        {
            name => 'Data::Clean-clone',
            module => 'Data::Clean',
            code_template => 'state $cl = Data::Clean->new(-obj => ["unbless"]); $cl->clone_and_clean(<data>)',
        },
        {
            name => 'JSON::PP',
            module => 'JSON::PP',
            code_template => 'state $json = JSON::PP->new->allow_blessed(1)->convert_blessed(1); $json->decode($json->encode(<data>))',
        },
        {
            name => 'Data::Rmap',
            module => 'Data::Rmap',
            code_template => 'my $data = <data>; Data::Rmap::rmap_ref(sub { Acme::Damn::damn($_) if Scalar::Util::blessed($_) }, $data); $data',
            tags => ['inplace'],
        },
    ],
    datasets => [
        {
            name => 'ary100-u1-obj',
            summary => 'A 100-element array containing 1 "unclean" data: object',
            args => {
                data => do {
                    my $data = [0..99];
                    $data->[49] = bless [], "Foo";
                    $data;
                },
            },
        },
        {
            name => 'ary100-u100-obj',
            summary => 'A 100-element array containing 100 "unclean" data: object',
            args => {
                data => do {
                    my $data = [map {bless [], "Foo"} 0..99];
                    $data;
                },
            },
        },
        {
            name => 'ary10k-u1-obj',
            summary => 'A 10k-element array containing 1 "unclean" data: object',
            args => {
                data => do {
                    my $data = [0..999];
                    $data->[499] = bless [], "Foo";
                    $data;
                },
            },
        },
        {
            name => 'ary10k-u10k-obj',
            summary => 'A 10k-element array containing 10k "unclean" data: object',
            args => {
                data => do {
                    my $data = [map {bless [], "Foo"} 0..999];
                    $data;
                },
            },
        },
    ],
};

1;
# ABSTRACT: Benchmark data cleansing (unblessing object)

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::DataCleansing::Object - Benchmark data cleansing (unblessing object)

=head1 VERSION

This document describes version 0.001 of Bencher::Scenario::DataCleansing::Object (from Perl distribution Bencher-Scenarios-DataCleansing), released on 2016-04-24.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m DataCleansing::Object

To run module startup overhead benchmark:

 % bencher --module-startup -m DataCleansing::Object

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Acme::Damn> 0.06

L<Data::Clean> 0.33

L<Data::Rmap> 0.64

L<JSON::PP> 2.27300

L<Scalar::Util> 1.41

=head1 BENCHMARK PARTICIPANTS

=over

=item * Data::Clean-inplace (perl_code) [inplace]

Code template:

 state $cl = Data::Clean->new(-obj => ["unbless"]); $cl->clean_in_place(<data>)



=item * Data::Clean-clone (perl_code)

Code template:

 state $cl = Data::Clean->new(-obj => ["unbless"]); $cl->clone_and_clean(<data>)



=item * JSON::PP (perl_code)

Code template:

 state $json = JSON::PP->new->allow_blessed(1)->convert_blessed(1); $json->decode($json->encode(<data>))



=item * Data::Rmap (perl_code) [inplace]

Code template:

 my $data = <data>; Data::Rmap::rmap_ref(sub { Acme::Damn::damn($_) if Scalar::Util::blessed($_) }, $data); $data



=back

=head1 BENCHMARK DATASETS

=over

=item * ary100-u1-obj

A 100-element array containing 1 "unclean" data: object

=item * ary100-u100-obj

A 100-element array containing 100 "unclean" data: object

=item * ary10k-u1-obj

A 10k-element array containing 1 "unclean" data: object

=item * ary10k-u10k-obj

A 10k-element array containing 10k "unclean" data: object

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default options (C<< bencher -m DataCleansing::Object >>):

 +---------------------+-----------------+-----------+-----------+------------+---------+---------+
 | participant         | dataset         | rate (/s) | time (ms) | vs_slowest | errors  | samples |
 +---------------------+-----------------+-----------+-----------+------------+---------+---------+
 | JSON::PP            | ary10k-u1-obj   | 2.6e+02   | 3.9       | 1          | 4.2e-06 | 21      |
 | Data::Rmap          | ary10k-u10k-obj | 369       | 2.71      | 1.44       | 1.2e-06 | 20      |
 | JSON::PP            | ary10k-u10k-obj | 369       | 2.71      | 1.44       | 4.1e-07 | 22      |
 | Data::Clean-clone   | ary10k-u10k-obj | 646       | 1.55      | 2.52       | 4.8e-07 | 20      |
 | Data::Rmap          | ary10k-u1-obj   | 742       | 1.35      | 2.9        | 1.3e-06 | 20      |
 | Data::Clean-inplace | ary10k-u10k-obj | 2.02e+03  | 0.495     | 7.89       | 2.1e-07 | 20      |
 | JSON::PP            | ary100-u1-obj   | 2.72e+03  | 0.367     | 10.6       | 1.8e-07 | 27      |
 | JSON::PP            | ary100-u100-obj | 3.56e+03  | 0.281     | 13.9       | 2.1e-07 | 20      |
 | Data::Rmap          | ary100-u100-obj | 3.65e+03  | 0.274     | 14.2       | 2e-07   | 22      |
 | Data::Clean-clone   | ary10k-u1-obj   | 6233.99   | 0.160411  | 24.3398    | 1.3e-10 | 20      |
 | Data::Clean-clone   | ary100-u100-obj | 6.27e+03  | 0.159     | 24.5       | 4.4e-08 | 30      |
 | Data::Rmap          | ary100-u1-obj   | 7.3e+03   | 0.14      | 28         | 2.1e-07 | 21      |
 | Data::Clean-inplace | ary10k-u1-obj   | 7.3e+03   | 0.14      | 29         | 1.4e-07 | 27      |
 | Data::Clean-inplace | ary100-u100-obj | 20640.1   | 0.0484493 | 80.5867    | 3.5e-11 | 20      |
 | Data::Clean-clone   | ary100-u1-obj   | 5.1e+04   | 0.02      | 2e+02      | 2e-08   | 20      |
 | Data::Clean-inplace | ary100-u1-obj   | 6.8e+04   | 0.015     | 2.7e+02    | 2.6e-08 | 21      |
 +---------------------+-----------------+-----------+-----------+------------+---------+---------+


Benchmark module startup overhead (C<< bencher -m DataCleansing::Object --module-startup >>):

 +---------------------+-----------+------------------------+------------+---------+---------+
 | participant         | time (ms) | mod_overhead_time (ms) | vs_slowest | errors  | samples |
 +---------------------+-----------+------------------------+------------+---------+---------+
 | JSON::PP            | 15        | 12.6                   | 1          | 5e-05   | 20      |
 | Data::Rmap          | 7.35      | 4.95                   | 2.03       | 7.2e-06 | 20      |
 | Data::Clean         | 6.6       | 4.2                    | 2.3        | 1.7e-05 | 22      |
 | perl -e1 (baseline) | 2.4       | 0                      | 6.21       | 2.4e-06 | 20      |
 +---------------------+-----------+------------------------+------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-DataCleansing>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-DataCleansing>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-DataCleansing>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
