package IO::Iron::Applications::IronCache::Command::list;

## no critic (Documentation::RequirePodAtEnd)
## no critic (Documentation::RequirePodSections)
## no critic (RegularExpressions::RequireExtendedFormatting)
## no critic (RegularExpressions::RequireLineBoundaryMatching)
## no critic (RegularExpressions::ProhibitEscapedMetacharacters)

use 5.010_000;
use strict;
use warnings FATAL => 'all';

# Global creator
BEGIN {
    # Inheritance
    use parent qw( IO::Iron::Applications::Command::CommandBase ); # Inheritance
}

# Global destructor
END {
}

# ABSTRACT: List caches or items in a cache.

our $VERSION = '0.11'; # VERSION: generated by DZP::OurPkgVersion


use Log::Any  qw{$log};

use IO::Iron::Applications::IronCache -command;

sub description {
	return "List caches or items.";
}

sub usage_desc {
	my ($self, $opt, $args) = @_;
    return 
            $opt->arg0() . " %o list caches <cache_name>[,<cache_name>]"
            . "\n" . 
            $opt->arg0() . " %o list items <item_key>[,<item_key>] --cache <cache_name>[,<cache_name>]";
}

sub opt_spec {
    # Note, dashes '-' become underscores '_' during opt_spec conversion!
    return (
        IO::Iron::Applications::Command::CommandBase::opt_spec_base(),
        [ 'cache=s',    "cache name or names (separated with \',\').", ],
        [ 'show-value!',    "Show the item value. Default: off.", { 'default' => 0, }, ],
        [ 'alternatives!',    "Only list possible alternatives, no network access. Default: off.", { 'default' => 0, }, ],
    );
}

sub validate_args {
	my ($self, $opt, $args) = @_;
    $self->validate_args_base($opt, $args);
    $self->usage_error("wrong arguments: use 'list caches' or 'list items'.") unless (defined $args->[0] && ($args->[0] eq 'caches' || $args->[0] eq 'items'));
    $self->usage_error("missing argument --cache name (must have if 'list items')") if ($args->[0] eq 'items' && !defined $opt->{'cache'});
    $self->usage_error("Wrong number of arguments") if ($args->[0] eq 'caches' && scalar @{$args} > 2);
    $self->usage_error("Wrong number of arguments") if ($args->[0] eq 'items' && scalar @{$args} > 2);
}

sub execute {
	my ($self, $opts, $args) = @_;

    $self->raise_logging_levels_from_options($opts);
    if($self->check_for_iron_io_config($opts)) {
        return 1;
    }
    $log->tracef('Entering execute(%s, %s)', $opts, $args);

    my %parameters;
    $parameters{'config'} = $opts->{'config'} if defined $opts->{'config'};
    $parameters{'policies'} = $opts->{'policies'} if defined $opts->{'policies'};
    $parameters{'no-policy'} = $opts->{'no-policy'};
    $parameters{'alternatives'} = $opts->{'alternatives'};
    my %output;
    if($args->[0] eq 'caches') {
        %output = IO::Iron::Applications::IronCache::Functionality::list_caches(%parameters);
        print $self->combine_template("list_caches", \%output);
    }
    elsif($args->[0] eq 'items') {
        $parameters{'item_key'} = [ split q{,}, $args->[1] ] if (scalar @{$args} > 1); # expects array
        $parameters{'cache_name'} = [ split q{,}, $opts->{'cache'} ]; # expects array;
        %output = IO::Iron::Applications::IronCache::Functionality::list_items(%parameters);
        my %instructions = ( 'show_value' => $opts->{'show_value'} );
        print $self->combine_template("list_items", \%output, \%instructions);
    }
    return 0;
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

IO::Iron::Applications::IronCache::Command::list - List caches or items in a cache.

=head1 VERSION

version 0.11

=head1 SYNOPSIS

This package is for internal use of IO::Iron packages.

=head1 AUTHOR

Mikko Koivunalho <mikko.koivunalho AT iki.fi>

=head1 BUGS

Please report any bugs or feature requests to bug-io-iron-applications@rt.cpan.org or through the web interface at:
 http://rt.cpan.org/Public/Dist/Display.html?Name=IO-Iron-Applications

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by Mikko Koivunalho.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

The full text of the license can be found in the
F<LICENSE> file included with this distribution.

=cut
