package # hide from PAUSE
App::YTDL::Merge;

use warnings;
use strict;
use 5.010000;

use Exporter qw( import );
our @EXPORT_OK = qw( search_and_merge );

use File::Copy            qw( move );
use File::Path            qw( make_path );
use File::Spec::Functions qw( catfile catdir );

use Term::Choose        qw( choose );
use Try::Tiny           qw( try catch );

use App::YTDL::Helper qw( get_filename timestamp_to_upload_date check_mapping_stdout encode_fs uni_capture uni_system );



sub search_and_merge {
    my ( $opt, $info ) = @_;
    return if ! merge_requirements_ok( $opt );
    my $first_line_out = 'Merged: ';

    EXTRACTOR: for my $ex ( keys %$info ) {
        VIDEO_ID: for my $video_id ( keys %{$info->{$ex}} ) {
            if ( keys %{$info->{$ex}{$video_id}{file_name}} == 2 ) {
                my ( $video, $audio );
                my $height = '';
                my $error = 0;
                try {
                    for my $fmt ( keys %{$info->{$ex}{$video_id}{file_name}} ) {
                        my $file = $info->{$ex}{$video_id}{file_name}{$fmt};
                        my $stream_info = _get_stream_info( $opt, $file );
                        if ( $stream_info =~ /^\s*codec_type=audio\s*$/m && $stream_info !~ /^\s*codec_type=video\s*$/m ) {
                            $audio = $file;
                        }
                        elsif ( $stream_info =~ /^\s*codec_type=video\s*$/m ) {
                            $video = $file;
                            if ( $stream_info =~ /^\s*height=(\S+)\s*$/m ) {
                                $height = $1;
                            }
                        }
                    }
                }
                catch {
                    say "$video_id - Merge: ", check_mapping_stdout( $opt, $_ );
                    push @{$opt->{error_merge}}, $video_id;
                    $error = 1;
                };
                next VIDEO_ID if $error;
                if ( ! $audio ) { #
                    say "$video_id: No file with audio-only stream!";
                    next VIDEO_ID;
                }
                if ( ! $video ) {
                    say "$video_id: No file with video stream!";
                    next VIDEO_ID;
                }
                if ( $first_line_out ) {
                    say $first_line_out;
                    $first_line_out = '';
                }
                my $video_dir = $info->{$ex}{$video_id}{video_dir};
                my $file_name = get_filename( $opt, $info->{$ex}{$video_id}{title}, $opt->{merge_ext}, $height );
                my $out = catfile $video_dir, $file_name;
                if ( -f $out && ! $opt->{merge_overwrite} ) {
                    my $overwrite_ok = choose(
                        [ '  Skip', '  Overwrite' ],
                        { prompt => 'Already exists: ' . $file_name, layout => 3, index => 1 } #
                    );
                    next VIDEO_ID if ! $overwrite_ok;
                }
                try {
                    _merge_video_with_audio( $opt, $video, $audio, $out );
                    say ' ' .  check_mapping_stdout( $opt, $out );
                    if ( $opt->{merged_in_files} == 1 ) {
                        my $dir_orig_files = catdir $video_dir, $opt->{dir_stream_files};
                        make_path encode_fs( $dir_orig_files );
                        move encode_fs( $video ), encode_fs( $dir_orig_files ) or die check_mapping_stdout( $opt, $video ) . " $!";
                        move encode_fs( $audio ), encode_fs( $dir_orig_files ) or die check_mapping_stdout( $opt, $audio ) . " $!";
                    }
                    elsif ( $opt->{merged_in_files} == 2 ) {
                        unlink encode_fs( $video ) or die check_mapping_stdout( $opt, $video ) . " $!";
                        unlink encode_fs( $audio ) or die check_mapping_stdout( $opt, $audio ) . " $!";
                    }
                    timestamp_to_upload_date( $info, $ex, $video_id, $out ) if $opt->{modify_timestamp};
                }
                catch {
                    say "$video_id - Merge: ", check_mapping_stdout( $opt, $_ );
                    push @{$opt->{error_merge}}, $video_id;
                };
            }
        }
    }
}


sub _merge_video_with_audio {
    my ( $opt, $video, $audio, $out ) = @_;
    my @cmd = ( $opt->{ffmpeg} );
    push @cmd, '-loglevel', $opt->{merge_loglevel};
    push @cmd, '-y';
    push @cmd, '-i', encode_fs( $video );
    push @cmd, '-i', encode_fs( $audio );
    push @cmd, '-codec', 'copy';
    push @cmd, '-map', '0:v:0';
    push @cmd, '-map', '1:a:0';
    push @cmd, encode_fs( $out );
    uni_system( @cmd );
}


sub _get_stream_info {
    my ( $opt, $file ) = @_;
    my @cmd = ( $opt->{ffprobe} );
    push @cmd, '-loglevel', 'warning';
    push @cmd, '-show_streams';
    push @cmd, encode_fs( $file );
    my $stream_info = uni_capture( @cmd );
    return $stream_info;
}


sub merge_requirements_ok {
    my ( $opt ) = @_;
    try {
        my @cmd = ( $opt->{ffmpeg}, '-version' );
        my $capture = uni_capture( @cmd );
    }
    catch {
        say "Could not find 'ffmpeg' - No merge.";
        return;
    };
    try {
        my @cmd = ( $opt->{ffprobe}, '-version' );
        my $capture = uni_capture( @cmd );
    }
    catch {
        say "Could not find 'ffprobe' - No merge.";
        return;
    };
    return 1;
}




1;


__END__
