package # hide from PAUSE
App::YTDL::Helper;

use warnings;
use strict;
use 5.010000;

use Exporter qw( import );
our @EXPORT_OK = qw( sec_to_time get_filename timestamp_to_upload_date encode_fs check_encode_stdout
                    write_json read_json sanitize_for_path uni_capture uni_system HIDE_CURSOR SHOW_CURSOR );

use Encode             qw( encode );
use Time::Local        qw( timelocal );
use Unicode::Normalize qw( NFC );

use Encode::Locale      qw();
use File::Touch         qw();
use IPC::System::Simple qw( capture system );
use JSON                qw();
use Term::Choose::Util  qw( unicode_trim );
#use Text::Unidecode     qw( unidecode ); # require-d
use Unicode::GCString   qw();

use if $^O eq 'MSWin32', 'Win32::ShellQuote';

use constant {
    HIDE_CURSOR => "\e[?25l",
    SHOW_CURSOR => "\e[?25h",
};



sub get_filename {
    my ( $opt, $title, $ext, $fmt ) = @_;
    $ext //= 'unknown';
    $fmt //= '';
    $fmt = '_' . $fmt if length $fmt;
    my $len_ext = Unicode::GCString->new( $ext )->columns();
    my $len_fmt = Unicode::GCString->new( $fmt )->columns();
    my $max_len_title = $opt->{max_len_f_name} - ( $len_fmt + 1 + $len_ext );
    $title = unicode_trim( sanitize_for_path( $opt, $title ), $max_len_title );
    my $file_name = $title . $fmt . '.' . $ext;
    return $file_name;
}


sub sanitize_for_path {
    my ( $opt, $str ) = @_;
    $str =~ s/^\s+|\s+\z//g;
    $str =~ s/\s/_/g                         if $opt->{replace_spaces};
    #$str =~ s/^\.+//                         if $^O eq 'Unix';
    $str =~ s/[\0\/<>&]/-/g                  if $opt->{sanitize_filename} == 0;
    $str =~ s/[\x{00}-\x{1F}"\/\\:*?<>|]/-/g if $opt->{sanitize_filename} == 1;
    # NTFS unsupported characters:  / \ : " * ? < > |       # 0x00-0x1F
    #$str =~ s/[^A-Za-z0-9_-]/-/g             if $opt->{sanitize_filename} == 2;
    return $str;
}


sub timestamp_to_upload_date {
    my ( $info, $ex, $video_id, $file ) = @_;
    if ( $info->{$ex}{$video_id}{upload_datetime} =~ /^(\d\d\d\d)-(\d\d)-(\d\d)T(\d\d):(\d\d):(\d\d)\z/ ) {
        my $time = timelocal( $6, $5, $4, $3, $2 - 1, $1 );
        my $ref = File::Touch->new( time => $time );
        my $count = $ref->touch( encode_fs( $file ) );
    }
}


sub encode_fs {
    my ( $filename ) = @_;
    my $encoded_filename;
    if ( eval { $encoded_filename = encode( 'locale_fs', NFC( $filename ), 1 ); 1 } ) {
        return $encoded_filename;
    }
    return $filename;
}


sub check_encode_stdout {
    my $opt = shift;
    if ( wantarray ) {
        return @_ if $Encode::Locale::ENCODING_CONSOLE_OUT =~ /^UTF-/i;
        if ( $opt->{text_unidecode} ) {
            my @unidecoded;
            if ( eval { require Text::Unidecode; @unidecoded = Text::Unidecode::unidecode( @_ ); 1 } ) {
                return @unidecoded;
            }
        }
        binmode STDOUT, ':pop';
        my @encoded_lax = map { encode( 'console_out', NFC( $_ ), sub { '*' } ) } @_;
        binmode STDOUT, ':encoding(console_out)';
        return @encoded_lax;
    }
    else {
        my $string = shift;
        return $string if $Encode::Locale::ENCODING_CONSOLE_OUT =~ /^UTF-/i;
        if ( $opt->{text_unidecode} ) {
            my $unidecoded;
            if ( eval { require Text::Unidecode; $unidecoded = Text::Unidecode::unidecode( $string ); 1 } ) {
                return $unidecoded;
            }
        }
        binmode STDOUT, ':pop';
        my $encoded_lax = encode( 'console_out', NFC( $string ), sub { '*' } );
        binmode STDOUT, ':encoding(console_out)';
        return $encoded_lax;
    }
}


sub uni_capture {
    my ( @cmd ) = @_;
    if ( wantarray ) {
        my @capture;
        if ( $^O eq 'MSWin32' ) {
            @capture = capture( Win32::ShellQuote::quote_native( @cmd ) );
        }
        else {
            @capture = capture( @cmd );
        }
        return @capture;
    }
    else {
        my $capture;
        if ( $^O eq 'MSWin32' ) {
            $capture = capture( Win32::ShellQuote::quote_native( @cmd ) );
        }
        else {
            $capture = capture( @cmd );
        }
        return $capture;
    }
}


sub uni_system {
    my ( @cmd ) = @_;
    if ( $^O eq 'MSWin32' ) {
        system( Win32::ShellQuote::quote_native( @cmd ) );
    }
    else {
        system( @cmd );
    }
}


sub sec_to_time {
    my ( $seconds, $long ) = @_;
    die 'seconds: not defined'                         if ! defined $seconds;
    die 'seconds: "' . $seconds . '" invalid datatype' if $seconds !~ /^[0-9]+\z/;
    my ( $minutes, $hours );
    if ( $seconds ) {
        $minutes = int( $seconds / 60 );
        $seconds = $seconds % 60;
    }
    if ( $minutes ) {
        $hours   = int( $minutes / 60 );
        $minutes = $minutes % 60;
    }
    return sprintf "%d:%02d:%02d", $hours // 0, $minutes // 0, $seconds if $long;
    return sprintf "%d:%02d:%02d", $hours,      $minutes,      $seconds if $hours;
    return sprintf "%d:%02d",                   $minutes,      $seconds if $minutes;
    return sprintf "0:%02d",                                   $seconds;
}


sub write_json {
    my ( $opt, $file, $h_ref ) = @_;
    my $json = JSON->new->pretty->canonical->utf8->encode( $h_ref );
    open my $fh, '>', encode_fs( $file ) or die check_encode_stdout( $opt, $file ) . " $!";
    print $fh $json;
    close $fh;
}


sub read_json {
    my ( $opt, $file ) = @_;
    return {} if ! -f encode_fs( $file );
    open my $fh, '<', encode_fs( $file ) or die check_encode_stdout( $opt, $file ) . " $!";
    my $json = do { local $/; <$fh> };
    close $fh;
    my $h_ref = JSON->new->utf8->decode( $json ) if $json;
    return $h_ref;
}



1;


__END__
