# $Id: Project.pm,v 1.33 2002/11/06 08:18:44 joern Exp $

#-----------------------------------------------------------------------
# Copyright (C) 2001-2002 Jrn Reder <joern@zyn.de> All Rights Reserved
# 
# This module is part of Video::DVDRip, which is free software; you can
# redistribute it and/or modify it under the same terms as Perl itself.
#-----------------------------------------------------------------------

package Video::DVDRip::Project;

use base Video::DVDRip::Base;

use Carp;
use strict;

use FileHandle;
use Data::Dumper;
use File::Basename;
use File::Path;
use File::Copy;

use Video::DVDRip::Content;

sub name			{ shift->{name}				}
sub version			{ shift->{version}			}
sub filename			{ shift->{filename}			}
sub dvd_device			{ shift->{dvd_device}			}
sub vob_dir			{ shift->{vob_dir}  			}
sub avi_dir			{ shift->{avi_dir}  			}
sub snap_dir			{ shift->{snap_dir}  			}
sub content			{ shift->{content}			}
sub last_saved_data		{ shift->{last_saved_data}		}
sub selected_title_nr		{ shift->{selected_title_nr}		}
sub rip_mode			{ shift->{rip_mode} 	       || 'rip'	}
sub dvd_image_dir		{ shift->{dvd_image_dir}		}
sub convert_message		{ shift->{convert_message}		}
sub last_selected_nb_page	{ shift->{last_selected_nb_page}	}

sub set_name			{ shift->{name}			= $_[1] }
sub set_version			{ shift->{version}		= $_[1] }
sub set_filename		{ shift->{filename}		= $_[1] }
sub set_dvd_device		{ shift->{dvd_device}		= $_[1]	}
sub set_vob_dir			{ shift->{vob_dir}  		= $_[1]	}
sub set_avi_dir			{ shift->{avi_dir}  		= $_[1]	}
sub set_snap_dir		{ shift->{snap_dir}		= $_[1]	}
sub set_content			{ shift->{content}		= $_[1] }
sub set_last_saved_data		{ shift->{last_saved_data}	= $_[1] }
sub set_selected_title_nr	{ shift->{selected_title_nr}	= $_[1] }
sub set_rip_mode		{ shift->{rip_mode}		= $_[1]	}
sub set_dvd_image_dir		{ shift->{dvd_image_dir}	= $_[1]	}
sub set_convert_message		{ shift->{convert_message}	= $_[1]	}
sub set_last_selected_nb_page	{ shift->{last_selected_nb_page}= $_[1]	}

sub logfile {
	my $self = shift;
	return $self->snap_dir."/logfile.txt";
}

sub ifo_dir {
	my $self = shift;
	
	return sprintf ("%s/ifo", $self->snap_dir);
}

sub new {
	my $class = shift;
	my %par = @_;
	my  ($name, $filename) =
	@par{'name','filename'};
	
	confess "missing name" if not $name;
	
	my $self = {
		name              => $name,
		version           => $Video::DVDRip::VERSION,
		filename          => $filename,
		dvd_device	  => "/dev/dvd",
		rip_mode          => "rip",
		vob_dir		  => undef,
		avi_dir		  => undef,
		snap_dir	  => undef,
		content  	  => undef,
		last_saved_data   => undef,
		selected_title_nr => undef,
	};
	
	bless $self, $class;

	my $content = Video::DVDRip::Content->new ( project => $self );

	$self->set_content ($content);

	return $self;
}

sub new_from_file {
	my $class = shift;
	my %par = @_;
	my  ($filename) =
	@par{'filename'};
	
	confess "missing filename" if not $filename;
	
	my $self = bless {
		filename => $filename,
	}, $class;
	
	$self->load;
	
	$self->set_filename ($filename);
	$self->set_version ($Video::DVDRip::VERSION);

	return $self;
}

sub save {
	my $self = shift; $self->trace_in;
	my %par = @_;
	my ($filename) = @par{'filename'};

	$filename ||= $self->filename;
	confess "not filename set" if not $filename;
	
	my $data_sref = $self->get_save_data;
	
	my $fh = FileHandle->new;

	open ($fh, "> $filename") or confess "can't write $filename";
	print $fh q{# $Id: Project.pm,v 1.33 2002/11/06 08:18:44 joern Exp $},"\n";
	print $fh "# This file was generated by Video::DVDRip Version $Video::DVDRip::VERSION\n\n";

	print $fh ${$data_sref};
	close $fh;
	
	$self->set_last_saved_data ($data_sref);

	$self->log ("Project file saved to '$filename'");

	1;
}

sub backup_copy {
	my $self = shift; $self->trace_in;
	
	my $filename = $self->snap_dir."/backup.rip";
	$self->save ( filename => $filename );
	
	1;
}

sub get_save_data {
	my $self = shift; $self->trace_in;
	
	my $last_saved_data = $self->last_saved_data;
	$self->set_last_saved_data(undef);

	my $filename = $self->filename;
	$self->set_filename(undef);

	my $dd = Data::Dumper->new ( [$self], ['project'] );
	$dd->Indent(1);
	$dd->Purity(1);
	my $data = $dd->Dump;

	$self->set_last_saved_data($last_saved_data);
	$self->set_filename ($filename);

	return \$data;
}

sub changed {
	my $self = shift; $self->trace_in;

	return 1 if not $self->last_saved_data;

	my $actual_data_sref = $self->get_save_data;
	my $saved_data_sref  = $self->last_saved_data;

	my $actual = join ("\n", map { $_.="," if !/,$/;$_ }
				 sort split (/\n/, $$actual_data_sref));
	my $saved  = join ("\n", map { $_.="," if !/,$/;$_ }
				 sort split (/\n/, $$saved_data_sref));

	if ( $self->debug_level ) {
		open (DBG,">/tmp/saved.txt"); print FOO $saved; close FOO;
		open (DBG,">/tmp/actual.txt"); print FOO $actual; close FOO;
	}

	return $actual ne $saved;
}

sub load {
	my $self = shift; $self->trace_in;
	
	my $filename = $self->filename;
	croak "no filename set" if not $filename;
	croak "can't read $filename" if not -r $filename;
	
	my $fh = FileHandle->new;
	open ($fh, $filename) or croak "can't read $filename";
	my $data = join ('', <$fh>);
	close $fh;

	croak "File is no dvd::rip file"
		if $data !~ /This file was generated by Video::DVDRip/;

	my ($version, undef, $pre) = $data =~ /DVDRip Version (\d+\.\d+)(_(\d+))?/;

	my $project;
	$project = eval($data);
	croak "can't load $filename. Perl error: $@" if $@;

	bless $project, ref($self);

	$self->convert_from_old_version (
		project => $project,
		version => $version,
		pre     => $pre,
	);

	%{$self} = %{$project};

	$self->content->set_project ($self);

	$self->set_last_saved_data ($self->get_save_data);

	1;
}

sub rip_data_source {
	my $self = shift; $self->trace_in;
	
	my $mode = $self->rip_mode;

	my $source;

	if ( $mode eq 'rip' or $mode eq 'dvd' ) {
		$source = $self->dvd_device;

	} elsif ( $mode eq 'dvd_image' ) {
		$source = $self->dvd_image_dir;

	}

	return $source;
}

sub convert_from_old_version {
	my $self = shift;
	my %par = @_;
	my  ($project, $version, $pre) =
	@par{'project','version','pre'};

	if ( $version < 0.45 or ( $version == 0.45 and
	     defined $pre and $pre < 4 ) ) {
	     	$self->convert_audio_tracks_0_45_04 (
			project => $project,
		);
	}

	if ( $version < 0.47 or ( $version == 0.47 and
	     defined $pre and $pre < 2 ) ) {
	     	$self->set_audio_bitrates_0_47_02 (
			project => $project,
		);
	}

	1;
}

sub convert_audio_tracks_0_45_04 {
	my $self = shift;
	my %par = @_;
	my ($project) = @par{'project'};

	print "[auto-conversion] Import audio configuration (0.45_04)\n";

	$project->set_convert_message (
		"WARNING: converted audio tracks to new data structure\n".
		"of version 0.45_04. After saving the project file you\n".
		"can't use it with an older version anymore!. Consider\n".
		"making a backup copy before saving!"
	);

	foreach my $title ( values %{$project->content->titles} ) {
		my $old_tracks = $title->audio_tracks;
		my @probe_tracks;
		my @tc_tracks;
		my $i = 0;
		foreach my $old ( @{$old_tracks} ) {
			my $probe_track = Video::DVDRip::ProbeAudio->new (
				type		=> $old->{type},
				lang		=> $old->{lang},
				channels	=> $old->{channels},
				bitrate		=> $old->{bitrate},
				sample_rate	=> $old->{sample_rate},
				volume_rescale  => $old->{scan_result}
						       ->{volume_rescale},
				scan_output	=> $old->{scan_result}
						       ->{scan_output},
			);

			my $tc_track = Video::DVDRip::Audio->new (
				tc_nr => $i
			);

			$tc_track->set_tc_option_n (
				$old->{tc_option_n}
			);
			$tc_track->set_tc_volume_rescale (
				$old->{tc_volume_rescale}
			);

			if ( $i == $title->{audio_channel} ) {
				$tc_track->set_tc_target_track(0);
				$tc_track->set_tc_audio_codec (
					$title->{tc_ac3_passthrough} ? "ac3" : "mp3"
				);
				$tc_track->set_tc_mp3_quality (
					$title->{tc_mp3_quality}
				);
				$tc_track->set_tc_audio_filter (
					$title->{tc_use_drc} ? 'a52drc' : 'rescale',
				);
				$tc_track->set_tc_bitrate (
					$title->{tc_audio_bitrate}
				);
			}
			push @probe_tracks, $probe_track;
			push @tc_tracks,    $tc_track;

			++$i;
		}

		$title->probe_result->set_audio_tracks ( \@probe_tracks );
		$title->set_tc_audio_tracks ( \@tc_tracks );
	}
	
	1;
}

sub set_audio_bitrates_0_47_02 {
	my $self = shift;
	my %par = @_;
	my ($project) = @par{'project'};

	print "[auto-conversion] Setting new audio bitrate attributes (0.47_02)\n";

	foreach my $title ( values %{$project->content->titles} ) {
		my $audio_info = $title->audio_tracks;
		my $i = 0;
		foreach my $audio ( @{$title->tc_audio_tracks} ) {
			$audio->set_tc_mp2_bitrate ( $audio->{tc_bitrate} );
			$audio->set_tc_mp3_bitrate ( $audio->{tc_bitrate} );
			$audio->set_tc_ogg_bitrate ( $audio->{tc_bitrate} );
			$audio->set_tc_ac3_bitrate ( $audio_info->[$i]->bitrate );
			++$i;
		}
	}

	1;
}

sub copy_ifo_files {
	my $self = shift;

	mkpath ( [ $self->ifo_dir  ], 0, 0755);

	my $mounted = $self->dvd_is_mounted;
	$self->mount_dvd if not $mounted;

	my @files = glob (
		$self->config('dvd_mount_point').
		"/{video_ts,VIDEO_TS}/{vts,VTS}*{ifo,IFO}"
	);

	if ( not @files ) {
		$self->log ("WARNING: no IFO files found - vobsub feature disabled.");
	}

	$self->log ("Copying IFO files to ".$self->ifo_dir);

	copy ($_, $self->ifo_dir."/".lc(basename($_))) for @files;

	$self->umount_dvd if not $mounted;

	1;
}

sub check_dvd_in_drive {
	my $self = shift;
	
	my $rip_mode = $self->rip_mode;

	my $dvd_device = $self->rip_data_source;

	my $command = "tcprobe -i $dvd_device >/dev/null 2>&1";

	system ($command);
	
	if ( $? ) {
		croak "No DVD found at $dvd_device.";
	}
	
	1;
}

sub dvd_is_mounted {
	my $self = shift;
	
	return 1 if $self->rip_mode eq 'dvd_image';
	
	my $dvd_mount_point = $self->config('dvd_mount_point');

	return 1 if -d "$dvd_mount_point/video_ts";
	return 1 if -d "$dvd_mount_point/VIDEO_TS";
	return;
}

sub mount_dvd {
	my $self = shift;
	
	return 1 if $self->rip_mode eq 'dvd_image';

	my $dvd_mount_point = $self->config('dvd_mount_point');

	$self->log ("Mounting DVD at $dvd_mount_point");

	my $mount = qx[ mount $dvd_mount_point 2>&1 && echo DVDRIP_SUCCESS ];

	$mount =~ s/\s$//;

	croak "msg:Failed to mount DVD at $dvd_mount_point ($mount)"
		if $mount !~ /DVDRIP_SUCCESS/;

	1;
}

sub umount_dvd {
	my $self = shift;
	
	return 1 if $self->rip_mode eq 'dvd_image';

	my $dvd_mount_point = $self->config('dvd_mount_point');

	my $mount = qx[ umount $dvd_mount_point 2>&1 ];

	$mount ||= "Ok";

	$self->log ("Umount $dvd_mount_point: $mount");

	1;
}

1;
