package Regexp::Pattern::License::Parts;

use utf8;
use strictures 2;

=head1 NAME

Regexp::Pattern::License::Parts - Regular expressions for licensing sub-parts

=head1 VERSION

Version v3.0.28

=cut

our $VERSION = version->declare("v3.0.28");

=head1 SYNOPSIS

 use Regexp::Pattern; # exports re()
 my $re = re("License::Parts::mit_advertising");

=head1 DESCRIPTION

L<Regexp::Pattern::License::Parts> provides a hash of regular expression patterns
related to sub-parts of legal software licenses.

L<Regexp::Pattern> is a convention for organizing reusable regex patterns.

=head1 PATTERNS

=head2 Single part

Patterns each covering a single part, e.g. a clause or a disclaimer phrase.

=over

=item * ad_ack_doc_pkg

=item * ad_ack_doc_sw

=item * ad_always

=item * ad_pub_ack_nosrc

=item * ad_verbatim

=item * asis_expr_warranty

=item * asis_name_sw

=item * asis_sw_by_name

=item * asis_sw_name_discl

=item * asis_sw_warranty

=item * cc

=item * cc_by

=item * cc_cc0

=item * cc_nc

=item * cc_nd

=item * cc_sa

=item * cc_sp

=item * cc_url

=item * cc_url_pd

=item * derive_altermark

=item * derive_no_relicense

=item * derive_share

=item * discl_name_warranties

=item * nopromo_except

=item * nopromo_neither

=item * nopromo_written_prior

=item * note_cpyr_perm

=item * note_cpyr_perm_doc

=item * note_invar

=item * note_marketing

=item * note_repro_notice_cond_discl

=item * note_retain_notice_cond_discl

=item * origin_no_misrepresent

=item * origin_no_misrepresent_ack

=item * permission_granted_agree_fee

=item * permission_use_deal_sw

=item * permission_use_deal_materials

=item * permission_use_fee_agree

=item * permission_person_materials

=item * permission_person_materials

=item * permission_person_sw_doc

=item * use_sw_doc_any

=item * version_later_para

=item * version_later_postfix

=item * version_number

=item * version_prefix

=back

=cut

my $D  = qr/[-–]/;     # dash
my $SD = qr/[ -–]/;    # space or dash

# permission clause
my $granted = qr/Permission is hereby granted/;

# purpose
my $to_use      = qr/to use, copy, modify, and distribute/;
my $to_deal_sw  = qr/to deal in the Software without restriction/;
my $to_deal_mat = qr/to deal in the Materials without restriction/;
my $any_purpose = qr/for any purpose/;
my $redist_src  = qr/Redistributions of source code/;
my $redist_bin  = qr/Redistributions in binary form/;
my $redist_any  = qr/Redistributions of any form whatsoever/;
my $promote     = qr/endorse or promote products derived from this software/;

# text
my $notice       = qr/the above copyright notice/;
my $cond         = qr/this list of conditions/;
my $discl        = qr/the following disclaimer/;
my $no_agreement = qr/without written agreement/;
my $no_permit    = qr/without specific prior written permission/;
my $ack          = qr/the following acknowledge?ment/;
my $product_by   = qr/This product includes software developed by/;

# money
my $no_charge = qr/free of charge/;
my $no_fee    = qr/without fee/;
my $no_fees   = qr/without license or royalty fees/;

# asset
my $sw_doc      = qr/this software and its documentation/;
my $sw_docfiles = qr/this software and associated documentation files/;
my $doc         = qr/the documentation/;
my $other_mat   = qr/other materials provided with the distribution/;
my $ad_mat
	= qr/All advertising materials mentioning features or use of this software/;

# agent
my $to_person = qr/to any person obtaining a copy/;
my $no_authors
	= qr/Neither the (?:names? .*|authors?) nor the names of(?: (?:its|their|other|any))? contributors/;

# origin
my $origin_no_misrepresent = qr/must not be misrepresented/;
my $origin_ack             = qr/an acknowledgment \(see the following\)/;

# forking
my $derive_altermark
	= qr/Altered source versions must be plainly marked as such/;

# invariant parts
my $invariant_notice = qr/This notice may not be removed or altered/;

our %RE = (
	ad_ack_doc_pkg => {
		caption => 'advertisement (aknowledge, documentation, packages)',
		tags    => ['clause'],
		pat =>
			qr/acknowledgment shall be given in the documentation and software packages that this Software was used/,
	},
	ad_ack_doc_sw => {
		caption => 'advertisement (documentation + source)',
		tags    => ['clause'],
		pat =>
			qr/acknowledgment shall be given in the documentation, materials and software packages that this Software was used/,
	},
	ad_always => {
		caption => 'advertisement',
		tags    => ['clause'],
		pat     => qr/$redist_any must retain $ack[^a-zA-Z]+$product_by/,
	},
	ad_pub_ack_nosrc => {
		caption => 'advertisement (public, aknowledge, no source)',
		tags    => ['clause'],
		pat =>
			qr/publicly documented acknowledgment must be given that this software has been used if no source/,
	},
	ad_verbatim => {
		caption => 'advertisement (verbatim)',
		tags    => ['clause'],
		pat     => qr/$ad_mat must display $ack[^a-zA-Z]+$product_by/,
	},
	asis_expr_warranty => {
		caption => 'as-is (express warranty)',
		tags    => ['clause'],
		pat     => qr/without express or implied warranty/,
	},
	asis_name_sw => {
		caption => 'as-is (name, software)',
		tags    => ['clause'],
		pat     => qr/(?:\S+ ){1,15}PROVIDES? THIS SOFTWARE/,
	},
	asis_sw_by_name => {
		caption => 'as-is (software, by name)',
		tags    => ['clause'],
		pat     => qr/THIS SOFTWARE IS PROVIDED BY/,
	},
	asis_sw_name_discl => {
		caption => 'as-is (software, name disclaims)',
		tags    => ['clause'],
		pat =>
			qr/THE SOFTWARE IS PROVIDED \W*AS\W+IS\W*(?:,?|AND) (?:\S+ ){1,15}DISCLAIMS/,
	},
	asis_sw_warranty => {
		caption => 'as-is (software, warranty)',
		tags    => ['clause'],
		pat => qr/THE SOFTWARE IS PROVIDED \W*AS\W+IS\W* WITHOUT WARRANTY/,
	},
	cc => {
		caption => 'Creative Commons qualifier',
		tags    => ['qualifier'],
		pat     => qr/(?:Creative Commons|CC)/,
	},
	cc_by => {
		caption => 'Creative Commons Attribution component',
		tags    => ['component'],
		pat     => qr/(?:Attribution)/,
	},
	cc_cc0 => {
		caption => 'Creative Commons CC0 component',
		tags    => ['component'],
		pat     => qr/(?:CC0|Zero|0)/,
	},
	cc_nc => {
		caption => 'Creative Commons NonCommercial component',
		tags    => ['component'],
		pat     => qr/(?:Non$SD?Commercial)/,
	},
	cc_nd => {
		caption => 'Creative Commons NoDerivatives component',
		tags    => ['component'],
		pat     => qr/(?:No$SD?Deriv(?:ative)?s)/,
	},
	cc_sa => {
		caption => 'Creative Commons ShareAlike component',
		tags    => ['component'],
		pat     => qr/(?:Share$SD?Alike)/,
	},
	cc_sp => {
		caption => 'Creative Commons Sampling Plus component',
		tags    => ['component'],
		pat     => qr/(?:Sampling$SD?Plus)/,
	},
	cc_url => {
		caption => 'Creative Commons URL',
		tags    => ['url'],
		pat     => qr"(?:(?:https?:?)?(?://)?creativecommons.org/licenses/)",
	},
	cc_url_pd => {
		caption => 'Creative Commons stem (public domain)',
		tags    => ['url'],
		pat => qr"(?:(?:https?:?)?(?://)?creativecommons.org/publicdomain/)",
	},
	derive_altermark => {
		caption => 'deriving (must plainly mark)',
		tags    => ['clause'],
		pat     => qr/$derive_altermark/,
	},
	derive_no_relicense => {
		caption => 'deriving (no relicense)',
		tags    => ['clause'],
		pat     => qr/may not be "?relicensed"?/,
	},
	derive_share => {
		caption => 'deriving (must share source)',
		tags    => ['clause'],
		pat =>
			qr/source code of the altered version of this software available/,
	},
	discl_name_warranties => {
		caption => 'disclaim (name, warranties)',
		tags    => ['clause'],
		pat     => qr/(?:\S+ ){1,15}DISCLAIMS? ALL WARRANTIES/,
	},
	nopromo_except => {
		caption => 'no promotion (except)',
		tags    => ['clause'],
		pat =>
			qr/Except as contained in this notice, the names? (?:\S+ ){1,15}shall not be used in advertising/,
	},
	nopromo_neither => {
		caption => 'no promotion (neither)',
		tags    => ['clause'],
		pat =>
			qr/(?:$no_authors may|The name.*? (?:may|must) not) be used to $promote $no_permit/,
	},
	nopromo_written_prior => {
		caption => 'no promo (advertising)',
		tags    => ['clause'],
		pat     => qr/without specific, written prior permission/,
	},
	note_cpyr_perm => {
		caption => 'notice (copyright + permission)',
		tags    => ['clause'],
		pat =>
			qr/provided that(?:\W*\S\W*)? the above copyright notice(?:\(?s\)?)? and this permission notice appear in all copies/,
	},
	note_cpyr_perm_doc => {
		caption => 'notice (copyright + permission, documentation)',
		tags    => ['clause'],
		pat =>
			qr/both the above copyright notice\(s\) and this permission notice appear in supporting documentation/,
	},
	note_marketing => {
		caption => 'notice (copyright + permission, marketing material)',
		tags    => ['clause'],
		pat =>
			qr/shall be included in all copies of the Software, its documentation and marketing/,
	},
	note_cpyr_perm_subst => {
		caption => 'notices (copyright, permission, substantial portions)',
		tags    => ['clause'],
		pat =>
			qr/The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software/,
	},
	note_invar => {
		caption => 'notice (invariable)',
		tags    => ['clause'],
		pat     => qr/This notice may not be removed or altered/,
	},
	note_repro_notice_cond_discl => {
		caption => 'notice (reproduce, notice + conditions + disclaimer)',
		tags    => ['clause'],
		pat =>
			qr/$redist_bin must reproduce $notice, $cond and $discl in $doc and\/or $other_mat/,
	},
	note_retain_notice_cond_discl => {
		caption => 'notice (retain, notice + conditions + disclaimer)',
		tags    => ['clause'],
		pat     => qr/$redist_src must retain $notice,? $cond and $discl/,
	},
	origin_no_misrepresent => {
		caption => 'origin (no misrepresentation)',
		tags    => ['clause'],
		pat     => qr/$origin_no_misrepresent/,
	},
	origin_no_misrepresent_ack => {
		caption => 'origin (no misrepresentation, acknowledgement)',
		tags    => ['clause'],
		pat     => qr/$origin_no_misrepresent.*, $origin_ack/,
	},
	permission_granted_agree_fee => {
		caption => 'permission (granted, no agreement or fee)',
		tags    => ['clause'],
		pat     => qr/$granted, $no_agreement and $no_fees/,
	},
	permission_use_deal_sw => {
		caption => 'usage (deal, software, any purpose)',
		tags    => ['clause'],
		pat     => qr/$to_deal_sw/,
	},
	permission_use_deal_materials => {
		caption => 'usage (deal, materials, any purpose)',
		tags    => ['clause'],
		pat     => qr/$to_deal_mat/,
	},
	permission_use_fee_agree => {
		caption => 'permission (use, no fee or agreement)',
		tags    => ['clause'],
		pat     => qr/and without a written agreement/,
	},
	permission_person_materials => {
		caption => 'permission (granted, no charge, person, materials)',
		tags    => ['clause'],
		pat =>
			qr/$granted, $no_charge, $to_person of $sw_docfiles \(the "?Materials"?\)/,
	},
	permission_person_sw_doc => {
		caption => 'permission (granted, no charge, person, software)',
		tags    => ['clause'],
		pat =>
			qr/$granted, $no_charge, $to_person of $sw_docfiles \(the "?Software"?\)/,
	},
	right_mod_sublic => {
		caption => 'rights (modify, sublicense)',
		tags    => ['clause'],
		pat =>
			qr/to use, copy, modify, merge, publish, distribute, sublicense, and\/or sell copies of the Software/,
	},
	use_sw_doc_any => {
		caption => 'usage (software and documentation, any purpose)',
		tags    => ['clause'],
		pat     => qr/$to_use $sw_doc $any_purpose/,
	},
	version_later_para => {
		caption => 'version "or later" (paragraph)',
		tags    => ['trait'],
		pat     => qr/Later versions are permitted/,
	},
	version_later_postfix => {
		caption => 'version "or later" (postfix)',
		tags    => ['trait'],
		pat =>
			qr/(?:and|or)(?: ?\(?at your option\)?)?(?: any)? (?:later|newer)(?: version)?/,
	},
	version_number => {
		caption => 'version number',
		tags    => ['component'],
		pat     => qr/\d(?:\.\d+)*/,
	},
	version_prefix => {
		caption => 'version prefix',
		tags    => ['component'],
		pat     => qr/[Vv](?:ersion |ERSION |\.? ?)/,
	},
);

=encoding UTF-8

=head1 AUTHOR

Jonas Smedegaard C<< <dr@jones.dk> >>

=head1 COPYRIGHT AND LICENSE

  Copyright © 2016 Jonas Smedegaard

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 3, or (at your option) any
later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program. If not, see <https://www.gnu.org/licenses/>.

=cut

1;
