aws-sdk-perl
============

Attempt to build a complete AWS SDK in Perl

This project is attempting to build an entire AWS SDK from the information
that is stored in other AWS SDKs. Other AWS SDKs have a "data-driven" approach,
meaning that the definitions for the method calls are stored in a data structure
describing input and output parameters.

The project is actually generating all of it's classes from botocore

Organization
============

build-lib: Contains classes that convert a datastructure into class definitions

auto-lib: Contains the auto-generated classes

lib: Contains roles and classes that the auto-generated classes use to call the API,
sign requests, handle responses, etc.

Generating API
============

Note: This step is not necessary if you want to try out the SDK. We commit in "auto-lib"
the classes generated by pulling other sdks and generating the classes from them. If you're
not developing the SDK, go directly to the "Trying it out" step :)

Execute command make pull-other-sdks This will do a git pull of some official AWS sdks
that are data-driven, and used to generate the SDK.

To generate the API for a given API call: ./gen_classes.pl botocore/botocore/data/aws/API.json.
This will generate file(s) in auto-lib.

To generate all files:

make gen-classes

Perl versions
============

The SDK is targeted at modern Perl versions (in support as per: http://perldoc.perl.org/perlpolicy.html#MAINTENANCE-AND-SUPPORT. 
Old versions may work, but no intention to support them is made. You can always install a modern version of perl with perlbrew or
plenv in a breeze.

Dependencies
============

Dependencies are versioned in a cpanfile. If you have carton, just execute 'carton' in the sdk directory, and all dependencies
will be pulled in automatically into a local library path. After that use 'carton exec ...' to execute your scripts.

Trying it out
============

Each class for each API can be constructed in the following way:

Create a Perl script (myscript.pl)

```
#!/usr/bin/env perl

use Paws;
use Data::Printer;

my $iam = Paws->service('IAM');

my $summary = $iam->GetAccountSummary;
p $summary->SummaryMap;
```

Credentials
============

There are various ways of transmitting credentials to the SDK. By default a 
ProviderChain is used. This chain tries to use the Environment, later the 
File, and later the InstanceProfile credential modules until it finds credentials. 
Other credential providers have to be passed explicitly when requesting a service.

```
my $svc = Paws->service('IAM', credentials => ...CredentialProvider...->new(...));
```

These Credential providers work as follows:

Paws::Credential::Environment

tries to find credentials in the process environment variables
 - Access Key in AWS_ACCESS_KEY or AWS_ACCESS_KEY_ID
 - Secret Key in AWS_SECRET_KEY or AWS_SECRET_ACCESS_KEY
 - Session Token [optional] in AWS_SESSION_TOKEN

Paws::Credential::File

tries to find credentials in ~/.aws/credentials. This file is an ini formatted file
as specified in http://docs.aws.amazon.com/cli/latest/userguide/cli-chap-getting-started.html#cli-config-files.

it will try to find keys aws_access_key_id, aws_secret_access_key and aws_session_token in the default profile,
or in the profile specified by ENV variable AWS_DEFAULT_PROFILE.

Paws::Credential::InstanceProfile

Instance Profiles (Roles) are also supported, so if you're running on an 
instance with a Role, the SDK will automatically pick up credentials. 

Paws::Credential::STS

With the STS Credential provider, you can use temporary federated credentials with 
optionally restricted permissions, obtained via the AWS STS service.

```
my $cred_provider = Paws::Credential::STS->new(
  Name => 'MyName',
  DurationSeconds => 900,
  Policy => '{"Version":"2012-10-17","Statement":[{"Effect": "Allow","Action":["ec2:DescribeInstances"],"Resource":"*"}]}'
);
my $ec2 = Paws->service('EC2', credentials => $cred_provider, region => 'eu-west-1');
$ec2->DescribeInstances;
```

Paws::Credential::AssumeRole

With the AssumeRole provider you can enable cross account access (call other accounts
APIs without needing them to provide you with access keys and secret keys.

```
my $ec2 = Paws->service('EC2',
  region => 'eu-west-1', 
  credentials => Paws::Credential::AssumeRole->new(
    RoleArn => 'arn:aws:iam::123456789012:role/AdminRole',
    RoleSessionName => 'CrossAccountTest',
    ExternalId => 'MyExternalId',
  )
);
# get security groups from account 123456789012
$ec2->DescribeSecurityGroups();
```

Status
================

Don't consider the SDK as "stable" code. There is a lot of experimenting going on. That said,
I'm using it in production, so changes to the way you call APIs, although not guaranteed, are
not prone to change because they are autogenerated. Expect changes around the way you obtain
service classes, transmit credentials, etc. Look at the TODO for expected changes to come.

As of 2015-02 I'm documenting breaking changes in the Changes file. API changes that break stuff
will be documented there. Please read the Changes file before updating your git clone.

Using the SDK in your code
================

Although the code isn't considered stable yet, it works, and more than one person is using it 
already. I recommend you to use a fixed checkout of the repo, via a git submodule, for example. 
Try to keep up-to-date by updating frequently, but read the Changes file before, so you can see 
if there is any before-seen breakage in the process.

Supported AWS Services
================

Just load a class (via Paws->service). (note that there is also a method in Paws that lists all
loadable service classes (the cli uses it, so try out the CLI!). If a service is not supported,
it will warn on construction with an explicit "non supported API" message. Basically all query and
json services are supported. RestXML and RestJSON services are in the coming.

Documentation
================
All services get auto-generated POD documentation. perldoc a file to take a look at the documentation.

CLI utility
================
Paws comes with a command-line utility to exercise the SDK. Just like Paws is the namespace
for the SDK, "paws" (in /bin) is the cli utility. It's quite rudimentary, but think of it as
a quick way to try out calling services. Just call:

```
paws
```

to list all services. If a service isn't supported yet, it will die explicitly advising you that
Paws doesn't support that service yet.

```
paws EC2 --region eu-west-1 DescribeInstances
```

Will output information from your instances

Thanks
================
Luis Alberto Gimenez (@agimenez) for 
 - The git-fu cleaning up the "pull other sdks" code
 - Credential Providers code

Srinvas (@kidambisrinivas) for testing, bug reporting and fixing

juair10 for corrections and testing
