package Vroom::Vroom;
use 5.006001;
use strict;
use warnings;

our $VERSION = '0.15';

use IO::All;
use YAML::XS;
use Class::Field 'field';
use Getopt::Long;
use File::HomeDir;
use Cwd;
use Carp;

field input => 'slides.vroom';
field stream => '';
field ext => '';
field help => 0;
field clean => 0;
field compile => 0;
field run => 0;
field html => 0;
field start => 0;
field digits => 0;
field config => {
    title => 'Untitled Presentation',
    height => 24,
    width => 80,
    list_indent => 10,
    skip => 0,
};

sub new {
    return bless {}, shift;
}

sub usage {
    return <<'...';
    Usage: vroom [options]

    -vroom      - Start slideshow
    -compile    - Generate files
    -clean      - Delete generated files
    -run        - Run a slide file (perl or yaml)
    -html       - Publish as HTML
    -help       - Get help!
...
}

sub vroom {
    my $self = ref($_[0]) ? shift : (shift)->new;

    $self->getOptions;

    if ($self->run) {
        $self->runSlide;
    }
    elsif ($self->clean) {
        $self->cleanAll;
    }
    elsif ($self->compile) {
        $self->makeSlides;
    }
    elsif ($self->start) {
        $self->makeSlides;
        $self->startUp;
    }
    elsif ($self->html) {
        $self->makeHTML;
    }
    elsif ($self->help) {
        warn $self->usage;
    }
    else {
        warn $self->usage;
    }
}

sub getOptions {
    my $self = shift;

    die <<'...' if cwd eq File::HomeDir->my_home;

Don't run vroom in your home directory.

Create a new directory for your slides and run vroom from there.
...

    GetOptions(
        "help" => \$self->{help},
        "clean" => \$self->{clean},
        "compile" => \$self->{compile},
        "run" => \$self->{run},
        "html" => \$self->{html},
        "input=s"  => \$self->{input},
        "vroom"  => \$self->{start},
    ) or die $self->usage;

    do { delete $self->{$_} unless defined $self->{$_} }
        for qw(clean compile input vroom);
}

sub cleanUp {
    my $self = shift;
    unlink(glob "0*");
    unlink('.vimrc');
    unlink('run.yaml');
}

sub cleanAll {
    my $self = shift;
    $self->cleanUp;
    io->dir('html')->rmtree;
}

sub runSlide {
    my $self = shift;
    my $slide = shift @ARGV;
    if ($slide =~ /\.pl$/) {
        exec "perl $slide";
    }
    if ($slide =~ /\.yaml/) {
        my $yaml < io($slide);
        $yaml =~ s/^\s*\n//;
        $yaml =~ s/\n\s*$/\n/;
        if ($yaml =~ /^    \S/) {
            $yaml =~ s/^    //mg;
        }
        $yaml > io('run.yaml');

        exec "perl -MYAML::XS -MData::Dumper -e '\$Data::Dumper::Terse = 1; \$Data::Dumper::Indent = 1; print Dumper YAML::XS::LoadFile(shift)' run.yaml";
    }
}

sub makeSlides {
    my $self = shift;
    $self->cleanUp;
    $self->getInput;
    $self->buildSlides;
    $self->writeVimrc;
}

sub makeHTML {
    my $self = shift;
    require Template::Toolkit::Simple;
    $self->cleanAll;
    $self->makeSlides;
    io('html')->mkdir;
    my @slides = glob('0*');
    for (my $i = 0; $i < @slides; $i++) {
        my $slide = $slides[$i];
        my $prev = ($i > 0) ? $slides[$i - 1] : '';
        my $next = ($i + 1 < @slides) ? $slides[$i + 1] : '';
        my $text = io($slide)->all;
        my $title = $text;
        $text = Template::Toolkit::Simple->new()->render(
            $self->slideTemplate,
            {
                title => "$slide",
                prev => $prev,
                next => $next,
                content => $text,
            }
        );
        io("html/$slide.html")->print($text);
    }

    my $index = [];
    for (my $i = 0; $i < @slides; $i++) {
        my $slide = $slides[$i];
        next if $slide =~ /^\d+[a-z]/;
        my $title = io($slide)->all;
        $title =~ s/.*?((?-s:\S.*)).*/$1/s;
        push @$index, [$slide, $title];
    }

    io("html/index.html")->print(
        Template::Toolkit::Simple->new()->render(
            $self->indexTemplate,
            {
                config => $self->config,
                index => $index,
            }
        )
    );
    $self->cleanUp;
}

sub indexTemplate {
    \ <<'...'
<html>
<head>
<title>[% config.title | html %]</title>
<meta http-equiv="Content-Type" content="text/html; charset=UTF-8">
</head>
<body>
<h1>[% config.title | html %]</h1>
<ul>
[% FOR entry = index -%]
[% slide = entry.shift() -%]
[% title = entry.shift() -%]
<li><a href="[% slide %].html">[% title | html %]</a></li>
[% END -%]
</ul>
<p>This presentation was generated by <a
href="http://ingydotnet.github.com/vroom-pm">Vroom</a>. Use &lt;SPACE&gt; key to go
forward and &lt;BACKSPACE&gt; to go backwards.
</p>
</body>
...
}

sub slideTemplate {
    \ <<'...'
<html>
<head>
<title>[% title | html %]</title>
<meta http-equiv="Content-Type" content="text/html; charset=UTF-8">
<script>
function navigate(e) {
    var keynum = (window.event) // IE
        ? e.keyCode
        : e.which;
    if (keynum == 8) {
[% IF prev -%]
        window.location = "[% prev %]" + ".html";
[% END -%]
        return false;
    }
[% IF next -%]
    if (keynum == 13 || keynum == 32) {
        window.location = "[% next %]" + ".html";
        return false;
    }
[% END -%]
    if (keynum == 73 || keynum == 105) {
        window.location = "index.html";
        return false;
    }
    return true;
}
</script>
</head>
<body onkeypress="return navigate(event)">
<pre>
[%- content | html -%]
</pre>
</body>
...
}

sub getInput {
    my $self = shift;
    my $stream = io($self->input)->all
        or croak "No input provided. Make a file called 'slides.vroom'";
    $self->stream($stream);
}

sub buildSlides {
    my $self = shift;
    my @split = grep length, split /^(----\ *.*)\n/m, $self->stream;
    push @split, '----' if $split[0] =~ /\n/;
    my (@raw_configs, @raw_slides);
    while (@split) {
        my ($config, $slide) = splice(@split, 0, 2);
        $config =~ s/^----\s*(.*?)\s*$/$1/;
        push @raw_configs, $config;
        push @raw_slides, $slide;
    }
    $self->{digits} = int(log(@raw_slides)/log(10)) + 2;

    my $number = 0;

    for my $raw_slide (@raw_slides) {
        my $config = $self->parseSlideConfig(shift @raw_configs);

        next if $config->{skip};

        $raw_slide = $self->applyOptions($raw_slide, $config)
            or next;

        $number++;

        if ($self->config->{skip}) {
            $self->config->{skip}--;
            next;
        }

        $raw_slide = $self->padVertical($raw_slide);

        my @slides;
        my $slide = '';
        for (split /^\+/m, $raw_slide) {
            $slide = '' if $config->{replace};
            $slide .= $_;
            $slide = $self->padVertical($slide)
                if $config->{replace};
            push @slides, $slide;
        }

        my $base_name = $self->formatNumber($number);

        my $suffix = 'a';
        for (my $i = 1; $i <= @slides; $i++) {
            my $slide = $self->padFullScreen($slides[$i - 1]);
            chomp $slide;
            $slide .= "\n";
            if ($slide =~ s/^\ *!(.*\n)//m) {
                $slide .= $1;
            }
            $slide =~ s{^\ *==\ *(.*?)\ *$}
                       {' ' x (($self->config->{width} - length($1)) / 2) . $1}gem;
            my $suf = $suffix++;
            $suf = $suf eq 'a'
                ? ''
                : $i == @slides
                    ? 'z'
                    : $suf;
            io("$base_name$suf" . $self->ext)->print($slide);
        }
    }
}

sub formatNumber {
    my $self = shift;
    my $number = shift;
    my $digits = $self->digits;
    return sprintf "%0${digits}d", $number;
}

sub parseSlideConfig {
    my $self = shift;
    my $string = shift;
    my $config = {};
    for my $option (split /\s*,\s*/, $string) {
        $config->{$1} = 1
            if $option =~ /^(
                config|skip|center|replace|
                perl|ruby|python|js|
                yaml|make|html
            )$/x;
        $config->{indent} = $1
            if $option =~ /i(\d+)/;
    }
    return $config;
}

sub applyOptions {
    my $self = shift;
    my ($slide, $config) = @_;

    $config = {
        %{$self->config},
        %$config,
    };

    if ($config->{config}) {
        $config = {
            %{$self->config},
            %{(YAML::XS::Load($slide))},
        };
        $self->config($config);
        return '';
    }

    if ($config->{center}) {
        $slide =~ s{^(\+?)\ *(.*?)\ *$}
                   {$1 . ' ' x (($self->config->{width} - length($2)) / 2) . $2}gem;
        $slide =~ s{^\s*$}{}gm;
    }

    if (defined $config->{indent}) {
        my $indent = $config->{indent};
        $slide =~ s{^(\+?)}{$1 . ' ' x $indent}gem;
    }
    elsif ($slide =~ /^\+?\*/m) {
        my $indent = $config->{list_indent};
        $slide =~ s{^(\+?)}{$1 . ' ' x $indent}gem;
    }

    my $ext = 
        $config->{perl} ? ".pl" :
        $config->{js} ? ".js" :
        $config->{python} ? ".py" :
        $config->{ruby} ? ".rb" :
        $config->{html} ? ".html" :
        $config->{shell} ? ".sh" :
        $config->{yaml} ? ".yaml" :
        $config->{make} ? ".mk" :
        "";
    $self->ext($ext);

    return $slide;
}

sub padVertical {
    my $self = shift;
    my $slide = shift;
    $slide =~ s/\A\s*\n//;
    $slide =~ s/\n\s*\z//;
    my @lines = split /\n/, $slide;
    my $lines = @lines;
    my $before = int(($self->config->{height} - $lines) / 2) - 1;
    return "\n" x $before . $slide;
}

sub padFullScreen {
    my $self = shift;
    my $slide = shift;
    chomp $slide;
    my @lines = split /\n/, $slide;
    my $lines = @lines;
    my $after = $self->config->{height} - $lines + 1;
    return $slide . "\n" x $after;
}

sub writeVimrc {
    my $self = shift;

    my $home_vroom = File::HomeDir->my_home . "/.vroom/vimrc";
    my $home_vimrc = -e $home_vroom ? io($home_vroom)->all : ''; 

    die <<'...'
The .vimrc in your current directory does not look like vroom created it.

If you are sure it can be overwritten, please delete it yourself this one
time, and rerun vroom. You should not get this message again.

...
    if -e '.vimrc' and io('.vimrc')->getline !~ /Vroom-\d\.\d\d/;

    my $title = "%f         " . $self->config->{title};
    $title =~ s/\s/_/g;
    io(".vimrc")->print(<<"...");
" This .vimrc file was created by Vroom-$VERSION
map <SPACE> :n<CR>:<CR>gg
map <BACKSPACE> :N<CR>:<CR>gg
map R :!vroom -run %<CR>
map Q :q!<CR>
map O :!open <cWORD><CR><CR>
map E :e <cWORD><CR>
map ! G:!open <cWORD><CR><CR>
set laststatus=2
set statusline=$title

" Overrides from $home_vroom
$home_vimrc
...
}

sub startUp {
    exec "vim 0*";
}

=encoding utf8

=head1 NAME

Vroom::Vroom - Slide Shows in Vim

=head1 SYNOPSIS

    > mkdir MySlides    # Make a Directory for Your Slides
    > cd MySlides       # Go In There
    > vim slides.vroom  # Write Some Slides
    > vroom --vroom     # Show Your Slides
    > vroom --html      # Publis Your Slides as HTML

=head1 DESCRIPTION

Ever given a Slide Show and needed to switch over to the shell?

Now you don't ever have to switch again. You're already there.

Vroom lets you create your slides in a single file using a Wiki-like
style, much like Spork and Sporx do. The difference is that your slides
don't compile to HTML or JavaScript or XUL. They get turned into a set
of files that begin with '0', like '03' or '07c' or '05b.pl'.

The slides are named in alpha order. That means you can bring them all
into a Vim session with the command: C<vim 0*>. C<vroom --vroom> does
exactly that.

Vroom creates a file called C<./.vimrc> with helpful key mappings for
navigating a slideshow. See L<KEY MAPPINGS> below.

Please note that you will need the following line in your
C<$HOME/.vimrc> file in order to pick up the local C<.vimrc> file.

    set exrc

Vroom takes advantage of Vim's syntax highlighting. It also lets you run
slides that contain code.

Since Vim is an editor, you can change your slides during the show.

=head1 COMMAND USAGE

Vroom has a few command line options:

=over

=item vroom

Just running vroom will compiles 'slides.vroom' into slide files.

=item vroom --vroom

Compile and start vim show.

=item vroom --clean

Clean up all the compiled  output files.

=back

=head1 INPUT FORMAT

Here is an example slides.vroom file:

    ---- config
    # These are YAML settings for Vroom
    title: My Spiffy Slideshow
    height: 84
    width: 20
    # skip: 12      # Skip 12 slides. Useful when making slides.
    ---- center
    My Presentation

    by Ingy
    ----
    == Stuff I care about:

    * Foo
    +* Bar
    +* Baz
    ---- perl,i10
    # Perl code indented 10 spaces
    use Vroom::Vroom;

    print "Hello World";
    ---- center
    THE END

A line that starts with '==' is a header line. It will be centered.

Lines that begin with a '+' cause vroom to split the slide there,
causing an animation effect.

=head1 CONFIGURATION OPTIONS

Each slide can have one or more configuration options. Options are
a comma separated list that follow the '----' header for a slide.
Like this:

    ---- center
    ---- html
    ---- perl,i20
    ---- config
    ---- skip

=over

=item skip

Ignore the following slide completely.

=item center

Center the contents of the slide.

=item i##

'i' followed by a number means to indent the contents by the number of
characters.

=item perl,ruby,python,js,yaml,make,html

Specifies that the slide is one of those syntaxen, and that the
appropriate file extension will be used, thus causing vim to syntax
highlight the slide.

=item config

The slide is really a yaml configuration. It will not be displayed
in the presentation, but will tell vroom what to do from that point
forward. You can use more than one config slide in your
C<slides.vroom> file.

=back

You can specify the following confguration options in a config slide:

=over

=item title <text>

The title of your presentation.

=item height <number>

The number of lines in the terminal you plan to use when presenting the
show. Used for centering the content.

=item width <number>

The number of columns in the terminal you plan to use when presenting
the show. Used for centering the content.

=item list_indent <number>

Auto detect slides that have lists in them, and indent them by the
specified number of columns.

=back

=head1 KEY MAPPINGS

These are the standard key mappings specified in the local C<.vimrc>.

=over

=item <SPACE>

Advance one slide.

=item <BACKSPACE>

Go back one slide.

=item <R>

Run current slide as Perl.
Turn a YAML slide into Data::Dumper Perl.

=item <Q>

Quit Vroom.

=back

=head1 CUSTOM CONFIGURATION

You can create a file called C<.vroom/vimrc> in your home directory. If
vroom sees this file, it will append it onto every local C<.vimrc> file
it creates.

Use this file to specify your own custom vim settings for all your vroom
presentations.

=head1 NOTE

Vroom is called Vroom but the module is Vroom::Vroom because the
CPAN shell sometimes thinks Vroom is Tim Vroom, and it refuses to
install him.

Use a shell command like this to install Vroom:

    sudo cpan Vroom::Vroom

=head1 AUTHOR

Ingy döt Net <ingy@cpan.org>

=head1 COPYRIGHT

Copyright (c) 2008, 2009. Ingy döt Net.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

See http://www.perl.com/perl/misc/Artistic.html

=cut
