package Net::JBoss::Management;

use 5.010;
use Carp;
use URL::Encode qw /url_encode_utf8/;
use Moo;

with 'Net::JBoss';

=head1 NAME

Net::JBoss::Management - Bindings for JBoss Management API

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.01';


=head1 SYNOPSIS

use Net::JBoss::Management;

 my %con = (
            username                => 'admin',
            password                => 'password',
            server                  => 'jboss1.example.com',
            port                    => 9990, #optional
 );
 
 my $jboss              = Net::JBoss::Management->new(%con);
 
 my $state              = $jboss->get_state();
 my $jvm_usage          = $jboss->get_jvm_usage();
 my $runtime_stats      = $jboss->get_runtime_stats();
 my $runtime_details    = $jboss->get_runtime_details();
 my $server_env         = $jboss->get_server_env();
 my $datasources        = $jboss->get_datasources();
 my $app_status         = $jboss->get_app_status('hawtio.war');
 my $active_session     = $jboss->get_active_sessions('hawtio.war');
 my $pool_stats         = $jboss->get_ds_pool_stats('java:jboss/datasources/jboss_Pool');
 
=head1 Attributes

 Other attributes is also inherited from JBoss.pm
 Check 'perldoc Net::JBoss' for detail
 
 notes :
 ro                     = read only, can be specified only during initialization
 rw                     = read write, user can set this attribute
 rwp                    = read write protected, for internal class
 
 management_url         = (ro) store default management url path

=cut

has 'management_url'    => ( is => 'ro', default => '/management' );

=head1 SUBROUTINES/METHODS

=head2 BUILD

 The Constructor, build logging, call pass_log_obj method
 Build final url
=cut

sub BUILD {
    my $self = shift;
    
    $self->pass_log_obj;

    my $url  = $self->base_url . $self->management_url;
    $self->_set_url($url);
    
    $self->log->debug($url);
}

=head2 get_state

 return server state
 my $state = $jboss->state();
    
=cut

sub get_state {
    my $self = shift;
    
    $self->_set_resource_url('/?operation=attribute&name=server-state&json.pretty');
    $self->log->debug($self->resource_url);
    
    $self->get_api_response();
}

=head2 get_jvm_usage

 return jvm usage
 my $jvm_usage = $jboss->get_jvm_usage();
    
=cut

sub get_jvm_usage {
    my $self = shift;
    
    $self->_set_resource_url('/core-service/platform-mbean/type/memory?operation=resource&include-runtime=true&json.pretty');
    $self->log->debug($self->resource_url);
    
    $self->get_api_response();
}

=head2 get_runtime_stats

 get HTTP connector runtime statistics
 my $runtime_stats = $jboss->runtime_stats();
    
=cut

sub get_runtime_stats {
    my $self = shift;
    
    $self->_set_resource_url('/subsystem/web/connector/http?operation=resource&include-runtime=true&recursive&json.pretty');
    $self->log->debug($self->resource_url);
    
    $self->get_api_response();
}

=head2 get_runtime_details

 get JBoss runtime details
 my $runtime_details = $jboss->get_runtime_details();
    
=cut

sub get_runtime_details {
    my $self = shift;
    
    $self->_set_resource_url('/core-service/platform-mbean/type/runtime?operation=resource&include-runtime=true&json.pretty');
    $self->log->debug($self->resource_url);
    
    $self->get_api_response();
}

=head2 get_server_env

 get JBoss server environment
 my $server_env = $jboss->get_server_env();
    
=cut

sub get_server_env {
    my $self = shift;
    
    $self->_set_resource_url('/core-service/server-environment?operation=resource&include-runtime=true&json.pretty');
    $self->log->debug($self->resource_url);
    
    $self->get_api_response();
}

=head2 get_datasources 

 get data source and driver
 my $datasources = $jboss->get_datasources();
    
=cut

sub get_datasources {
    my $self = shift;
    
    $self->_set_resource_url('/subsystem/datasources/?operation=resource&recursive=true&json.pretty');
    $self->log->debug($self->resource_url);
    
    $self->get_api_response();
}

=head2 get_app_status

 get web application status
 web application name is required
 my $app_status = $jboss->get_app_status('hawtio.war');
    
=cut

sub get_app_status {
    my $self        = shift;
    
    my $app_name    = shift; 
    croak "web application name is required"
        unless $app_name;
    
    $self->_set_resource_url(qq|/deployment/$app_name?operation=attribute&name=status&json.pretty|);
    $self->log->debug($self->resource_url);
    
    $self->get_api_response();
}

=head2 get_active_sessions

 get web application active sessions
 web application name is required
 my $active_session = $jboss->get_active_sessions('hawtio.war');
    
=cut

sub get_active_sessions {
    my $self        = shift;
    
    my $app_name    = shift; 
    croak "web application name is required"
        unless $app_name;
    
    $self->_set_resource_url(qq|/deployment/$app_name/subsystem/web?operation=attribute&name=active-sessions&json.pretty|);
    $self->log->debug($self->resource_url);
    
    $self->get_api_response();
}

=head2 get_ds_pool_stats

 get usage metric of connection pooling of the data source
 my $pool_stats = $jboss->get_ds_pool_stats('java:jboss/datasources/jboss_Pool');

=cut

sub get_ds_pool_stats {
    my $self        = shift;
    my $ds_name     = shift; 
    
    croak "data source name is required"
        unless $ds_name;
    $ds_name = url_encode_utf8($ds_name);
    $self->log->info("encode : $ds_name");
    
    $self->_set_resource_url(qq|/subsystem/datasources/data-source/$ds_name/statistics/pool/?operation=resource&recursive=true&include-runtime=true&json.pretty|);
    $self->log->debug($self->resource_url);
    
    $self->get_api_response();   
}

=head1 AUTHOR

"Heince Kurniawan", C<< <"heince at gmail.com"> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-jboss at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Net::JBoss::Management>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.


=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Net::JBoss
    perldoc Net::JBoss::Management

You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Net::JBoss>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Net::JBoss>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Net::JBoss>

=item * Search CPAN

L<http://search.cpan.org/dist/Net::JBoss/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2015 "Heince Kurniawan".

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of JBoss
