#!/usr/bin/perl
use strict;
use warnings;
use File::Spec;

use lib qw(. t);
BEGIN { require "models.pl"; }

use Test::More;

my $tests	= 11;
my @models	= test_models( qw(data/foaf.xrdf data/about.xrdf) );
plan tests => 1 + ($tests * scalar(@models));

use_ok( 'RDF::Query' );
foreach my $model (@models) {
	print "\n#################################\n";
	print "### Using model: $model\n";
	SKIP: {
		{
			print "# DATATYPE() comparison\n";
			my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
				PREFIX foaf: <http://xmlns.com/foaf/0.1/>
				PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
				PREFIX dc: <http://purl.org/dc/elements/1.1/>
				SELECT ?image ?date
				WHERE {
					?image a foaf:Image ;
						dc:date ?date .
					FILTER ( datatype(?date) = xsd:dateTime )
					
				}
END
			
			my $count	= 0;
			my $stream	= $query->execute( $model );
			while (my $row = $stream->()) {
				my ($image, $dt)	= @{ $row };
				my $url		= $query->bridge->uri_value( $image );
				my $date	= $query->bridge->literal_value( $dt );
				like( $date, qr/^\d{4}-\d\d-\d\dT\d\d:\d\d:\d\d[-+]\d\d:\d\d$/, "valid date: $date" );
				$count++;
			}
			is( $count, 2, "2 photo found with typed date" );
		}

		{
			print "# LANG() comparison\n";
			my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
				PREFIX foaf: <http://xmlns.com/foaf/0.1/>
				PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
				PREFIX dc: <http://purl.org/dc/elements/1.1/>
				SELECT ?person ?name
				WHERE {
					?person a foaf:Person ;
						foaf:name ?name .
					FILTER ( LANG(?name) = "en" )
					
				}
END
			
			my $count	= 0;
			my $stream	= $query->execute( $model );
			while (my $row = $stream->()) {
				my ($p, $n)	= @{ $row };
				my $person	= $query->bridge->as_string( $p );
				my $name	= $query->bridge->literal_value( $n );
				is( $name, 'Gary Peck', "english name: $name" );
				$count++;
			}
			is( $count, 1, "1 person found with an english-typed name" );
		}

		{
			print "# LANGMATCHES()\n";
			my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
				PREFIX foaf: <http://xmlns.com/foaf/0.1/>
				PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
				PREFIX dc: <http://purl.org/dc/elements/1.1/>
				SELECT ?person ?name
				WHERE {
					?person a foaf:Person ;
						foaf:name ?name .
					FILTER ( LANGMATCHES(?name, "en") )
					
				}
END
			
			my $count	= 0;
			my $stream	= $query->execute( $model );
			while (my $row = $stream->()) {
				my ($p, $n)	= @{ $row };
				my $person	= $query->bridge->as_string( $p );
				my $name	= $query->bridge->literal_value( $n );
				is( $name, 'Gary Peck', "english name: $name" );
				$count++;
			}
			is( $count, 1, "1 person found with an english-typed name" );
		}

		{
			my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
				PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
				ASK {
					FILTER ( xsd:dateTime("1994-11-05T08:15:30-05:00") = "1994-11-05T13:15:30Z"^^xsd:dateTime ) .
				}
END
			
			my $count	= 0;
			my $stream	= $query->execute( $model );
			my $ok		= $stream->get_boolean();
			ok( $ok, 'op:dateTime-equal' );
		}
		
		{
			my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
				PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
				ASK {
					FILTER ( xsd:dateTime("1994-11-05T08:15:30-08:00") = "1994-11-05T13:15:30Z"^^xsd:dateTime ) .
				}
END
			
			my $count	= 0;
			my $stream	= $query->execute( $model );
			my $ok		= $stream->get_boolean();
			ok( not($ok), 'not op:dateTime-equal' );
		}
		
		{
			my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
				PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
				ASK {
					FILTER ( "1995-11-05"^^xsd:dateTime > "1994-11-05T13:15:30Z"^^xsd:dateTime ) .
				}
END
			
			my $count	= 0;
			my $stream	= $query->execute( $model );
			my $ok		= $stream->get_boolean();
			ok( $ok, 'dateTime-greater-than' );
		}
		
		{
			my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
				PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
				ASK {
					FILTER ( "1995-11-05"^^xsd:dateTime <= "1994-11-05T13:15:30Z"^^xsd:dateTime ) .
				}
END
			
			my $count	= 0;
			my $stream	= $query->execute( $model );
			my $ok		= $stream->get_boolean();
			ok( not($ok), 'not dateTime-less-than-or-equal' );
		}
		
	}
}
