package Prty::ContentProcessor::Type;
use base qw/Prty::Section::Object/;

use strict;
use warnings;

our $VERSION = 1.097;

use Prty::Perl;

# -----------------------------------------------------------------------------

=encoding utf8

=head1 NAME

Prty::ContentProcessor::Type - Basisklasse für Entitäts-Typen

=head1 BASE CLASS

L<Prty::Section::Object>

=head1 DESCRIPTION

Diese Klasse ist die Basisklasse für alle Plugin-Klassen, die im
ContentProcessor mit registerType() definiert werden.

Eine Plugin-Klasse wird in folgenden Schritten definiert

=head2 Prüfung Abschnittsattribute

Die zulässigen Abschnittsattribute werden in der Plugin-Klasse per

    our @Attributs = qw/
        <Liste der Attributnamen>
    /;

definiert.

=head1 METHODS

=head2 Erzeugung

=head3 create() - Überführe Abschnitts-Objekt in Entität

=head4 Synopsis

    $ent = $class->create($cop,$plg,$sec);

=cut

# -----------------------------------------------------------------------------

sub create {
    my ($class,$sec,$cop,$plg) = splice @_,0,4;
    # @_: @keyVal

    # Abschnitts-Attribute prüfen
    # FIXME: Ob Content erlaubt/verboten ist, konfigurierbar machen
    $sec->validate(1,scalar $class->attributes);
        
    $sec->set(
        processor=>$cop,
        plugin=>$plg,
        testable=>0,
        # memoize
        name=>undef,
        entityId=>undef,
        # Subklassen-Attribute
        @_,
    );
    $sec->weaken('processor');
    $sec->weaken('plugin');
    
    return bless $sec,$class;
}

# -----------------------------------------------------------------------------

=head2 Sub-Abschnitte

=head3 addSubSection() - Füge Sub-Abschnitt hinzu

=head4 Synopsis

    $ent->addSubSection($sec);

=cut

# -----------------------------------------------------------------------------

sub addSubSection {
    my ($self,$sec) = @_;
    return;
}

# -----------------------------------------------------------------------------

=head2 Objektmethoden

=head3 attributes() - Liste der zulssigen Abschnitts-Attribute

=head4 Synopsis

    @attributes | $attributeA = $class->attributes;

=cut

# -----------------------------------------------------------------------------

sub attributes {
    my $class = shift;

    my $a = Prty::Perl->getVar($class,'@','AllAttributes');
    if (!$a) {
        my @attributes;
        for my $class ($class,Prty::Perl->baseClassesISA($class)) {
            my $a = Prty::Perl->getVar($class,'@','Attributes');
            if ($a) {
                push @attributes,@$a;
            }
        }
        @attributes = sort @attributes;
        $a = Prty::Perl->setVar($class,'@','AllAttributes',\@attributes);

        # Gib die Liste der Attribute der Entity-Klassen aus
        # warn "$class - ",join(', ',@$a),"\n";
    }

    return wantarray? @$a: $a;
}

# -----------------------------------------------------------------------------

=head3 entityId() - Eindeutiger Entitätsbezeichner

=head4 Synopsis

    $path = $ent->entityId;

=cut

# -----------------------------------------------------------------------------

sub entityId {
    my $self = shift;

    return $self->memoize('entityId',sub {
        my ($self,$key) = @_;

        # Abschnittytyp, z.B. 'Class'
        my $entityId = $self->type;

        # Abschnittskriterien, z.B. 'Perl' von Language=>'Perl'
    
        my $a = $self->plugin->keyValA;
        for (my $i = 0; $i < @$a; $i += 2) {
            $entityId .= '/'.$a->[$i+1];
        }

        # Entitäts-Name (Pflichtangabe)
        $entityId .= '/'.$self->name;

        return $entityId;
    });
}

# -----------------------------------------------------------------------------

=head3 files() - Liste der Ausgebadateien

=head4 Synopsis

    @files = $ent->files;

=cut

# -----------------------------------------------------------------------------

sub files {
    my $self = shift;
    return;
}

# -----------------------------------------------------------------------------

=head3 name() - Name der Entität

=head4 Synopsis

    $name = $ent->name;

=head4 Description

Liefere den Namen der Entität. Dies ist der Wert
des Attributs C<Name:>, bereinigt um Besonderheiten:

=over 2

=item *

ein Sigil am Namensanfang (z.B. C<@@>) wird entfernt

=item *

Gleichheitszeichen (C<=>) innerhalb des Namens (z.B. bei Klassen)
werden durch einen Slash (C</>) ersetzt

=back

=cut

# -----------------------------------------------------------------------------

sub name {
    my $self = shift;

    return $self->memoize('name',sub {
        my ($self,$key) = @_;
        
        my ($name) = $self->get('Name');
        if (!$name) {
            $self->throw;
        }
        $name =~ s/^\W+//; # Sigil entfernen
        $name =~ s|=|/|g;

        return $name;
    });
}

# -----------------------------------------------------------------------------

=head3 pureCode() - Quelltext ohne Kommentare und Inline-Doku

=head4 Synopsis

    $str = $ent->pureCode;

=cut

# -----------------------------------------------------------------------------

sub pureCode {
    my $self = shift;
    return;
}

# -----------------------------------------------------------------------------

=head1 VERSION

1.097

=head1 AUTHOR

Frank Seitz, L<http://fseitz.de/>

=head1 COPYRIGHT

Copyright (C) 2016 Frank Seitz

=head1 LICENSE

This code is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

# -----------------------------------------------------------------------------

1;

# eof
