#! /usr/bin/perl

use strict;
use warnings;
BEGIN: {
    if ($^O ne 'MSWin32') {
		require Test::NeedsDisplay;
		Test::NeedsDisplay->import();
	}
}
use Digest::SHA();
use MIME::Base64();
use Test::More tests => 107;
use Firefox::Marionette();

umask 0;
diag( "Testing Firefox::Marionette $Firefox::Marionette::VERSION" );
ok(my $firefox = Firefox::Marionette->new(), "Firefox has started in Marionette mode");
ok($firefox->application_type(), "\$firefox->application_type() returns " . $firefox->application_type());
ok($firefox->marionette_protocol(), "\$firefox->marionette_protocol() returns " . $firefox->marionette_protocol());
ok($firefox->quit(), "Firefox has closed");

ok($firefox = Firefox::Marionette->new(capabilities => Firefox::Marionette::Capabilities->new(moz_headless => 1, accept_insecure_certs => 1, page_load_strategy => 'eager', moz_webdriver_click => 1, moz_accessibility_checks => 1)), "Firefox has started in Marionette mode with definable capabilities set to known values");
my $capabilities = $firefox->capabilities();
ok((ref $capabilities) eq 'Firefox::Marionette::Capabilities', "\$firefox->capabilities() returns a Firefox::Marionette::Capabilities object");
ok($capabilities->page_load_strategy() eq 'eager', "\$capabilities->page_load_strategy() is 'eager'");
ok($capabilities->accept_insecure_certs() == 1, "\$capabilities->accept_insecure_certs() is set to true");
ok($capabilities->moz_webdriver_click() == 1, "\$capabilities->moz_webdriver_click() is set to true");
ok($capabilities->moz_accessibility_checks() == 1, "\$capabilities->moz_accessibility_checks() is set to true");
ok($capabilities->moz_headless() == 1, "\$capabilities->moz_headless() is set to true");
ok($firefox->quit(), "Firefox has closed");

ok($firefox = Firefox::Marionette->new(capabilities => Firefox::Marionette::Capabilities->new(moz_headless => 0, accept_insecure_certs => 0, page_load_strategy => 'none', moz_webdriver_click => 0, moz_accessibility_checks => 0)), "Firefox has started in Marionette mode with definable capabilities set to different values");
$capabilities = $firefox->capabilities();
ok((ref $capabilities) eq 'Firefox::Marionette::Capabilities', "\$firefox->capabilities() returns a Firefox::Marionette::Capabilities object");
ok($capabilities->page_load_strategy() eq 'none', "\$capabilities->page_load_strategy() is 'none'");
ok($capabilities->accept_insecure_certs() == 0, "\$capabilities->accept_insecure_certs() is set to false");
ok($capabilities->moz_webdriver_click() == 0, "\$capabilities->moz_webdriver_click() is set to false");
ok($capabilities->moz_accessibility_checks() == 0, "\$capabilities->moz_accessibility_checks() is set to false");
ok(not($capabilities->moz_headless()), "\$capabilities->moz_headless() is set to false");
ok($firefox->quit(), "Firefox has closed");

ok($firefox = Firefox::Marionette->new(), "Firefox has started in Marionette mode without defined capabilities");
ok($firefox->go(URI->new("https://www.w3.org/WAI/UA/TS/html401/cp0101/0101-FRAME-TEST.html")), "https://www.w3.org/WAI/UA/TS/html401/cp0101/0101-FRAME-TEST.html has been loaded");
ok($firefox->window_handle() =~ /^\d+$/, "\$firefox->window_handle() is an integer:" . $firefox->window_handle());
ok($firefox->chrome_window_handle() =~ /^\d+$/, "\$firefox->chrome_window_handle() is an integer:" . $firefox->chrome_window_handle());
foreach my $handle ($firefox->chrome_window_handles()) {
	ok($handle =~ /^\d+$/, "\$firefox->chrome_window_handles() returns a list of integers:" . $handle);
}
TODO: {
	my $screen_orientation = q[];
	eval {
		$screen_orientation = $firefox->screen_orientation();
		ok($screen_orientation, "\$firefox->screen_orientation() is " . $screen_orientation);
	} or do {
		if ($@ =~ /Only supported in Fennec/) {
			local $TODO = "Only supported in Fennec";
			ok($screen_orientation, "\$firefox->screen_orientation() is " . $screen_orientation);
		} else {
			ok($screen_orientation, "\$firefox->screen_orientation() is " . $screen_orientation);
		}
	};
}
ok($firefox->find_element('//frame[@name="target1"]')->switch_to_frame(), "Switched to target1 frame");
ok($firefox->active_frame()->isa('Firefox::Marionette::Element'), "\$firefox->active_frame() returns a Firefox::Marionette::Element object");
ok($firefox->switch_to_parent_frame(), "Switched to parent frame");
ok($firefox->go("https://metacpan.org/"), "metacpan.org has been loaded");
my $uri = $firefox->uri();
ok($uri eq 'https://metacpan.org/', "\$firefox->uri() is equal to https://metacpan.org/:$uri");
ok($firefox->title() =~ /Search/, "metacpan.org has a title containing Search");
my $timeouts = $firefox->timeouts();
ok((ref $timeouts) eq 'Firefox::Marionette::Timeouts', "\$firefox->timeouts() returns a Firefox::Marionette::Timeouts object");
ok($timeouts->page_load() =~ /^\d+$/, "\$timeouts->page_load() is an integer");
ok($timeouts->script() =~ /^\d+$/, "\$timeouts->script() is an integer");
ok($timeouts->implicit() =~ /^\d+$/, "\$timeouts->implicit() is an integer");
SKIP: {
	local $SIG{ALRM} = sub { die "alarm\n" };
	alarm 10;
	if ((exists $ENV{XAUTHORITY}) && (defined $ENV{XAUTHORITY}) && ($ENV{XAUTHORITY} =~ /xvfb/smxi)) {
		skip("Unable to change firefox screen size when xvfb is running", 3);	
	}
	ok($firefox->full_screen(), "\$firefox->full_screen()");
	if ($capabilities->moz_headless()) {
		skip("Unable to minimise/maximise firefox screen size when operating in headless mode", 2);	
	}
	ok($firefox->minimise(), "\$firefox->minimise()");
	ok($firefox->maximise(), "\$firefox->maximise()");
}
alarm 0;
ok($firefox->context('chrome') eq 'content', "Initial context of the browser is 'content'");
ok($firefox->context('content') eq 'chrome', "Changed context of the browser is 'chrome'");
ok($firefox->page_source() =~ /lucky/smx, "metacpan.org contains the phrase 'lucky' in page source");
ok($firefox->refresh(), "\$firefox->refresh()");
my $element = $firefox->active_element();
ok($element, "\$firefox->active_element() returns an element");
ok(not(defined $firefox->active_frame()), "\$firefox->active_frame() is undefined for " . $firefox->uri());
ok($firefox->find_element('//input[@id="search-input"]')->send_keys('Test::More'), "Sent 'Test::More' to the 'search-input' field directly to the element");
ok($firefox->find_element('//input[@id="search-input"]')->clear(), "Clearing the element directly");
foreach my $element ($firefox->find_elements('//input[@id="search-input"]')) {
	ok($firefox->send_keys($element, 'Test::More'), "Sent 'Test::More' to the 'search-input' field via the browser");
	last;
}
my $text = $firefox->find_element('//button[@name="lucky"]')->text();
ok($text, "Read '$text' directly from 'Lucky' button");
my $tag_name = $firefox->find_element('//button[@name="lucky"]')->tag_name();
ok($tag_name, "'Lucky' button has a tag name of '$tag_name'");
my $rect = $firefox->find_element('//button[@name="lucky"]')->rect();
ok($rect->pos_x() =~ /^\d+([.]\d+)?$/, "'Lucky' button has a X position of " . $rect->pos_x());
ok($rect->pos_y() =~ /^\d+([.]\d+)?$/, "'Lucky' button has a Y position of " . $rect->pos_y());
ok($rect->width() =~ /^\d+([.]\d+)?$/, "'Lucky' button has a width of " . $rect->width());
ok($rect->height() =~ /^\d+([.]\d+)?$/, "'Lucky' button has a height of " . $rect->height());
ok(((scalar $firefox->cookies()) > 0), "\$firefox->cookies() shows cookies on " . $firefox->uri());
ok($firefox->delete_cookies() && ((scalar $firefox->cookies()) == 0), "\$firefox->delete_cookies() clears all cookies");
my $handle = $firefox->selfie();
$handle->read(my $buffer, 20);
ok($buffer =~ /^\x89\x50\x4E\x47\x0D\x0A\x1A\x0A/smx, "\$firefox->selfie() returns a PNG file");
$handle = $firefox->find_element('//button[@name="lucky"]')->selfie();
$handle->read($buffer, 20);
ok($buffer =~ /^\x89\x50\x4E\x47\x0D\x0A\x1A\x0A/smx, "\$firefox->find_element('//button[\@name=\"lucky\"]')->selfie() returns a PNG file");
my $actual_digest = $firefox->selfie(hash => 1, highlights => [ $firefox->find_element('//button[@name="lucky"]') ]);
ok($actual_digest =~ /^[a-f0-9]+$/smx, "\$firefox->selfie(hash => 1, highlights => [ \$firefox->find_element('//button[\@name=\"lucky\"]') ]) returns a hex encoded SHA256 digest");
$handle = $firefox->selfie(highlights => [ $firefox->find_element('//button[@name="lucky"]') ]);
$handle->read($buffer, 20);
ok($buffer =~ /^\x89\x50\x4E\x47\x0D\x0A\x1A\x0A/smx, "\$firefox->selfie(highlights => [ \$firefox->find_element('//button[\@name=\"lucky\"]') ]) returns a PNG file");
$handle->seek(0,0) or die "Failed to seek:$!";
$handle->read($buffer, 1_000_000) or die "Failed to read:$!";
my $correct_digest = Digest::SHA::sha256_hex(MIME::Base64::encode_base64($buffer, q[]));
TODO: {
	local $TODO = "Sometimes the digests do not equal each other";
	ok($actual_digest eq $correct_digest, "\$firefox->selfie(hash => 1, highlights => [ \$firefox->find_element('//button[\@name=\"lucky\"]') ]) returns the correct hex encoded SHA256 hash of the base64 encoded image");
}
foreach my $element ($firefox->find_elements('//a[contains(text(), "API")]')) {
	ok($firefox->click($element), "Clicked the API link");
	last;
}
my @cookies = $firefox->cookies();
ok($cookies[0]->name() =~ /\w/, "The first cookie name is '" . $cookies[0]->name() . "'");
ok($cookies[0]->value() =~ /\w/, "The first cookie value is '" . $cookies[0]->value() . "'");
ok($cookies[0]->expiry() =~ /^\d+$/, "The first cookie name has an integer expiry date of '" . $cookies[0]->expiry() . "'");
ok($cookies[0]->http_only() =~ /^[01]$/, "The first cookie httpOnly flag is a boolean set to '" . $cookies[0]->http_only() . "'");
ok($cookies[0]->secure() =~ /^[01]$/, "The first cookie secure flag is a boolean set to '" . $cookies[0]->secure() . "'");
ok($cookies[0]->path() =~ /\S/, "The first cookie path is a string set to '" . $cookies[0]->path() . "'");
ok($cookies[0]->domain() =~ /^[\w\-.]+$/, "The first cookie domain is a domain set to '" . $cookies[0]->domain() . "'");
my $original_number_of_cookies = scalar @cookies;
ok(($original_number_of_cookies > 1) && ((ref $cookies[0]) eq 'Firefox::Marionette::Cookie'), "\$firefox->cookies() returns more than 1 cookie on " . $firefox->uri());
ok($firefox->delete_cookie($cookies[0]->name()), "\$firefox->delete_cookie('" . $cookies[0]->name() . "') deletes the specified cookie name");
ok(not(grep { $_->name() eq $cookies[0]->name() } $firefox->cookies()), "List of cookies no longer includes " . $cookies[0]->name());
ok($firefox->back(), "\$firefox->back() goes back one page");
ok($firefox->uri()->host() eq 'metacpan.org', "\$firefox->uri()->host() is equal to metacpan.org:" . $firefox->uri());
ok($firefox->forward(), "\$firefox->forward() goes forward one page");
ok($firefox->uri()->host() eq 'github.com', "\$firefox->uri()->host() is equal to github.com:" . $firefox->uri());
ok($firefox->back(), "\$firefox->back() goes back one page");
ok($firefox->uri()->host() eq 'metacpan.org', "\$firefox->uri()->host() is equal to metacpan.org:" . $firefox->uri());
ok($firefox->script('return window.find("lucky");'), "metacpan.org contains the 'phrase 'lucky' in a 'window.find' javascript command");
my $cookie = Firefox::Marionette::Cookie->new(name => 'BonusCookie', value => 'who really cares about privacy', expiry => time + 500000);
ok($firefox->add_cookie($cookie), "\$firefox->add_cookie() adds a Firefox::Marionette::Cookie without a domain");
foreach my $element ($firefox->find_elements('//button[@name="lucky"]')) {
	ok($firefox->click($element), "Clicked the \"I'm Feeling Lucky\" button");
}
my $alert_text = 'testing alert';
$firefox->script(qq[alert('$alert_text')]);
ok($firefox->alert_text() eq $alert_text, "\$firefox->alert_text() correctly detects alert text");
TODO: {
	local $TODO = $^O eq 'MSWin32' ? "\$firefox->dismiss_alert() not perfect in Win32 yet" : undef;
	my $result;
	eval {
		$result = $firefox->dismiss_alert();
	};
	ok($result, "\$firefox->dismiss_alert() dismisses alert box");
}
TODO: {
	local $TODO = $^O eq 'MSWin32' ? "\$firefox->dismiss_alert() not perfect in Win32 yet" : undef;
	my $result;
	eval {
		$result = $firefox->async_script(qq[prompt("Please enter your name", "Roland Grelewicz");]);
	};
	ok($result, "Started async script containing a prompt");

}
TODO: {
	local $TODO = $^O eq 'MSWin32' ? "\$firefox->dismiss_alert() not perfect in Win32 yet" : undef;
	my $result;
	eval {
		$result = $firefox->send_alert_text("John Cole");
	};
	ok($result, "\$firefox->send_alert_text() sends alert text");
}
TODO: {
	local $TODO = $^O eq 'MSWin32' ? "\$firefox->dismiss_alert() not perfect in Win32 yet" : undef;
	my $result;
	eval {
		$result = $firefox->accept_dialog();
	};
	ok($result, "\$firefox->accept_dialog() accepts dialog box");
}
ok($firefox->current_chrome_window_handle() =~ /^\d+$/, "Returned the current chrome window handle as an integer");
$capabilities = $firefox->capabilities();
ok((ref $capabilities) eq 'Firefox::Marionette::Capabilities', "\$firefox->capabilities() returns a Firefox::Marionette::Capabilities object");
ok($capabilities->page_load_strategy() =~ /^\w+$/, "\$capabilities->page_load_strategy() is a string:" . $capabilities->page_load_strategy());
ok($capabilities->moz_headless() =~ /^(1|0)$/, "\$capabilities->moz_headless() is a boolean:" . $capabilities->moz_headless());
ok($capabilities->accept_insecure_certs() =~ /^(1|0)$/, "\$capabilities->accept_insecure_certs() is a boolean:" . $capabilities->accept_insecure_certs());
ok($capabilities->moz_process_id() =~ /^\d+$/, "\$capabilities->moz_process_id() is an integer:" . $capabilities->moz_process_id());
ok($capabilities->browser_name() =~ /^\w+$/, "\$capabilities->browser_name() is a string:" . $capabilities->browser_name());
ok($capabilities->rotatable() =~ /^(1|0)$/, "\$capabilities->rotatable() is a boolean:" . $capabilities->rotatable());
ok($capabilities->moz_accessibility_checks() =~ /^(1|0)$/, "\$capabilities->moz_accessibility_checks() is a boolean:" . $capabilities->moz_accessibility_checks());
ok((ref $capabilities->timeouts()) eq 'Firefox::Marionette::Timeouts', "\$capabilities->timeouts() returns a Firefox::Marionette::Timeouts object");
ok($capabilities->timeouts()->page_load() =~ /^\d+$/, "\$capabilities->timeouts->page_load() is an integer:" . $capabilities->timeouts()->page_load());
ok($capabilities->timeouts()->script() =~ /^\d+$/, "\$capabilities->timeouts->script() is an integer:" . $capabilities->timeouts()->script());
ok($capabilities->timeouts()->implicit() =~ /^\d+$/, "\$capabilities->timeouts->implicit() is an integer:" . $capabilities->timeouts()->implicit());
ok($capabilities->browser_version() =~ /^\d+[.]\d+[.]\d+$/, "\$capabilities->browser_version() is a major.minor.patch version number:" . $capabilities->browser_version());
ok($capabilities->platform_version() =~ /\d+/, "\$capabilities->platform_version() contains a number:" . $capabilities->platform_version());
ok($capabilities->moz_profile() =~ /firefox_marionette/, "\$capabilities->moz_profile() contains 'firefox_marionette':" . $capabilities->moz_profile());
ok($capabilities->moz_webdriver_click() =~ /^(1|0)$/, "\$capabilities->moz_webdriver_click() is a boolean:" . $capabilities->moz_webdriver_click());
ok($capabilities->platform_name() =~ /\w+/, "\$capabilities->platform_version() contains alpha characters:" . $capabilities->platform_name());
TODO: {
	local $TODO = "\$firefox->dismiss_alert() not perfect in Win32 yet";
	eval {
		$firefox->dismiss_alert();
	};
	ok($@, "Dismiss non-existant alert caused an exception to be thrown");
}
ok($firefox->accept_connections(0), "Refusing future connections");
ok($firefox->quit(), "Quit is ok");
	
