package Firewall::Config::Parser::Hillstone;

#------------------------------------------------------------------------------
# 加载扩展模块方法属性
#------------------------------------------------------------------------------
use Moose;
use namespace::autoclean;

#------------------------------------------------------------------------------
# 加载 Firewall::Config::Element::Hillstone 相关组件
#------------------------------------------------------------------------------
use Firewall::Config::Element::Zone::Hillstone;
use Firewall::Config::Element::Route::Hillstone;
use Firewall::Config::Element::Interface::Hillstone;
use Firewall::Config::Element::Address::Hillstone;
use Firewall::Config::Element::AddressGroup::Hillstone;
use Firewall::Config::Element::Service::Hillstone;
use Firewall::Config::Element::ServiceGroup::Hillstone;
use Firewall::Config::Element::Rule::Hillstone;
use Firewall::Config::Element::StaticNat::Hillstone;
use Firewall::Config::Element::DynamicNat::Hillstone;
use Firewall::Config::Element::Schedule::Hillstone;

#------------------------------------------------------------------------------
# 引用 'Firewall::Config::Parser::Role' 角色，必须实现 parser 方法
#------------------------------------------------------------------------------
with 'Firewall::Config::Parser::Role';

#------------------------------------------------------------------------------
# 具体实现 'Firewall::Config::Parser::Role' 角色 parser 方法
#------------------------------------------------------------------------------
sub parse {
  my $self = shift;

  # 初始化策略计数器
  $self->{ruleNum} = 0;

  # 遍历未解析的配置命令行
  while (my $string = $self->nextUnParsedLine) {
    if    ($self->isZone($string))         { $self->parseZone($string) }
    elsif ($self->isAddress($string))      { $self->parseAddress($string) }
    elsif ($self->isService($string))      { $self->parseService($string) }
    elsif ($self->isServiceGroup($string)) { $self->parseServiceGroup($string) }
    elsif ($self->isRoute($string))        { $self->parseRoute($string) }
    elsif ($self->isInterface($string))    { $self->parseInterface($string) }
    elsif ($self->isSchedule($string))     { $self->parseSchedule($string) }

    # 暂时不支持主备防火墙判断
    # elsif ($self->isActive($string)) {$self->setActive($string)}
    else { $self->ignoreLine }
  }

  # 追加接口路由并解析防火墙策略
  $self->addRouteToInterface;
  $self->addZoneRange;

  # 重头开始解析
  $self->goToHeadLine;
  while (my $string = $self->nextUnParsedLine) {
    if    ($self->isNat($string))  { $self->parseNat($string) }
    elsif ($self->isRule($string)) { $self->parseRule($string) }
    else                           { $self->ignoreLine }
  }

  # 优化快照缓存加载速度
  $self->{config} = "";
}

#------------------------------------------------------------------------------
# isInterface 捕捉接口代码块
#------------------------------------------------------------------------------
sub isInterface {
  my ($self, $string) = @_;
  if ($string =~ /^interface /i) {
    $self->setElementType('interface');
    return 1;
  }
  else {
    $self->setElementType();
  }
}

#------------------------------------------------------------------------------
# getInterface 查询指定接口的相关信息 | 查询特定的签名
#------------------------------------------------------------------------------
sub getInterface {
  my ($self, $name) = @_;
  return $self->getElement('interface', $name);
}

#------------------------------------------------------------------------------
# parseInterface 解析特定接口代码块 | 实现过滤端口扫描
#------------------------------------------------------------------------------
sub parseInterface {
  my ($self, $string) = @_;
  if ($string =~ /^interface\s+(?<name>\S+)/i) {
    my $name = $+{name};

    # 过滤部分端口：vswitch、tunnel和HA
    if ($name !~ /vswitchif|ha|null/i) {
      my $interface;
      unless ($interface = $self->getInterface($name)) {
        $interface = Firewall::Config::Element::Interface::Hillstone->new(name => $name);
        $self->addElement($interface);
      }
      $interface->addConfig("$string\n");

      # 继续解析其余配置
      while ($string = $self->nextUnParsedLine) {
        $interface->addConfig("$string\n");
        if ($string =~ /^exit/) {
          last;
        }
        if ($string =~ /zone\s+"(?<zoneName>\S+)"/) {
          my $zoneName = $+{zoneName};
          $interface->{zoneName} = $zoneName;
          my $zone = $self->getZone($zoneName);
          $zone->addInterface($interface);
        }
        elsif ($string =~ /ip\s+address\s+(?<ip>\S+)\s+(?<mask>\S+)/) {
          $interface->{ipAddress}     = $+{ip};
          $interface->{mask}          = $+{mask};
          $interface->{interfaceType} = 'layer3';

          # 转换点分十进制掩码为纯数字格式
          my $maskNum = Firewall::Utils::Ip->new->changeMaskToNumForm($+{mask});

          # 实例化路由对象
          my $route = Firewall::Config::Element::Route::Hillstone->new(
            network      => $+{ip},
            mask         => $maskNum,
            dstInterface => $name,
            nextHop      => $+{ip},
            type         => "connected",
          );
          my $zoneName = $interface->{zoneName};
          $route->{routeInstance} = $self->getRouteInstance($zoneName);
          $interface->addRoute($route);
          $self->addElement($route);
        }
      }
    }
  }
}

#------------------------------------------------------------------------------
# isZone 判断是否为 Zone 的代码块
#------------------------------------------------------------------------------
sub isZone {
  my ($self, $string) = @_;
  if ($string =~ /^zone "\S+"/i) {
    $self->setElementType('zone');
    return 1;
  }
  else {
    $self->setElementType();
  }
}

#------------------------------------------------------------------------------
# getZone 查找特定 zone
#------------------------------------------------------------------------------
sub getZone {
  my ($self, $name) = @_;
  return $self->getElement('zone', $name);
}

#------------------------------------------------------------------------------
# parseZone 解析 ZONE 代码块
#------------------------------------------------------------------------------
sub parseZone {
  my ($self, $string) = @_;
  if ($string =~ /^zone "(?<name>\S+)"/i) {
    my $name = $+{name};
    my $zone;

    # 查询是否存在网络区域对象，不存在则新建
    unless ($zone = $self->getZone($name)) {
      $zone = Firewall::Config::Element::Zone::Hillstone->new(name => $name,);
      $self->addElement($zone);
    }
    $zone->addConfig("$string\n");

    # 继续解析代码块配置
    while ($string = $self->nextUnParsedLine) {
      $zone->addConfig("$string\n");

      # 跳出代码块标识符
      if ($string =~ /^exit/) {
        last;
      }
      elsif ($string =~ /^\s+vrouter "(?<vRouter>.*)"/) {
        $zone->{routeInstance} = $+{vRouter};
      }
    }
  }
}

#------------------------------------------------------------------------------
# getRouteInstance 查询区域关联的路由作用域
#------------------------------------------------------------------------------
sub getRouteInstance {
  my ($self, $zoneName) = @_;
  my $zone = $self->getZone($zoneName);
  return $zone ? $zone->{routeInstance} // "default" : undef;
}

#------------------------------------------------------------------------------
# isAddress 判断是否为 address 代码块
#------------------------------------------------------------------------------
sub isAddress {
  my ($self, $string) = @_;
  if ($string =~ /^address\s+".*"/i) {
    $self->setElementType('address');
    return 1;
  }
  else {
    $self->setElementType();
  }
}

#------------------------------------------------------------------------------
# getAddress 查询特定的 addressName
#------------------------------------------------------------------------------
sub getAddress {
  my ($self, $name) = @_;
  return $self->getElement('address', $name);
}

#------------------------------------------------------------------------------
# parseAddress 解析地址对象代码块
#------------------------------------------------------------------------------
sub parseAddress {
  my ($self, $string) = @_;

  # 捕捉特定格式的代码块
  if ($string =~ /^address\s+"(?<name>\S+)"/i) {
    my $name = $+{name};

    # 判断是否已存在该地址对象
    my $address;
    unless ($address = $self->getAddress($name)) {
      $address = Firewall::Config::Element::Address::Hillstone->new(addrName => $name);
      $self->addElement($address);
    }
    $address->addConfig("$string\n");

    # 继续解析 address 代码块
    while ($string = $self->nextUnParsedLine) {
      $address->addConfig("$string\n");

      # 代码块结束符
      if ($string =~ /^exit/) {
        last;
      }
      elsif ($string =~ /ip\s+(?<ipMask>\S+)/) {
        $address->addMember({ipmask => $+{ipMask}});
      }
      elsif ($string =~ /range\s+(?<range>(\S+)\s+(\S+))/) {
        $address->addMember({range => $+{range}});
      }
      elsif ($string =~ /member\s+"(?<addName>\S+)"/) {
        if (my $addrObj = $self->getAddress($+{addName})) {
          $address->addMember({addr => $addrObj});
        }
        else {
          $self->warn("解析字串 $string 期间查询地址对象 $+{addName} 异常");
        }
      }
      elsif ($string =~ /reference-zone\s+"(?<zone>\S+)"/) {
        $address->{zone} = $+{zone};
      }
    }
  }
}

#------------------------------------------------------------------------------
# isService 捕捉到 服务端口 代码块
#------------------------------------------------------------------------------
sub isService {
  my ($self, $string) = @_;
  if ($string =~ /^service\s+".*"/ox) {
    $self->setElementType('service');
    return 1;
  }
  else {
    $self->setElementType();
  }
}

#------------------------------------------------------------------------------
# getService 查询特定的 服务端口信息 | 服务端口名称作为签名
#------------------------------------------------------------------------------
sub getService {
  my ($self, $serviceName) = @_;
  return $self->getElement('service', $serviceName);
}

#------------------------------------------------------------------------------
# parseService 解析服务端口代码块
#------------------------------------------------------------------------------
sub parseService {
  my ($self, $string) = @_;

  # 解析服务端口代码块
  if ($string =~ /^service "(?<name>\S+)"/) {
    my $service;
    my $name = $+{name};

    # 查询服务端口对象，不存在则新建
    unless ($service = $self->getService($name)) {
      $service = Firewall::Config::Element::Service::Hillstone->new(srvName => $name);
      $self->addElement($service);
    }
    $service->addConfig("$string\n");

    my %params;
    while ($string = $self->nextUnParsedLine) {
      $service->addConfig("$string\n");

      # 跳出巡检代码块
      if ($string =~ /^exit/) {
        last;
      }

      # 定义具体的服务端口信息
      elsif (
        $string =~ /(?<proto>\S+) \s+
                      dst-port \s+
                      (?<dstPort>\d+(\s+\d+)?)
                      (\s*src-port \s+
                      (?<srcPort>\d+(\s+\d+)?))?
                    /ox
        )
      {
        $params{srvName}  = $name;
        $params{protocol} = $+{proto};
        $params{dstPort}  = $+{dstPort};
        $params{srcPort}  = $+{srcPort} // '1-65535';
        $service->addMeta(%params);
      }
      elsif ($string =~ /icmp/i) {
        $params{srvName}  = $name;
        $params{protocol} = 'icmp';
        $params{dstPort}  = '1-65535';
        $params{srcPort}  = '1-65535';
        $service->addMeta(%params);
      }
      elsif ($string =~ /description "(?<description>.*?)"/) {
        $service->{description} = $+{description};
      }
      else {
        $self->warn("$string 未解析成功");
      }
    }
  }
}

#------------------------------------------------------------------------------
# getPreDefinedService 厂商相关预定义的服务端口
#------------------------------------------------------------------------------
sub getPreDefinedService {
  my ($self, $srvName) = @_;

  # 根据服务端口签名查询预定义的端口信息
  my $sign = Firewall::Config::Element::Service::Hillstone->createSign($srvName);
  return $self->preDefinedService->{$sign};
}

#------------------------------------------------------------------------------
# isServiceGroup 服务端口组代码块
#------------------------------------------------------------------------------
sub isServiceGroup {
  my ($self, $string) = @_;
  if ($string =~ /^servgroup "\S+"/) {
    $self->setElementType('serviceGroup');
    return 1;
  }
  else {
    $self->setElementType();
  }
}

#------------------------------------------------------------------------------
# getServiceGroup 查询指定的服务端口组
#------------------------------------------------------------------------------
sub getServiceGroup {
  my ($self, $srvGroupName) = @_;

  # 根据签名查询 serviceGroup 对象
  return $self->getElement('serviceGroup', $srvGroupName);
}

#------------------------------------------------------------------------------
# parseServiceGroup 解析服务端口代码块
#------------------------------------------------------------------------------
sub parseServiceGroup {
  my ($self, $string) = @_;
  if ($string =~ /^servgroup "(?<name>\S+)"/i) {
    my $serviceGroup;
    my $name = $+{name};

    # 检查是否已定义该对象
    unless ($serviceGroup = $self->getServiceGroup($name)) {
      $serviceGroup = Firewall::Config::Element::ServiceGroup::Hillstone->new(srvGroupName => $name);
      $self->addElement($serviceGroup);
    }
    $serviceGroup->addConfig("$string\n");

    # 解析服务端口组代码块
    while ($string = $self->nextUnParsedLine) {
      $serviceGroup->addConfig("$string\n");

      # 跳出代码解析标识符
      if ($string =~ /^exit/i) {
        last;
      }
      elsif ($string =~ /service "(?<serName>\S+)"/) {
        my $obj = $self->getServiceOrServiceGroupFromSrvGroupMemberName($+{serName});

        # 此处仅打印异常，未作拦截 | TODO
        unless (!!$obj) {
          $self->warn("srvGroup $name Member $+{serName} 既不是 service 不是 pre-defined service 也不是 service Group\n");
        }
        else {
          # 查询到服务端口组成员对象则自动关联到服务端口组
          $serviceGroup->addSrvGroupMember($+{serName}, $obj);
        }
      }
    }
  }
}

#------------------------------------------------------------------------------
# getServiceOrServiceGroupFromSrvGroupMemberName => 查询服务端口信息
#------------------------------------------------------------------------------
sub getServiceOrServiceGroupFromSrvGroupMemberName {
  my ($self, $srvGroupMemberName) = @_;

  # 依次检索预定服务端口、自定义服务端口、服务端口组成员对象
  my $obj = $self->getPreDefinedService($srvGroupMemberName) // $self->getService($srvGroupMemberName)
    // $self->getServiceGroup($srvGroupMemberName);

  # 返回查询结果 | 可能为空对象 undef
  return $obj;
}

#------------------------------------------------------------------------------
# isSchedule 是否计划性任务
#------------------------------------------------------------------------------
sub isSchedule {
  my ($self, $string) = @_;
  if ($string =~ /^schedule\s+"\S+"/i) {
    $self->setElementType('schedule');
    return 1;
  }
  else {
    $self->setElementType();
  }
}

#------------------------------------------------------------------------------
# getSchedule 查询特定的计划任务
#------------------------------------------------------------------------------
sub getSchedule {
  my ($self, $schName) = @_;
  return $self->getElement('schedule', $schName);
}

#------------------------------------------------------------------------------
# parseSchedule 查询指定的服务端口组
#------------------------------------------------------------------------------
sub parseSchedule {
  my ($self, $string) = @_;
  if ($string =~ /^schedule\s+"(?<name>\S+)"/i) {
    my $schedule;
    my %params;
    $params{schName} = $+{name};

    # 解析计划任务代码块
    while ($string = $self->nextUnParsedLine) {

      # 跳出代码块标识符
      if ($string =~ /^exit/) {
        last;
      }
      elsif ($string =~ /absolute\s+(start\s+(?<startDate>\S+\s+\S+)\s+)?(end\s+(?<endDate>\S+\s+\S+))/) {
        $params{schType}   = 'onetime';
        $params{startDate} = $+{startDate} if defined $+{startDate};
        $params{endDate}   = $+{endDate};

        # 实例化计划任务对象
        $schedule = Firewall::Config::Element::Schedule::Hillstone->new(%params);
        $self->addElement($schedule);
        $schedule->addConfig("$string\n");
      }
      elsif ($string =~ /periodic\s+(?<day>\S+((\s+\S+)+?)?)\s+(?<startDate>\d+:\d+)\s+to\s+(?<endDate>\S+)/) {
        $params{schType}   = 'recurring';
        $params{startTime} = $+{startDate} if defined $+{startDate};
        $params{endTime}   = $+{endDate};
        $params{day}       = $+{day};

        # 实例化计划任务对象
        $schedule = Firewall::Config::Element::Schedule::Hillstone->new(%params);
        $self->addElement($schedule);
        $schedule->addConfig("$string\n");
      }
      else {
        $self->warn("字串解析异常 $string");
      }
    }
  }
}

#------------------------------------------------------------------------------
# isRoute 是否为路由的代码块
#------------------------------------------------------------------------------
sub isRoute {
  my ($self, $string) = @_;
  if ($string =~ /^ip vrouter ".*"/i) {
    $self->setElementType('route');
    return 1;
  }
  else {
    $self->setElementType();
  }
}

#------------------------------------------------------------------------------
# isRoute 是否为路由的代码块
#------------------------------------------------------------------------------
sub getRoute {
  my ($self, $type, $network, $mask, $routeInstance) = @_;
  return $self->getElement('route', $type, $network, $mask, $routeInstance);
}

#------------------------------------------------------------------------------
# parseRoute 解析路由的代码块
# 暂时只支持简单的路由解析，更复杂的需要具体适配 https://regex101.com/
#------------------------------------------------------------------------------
sub parseRoute {
  my ($self, $string) = @_;

  # 解析 vrouter 代码块
  if ($string =~ /^ip vrouter "(?<vRouter>.*)"/) {
    my $vRouter = $+{vRouter};
    my %params;

    # 遍历代码块
    while ($string = $self->nextUnParsedLine) {
      if ($string =~ /^exit/) {
        last;
      }
      elsif (
        $string =~ /ip\s+route\s+
                      (?<network>.*?)\s+
                      ("(?<dstInt>.*?)"\s+)?
                      (?<nexthop>(\d+\.){3}\d+)?/ox
        )
      {
        my ($subnet, $mask) = split('/', $+{network});
        $params{network}       = $subnet;
        $params{mask}          = $mask;
        $params{type}          = "static";
        $params{routeInstance} = $vRouter;
        $params{dstInterface}  = $+{dstInt} if !!$+{dstInt};
        $params{nextHop}       = $+{nexthop} // "127.0.0.1";

        # 实例化路由对象
        my $route = Firewall::Config::Element::Route::Hillstone->new(%params);
        $route->addConfig("$string\n");
        $self->addElement($route);
      }
      else {
        $self->warn("字串解析异常 " . $string);
      }
    }
  }
}

#------------------------------------------------------------------------------
# getServiceGroup 查询指定的服务端口组
#------------------------------------------------------------------------------
sub isNat {
  my ($self, $string) = @_;
  if ($string =~ /^nat-policy/i) {
    $self->setElementType('nat');
    return 1;
  }
  else {
    $self->setElementType();
  }
}

#------------------------------------------------------------------------------
# getServiceGroup 查询静态 NAT | natId 签名
#------------------------------------------------------------------------------
sub getStaticNat {
  my ($self, $id) = @_;
  return $self->getElement('staticNat', $id);
}

#------------------------------------------------------------------------------
# getDynamicNat 查询动态 NAT
#------------------------------------------------------------------------------
sub getDynamicNat {
  my ($self, $id, $natDirection) = @_;
  return $self->getElement('dynamicNat', $natDirection, $id);
}

#------------------------------------------------------------------------------
# parseNat 查询指定的服务端口组
#------------------------------------------------------------------------------
sub parseNat {
  my ($self, $string) = @_;

  # 解析地址转换对象代码块
  if ($string =~ /^nat$/i) {

    # 继续解析代码块下其他策略 | 均为单条策略即配置
    while ($string = $self->nextUnParsedLine) {
      if ($string =~ /^exit/) {
        last;
      }

      # 静态地址转换 | 双向地址转换
      if (
        $string =~ /bnatrule \s+ id \s+
                    (?<id>\d+) \s+
                    virtual \s+
                    ((address-book \s+ "(?<natIpBook>\S+)")
                    |
                    (ip \s+ (?<natIp>\S+)))
                    \s+
                    real \s+
                    ((address-book \s+ "(?<realIpBook>\S+)")
                    |
                    (ip \s+ (?<realIp>\S+)))/ox
        )
      {
        my %params;
        $params{config} = ($string);
        $params{id}     = $+{id};
        if (defined $+{natIpBook}) {
          my $natIpObj = $self->getAddress($+{natIpBook});

          # 查询到了地址转换对象 | TODO
          $params{natIp}      = (values %{$natIpObj->members})[0];
          $params{natIpRange} = $natIpObj->range;
        }
        elsif (defined $+{natIp}) {
          $params{natIp} = $+{natIp};
          my ($ip, $mask) = split('/', $+{natIp});
          my $range = Firewall::Utils::Ip->new->getRangeFromIpMask($ip, $mask);
          $params{natIpRange} = $range;
        }
        if (defined $+{realIpBook}) {
          my $realIpObj = $self->getAddress($+{realIpBook});
          $params{realIp}     = (values %{$realIpObj->members})[0];
          $params{natIpRange} = $realIpObj->range;
        }
        elsif (defined $+{realIp}) {
          $params{realIp} = $+{realIp};
          my ($ip, $mask) = split('/', $+{realIp});
          my $range = Firewall::Utils::Ip->new->getRangeFromIpMask($ip, $mask);
          $params{realIpRange} = $range;
        }

        # 实例化静态地址转换对象
        my $staticNat = Firewall::Config::Element::StaticNat::Hillstone->new(%params);
        $self->addElement($staticNat);
        $staticNat->addConfig("$string\n");
      }
      elsif (
        $string =~ /snatrule \s+ id\s+
                          (?<id>\d+) \s+
                          from \s+ "(?<src>\S+)" \s+
                          to \s+ "(?<dst>\S+)" \s+
                          (service \s+ "(?<srv>\S+)" \s+ )?
                          trans-to \s+
                          ((address-book \s+ "(?<natIpBook>\S+)")
                          |
                          (ip \s+ (?<natIp>(\d+\.){3}\d+))
                          |
                          (eif-ip \s+ (?<int>\S+)) \s+ )
                          /ox
        )
      {
        # 绑定正则表达式捕捉的变量
        my %params;
        $params{id}           = $+{id};
        $params{natDirection} = 'source';
        $params{config}       = ($string);
        my ($src, $dst, $srv, $natSrc, $natIp, $natInt) = ($+{src}, $+{dst}, $+{srv}, $+{natIpBook}, $+{natIp}, $+{int});

        # 根据变量绑定执行不同的处理逻辑
        if (defined $src) {
          my $range;
          if ($src =~ /any/i) {
            $range = Firewall::Utils::Set->new(0, 4294967295);
          }
          else {
            my $addr;
            if ($addr = $self->getAddress($src)) {
              $range = $addr->range;
            }
            else {
              my ($ip, $mask) = split('/', $src);
              $range = Firewall::Utils::Ip->new->getRangeFromIpMask($ip, $mask // 32);
            }
          }
          $params{srcIpRange} = $range;
        }
        if (defined $dst) {
          my $range;
          if ($dst =~ /any/i) {
            $range = Firewall::Utils::Set->new(0, 4294967295);
          }
          else {
            my $addr;
            if ($addr = $self->getAddress($dst)) {
              $range = $addr->range;
            }
            else {
              my ($ip, $mask) = split('/', $dst);
              $range = Firewall::Utils::Ip->new->getRangeFromIpMask($ip, $mask // 32);
            }
          }
          $params{dstIpRange} = $range;
        }
        if ($srv && $srv !~ /any/i) {
          my $range = $self->getServiceOrServiceGroupFromSrvGroupMemberName($srv);
          $params{srvRange} = $range->range;
        }
        if (!!$natSrc) {
          my $natAddr = $self->getAddress($natSrc);
          $params{natSrcIpRange} = $natAddr->range;
        }
        elsif (!!$natInt) {
          my ($ip, $mask) = split('/', $natIp);
          my $range = Firewall::Utils::Ip->new->getRangeFromIpMask($ip, $mask // 32);
          $params{natSrcIpRange} = $range;
        }

        # 实例化动态源地址转换对象
        my $dyNat = Firewall::Config::Element::DynamicNat::Hillstone->new(%params);
        $self->addElement($dyNat);
        $dyNat->addConfig("$string\n");

        # 绑定接口地址作为大网地址
        if (!!$natInt) {
          $self->setInterfaceAsNatIp($dyNat);
        }
      }
      elsif (
        $string =~ /dnatrule \s+ id \s+
                    (?<id>\d+) \s+
                    from \s+ "(?<src>\S+)" \s+
                    to \s+ "(?<dst>\S+)" \s+
                    (service \s+ "(?<srv>\S+)" \s+ )?
                    trans-to \s+ "(?<natDst>\S+)" \s+
                    (port \s+ (?<natPort>\d+))?
                  /ox
        )
      {
        my %params;
        $params{id}           = $+{id};
        $params{natDirection} = 'destination';
        $params{config}       = $string;

        # 绑定正则表达式捕捉的变量
        my ($src, $dst, $srv, $natDst, $natPort) = ($+{src}, $+{dst}, $+{srv}, $+{natDst}, $+{natPort});

        # natSrcRange
        if (!!$src) {
          my $range;
          if ($src =~ /any/i) {
            $range = Firewall::Utils::Ip->new->getRangeFromIpMask(0, 4294967295);
          }
          else {
            my $addr;
            if ($addr = $self->getAddress($src)) {
              $range = $addr->range;
            }
            else {
              my ($ip, $mask) = split('/', $src);
              $range = Firewall::Utils::Ip->new->getRangeFromIpMask($ip, $mask // 32);
            }
          }
          $params{srcIpRange} = $range;
        }

        # natDstRange
        if (!!$dst) {
          my $range;
          if ($dst =~ /any/i) {
            $range = Firewall::Utils::Ip->new->getRangeFromIpMask(0, 4294967295);
          }
          else {
            my $addr;
            if ($addr = $self->getAddress($dst)) {
              $range = $addr->range;
            }
            else {
              my ($ip, $mask) = split('/', $dst);
              $range = Firewall::Utils::Ip->new->getRangeFromIpMask($ip, $mask // 32);
            }
          }
          $params{dstIpRange} = $range;
        }

        # natDstPort
        if ($srv and $srv !~ /any/i) {
          my $range = $self->getServiceOrServiceGroupFromSrvGroupMemberName($srv);
          $params{srvRange} = $range->range;
        }

        # natIp
        if (!!$natDst) {
          $params{natDstIp} = $natDst;
          my $natAddr = $self->getAddress($natDst);
          my $range;
          if (!!$natAddr) {
            $range = $natAddr->range;
          }
          else {
            my ($ip, $mask) = split('/', $natDst);
            $range = Firewall::Utils::Ip->new->getRangeFromIpMask($ip, $mask // 32);
          }
          $params{natDstIpRange} = $range;
        }

        if (!!$natPort) {
          $params{natDstPort} = $natPort;
          my $min = $params{srvRange}->mins->[0];

          # 此处作用不明确
          my $mins = ($min & 0xFF0000) + $natPort;
          $params{natSrvRange} = Firewall::Utils::Set->new($mins, $mins);
        }
        my $dyNat = Firewall::Config::Element::DynamicNat::Hillstone->new(%params);
        $self->addElement($dyNat);
        $dyNat->addConfig("$string\n");
      }
      elsif (
        $string =~ /ip \s+ route \s+
                    (source \s+
                    (in-interface \s+ (?<srcInt>\S+) \s+ )?
                    (?<srcAddr>\S+)? \s+ )?
                    ((?<network>\d+\.\d+\S+)\s+)?
                    ((?<dstInt>[a-zA-Z]+\S+)\s+)?
                    (?<nexthop>\d+\.\d+\.\S+)?
                  /ox
        )
      {
        my %params;
        $params{config} = ($string);
        if ($+{srcInt} || $+{srcAddr}) {
          $params{type}         = 'policy';
          $params{srcInterface} = $+{srcInt}  if defined $+{srcInt};
          $params{srcIpmask}    = $+{srcAddr} if defined $+{srcAddr};
        }
        if (!!$+{network}) {
          my ($ip, $mask) = split('/', $+{network});
          $params{network} = $ip;
          $params{mask}    = $mask // 32;
        }
        $params{natInterface} = $+{dstInt} if defined $+{dstInt};
        $params{nextHop}      = $+{nexthop};

        # 实例化路由对象
        my $route = Firewall::Config::Element::Route::Hillstone->new(%params);
        $self->addElement($route);
        $route->addConfig("$string\n");
      }
      else {
        $self->warn("暂不支持解析 " . $string);
      }
    }
  }
}

#------------------------------------------------------------------------------
# addRouteToInterface 关联接口路由
#------------------------------------------------------------------------------
sub addRouteToInterface {
  my $self   = shift;
  my $routes = $self->{elements}{route};

  # 遍历解析到的路由条目
  for my $route (values %{$routes}) {

    # 过滤静态或直连路由
    if ($route->{type} =~ /static/i) {

      # 路由条件绑定的出站接口
      if (my $dstInt = $route->{dstInterface}) {

        # 查询路由下一跳接口，为路由绑定 zone 同时为该接口绑定路由
        my $interface = $self->getInterface($dstInt);
        $route->{zoneName} = $interface->{zoneName};
        $interface->addRoute($route);
      }
      else {
        # 遍历解析到的接口信息
        for my $interface (values $self->{elements}{interface}->%*) {
          if (!!$interface->{ipAddress}) {
            my $addr = $interface->{ipAddress};
            my $mask = $interface->{mask};

            # 实例化接口地址对象
            my $ipRange = Firewall::Utils::Ip->new->getRangeFromIpMask($addr, $mask);

            # 实例化下一跳地址对象
            my $nexthop = Firewall::Utils::Ip->new->getRangeFromIpMask($route->{nextHop});

            # 接口地址包含出站接口集合对象
            if ($ipRange->isContain($nexthop)) {

              # 绑定接口的网络区域和出站接口
              $route->{zoneName}     = $interface->{zoneName};
              $route->{dstInterface} = $interface->{name};

              # 为接口对象绑定路由
              $interface->addRoute($route);
            }
          }
        }
      }
    }
  }
}

#------------------------------------------------------------------------------
# addZoneRange 查询指定的服务端口组
#------------------------------------------------------------------------------
sub addZoneRange {
  my $self = shift;

  # 网络区域内可能存在多个接口，将各个接口地址转换为地址对象并合并到区间集合
  for my $zone (values %{$self->{elements}{zone}}) {
    for my $interface (values %{$zone->{interfaces}}) {

      # 合并接口的集合区间值到网络区域对象
      $zone->{range}->mergeToSet($interface->{range});
    }
  }
}

#------------------------------------------------------------------------------
# isRule 是否为 rule 代码块
#------------------------------------------------------------------------------
sub isRule {
  my ($self, $string) = @_;
  if (
    $string =~ /^rule \s+ id \s+ \d+
                  |
                  policy \s+
                  from \s+ "\S+" \s+
                  to \s+ "\S+"/x
    )
  {
    $self->setElementType('rule');
    return 1;
  }
  else {
    $self->setElementType();
  }
}

#------------------------------------------------------------------------------
# getRule 查询指定规则
#------------------------------------------------------------------------------
sub getRule {
  my ($self, $ruleId) = @_;
  return $self->getElement('rule', $ruleId);
}

#------------------------------------------------------------------------------
# parseRule 解析规则代码块
#------------------------------------------------------------------------------
sub parseRule {
  my ($self, $string) = @_;

  # 命中策略对象代码块
  if ($string =~ /^policy from "\S+" to "\S+"/oxi) {

    # 绑定正则表达式捕捉的变量
    my %params;
    $params{fromZone} = $+{fromZone};
    $params{toZone}   = $+{toZone};

    # 检索代码块
    while ($string = $self->nextUnParsedLine) {

      # 跳出代码解析
      if ($string =~ /^exit/i) {
        last;
      }
      elsif ($string =~ /^rule id (?<id>\d+)/i) {
        $self->_parseRule($string, \%params);
      }
      elsif ($string =~ /^policy from "\S+" to "\S+"/) {
        $self->backtrackLine;
        last;
      }
    }
  }
  elsif ($string =~ /^rule id \d+/) {
    $self->_parseRule($string);
  }
}

#------------------------------------------------------------------------------
# _parseRule 解析策略逻辑
#------------------------------------------------------------------------------
sub _parseRule {
  my ($self, $string, $param) = @_;

  # 解析防火墙策略代码块
  if ($string =~ /^rule id (?<id>\d+)/i) {
    my %params = $param ? %{$param} : ();
    $params{policyId} = $+{id};
    $params{content}  = "$string\n";
    $self->{ruleNum} += 1;

    # 实例化策略对象
    my $rule = Firewall::Config::Element::Rule::Hillstone->new(%params);
    $self->addElement($rule);
    $rule->addConfig("$string\n");

    # 添加查询索引
    my $index = $params{policyId};
    push @{$self->{ruleIndex}{$index}}, $rule;

    # 继续解析 rule 代码块
    while ($string = $self->nextUnParsedLine) {
      $rule->addConfig("$string\n");
      $rule->{content} .= "$string\n";

      # 代码块结束符
      if ($string =~ /exit/i) {
        last;
      }

      # 策略动作
      if ($string =~ /action (?<action>\S+)/) {
        $rule->{action} = $+{action};
      }

      # 命中源网络区域
      if ($string =~ /src-zone "(?<srcZone>\S+)"/) {
        $rule->{fromZone} = $+{srcZone};
      }

      # 命中目的网络区域
      if ($string =~ /dst-zone "(?<dstZone>\S+)"/) {
        $rule->{toZone} = $+{dstZone};
      }

      # 捕捉源地址 | 地址对象
      if ($string =~ /src-addr "(?<srcAddr>\S+)"/) {
        $self->addToRuleSrcAddressGroup($rule, $+{srcAddr}, "addr");
      }

      # 捕捉目的地址 | 地址对象
      if ($string =~ /dst-addr "(?<dstAddr>\S+)"/) {
        $self->addToRuleDstAddressGroup($rule, $+{dstAddr}, "addr");
      }

      # 捕捉服务端口 | 服务端口
      if ($string =~ /service "(?<srv>\S+)"/) {
        $self->addToRuleServiceGroup($rule, $+{srv});
      }

      # 捕捉计划任务 | 计划任务
      if ($string =~ /schedule "(?<schedule>\S+)"/) {
        my $schedule = $self->getSchedule($+{schedule});
        if (defined $schedule) {
          $rule->setSchedule($schedule);
        }
        else {
          $self->warn("schName $+{schedule} 不是 schedule\n");
        }
      }

      # 命中源地址对象 | 地址/掩码
      if ($string =~ /src-ip (?<srcIp>\S+)/) {
        $self->addToRuleSrcAddressGroup($rule, $+{srcIp}, "ipmask");
      }

      # 命中目的地址对象 | 地址/掩码
      if ($string =~ /dst-ip (?<dstIp>\S+)/) {
        $self->addToRuleDstAddressGroup($rule, $+{dstIp}, "ipmask");
      }

      # 命中连续地址对象 | 连续地址
      if ($string =~ /src-range (?<srcRange>\S+ \S+)/) {
        $self->addToRuleSrcAddressGroup($rule, $+{srcRange}, "range");
      }

      # 命中目的地址对象 | 连续地址
      if ($string =~ /dst-range (?<dstRange>\S+ \S+)/) {
        $self->addToRuleDstAddressGroup($rule, $+{dstRange}, "range");
      }

      # 命中目的地址对象
      if ($string =~ /log (?<hasLog>policy-deny|session-start|session-end)/) {
        $rule->{hasLog} = $+{hasLog};
      }

      # 策略是否生效
      if ($string =~ /disable/ox) {
        $rule->{isDisable} = 'disable';
      }

      # 判断是否存在安全区属性
      $rule->{fromZone} = "any" unless exists($rule->{fromZone});
      $rule->{toZone}   = "any" unless exists($rule->{toZone});
    }
  }
}

#------------------------------------------------------------------------------
# _addSrcOrDstAddressGroup 根据入参地址型别添加成员地址
#------------------------------------------------------------------------------
sub _addSrcOrDstAddressGroup {
  my ($self, $addrName, $type) = @_;
  my $name = $addrName;
  my $obj;

  # 绑定厂商
  my $vendor = (split /::/, __PACKAGE__)[-1];
  my $plugin = qq{Firewall::Config::Element::Address::$vendor};

  # 根据入参型别添加成员对象
  if ($type eq 'addr') {
    if ($addrName =~ /^(?:Any|any-ipv4)$/io) {

      # 检查是否已经定义该地址对象
      unless ($obj = $self->getAddress($addrName)) {
        $obj = $plugin->new(addrName => $addrName);
        $obj->addMember({ipmask => '0.0.0.0/0'});
        $self->addElement($obj);
      }
    }
    elsif ($obj = $self->getAddress($addrName)) {
      $obj->{refnum} += 1;
    }
    else {
      $self->warn("srcAddrName $addrName 不是address 也不是 addressGroup");
      return;
    }
  }
  elsif ($type eq 'ipmask') {

    # 实例化匿名地址对象
    $obj = $plugin->new(addrName => $addrName);

    # 添加成员地址
    $obj->addMember({ipmask => $addrName});
  }
  elsif ($type eq 'range') {
    my ($ipmin, $ipmax) = split('\s+|\-', $addrName);

    # 实例化匿名连续地址对象
    $name = qq{$ipmin-$ipmax};
    $obj  = $plugin->new(addrName => $name);

    # 添加成员地址
    $obj->addMember({range => $addrName});
  }

  # 返回计算结果
  return ($name, $obj);
}

#------------------------------------------------------------------------------
# addToRuleSrcAddressGroup 查询指定的服务端口组
#------------------------------------------------------------------------------
sub addToRuleSrcAddressGroup {
  my ($self, $rule, $addrName, $type) = @_;

  # 生成地址对象和策略签名
  my ($name, $obj) = $self->_addSrcOrDstAddressGroup($addrName, $type);

  # 绑定策略源地址成员
  $rule->addSrcAddressMembers($name, $obj);
}

#------------------------------------------------------------------------------
# getServiceGroup 查询指定的服务端口组
#------------------------------------------------------------------------------
sub addToRuleDstAddressGroup {
  my ($self, $rule, $addrName, $type) = @_;

  # 生成地址对象和策略签名
  my ($name, $obj) = $self->_addSrcOrDstAddressGroup($addrName, $type);

  # 添加目的地址成员对象
  $rule->addDstAddressMembers($name, $obj);
}

#------------------------------------------------------------------------------
# getServiceGroup 查询指定的服务端口组
#------------------------------------------------------------------------------
sub addToRuleServiceGroup {
  my ($self, $rule, $srvName) = @_;

  # 可能查询不到数据
  if (my $obj = $self->getServiceOrServiceGroupFromSrvGroupMemberName($srvName)) {

    # 对象应用计数器自增
    $obj->{refnum} += 1;

    # 添加成员对象
    $rule->addServiceMembers($srvName, $obj);
  }
  else {
    $self->warn("srvName $srvName 不是 service、preDefinedService 也不是 serviceGroup");
  }
}

#------------------------------------------------------------------------------
# getInterfaceFromRoute 通过路由查询指定接口信息
#------------------------------------------------------------------------------
sub getInterfaceFromRoute {
  my ($self, $srcSet, $dstSet) = @_;

  # 防火墙快照下路由条目
  my $routes = $self->{elements}{route};

  # 优先查找策略路由是否包含源地址区间
  for my $route (grep { $_->type eq 'policy' } values %{$routes}) {
    if ($route->{srcRange}->isContain($srcSet)) {
      return $route->{dstInterface};
    }
  }

  # 查找静态路由是否包含目的地址区间 | 最长掩码
  for my $route (sort { $b->{mask} <=> $a->{mask} } grep { $_->{type} eq 'static' } values %{$routes}) {

    # 路由区间集合对象
    if ($route->{range}->isContain($dstSet)) {
      return $route->{dstInterface};
    }
  }
}

#------------------------------------------------------------------------------
# getServiceGroup 查询指定的服务端口组
#------------------------------------------------------------------------------
sub setInterfaceAsNatIp {
  my ($self, $dyNat) = @_;

  # 通过路由定位到接口信息
  my $interface = $self->getInterfaceFromRoute($dyNat->{srcIpRange}, $dyNat->{dstIpRange});

  # 查询不到直接返回
  return unless defined $interface;

  # 绑定接口信息
  my $iface = $self->getInterface($interface);
  $dyNat->{natSrcIpRange} = Firewall::Utils::Ip->new->getRangeFromIpMask($iface->{ipAddress});
}

__PACKAGE__->meta->make_immutable;
1;
