package App::WRT::Renderer;

use strict;
use warnings;
use 5.10.0;

use base qw(Exporter);
our @EXPORT_OK = qw(render);

use Data::Dumper;
use File::Basename;
use File::Copy;
use File::Find;
use File::Path qw(make_path);

# TODO:  In general, this should not print anything.  It should
#        either create a log or call a callback with log data, I think.
#        It should, further, probably allow the user to render selectively.

sub render {
  # this is invoked off of $w, so it's passing in $self there:
  my ($wrt) = shift;
  my $entry_dir = $wrt->entry_dir;
  my $publish_dir = $wrt->publish_dir;

  # Insure that publication path exists and is a directory:
  if (-e $publish_dir) {
    unless (-d $publish_dir) {
      die("$publish_dir exists but is not a directory");
    }
  } else {
    my $path_err;
    make_path($publish_dir, { error => \$path_err });
    if (@{ $path_err }) {
      print Dumper($path_err);
      die("Could not create $publish_dir: " . Dumper($path_err));
    }
  }

  # Handle the front page and Atom feed:
  file_put_contents("${publish_dir}/index.html", $wrt->display('new'));
  file_put_contents("${publish_dir}/feed", $wrt->display('feed'));

  # Handle any other paths that aren't derived direct from files:
  my @meta_paths = qw(all);

  my $rendered_count = 0;
  my $copied_count   = 0;
  for my $target (get_source_files($entry_dir), @meta_paths)
  {
    my $path_err;

    # Lowercase and alpanumeric + underscores + dashes, no dots - an entry:
    if ($target =~ $wrt->entrypath_expr) {
      make_path("${publish_dir}/$target", { error => \$path_err });
      print Dumper($path_err) if (@{ $path_err });

      my $rendered = $wrt->display($target);

      my $target_file = "${publish_dir}/$target/index.html";
      say "[write] $target_file " . length($rendered);
      file_put_contents($target_file, $rendered);
      $rendered_count++;
      next;
    }

    # A directory - no-op:
    if (-d "$entry_dir/$target") {
      say "[directory] $entry_dir/$target";
      next;
    }

    # Some other file - a static asset of some kind:
    my $dirname = dirname($target);
    say "[copy] archives/$target -> ${publish_dir}/$target";
    make_path("public/$dirname", { error => \$path_err });
    print Dumper($path_err) if (@{ $path_err });
    copy("$entry_dir/$target", "${publish_dir}/$target");
    $copied_count++;
  }

  say "rendered $rendered_count entries";
  say "copied $copied_count static files";
}

sub file_put_contents {
  my ($file, $contents) = @_;
  open(my $fh, '>', $file)
    or die "Unable to open $file for writing: $!";
  print $fh $contents;
  close $fh;
}

# Collect a list of things to render:
sub get_source_files {
  my ($entry_dir) = @_;

  my @source_files;
  find(
    sub {
      # We skip index files, because they'll be rendered from the dir path:
      return if /index$/;
      if ($File::Find::name =~ m{^ \Q$entry_dir\E / (.*) $}x) {
        my $target = $1;
        push @source_files, $target;
      }
    },
    $entry_dir
  );
  return @source_files;
}
