package DR::SAP::Meta::Types;
BEGIN {
  $DR::SAP::Meta::Types::VERSION = '0.15';
}
BEGIN {
  $DR::SAP::Meta::Types::VERSION = '0.15';
}

# ABSTRACT: Moose types used by the DR::SAP library

use MooseX::Types -declare => [qw(PlatformID Uri Country Currency PaymentMethod PaymentTerms Region AccountGroup)];
use MooseX::Types::Moose qw(Str Object);
use Moose::Util::TypeConstraints;
use namespace::autoclean;
{
	my %account_groups = (
		vendor      => 'ZCLN',
		affiliate   => 'ZAFF',
		third_party => 'Z3RD',
	);
	my %SAP_account_groups = reverse %account_groups;
	enum( AccountGroup, [ keys %account_groups ] );

	sub _translate_account_group_to_SAP {
		my $g = shift;
		return $account_groups{$g};
	}

	sub _translate_account_group_from_SAP {
		my $g = shift;
		return $SAP_account_groups{$g} || "UNKNOWN: $g";
	}
}

my %platform_ids_to_names = (
	AT => 'Atlantic',
	CC => 'CCNow',
	DL => 'Direct lead',
	E5 => 'element5',
	GC => 'Global Commerce',
	GT => 'Global Tech',
	MV => 'eSellerate',
	ON => 'One Network Direct',
	RS => 'RegSoft',
	RT => 'RegNet',
	RW => 'RegNow',
	SW => 'SWReg',
);
my %platform_names_to_ids = map {(my $normalized = lc $platform_ids_to_names{$_}) =~ s/\s//g; ($normalized => $_)} keys %platform_ids_to_names;
enum PlatformID, [keys %platform_ids_to_names];
coerce PlatformID, from Str, via {
	(my $p = lc($_)) =~ s/\s//g;
	return $platform_names_to_ids{$p} || "Unknown Platform: $_";
};

subtype Uri, as Object, where { $_->isa('URI') };
coerce Uri, from Str, via { URI->new($_) };

{
	my %countries = (
		'AD' => q(Andorra),
		'AE' => q(United Arab Emirates),
		'AF' => q(Afghanistan),
		'AG' => q(Antigua and Barbuda),
		'AI' => q(Anguilla),
		'AL' => q(Albania),
		'AM' => q(Armenia),
		'AN' => q(Dutch Antilles),
		'AO' => q(Angola),
		'AQ' => q(Antarctica),
		'AR' => q(Argentina),
		'AS' => q(American Samoa),
		'AT' => q(Austria),
		'AU' => q(Australia),
		'AW' => q(Aruba),
		'AX' => q(Aland Islands),
		'AZ' => q(Azerbaijan),
		'BA' => q(Bosnia and Herzegovina),
		'BB' => q(Barbados),
		'BD' => q(Bangladesh),
		'BE' => q(Belgium),
		'BF' => q(Burkina Faso),
		'BG' => q(Bulgaria),
		'BH' => q(Bahrain),
		'BI' => q(Burundi),
		'BJ' => q(Benin),
		'BL' => q(Blue),
		'BM' => q(Bermuda),
		'BN' => q(Brunei Darussalam),
		'BO' => q(Bolivia),
		'BR' => q(Brazil),
		'BS' => q(Bahamas),
		'BT' => q(Bhutan),
		'BV' => q(Bouvet Islands),
		'BW' => q(Botswana),
		'BY' => q(Belarus),
		'BZ' => q(Belize),
		'CA' => q(Canada),
		'CC' => q(Coconut Islands),
		'CD' => q(Democratic Republic of the Congo),
		'CF' => q(Central African Republic),
		'CG' => q(Republic of the Congo),
		'CH' => q(Switzerland),
		'CI' => q(Cote d'Ivoire),
		'CK' => q(Cook Islands),
		'CL' => q(Chile),
		'CM' => q(Cameroon),
		'CN' => q(China),
		'CO' => q(Colombia),
		'CR' => q(Costa Rica),
		'CS' => q(Serbia and Montenegro),
		'CU' => q(Cuba),
		'CV' => q(Cape Verde),
		'CX' => q(Christmas Islnd),
		'CY' => q(Cyprus),
		'CZ' => q(Czech Republic),
		'DE' => q(Germany),
		'DJ' => q(Djibouti),
		'DK' => q(Denmark),
		'DM' => q(Dominica),
		'DO' => q(Dominican Republic),
		'DZ' => q(Algeria),
		'EC' => q(Ecuador),
		'EE' => q(Estonia),
		'EG' => q(Egypt),
		'EH' => q(West Sahara),
		'ER' => q(Eritrea),
		'ES' => q(Spain),
		'ET' => q(Ethiopia),
		'EU' => q(European Union),
		'FI' => q(Finland),
		'FJ' => q(Fiji),
		'FK' => q(Falkland Islands),
		'FM' => q(Micronesia),
		'FO' => q(Faroe Islands),
		'FR' => q(France),
		'GA' => q(Gabon),
		'GB' => q(United Kingdom),
		'GD' => q(Grenada),
		'GE' => q(Georgia),
		'GF' => q(French Guyana),
		'GG' => q(Guernsey),
		'GH' => q(Ghana),
		'GI' => q(Gibraltar),
		'GL' => q(Greenland),
		'GM' => q(Gambia),
		'GN' => q(Guinea),
		'GP' => q(Guadeloupe),
		'GQ' => q(Equatorial Guinea),
		'GR' => q(Greece),
		'GS' => q(South Georgia and the Southern Sandwich Islands),
		'GT' => q(Guatemala),
		'GU' => q(Guam),
		'GW' => q(Guinea-Bissau),
		'GY' => q(Guyana),
		'HK' => q(Hong Kong),
		'HM' => q(Heard and McDonald Islands),
		'HN' => q(Honduras),
		'HR' => q(Croatia),
		'HT' => q(Haiti),
		'HU' => q(Hungary),
		'ID' => q(Indonesia),
		'IE' => q(Ireland),
		'IL' => q(Israel),
		'IM' => q(Isle of Man),
		'IN' => q(India),
		'IO' => q(British Indian Ocean Territory),
		'IQ' => q(Iraq),
		'IR' => q(Iran),
		'IS' => q(Iceland),
		'IT' => q(Italy),
		'JE' => q(Jersey),
		'JM' => q(Jamaica),
		'JO' => q(Jordan),
		'JP' => q(Japan),
		'KE' => q(Kenya),
		'KG' => q(Kyrgyzstan),
		'KH' => q(Cambodia),
		'KI' => q(Kiribati),
		'KM' => q(Comoros),
		'KN' => q(Saint Kitts and Nevis),
		'KP' => q(North Korea),
		'KR' => q(South Korea),
		'KW' => q(Kuwait),
		'KY' => q(Cayman Islands),
		'KZ' => q(Kazakhstan),
		'LA' => q(Laos),
		'LB' => q(Lebanon),
		'LC' => q(St. Lucia),
		'LI' => q(Liechtenstein),
		'LK' => q(Sri Lanka),
		'LR' => q(Liberia),
		'LS' => q(Lesotho),
		'LT' => q(Lithuania),
		'LU' => q(Luxembourg),
		'LV' => q(Latvia),
		'LY' => q(Libya),
		'MA' => q(Morocco),
		'MC' => q(Monaco),
		'MD' => q(Moldova),
		'MF' => q(Saint Martin),
		'MG' => q(Madagascar),
		'MH' => q(Marshall Islands),
		'MK' => q(Macedonia),
		'ML' => q(Mali),
		'MM' => q(Burma),
		'MN' => q(Mongolia),
		'MO' => q(Macau),
		'MP' => q(North Mariana Islands),
		'MQ' => q(Martinique),
		'MR' => q(Mauretania),
		'MS' => q(Montserrat),
		'MT' => q(Malta),
		'MU' => q(Mauritius),
		'MV' => q(Maldives),
		'MW' => q(Malawi),
		'MX' => q(Mexico),
		'MY' => q(Malaysia),
		'MZ' => q(Mozambique),
		'NA' => q(Namibia),
		'NC' => q(New Caledonia),
		'NE' => q(Niger),
		'NF' => q(Norfolk Islands),
		'NG' => q(Nigeria),
		'NI' => q(Nicaragua),
		'NL' => q(Netherlands),
		'NO' => q(Norway),
		'NP' => q(Nepal),
		'NR' => q(Nauru),
		'NT' => q(NATO),
		'NU' => q(Niue),
		'NZ' => q(New Zealand),
		'OM' => q(Oman),
		'OR' => q(Orange),
		'PA' => q(Panama),
		'PE' => q(Peru),
		'PF' => q(French Polynesia),
		'PG' => q(Papua New Guinea),
		'PH' => q(Philippines),
		'PK' => q(Pakistan),
		'PL' => q(Poland),
		'PM' => q(St. Pierre and Miquelon),
		'PN' => q(Pitcairn Islands),
		'PR' => q(Puerto Rico),
		'PS' => q(Palestine),
		'PT' => q(Portugal),
		'PW' => q(Palau),
		'PY' => q(Paraguay),
		'QA' => q(Qatar),
		'RE' => q(Reunion),
		'RO' => q(Romania),
		'RS' => q(Serbia),
		'RU' => q(Russian Federation),
		'RW' => q(Rwanda),
		'SA' => q(Saudi Arabia),
		'SB' => q(Solomon Islands),
		'SC' => q(Seychelles),
		'SD' => q(Sudan),
		'SE' => q(Sweden),
		'SG' => q(Singapore),
		'SH' => q(Saint Helena),
		'SI' => q(Slovenia),
		'SJ' => q(Svalbard),
		'SK' => q(Slovakia),
		'SL' => q(Sierra Leone),
		'SM' => q(San Marino),
		'SN' => q(Senegal),
		'SO' => q(Somalia),
		'SR' => q(Suriname),
		'ST' => q(Sao Tome and Principe),
		'SV' => q(El Salvador),
		'SY' => q(Syria),
		'SZ' => q(Swaziland),
		'TC' => q(Turks and Caicos Islands),
		'TD' => q(Chad),
		'TG' => q(Togo),
		'TH' => q(Thailand),
		'TJ' => q(Tajikistan),
		'TK' => q(Tokelau Islands),
		'TL' => q(East Timor),
		'TM' => q(Turkmenistan),
		'TN' => q(Tunisia),
		'TO' => q(Tonga),
		'TP' => q(East Timor),
		'TR' => q(Turkey),
		'TT' => q(Trinidad and Tobago),
		'TV' => q(Tuvalu),
		'TW' => q(Taiwan),
		'TZ' => q(Tanzania),
		'UA' => q(Ukraine),
		'UG' => q(Uganda),
		'UM' => q(American Minor Outlying Islands),
		'UN' => q(United Nations),
		'US' => q(United States),
		'UY' => q(Uruguay),
		'UZ' => q(Uzbekistan),
		'VA' => q(Vatican City),
		'VC' => q(St. Vincent and the Grenadines),
		'VE' => q(Venezuela),
		'VG' => q(British Virgin Islands),
		'VI' => q(American Virgin Islands),
		'VN' => q(Vietnam),
		'VU' => q(Vanuatu),
		'WF' => q(Wallis and Futuna Islands),
		'WS' => q(Samoa),
		'YE' => q(Yemen),
		'YT' => q(Mayotte),
		'ZA' => q(South Africa),
		'ZM' => q(Zambia),
		'ZW' => q(Zimbabwe),
	);
	enum Country, [keys %countries];

	sub _get_countries {
		return \%countries;
	}
	sub _get_country_name {
		my $country = shift;
		return $countries{$country};
	}

}

{
	my @currencies = (
		'ADP',  # Andoran peseta
		'AED',  # United Arab Emirates Dirham
		'AFA',  # Afghani (Old)
		'AFN',  # Afghani
		'ALL',  # Albanian Lek
		'AMD',  # Armenian Dram
		'ANG',  # West Indian Guilder
		'AOA',  # Angolanische Kwanza
		'AON',  # Angolan New Kwanza (Old)
		'AOR',  # Angolan Kwanza Reajustado (Old)
		'ARS',  # Argentine Peso
		'ATS',  # Austrian Schilling
		'AUD',  # Australian Dollar
		'AWG',  # Aruban Guilder
		'AZM',  # Azerbaijan Manat
		'BAM',  # Bosnia and Herzegovina Convertible Mark
		'BBD',  # Barbados Dollar
		'BDT',  # Bangladesh Taka
		'BEF',  # Belgian Franc
		'BGN',  # Bulgarian Lev
		'BHD',  # Bahrain Dinar
		'BIF',  # Burundi Franc
		'BMD',  # Bermudan Dollar
		'BND',  # Brunei Dollar
		'BOB',  # Boliviano
		'BRL',  # Brazilian Real
		'BSD',  # Bahaman Dollar
		'BTN',  # Bhutan Ngultrum
		'BWP',  # Botswana Pula
		'BYR',  # Belorussian Ruble 
		'BZD',  # Belize Dollar
		'CAD',  # Canadian Dollar
		'CDF',  # Congolese Franc
		'CFP',  # French Franc (Pacific Islands)
		'CHF',  # Swiss Franc
		'CLP',  # Chilean Peso
		'CNY',  # Chinesische Yuan (international)
		'COP',  # Colombian Peso
		'CRC',  # Costa Rica Colon
		'CSD',  # Serbian Dinar
		'CUP',  # Cuban Peso
		'CVE',  # Cape Verde Escudo
		'CYP',  # Cyprus Pound
		'CZK',  # Czech Krona
		'DEM',  # German Mark
		'DEM3', # (Internal) German Mark (3 dec.places)
		'DJF',  # Djibouti Franc
		'DKK',  # Danish Krone
		'DOP',  # Dominican Peso
		'DZD',  # Algerian Dinar
		'ECS',  # Ecuadorian Sucre (  > USD)
		'EEK',  # Estonian Krone
		'EGP',  # Egyptian Pound
		'ERN',  # Eritrean Nafka
		'ESP',  # Spanish Peseta
		'ETB',  # Ethiopian Birr
		'EUR',  # European Euro
		'FIM',  # Finnish markka
		'FJD',  # Fiji Dollar
		'FKP',  # Falkland Pound
		'FRF',  # French Franc
		'GBP',  # British Pound
		'GEL',  # Georgian Lari
		'GHC',  # Ghanian Cedi
		'GHS',  # Ghanaian Cedi
		'GIP',  # Gibraltar Pound
		'GMD',  # Gambian Dalasi
		'GNF',  # Guinean Franc
		'GRD',  # Greek Drachma
		'GTQ',  # Guatemalan Quetzal
		'GWP',  # Guinea Peso
		'GYD',  # Guyana Dollar
		'HKD',  # Hong Kong Dollar
		'HNL',  # Honduran Lempira
		'HRK',  # Croatian Kuna
		'HTG',  # Haitian Gourde
		'HUF',  # Hungarian Forint
		'IDR',  # Indonesian Rupiah
		'IEP',  # Irish Punt
		'ILS',  # Israeli Scheckel
		'INR',  # Indian Rupee
		'IQD',  # Iraqui Dinar
		'IRR',  # Iranian Rial
		'ISK',  # Iceland Krona
		'ITL',  # Italian Lira
		'JMD',  # Jamaican Dollar
		'JOD',  # Jordanian Dinar
		'JPY',  # Japanese Yen
		'KES',  # Kenyan Shilling
		'KGS',  # Kyrgyzstan Som
		'KHR',  # Cambodian Riel
		'KMF',  # Comoros Franc
		'KPW',  # North Korean Won
		'KRW',  # South Korean Won
		'KWD',  # Kuwaiti Dinar
		'KYD',  # Cayman Dollar
		'KZT',  # Kazakstanian Tenge
		'LAK',  # Laotian Kip
		'LBP',  # Lebanese Pound
		'LKR',  # Sri Lankan Rupee
		'LRD',  # Liberian Dollar
		'LSL',  # Lesotho Loti
		'LTL',  # Lithuanian Lita
		'LUF',  # Luxembourg Franc
		'LVL',  # Latvian Lat
		'LYD',  # Libyan Dinar
		'MAD',  # Moroccan Dirham
		'MDL',  # Moldavian Leu
		'MGA',  # Madagascan Ariary (New)
		'MGF',  # Madagascan Franc (Old
		'MKD',  # Macedonian Denar
		'MMK',  # Myanmar Kyat
		'MNT',  # Mongolian Tugrik
		'MOP',  # Macao Pataca
		'MRO',  # Mauritanian Ouguiya
		'MTL',  # Maltese Lira
		'MUR',  # Mauritian Rupee
		'MVR',  # Maldive Rufiyaa
		'MWK',  # Malawi Kwacha
		'MXN',  # Mexican Pesos
		'MYR',  # Malaysian Ringgit
		'MZM',  # Mozambique Metical
		'NAD',  # Namibian Dollar
		'NGN',  # Nigerian Naira
		'NIO',  # Nicaraguan Cordoba Oro
		'NLG',  # Dutch Guilder
		'NOK',  # Norwegian Krone
		'NPR',  # Nepalese Rupee
		'NZD',  # New Zealand Dollars
		'OMR',  # Omani Rial
		'PAB',  # Panamanian Balboa
		'PEN',  # Peruvian New Sol
		'PGK',  # Papua New Guinea Kina
		'PHP',  # Philippine Peso
		'PKR',  # Pakistani Rupee
		'PLN',  # Polish Zloty (new)
		'PTE',  # Portuguese Escudo
		'PYG',  # Paraguayan Guarani
		'QAR',  # Qatar Rial
		'RMB',  # Chinesische Renminbi (national)
		'ROL',  # Romanian Leu
		'RON',  # Romanian Leu
		'RUB',  # Russian Ruble
		'RWF',  # Rwandan Franc
		'SAR',  # Saudi Riyal
		'SBD',  # Solomon Islands Dollar
		'SCR',  # Seychelles Rupee
		'SDD',  # Sudanese Dinar
		'SDP',  # Sudanese Pound
		'SEK',  # Swedish Krona
		'SGD',  # Singapore Dollar
		'SHP',  # St.Helena Pound
		'SIT',  # Slovenian Tolar
		'SKK',  # Slovakian Krona
		'SLL',  # Sierra Leone Leone
		'SOS',  # Somalian Shilling
		'SRD',  # Surinam Dollar
		'SRG',  # Surinam Guilder (Old)
		'STD',  # Sao Tome / Principe Dobra
		'SVC',  # El Salvador Colon
		'SYP',  # Syrian Pound
		'SZL',  # Swaziland Lilangeni
		'THB',  # Thailand Baht
		'TJR',  # Tajikistani Ruble (Old)
		'TJS',  # Tajikistani Somoni
		'TMM',  # Turkmenistani Manat
		'TND',  # Tunisian Dinar
		'TOP',  # Tongan Pa\'anga
		'TPE',  # Timor Escudo
		'TRL',  # Turkish Lira (Old)
		'TRY',  # Turkish Lira
		'TTD',  # Trinidad and Tobago Dollar
		'TWD',  # New Taiwan Dollar
		'TZS',  # Tanzanian Shilling
		'UAH',  # Ukraine Hryvnia
		'UGX',  # Ugandan Shilling
		'USD',  # United States Dollar
		'USDN', # (Internal) United States Dollar (5 Dec.)
		'UYU',  # Uruguayan Peso (new)
		'UZS',  # Uzbekistan Som
		'VEB',  # Venezuelan Bolivar
		'VEF',  # Venezuelan Bolivar Hard
		'VND',  # Vietnamese Dong
		'VUV',  # Vanuatu Vatu
		'WST',  # Samoan Tala
		'XAF',  # Gabon CFA Franc BEAC
		'XCD',  # East Carribean Dollar
		'XDR',  # IMF Special Drawing Rate
		'XDS',  # St. Christopher Dollar
		'XEU',  # European Currency Unit (E.C.U.)
		'XOF',  # Benin CFA Franc BCEAO
		'XPF',  # CFP Franc
		'YER',  # Yemeni Ryal
		'YUM',  # New Yugoslavian Dinar (Old)
		'ZAR',  # South African Rand
		'ZMK',  # Zambian Kwacha
		'ZRN',  # Zaire (Old)
		'ZWD',  # Zimbabwean Dollar
	);
	enum Currency, \@currencies;
}

{
	my %payment_methods = (
		A => 'Domestic ACH',
		C => 'Check',
		D => 'Drafts',
		E => 'ePassporte',
		F => 'Check File',
		I => 'IACH',
		M => 'Web Money',
		P => 'Paypal',
		S => 'IACH Sepa',
		T => 'Bank Transfer',
		X => 'e4x wires',
		Y => 'e4x drafts',

		0 => 'Wire Transfer', # mapped by SAP to G, H, J, K, V, W
		G => 'Global Wires',
		H => 'Global Wires Fees',
		J => 'Domestic Wires CWT - Wire Feeds',
		K => 'Domestic Wires CWT',
		V => 'Domestic FWT - Wires Fees',
		W => 'Domestic Wires FWT',
	);
	enum PaymentMethod, [ keys %payment_methods ];
}
{
	my %payment_terms = (
		'0001' => 'Due on Receipt',
		'0002' => '14 days 2%, 30 net',
		'0003' => '14 days 3%, 20/2%, 30 net',
		'0004' => 'as of end of month',
		'0005' => 'from the 10th of subs. month',
		'0006' => 'End of month 4%, Mid-month 2%',
		'0007' => '15th/31st subs. month 2%, ...',
		'0007' => '15th/31st subs. month 2%, ...',
		'0008' => '14 days 2%, 30/1,5%, 45 net',
		'0008' => '14 days 2%, 30/1,5%, 45 net',
		'0009' => 'Payable in 3 installments',
		'0012' => 'Due On Receipt',
		'0013' => 'Net Due 30 Days',
		'0014' => '2%, 10D, Due 45 Days',
		'0015' => 'Net Due 15 Days',
		'0016' => 'Net Due 60 Days',
		'0017' => 'Net Due 10 Days',
		'0018' => 'Due 5th of Month',
		'0019' => 'Net Due 90 Days',
		'0020' => 'Net Due 7 Days',
		'0021' => 'Net Due, 2 Weeks',
		'0022' => 'Net Due, 1 Month',
		'0023' => 'Net Due, 3 Weeks',
		'0024' => 'Due 10th of Month',
		'0025' => 'Net Due 19 Days',
		'0026' => 'Net Due 3 Months',
		'0027' => 'Net Due 20 Days',
		'0028' => 'Due 30th of Month',
		'0029' => 'Net Due 3 Days',
		'0030' => 'Net Due 5 days',
		'0031' => 'Net Due 75 Days',
		'0032' => 'Net Due 60 Days',
		'0033' => 'Net Due 45 Days',
		'0034' => '2% 10, Net 30',
		'0035' => '1% 10, Net 45',
		'0036' => '1% 10, Net 30',
		'0037' => 'Net Due 180 Days',
		'0038' => 'Net Due 40 Days',
		'0039' => '10 days 0.5%, 30days  net',
		'0040' => '10 days 0.75%, 30days  net',
		'0041' => '1% 10, Net 35',
		'Z001' => 'Before 5 of the next month',
		'Z002' => 'Before 10 of the next month',
		'Z003' => 'Before 15 of the next month',
		'Z004' => 'Before 20 of the next month',
		'Z005' => 'Before 23 of the next month',
		'Z006' => 'Before 25th of the next month',
		'Z007' => 'Monthly, Pay 10th Second Month (oND Commissions)',
		'Z008' => 'Before 25 in 3 months Due net',
		'Z010' => 'Before 20 for 1-15, Before 5 for 16-31 ;; Due net',
		'Z010' => 'Before 20 for 1-15, Before 5 for 16-31 ;; Due net',
		'Z011' => '2% 10, Net 45 from month end',
		'Z012' => 'Net 30 from End of Month',
		'Z013' => 'Net 45 from End of Month',
		'Z015' => 'Weekly, pay +5 days (pay 12th, 19th, 26th, 5th)',
		'Z016' => 'Weekly, pay +7 days (pay 14th, 21st, 28th, 7th)',
		'Z017' => 'Weekly Pay, Wednesday;; Due Net',
		'Z018' => 'Weekly Pay, Thursday;; Due Net',
		'Z019' => 'Weekly Pay, Friday;; Due Net',
		'Z102' => 'Net Due 2 Days',
		'Z104' => 'Net Due 4 Days',
		'Z106' => 'Net Due 6 Days',
		'Z108' => 'Net Due 8 Days',
		'Z110' => 'Monthly, Pay 30th of Second Month',
		'Z112' => '20th, 30th, Next 10th',
		'Z114' => '22nd, 1st next, Next 12th',
		'Z116' => 'Due on Receipt, 1.00% Discount',
		'Z118' => 'Due on Receipt, .75% Discount',
		'Z120' => 'Due on Receipt, .50% Discount',
	);
	enum PaymentTerms, [ keys %payment_terms ];
}

{
	my (%country_regions, %regions);
	my @data = (
		[ qw(AR 0),   q(Capital Federal) ],
		[ qw(AR 1),   q(Buenos Aires) ],
		[ qw(AR 2),   q(Catamarca) ],
		[ qw(AR 3),   q(Cordoba) ],
		[ qw(AR 4),   q(Corrientes) ],
		[ qw(AR 5),   q(Entre Rios) ],
		[ qw(AR 6),   q(Jujuy) ],
		[ qw(AR 7),   q(Mendoza) ],
		[ qw(AR 8),   q(La Rioja) ],
		[ qw(AR 9),   q(Salta) ],
		[ qw(AR 10),  q(San Juan) ],
		[ qw(AR 11),  q(San Luis) ],
		[ qw(AR 12),  q(Santa Fe) ],
		[ qw(AR 13),  q(Santiago del Estero) ],
		[ qw(AR 14),  q(Tucuman) ],
		[ qw(AR 16),  q(Chaco) ],
		[ qw(AR 17),  q(Chubut) ],
		[ qw(AR 18),  q(Formosa) ],
		[ qw(AR 19),  q(Misiones) ],
		[ qw(AR 20),  q(Neuquen) ],
		[ qw(AR 21),  q(La Pampa) ],
		[ qw(AR 22),  q(Rio Negro) ],
		[ qw(AR 23),  q(Santa Cruz) ],
		[ qw(AR 24),  q(Tierra de Fuego) ],
		[ qw(AT B),   q(Burgenland) ],
		[ qw(AT K),   q(Carinthia) ],
		[ qw(AT NOE), q(Lower Austria) ],
		[ qw(AT OOE), q(Upper Austria) ],
		[ qw(AT S),   q(Salzburg) ],
		[ qw(AT ST),  q(Styria) ],
		[ qw(AT T),   q(Tyrol) ],
		[ qw(AT V),   q(Vorarlberg) ],
		[ qw(AT W),   q(Vienna) ],
		[ qw(AU ACT), q(Aust Capital Terr) ],
		[ qw(AU NSW), q(New South Wales) ],
		[ qw(AU NT),  q(Northern Territory) ],
		[ qw(AU QLD), q(Queensland) ],
		[ qw(AU SA),  q(South Australia) ],
		[ qw(AU TAS), q(Tasmania) ],
		[ qw(AU VIC), q(Victoria) ],
		[ qw(AU WA),  q(Western Australia) ],
		[ qw(BE 1),   q(Antwerp) ],
		[ qw(BE 2),   q(Brabant (Flemish)) ],
		[ qw(BE 3),   q(Hainaut) ],
		[ qw(BE 4),   q(Liege) ],
		[ qw(BE 5),   q(Limburg) ],
		[ qw(BE 6),   q(Luxembourg) ],
		[ qw(BE 7),   q(Namur) ],
		[ qw(BE 8),   q(Oost-Vlaanderen) ],
		[ qw(BE 9),   q(West-Vlaanderen) ],
		[ qw(BE 10),  q(Brabant (Walloon)) ],
		[ qw(BE 11),  q(Brussels (Capital)) ],
		[ qw(BG 1),   q(Burgas) ],
		[ qw(BG 2),   q(Grad Sofiya) ],
		[ qw(BG 3),   q(Khaskovo) ],
		[ qw(BG 4),   q(Lovech) ],
		[ qw(BG 5),   q(Montana) ],
		[ qw(BG 6),   q(Plovdiv) ],
		[ qw(BG 7),   q(Ruse) ],
		[ qw(BG 8),   q(Sofiya) ],
		[ qw(BG 9),   q(Varna) ],
		[ qw(BR AC),  q(Acre) ],
		[ qw(BR AL),  q(Alagoas) ],
		[ qw(BR AM),  q(Amazon) ],
		[ qw(BR AP),  q(Amapa) ],
		[ qw(BR BA),  q(Bahia) ],
		[ qw(BR CE),  q(Ceara) ],
		[ qw(BR DF),  q(Brasilia) ],
		[ qw(BR ES),  q(Espirito Santo) ],
		[ qw(BR GO),  q(Goias) ],
		[ qw(BR MA),  q(Maranhao) ],
		[ qw(BR MG),  q(Minas Gerais) ],
		[ qw(BR MS),  q(Mato Grosso do Sul) ],
		[ qw(BR MT),  q(Mato Grosso) ],
		[ qw(BR PA),  q(Para) ],
		[ qw(BR PB),  q(Paraiba) ],
		[ qw(BR PE),  q(Pernambuco) ],
		[ qw(BR PI),  q(Piaui) ],
		[ qw(BR PR),  q(Parana) ],
		[ qw(BR RJ),  q(Rio de Janeiro) ],
		[ qw(BR RN),  q(Rio Grande do Norte) ],
		[ qw(BR RO),  q(Rondonia) ],
		[ qw(BR RR),  q(Roraima) ],
		[ qw(BR RS),  q(Rio Grande do Sul) ],
		[ qw(BR SC),  q(Santa Catarina) ],
		[ qw(BR SE),  q(Sergipe) ],
		[ qw(BR SP),  q(Sao Paulo) ],
		[ qw(BR TO),  q(Tocantins) ],
		[ qw(BS BS),  q(Bahamas) ],
		[ qw(CA AB),  q(Alberta) ],
		[ qw(CA BC),  q(British Columbia) ],
		[ qw(CA MB),  q(Manitoba) ],
		[ qw(CA NB),  q(New Brunswick) ],
		[ qw(CA NL),  q(Newfoundland & Labr.) ],
		[ qw(CA NS),  q(Nova Scotia) ],
		[ qw(CA NT),  q(Northwest Terr.) ],
		[ qw(CA NU),  q(Nunavut) ],
		[ qw(CA ON),  q(Ontario) ],
		[ qw(CA PE),  q(Prince Edward Island) ],
		[ qw(CA QC),  q(Quebec) ],
		[ qw(CA SK),  q(Saskatchewan) ],
		[ qw(CA YT),  q(Yukon Territory) ],
		[ qw(CH AG),  q(Aargau) ],
		[ qw(CH AI),  q(Inner-Rhoden) ],
		[ qw(CH AR),  q(Ausser-Rhoden) ],
		[ qw(CH BE),  q(Bern) ],
		[ qw(CH BL),  q(Basel Land) ],
		[ qw(CH BS),  q(Basel Stadt) ],
		[ qw(CH FR),  q(Fribourg) ],
		[ qw(CH GE),  q(Geneva) ],
		[ qw(CH GL),  q(Glarus) ],
		[ qw(CH GR),  q(Graubuenden) ],
		[ qw(CH JU),  q(Jura) ],
		[ qw(CH LU),  q(Lucerne) ],
		[ qw(CH NE),  q(Neuchatel) ],
		[ qw(CH NW),  q(Nidwalden) ],
		[ qw(CH OW),  q(Obwalden) ],
		[ qw(CH SG),  q(St. Gallen) ],
		[ qw(CH SH),  q(Schaffhausen) ],
		[ qw(CH SO),  q(Solothurn) ],
		[ qw(CH SZ),  q(Schwyz) ],
		[ qw(CH TG),  q(Thurgau) ],
		[ qw(CH TI),  q(Ticino) ],
		[ qw(CH UR),  q(Uri) ],
		[ qw(CH VD),  q(Vaud) ],
		[ qw(CH VS),  q(Valais) ],
		[ qw(CH ZG),  q(Zug) ],
		[ qw(CH ZH),  q(Zurich) ],
		[ qw(CL 1),   q(I - Iquique) ],
		[ qw(CL 2),   q(II - Antofagasta) ],
		[ qw(CL 3),   q(III - Copiapo) ],
		[ qw(CL 4),   q(IV - La Serena) ],
		[ qw(CL 5),   q(V - Valparaiso) ],
		[ qw(CL 6),   q(VI - Rancagua) ],
		[ qw(CL 7),   q(VII - Talca) ],
		[ qw(CL 8),   q(VIII - Concepcion) ],
		[ qw(CL 9),   q(IX - Temuco) ],
		[ qw(CL 10),  q(X - Puerto Montt) ],
		[ qw(CL 11),  q(XI - Coyhaique) ],
		[ qw(CL 12),  q(XII - Punta Arenas) ],
		[ qw(CL 13),  q(RM - Santiago) ],
		[ qw(CN 10),  q(Beijing) ],
		[ qw(CN 20),  q(Shanghai) ],
		[ qw(CN 30),  q(Tianjin) ],
		[ qw(CN 40),  q(Nei Mongol) ],
		[ qw(CN 50),  q(Shanxi) ],
		[ qw(CN 60),  q(Hebei) ],
		[ qw(CN 70),  q(Liaoning) ],
		[ qw(CN 80),  q(Jilin) ],
		[ qw(CN 90),  q(Heilongjiang) ],
		[ qw(CN 100), q(Jiangsu) ],
		[ qw(CN 110), q(Anhui) ],
		[ qw(CN 120), q(Shandong) ],
		[ qw(CN 130), q(Zhejiang) ],
		[ qw(CN 140), q(Jiangxi) ],
		[ qw(CN 150), q(Fujian) ],
		[ qw(CN 160), q(Hunan) ],
		[ qw(CN 170), q(Hubei) ],
		[ qw(CN 180), q(Henan) ],
		[ qw(CN 190), q(Guangdong) ],
		[ qw(CN 200), q(Hainan) ],
		[ qw(CN 210), q(Guangxi) ],
		[ qw(CN 220), q(Guizhou) ],
		[ qw(CN 230), q(Sichuan) ],
		[ qw(CN 240), q(Yunnan) ],
		[ qw(CN 250), q(Shaanxi) ],
		[ qw(CN 260), q(Gansu) ],
		[ qw(CN 270), q(Ningxia) ],
		[ qw(CN 280), q(Qinghai) ],
		[ qw(CN 290), q(Xinjiang) ],
		[ qw(CN 300), q(Xizang) ],
		[ qw(CN 320), q(Chong Qing) ],
		[ qw(CO 5),   q(Antioquia) ],
		[ qw(CO 8),   q(Atlantico) ],
		[ qw(CO 11),  q(Bogota) ],
		[ qw(CO 13),  q(Bolivar) ],
		[ qw(CO 15),  q(Boyaca) ],
		[ qw(CO 17),  q(Caldas) ],
		[ qw(CO 18),  q(Caqueta) ],
		[ qw(CO 19),  q(Cauca) ],
		[ qw(CO 20),  q(Cesar) ],
		[ qw(CO 23),  q(CORDOBA) ],
		[ qw(CO 25),  q(Cundinamarca) ],
		[ qw(CO 27),  q(Choco) ],
		[ qw(CO 41),  q(Huila) ],
		[ qw(CO 44),  q(La Guajira) ],
		[ qw(CO 47),  q(Magdalena) ],
		[ qw(CO 50),  q(Meta) ],
		[ qw(CO 52),  q(Narino) ],
		[ qw(CO 54),  q(Norte Santander) ],
		[ qw(CO 63),  q(Quindio) ],
		[ qw(CO 66),  q(Risaralda) ],
		[ qw(CO 68),  q(Santander) ],
		[ qw(CO 70),  q(Sucre) ],
		[ qw(CO 73),  q(Tolima) ],
		[ qw(CO 76),  q(Valle) ],
		[ qw(CO 81),  q(Arauca) ],
		[ qw(CO 85),  q(Casanare) ],
		[ qw(CO 86),  q(Putumayo) ],
		[ qw(CO 88),  q(San Andres) ],
		[ qw(CO 91),  q(Amazonas) ],
		[ qw(CO 94),  q(Guainia) ],
		[ qw(CO 95),  q(Guaviare) ],
		[ qw(CO 97),  q(Vaupes) ],
		[ qw(CO 99),  q(Vichada) ],
		[ qw(CZ 11),  q(Praha) ],
		[ qw(CZ 21),  q(Stredocesky) ],
		[ qw(CZ 31),  q(Jihocesky) ],
		[ qw(CZ 32),  q(Plzensky) ],
		[ qw(CZ 41),  q(Karlovarsky) ],
		[ qw(CZ 42),  q(Ustecky) ],
		[ qw(CZ 51),  q(Liberecky) ],
		[ qw(CZ 52),  q(Kralovehradecky) ],
		[ qw(CZ 53),  q(Pardubicky) ],
		[ qw(CZ 61),  q(Vysocina) ],
		[ qw(CZ 62),  q(Jihomoravsky) ],
		[ qw(CZ 71),  q(Olomoucky) ],
		[ qw(CZ 72),  q(Zlinsky) ],
		[ qw(CZ 81),  q(Moravskoslezsky) ],
		[ qw(DE 1),   q(Schleswig-Holstein) ],
		[ qw(DE 2),   q(Hamburg) ],
		[ qw(DE 3),   q(Lower Saxony) ],
		[ qw(DE 4),   q(Bremen) ],
		[ qw(DE 5),   q(Nrth Rhine Westfalia) ],
		[ qw(DE 6),   q(Hessen) ],
		[ qw(DE 7),   q(Rhineland Palatinate) ],
		[ qw(DE 8),   q(Baden-Wurttemberg) ],
		[ qw(DE 9),   q(Bavaria) ],
		[ qw(DE 10),  q(Saarland) ],
		[ qw(DE 11),  q(Berlin) ],
		[ qw(DE 12),  q(Brandenburg) ],
		[ qw(DE 13),  q(Mecklenburg-Vorpomm.) ],
		[ qw(DE 14),  q(Saxony) ],
		[ qw(DE 15),  q(Saxony-Anhalt) ],
		[ qw(DE 16),  q(Thuringia) ],
		[ qw(DK 1),   q(Danish Capital Reg.) ],
		[ qw(DK 2),   q(Central Jutland) ],
		[ qw(DK 3),   q(North Jutland) ],
		[ qw(DK 4),   q(Zealand) ],
		[ qw(DK 5),   q(South Denmark) ],
		[ qw(ES 1),   q(Alava) ],
		[ qw(ES 2),   q(Albacete) ],
		[ qw(ES 3),   q(Alicante) ],
		[ qw(ES 4),   q(Almeria) ],
		[ qw(ES 5),   q(Avila) ],
		[ qw(ES 6),   q(Badajoz) ],
		[ qw(ES 7),   q(Baleares) ],
		[ qw(ES 8),   q(Barcelona) ],
		[ qw(ES 9),   q(Burgos) ],
		[ qw(ES 10),  q(Caceres) ],
		[ qw(ES 11),  q(Cadiz) ],
		[ qw(ES 12),  q(Castellon) ],
		[ qw(ES 13),  q(Ciudad Real) ],
		[ qw(ES 14),  q(Cordoba) ],
		[ qw(ES 15),  q(La Coruna) ],
		[ qw(ES 16),  q(Cuenca) ],
		[ qw(ES 17),  q(Gerona) ],
		[ qw(ES 18),  q(Granada) ],
		[ qw(ES 19),  q(Guadalajara) ],
		[ qw(ES 20),  q(Guipuzcoa) ],
		[ qw(ES 21),  q(Huelva) ],
		[ qw(ES 22),  q(Huesca) ],
		[ qw(ES 23),  q(Jaen) ],
		[ qw(ES 24),  q(Leon) ],
		[ qw(ES 25),  q(Lerida) ],
		[ qw(ES 26),  q(La Rioja) ],
		[ qw(ES 27),  q(Lugo) ],
		[ qw(ES 28),  q(Madrid) ],
		[ qw(ES 29),  q(Malaga) ],
		[ qw(ES 30),  q(Murcia) ],
		[ qw(ES 31),  q(Navarra) ],
		[ qw(ES 32),  q(Orense) ],
		[ qw(ES 33),  q(Asturias) ],
		[ qw(ES 34),  q(Palencia) ],
		[ qw(ES 35),  q(Las Palmas) ],
		[ qw(ES 36),  q(Pontevedra) ],
		[ qw(ES 37),  q(Salamanca) ],
		[ qw(ES 38),  q(Sta. Cruz Tenerife) ],
		[ qw(ES 39),  q(Cantabria) ],
		[ qw(ES 40),  q(Segovia) ],
		[ qw(ES 41),  q(Sevilla) ],
		[ qw(ES 42),  q(Soria) ],
		[ qw(ES 43),  q(Tarragona) ],
		[ qw(ES 44),  q(Teruel) ],
		[ qw(ES 45),  q(Toledo) ],
		[ qw(ES 46),  q(Valencia) ],
		[ qw(ES 47),  q(Valladolid) ],
		[ qw(ES 48),  q(Vizcaya) ],
		[ qw(ES 49),  q(Zamora) ],
		[ qw(ES 50),  q(Zaragoza) ],
		[ qw(FI 1),   q(Ahvenanmaa) ],
		[ qw(FI 2),   q(Southern Finnland) ],
		[ qw(FI 3),   q(Eastern Finnland) ],
		[ qw(FI 4),   q(Lappi) ],
		[ qw(FI 5),   q(Western Finnland) ],
		[ qw(FI 6),   q(Oulu) ],
		[ qw(FR 1),   q(Ain) ],
		[ qw(FR 2),   q(Aisne) ],
		[ qw(FR 3),   q(Allier) ],
		[ qw(FR 4),   q(Alpes (Hte-Provence)) ],
		[ qw(FR 5),   q(Alpes (Hautes)) ],
		[ qw(FR 6),   q(Alpes-Maritimes) ],
		[ qw(FR 7),   q(Ardeche) ],
		[ qw(FR 8),   q(Ardennes) ],
		[ qw(FR 9),   q(Ariege) ],
		[ qw(FR 10),  q(Aube) ],
		[ qw(FR 11),  q(Aude) ],
		[ qw(FR 12),  q(Aveyron) ],
		[ qw(FR 13),  q(Bouches-du-Rhone) ],
		[ qw(FR 14),  q(Calvados) ],
		[ qw(FR 15),  q(Cantal) ],
		[ qw(FR 16),  q(Charente) ],
		[ qw(FR 17),  q(Charente-Maritime) ],
		[ qw(FR 18),  q(Cher) ],
		[ qw(FR 19),  q(Correze) ],
		[ qw(FR 21),  q(Cote-d'Or) ],
		[ qw(FR 22),  q(Cotes-d'Armor) ],
		[ qw(FR 23),  q(Creuse) ],
		[ qw(FR 24),  q(Dordogne) ],
		[ qw(FR 25),  q(Doubs) ],
		[ qw(FR 26),  q(Drome) ],
		[ qw(FR 27),  q(Eure) ],
		[ qw(FR 28),  q(Eure-et-Loir) ],
		[ qw(FR 29),  q(Finistere) ],
		[ qw(FR 2A),  q(Corse-du-Sud) ],
		[ qw(FR 2B),  q(Corse-du-Nord) ],
		[ qw(FR 30),  q(Gard) ],
		[ qw(FR 31),  q(Garonne (Haute)) ],
		[ qw(FR 32),  q(Gers) ],
		[ qw(FR 33),  q(Gironde) ],
		[ qw(FR 34),  q(Herault) ],
		[ qw(FR 35),  q(Ille-et-Vilaine) ],
		[ qw(FR 36),  q(Indre) ],
		[ qw(FR 37),  q(Indre-et-Loire) ],
		[ qw(FR 38),  q(Isere) ],
		[ qw(FR 39),  q(Jura) ],
		[ qw(FR 40),  q(Landes) ],
		[ qw(FR 41),  q(Loir-et-Cher) ],
		[ qw(FR 42),  q(Loire) ],
		[ qw(FR 43),  q(Loire (Haute)) ],
		[ qw(FR 44),  q(Loire-Atlantique) ],
		[ qw(FR 45),  q(Loiret) ],
		[ qw(FR 46),  q(Lot) ],
		[ qw(FR 47),  q(Lot-et-Garonne) ],
		[ qw(FR 48),  q(Lozere) ],
		[ qw(FR 49),  q(Maine-et-Loire) ],
		[ qw(FR 50),  q(Manche) ],
		[ qw(FR 51),  q(Marne) ],
		[ qw(FR 52),  q(Marne (Haute)) ],
		[ qw(FR 53),  q(Mayenne) ],
		[ qw(FR 54),  q(Meurthe-et-Moselle) ],
		[ qw(FR 55),  q(Meuse) ],
		[ qw(FR 56),  q(Morbihan) ],
		[ qw(FR 57),  q(Moselle) ],
		[ qw(FR 58),  q(Nievre) ],
		[ qw(FR 59),  q(Nord) ],
		[ qw(FR 60),  q(Oise) ],
		[ qw(FR 61),  q(Orne) ],
		[ qw(FR 62),  q(Pas-de-Calais) ],
		[ qw(FR 63),  q(Puy-de-Dome) ],
		[ qw(FR 64),  q(Pyrenees-Atlantiques) ],
		[ qw(FR 65),  q(Pyrenees (Hautes)) ],
		[ qw(FR 66),  q(Pyrenees-Orientales) ],
		[ qw(FR 67),  q(Bas-Rhin) ],
		[ qw(FR 68),  q(Haut-Rhin) ],
		[ qw(FR 69),  q(Rhone) ],
		[ qw(FR 70),  q(Saone (Haute)) ],
		[ qw(FR 71),  q(Saone-et-Loire) ],
		[ qw(FR 72),  q(Sarthe) ],
		[ qw(FR 73),  q(Savoie) ],
		[ qw(FR 74),  q(Savoie (Haute)) ],
		[ qw(FR 75),  q(Paris) ],
		[ qw(FR 76),  q(Seine-Maritime) ],
		[ qw(FR 77),  q(Seine-et-Marne) ],
		[ qw(FR 78),  q(Yvelines) ],
		[ qw(FR 79),  q(Sevres (Deux)) ],
		[ qw(FR 80),  q(Somme) ],
		[ qw(FR 81),  q(Tarn) ],
		[ qw(FR 82),  q(Tarn-et-Garonne) ],
		[ qw(FR 83),  q(Var) ],
		[ qw(FR 84),  q(Vaucluse) ],
		[ qw(FR 85),  q(Vendee) ],
		[ qw(FR 86),  q(Vienne) ],
		[ qw(FR 87),  q(Vienne (Haute)) ],
		[ qw(FR 88),  q(Vosges) ],
		[ qw(FR 89),  q(Yonne) ],
		[ qw(FR 90),  q(Territ.-de-Belfort) ],
		[ qw(FR 91),  q(Essonne) ],
		[ qw(FR 92),  q(Hauts-de-Seine) ],
		[ qw(FR 93),  q(Seine-Saint-Denis) ],
		[ qw(FR 94),  q(Val-de-Marne) ],
		[ qw(FR 95),  q(Val-d'Oise) ],
		[ qw(FR 97),  q(D.O.M.-T.O.M.) ],
		[ qw(FR 971), q(Guadeloupe) ],
		[ qw(FR 972), q(Martinique) ],
		[ qw(FR 973), q(Guyane) ],
		[ qw(FR 974), q(Reunion) ],
		[ qw(FR 975), q(Saint-Pierre-et-Miq.) ],
		[ qw(FR 976), q(Wallis-et-Futuna) ],
		[ qw(FR 99),  q(Hors-France) ],
		[ qw(GB AB),  q(Aberdeenshire) ],
		[ qw(GB AG),  q(Argyllshire) ],
		[ qw(GB AL),  q(Anglesey) ],
		[ qw(GB AM),  q(Armagh) ],
		[ qw(GB AN),  q(Angus/Forfarshire) ],
		[ qw(GB AT),  q(Antrim) ],
		[ qw(GB AY),  q(Ayrshire) ],
		[ qw(GB BE),  q(Bedfordshire) ],
		[ qw(GB BF),  q(Banffshire) ],
		[ qw(GB BK),  q(Berkshire) ],
		[ qw(GB BR),  q(Brecknockshire) ],
		[ qw(GB BS),  q(Bath&NthEstSomerset) ],
		[ qw(GB BT),  q(Buteshire) ],
		[ qw(GB BU),  q(Buckinghamshire) ],
		[ qw(GB BW),  q(Berwickshire) ],
		[ qw(GB CA),  q(Cambridgeshire) ],
		[ qw(GB CB),  q(Carmarthenshire) ],
		[ qw(GB CD),  q(Cardiganshire) ],
		[ qw(GB CF),  q(Caernarfonshire) ],
		[ qw(GB CH),  q(Cheshire) ],
		[ qw(GB CM),  q(Cromartyshire) ],
		[ qw(GB CN),  q(Clackmannanshire) ],
		[ qw(GB CO),  q(Cornwall) ],
		[ qw(GB CT),  q(Caithness) ],
		[ qw(GB CU),  q(Cumberland) ],
		[ qw(GB DB),  q(Derbyshire) ],
		[ qw(GB DD),  q(Denbighshire) ],
		[ qw(GB DF),  q(Dumfriesshire) ],
		[ qw(GB DN),  q(Down) ],
		[ qw(GB DO),  q(Dorset) ],
		[ qw(GB DT),  q(Dunbartonshire) ],
		[ qw(GB DU),  q(Durham) ],
		[ qw(GB DV),  q(Devon) ],
		[ qw(GB EL),  q(East Lothian) ],
		[ qw(GB ES),  q(Essex) ],
		[ qw(GB FI),  q(Fife) ],
		[ qw(GB FL),  q(Flintshire) ],
		[ qw(GB FM),  q(Fermanagh) ],
		[ qw(GB GL),  q(Gloucestershire) ],
		[ qw(GB HA),  q(Hampshire) ],
		[ qw(GB HT),  q(Hertfordshire) ],
		[ qw(GB HU),  q(Huntingdonshire) ],
		[ qw(GB HW),  q(Hereford and Worcs.) ],
		[ qw(GB IN),  q(Invernesshire) ],
		[ qw(GB IW),  q(Isle of Wight) ],
		[ qw(GB KE),  q(Kent) ],
		[ qw(GB KI),  q(Kincardineshire) ],
		[ qw(GB KK),  q(Kirkcudbrightshire) ],
		[ qw(GB KN),  q(Kinross-shire) ],
		[ qw(GB LA),  q(Lancashire) ],
		[ qw(GB LD),  q(Londonderry) ],
		[ qw(GB LE),  q(Leicestershire) ],
		[ qw(GB LI),  q(Lincolnshire) ],
		[ qw(GB LN),  q(Lanarkshire) ],
		[ qw(GB MD),  q(Midlothian) ],
		[ qw(GB ME),  q(Merioneth) ],
		[ qw(GB MG),  q(Mid Glamorgan) ],
		[ qw(GB MM),  q(Monmouthshire) ],
		[ qw(GB MR),  q(Morayshire) ],
		[ qw(GB MT),  q(Montgomeryshire) ],
		[ qw(GB MX),  q(Middlesex) ],
		[ qw(GB NH),  q(Northamptonshire) ],
		[ qw(GB NK),  q(Norfolk) ],
		[ qw(GB NR),  q(Nairnshire) ],
		[ qw(GB NT),  q(Nottinghamshire) ],
		[ qw(GB NU),  q(Northumberland) ],
		[ qw(GB OR),  q(Orkney) ],
		[ qw(GB OX),  q(Oxfordshire) ],
		[ qw(GB PE),  q(Peeblesshire) ],
		[ qw(GB PM),  q(Pembrokeshire) ],
		[ qw(GB PR),  q(Perthshire) ],
		[ qw(GB RA),  q(Radnorshire) ],
		[ qw(GB RE),  q(Renfrewshire) ],
		[ qw(GB RO),  q(Ross-shire) ],
		[ qw(GB RU),  q(Rutland) ],
		[ qw(GB RX),  q(Roxburghshire) ],
		[ qw(GB SE),  q(East Sussex) ],
		[ qw(GB SF),  q(Selkirkshire) ],
		[ qw(GB SG),  q(South Glamorgan) ],
		[ qw(GB SH),  q(Shropshire) ],
		[ qw(GB SK),  q(Suffolk) ],
		[ qw(GB SL),  q(Shetland) ],
		[ qw(GB SO),  q(Somerset) ],
		[ qw(GB ST),  q(Staffordshire) ],
		[ qw(GB SU),  q(Sutherland) ],
		[ qw(GB SV),  q(Stirlingshire) ],
		[ qw(GB SW),  q(West Sussex) ],
		[ qw(GB SY),  q(Surrey) ],
		[ qw(GB TY),  q(Tyrone) ],
		[ qw(GB WA),  q(Warwickshire) ],
		[ qw(GB WC),  q(Worcestershire) ],
		[ qw(GB WE),  q(Westmorland) ],
		[ qw(GB WG),  q(West Glamorgan) ],
		[ qw(GB WI),  q(Wiltshire) ],
		[ qw(GB WK),  q(West Lothian) ],
		[ qw(GB WT),  q(Wigtownshire) ],
		[ qw(GB YN),  q(North Yorkshire) ],
		[ qw(GB YS),  q(South Yorkshire) ],
		[ qw(GB YW),  q(West Yorkshire) ],
		[ qw(GR 1),   q(Aitolia kai Akarnan.) ],
		[ qw(GR 2),   q(Akhaia) ],
		[ qw(GR 3),   q(Argolis) ],
		[ qw(GR 4),   q(Arkadhia) ],
		[ qw(GR 5),   q(Arta) ],
		[ qw(GR 6),   q(Attiki) ],
		[ qw(GR 7),   q(Dhodhekanisos) ],
		[ qw(GR 8),   q(Dhrama) ],
		[ qw(GR 9),   q(Evritania) ],
		[ qw(GR 10),  q(Evros) ],
		[ qw(GR 11),  q(Evvoia) ],
		[ qw(GR 12),  q(Florina) ],
		[ qw(GR 13),  q(Fokis) ],
		[ qw(GR 14),  q(Fthiotis) ],
		[ qw(GR 15),  q(Grevena) ],
		[ qw(GR 16),  q(Ilia) ],
		[ qw(GR 17),  q(Imathia) ],
		[ qw(GR 18),  q(Ioannina) ],
		[ qw(GR 19),  q(Iraklion) ],
		[ qw(GR 20),  q(Kardhitsa) ],
		[ qw(GR 21),  q(Kastoria) ],
		[ qw(GR 22),  q(Kavala) ],
		[ qw(GR 23),  q(Kefallinia) ],
		[ qw(GR 24),  q(Kerkira) ],
		[ qw(GR 25),  q(Khalkidhiki) ],
		[ qw(GR 26),  q(Khania) ],
		[ qw(GR 27),  q(Khios) ],
		[ qw(GR 28),  q(Kikladhes) ],
		[ qw(GR 29),  q(Kilkis) ],
		[ qw(GR 30),  q(Korinthia) ],
		[ qw(GR 31),  q(Kozani) ],
		[ qw(GR 32),  q(Lakonia) ],
		[ qw(GR 33),  q(Larisa) ],
		[ qw(GR 34),  q(Lasithi) ],
		[ qw(GR 35),  q(Lesvos) ],
		[ qw(GR 36),  q(Levkas) ],
		[ qw(GR 37),  q(Magnisia) ],
		[ qw(GR 38),  q(Messinia) ],
		[ qw(GR 39),  q(Pella) ],
		[ qw(GR 40),  q(Pieria) ],
		[ qw(GR 41),  q(Piraievs) ],
		[ qw(GR 42),  q(Preveza) ],
		[ qw(GR 43),  q(Rethimni) ],
		[ qw(GR 44),  q(Rodhopi) ],
		[ qw(GR 45),  q(Samos) ],
		[ qw(GR 46),  q(Serrai) ],
		[ qw(GR 47),  q(Thesprotia) ],
		[ qw(GR 48),  q(Thessaloniki) ],
		[ qw(GR 49),  q(Trikala) ],
		[ qw(GR 50),  q(Voiotia) ],
		[ qw(GR 51),  q(Xanthi) ],
		[ qw(GR 52),  q(Zakinthos) ],
		[ qw(HK HK),  q(Hong Kong Island) ],
		[ qw(HK KLN), q(Kowloon) ],
		[ qw(HK NT),  q(New Territories) ],
		[ qw(HR A00), q(Zagrebacka) ],
		[ qw(HR B00), q(Krapinsko-zagorska) ],
		[ qw(HR C00), q(Sisacko-moslavacka) ],
		[ qw(HR D00), q(Karlovacka) ],
		[ qw(HR E00), q(Varazdinska) ],
		[ qw(HR F00), q(Koprivnicko-krizevac) ],
		[ qw(HR G00), q(Bjelovarsko-bilogors) ],
		[ qw(HR H00), q(Rijecko-goranska) ],
		[ qw(HR I00), q(Licko-senjska) ],
		[ qw(HR J00), q(Viroviticko-podravac) ],
		[ qw(HR K00), q(Pozesko-slavonska) ],
		[ qw(HR L00), q(Slavonskobrodska) ],
		[ qw(HR M00), q(Zadarska) ],
		[ qw(HR N00), q(Osjecko-baranjska) ],
		[ qw(HR O00), q(Sibensko-kninska) ],
		[ qw(HR P00), q(Vukovarsko-srijemska) ],
		[ qw(HR R00), q(Splitsko-dalmatinska) ],
		[ qw(HR S00), q(Istarska) ],
		[ qw(HR T00), q(Dubrovacko-neretvans) ],
		[ qw(HR U00), q(Medjimurska) ],
		[ qw(HR V00), q(Zagreb) ],
		[ qw(HU 1),   q(Bacs-Kiskun) ],
		[ qw(HU 2),   q(Baranya) ],
		[ qw(HU 3),   q(Bekes) ],
		[ qw(HU 4),   q(Bekescsaba) ],
		[ qw(HU 5),   q(Borsod-Abauj-Zemplen) ],
		[ qw(HU 6),   q(Budapest) ],
		[ qw(HU 7),   q(Csongrad) ],
		[ qw(HU 8),   q(Debrecen) ],
		[ qw(HU 9),   q(Dunaujvaros) ],
		[ qw(HU 10),  q(Eger) ],
		[ qw(HU 11),  q(Fejer) ],
		[ qw(HU 12),  q(Gyor) ],
		[ qw(HU 13),  q(Gyor-Moson-Sopron) ],
		[ qw(HU 14),  q(Hajdu-Bihar) ],
		[ qw(HU 15),  q(Heves) ],
		[ qw(HU 16),  q(Hodmezovasarhely) ],
		[ qw(HU 17),  q(Jasz-Nagykun-Szolnok) ],
		[ qw(HU 18),  q(Kaposvar) ],
		[ qw(HU 19),  q(Kecskemet) ],
		[ qw(HU 20),  q(Komarom-Esztergom) ],
		[ qw(HU 21),  q(Miskolc) ],
		[ qw(HU 22),  q(Nagykanizsa) ],
		[ qw(HU 23),  q(Nograd) ],
		[ qw(HU 24),  q(Nyiregyhaza) ],
		[ qw(HU 25),  q(Pecs) ],
		[ qw(HU 26),  q(Pest) ],
		[ qw(HU 27),  q(Somogy) ],
		[ qw(HU 28),  q(Sopron) ],
		[ qw(HU 29),  q(Szabolcs-Szat.-Bereg) ],
		[ qw(HU 30),  q(Szeged) ],
		[ qw(HU 31),  q(Szekesfehervar) ],
		[ qw(HU 32),  q(Szolnok) ],
		[ qw(HU 33),  q(Szombathely) ],
		[ qw(HU 34),  q(Tatabanya) ],
		[ qw(HU 35),  q(Tolna) ],
		[ qw(HU 36),  q(Vas) ],
		[ qw(HU 37),  q(Veszprem) ],
		[ qw(HU 38),  q(Zala) ],
		[ qw(HU 39),  q(Zalaegerszeg) ],
		[ qw(ID 1),   q(DKI Jakarta Jakarta) ],
		[ qw(ID 2),   q(Jawa Barat West Java) ],
		[ qw(ID 3),   q(Jawa Tengah Central) ],
		[ qw(ID 4),   q(Jawa Timur East Java) ],
		[ qw(ID 5),   q(DI Yogyakarta Yogyak) ],
		[ qw(ID 6),   q(DI Aceh Aceh) ],
		[ qw(ID 7),   q(Sumatera Utara North) ],
		[ qw(ID 8),   q(Sumatera Barat West) ],
		[ qw(ID 9),   q(Riau Riau) ],
		[ qw(ID 10),  q(Jambi Jambi) ],
		[ qw(ID 11),  q(Sumatera Selatan Sou) ],
		[ qw(ID 12),  q(Bengkulu Bengkulu) ],
		[ qw(ID 13),  q(Lampung Lampung) ],
		[ qw(ID 14),  q(Kalimantan Selatan S) ],
		[ qw(ID 15),  q(Kalimantan Barat Wes) ],
		[ qw(ID 16),  q(Kalimantan Tengah Ce) ],
		[ qw(ID 17),  q(Kalimantan Timur Eas) ],
		[ qw(ID 18),  q(Sulawesi Selatan Sou) ],
		[ qw(ID 19),  q(Sulawesi Tenggara So) ],
		[ qw(ID 20),  q(Sulawesi Tengah Cent) ],
		[ qw(ID 21),  q(Sulawesi Utara North) ],
		[ qw(ID 22),  q(Bali Bali) ],
		[ qw(ID 23),  q(Nusa Tenggara Barat) ],
		[ qw(ID 24),  q(Nusa Tenggara Timur) ],
		[ qw(ID 25),  q(Maluku Maluku) ],
		[ qw(ID 26),  q(Irian Jaya Irian Jay) ],
		[ qw(ID 27),  q(Timor Timur East Tim) ],
		[ qw(IE CK),  q(Cork) ],
		[ qw(IE CL),  q(Clare) ],
		[ qw(IE CV),  q(Cavan) ],
		[ qw(IE CW),  q(Carlow) ],
		[ qw(IE DB),  q(Dublin) ],
		[ qw(IE DG),  q(Donegal) ],
		[ qw(IE GW),  q(Galway) ],
		[ qw(IE KD),  q(Kildare) ],
		[ qw(IE KK),  q(Kilkenny) ],
		[ qw(IE KY),  q(Kerry) ],
		[ qw(IE LF),  q(Longford) ],
		[ qw(IE LI),  q(Limerick) ],
		[ qw(IE LM),  q(Leitrim) ],
		[ qw(IE LS),  q(Laois) ],
		[ qw(IE LT),  q(Louth) ],
		[ qw(IE MH),  q(Monaghan) ],
		[ qw(IE MT),  q(Meath) ],
		[ qw(IE MY),  q(Mayo) ],
		[ qw(IE OF),  q(Offaly) ],
		[ qw(IE RC),  q(Rosscommon) ],
		[ qw(IE SG),  q(Sligo) ],
		[ qw(IE TP),  q(Tipperary) ],
		[ qw(IE WF),  q(Waterford) ],
		[ qw(IE WK),  q(Wicklow) ],
		[ qw(IE WM),  q(Westmeath) ],
		[ qw(IE WX),  q(Wexford) ],
		[ qw(IL 1),   q(Central) ],
		[ qw(IL 2),   q(Haifa) ],
		[ qw(IL 3),   q(Jerusalem) ],
		[ qw(IL 4),   q(Northern) ],
		[ qw(IL 5),   q(Southern) ],
		[ qw(IL 6),   q(Tel Aviv) ],
		[ qw(IN 1),   q(Andra Pradesh) ],
		[ qw(IN 2),   q(Arunachal Pradesh) ],
		[ qw(IN 3),   q(Assam) ],
		[ qw(IN 4),   q(Bihar) ],
		[ qw(IN 5),   q(Goa) ],
		[ qw(IN 6),   q(Gujarat) ],
		[ qw(IN 7),   q(Haryana) ],
		[ qw(IN 8),   q(Himachal Pradesh) ],
		[ qw(IN 9),   q(Jammu und Kashmir) ],
		[ qw(IN 10),  q(Karnataka) ],
		[ qw(IN 11),  q(Kerala) ],
		[ qw(IN 12),  q(Madhya Pradesh) ],
		[ qw(IN 13),  q(Maharashtra) ],
		[ qw(IN 14),  q(Manipur) ],
		[ qw(IN 15),  q(Megalaya) ],
		[ qw(IN 16),  q(Mizoram) ],
		[ qw(IN 17),  q(Nagaland) ],
		[ qw(IN 18),  q(Orissa) ],
		[ qw(IN 19),  q(Punjab) ],
		[ qw(IN 20),  q(Rajasthan) ],
		[ qw(IN 21),  q(Sikkim) ],
		[ qw(IN 22),  q(Tamil Nadu) ],
		[ qw(IN 23),  q(Tripura) ],
		[ qw(IN 24),  q(Uttar Pradesh) ],
		[ qw(IN 25),  q(West Bengal) ],
		[ qw(IN 26),  q(Andaman und Nico.In.) ],
		[ qw(IN 27),  q(Chandigarh) ],
		[ qw(IN 28),  q(Dadra und Nagar Hav.) ],
		[ qw(IN 29),  q(Daman und Diu) ],
		[ qw(IN 30),  q(Delhi) ],
		[ qw(IN 31),  q(Lakshadweep) ],
		[ qw(IN 32),  q(Pondicherry) ],
		[ qw(IN 33),  q(Chhaattisgarh) ],
		[ qw(IN 34),  q(Jharkhand) ],
		[ qw(IN 35),  q(Uttaranchal) ],
		[ qw(IN A1),  q() ],
		[ qw(IN A2),  q() ],
		[ qw(IN A3),  q() ],
		[ qw(IN A4),  q() ],
		[ qw(IT AG),  q(Agriento) ],
		[ qw(IT AL),  q(Alessandria) ],
		[ qw(IT AN),  q(Ancona) ],
		[ qw(IT AO),  q(Aosta) ],
		[ qw(IT AP),  q(Ascoli Piceno) ],
		[ qw(IT AQ),  q(L'Aquila) ],
		[ qw(IT AR),  q(Arezzo) ],
		[ qw(IT AT),  q(Asti) ],
		[ qw(IT AV),  q(Avellino) ],
		[ qw(IT BA),  q(Bari) ],
		[ qw(IT BG),  q(Bergamo) ],
		[ qw(IT BI),  q(Biella) ],
		[ qw(IT BL),  q(Belluno) ],
		[ qw(IT BN),  q(Benevento) ],
		[ qw(IT BO),  q(Bologna) ],
		[ qw(IT BR),  q(Brindisi) ],
		[ qw(IT BS),  q(Brescia) ],
		[ qw(IT BZ),  q(Bolzano) ],
		[ qw(IT CA),  q(Cagliari) ],
		[ qw(IT CB),  q(Campobasso) ],
		[ qw(IT CE),  q(Caserta) ],
		[ qw(IT CH),  q(Chieti) ],
		[ qw(IT CL),  q(Caltanisetta) ],
		[ qw(IT CN),  q(Cuneo) ],
		[ qw(IT CO),  q(Como) ],
		[ qw(IT CR),  q(Cremona) ],
		[ qw(IT CS),  q(Cosenza) ],
		[ qw(IT CT),  q(Catania) ],
		[ qw(IT CZ),  q(Catanzaro) ],
		[ qw(IT EN),  q(Enna) ],
		[ qw(IT FC),  q(Forli/Cesana) ],
		[ qw(IT FE),  q(Ferrara) ],
		[ qw(IT FG),  q(Foggia) ],
		[ qw(IT FI),  q(Florence) ],
		[ qw(IT FR),  q(Frosinone) ],
		[ qw(IT GE),  q(Genova) ],
		[ qw(IT GO),  q(Gorizia) ],
		[ qw(IT GR),  q(Grosseto) ],
		[ qw(IT IM),  q(Imperia) ],
		[ qw(IT IS),  q(Isernia) ],
		[ qw(IT KR),  q(Crotone) ],
		[ qw(IT LC),  q(Lecco) ],
		[ qw(IT LE),  q(Lecce) ],
		[ qw(IT LI),  q(Livorno) ],
		[ qw(IT LO),  q(Lodi) ],
		[ qw(IT LT),  q(Latina) ],
		[ qw(IT LU),  q(Lucca) ],
		[ qw(IT MC),  q(Macerata) ],
		[ qw(IT ME),  q(Messina) ],
		[ qw(IT MI),  q(Milan) ],
		[ qw(IT MN),  q(Mantova) ],
		[ qw(IT MO),  q(Modena) ],
		[ qw(IT MS),  q(Massa Carrara) ],
		[ qw(IT MT),  q(Matera) ],
		[ qw(IT NA),  q(Naples) ],
		[ qw(IT NO),  q(Novara) ],
		[ qw(IT NU),  q(Nuoro) ],
		[ qw(IT OR),  q(Oristano) ],
		[ qw(IT PA),  q(Palermo) ],
		[ qw(IT PC),  q(Piacenza) ],
		[ qw(IT PD),  q(Padova) ],
		[ qw(IT PE),  q(Pescara) ],
		[ qw(IT PG),  q(Perugia) ],
		[ qw(IT PI),  q(Pisa) ],
		[ qw(IT PN),  q(Pordenone) ],
		[ qw(IT PO),  q(Prato) ],
		[ qw(IT PR),  q(Parma) ],
		[ qw(IT PT),  q(Pistoia) ],
		[ qw(IT PU),  q(Pesaro-Urbino) ],
		[ qw(IT PV),  q(Pavia) ],
		[ qw(IT PZ),  q(Potenza) ],
		[ qw(IT RA),  q(Ravenna) ],
		[ qw(IT RC),  q(Reggio Calabria) ],
		[ qw(IT RE),  q(Reggio Emilia) ],
		[ qw(IT RG),  q(Ragusa) ],
		[ qw(IT RI),  q(Rieti) ],
		[ qw(IT RM),  q(Rome) ],
		[ qw(IT RN),  q(Rimini) ],
		[ qw(IT RO),  q(Rovigo) ],
		[ qw(IT SA),  q(Salerno) ],
		[ qw(IT SI),  q(Siena) ],
		[ qw(IT SO),  q(Sondrio) ],
		[ qw(IT SP),  q(La Spezia) ],
		[ qw(IT SR),  q(Siracusa) ],
		[ qw(IT SS),  q(Sassari) ],
		[ qw(IT SV),  q(Savona) ],
		[ qw(IT TA),  q(Taranto) ],
		[ qw(IT TE),  q(Teramo) ],
		[ qw(IT TN),  q(Trento) ],
		[ qw(IT TO),  q(Turin) ],
		[ qw(IT TP),  q(Trapani) ],
		[ qw(IT TR),  q(Terni) ],
		[ qw(IT TS),  q(Trieste) ],
		[ qw(IT TV),  q(Treviso) ],
		[ qw(IT UD),  q(Udine) ],
		[ qw(IT VA),  q(Varese) ],
		[ qw(IT VB),  q(Verbania) ],
		[ qw(IT VC),  q(Vercelli) ],
		[ qw(IT VE),  q(Venice) ],
		[ qw(IT VI),  q(Vicenza) ],
		[ qw(IT VR),  q(Verona) ],
		[ qw(IT VT),  q(Viterbo) ],
		[ qw(IT VV),  q(Vibo Valentia) ],
		[ qw(JP 1),   q(Hokkaido) ],
		[ qw(JP 2),   q(Aomori) ],
		[ qw(JP 3),   q(Iwate) ],
		[ qw(JP 4),   q(Miyagi) ],
		[ qw(JP 5),   q(Akita) ],
		[ qw(JP 6),   q(Yamagata) ],
		[ qw(JP 7),   q(Fukushima) ],
		[ qw(JP 8),   q(Ibaraki) ],
		[ qw(JP 9),   q(Tochigi) ],
		[ qw(JP 10),  q(Gunma) ],
		[ qw(JP 11),  q(Saitama) ],
		[ qw(JP 12),  q(Chiba) ],
		[ qw(JP 13),  q(Tokyo) ],
		[ qw(JP 14),  q(Kanagawa) ],
		[ qw(JP 15),  q(Niigata) ],
		[ qw(JP 16),  q(Toyama) ],
		[ qw(JP 17),  q(Ishikawa) ],
		[ qw(JP 18),  q(Fukui) ],
		[ qw(JP 19),  q(Yamanashi) ],
		[ qw(JP 20),  q(Nagano) ],
		[ qw(JP 21),  q(Gifu) ],
		[ qw(JP 22),  q(Shizuoka) ],
		[ qw(JP 23),  q(Aichi) ],
		[ qw(JP 24),  q(Mie) ],
		[ qw(JP 25),  q(Shiga) ],
		[ qw(JP 26),  q(Kyoto) ],
		[ qw(JP 27),  q(Osaka) ],
		[ qw(JP 28),  q(Hyogo) ],
		[ qw(JP 29),  q(Nara) ],
		[ qw(JP 30),  q(Wakayama) ],
		[ qw(JP 31),  q(Tottori) ],
		[ qw(JP 32),  q(Shimane) ],
		[ qw(JP 33),  q(Okayama) ],
		[ qw(JP 34),  q(Hiroshima) ],
		[ qw(JP 35),  q(Yamaguchi) ],
		[ qw(JP 36),  q(Tokushima) ],
		[ qw(JP 37),  q(Kagawa) ],
		[ qw(JP 38),  q(Ehime) ],
		[ qw(JP 39),  q(Kochi) ],
		[ qw(JP 40),  q(Fukuoka) ],
		[ qw(JP 41),  q(Saga) ],
		[ qw(JP 42),  q(Nagasaki) ],
		[ qw(JP 43),  q(Kumamoto) ],
		[ qw(JP 44),  q(Oita) ],
		[ qw(JP 45),  q(Miyazaki) ],
		[ qw(JP 46),  q(Kagoshima) ],
		[ qw(JP 47),  q(Okinawa) ],
		[ qw(KR 1),   q(Cheju-do) ],
		[ qw(KR 2),   q(Cholla-bukto) ],
		[ qw(KR 3),   q(Cholla-namdo) ],
		[ qw(KR 4),   q(Ch'ungch'ong-bukto) ],
		[ qw(KR 5),   q(Ch'ungch'ong-namdo) ],
		[ qw(KR 6),   q(Inch'on-jikhalsi) ],
		[ qw(KR 7),   q(Kangwon-do) ],
		[ qw(KR 8),   q(Kwangju-jikhalsi) ],
		[ qw(KR 9),   q(Kyonggi-do) ],
		[ qw(KR 10),  q(Kyongsang-bukto) ],
		[ qw(KR 11),  q(Kyongsang-namdo) ],
		[ qw(KR 12),  q(Pusan-jikhalsi) ],
		[ qw(KR 13),  q(Soul-t'ukpyolsi) ],
		[ qw(KR 14),  q(Taegu-jikhalsi) ],
		[ qw(KR 15),  q(Taejon-jikhalsi) ],
		[ qw(KZ 0),   q(Almatynskaia) ],
		[ qw(KZ 1),   q(Kostanaiskaia) ],
		[ qw(KZ 2),   q(Severo-Kazakhstansk) ],
		[ qw(KZ 3),   q(Pavlodarskaia) ],
		[ qw(KZ 4),   q(Akmolinskaia) ],
		[ qw(KZ 5),   q(Aktubinskaia) ],
		[ qw(KZ 6),   q(Atyrauskaia) ],
		[ qw(KZ 7),   q(Zapadno-Kazakhst) ],
		[ qw(KZ 8),   q(Mangystayskaia) ],
		[ qw(KZ 9),   q(Karagandinskaia) ],
		[ qw(KZ 10),  q(Vostochno-Kazakhstan) ],
		[ qw(KZ 11),  q(Gambilskaia) ],
		[ qw(KZ 12),  q(Kyzilordinskaia) ],
		[ qw(LU LU),  q() ],
		[ qw(MT MT),  q(Malta) ],
		[ qw(MX AGS), q(Aguascalientes) ],
		[ qw(MX BC),  q(Baja California) ],
		[ qw(MX BCS), q(Baja California S) ],
		[ qw(MX CHI), q(Chihuahua) ],
		[ qw(MX CHS), q(Chiapas) ],
		[ qw(MX CMP), q(Campeche) ],
		[ qw(MX COA), q(Coahuila) ],
		[ qw(MX COL), q(Colima) ],
		[ qw(MX DF),  q(Distrito Federal) ],
		[ qw(MX DGO), q(Durango) ],
		[ qw(MX GRO), q(Guerrero) ],
		[ qw(MX GTO), q(Guanajuato) ],
		[ qw(MX HGO), q(Hidalgo) ],
		[ qw(MX JAL), q(Jalisco) ],
		[ qw(MX MCH), q(Michoacan) ],
		[ qw(MX MEX), q(Estado de Mexico) ],
		[ qw(MX MOR), q(Morelos) ],
		[ qw(MX NAY), q(Nayarit) ],
		[ qw(MX NL),  q(Nuevo Leon) ],
		[ qw(MX OAX), q(Oaxaca) ],
		[ qw(MX PUE), q(Puebla) ],
		[ qw(MX QR),  q(Quintana Roo) ],
		[ qw(MX QRO), q(Queretaro) ],
		[ qw(MX SIN), q(Sinaloa) ],
		[ qw(MX SLP), q(San Luis Potosi) ],
		[ qw(MX SON), q(Sonora) ],
		[ qw(MX TAB), q(Tabasco) ],
		[ qw(MX TLX), q(Tlaxcala) ],
		[ qw(MX TMS), q(Tamaulipas) ],
		[ qw(MX VER), q(Veracruz) ],
		[ qw(MX YUC), q(Yucatan) ],
		[ qw(MX ZAC), q(Zacatecas) ],
		[ qw(MY JOH), q(Johor) ],
		[ qw(MY KED), q(Kedah) ],
		[ qw(MY KEL), q(Kelantan) ],
		[ qw(MY KUL), q(Kuala Lumpur) ],
		[ qw(MY LAB), q(Labuan) ],
		[ qw(MY MEL), q(Melaka) ],
		[ qw(MY PAH), q(Pahang) ],
		[ qw(MY PEL), q(Perlis) ],
		[ qw(MY PER), q(Perak) ],
		[ qw(MY PIN), q(Pulau Pinang) ],
		[ qw(MY PSK), q(Wil. Persekutuan) ],
		[ qw(MY SAB), q(Sabah) ],
		[ qw(MY SAR), q(Sarawak) ],
		[ qw(MY SEL), q(Selangor) ],
		[ qw(MY SER), q(Negeri Sembilan) ],
		[ qw(MY TRE), q(Trengganu) ],
		[ qw(NL 1),   q(Drenthe) ],
		[ qw(NL 2),   q(Flevoland) ],
		[ qw(NL 3),   q(Friesland) ],
		[ qw(NL 4),   q(Gelderland) ],
		[ qw(NL 5),   q(Groningen) ],
		[ qw(NL 6),   q(Limburg) ],
		[ qw(NL 7),   q(Noord-Brabant) ],
		[ qw(NL 8),   q(Noord-Holland) ],
		[ qw(NL 9),   q(Overijssel) ],
		[ qw(NL 10),  q(Utrecht) ],
		[ qw(NL 11),  q(Zeeland) ],
		[ qw(NL 12),  q(Zuid-Holland) ],
		[ qw(NO 1),   q(Ostfold County) ],
		[ qw(NO 2),   q(Akershus County) ],
		[ qw(NO 3),   q(Oslo) ],
		[ qw(NO 4),   q(Hedmark County) ],
		[ qw(NO 5),   q(Oppland County) ],
		[ qw(NO 6),   q(Buskerud County) ],
		[ qw(NO 7),   q(Vestfold County) ],
		[ qw(NO 8),   q(Telemark County) ],
		[ qw(NO 9),   q(Aust-Agder County) ],
		[ qw(NO 10),  q(Vest-Agder County) ],
		[ qw(NO 11),  q(Rogaland County) ],
		[ qw(NO 12),  q(Hordaland County) ],
		[ qw(NO 14),  q(Sogn og Fjordane C.) ],
		[ qw(NO 15),  q(More og Romsdal C.) ],
		[ qw(NO 16),  q(Sor-Trondelag County) ],
		[ qw(NO 17),  q(Nord-Trondelag Cnty) ],
		[ qw(NO 18),  q(Nordland County) ],
		[ qw(NO 19),  q(Troms County) ],
		[ qw(NO 20),  q(Finnmark County) ],
		[ qw(NZ AKL), q(Auckland) ],
		[ qw(NZ BOP), q(Bay of Plenty) ],
		[ qw(NZ CAN), q(Canterbury) ],
		[ qw(NZ HAB), q(Hawke's Bay) ],
		[ qw(NZ MAN), q(Manawatu-Wanganui) ],
		[ qw(NZ NTL), q(Northland) ],
		[ qw(NZ OTA), q(Otago) ],
		[ qw(NZ STL), q(Southland) ],
		[ qw(NZ TAR), q(Taranaki) ],
		[ qw(NZ WAI), q(Waikato) ],
		[ qw(NZ WEC), q(West Coast) ],
		[ qw(NZ WLG), q(Wellington) ],
		[ qw(PE 1),   q(Tumbes) ],
		[ qw(PE 2),   q(Piura) ],
		[ qw(PE 3),   q(Lambayeque) ],
		[ qw(PE 4),   q(La Libertad) ],
		[ qw(PE 5),   q(Ancash) ],
		[ qw(PE 6),   q(Lima y Callao) ],
		[ qw(PE 7),   q(Ica) ],
		[ qw(PE 8),   q(Arequipa) ],
		[ qw(PE 9),   q(Moquegua) ],
		[ qw(PE 10),  q(Tacna) ],
		[ qw(PE 11),  q(Amazon) ],
		[ qw(PE 12),  q(Cajamarca) ],
		[ qw(PE 13),  q(San Martin) ],
		[ qw(PE 14),  q(Huanuco) ],
		[ qw(PE 15),  q(Pasco) ],
		[ qw(PE 16),  q(Junin) ],
		[ qw(PE 17),  q(Huancavelica) ],
		[ qw(PE 18),  q(Ayacucho) ],
		[ qw(PE 19),  q(Apurimac) ],
		[ qw(PE 20),  q(Cuzco) ],
		[ qw(PE 21),  q(Puno) ],
		[ qw(PE 22),  q(Loreto) ],
		[ qw(PE 23),  q(Ucayali) ],
		[ qw(PE 24),  q(Madre de Dios) ],
		[ qw(PH 1),   q(Ilocos) ],
		[ qw(PH 2),   q(Cagayan Valley) ],
		[ qw(PH 3),   q(Central Luzon) ],
		[ qw(PH 4),   q(South Luzon) ],
		[ qw(PH 5),   q(Bicol) ],
		[ qw(PH 6),   q(West Visayas) ],
		[ qw(PH 7),   q(Central Visayas) ],
		[ qw(PH 8),   q(Eastern Visayas) ],
		[ qw(PH 9),   q(Western Mindanao) ],
		[ qw(PH 10),  q(Northern Mindanao) ],
		[ qw(PH 11),  q(Central Mindanao) ],
		[ qw(PH 12),  q(South Mindanao) ],
		[ qw(PL DSL), q(Dolnoslaskie) ],
		[ qw(PL K-P), q(Kujawsko-Pomorskie)],
		[ qw(PL LBL), q(Lubelskie) ],
		[ qw(PL LBS), q(Lubuskie) ],
		[ qw(PL LDZ), q(Lodzkie) ],
		[ qw(PL MAL), q(Malopolskie) ],
		[ qw(PL MAZ), q(Mazowieckie) ],
		[ qw(PL OPO), q(Opolskie) ],
		[ qw(PL PDK), q(Podkarpackie) ],
		[ qw(PL PDL), q(Podlaskie) ],
		[ qw(PL POM), q(Pomorskie) ],
		[ qw(PL SLS), q(Slaskie) ],
		[ qw(PL SWK), q(Swietokrzyskie) ],
		[ qw(PL W-M), q(Warminsko-mazurskie)],
		[ qw(PL WLK), q(Wielkopolskie) ],
		[ qw(PL Z-P), q(Zachodnio-Pomorskie)],
		[ qw(PT 10),  q(Minho-Lima) ],
		[ qw(PT 11),  q(Cavado) ],
		[ qw(PT 12),  q(Ave) ],
		[ qw(PT 13),  q(Grande Porto) ],
		[ qw(PT 14),  q(Tamega) ],
		[ qw(PT 15),  q(Entre Douro e Vouga) ],
		[ qw(PT 16),  q(Douro) ],
		[ qw(PT 17),  q(Alto Tras-os-Montes) ],
		[ qw(PT 20),  q(Baixo Vouga) ],
		[ qw(PT 21),  q(Baixo Mondego) ],
		[ qw(PT 22),  q(Pinhal Litoral) ],
		[ qw(PT 23),  q(Pinhal Interior N.) ],
		[ qw(PT 24),  q(Pinhal Interior Sul) ],
		[ qw(PT 25),  q(Dao-Lafoes) ],
		[ qw(PT 26),  q(Serra da Estrela) ],
		[ qw(PT 27),  q(Beira Interior Norte) ],
		[ qw(PT 28),  q(Beira Interior Sul) ],
		[ qw(PT 29),  q(Cova da Beira) ],
		[ qw(PT 30),  q(Oeste) ],
		[ qw(PT 31),  q(Grande Lisboa) ],
		[ qw(PT 32),  q(Peninsula de Setubal) ],
		[ qw(PT 33),  q(Medio Tejo) ],
		[ qw(PT 34),  q(Leziria do Tejo) ],
		[ qw(PT 40),  q(Alentejo Litoral) ],
		[ qw(PT 41),  q(Alto Alentejo) ],
		[ qw(PT 42),  q(Alentejo Central) ],
		[ qw(PT 43),  q(Baixo Alentejo) ],
		[ qw(PT 50),  q(Algarve) ],
		[ qw(PT 60),  q(Reg. Aut. dos Atores) ],
		[ qw(PT 70),  q(Reg. Aut. da Madeira) ],
		[ qw(RO 1),   q(Alba) ],
		[ qw(RO 2),   q(Arad) ],
		[ qw(RO 3),   q(Arges) ],
		[ qw(RO 4),   q(Bacau) ],
		[ qw(RO 5),   q(Bihor) ],
		[ qw(RO 6),   q(Bistrita-Nasaud) ],
		[ qw(RO 7),   q(Botosani) ],
		[ qw(RO 8),   q(Braila) ],
		[ qw(RO 9),   q(Brasov) ],
		[ qw(RO 10),  q(Bucuresti) ],
		[ qw(RO 11),  q(Buzau) ],
		[ qw(RO 12),  q(Calarasi) ],
		[ qw(RO 13),  q(Caras-Severin) ],
		[ qw(RO 14),  q(Cluj) ],
		[ qw(RO 15),  q(Constanta) ],
		[ qw(RO 16),  q(Covasna) ],
		[ qw(RO 17),  q(Dimbovita) ],
		[ qw(RO 18),  q(Dolj) ],
		[ qw(RO 19),  q(Galati) ],
		[ qw(RO 20),  q(Gorj) ],
		[ qw(RO 21),  q(Giurgiu) ],
		[ qw(RO 22),  q(Harghita) ],
		[ qw(RO 23),  q(Hunedoara) ],
		[ qw(RO 24),  q(Ialomita) ],
		[ qw(RO 25),  q(Iasi) ],
		[ qw(RO 26),  q(Maramures) ],
		[ qw(RO 27),  q(Mehedinti) ],
		[ qw(RO 28),  q(Mures) ],
		[ qw(RO 29),  q(Neamt) ],
		[ qw(RO 30),  q(Olt) ],
		[ qw(RO 31),  q(Prahova) ],
		[ qw(RO 32),  q(Salaj) ],
		[ qw(RO 33),  q(Satu Mare) ],
		[ qw(RO 34),  q(Sibiu) ],
		[ qw(RO 35),  q(Suceava) ],
		[ qw(RO 36),  q(Teleorman) ],
		[ qw(RO 37),  q(Timis) ],
		[ qw(RO 38),  q(Tulcea) ],
		[ qw(RO 39),  q(Vaslui) ],
		[ qw(RO 40),  q(Vilcea) ],
		[ qw(RO 41),  q(Vrancea) ],
		[ qw(RU 1),   q(Adigeja Republic) ],
		[ qw(RU 2),   q(Highlands-Altay Rep.) ],
		[ qw(RU 3),   q(Republ.of Bashkortos) ],
		[ qw(RU 4),   q(Buryat Republic) ],
		[ qw(RU 5),   q(Dagestan Republic) ],
		[ qw(RU 6),   q(Ingushetija Republic) ],
		[ qw(RU 7),   q(Kabardino-Balkar.Rep) ],
		[ qw(RU 8),   q(Kalmyk Republic) ],
		[ qw(RU 9),   q(Karach.-Cherkessk Re) ],
		[ qw(RU 10),  q(Karelian Republic) ],
		[ qw(RU 11),  q(Komi Republic) ],
		[ qw(RU 12),  q(Marijskaya Republic) ],
		[ qw(RU 13),  q(Mordovian Republic) ],
		[ qw(RU 14),  q(Yakutiya-Saha Rrepub) ],
		[ qw(RU 15),  q(North-Osetiya Republ) ],
		[ qw(RU 16),  q(Tatarstan Republic) ],
		[ qw(RU 17),  q(Tuva Republic) ],
		[ qw(RU 18),  q(The Udmurt Republic) ],
		[ qw(RU 19),  q(Chakassky Republic) ],
		[ qw(RU 20),  q(Chechenskaya Republ.) ],
		[ qw(RU 21),  q(Chuvash Republic) ],
		[ qw(RU 22),  q(Altay Territory) ],
		[ qw(RU 23),  q(Krasnodar Territory) ],
		[ qw(RU 24),  q(Krasnoyarsk Territor) ],
		[ qw(RU 25),  q(Primorye Territory) ],
		[ qw(RU 26),  q(Stavropol Territory) ],
		[ qw(RU 27),  q(Khabarovsk Territory) ],
		[ qw(RU 28),  q(The Amur Area) ],
		[ qw(RU 29),  q(The Arkhangelsk Area) ],
		[ qw(RU 30),  q(The Astrakhan Area) ],
		[ qw(RU 31),  q(The Belgorod Area) ],
		[ qw(RU 32),  q(The Bryansk Area) ],
		[ qw(RU 33),  q(The Vladimir Area) ],
		[ qw(RU 34),  q(The Volgograd Area) ],
		[ qw(RU 35),  q(The Vologda Area) ],
		[ qw(RU 36),  q(The Voronezh Area) ],
		[ qw(RU 37),  q(The Ivanovo Area) ],
		[ qw(RU 38),  q(The Irkutsk Area) ],
		[ qw(RU 39),  q(The Kaliningrad Area) ],
		[ qw(RU 40),  q(The Kaluga Area) ],
		[ qw(RU 41),  q(The Kamchatka Area) ],
		[ qw(RU 42),  q(The Kemerovo Area) ],
		[ qw(RU 43),  q(The Kirov Area) ],
		[ qw(RU 44),  q(The Kostroma Area) ],
		[ qw(RU 45),  q(The Kurgan Area) ],
		[ qw(RU 46),  q(The Kursk Area) ],
		[ qw(RU 47),  q(The Leningrad Area) ],
		[ qw(RU 48),  q(The Lipetsk Area) ],
		[ qw(RU 49),  q(The Magadan Area) ],
		[ qw(RU 50),  q(The Moscow Area) ],
		[ qw(RU 51),  q(The Murmansk Area) ],
		[ qw(RU 52),  q(The Nizhniy Novgorod) ],
		[ qw(RU 53),  q(The Novgorod Area) ],
		[ qw(RU 54),  q(The Novosibirsk Area) ],
		[ qw(RU 55),  q(The Omsk Area) ],
		[ qw(RU 56),  q(The Orenburg Area) ],
		[ qw(RU 57),  q(The Oryol Area) ],
		[ qw(RU 58),  q(The Penza Area) ],
		[ qw(RU 59),  q(The Perm Area) ],
		[ qw(RU 60),  q(The Pskov Area) ],
		[ qw(RU 61),  q(The Rostov Area) ],
		[ qw(RU 62),  q(The Ryazan Area) ],
		[ qw(RU 63),  q(The Samara Area) ],
		[ qw(RU 64),  q(The Saratov Area) ],
		[ qw(RU 65),  q(The Sakhalin Area) ],
		[ qw(RU 66),  q(The Sverdlovsk Area) ],
		[ qw(RU 67),  q(The Smolensk Area) ],
		[ qw(RU 68),  q(The Tambov Area) ],
		[ qw(RU 69),  q(The Tver Area) ],
		[ qw(RU 70),  q(The Tomsk Area) ],
		[ qw(RU 71),  q(The Tula Area) ],
		[ qw(RU 72),  q(The Tyumen Area) ],
		[ qw(RU 73),  q(The Ulyanovsk Area) ],
		[ qw(RU 74),  q(The Chelyabinsk Area) ],
		[ qw(RU 75),  q(The Chita Area) ],
		[ qw(RU 76),  q(The Yaroslavl Area) ],
		[ qw(RU 77),  q(Moscow) ],
		[ qw(RU 78),  q(St-Peterburg) ],
		[ qw(RU 79),  q(The Jewish Auton.are) ],
		[ qw(RU 80),  q(Aginsk Buryat Aut.di) ],
		[ qw(RU 81),  q(Komy Permjats.Aut.di) ],
		[ qw(RU 82),  q(Korjacs Auton.distri) ],
		[ qw(RU 83),  q(Nenekchky Auton.dist) ],
		[ qw(RU 84),  q(The Taymir Auton.dis) ],
		[ qw(RU 85),  q(Ust-Ordinsky Buryat) ],
		[ qw(RU 86),  q(Chanti-Mansyjsky Aut) ],
		[ qw(RU 87),  q(Chukotka Auton. dist) ],
		[ qw(RU 88),  q(Evensky Auton.distri) ],
		[ qw(RU 89),  q(Jamalo-Nenekchky Aut) ],
		[ qw(SE 1),   q(Blekinge County) ],
		[ qw(SE 2),   q(Dalarnas County) ],
		[ qw(SE 3),   q(Gotland County) ],
		[ qw(SE 4),   q(Gaevleborg County) ],
		[ qw(SE 5),   q(Halland County) ],
		[ qw(SE 6),   q(Jaemtland County) ],
		[ qw(SE 7),   q(Joenkoeping County) ],
		[ qw(SE 8),   q(Kalmar County) ],
		[ qw(SE 9),   q(Kronoberg County) ],
		[ qw(SE 10),  q(Norrbotten County) ],
		[ qw(SE 11),  q(Skaane County) ],
		[ qw(SE 12),  q(Stockholm County) ],
		[ qw(SE 13),  q(Soedermanland County) ],
		[ qw(SE 14),  q(Uppsala County) ],
		[ qw(SE 15),  q(Vaermland County) ],
		[ qw(SE 16),  q(Vaesterbotten County) ],
		[ qw(SE 17),  q(Vaesternorrland Cnty) ],
		[ qw(SE 18),  q(Vaestmanland County) ],
		[ qw(SE 19),  q(Vaestra Goetaland C.) ],
		[ qw(SE 20),  q(Oerebro County) ],
		[ qw(SE 21),  q(Oestergoetland Cnty) ],
		[ qw(SG SG),  q(Singapore) ],
		[ qw(SI 1),   q(Ajdovscina) ],
		[ qw(SI 2),   q(Brezice) ],
		[ qw(SI 3),   q(Celje) ],
		[ qw(SI 4),   q(Cerknica) ],
		[ qw(SI 5),   q(Crnomelj) ],
		[ qw(SI 6),   q(Dravograd) ],
		[ qw(SI 7),   q(Gornja Radgona) ],
		[ qw(SI 8),   q(Grosuplje) ],
		[ qw(SI 9),   q(Hrastnik Lasko) ],
		[ qw(SI 10),  q(Idrija) ],
		[ qw(SI 11),  q(Ilirska Bistrica) ],
		[ qw(SI 12),  q(Izola) ],
		[ qw(SI 13),  q(Jesenice) ],
		[ qw(SI 14),  q(Kamnik) ],
		[ qw(SI 15),  q(Kocevje) ],
		[ qw(SI 16),  q(Koper) ],
		[ qw(SI 17),  q(Kranj) ],
		[ qw(SI 18),  q(Krsko) ],
		[ qw(SI 19),  q(Lenart) ],
		[ qw(SI 20),  q(Lendava) ],
		[ qw(SI 21),  q(Litija) ],
		[ qw(SI 22),  q(Ljubljana-Bezigrad) ],
		[ qw(SI 23),  q(Ljubljana-Center) ],
		[ qw(SI 24),  q(Ljubljana-Moste-Polj) ],
		[ qw(SI 25),  q(Ljubljana-Siska) ],
		[ qw(SI 26),  q(Ljubljana-Vic-Rudnik) ],
		[ qw(SI 27),  q(Ljutomer) ],
		[ qw(SI 28),  q(Logatec) ],
		[ qw(SI 29),  q(Maribor) ],
		[ qw(SI 30),  q(Metlika) ],
		[ qw(SI 31),  q(Mozirje) ],
		[ qw(SI 32),  q(Murska Sobota) ],
		[ qw(SI 33),  q(Nova Gorica) ],
		[ qw(SI 34),  q(Novo Mesto) ],
		[ qw(SI 35),  q(Ormoz) ],
		[ qw(SI 36),  q(Pesnica) ],
		[ qw(SI 37),  q(Piran) ],
		[ qw(SI 38),  q(Postojna) ],
		[ qw(SI 39),  q(Ptuj) ],
		[ qw(SI 40),  q(Radlje Ob Dravi) ],
		[ qw(SI 41),  q(Radovljica) ],
		[ qw(SI 42),  q(Ravne Na Koroskem) ],
		[ qw(SI 43),  q(Ribnica) ],
		[ qw(SI 44),  q(Ruse) ],
		[ qw(SI 45),  q(Sentjur Pri Celju) ],
		[ qw(SI 46),  q(Sevnica) ],
		[ qw(SI 47),  q(Sezana) ],
		[ qw(SI 48),  q(Skofja Loka) ],
		[ qw(SI 49),  q(Slovenj Gradec) ],
		[ qw(SI 50),  q(Slovenska Bistrica) ],
		[ qw(SI 51),  q(Slovenske Konjice) ],
		[ qw(SI 52),  q(Smarje Pri Jelsah) ],
		[ qw(SI 53),  q(Tolmin) ],
		[ qw(SI 54),  q(Trbovlje) ],
		[ qw(SI 55),  q(Trebnje) ],
		[ qw(SI 56),  q(Trzic) ],
		[ qw(SI 57),  q(Velenje) ],
		[ qw(SI 58),  q(Vrhnika) ],
		[ qw(SI 59),  q(Zagorje Ob Savi) ],
		[ qw(SI 60),  q(Zalec) ],
		[ qw(SK 1),   q(Bratislava) ],
		[ qw(SK 2),   q(Zapadoslovensky) ],
		[ qw(SK 3),   q(Stredoslovensky) ],
		[ qw(SK 4),   q(Vychodoslovensky) ],
		[ qw(TH 1),   q(Amnat Charoen) ],
		[ qw(TH 2),   q(Ang Thong) ],
		[ qw(TH 3),   q(Buriram) ],
		[ qw(TH 4),   q(Chachoengsao) ],
		[ qw(TH 5),   q(Chai Nat) ],
		[ qw(TH 6),   q(Chaiyaphum) ],
		[ qw(TH 7),   q(Chanthaburi) ],
		[ qw(TH 8),   q(Chiang Mai) ],
		[ qw(TH 9),   q(Chiang Rai) ],
		[ qw(TH 10),  q(Chon Buri) ],
		[ qw(TH 11),  q(Chumphon) ],
		[ qw(TH 12),  q(Kalasin) ],
		[ qw(TH 13),  q(Kamphaeng Phet) ],
		[ qw(TH 14),  q(Kanchanaburi) ],
		[ qw(TH 15),  q(Khon Kaen) ],
		[ qw(TH 16),  q(Krabi) ],
		[ qw(TH 17),  q(Krung Thep) ],
		[ qw(TH 18),  q(Mahanakhon) ],
		[ qw(TH 19),  q(Lampang) ],
		[ qw(TH 20),  q(Lamphun) ],
		[ qw(TH 21),  q(Loei) ],
		[ qw(TH 22),  q(Lop Buri) ],
		[ qw(TH 23),  q(Mae Hong Son) ],
		[ qw(TH 24),  q(Maha Sarakham) ],
		[ qw(TH 25),  q(Mukdahan) ],
		[ qw(TH 26),  q(Nakhon Nayok) ],
		[ qw(TH 27),  q(Nakhon Pathom) ],
		[ qw(TH 28),  q(Nakhon Phanom) ],
		[ qw(TH 29),  q(Nakhon Ratchasima) ],
		[ qw(TH 30),  q(Nakhon Sawan) ],
		[ qw(TH 31),  q(Nakhon Si Thammarat) ],
		[ qw(TH 32),  q(Nan) ],
		[ qw(TH 33),  q(Narathiwat) ],
		[ qw(TH 34),  q(Nong Bua Lamphu) ],
		[ qw(TH 35),  q(Nong Khai) ],
		[ qw(TH 36),  q(Nonthaburi) ],
		[ qw(TH 37),  q(Pathum Thani) ],
		[ qw(TH 38),  q(Pattani) ],
		[ qw(TH 39),  q(Phangnga) ],
		[ qw(TH 40),  q(Phatthalung) ],
		[ qw(TH 41),  q(Phayao) ],
		[ qw(TH 42),  q(Phetchabun) ],
		[ qw(TH 43),  q(Phetchaburi) ],
		[ qw(TH 44),  q(Phichit) ],
		[ qw(TH 45),  q(Phitsanulok) ],
		[ qw(TH 46),  q(Phra Nakhon Si Ayut.) ],
		[ qw(TH 47),  q(Phrae) ],
		[ qw(TH 48),  q(Phuket) ],
		[ qw(TH 49),  q(Prachin Buri) ],
		[ qw(TR 1),   q(Adana) ],
		[ qw(TR 2),   q(Adiyaman) ],
		[ qw(TR 3),   q(Afyon) ],
		[ qw(TR 4),   q(Agri) ],
		[ qw(TR 5),   q(Amasya) ],
		[ qw(TR 6),   q(Ankara) ],
		[ qw(TR 7),   q(Antalya) ],
		[ qw(TR 8),   q(Artvin) ],
		[ qw(TR 9),   q(Aydin) ],
		[ qw(TR 10),  q(Balikesir) ],
		[ qw(TR 11),  q(Bilecik) ],
		[ qw(TR 12),  q(Bing÷l) ],
		[ qw(TR 13),  q(Bitlis) ],
		[ qw(TR 14),  q(Bolu) ],
		[ qw(TR 15),  q(Burdur) ],
		[ qw(TR 16),  q(Bursa) ],
		[ qw(TR 17),  q(Canakkale) ],
		[ qw(TR 18),  q(Cankiri) ],
		[ qw(TR 19),  q(Corum) ],
		[ qw(TR 20),  q(Denizli) ],
		[ qw(TR 21),  q(Diyarbakir) ],
		[ qw(TR 22),  q(Edirne) ],
		[ qw(TR 23),  q(Elazig) ],
		[ qw(TR 24),  q(Erzincan) ],
		[ qw(TR 25),  q(Erzurum) ],
		[ qw(TR 26),  q(Eskisehir) ],
		[ qw(TR 27),  q(Gaziantep) ],
		[ qw(TR 28),  q(Giresun) ],
		[ qw(TR 29),  q(Guemueshane) ],
		[ qw(TR 30),  q(Hakkari) ],
		[ qw(TR 31),  q(Hatay) ],
		[ qw(TR 32),  q(Isparta) ],
		[ qw(TR 33),  q(Icel) ],
		[ qw(TR 34),  q(Istanbul) ],
		[ qw(TR 35),  q(Izmir) ],
		[ qw(TR 36),  q(Kars) ],
		[ qw(TR 37),  q(Kastamonu) ],
		[ qw(TR 38),  q(Kayseri) ],
		[ qw(TR 39),  q(Kirklareli) ],
		[ qw(TR 40),  q(Kirshehir) ],
		[ qw(TR 41),  q(Kocaeli) ],
		[ qw(TR 42),  q(Konya) ],
		[ qw(TR 43),  q(Kuetahya) ],
		[ qw(TR 44),  q(Malatya) ],
		[ qw(TR 45),  q(Manisa) ],
		[ qw(TR 46),  q(K.Marash) ],
		[ qw(TR 47),  q(Mardin) ],
		[ qw(TR 48),  q(Mugla) ],
		[ qw(TR 49),  q(Mush) ],
		[ qw(TR 50),  q(Nevshehir) ],
		[ qw(TR 51),  q(Nigde) ],
		[ qw(TR 52),  q(Ordu) ],
		[ qw(TR 53),  q(Rize) ],
		[ qw(TR 54),  q(Sakarya) ],
		[ qw(TR 55),  q(Samsun) ],
		[ qw(TR 56),  q(Siirt) ],
		[ qw(TR 57),  q(Sinop) ],
		[ qw(TR 58),  q(Sivas) ],
		[ qw(TR 59),  q(Tekirdag) ],
		[ qw(TR 60),  q(Tokat) ],
		[ qw(TR 61),  q(Trabzon) ],
		[ qw(TR 62),  q(Tunceli) ],
		[ qw(TR 63),  q(Shanliurfa) ],
		[ qw(TR 64),  q(Ushak) ],
		[ qw(TR 65),  q(Van) ],
		[ qw(TR 66),  q(Yozgat) ],
		[ qw(TR 67),  q(Zonguldak) ],
		[ qw(TR 68),  q(Aksaray) ],
		[ qw(TR 69),  q(Bayburt) ],
		[ qw(TR 70),  q(Karaman) ],
		[ qw(TR 71),  q(Kirikkale) ],
		[ qw(TR 72),  q(Batman) ],
		[ qw(TR 73),  q(Shirnak) ],
		[ qw(TR 74),  q(Bartin) ],
		[ qw(TR 75),  q(Ardahan) ],
		[ qw(TR 76),  q(Igdir) ],
		[ qw(TR 77),  q(Yalova) ],
		[ qw(TW FJN), q(Fu-chien) ],
		[ qw(TW KSH), q(Kao-hsiung) ],
		[ qw(TW TPE), q(Tai-pei) ],
		[ qw(TW TWN), q(Taiwan) ],
		[ qw(UA CHG), q(Chernigivs'ka) ],
		[ qw(UA CHR), q(Cherkas'ka) ],
		[ qw(UA CHV), q(Chernovits'ka) ],
		[ qw(UA DNP), q(Dnipropetrovs'ka) ],
		[ qw(UA DON), q(Donets'ka) ],
		[ qw(UA HAR), q(Harkivs'ka) ],
		[ qw(UA HML), q(Hmel'nits'ka) ],
		[ qw(UA HRS), q(Hersons'ka) ],
		[ qw(UA IVF), q(Ivano-Frankivs'ka) ],
		[ qw(UA KIE), q(Kievs'ka) ],
		[ qw(UA KIR), q(Kirovograds'ka) ],
		[ qw(UA KRM), q(Respublika Krim) ],
		[ qw(UA L'V), q(L'vivsbka)],
		[ qw(UA LUG), q(Lugans'ka) ],
		[ qw(UA MIK), q(Mikolaivs'ka) ],
		[ qw(UA M_K), q(Kiev) ],
		[ qw(UA M_S), q(Sevastopil) ],
		[ qw(UA ODS), q(Odes'ka) ],
		[ qw(UA POL), q(Poltavs'ka) ],
		[ qw(UA RIV), q(Rivnens'ka) ],
		[ qw(UA SUM), q(Sums'ka) ],
		[ qw(UA TER), q(Ternopil's'ka) ],
		[ qw(UA VIN), q(Vinnits'ka) ],
		[ qw(UA VOL), q(Volins'ka) ],
		[ qw(UA ZAK), q(Zakarpats'ka) ],
		[ qw(UA ZAP), q(Zaporiz'ka) ],
		[ qw(UA ZHI), q(Zhitomirs'ka) ],
		[ qw(US AK),  q(Alaska) ],
		[ qw(US AL),  q(Alabama) ],
		[ qw(US AR),  q(Arkansas) ],
		[ qw(US AS),  q(American Samoa) ],
		[ qw(US AZ),  q(Arizona) ],
		[ qw(US CA),  q(California) ],
		[ qw(US CO),  q(Colorado) ],
		[ qw(US CT),  q(Connecticut) ],
		[ qw(US DC),  q(District of Columbia) ],
		[ qw(US DE),  q(Delaware) ],
		[ qw(US FL),  q(Florida) ],
		[ qw(US GA),  q(Georgia) ],
		[ qw(US GU),  q(Guam) ],
		[ qw(US HI),  q(Hawaii) ],
		[ qw(US IA),  q(Iowa) ],
		[ qw(US ID),  q(Idaho) ],
		[ qw(US IL),  q(Illinois) ],
		[ qw(US IN),  q(Indiana) ],
		[ qw(US KS),  q(Kansas) ],
		[ qw(US KY),  q(Kentucky) ],
		[ qw(US LA),  q(Louisiana) ],
		[ qw(US MA),  q(Massachusetts) ],
		[ qw(US MD),  q(Maryland) ],
		[ qw(US ME),  q(Maine) ],
		[ qw(US MI),  q(Michigan) ],
		[ qw(US MN),  q(Minnesota) ],
		[ qw(US MO),  q(Missouri) ],
		[ qw(US MP),  q(Northern Mariana Isl) ],
		[ qw(US MS),  q(Mississippi) ],
		[ qw(US MT),  q(Montana) ],
		[ qw(US NC),  q(North Carolina) ],
		[ qw(US ND),  q(North Dakota) ],
		[ qw(US NE),  q(Nebraska) ],
		[ qw(US NH),  q(New Hampshire) ],
		[ qw(US NJ),  q(New Jersey) ],
		[ qw(US NM),  q(New Mexico) ],
		[ qw(US NV),  q(Nevada) ],
		[ qw(US NY),  q(New York) ],
		[ qw(US OH),  q(Ohio) ],
		[ qw(US OK),  q(Oklahoma) ],
		[ qw(US OR),  q(Oregon) ],
		[ qw(US PA),  q(Pennsylvania) ],
		[ qw(US PR),  q(Puerto Rico) ],
		[ qw(US RI),  q(Rhode Island) ],
		[ qw(US SC),  q(South Carolina) ],
		[ qw(US SD),  q(South Dakota) ],
		[ qw(US TN),  q(Tennessee) ],
		[ qw(US TX),  q(Texas) ],
		[ qw(US UT),  q(Utah) ],
		[ qw(US VA),  q(Virginia) ],
		[ qw(US VI),  q(Virgin Islands) ],
		[ qw(US VT),  q(Vermont) ],
		[ qw(US WA),  q(Washington) ],
		[ qw(US WI),  q(Wisconsin) ],
		[ qw(US WV),  q(West Virginia) ],
		[ qw(US WY),  q(Wyoming) ],
		[ qw(VE AMA), q(Amazon) ],
		[ qw(VE ANZ), q(Anzoategui) ],
		[ qw(VE APU), q(Apure) ],
		[ qw(VE ARA), q(Aragua) ],
		[ qw(VE BAR), q(Barinas) ],
		[ qw(VE BOL), q(Bolivar) ],
		[ qw(VE CAR), q(Carabobo) ],
		[ qw(VE COJ), q(Cojedes) ],
		[ qw(VE DA),  q(Delta Amacuro) ],
		[ qw(VE DF),  q(Distrito Federal) ],
		[ qw(VE FAL), q(Falcon) ],
		[ qw(VE GUA), q(Guarico) ],
		[ qw(VE LAR), q(Lara) ],
		[ qw(VE MER), q(Merida) ],
		[ qw(VE MIR), q(Miranda) ],
		[ qw(VE MON), q(Monagas) ],
		[ qw(VE NE),  q(Nueva Esparta) ],
		[ qw(VE POR), q(Portuguesa) ],
		[ qw(VE SUC), q(Sucre) ],
		[ qw(VE TAC), q(Tachira) ],
		[ qw(VE TRU), q(Trujillo) ],
		[ qw(VE VAR), q(Vargas) ],
		[ qw(VE YAR), q(Yaracuy) ],
		[ qw(VE ZUL), q(Zulia) ],
		[ qw(ZA EC),  q(Eastern Cape) ],
		[ qw(ZA FS),  q(Freestate) ],
		[ qw(ZA GP),  q(Gauteng) ],
		[ qw(ZA KZN), q(Kwazulu/Natal) ],
		[ qw(ZA MP),  q(Mpumalanga) ],
		[ qw(ZA NC),  q(Northern Cape) ],
		[ qw(ZA NP),  q(Northern Province) ],
		[ qw(ZA NW),  q(North-West) ],
		[ qw(ZA WC),  q(Western Cape) ],
	);
	while(@data){
		my $entry = shift @data;
		my($country, $region, $name) = @$entry;
		($country_regions{$country} ||= {})->{$region} = $regions{$region} = $name;
	}
	enum Region, [ '', keys %regions ];


	sub validate_region_in_country {
		my($country, $region) = @_;
		return exists($regions{$country}) && exists($regions{$country}->{$region});
	}

	sub _get_regions {
		my ($country, $region) = @_;
		my $return = \%country_regions;
		if($country){
			$return = $return->{$country};
			if($region){
				$return = { $region => $return->{$region} };
			}
		}
		return $return;
	}

}

1;



=pod

=head1 NAME

DR::SAP::Meta::Types - Moose types used by the DR::SAP library

=head1 VERSION

version 0.15

=head1 METHODS

=head2 validate_region_in_country

Accepts a country code and a region code and validates that they are matched.

=head1 TYPES

=over 4

=item * Country

ISO-standard 2-letter country codes

=item * Currency

ISO-standard 3-letter currency codes

=item * PaymentMethod

	A => 'Domestic ACH',
	C => 'Check',
	D => 'Drafts',
	E => 'ePassporte',
	F => 'Check File',
	I => 'IACH',
	M => 'Web Money',
	P => 'Paypal',
	S => 'IACH Sepa',
	T => 'Bank Transfer',
	X => 'e4x wires',
	Y => 'e4x drafts',
	0 => 'Wire Transfer'

=item * PlatformID

	AT => 'Atlantic',
	CC => 'CCNow',
	DL => 'Direct lead',
	E5 => 'element5',
	GC => 'Global Commerce',
	GT => 'Global Tech',
	MV => 'eSellerate',
	ON => 'One Network Direct',
	RS => 'RegSoft',
	RT => 'RegNet',
	RW => 'RegNow',
	SW => 'SWReg',

=item * PaymentTerms

	'0001' => 'Due on Receipt',
	'0002' => '14 days 2%, 30 net',
	'0003' => '14 days 3%, 20/2%, 30 net',
	'0004' => 'as of end of month',
	'0005' => 'from the 10th of subs. month',
	'0006' => 'End of month 4%, Mid-month 2%',
	'0007' => '15th/31st subs. month 2%, ...',
	'0007' => '15th/31st subs. month 2%, ...',
	'0008' => '14 days 2%, 30/1,5%, 45 net',
	'0008' => '14 days 2%, 30/1,5%, 45 net',
	'0009' => 'Payable in 3 installments',
	'0012' => 'Due On Receipt',
	'0013' => 'Net Due 30 Days',
	'0014' => '2%, 10D, Due 45 Days',
	'0015' => 'Net Due 15 Days',
	'0016' => 'Net Due 60 Days',
	'0017' => 'Net Due 10 Days',
	'0018' => 'Due 5th of Month',
	'0019' => 'Net Due 90 Days',
	'0020' => 'Net Due 7 Days',
	'0021' => 'Net Due, 2 Weeks',
	'0022' => 'Net Due, 1 Month',
	'0023' => 'Net Due, 3 Weeks',
	'0024' => 'Due 10th of Month',
	'0025' => 'Net Due 19 Days',
	'0026' => 'Net Due 3 Months',
	'0027' => 'Net Due 20 Days',
	'0028' => 'Due 30th of Month',
	'0029' => 'Net Due 3 Days',
	'0030' => 'Net Due 5 days',
	'0031' => 'Net Due 75 Days',
	'0032' => 'Net Due 60 Days',
	'0033' => 'Net Due 45 Days',
	'0034' => '2% 10, Net 30',
	'0035' => '1% 10, Net 45',
	'0036' => '1% 10, Net 30',
	'0037' => 'Net Due 180 Days',
	'0038' => 'Net Due 40 Days',
	'0039' => '10 days 0.5%, 30days  net',
	'0040' => '10 days 0.75%, 30days  net',
	'0041' => '1% 10, Net 35',
	'Z001' => 'Before 5 of the next month',
	'Z002' => 'Before 10 of the next month',
	'Z003' => 'Before 15 of the next month',
	'Z004' => 'Before 20 of the next month',
	'Z005' => 'Before 23 of the next month',
	'Z006' => 'Before 25th of the next month',
	'Z007' => 'Monthly, Pay 10th Second Month (oND Commissions)',
	'Z008' => 'Before 25 in 3 months Due net',
	'Z010' => 'Before 20 for 1-15, Before 5 for 16-31 ;; Due net',
	'Z010' => 'Before 20 for 1-15, Before 5 for 16-31 ;; Due net',
	'Z011' => '2% 10, Net 45 from month end',
	'Z012' => 'Net 30 from End of Month',
	'Z013' => 'Net 45 from End of Month',
	'Z015' => 'Weekly, pay +5 days (pay 12th, 19th, 26th, 5th)',
	'Z015' => 'Weekly, pay +5 days (pay 12th, 19th, 26th, 5th)',
	'Z015' => 'Weekly, pay +5 days (pay 12th, 19th, 26th, 5th)',
	'Z015' => 'Weekly, pay +5 days (pay 12th, 19th, 26th, 5th)',
	'Z016' => 'Weekly, pay +7 days (pay 14th, 21st, 28th, 7th)',
	'Z016' => 'Weekly, pay +7 days (pay 14th, 21st, 28th, 7th)',
	'Z016' => 'Weekly, pay +7 days (pay 14th, 21st, 28th, 7th)',
	'Z016' => 'Weekly, pay +7 days (pay 14th, 21st, 28th, 7th)',
	'Z017' => 'Weekly Pay, Wednesday;; Due Net',
	'Z018' => 'Weekly Pay, Thursday;; Due Net',
	'Z019' => 'Weekly Pay, Friday;; Due Net',
	'Z102' => 'Net Due 2 Days',
	'Z104' => 'Net Due 4 Days',
	'Z106' => 'Net Due 6 Days',
	'Z108' => 'Net Due 8 Days',
	'Z110' => 'Monthly, Pay 30th of Second Month',
	'Z112' => '20th, 30th, Next 10th',
	'Z112' => '20th, 30th, Next 10th',
	'Z112' => '20th, 30th, Next 10th',
	'Z114' => '22nd, 1st next, Next 12th',
	'Z114' => '22th, 1st next, Next 12th',
	'Z114' => '22th, 1st next, Next 12th',
	'Z116' => 'Due on Receipt, 1.00% Discount',
	'Z118' => 'Due on Receipt, .75% Discount',
	'Z120' => 'Due on Receipt, .50% Discount',

=back

=over 4

=item *

PlatformID

The accepted platform IDs according to SAP (AT, CC, DL, E5, GC, GT, MV, ON, RS, RT, RW, SW)

=item *

Region

State or province codes

=item *

Uri

=back

=head1 AUTHOR

  Brian Phillips <bphillips@digitalriver.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2010 by Digital River, Inc

All rights reserved.

=cut


__END__

